# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/00_core copy.ipynb.

# %% auto 0
__all__ = ['logger', 'write_company_name']

# %% ../nbs/00_core copy.ipynb 3
import pandas as pd
import psycopg2
import typing as t
from tqdm import tqdm

import os
import csv
from tqdm import tqdm
import shutil

import logging

from .db_mgmt import get_db_credentials

# %% ../nbs/00_core copy.ipynb 4
logger = logging.getLogger(__name__)

def write_company_name(name: str, additional_info: pd.Series = None, ignore_company_if_exist: bool = True) -> int:
     
    """
    This function writes the company name to the database and any additional info.
    Each key in `additional_info` becomes a column in the database table if it doesn't exist,
    and the associated value is written to that column.
    
    If `ignore_company_if_exist` is False and the company name already exists, an error is raised.
    If `ignore_company_if_exist` is True, a warning is logged and the existing record is updated if additional info differs.
    
    Returns the ID that the database has assigned to the company name.
    """

    db_credentials = get_db_credentials()["con"]

    try:
        with psycopg2.connect(db_credentials) as conn:
            with conn.cursor() as cur:
                # Try to insert the company name
                cur.execute(
                    """
                    INSERT INTO companies (name)
                    VALUES (%s)
                    ON CONFLICT (name) DO NOTHING
                    RETURNING "ID";
                    """,
                    (name,)
                )
                result = cur.fetchone()
                
                if result is None:
                    # Company exists, handle based on ignore_company_if_exist flag
                    cur.execute(
                        """
                        SELECT "ID" FROM companies WHERE name = %s;
                        """,
                        (name,)
                    )
                    company_id = cur.fetchone()[0]
                    
                    if not ignore_company_if_exist:
                        raise ValueError(f"Company '{name}' already exists.")
                    
                    logger.warning("Company already exists, ignoring new entry")

                    # Check if additional info needs to be updated
                    if additional_info is not None:
                        for key, value in additional_info.to_dict().items():
                            # Add column if it doesn't exist
                            cur.execute(
                                f"""
                                ALTER TABLE companies
                                ADD COLUMN IF NOT EXISTS {key} TEXT;
                                """
                            )
                            
                            # Check current value before updating
                            cur.execute(
                                f"""
                                SELECT {key} FROM companies WHERE "ID" = %s;
                                """,
                                (company_id,)
                            )
                            current_value = cur.fetchone()[0]
                            
                            # Only update if the value is different
                            if current_value != value:
                                logger.warning(f"Overwriting '{key}' for company '{name}' from '{current_value}' to '{value}'.")
                                cur.execute(
                                    f"""
                                    UPDATE companies
                                    SET {key} = %s
                                    WHERE "ID" = %s;
                                    """,
                                    (value, company_id)
                                )
                else:
                    company_id = result[0]
                    
                    # Insert additional information for new entry
                    if additional_info is not None:
                        for key, value in additional_info.to_dict().items():
                            # Add column if it doesn't exist
                            cur.execute(
                                f"""
                                ALTER TABLE companies
                                ADD COLUMN IF NOT EXISTS {key} TEXT;
                                """
                            )
                            # Update the row with the additional information
                            cur.execute(
                                f"""
                                UPDATE companies
                                SET {key} = %s
                                WHERE "ID" = %s;
                                """,
                                (value, company_id)
                            )
                    
    except Exception as e:
        logger.error(f"Got error while writing company name to database: {e}")
        raise e

    return company_id
