#!/usr/bin/env python

"""
camcops_server/alembic/versions/0001_start.py

===============================================================================

    Copyright (C) 2012, University of Cambridge, Department of Psychiatry.
    Created by Rudolf Cardinal (rnc1001@cam.ac.uk).

    This file is part of CamCOPS.

    CamCOPS is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    CamCOPS is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with CamCOPS. If not, see <https://www.gnu.org/licenses/>.

===============================================================================

start
Revision ID: 0001
Revises:
Creation date: 2017-10-22 02:13:14.498367
DATABASE REVISION SCRIPT
"""

# =============================================================================
# Imports
# =============================================================================

from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects import mysql
import cardinal_pythonlib.sqlalchemy.list_types

from camcops_server.cc_modules.cc_sqla_coltypes import (
    IdNumReferenceListColType,
    PendulumDateTimeAsIsoTextColType,
    SemanticVersionColType,
)


# =============================================================================
# Revision identifiers, used by Alembic.
# =============================================================================

revision = "0001"
down_revision = None
branch_labels = None
depends_on = None


# =============================================================================
# The upgrade/downgrade steps
# =============================================================================

# noinspection PyPep8,PyTypeChecker
def upgrade():
    # ### commands auto generated by Alembic - please adjust! ###
    op.create_table(
        "_idnum_definitions",
        sa.Column("which_idnum", sa.Integer(), nullable=False),
        sa.Column("description", sa.Unicode(length=255), nullable=True),
        sa.Column("short_description", sa.Unicode(length=255), nullable=True),
        sa.Column("hl7_id_type", sa.String(length=5), nullable=True),
        sa.Column(
            "hl7_assigning_authority", sa.String(length=20), nullable=True
        ),
        sa.PrimaryKeyConstraint(
            "which_idnum", name=op.f("pk__idnum_definitions")
        ),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("_idnum_definitions", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix__idnum_definitions_which_idnum"),
            ["which_idnum"],
            unique=False,
        )

    op.create_table(
        "_security_account_lockouts",
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column("username", sa.String(length=255), nullable=False),
        sa.Column("locked_until", sa.DateTime(), nullable=False),
        sa.PrimaryKeyConstraint(
            "id", name=op.f("pk__security_account_lockouts")
        ),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table(
        "_security_account_lockouts", schema=None
    ) as batch_op:
        batch_op.create_index(
            batch_op.f("ix__security_account_lockouts_locked_until"),
            ["locked_until"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix__security_account_lockouts_username"),
            ["username"],
            unique=False,
        )

    op.create_table(
        "_security_groups",
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column("name", sa.Unicode(length=255), nullable=False),
        sa.Column("description", sa.Unicode(length=255), nullable=True),
        sa.Column("upload_policy", sa.String(length=255), nullable=True),
        sa.Column("finalize_policy", sa.String(length=255), nullable=True),
        sa.PrimaryKeyConstraint("id", name=op.f("pk__security_groups")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("_security_groups", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix__security_groups_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix__security_groups_name"), ["name"], unique=True
        )

    op.create_table(
        "_security_login_failures",
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column("username", sa.String(length=255), nullable=False),
        sa.Column("login_failure_at", sa.DateTime(), nullable=False),
        sa.PrimaryKeyConstraint(
            "id", name=op.f("pk__security_login_failures")
        ),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table(
        "_security_login_failures", schema=None
    ) as batch_op:
        batch_op.create_index(
            batch_op.f("ix__security_login_failures_login_failure_at"),
            ["login_failure_at"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix__security_login_failures_username"),
            ["username"],
            unique=False,
        )

    op.create_table(
        "_server_settings",
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column("database_title", sa.Unicode(length=255), nullable=True),
        sa.Column(
            "last_dummy_login_failure_clearance_at_utc",
            sa.DateTime(),
            nullable=True,
        ),
        sa.PrimaryKeyConstraint("id", name=op.f("pk__server_settings")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("_server_settings", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix__server_settings_id"), ["id"], unique=False
        )

    op.create_table(
        "_task_filters",
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "task_types",
            cardinal_pythonlib.sqlalchemy.list_types.StringListType(),
            nullable=True,
        ),
        sa.Column("tasks_offering_trackers_only", sa.Boolean(), nullable=True),
        sa.Column("tasks_with_patient_only", sa.Boolean(), nullable=True),
        sa.Column("surname", sa.Unicode(length=255), nullable=True),
        sa.Column("forename", sa.Unicode(length=255), nullable=True),
        sa.Column("dob", sa.Date(), nullable=True),
        sa.Column("sex", sa.String(length=1), nullable=True),
        sa.Column(
            "idnum_criteria", IdNumReferenceListColType(), nullable=True
        ),
        sa.Column(
            "device_ids",
            cardinal_pythonlib.sqlalchemy.list_types.IntListType(),
            nullable=True,
        ),
        sa.Column(
            "user_ids",
            cardinal_pythonlib.sqlalchemy.list_types.IntListType(),
            nullable=True,
        ),
        sa.Column(
            "group_ids",
            cardinal_pythonlib.sqlalchemy.list_types.IntListType(),
            nullable=True,
        ),
        sa.Column(
            "start_datetime_iso8601",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column(
            "end_datetime_iso8601",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column(
            "text_contents",
            cardinal_pythonlib.sqlalchemy.list_types.StringListType(),
            nullable=True,
        ),
        sa.Column("complete_only", sa.Boolean(), nullable=True),
        sa.PrimaryKeyConstraint("id", name=op.f("pk__task_filters")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("_task_filters", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix__task_filters_id"), ["id"], unique=False
        )

    op.create_table(
        "_hl7_run_log",
        sa.Column("run_id", sa.BigInteger(), nullable=False),
        sa.Column("start_at_utc", sa.DateTime(), nullable=False),
        sa.Column("finish_at_utc", sa.DateTime(), nullable=True),
        sa.Column("recipient", sa.String(length=255), nullable=True),
        sa.Column("type", sa.String(length=50), nullable=False),
        sa.Column("group_id", sa.Integer(), nullable=False),
        sa.Column("primary_idnum", sa.Integer(), nullable=False),
        sa.Column("require_idnum_mandatory", sa.Boolean(), nullable=True),
        sa.Column("start_date", sa.DateTime(), nullable=True),
        sa.Column("end_date", sa.DateTime(), nullable=True),
        sa.Column("finalized_only", sa.Boolean(), nullable=True),
        sa.Column("task_format", sa.String(length=50), nullable=True),
        sa.Column("xml_field_comments", sa.Boolean(), nullable=True),
        sa.Column("host", sa.String(length=255), nullable=True),
        sa.Column("port", sa.Integer(), nullable=True),
        sa.Column("divert_to_file", sa.Text(), nullable=True),
        sa.Column("treat_diverted_as_sent", sa.Boolean(), nullable=True),
        sa.Column("include_anonymous", sa.Boolean(), nullable=True),
        sa.Column("overwrite_files", sa.Boolean(), nullable=True),
        sa.Column("rio_metadata", sa.Boolean(), nullable=True),
        sa.Column("rio_idnum", sa.Integer(), nullable=True),
        sa.Column("rio_uploading_user", sa.Text(), nullable=True),
        sa.Column("rio_document_type", sa.Text(), nullable=True),
        sa.Column("script_after_file_export", sa.Text(), nullable=True),
        sa.Column("script_retcode", sa.Integer(), nullable=True),
        sa.Column("script_stdout", sa.UnicodeText(), nullable=True),
        sa.Column("script_stderr", sa.UnicodeText(), nullable=True),
        sa.ForeignKeyConstraint(
            ["group_id"],
            ["_security_groups.id"],
            name=op.f("fk__hl7_run_log_group_id"),
        ),
        sa.PrimaryKeyConstraint("run_id", name=op.f("pk__hl7_run_log")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("_hl7_run_log", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix__hl7_run_log_group_id"), ["group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix__hl7_run_log_recipient"),
            ["recipient"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix__hl7_run_log_start_at_utc"),
            ["start_at_utc"],
            unique=False,
        )

    op.create_table(
        "_security_group_group",
        sa.Column("group_id", sa.Integer(), nullable=False),
        sa.Column("can_see_group_id", sa.Integer(), nullable=False),
        sa.ForeignKeyConstraint(
            ["can_see_group_id"],
            ["_security_groups.id"],
            name=op.f("fk__security_group_group_can_see_group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["group_id"],
            ["_security_groups.id"],
            name=op.f("fk__security_group_group_group_id"),
        ),
        sa.PrimaryKeyConstraint(
            "group_id",
            "can_see_group_id",
            name=op.f("pk__security_group_group"),
        ),
    )

    op.create_table(
        "_security_users",
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column("username", sa.String(length=255), nullable=False),
        sa.Column("fullname", sa.Unicode(length=255), nullable=True),
        sa.Column("email", sa.Unicode(length=255), nullable=True),
        sa.Column("hashedpw", sa.String(length=60), nullable=False),
        sa.Column("last_login_at_utc", sa.DateTime(), nullable=True),
        sa.Column("last_password_change_utc", sa.DateTime(), nullable=True),
        sa.Column("superuser", sa.Boolean(), nullable=True),
        sa.Column("must_change_password", sa.Boolean(), nullable=True),
        sa.Column(
            "when_agreed_terms_of_use",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("upload_group_id", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["upload_group_id"],
            ["_security_groups.id"],
            name=op.f("fk__security_users_upload_group_id"),
        ),
        sa.PrimaryKeyConstraint("id", name=op.f("pk__security_users")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("_security_users", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix__security_users_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix__security_users_username"),
            ["username"],
            unique=True,
        )

    op.create_table(
        "_hl7_message_log",
        sa.Column("msg_id", sa.Integer(), nullable=False),
        sa.Column("run_id", sa.BigInteger(), nullable=True),
        sa.Column("basetable", sa.String(length=128), nullable=True),
        sa.Column("serverpk", sa.Integer(), nullable=True),
        sa.Column("sent_at_utc", sa.DateTime(), nullable=True),
        sa.Column("reply_at_utc", sa.DateTime(), nullable=True),
        sa.Column("success", sa.Boolean(), nullable=True),
        sa.Column("failure_reason", sa.Text(), nullable=True),
        # Next one altered 2018-11-09 for SQL Server:
        # sa.Column('message', sa.UnicodeText(length=4294967295), nullable=True),  # noqa: E501
        sa.Column(
            "message",
            sa.UnicodeText().with_variant(mysql.LONGTEXT, "mysql"),
            nullable=True,
        ),
        sa.Column("reply", sa.Text(), nullable=True),
        sa.Column("filename", sa.Text(), nullable=True),
        sa.Column("rio_metadata_filename", sa.Text(), nullable=True),
        sa.Column("cancelled", sa.Boolean(), nullable=True),
        sa.Column("cancelled_at_utc", sa.DateTime(), nullable=True),
        sa.ForeignKeyConstraint(
            ["run_id"],
            ["_hl7_run_log.run_id"],
            name=op.f("fk__hl7_message_log_run_id"),
        ),
        sa.PrimaryKeyConstraint("msg_id", name=op.f("pk__hl7_message_log")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("_hl7_message_log", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix__hl7_message_log_basetable"),
            ["basetable"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix__hl7_message_log_serverpk"),
            ["serverpk"],
            unique=False,
        )

    op.create_table(
        "_security_devices",
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column("name", sa.String(length=255), nullable=True),
        sa.Column("registered_by_user_id", sa.Integer(), nullable=True),
        sa.Column("when_registered_utc", sa.DateTime(), nullable=True),
        sa.Column("friendly_name", sa.Text(), nullable=True),
        sa.Column(
            "camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column("last_upload_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("ongoing_upload_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("uploading_user_id", sa.Integer(), nullable=True),
        sa.Column("currently_preserving", sa.Boolean(), nullable=True),
        sa.ForeignKeyConstraint(
            ["registered_by_user_id"],
            ["_security_users.id"],
            name=op.f("fk__security_devices_registered_by_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["uploading_user_id"],
            ["_security_users.id"],
            name=op.f("fk__security_devices_uploading_user_id"),
        ),
        sa.PrimaryKeyConstraint("id", name=op.f("pk__security_devices")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("_security_devices", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix__security_devices_name"), ["name"], unique=True
        )

    op.create_table(
        "_security_user_group",
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column("user_id", sa.Integer(), nullable=True),
        sa.Column("group_id", sa.Integer(), nullable=True),
        sa.Column("groupadmin", sa.Boolean(), nullable=True),
        sa.Column("may_upload", sa.Boolean(), nullable=True),
        sa.Column("may_register_devices", sa.Boolean(), nullable=True),
        sa.Column("may_use_webviewer", sa.Boolean(), nullable=True),
        sa.Column(
            "view_all_patients_when_unfiltered", sa.Boolean(), nullable=True
        ),
        sa.Column("may_dump_data", sa.Boolean(), nullable=True),
        sa.Column("may_run_reports", sa.Boolean(), nullable=True),
        sa.Column("may_add_notes", sa.Boolean(), nullable=True),
        sa.ForeignKeyConstraint(
            ["group_id"],
            ["_security_groups.id"],
            name=op.f("fk__security_user_group_group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["user_id"],
            ["_security_users.id"],
            name=op.f("fk__security_user_group_user_id"),
        ),
        sa.PrimaryKeyConstraint("id", name=op.f("pk__security_user_group")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    op.create_table(
        "_security_webviewer_sessions",
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column("token", sa.String(length=88), nullable=True),
        sa.Column("ip_address", sa.String(length=45), nullable=True),
        sa.Column("user_id", sa.Integer(), nullable=True),
        sa.Column("last_activity_utc", sa.DateTime(), nullable=True),
        sa.Column("number_to_view", sa.Integer(), nullable=True),
        sa.Column("task_filter_id", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["task_filter_id"],
            ["_task_filters.id"],
            name=op.f("fk__security_webviewer_sessions_task_filter_id"),
        ),
        sa.ForeignKeyConstraint(
            ["user_id"],
            ["_security_users.id"],
            name=op.f("fk__security_webviewer_sessions_user_id"),
            ondelete="CASCADE",
        ),
        sa.PrimaryKeyConstraint(
            "id", name=op.f("pk__security_webviewer_sessions")
        ),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table(
        "_security_webviewer_sessions", schema=None
    ) as batch_op:
        batch_op.create_index(
            batch_op.f("ix__security_webviewer_sessions_id"),
            ["id"],
            unique=False,
        )

    op.create_table(
        "_special_notes",
        sa.Column("note_id", sa.Integer(), nullable=False),
        sa.Column("basetable", sa.String(length=128), nullable=True),
        sa.Column("task_id", sa.Integer(), nullable=True),
        sa.Column("device_id", sa.Integer(), nullable=True),
        sa.Column("era", sa.String(length=32), nullable=True),
        sa.Column(
            "note_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("user_id", sa.Integer(), nullable=True),
        sa.Column("note", sa.UnicodeText(), nullable=True),
        sa.ForeignKeyConstraint(
            ["user_id"],
            ["_security_users.id"],
            name=op.f("fk__special_notes_user_id"),
        ),
        sa.PrimaryKeyConstraint("note_id", name=op.f("pk__special_notes")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("_special_notes", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix__special_notes_basetable"),
            ["basetable"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix__special_notes_device_id"),
            ["device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix__special_notes_era"), ["era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix__special_notes_task_id"), ["task_id"], unique=False
        )

    op.create_table(
        "_dirty_tables",
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column("device_id", sa.Integer(), nullable=True),
        sa.Column("tablename", sa.String(length=128), nullable=True),
        sa.ForeignKeyConstraint(
            ["device_id"],
            ["_security_devices.id"],
            name=op.f("fk__dirty_tables_device_id"),
        ),
        sa.PrimaryKeyConstraint("id", name=op.f("pk__dirty_tables")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )

    op.create_table(
        "_security_audit",
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column("when_access_utc", sa.DateTime(), nullable=False),
        sa.Column("source", sa.String(length=20), nullable=False),
        sa.Column("remote_addr", sa.String(length=45), nullable=True),
        sa.Column("user_id", sa.Integer(), nullable=True),
        sa.Column("device_id", sa.Integer(), nullable=True),
        sa.Column("table_name", sa.String(length=128), nullable=True),
        sa.Column("server_pk", sa.Integer(), nullable=True),
        sa.Column("patient_server_pk", sa.Integer(), nullable=True),
        sa.Column("details", sa.UnicodeText(), nullable=True),
        sa.ForeignKeyConstraint(
            ["device_id"],
            ["_security_devices.id"],
            name=op.f("fk__security_audit_device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["user_id"],
            ["_security_users.id"],
            name=op.f("fk__security_audit_user_id"),
        ),
        sa.PrimaryKeyConstraint("id", name=op.f("pk__security_audit")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("_security_audit", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix__security_audit_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix__security_audit_when_access_utc"),
            ["when_access_utc"],
            unique=False,
        )

    op.create_table(
        "ace3",
        sa.Column(
            "age_at_leaving_full_time_education", sa.Integer(), nullable=True
        ),
        sa.Column("occupation", sa.UnicodeText(), nullable=True),
        sa.Column("handedness", sa.String(length=1), nullable=True),
        sa.Column("attn_num_registration_trials", sa.Integer(), nullable=True),
        sa.Column("fluency_letters_score", sa.Integer(), nullable=True),
        sa.Column("fluency_animals_score", sa.Integer(), nullable=True),
        sa.Column("lang_follow_command_practice", sa.Integer(), nullable=True),
        sa.Column("lang_read_words_aloud", sa.Integer(), nullable=True),
        sa.Column("vsp_copy_infinity", sa.Integer(), nullable=True),
        sa.Column("vsp_copy_cube", sa.Integer(), nullable=True),
        sa.Column("vsp_draw_clock", sa.Integer(), nullable=True),
        sa.Column("picture1_blobid", sa.Integer(), nullable=True),
        sa.Column("picture1_rotation", sa.Integer(), nullable=True),
        sa.Column("picture2_blobid", sa.Integer(), nullable=True),
        sa.Column("picture2_rotation", sa.Integer(), nullable=True),
        sa.Column("comments", sa.UnicodeText(), nullable=True),
        sa.Column("attn_time1", sa.Integer(), nullable=True),
        sa.Column("attn_time2", sa.Integer(), nullable=True),
        sa.Column("attn_time3", sa.Integer(), nullable=True),
        sa.Column("attn_time4", sa.Integer(), nullable=True),
        sa.Column("attn_time5", sa.Integer(), nullable=True),
        sa.Column("attn_place1", sa.Integer(), nullable=True),
        sa.Column("attn_place2", sa.Integer(), nullable=True),
        sa.Column("attn_place3", sa.Integer(), nullable=True),
        sa.Column("attn_place4", sa.Integer(), nullable=True),
        sa.Column("attn_place5", sa.Integer(), nullable=True),
        sa.Column("attn_repeat_word1", sa.Integer(), nullable=True),
        sa.Column("attn_repeat_word2", sa.Integer(), nullable=True),
        sa.Column("attn_repeat_word3", sa.Integer(), nullable=True),
        sa.Column("attn_serial7_subtraction1", sa.Integer(), nullable=True),
        sa.Column("attn_serial7_subtraction2", sa.Integer(), nullable=True),
        sa.Column("attn_serial7_subtraction3", sa.Integer(), nullable=True),
        sa.Column("attn_serial7_subtraction4", sa.Integer(), nullable=True),
        sa.Column("attn_serial7_subtraction5", sa.Integer(), nullable=True),
        sa.Column("mem_recall_word1", sa.Integer(), nullable=True),
        sa.Column("mem_recall_word2", sa.Integer(), nullable=True),
        sa.Column("mem_recall_word3", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial1_1", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial1_2", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial1_3", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial1_4", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial1_5", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial1_6", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial1_7", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial2_1", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial2_2", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial2_3", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial2_4", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial2_5", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial2_6", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial2_7", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial3_1", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial3_2", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial3_3", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial3_4", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial3_5", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial3_6", sa.Integer(), nullable=True),
        sa.Column("mem_repeat_address_trial3_7", sa.Integer(), nullable=True),
        sa.Column("mem_famous1", sa.Integer(), nullable=True),
        sa.Column("mem_famous2", sa.Integer(), nullable=True),
        sa.Column("mem_famous3", sa.Integer(), nullable=True),
        sa.Column("mem_famous4", sa.Integer(), nullable=True),
        sa.Column("lang_follow_command1", sa.Integer(), nullable=True),
        sa.Column("lang_follow_command2", sa.Integer(), nullable=True),
        sa.Column("lang_follow_command3", sa.Integer(), nullable=True),
        sa.Column("lang_write_sentences_point1", sa.Integer(), nullable=True),
        sa.Column("lang_write_sentences_point2", sa.Integer(), nullable=True),
        sa.Column("lang_repeat_word1", sa.Integer(), nullable=True),
        sa.Column("lang_repeat_word2", sa.Integer(), nullable=True),
        sa.Column("lang_repeat_word3", sa.Integer(), nullable=True),
        sa.Column("lang_repeat_word4", sa.Integer(), nullable=True),
        sa.Column("lang_repeat_sentence1", sa.Integer(), nullable=True),
        sa.Column("lang_repeat_sentence2", sa.Integer(), nullable=True),
        sa.Column("lang_name_picture1", sa.Integer(), nullable=True),
        sa.Column("lang_name_picture2", sa.Integer(), nullable=True),
        sa.Column("lang_name_picture3", sa.Integer(), nullable=True),
        sa.Column("lang_name_picture4", sa.Integer(), nullable=True),
        sa.Column("lang_name_picture5", sa.Integer(), nullable=True),
        sa.Column("lang_name_picture6", sa.Integer(), nullable=True),
        sa.Column("lang_name_picture7", sa.Integer(), nullable=True),
        sa.Column("lang_name_picture8", sa.Integer(), nullable=True),
        sa.Column("lang_name_picture9", sa.Integer(), nullable=True),
        sa.Column("lang_name_picture10", sa.Integer(), nullable=True),
        sa.Column("lang_name_picture11", sa.Integer(), nullable=True),
        sa.Column("lang_name_picture12", sa.Integer(), nullable=True),
        sa.Column("lang_identify_concept1", sa.Integer(), nullable=True),
        sa.Column("lang_identify_concept2", sa.Integer(), nullable=True),
        sa.Column("lang_identify_concept3", sa.Integer(), nullable=True),
        sa.Column("lang_identify_concept4", sa.Integer(), nullable=True),
        sa.Column("vsp_count_dots1", sa.Integer(), nullable=True),
        sa.Column("vsp_count_dots2", sa.Integer(), nullable=True),
        sa.Column("vsp_count_dots3", sa.Integer(), nullable=True),
        sa.Column("vsp_count_dots4", sa.Integer(), nullable=True),
        sa.Column("vsp_identify_letter1", sa.Integer(), nullable=True),
        sa.Column("vsp_identify_letter2", sa.Integer(), nullable=True),
        sa.Column("vsp_identify_letter3", sa.Integer(), nullable=True),
        sa.Column("vsp_identify_letter4", sa.Integer(), nullable=True),
        sa.Column("mem_recall_address1", sa.Integer(), nullable=True),
        sa.Column("mem_recall_address2", sa.Integer(), nullable=True),
        sa.Column("mem_recall_address3", sa.Integer(), nullable=True),
        sa.Column("mem_recall_address4", sa.Integer(), nullable=True),
        sa.Column("mem_recall_address5", sa.Integer(), nullable=True),
        sa.Column("mem_recall_address6", sa.Integer(), nullable=True),
        sa.Column("mem_recall_address7", sa.Integer(), nullable=True),
        sa.Column("mem_recognize_address1", sa.Integer(), nullable=True),
        sa.Column("mem_recognize_address2", sa.Integer(), nullable=True),
        sa.Column("mem_recognize_address3", sa.Integer(), nullable=True),
        sa.Column("mem_recognize_address4", sa.Integer(), nullable=True),
        sa.Column("mem_recognize_address5", sa.Integer(), nullable=True),
        sa.Column(
            "mem_recognize_address_choice1", sa.String(length=1), nullable=True
        ),
        sa.Column(
            "mem_recognize_address_choice2", sa.String(length=1), nullable=True
        ),
        sa.Column(
            "mem_recognize_address_choice3", sa.String(length=1), nullable=True
        ),
        sa.Column(
            "mem_recognize_address_choice4", sa.String(length=1), nullable=True
        ),
        sa.Column(
            "mem_recognize_address_choice5", sa.String(length=1), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ace3__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_ace3__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_ace3__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ace3__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ace3__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ace3__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_ace3")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("ace3", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_ace3__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ace3__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ace3__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ace3__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_ace3__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_ace3_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_ace3_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ace3_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "aims",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_aims__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_aims__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_aims__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_aims__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_aims__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_aims__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_aims")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("aims", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_aims__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_aims__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_aims__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_aims__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_aims__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_aims_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_aims_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_aims_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "audit",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_audit__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_audit__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_audit__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_audit__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_audit__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_audit__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_audit")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("audit", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_audit__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_audit__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_audit__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_audit__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_audit__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_audit_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_audit_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_audit_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "audit_c",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_audit_c__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_audit_c__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_audit_c__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_audit_c__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_audit_c__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_audit_c__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_audit_c")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("audit_c", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_audit_c__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_audit_c__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_audit_c__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_audit_c__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_audit_c__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_audit_c_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_audit_c_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_audit_c_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "badls",
        sa.Column("q1", sa.String(length=1), nullable=True),
        sa.Column("q2", sa.String(length=1), nullable=True),
        sa.Column("q3", sa.String(length=1), nullable=True),
        sa.Column("q4", sa.String(length=1), nullable=True),
        sa.Column("q5", sa.String(length=1), nullable=True),
        sa.Column("q6", sa.String(length=1), nullable=True),
        sa.Column("q7", sa.String(length=1), nullable=True),
        sa.Column("q8", sa.String(length=1), nullable=True),
        sa.Column("q9", sa.String(length=1), nullable=True),
        sa.Column("q10", sa.String(length=1), nullable=True),
        sa.Column("q11", sa.String(length=1), nullable=True),
        sa.Column("q12", sa.String(length=1), nullable=True),
        sa.Column("q13", sa.String(length=1), nullable=True),
        sa.Column("q14", sa.String(length=1), nullable=True),
        sa.Column("q15", sa.String(length=1), nullable=True),
        sa.Column("q16", sa.String(length=1), nullable=True),
        sa.Column("q17", sa.String(length=1), nullable=True),
        sa.Column("q18", sa.String(length=1), nullable=True),
        sa.Column("q19", sa.String(length=1), nullable=True),
        sa.Column("q20", sa.String(length=1), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("respondent_relationship", sa.Text(), nullable=True),
        sa.Column("respondent_name", sa.Text(), nullable=True),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_badls__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_badls__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_badls__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_badls__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_badls__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_badls__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_badls")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("badls", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_badls__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_badls__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_badls__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_badls__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_badls__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_badls_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_badls_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_badls_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "bdi",
        sa.Column("bdi_scale", sa.String(length=10), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("q18", sa.Integer(), nullable=True),
        sa.Column("q19", sa.Integer(), nullable=True),
        sa.Column("q20", sa.Integer(), nullable=True),
        sa.Column("q21", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bdi__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_bdi__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_bdi__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bdi__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bdi__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bdi__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_bdi")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("bdi", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_bdi__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bdi__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bdi__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bdi__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_bdi__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_bdi_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_bdi_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bdi_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "blobs",
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column("tablename", sa.String(length=128), nullable=False),
        sa.Column("tablepk", sa.Integer(), nullable=False),
        sa.Column("fieldname", sa.String(length=128), nullable=False),
        sa.Column("filename", sa.Text(), nullable=True),
        sa.Column("mimetype", sa.String(length=255), nullable=True),
        sa.Column("image_rotation_deg_cw", sa.Integer(), nullable=True),
        # Next one altered 2018-11-09 for SQL Server:
        # sa.Column('theblob', sa.LargeBinary(length=4294967295), nullable=True),  # noqa: E501
        sa.Column(
            "theblob",
            sa.LargeBinary().with_variant(mysql.LONGBLOB, "mysql"),
            nullable=True,
        ),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_blobs__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_blobs__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_blobs__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_blobs__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_blobs__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_blobs__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_blobs")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("blobs", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_blobs__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_blobs__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_blobs__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_blobs__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_blobs__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_blobs_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "bmi",
        sa.Column("height_m", sa.Float(), nullable=True),
        sa.Column("mass_kg", sa.Float(), nullable=True),
        sa.Column("comment", sa.UnicodeText(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bmi__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_bmi__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_bmi__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bmi__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bmi__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bmi__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_bmi")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("bmi", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_bmi__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bmi__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bmi__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bmi__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_bmi__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_bmi_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_bmi_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bmi_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "bprs",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("q18", sa.Integer(), nullable=True),
        sa.Column("q19", sa.Integer(), nullable=True),
        sa.Column("q20", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bprs__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_bprs__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_bprs__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bprs__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bprs__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bprs__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_bprs")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("bprs", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_bprs__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bprs__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bprs__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bprs__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_bprs__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_bprs_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_bprs_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bprs_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "bprse",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("q18", sa.Integer(), nullable=True),
        sa.Column("q19", sa.Integer(), nullable=True),
        sa.Column("q20", sa.Integer(), nullable=True),
        sa.Column("q21", sa.Integer(), nullable=True),
        sa.Column("q22", sa.Integer(), nullable=True),
        sa.Column("q23", sa.Integer(), nullable=True),
        sa.Column("q24", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bprse__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_bprse__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_bprse__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bprse__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bprse__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_bprse__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_bprse")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("bprse", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_bprse__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bprse__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bprse__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bprse__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bprse__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_bprse_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_bprse_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_bprse_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cage",
        sa.Column("q1", sa.String(length=1), nullable=True),
        sa.Column("q2", sa.String(length=1), nullable=True),
        sa.Column("q3", sa.String(length=1), nullable=True),
        sa.Column("q4", sa.String(length=1), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cage__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_cage__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_cage__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cage__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cage__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cage__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_cage")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("cage", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cage__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cage__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cage__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cage__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_cage__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_cage_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_cage_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cage_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cape42",
        sa.Column("frequency1", sa.Integer(), nullable=True),
        sa.Column("frequency2", sa.Integer(), nullable=True),
        sa.Column("frequency3", sa.Integer(), nullable=True),
        sa.Column("frequency4", sa.Integer(), nullable=True),
        sa.Column("frequency5", sa.Integer(), nullable=True),
        sa.Column("frequency6", sa.Integer(), nullable=True),
        sa.Column("frequency7", sa.Integer(), nullable=True),
        sa.Column("frequency8", sa.Integer(), nullable=True),
        sa.Column("frequency9", sa.Integer(), nullable=True),
        sa.Column("frequency10", sa.Integer(), nullable=True),
        sa.Column("frequency11", sa.Integer(), nullable=True),
        sa.Column("frequency12", sa.Integer(), nullable=True),
        sa.Column("frequency13", sa.Integer(), nullable=True),
        sa.Column("frequency14", sa.Integer(), nullable=True),
        sa.Column("frequency15", sa.Integer(), nullable=True),
        sa.Column("frequency16", sa.Integer(), nullable=True),
        sa.Column("frequency17", sa.Integer(), nullable=True),
        sa.Column("frequency18", sa.Integer(), nullable=True),
        sa.Column("frequency19", sa.Integer(), nullable=True),
        sa.Column("frequency20", sa.Integer(), nullable=True),
        sa.Column("frequency21", sa.Integer(), nullable=True),
        sa.Column("frequency22", sa.Integer(), nullable=True),
        sa.Column("frequency23", sa.Integer(), nullable=True),
        sa.Column("frequency24", sa.Integer(), nullable=True),
        sa.Column("frequency25", sa.Integer(), nullable=True),
        sa.Column("frequency26", sa.Integer(), nullable=True),
        sa.Column("frequency27", sa.Integer(), nullable=True),
        sa.Column("frequency28", sa.Integer(), nullable=True),
        sa.Column("frequency29", sa.Integer(), nullable=True),
        sa.Column("frequency30", sa.Integer(), nullable=True),
        sa.Column("frequency31", sa.Integer(), nullable=True),
        sa.Column("frequency32", sa.Integer(), nullable=True),
        sa.Column("frequency33", sa.Integer(), nullable=True),
        sa.Column("frequency34", sa.Integer(), nullable=True),
        sa.Column("frequency35", sa.Integer(), nullable=True),
        sa.Column("frequency36", sa.Integer(), nullable=True),
        sa.Column("frequency37", sa.Integer(), nullable=True),
        sa.Column("frequency38", sa.Integer(), nullable=True),
        sa.Column("frequency39", sa.Integer(), nullable=True),
        sa.Column("frequency40", sa.Integer(), nullable=True),
        sa.Column("frequency41", sa.Integer(), nullable=True),
        sa.Column("frequency42", sa.Integer(), nullable=True),
        sa.Column("distress1", sa.Integer(), nullable=True),
        sa.Column("distress2", sa.Integer(), nullable=True),
        sa.Column("distress3", sa.Integer(), nullable=True),
        sa.Column("distress4", sa.Integer(), nullable=True),
        sa.Column("distress5", sa.Integer(), nullable=True),
        sa.Column("distress6", sa.Integer(), nullable=True),
        sa.Column("distress7", sa.Integer(), nullable=True),
        sa.Column("distress8", sa.Integer(), nullable=True),
        sa.Column("distress9", sa.Integer(), nullable=True),
        sa.Column("distress10", sa.Integer(), nullable=True),
        sa.Column("distress11", sa.Integer(), nullable=True),
        sa.Column("distress12", sa.Integer(), nullable=True),
        sa.Column("distress13", sa.Integer(), nullable=True),
        sa.Column("distress14", sa.Integer(), nullable=True),
        sa.Column("distress15", sa.Integer(), nullable=True),
        sa.Column("distress16", sa.Integer(), nullable=True),
        sa.Column("distress17", sa.Integer(), nullable=True),
        sa.Column("distress18", sa.Integer(), nullable=True),
        sa.Column("distress19", sa.Integer(), nullable=True),
        sa.Column("distress20", sa.Integer(), nullable=True),
        sa.Column("distress21", sa.Integer(), nullable=True),
        sa.Column("distress22", sa.Integer(), nullable=True),
        sa.Column("distress23", sa.Integer(), nullable=True),
        sa.Column("distress24", sa.Integer(), nullable=True),
        sa.Column("distress25", sa.Integer(), nullable=True),
        sa.Column("distress26", sa.Integer(), nullable=True),
        sa.Column("distress27", sa.Integer(), nullable=True),
        sa.Column("distress28", sa.Integer(), nullable=True),
        sa.Column("distress29", sa.Integer(), nullable=True),
        sa.Column("distress30", sa.Integer(), nullable=True),
        sa.Column("distress31", sa.Integer(), nullable=True),
        sa.Column("distress32", sa.Integer(), nullable=True),
        sa.Column("distress33", sa.Integer(), nullable=True),
        sa.Column("distress34", sa.Integer(), nullable=True),
        sa.Column("distress35", sa.Integer(), nullable=True),
        sa.Column("distress36", sa.Integer(), nullable=True),
        sa.Column("distress37", sa.Integer(), nullable=True),
        sa.Column("distress38", sa.Integer(), nullable=True),
        sa.Column("distress39", sa.Integer(), nullable=True),
        sa.Column("distress40", sa.Integer(), nullable=True),
        sa.Column("distress41", sa.Integer(), nullable=True),
        sa.Column("distress42", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cape42__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_cape42__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_cape42__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cape42__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cape42__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cape42__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_cape42")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("cape42", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cape42__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cape42__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cape42__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cape42__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cape42__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_cape42_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_cape42_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cape42_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "caps",
        sa.Column("endorse1", sa.Integer(), nullable=True),
        sa.Column("endorse2", sa.Integer(), nullable=True),
        sa.Column("endorse3", sa.Integer(), nullable=True),
        sa.Column("endorse4", sa.Integer(), nullable=True),
        sa.Column("endorse5", sa.Integer(), nullable=True),
        sa.Column("endorse6", sa.Integer(), nullable=True),
        sa.Column("endorse7", sa.Integer(), nullable=True),
        sa.Column("endorse8", sa.Integer(), nullable=True),
        sa.Column("endorse9", sa.Integer(), nullable=True),
        sa.Column("endorse10", sa.Integer(), nullable=True),
        sa.Column("endorse11", sa.Integer(), nullable=True),
        sa.Column("endorse12", sa.Integer(), nullable=True),
        sa.Column("endorse13", sa.Integer(), nullable=True),
        sa.Column("endorse14", sa.Integer(), nullable=True),
        sa.Column("endorse15", sa.Integer(), nullable=True),
        sa.Column("endorse16", sa.Integer(), nullable=True),
        sa.Column("endorse17", sa.Integer(), nullable=True),
        sa.Column("endorse18", sa.Integer(), nullable=True),
        sa.Column("endorse19", sa.Integer(), nullable=True),
        sa.Column("endorse20", sa.Integer(), nullable=True),
        sa.Column("endorse21", sa.Integer(), nullable=True),
        sa.Column("endorse22", sa.Integer(), nullable=True),
        sa.Column("endorse23", sa.Integer(), nullable=True),
        sa.Column("endorse24", sa.Integer(), nullable=True),
        sa.Column("endorse25", sa.Integer(), nullable=True),
        sa.Column("endorse26", sa.Integer(), nullable=True),
        sa.Column("endorse27", sa.Integer(), nullable=True),
        sa.Column("endorse28", sa.Integer(), nullable=True),
        sa.Column("endorse29", sa.Integer(), nullable=True),
        sa.Column("endorse30", sa.Integer(), nullable=True),
        sa.Column("endorse31", sa.Integer(), nullable=True),
        sa.Column("endorse32", sa.Integer(), nullable=True),
        sa.Column("distress1", sa.Integer(), nullable=True),
        sa.Column("distress2", sa.Integer(), nullable=True),
        sa.Column("distress3", sa.Integer(), nullable=True),
        sa.Column("distress4", sa.Integer(), nullable=True),
        sa.Column("distress5", sa.Integer(), nullable=True),
        sa.Column("distress6", sa.Integer(), nullable=True),
        sa.Column("distress7", sa.Integer(), nullable=True),
        sa.Column("distress8", sa.Integer(), nullable=True),
        sa.Column("distress9", sa.Integer(), nullable=True),
        sa.Column("distress10", sa.Integer(), nullable=True),
        sa.Column("distress11", sa.Integer(), nullable=True),
        sa.Column("distress12", sa.Integer(), nullable=True),
        sa.Column("distress13", sa.Integer(), nullable=True),
        sa.Column("distress14", sa.Integer(), nullable=True),
        sa.Column("distress15", sa.Integer(), nullable=True),
        sa.Column("distress16", sa.Integer(), nullable=True),
        sa.Column("distress17", sa.Integer(), nullable=True),
        sa.Column("distress18", sa.Integer(), nullable=True),
        sa.Column("distress19", sa.Integer(), nullable=True),
        sa.Column("distress20", sa.Integer(), nullable=True),
        sa.Column("distress21", sa.Integer(), nullable=True),
        sa.Column("distress22", sa.Integer(), nullable=True),
        sa.Column("distress23", sa.Integer(), nullable=True),
        sa.Column("distress24", sa.Integer(), nullable=True),
        sa.Column("distress25", sa.Integer(), nullable=True),
        sa.Column("distress26", sa.Integer(), nullable=True),
        sa.Column("distress27", sa.Integer(), nullable=True),
        sa.Column("distress28", sa.Integer(), nullable=True),
        sa.Column("distress29", sa.Integer(), nullable=True),
        sa.Column("distress30", sa.Integer(), nullable=True),
        sa.Column("distress31", sa.Integer(), nullable=True),
        sa.Column("distress32", sa.Integer(), nullable=True),
        sa.Column("intrusiveness1", sa.Integer(), nullable=True),
        sa.Column("intrusiveness2", sa.Integer(), nullable=True),
        sa.Column("intrusiveness3", sa.Integer(), nullable=True),
        sa.Column("intrusiveness4", sa.Integer(), nullable=True),
        sa.Column("intrusiveness5", sa.Integer(), nullable=True),
        sa.Column("intrusiveness6", sa.Integer(), nullable=True),
        sa.Column("intrusiveness7", sa.Integer(), nullable=True),
        sa.Column("intrusiveness8", sa.Integer(), nullable=True),
        sa.Column("intrusiveness9", sa.Integer(), nullable=True),
        sa.Column("intrusiveness10", sa.Integer(), nullable=True),
        sa.Column("intrusiveness11", sa.Integer(), nullable=True),
        sa.Column("intrusiveness12", sa.Integer(), nullable=True),
        sa.Column("intrusiveness13", sa.Integer(), nullable=True),
        sa.Column("intrusiveness14", sa.Integer(), nullable=True),
        sa.Column("intrusiveness15", sa.Integer(), nullable=True),
        sa.Column("intrusiveness16", sa.Integer(), nullable=True),
        sa.Column("intrusiveness17", sa.Integer(), nullable=True),
        sa.Column("intrusiveness18", sa.Integer(), nullable=True),
        sa.Column("intrusiveness19", sa.Integer(), nullable=True),
        sa.Column("intrusiveness20", sa.Integer(), nullable=True),
        sa.Column("intrusiveness21", sa.Integer(), nullable=True),
        sa.Column("intrusiveness22", sa.Integer(), nullable=True),
        sa.Column("intrusiveness23", sa.Integer(), nullable=True),
        sa.Column("intrusiveness24", sa.Integer(), nullable=True),
        sa.Column("intrusiveness25", sa.Integer(), nullable=True),
        sa.Column("intrusiveness26", sa.Integer(), nullable=True),
        sa.Column("intrusiveness27", sa.Integer(), nullable=True),
        sa.Column("intrusiveness28", sa.Integer(), nullable=True),
        sa.Column("intrusiveness29", sa.Integer(), nullable=True),
        sa.Column("intrusiveness30", sa.Integer(), nullable=True),
        sa.Column("intrusiveness31", sa.Integer(), nullable=True),
        sa.Column("intrusiveness32", sa.Integer(), nullable=True),
        sa.Column("frequency1", sa.Integer(), nullable=True),
        sa.Column("frequency2", sa.Integer(), nullable=True),
        sa.Column("frequency3", sa.Integer(), nullable=True),
        sa.Column("frequency4", sa.Integer(), nullable=True),
        sa.Column("frequency5", sa.Integer(), nullable=True),
        sa.Column("frequency6", sa.Integer(), nullable=True),
        sa.Column("frequency7", sa.Integer(), nullable=True),
        sa.Column("frequency8", sa.Integer(), nullable=True),
        sa.Column("frequency9", sa.Integer(), nullable=True),
        sa.Column("frequency10", sa.Integer(), nullable=True),
        sa.Column("frequency11", sa.Integer(), nullable=True),
        sa.Column("frequency12", sa.Integer(), nullable=True),
        sa.Column("frequency13", sa.Integer(), nullable=True),
        sa.Column("frequency14", sa.Integer(), nullable=True),
        sa.Column("frequency15", sa.Integer(), nullable=True),
        sa.Column("frequency16", sa.Integer(), nullable=True),
        sa.Column("frequency17", sa.Integer(), nullable=True),
        sa.Column("frequency18", sa.Integer(), nullable=True),
        sa.Column("frequency19", sa.Integer(), nullable=True),
        sa.Column("frequency20", sa.Integer(), nullable=True),
        sa.Column("frequency21", sa.Integer(), nullable=True),
        sa.Column("frequency22", sa.Integer(), nullable=True),
        sa.Column("frequency23", sa.Integer(), nullable=True),
        sa.Column("frequency24", sa.Integer(), nullable=True),
        sa.Column("frequency25", sa.Integer(), nullable=True),
        sa.Column("frequency26", sa.Integer(), nullable=True),
        sa.Column("frequency27", sa.Integer(), nullable=True),
        sa.Column("frequency28", sa.Integer(), nullable=True),
        sa.Column("frequency29", sa.Integer(), nullable=True),
        sa.Column("frequency30", sa.Integer(), nullable=True),
        sa.Column("frequency31", sa.Integer(), nullable=True),
        sa.Column("frequency32", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_caps__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_caps__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_caps__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_caps__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_caps__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_caps__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_caps")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("caps", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_caps__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_caps__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_caps__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_caps__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_caps__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_caps_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_caps_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_caps_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cardinal_expdet",
        sa.Column("num_blocks", sa.Integer(), nullable=True),
        sa.Column("stimulus_counterbalancing", sa.Integer(), nullable=True),
        sa.Column(
            "is_detection_response_on_right", sa.Integer(), nullable=True
        ),
        sa.Column("pause_every_n_trials", sa.Integer(), nullable=True),
        sa.Column("cue_duration_s", sa.Float(), nullable=True),
        sa.Column("visual_cue_intensity", sa.Float(), nullable=True),
        sa.Column("auditory_cue_intensity", sa.Float(), nullable=True),
        sa.Column("isi_duration_s", sa.Float(), nullable=True),
        sa.Column("visual_target_duration_s", sa.Float(), nullable=True),
        sa.Column("visual_background_intensity", sa.Float(), nullable=True),
        sa.Column("visual_target_0_intensity", sa.Float(), nullable=True),
        sa.Column("visual_target_1_intensity", sa.Float(), nullable=True),
        sa.Column("auditory_background_intensity", sa.Float(), nullable=True),
        sa.Column("auditory_target_0_intensity", sa.Float(), nullable=True),
        sa.Column("auditory_target_1_intensity", sa.Float(), nullable=True),
        sa.Column("iti_min_s", sa.Float(), nullable=True),
        sa.Column("iti_max_s", sa.Float(), nullable=True),
        sa.Column("aborted", sa.Integer(), nullable=True),
        sa.Column("finished", sa.Integer(), nullable=True),
        sa.Column("last_trial_completed", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdet__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_cardinal_expdet__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_cardinal_expdet__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdet__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdet__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdet__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_cardinal_expdet")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("cardinal_expdet", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cardinal_expdet_trialgroupspec",
        sa.Column("cardinal_expdet_id", sa.Integer(), nullable=False),
        sa.Column("group_num", sa.Integer(), nullable=False),
        sa.Column("cue", sa.Integer(), nullable=True),
        sa.Column("target_modality", sa.Integer(), nullable=True),
        sa.Column("target_number", sa.Integer(), nullable=True),
        sa.Column("n_target", sa.Integer(), nullable=True),
        sa.Column("n_no_target", sa.Integer(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdet_trialgroupspec__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_cardinal_expdet_trialgroupspec__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_cardinal_expdet_trialgroupspec__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f(
                "fk_cardinal_expdet_trialgroupspec__manually_erasing_user_id"
            ),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdet_trialgroupspec__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdet_trialgroupspec__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint(
            "_pk", name=op.f("pk_cardinal_expdet_trialgroupspec")
        ),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table(
        "cardinal_expdet_trialgroupspec", schema=None
    ) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_trialgroupspec__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_trialgroupspec__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_trialgroupspec__era"),
            ["_era"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_trialgroupspec__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_trialgroupspec__pk"),
            ["_pk"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_trialgroupspec_id"),
            ["id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_trialgroupspec_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cardinal_expdet_trials",
        sa.Column("cardinal_expdet_id", sa.Integer(), nullable=False),
        sa.Column("trial", sa.Integer(), nullable=False),
        sa.Column("block", sa.Integer(), nullable=True),
        sa.Column("group_num", sa.Integer(), nullable=True),
        sa.Column("cue", sa.Integer(), nullable=True),
        sa.Column("raw_cue_number", sa.Integer(), nullable=True),
        sa.Column("target_modality", sa.Integer(), nullable=True),
        sa.Column("target_number", sa.Integer(), nullable=True),
        sa.Column("target_present", sa.Integer(), nullable=True),
        sa.Column("iti_length_s", sa.Float(), nullable=True),
        sa.Column("pause_given_before_trial", sa.Integer(), nullable=True),
        sa.Column(
            "pause_start_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column(
            "pause_end_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column(
            "trial_start_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column(
            "cue_start_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column(
            "target_start_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column(
            "detection_start_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column(
            "iti_start_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column(
            "iti_end_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column(
            "trial_end_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("responded", sa.Integer(), nullable=True),
        sa.Column(
            "response_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("response_latency_ms", sa.Integer(), nullable=True),
        sa.Column("rating", sa.Integer(), nullable=True),
        sa.Column("correct", sa.Integer(), nullable=True),
        sa.Column("points", sa.Integer(), nullable=True),
        sa.Column("cumulative_points", sa.Integer(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdet_trials__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_cardinal_expdet_trials__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_cardinal_expdet_trials__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdet_trials__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdet_trials__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdet_trials__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_cardinal_expdet_trials")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table(
        "cardinal_expdet_trials", schema=None
    ) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_trials__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_trials__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_trials__era"),
            ["_era"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_trials__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_trials__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_trials_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdet_trials_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cardinal_expdetthreshold",
        sa.Column("modality", sa.Integer(), nullable=True),
        sa.Column("target_number", sa.Integer(), nullable=True),
        sa.Column("background_filename", sa.Text(), nullable=True),
        sa.Column("target_filename", sa.Text(), nullable=True),
        sa.Column("visual_target_duration_s", sa.Float(), nullable=True),
        sa.Column("background_intensity", sa.Float(), nullable=True),
        sa.Column("start_intensity_min", sa.Float(), nullable=True),
        sa.Column("start_intensity_max", sa.Float(), nullable=True),
        sa.Column("initial_large_intensity_step", sa.Float(), nullable=True),
        sa.Column("main_small_intensity_step", sa.Float(), nullable=True),
        sa.Column("num_trials_in_main_sequence", sa.Integer(), nullable=True),
        sa.Column("p_catch_trial", sa.Float(), nullable=True),
        sa.Column("prompt", sa.UnicodeText(), nullable=True),
        sa.Column("iti_s", sa.Float(), nullable=True),
        sa.Column("finished", sa.Integer(), nullable=True),
        sa.Column("intercept", sa.Float(), nullable=True),
        sa.Column("slope", sa.Float(), nullable=True),
        sa.Column("k", sa.Float(), nullable=True),
        sa.Column("theta", sa.Float(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdetthreshold__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_cardinal_expdetthreshold__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_cardinal_expdetthreshold__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdetthreshold__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdetthreshold__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdetthreshold__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint(
            "_pk", name=op.f("pk_cardinal_expdetthreshold")
        ),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table(
        "cardinal_expdetthreshold", schema=None
    ) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdetthreshold__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdetthreshold__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdetthreshold__era"),
            ["_era"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdetthreshold__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdetthreshold__pk"),
            ["_pk"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdetthreshold_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdetthreshold_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdetthreshold_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cardinal_expdetthreshold_trials",
        sa.Column("cardinal_expdetthreshold_id", sa.Integer(), nullable=False),
        sa.Column("trial", sa.Integer(), nullable=False),
        sa.Column("trial_ignoring_catch_trials", sa.Integer(), nullable=True),
        sa.Column("target_presented", sa.Integer(), nullable=True),
        sa.Column(
            "target_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("intensity", sa.Float(), nullable=True),
        sa.Column(
            "choice_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("responded", sa.Integer(), nullable=True),
        sa.Column(
            "response_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("response_latency_ms", sa.Integer(), nullable=True),
        sa.Column("yes", sa.Integer(), nullable=True),
        sa.Column("no", sa.Integer(), nullable=True),
        sa.Column("caught_out_reset", sa.Integer(), nullable=True),
        sa.Column(
            "trial_num_in_calculation_sequence", sa.Integer(), nullable=True
        ),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdetthreshold_trials__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_cardinal_expdetthreshold_trials__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_cardinal_expdetthreshold_trials__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f(
                "fk_cardinal_expdetthreshold_trials__manually_erasing_user_id"
            ),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f(
                "fk_cardinal_expdetthreshold_trials__preserving_user_id"
            ),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cardinal_expdetthreshold_trials__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint(
            "_pk", name=op.f("pk_cardinal_expdetthreshold_trials")
        ),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table(
        "cardinal_expdetthreshold_trials", schema=None
    ) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdetthreshold_trials__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdetthreshold_trials__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdetthreshold_trials__era"),
            ["_era"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdetthreshold_trials__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdetthreshold_trials__pk"),
            ["_pk"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cardinal_expdetthreshold_trials_id"),
            ["id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f(
                "ix_cardinal_expdetthreshold_trials_when_last_modified"
            ),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cbir",
        sa.Column("confirm_blanks", sa.Integer(), nullable=True),
        sa.Column("comments", sa.UnicodeText(), nullable=True),
        sa.Column("frequency1", sa.Integer(), nullable=True),
        sa.Column("frequency2", sa.Integer(), nullable=True),
        sa.Column("frequency3", sa.Integer(), nullable=True),
        sa.Column("frequency4", sa.Integer(), nullable=True),
        sa.Column("frequency5", sa.Integer(), nullable=True),
        sa.Column("frequency6", sa.Integer(), nullable=True),
        sa.Column("frequency7", sa.Integer(), nullable=True),
        sa.Column("frequency8", sa.Integer(), nullable=True),
        sa.Column("frequency9", sa.Integer(), nullable=True),
        sa.Column("frequency10", sa.Integer(), nullable=True),
        sa.Column("frequency11", sa.Integer(), nullable=True),
        sa.Column("frequency12", sa.Integer(), nullable=True),
        sa.Column("frequency13", sa.Integer(), nullable=True),
        sa.Column("frequency14", sa.Integer(), nullable=True),
        sa.Column("frequency15", sa.Integer(), nullable=True),
        sa.Column("frequency16", sa.Integer(), nullable=True),
        sa.Column("frequency17", sa.Integer(), nullable=True),
        sa.Column("frequency18", sa.Integer(), nullable=True),
        sa.Column("frequency19", sa.Integer(), nullable=True),
        sa.Column("frequency20", sa.Integer(), nullable=True),
        sa.Column("frequency21", sa.Integer(), nullable=True),
        sa.Column("frequency22", sa.Integer(), nullable=True),
        sa.Column("frequency23", sa.Integer(), nullable=True),
        sa.Column("frequency24", sa.Integer(), nullable=True),
        sa.Column("frequency25", sa.Integer(), nullable=True),
        sa.Column("frequency26", sa.Integer(), nullable=True),
        sa.Column("frequency27", sa.Integer(), nullable=True),
        sa.Column("frequency28", sa.Integer(), nullable=True),
        sa.Column("frequency29", sa.Integer(), nullable=True),
        sa.Column("frequency30", sa.Integer(), nullable=True),
        sa.Column("frequency31", sa.Integer(), nullable=True),
        sa.Column("frequency32", sa.Integer(), nullable=True),
        sa.Column("frequency33", sa.Integer(), nullable=True),
        sa.Column("frequency34", sa.Integer(), nullable=True),
        sa.Column("frequency35", sa.Integer(), nullable=True),
        sa.Column("frequency36", sa.Integer(), nullable=True),
        sa.Column("frequency37", sa.Integer(), nullable=True),
        sa.Column("frequency38", sa.Integer(), nullable=True),
        sa.Column("frequency39", sa.Integer(), nullable=True),
        sa.Column("frequency40", sa.Integer(), nullable=True),
        sa.Column("frequency41", sa.Integer(), nullable=True),
        sa.Column("frequency42", sa.Integer(), nullable=True),
        sa.Column("frequency43", sa.Integer(), nullable=True),
        sa.Column("frequency44", sa.Integer(), nullable=True),
        sa.Column("frequency45", sa.Integer(), nullable=True),
        sa.Column("distress1", sa.Integer(), nullable=True),
        sa.Column("distress2", sa.Integer(), nullable=True),
        sa.Column("distress3", sa.Integer(), nullable=True),
        sa.Column("distress4", sa.Integer(), nullable=True),
        sa.Column("distress5", sa.Integer(), nullable=True),
        sa.Column("distress6", sa.Integer(), nullable=True),
        sa.Column("distress7", sa.Integer(), nullable=True),
        sa.Column("distress8", sa.Integer(), nullable=True),
        sa.Column("distress9", sa.Integer(), nullable=True),
        sa.Column("distress10", sa.Integer(), nullable=True),
        sa.Column("distress11", sa.Integer(), nullable=True),
        sa.Column("distress12", sa.Integer(), nullable=True),
        sa.Column("distress13", sa.Integer(), nullable=True),
        sa.Column("distress14", sa.Integer(), nullable=True),
        sa.Column("distress15", sa.Integer(), nullable=True),
        sa.Column("distress16", sa.Integer(), nullable=True),
        sa.Column("distress17", sa.Integer(), nullable=True),
        sa.Column("distress18", sa.Integer(), nullable=True),
        sa.Column("distress19", sa.Integer(), nullable=True),
        sa.Column("distress20", sa.Integer(), nullable=True),
        sa.Column("distress21", sa.Integer(), nullable=True),
        sa.Column("distress22", sa.Integer(), nullable=True),
        sa.Column("distress23", sa.Integer(), nullable=True),
        sa.Column("distress24", sa.Integer(), nullable=True),
        sa.Column("distress25", sa.Integer(), nullable=True),
        sa.Column("distress26", sa.Integer(), nullable=True),
        sa.Column("distress27", sa.Integer(), nullable=True),
        sa.Column("distress28", sa.Integer(), nullable=True),
        sa.Column("distress29", sa.Integer(), nullable=True),
        sa.Column("distress30", sa.Integer(), nullable=True),
        sa.Column("distress31", sa.Integer(), nullable=True),
        sa.Column("distress32", sa.Integer(), nullable=True),
        sa.Column("distress33", sa.Integer(), nullable=True),
        sa.Column("distress34", sa.Integer(), nullable=True),
        sa.Column("distress35", sa.Integer(), nullable=True),
        sa.Column("distress36", sa.Integer(), nullable=True),
        sa.Column("distress37", sa.Integer(), nullable=True),
        sa.Column("distress38", sa.Integer(), nullable=True),
        sa.Column("distress39", sa.Integer(), nullable=True),
        sa.Column("distress40", sa.Integer(), nullable=True),
        sa.Column("distress41", sa.Integer(), nullable=True),
        sa.Column("distress42", sa.Integer(), nullable=True),
        sa.Column("distress43", sa.Integer(), nullable=True),
        sa.Column("distress44", sa.Integer(), nullable=True),
        sa.Column("distress45", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("respondent_relationship", sa.Text(), nullable=True),
        sa.Column("respondent_name", sa.Text(), nullable=True),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cbir__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_cbir__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_cbir__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cbir__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cbir__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cbir__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_cbir")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("cbir", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cbir__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cbir__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cbir__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cbir__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_cbir__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_cbir_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_cbir_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cbir_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cecaq3",
        *[
            sa.Column(
                "s1a_motherfigure_birthmother", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s1a_motherfigure_stepmother", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s1a_motherfigure_femalerelative", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s1a_motherfigure_femalerelative_detail",
                sa.UnicodeText(),
                nullable=True,
            ),
            sa.Column(
                "s1a_motherfigure_familyfriend", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s1a_motherfigure_fostermother", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s1a_motherfigure_adoptivemother", sa.Boolean(), nullable=True
            ),
            sa.Column("s1a_motherfigure_other", sa.Boolean(), nullable=True),
            sa.Column(
                "s1a_motherfigure_other_detail",
                sa.UnicodeText(),
                nullable=True,
            ),
            sa.Column(
                "s1a_fatherfigure_birthfather", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s1a_fatherfigure_stepfather", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s1a_fatherfigure_malerelative", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s1a_fatherfigure_malerelative_detail",
                sa.UnicodeText(),
                nullable=True,
            ),
            sa.Column(
                "s1a_fatherfigure_familyfriend", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s1a_fatherfigure_fosterfather", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s1a_fatherfigure_adoptivefather", sa.Boolean(), nullable=True
            ),
            sa.Column("s1a_fatherfigure_other", sa.Boolean(), nullable=True),
            sa.Column(
                "s1a_fatherfigure_other_detail",
                sa.UnicodeText(),
                nullable=True,
            ),
            sa.Column("s1b_institution", sa.Boolean(), nullable=True),
            sa.Column("s1b_institution_time_years", sa.Float(), nullable=True),
            sa.Column("s1c_mother_died", sa.Boolean(), nullable=True),
            sa.Column("s1c_father_died", sa.Boolean(), nullable=True),
            sa.Column(
                "s1c_mother_died_subject_aged", sa.Float(), nullable=True
            ),
            sa.Column(
                "s1c_father_died_subject_aged", sa.Float(), nullable=True
            ),
            sa.Column(
                "s1c_separated_from_mother", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s1c_separated_from_father", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s1c_first_separated_from_mother_aged",
                sa.Float(),
                nullable=True,
            ),
            sa.Column(
                "s1c_first_separated_from_father_aged",
                sa.Float(),
                nullable=True,
            ),
            sa.Column(
                "s1c_mother_how_long_first_separation_years",
                sa.Float(),
                nullable=True,
            ),
            sa.Column(
                "s1c_father_how_long_first_separation_years",
                sa.Float(),
                nullable=True,
            ),
            sa.Column(
                "s1c_mother_separation_reason", sa.Integer(), nullable=True
            ),
            sa.Column(
                "s1c_father_separation_reason", sa.Integer(), nullable=True
            ),
            sa.Column(
                "s1c_describe_experience", sa.UnicodeText(), nullable=True
            ),
            sa.Column("s2a_which_mother_figure", sa.Integer(), nullable=True),
            sa.Column(
                "s2a_which_mother_figure_other_detail",
                sa.UnicodeText(),
                nullable=True,
            ),
            sa.Column("s2a_q1", sa.Integer(), nullable=True),
            sa.Column("s2a_q2", sa.Integer(), nullable=True),
            sa.Column("s2a_q3", sa.Integer(), nullable=True),
            sa.Column("s2a_q4", sa.Integer(), nullable=True),
            sa.Column("s2a_q5", sa.Integer(), nullable=True),
            sa.Column("s2a_q6", sa.Integer(), nullable=True),
            sa.Column("s2a_q7", sa.Integer(), nullable=True),
            sa.Column("s2a_q8", sa.Integer(), nullable=True),
            sa.Column("s2a_q9", sa.Integer(), nullable=True),
            sa.Column("s2a_q10", sa.Integer(), nullable=True),
            sa.Column("s2a_q11", sa.Integer(), nullable=True),
            sa.Column("s2a_q12", sa.Integer(), nullable=True),
            sa.Column("s2a_q13", sa.Integer(), nullable=True),
            sa.Column("s2a_q14", sa.Integer(), nullable=True),
            sa.Column("s2a_q15", sa.Integer(), nullable=True),
            sa.Column("s2a_q16", sa.Integer(), nullable=True),
            sa.Column("s2a_extra", sa.UnicodeText(), nullable=True),
            sa.Column("s2b_q1", sa.Integer(), nullable=True),
            sa.Column("s2b_q2", sa.Integer(), nullable=True),
            sa.Column("s2b_q3", sa.Integer(), nullable=True),
            sa.Column("s2b_q4", sa.Integer(), nullable=True),
            sa.Column("s2b_q5", sa.Integer(), nullable=True),
            sa.Column("s2b_q6", sa.Integer(), nullable=True),
            sa.Column("s2b_q7", sa.Integer(), nullable=True),
            sa.Column("s2b_q8", sa.Integer(), nullable=True),
            sa.Column("s2b_q9", sa.Integer(), nullable=True),
            sa.Column("s2b_q10", sa.Integer(), nullable=True),
            sa.Column("s2b_q11", sa.Integer(), nullable=True),
            sa.Column("s2b_q12", sa.Integer(), nullable=True),
            sa.Column("s2b_q13", sa.Integer(), nullable=True),
            sa.Column("s2b_q14", sa.Integer(), nullable=True),
            sa.Column("s2b_q15", sa.Integer(), nullable=True),
            sa.Column("s2b_q16", sa.Integer(), nullable=True),
            sa.Column("s2b_q17", sa.Integer(), nullable=True),
            sa.Column("s2b_q1_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q2_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q3_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q4_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q5_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q6_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q7_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q8_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q9_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q10_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q11_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q12_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q13_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q14_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q15_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q16_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_q17_frequency", sa.Integer(), nullable=True),
            sa.Column("s2b_age_began", sa.Float(), nullable=True),
            sa.Column("s2b_extra", sa.UnicodeText(), nullable=True),
            sa.Column("s3a_which_father_figure", sa.Integer(), nullable=True),
            sa.Column(
                "s3a_which_father_figure_other_detail",
                sa.UnicodeText(),
                nullable=True,
            ),
            sa.Column("s3a_q1", sa.Integer(), nullable=True),
            sa.Column("s3a_q2", sa.Integer(), nullable=True),
            sa.Column("s3a_q3", sa.Integer(), nullable=True),
            sa.Column("s3a_q4", sa.Integer(), nullable=True),
            sa.Column("s3a_q5", sa.Integer(), nullable=True),
            sa.Column("s3a_q6", sa.Integer(), nullable=True),
            sa.Column("s3a_q7", sa.Integer(), nullable=True),
            sa.Column("s3a_q8", sa.Integer(), nullable=True),
            sa.Column("s3a_q9", sa.Integer(), nullable=True),
            sa.Column("s3a_q10", sa.Integer(), nullable=True),
            sa.Column("s3a_q11", sa.Integer(), nullable=True),
            sa.Column("s3a_q12", sa.Integer(), nullable=True),
            sa.Column("s3a_q13", sa.Integer(), nullable=True),
            sa.Column("s3a_q14", sa.Integer(), nullable=True),
            sa.Column("s3a_q15", sa.Integer(), nullable=True),
            sa.Column("s3a_q16", sa.Integer(), nullable=True),
            sa.Column("s3a_extra", sa.UnicodeText(), nullable=True),
            sa.Column("s3b_q1", sa.Integer(), nullable=True),
            sa.Column("s3b_q2", sa.Integer(), nullable=True),
            sa.Column("s3b_q3", sa.Integer(), nullable=True),
            sa.Column("s3b_q4", sa.Integer(), nullable=True),
            sa.Column("s3b_q5", sa.Integer(), nullable=True),
            sa.Column("s3b_q6", sa.Integer(), nullable=True),
            sa.Column("s3b_q7", sa.Integer(), nullable=True),
            sa.Column("s3b_q8", sa.Integer(), nullable=True),
            sa.Column("s3b_q9", sa.Integer(), nullable=True),
            sa.Column("s3b_q10", sa.Integer(), nullable=True),
            sa.Column("s3b_q11", sa.Integer(), nullable=True),
            sa.Column("s3b_q12", sa.Integer(), nullable=True),
            sa.Column("s3b_q13", sa.Integer(), nullable=True),
            sa.Column("s3b_q14", sa.Integer(), nullable=True),
            sa.Column("s3b_q15", sa.Integer(), nullable=True),
            sa.Column("s3b_q16", sa.Integer(), nullable=True),
            sa.Column("s3b_q17", sa.Integer(), nullable=True),
            sa.Column("s3b_q1_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q2_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q3_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q4_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q5_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q6_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q7_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q8_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q9_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q10_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q11_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q12_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q13_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q14_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q15_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q16_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_q17_frequency", sa.Integer(), nullable=True),
            sa.Column("s3b_age_began", sa.Float(), nullable=True),
            sa.Column("s3b_extra", sa.UnicodeText(), nullable=True),
            sa.Column("s3c_q1", sa.Integer(), nullable=True),
            sa.Column("s3c_q2", sa.Integer(), nullable=True),
            sa.Column("s3c_q3", sa.Integer(), nullable=True),
            sa.Column("s3c_q4", sa.Integer(), nullable=True),
            sa.Column("s3c_q5", sa.Integer(), nullable=True),
            sa.Column("s3c_q6", sa.Integer(), nullable=True),
            sa.Column("s3c_q7", sa.Integer(), nullable=True),
            sa.Column("s3c_q8", sa.Integer(), nullable=True),
            sa.Column("s3c_q9", sa.Integer(), nullable=True),
            sa.Column("s3c_q10", sa.Integer(), nullable=True),
            sa.Column("s3c_q11", sa.Integer(), nullable=True),
            sa.Column("s3c_q12", sa.Integer(), nullable=True),
            sa.Column("s3c_q13", sa.Integer(), nullable=True),
            sa.Column("s3c_q14", sa.Integer(), nullable=True),
            sa.Column("s3c_q15", sa.Integer(), nullable=True),
            sa.Column("s3c_q16", sa.Integer(), nullable=True),
            sa.Column("s3c_q17", sa.Integer(), nullable=True),
            sa.Column(
                "s3c_which_parent_cared_for", sa.Integer(), nullable=True
            ),
            sa.Column(
                "s3c_parent_mental_problem", sa.Integer(), nullable=True
            ),
            sa.Column(
                "s3c_parent_physical_problem", sa.Integer(), nullable=True
            ),
            sa.Column("s4a_adultconfidant", sa.Boolean(), nullable=True),
            sa.Column(
                "s4a_adultconfidant_mother", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s4a_adultconfidant_father", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s4a_adultconfidant_otherrelative", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s4a_adultconfidant_familyfriend", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s4a_adultconfidant_responsibleadult",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column("s4a_adultconfidant_other", sa.Boolean(), nullable=True),
            sa.Column(
                "s4a_adultconfidant_other_detail",
                sa.UnicodeText(),
                nullable=True,
            ),
            sa.Column(
                "s4a_adultconfidant_additional",
                sa.UnicodeText(),
                nullable=True,
            ),
            sa.Column("s4b_childconfidant", sa.Boolean(), nullable=True),
            sa.Column(
                "s4b_childconfidant_sister", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s4b_childconfidant_brother", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s4b_childconfidant_otherrelative", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s4b_childconfidant_closefriend", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s4b_childconfidant_otherfriend", sa.Boolean(), nullable=True
            ),
            sa.Column("s4b_childconfidant_other", sa.Boolean(), nullable=True),
            sa.Column(
                "s4b_childconfidant_other_detail",
                sa.UnicodeText(),
                nullable=True,
            ),
            sa.Column(
                "s4b_childconfidant_additional",
                sa.UnicodeText(),
                nullable=True,
            ),
            sa.Column("s4c_closest_mother", sa.Boolean(), nullable=True),
            sa.Column("s4c_closest_father", sa.Boolean(), nullable=True),
            sa.Column("s4c_closest_sibling", sa.Boolean(), nullable=True),
            sa.Column(
                "s4c_closest_otherrelative", sa.Boolean(), nullable=True
            ),
            sa.Column("s4c_closest_adultfriend", sa.Boolean(), nullable=True),
            sa.Column("s4c_closest_childfriend", sa.Boolean(), nullable=True),
            sa.Column("s4c_closest_other", sa.Boolean(), nullable=True),
            sa.Column(
                "s4c_closest_other_detail", sa.UnicodeText(), nullable=True
            ),
            sa.Column(
                "s4c_closest_additional", sa.UnicodeText(), nullable=True
            ),
            sa.Column("s5c_physicalabuse", sa.Boolean(), nullable=True),
            sa.Column("s5c_abused_by_mother", sa.Boolean(), nullable=True),
            sa.Column("s5c_abused_by_father", sa.Boolean(), nullable=True),
            sa.Column("s5c_mother_abuse_age_began", sa.Float(), nullable=True),
            sa.Column("s5c_father_abuse_age_began", sa.Float(), nullable=True),
            sa.Column(
                "s5c_mother_hit_more_than_once", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s5c_father_hit_more_than_once", sa.Boolean(), nullable=True
            ),
            sa.Column("s5c_mother_hit_how", sa.Integer(), nullable=True),
            sa.Column("s5c_father_hit_how", sa.Integer(), nullable=True),
            sa.Column("s5c_mother_injured", sa.Boolean(), nullable=True),
            sa.Column("s5c_father_injured", sa.Boolean(), nullable=True),
            sa.Column(
                "s5c_mother_out_of_control", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s5c_father_out_of_control", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s5c_parental_abuse_description",
                sa.UnicodeText(),
                nullable=True,
            ),
            sa.Column("s5c_abuse_by_nonparent", sa.Boolean(), nullable=True),
            sa.Column(
                "s5c_nonparent_abuse_description",
                sa.UnicodeText(),
                nullable=True,
            ),
            sa.Column(
                "s6_any_unwanted_sexual_experience",
                sa.Integer(),
                nullable=True,
            ),
            sa.Column("s6_unwanted_intercourse", sa.Integer(), nullable=True),
            sa.Column(
                "s6_upsetting_sexual_adult_authority",
                sa.Integer(),
                nullable=True,
            ),
            sa.Column("s6_first_age", sa.Float(), nullable=True),
            sa.Column("s6_other_age", sa.Float(), nullable=True),
            sa.Column("s6_first_person_known", sa.Boolean(), nullable=True),
            sa.Column("s6_other_person_known", sa.Boolean(), nullable=True),
            sa.Column("s6_first_relative", sa.Boolean(), nullable=True),
            sa.Column("s6_other_relative", sa.Boolean(), nullable=True),
            sa.Column("s6_first_in_household", sa.Boolean(), nullable=True),
            sa.Column("s6_other_in_household", sa.Boolean(), nullable=True),
            sa.Column("s6_first_more_than_once", sa.Boolean(), nullable=True),
            sa.Column("s6_other_more_than_once", sa.Boolean(), nullable=True),
            sa.Column(
                "s6_first_touch_privates_subject", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s6_other_touch_privates_subject", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s6_first_touch_privates_other", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "s6_other_touch_privates_other", sa.Boolean(), nullable=True
            ),
            sa.Column("s6_first_intercourse", sa.Boolean(), nullable=True),
            sa.Column("s6_other_intercourse", sa.Boolean(), nullable=True),
            sa.Column(
                "s6_unwanted_sexual_description",
                sa.UnicodeText(),
                nullable=True,
            ),
            sa.Column("any_other_comments", sa.UnicodeText(), nullable=True),
            sa.Column("patient_id", sa.Integer(), nullable=False),
            sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
            sa.Column(
                "when_created",
                PendulumDateTimeAsIsoTextColType(length=32),
                nullable=False,
            ),
            sa.Column(
                "when_firstexit",
                PendulumDateTimeAsIsoTextColType(length=32),
                nullable=True,
            ),
            sa.Column("editing_time_s", sa.Float(), nullable=True),
            sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
            sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
            sa.Column("_manually_erased", sa.Boolean(), nullable=True),
            sa.Column("_removal_pending", sa.Boolean(), nullable=True),
            sa.Column("_group_id", sa.Integer(), nullable=False),
            sa.Column(
                "_manually_erasing_user_id", sa.Integer(), nullable=True
            ),
            sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
            sa.Column("_adding_user_id", sa.Integer(), nullable=True),
            sa.Column(
                "when_last_modified",
                PendulumDateTimeAsIsoTextColType(length=32),
                nullable=True,
            ),
            sa.Column("_removing_user_id", sa.Integer(), nullable=True),
            sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
            sa.Column("_device_id", sa.Integer(), nullable=False),
            sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
            sa.Column("_pk", sa.Integer(), nullable=False),
            sa.Column("_current", sa.Boolean(), nullable=False),
            sa.Column("_addition_pending", sa.Boolean(), nullable=False),
            sa.Column(
                "_manually_erased_at",
                PendulumDateTimeAsIsoTextColType(length=32),
                nullable=True,
            ),
            sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
            sa.Column("_era", sa.String(length=32), nullable=False),
            sa.Column(
                "_when_added_exact",
                PendulumDateTimeAsIsoTextColType(length=32),
                nullable=True,
            ),
            sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
            sa.Column("id", sa.Integer(), nullable=False),
            sa.Column(
                "_camcops_version",
                SemanticVersionColType(length=147),
                nullable=True,
            ),
            sa.Column(
                "_when_removed_exact",
                PendulumDateTimeAsIsoTextColType(length=32),
                nullable=True,
            ),
            sa.Column("_successor_pk", sa.Integer(), nullable=True),
            sa.ForeignKeyConstraint(
                ["_adding_user_id"],
                ["_security_users.id"],
                name=op.f("fk_cecaq3__adding_user_id"),
            ),
            sa.ForeignKeyConstraint(
                ["_device_id"],
                ["_security_devices.id"],
                name=op.f("fk_cecaq3__device_id"),
            ),
            sa.ForeignKeyConstraint(
                ["_group_id"],
                ["_security_groups.id"],
                name=op.f("fk_cecaq3__group_id"),
            ),
            sa.ForeignKeyConstraint(
                ["_manually_erasing_user_id"],
                ["_security_users.id"],
                name=op.f("fk_cecaq3__manually_erasing_user_id"),
            ),
            sa.ForeignKeyConstraint(
                ["_preserving_user_id"],
                ["_security_users.id"],
                name=op.f("fk_cecaq3__preserving_user_id"),
            ),
            sa.ForeignKeyConstraint(
                ["_removing_user_id"],
                ["_security_users.id"],
                name=op.f("fk_cecaq3__removing_user_id"),
            ),
            sa.PrimaryKeyConstraint("_pk", name=op.f("pk_cecaq3")),
        ],
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC"
    )
    with op.batch_alter_table("cecaq3", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cecaq3__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cecaq3__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cecaq3__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cecaq3__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cecaq3__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_cecaq3_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_cecaq3_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cecaq3_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cgi",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3t", sa.Integer(), nullable=True),
        sa.Column("q3s", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cgi__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_cgi__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_cgi__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cgi__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cgi__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cgi__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_cgi")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("cgi", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cgi__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cgi__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cgi__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cgi__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_cgi__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_cgi_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_cgi_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cgi_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cgi_i",
        sa.Column("q", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cgi_i__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_cgi_i__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_cgi_i__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cgi_i__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cgi_i__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cgi_i__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_cgi_i")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("cgi_i", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cgi_i__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cgi_i__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cgi_i__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cgi_i__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cgi_i__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_cgi_i_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_cgi_i_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cgi_i_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cgisch",
        sa.Column("severity1", sa.Integer(), nullable=True),
        sa.Column("severity2", sa.Integer(), nullable=True),
        sa.Column("severity3", sa.Integer(), nullable=True),
        sa.Column("severity4", sa.Integer(), nullable=True),
        sa.Column("severity5", sa.Integer(), nullable=True),
        sa.Column("change1", sa.Integer(), nullable=True),
        sa.Column("change2", sa.Integer(), nullable=True),
        sa.Column("change3", sa.Integer(), nullable=True),
        sa.Column("change4", sa.Integer(), nullable=True),
        sa.Column("change5", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cgisch__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_cgisch__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_cgisch__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cgisch__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cgisch__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cgisch__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_cgisch")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("cgisch", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cgisch__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cgisch__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cgisch__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cgisch__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cgisch__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_cgisch_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_cgisch_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cgisch_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "ciwa",
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("t", sa.Float(), nullable=True),
        sa.Column("hr", sa.Integer(), nullable=True),
        sa.Column("sbp", sa.Integer(), nullable=True),
        sa.Column("dbp", sa.Integer(), nullable=True),
        sa.Column("rr", sa.Integer(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ciwa__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_ciwa__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_ciwa__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ciwa__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ciwa__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ciwa__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_ciwa")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("ciwa", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_ciwa__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ciwa__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ciwa__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ciwa__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_ciwa__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_ciwa_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_ciwa_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ciwa_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "contactlog",
        sa.Column("location", sa.UnicodeText(), nullable=True),
        sa.Column(
            "start", PendulumDateTimeAsIsoTextColType(length=32), nullable=True
        ),
        sa.Column(
            "end", PendulumDateTimeAsIsoTextColType(length=32), nullable=True
        ),
        sa.Column("patient_contact", sa.Integer(), nullable=True),
        sa.Column("staff_liaison", sa.Integer(), nullable=True),
        sa.Column("other_liaison", sa.Integer(), nullable=True),
        sa.Column("comment", sa.UnicodeText(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_contactlog__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_contactlog__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_contactlog__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_contactlog__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_contactlog__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_contactlog__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_contactlog")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("contactlog", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_contactlog__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_contactlog__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_contactlog__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_contactlog__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_contactlog__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_contactlog_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_contactlog_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_contactlog_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cope_brief",
        sa.Column("completed_by_patient", sa.Integer(), nullable=True),
        sa.Column("completed_by", sa.UnicodeText(), nullable=True),
        sa.Column("relationship_to_patient", sa.Integer(), nullable=True),
        sa.Column(
            "relationship_to_patient_other", sa.UnicodeText(), nullable=True
        ),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("q18", sa.Integer(), nullable=True),
        sa.Column("q19", sa.Integer(), nullable=True),
        sa.Column("q20", sa.Integer(), nullable=True),
        sa.Column("q21", sa.Integer(), nullable=True),
        sa.Column("q22", sa.Integer(), nullable=True),
        sa.Column("q23", sa.Integer(), nullable=True),
        sa.Column("q24", sa.Integer(), nullable=True),
        sa.Column("q25", sa.Integer(), nullable=True),
        sa.Column("q26", sa.Integer(), nullable=True),
        sa.Column("q27", sa.Integer(), nullable=True),
        sa.Column("q28", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cope_brief__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_cope_brief__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_cope_brief__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cope_brief__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cope_brief__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cope_brief__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_cope_brief")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("cope_brief", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cope_brief__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cope_brief__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cope_brief__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cope_brief__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cope_brief__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cope_brief_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cope_brief_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cope_brief_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cpft_lps_discharge",
        sa.Column("discharge_date", sa.Date(), nullable=True),
        sa.Column("discharge_reason_code", sa.UnicodeText(), nullable=True),
        sa.Column(
            "leaflet_or_discharge_card_given",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_lodcg")),
            nullable=True,
        ),
        sa.Column("frequent_attender", sa.Boolean(), nullable=True),
        sa.Column(
            "patient_wanted_copy_of_letter", sa.Boolean(), nullable=True
        ),
        sa.Column("gaf_at_first_assessment", sa.Integer(), nullable=True),
        sa.Column("gaf_at_discharge", sa.Integer(), nullable=True),
        sa.Column(
            "referral_reason_self_harm_overdose",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_rrshoverdose")),
            nullable=True,
        ),
        sa.Column(
            "referral_reason_self_harm_other",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_rrshother")),
            nullable=True,
        ),
        sa.Column(
            "referral_reason_suicidal_ideas",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_rrsuicidal")),
            nullable=True,
        ),
        sa.Column(
            "referral_reason_behavioural_disturbance",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_behavdisturb")),
            nullable=True,
        ),
        sa.Column("referral_reason_low_mood", sa.Boolean(), nullable=True),
        sa.Column(
            "referral_reason_elevated_mood", sa.Boolean(), nullable=True
        ),
        sa.Column("referral_reason_psychosis", sa.Boolean(), nullable=True),
        sa.Column(
            "referral_reason_pre_transplant",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_pretransplant")),
            nullable=True,
        ),
        sa.Column(
            "referral_reason_post_transplant",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_posttransplant")),
            nullable=True,
        ),
        sa.Column("referral_reason_delirium", sa.Boolean(), nullable=True),
        sa.Column("referral_reason_anxiety", sa.Boolean(), nullable=True),
        sa.Column(
            "referral_reason_somatoform_mus",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_mus")),
            nullable=True,
        ),
        sa.Column(
            "referral_reason_motivation_adherence",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_motivadherence")),
            nullable=True,
        ),
        sa.Column("referral_reason_capacity", sa.Boolean(), nullable=True),
        sa.Column(
            "referral_reason_eating_disorder",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_eatingdis")),
            nullable=True,
        ),
        sa.Column("referral_reason_safeguarding", sa.Boolean(), nullable=True),
        sa.Column(
            "referral_reason_discharge_placement",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_dcplacement")),
            nullable=True,
        ),
        sa.Column(
            "referral_reason_cognitive_problem",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_cognitiveprob")),
            nullable=True,
        ),
        sa.Column(
            "referral_reason_substance_alcohol",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_alcohol")),
            nullable=True,
        ),
        sa.Column(
            "referral_reason_substance_other",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_substanceother")),
            nullable=True,
        ),
        sa.Column("referral_reason_other", sa.Boolean(), nullable=True),
        sa.Column(
            "referral_reason_transplant_organ", sa.UnicodeText(), nullable=True
        ),
        sa.Column(
            "referral_reason_other_detail", sa.UnicodeText(), nullable=True
        ),
        sa.Column(
            "diagnosis_no_active_mental_health_problem",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_nomhprob")),
            nullable=True,
        ),
        sa.Column(
            "diagnosis_psych_1_icd10code", sa.String(length=7), nullable=True
        ),
        sa.Column(
            "diagnosis_psych_1_description", sa.UnicodeText(), nullable=True
        ),
        sa.Column(
            "diagnosis_psych_2_icd10code", sa.String(length=7), nullable=True
        ),
        sa.Column(
            "diagnosis_psych_2_description", sa.UnicodeText(), nullable=True
        ),
        sa.Column(
            "diagnosis_psych_3_icd10code", sa.String(length=7), nullable=True
        ),
        sa.Column(
            "diagnosis_psych_3_description", sa.UnicodeText(), nullable=True
        ),
        sa.Column(
            "diagnosis_psych_4_icd10code", sa.String(length=7), nullable=True
        ),
        sa.Column(
            "diagnosis_psych_4_description", sa.UnicodeText(), nullable=True
        ),
        sa.Column("diagnosis_medical_1", sa.UnicodeText(), nullable=True),
        sa.Column("diagnosis_medical_2", sa.UnicodeText(), nullable=True),
        sa.Column("diagnosis_medical_3", sa.UnicodeText(), nullable=True),
        sa.Column("diagnosis_medical_4", sa.UnicodeText(), nullable=True),
        sa.Column(
            "management_assessment_diagnostic",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_mx_ass_diag")),
            nullable=True,
        ),
        sa.Column("management_medication", sa.Boolean(), nullable=True),
        sa.Column(
            "management_specialling_behavioural_disturbance",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_msbd")),
            nullable=True,
        ),
        sa.Column(
            "management_supportive_patient", sa.Boolean(), nullable=True
        ),
        sa.Column("management_supportive_carers", sa.Boolean(), nullable=True),
        sa.Column("management_supportive_staff", sa.Boolean(), nullable=True),
        sa.Column(
            "management_nursing_management", sa.Boolean(), nullable=True
        ),
        sa.Column("management_therapy_cbt", sa.Boolean(), nullable=True),
        sa.Column("management_therapy_cat", sa.Boolean(), nullable=True),
        sa.Column("management_therapy_other", sa.Boolean(), nullable=True),
        sa.Column(
            "management_treatment_adherence",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_mx_rx_adhere")),
            nullable=True,
        ),
        sa.Column("management_capacity", sa.Boolean(), nullable=True),
        sa.Column("management_education_patient", sa.Boolean(), nullable=True),
        sa.Column("management_education_carers", sa.Boolean(), nullable=True),
        sa.Column("management_education_staff", sa.Boolean(), nullable=True),
        sa.Column(
            "management_accommodation_placement",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_accom")),
            nullable=True,
        ),
        sa.Column(
            "management_signposting_external_referral",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_mx_signpostrefer")),
            nullable=True,
        ),
        sa.Column("management_mha_s136", sa.Boolean(), nullable=True),
        sa.Column("management_mha_s5_2", sa.Boolean(), nullable=True),
        sa.Column("management_mha_s2", sa.Boolean(), nullable=True),
        sa.Column("management_mha_s3", sa.Boolean(), nullable=True),
        sa.Column(
            "management_complex_case_conference",
            sa.Boolean(name=op.f("ck_cpft_lps_discharge_caseconf")),
            nullable=True,
        ),
        sa.Column("management_other", sa.Boolean(), nullable=True),
        sa.Column("management_other_detail", sa.UnicodeText(), nullable=True),
        sa.Column("outcome", sa.UnicodeText(), nullable=True),
        sa.Column(
            "outcome_hospital_transfer_detail", sa.UnicodeText(), nullable=True
        ),
        sa.Column("outcome_other_detail", sa.UnicodeText(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cpft_lps_discharge__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_cpft_lps_discharge__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_cpft_lps_discharge__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cpft_lps_discharge__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cpft_lps_discharge__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cpft_lps_discharge__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_cpft_lps_discharge")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("cpft_lps_discharge", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_discharge__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_discharge__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_discharge__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_discharge__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_discharge__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_discharge_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_discharge_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_discharge_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cpft_lps_referral",
        sa.Column(
            "referral_date_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("lps_division", sa.UnicodeText(), nullable=True),
        sa.Column("referral_priority", sa.UnicodeText(), nullable=True),
        sa.Column("referral_method", sa.UnicodeText(), nullable=True),
        sa.Column("referrer_name", sa.UnicodeText(), nullable=True),
        sa.Column("referrer_contact_details", sa.UnicodeText(), nullable=True),
        sa.Column("referring_consultant", sa.UnicodeText(), nullable=True),
        sa.Column("referring_specialty", sa.UnicodeText(), nullable=True),
        sa.Column(
            "referring_specialty_other", sa.UnicodeText(), nullable=True
        ),
        sa.Column("patient_location", sa.UnicodeText(), nullable=True),
        sa.Column("admission_date", sa.Date(), nullable=True),
        sa.Column("estimated_discharge_date", sa.Date(), nullable=True),
        sa.Column("patient_aware_of_referral", sa.Boolean(), nullable=True),
        sa.Column("interpreter_required", sa.Boolean(), nullable=True),
        sa.Column("sensory_impairment", sa.Boolean(), nullable=True),
        sa.Column("marital_status_code", sa.String(length=1), nullable=True),
        sa.Column("ethnic_category_code", sa.String(length=1), nullable=True),
        sa.Column("admission_reason_overdose", sa.Boolean(), nullable=True),
        sa.Column(
            "admission_reason_self_harm_not_overdose",
            sa.Boolean(name=op.f("ck_cpft_lps_referral_arshno")),
            nullable=True,
        ),
        sa.Column("admission_reason_confusion", sa.Boolean(), nullable=True),
        sa.Column("admission_reason_trauma", sa.Boolean(), nullable=True),
        sa.Column("admission_reason_falls", sa.Boolean(), nullable=True),
        sa.Column("admission_reason_infection", sa.Boolean(), nullable=True),
        sa.Column(
            "admission_reason_poor_adherence",
            sa.Boolean(name=op.f("ck_cpft_lps_referral_adpa")),
            nullable=True,
        ),
        sa.Column("admission_reason_other", sa.Boolean(), nullable=True),
        sa.Column(
            "existing_psychiatric_teams", sa.UnicodeText(), nullable=True
        ),
        sa.Column("care_coordinator", sa.UnicodeText(), nullable=True),
        sa.Column("other_contact_details", sa.UnicodeText(), nullable=True),
        sa.Column("referral_reason", sa.UnicodeText(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cpft_lps_referral__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_cpft_lps_referral__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_cpft_lps_referral__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cpft_lps_referral__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cpft_lps_referral__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cpft_lps_referral__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_cpft_lps_referral")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("cpft_lps_referral", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_referral__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_referral__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_referral__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_referral__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_referral__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_referral_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_referral_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_referral_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "cpft_lps_resetresponseclock",
        sa.Column(
            "reset_start_time_to",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("reason", sa.UnicodeText(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cpft_lps_resetresponseclock__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_cpft_lps_resetresponseclock__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_cpft_lps_resetresponseclock__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f(
                "fk_cpft_lps_resetresponseclock__manually_erasing_user_id"
            ),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cpft_lps_resetresponseclock__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_cpft_lps_resetresponseclock__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint(
            "_pk", name=op.f("pk_cpft_lps_resetresponseclock")
        ),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table(
        "cpft_lps_resetresponseclock", schema=None
    ) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_resetresponseclock__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_resetresponseclock__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_resetresponseclock__era"),
            ["_era"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_resetresponseclock__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_resetresponseclock__pk"),
            ["_pk"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_resetresponseclock_id"),
            ["id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_resetresponseclock_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_cpft_lps_resetresponseclock_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "dad",
        sa.Column("hygiene_init_wash", sa.Integer(), nullable=True),
        sa.Column("hygiene_init_teeth", sa.Integer(), nullable=True),
        sa.Column("hygiene_init_hair", sa.Integer(), nullable=True),
        sa.Column("hygiene_plan_wash", sa.Integer(), nullable=True),
        sa.Column("hygiene_exec_wash", sa.Integer(), nullable=True),
        sa.Column("hygiene_exec_hair", sa.Integer(), nullable=True),
        sa.Column("hygiene_exec_teeth", sa.Integer(), nullable=True),
        sa.Column("dressing_init_dress", sa.Integer(), nullable=True),
        sa.Column("dressing_plan_clothing", sa.Integer(), nullable=True),
        sa.Column("dressing_plan_order", sa.Integer(), nullable=True),
        sa.Column("dressing_exec_dress", sa.Integer(), nullable=True),
        sa.Column("dressing_exec_undress", sa.Integer(), nullable=True),
        sa.Column("continence_init_toilet", sa.Integer(), nullable=True),
        sa.Column("continence_exec_toilet", sa.Integer(), nullable=True),
        sa.Column("eating_init_eat", sa.Integer(), nullable=True),
        sa.Column("eating_plan_utensils", sa.Integer(), nullable=True),
        sa.Column("eating_exec_eat", sa.Integer(), nullable=True),
        sa.Column("mealprep_init_meal", sa.Integer(), nullable=True),
        sa.Column("mealprep_plan_meal", sa.Integer(), nullable=True),
        sa.Column("mealprep_exec_meal", sa.Integer(), nullable=True),
        sa.Column("telephone_init_phone", sa.Integer(), nullable=True),
        sa.Column("telephone_plan_dial", sa.Integer(), nullable=True),
        sa.Column("telephone_exec_conversation", sa.Integer(), nullable=True),
        sa.Column("telephone_exec_message", sa.Integer(), nullable=True),
        sa.Column("outing_init_outing", sa.Integer(), nullable=True),
        sa.Column("outing_plan_outing", sa.Integer(), nullable=True),
        sa.Column(
            "outing_exec_reach_destination", sa.Integer(), nullable=True
        ),
        sa.Column(
            "outing_exec_mode_transportation", sa.Integer(), nullable=True
        ),
        sa.Column(
            "outing_exec_return_with_shopping", sa.Integer(), nullable=True
        ),
        sa.Column("finance_init_interest", sa.Integer(), nullable=True),
        sa.Column("finance_plan_pay_bills", sa.Integer(), nullable=True),
        sa.Column(
            "finance_plan_organise_correspondence", sa.Integer(), nullable=True
        ),
        sa.Column("finance_exec_handle_money", sa.Integer(), nullable=True),
        sa.Column("medications_init_medication", sa.Integer(), nullable=True),
        sa.Column(
            "medications_exec_take_medications", sa.Integer(), nullable=True
        ),
        sa.Column(
            "leisure_init_interest_leisure", sa.Integer(), nullable=True
        ),
        sa.Column("leisure_init_interest_chores", sa.Integer(), nullable=True),
        sa.Column("leisure_plan_chores", sa.Integer(), nullable=True),
        sa.Column("leisure_exec_complete_chores", sa.Integer(), nullable=True),
        sa.Column("leisure_exec_safe_at_home", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("respondent_relationship", sa.Text(), nullable=True),
        sa.Column("respondent_name", sa.Text(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_dad__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_dad__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_dad__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_dad__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_dad__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_dad__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_dad")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("dad", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_dad__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_dad__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_dad__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_dad__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_dad__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_dad_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_dad_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_dad_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "dast",
        sa.Column("q1", sa.String(length=1), nullable=True),
        sa.Column("q2", sa.String(length=1), nullable=True),
        sa.Column("q3", sa.String(length=1), nullable=True),
        sa.Column("q4", sa.String(length=1), nullable=True),
        sa.Column("q5", sa.String(length=1), nullable=True),
        sa.Column("q6", sa.String(length=1), nullable=True),
        sa.Column("q7", sa.String(length=1), nullable=True),
        sa.Column("q8", sa.String(length=1), nullable=True),
        sa.Column("q9", sa.String(length=1), nullable=True),
        sa.Column("q10", sa.String(length=1), nullable=True),
        sa.Column("q11", sa.String(length=1), nullable=True),
        sa.Column("q12", sa.String(length=1), nullable=True),
        sa.Column("q13", sa.String(length=1), nullable=True),
        sa.Column("q14", sa.String(length=1), nullable=True),
        sa.Column("q15", sa.String(length=1), nullable=True),
        sa.Column("q16", sa.String(length=1), nullable=True),
        sa.Column("q17", sa.String(length=1), nullable=True),
        sa.Column("q18", sa.String(length=1), nullable=True),
        sa.Column("q19", sa.String(length=1), nullable=True),
        sa.Column("q20", sa.String(length=1), nullable=True),
        sa.Column("q21", sa.String(length=1), nullable=True),
        sa.Column("q22", sa.String(length=1), nullable=True),
        sa.Column("q23", sa.String(length=1), nullable=True),
        sa.Column("q24", sa.String(length=1), nullable=True),
        sa.Column("q25", sa.String(length=1), nullable=True),
        sa.Column("q26", sa.String(length=1), nullable=True),
        sa.Column("q27", sa.String(length=1), nullable=True),
        sa.Column("q28", sa.String(length=1), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_dast__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_dast__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_dast__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_dast__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_dast__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_dast__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_dast")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("dast", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_dast__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_dast__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_dast__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_dast__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_dast__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_dast_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_dast_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_dast_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "deakin_1_healthreview",
        sa.Column("ethnicity", sa.Integer(), nullable=True),
        sa.Column("ethnicity_text", sa.UnicodeText(), nullable=True),
        sa.Column("ethnicity_other_details", sa.UnicodeText(), nullable=True),
        sa.Column("handedness", sa.String(length=1), nullable=True),
        sa.Column("education", sa.Text(), nullable=True),
        sa.Column("allergies", sa.Boolean(), nullable=True),
        sa.Column("allergy_asthma", sa.Boolean(), nullable=True),
        sa.Column("allergy_pollen_dust", sa.Boolean(), nullable=True),
        sa.Column("allergy_dermatitis", sa.Boolean(), nullable=True),
        sa.Column("allergy_food", sa.Boolean(), nullable=True),
        sa.Column("allergy_dander", sa.Boolean(), nullable=True),
        sa.Column("allergy_other", sa.Boolean(), nullable=True),
        sa.Column("allergy_details", sa.Text(), nullable=True),
        sa.Column("vaccinations_last3months", sa.Boolean(), nullable=True),
        sa.Column("vaccination_details", sa.Text(), nullable=True),
        sa.Column("infections_last3months", sa.Boolean(), nullable=True),
        sa.Column("infection_recent_respiratory", sa.Boolean(), nullable=True),
        sa.Column(
            "infection_recent_gastroenteritis",
            sa.Boolean(
                name=op.f("ck_deakin_1_healthreview_inf_recent_gastro")
            ),
            nullable=True,
        ),
        sa.Column("infection_recent_urinary", sa.Boolean(), nullable=True),
        sa.Column("infection_recent_sexual", sa.Boolean(), nullable=True),
        sa.Column("infection_recent_hepatitis", sa.Boolean(), nullable=True),
        sa.Column("infection_recent_other", sa.Boolean(), nullable=True),
        sa.Column("infection_recent_details", sa.Text(), nullable=True),
        sa.Column("infections_chronic", sa.Boolean(), nullable=True),
        sa.Column(
            "infection_chronic_respiratory", sa.Boolean(), nullable=True
        ),
        sa.Column(
            "infection_chronic_gastroenteritis",
            sa.Boolean(
                name=op.f("ck_deakin_1_healthreview_inf_chronic_gastro")
            ),
            nullable=True,
        ),
        sa.Column("infection_chronic_urinary", sa.Boolean(), nullable=True),
        sa.Column("infection_chronic_sexual", sa.Boolean(), nullable=True),
        sa.Column("infection_chronic_hepatitis", sa.Boolean(), nullable=True),
        sa.Column("infection_chronic_other", sa.Boolean(), nullable=True),
        sa.Column("infection_chronic_details", sa.Text(), nullable=True),
        sa.Column("immune_disorders", sa.Boolean(), nullable=True),
        sa.Column("immunity_ms", sa.Boolean(), nullable=True),
        sa.Column("immunity_sle", sa.Boolean(), nullable=True),
        sa.Column("immunity_arthritis", sa.Boolean(), nullable=True),
        sa.Column("immunity_hiv", sa.Boolean(), nullable=True),
        sa.Column("immunity_graves", sa.Boolean(), nullable=True),
        sa.Column("immunity_diabetes", sa.Boolean(), nullable=True),
        sa.Column("immunity_other", sa.Boolean(), nullable=True),
        sa.Column("immunity_details", sa.Text(), nullable=True),
        sa.Column("family_history", sa.Boolean(), nullable=True),
        sa.Column("familyhistory_ms", sa.Boolean(), nullable=True),
        sa.Column("familyhistory_sle", sa.Boolean(), nullable=True),
        sa.Column("familyhistory_arthritis", sa.Boolean(), nullable=True),
        sa.Column("familyhistory_graves", sa.Boolean(), nullable=True),
        sa.Column("familyhistory_diabetes", sa.Boolean(), nullable=True),
        sa.Column("familyhistory_psychosis_sz", sa.Boolean(), nullable=True),
        sa.Column("familyhistory_bipolar", sa.Boolean(), nullable=True),
        sa.Column("familyhistory_details", sa.Text(), nullable=True),
        sa.Column("health_anything_else", sa.Boolean(), nullable=True),
        sa.Column(
            "health_anything_else_details", sa.UnicodeText(), nullable=True
        ),
        sa.Column("drug_history", sa.UnicodeText(), nullable=True),
        sa.Column(
            "first_antipsychotic_medication", sa.UnicodeText(), nullable=True
        ),
        sa.Column(
            "recreational_drug_in_last_3_months",
            sa.Boolean(name=op.f("ck_deakin_1_healthreview_recdruglast3mo")),
            nullable=True,
        ),
        sa.Column("recdrug_tobacco_frequency", sa.Integer(), nullable=True),
        sa.Column("recdrug_tobacco_cigsperweek", sa.Integer(), nullable=True),
        sa.Column("recdrug_tobacco_prevheavy", sa.Boolean(), nullable=True),
        sa.Column("recdrug_cannabis_frequency", sa.Integer(), nullable=True),
        sa.Column(
            "recdrug_cannabis_jointsperweek", sa.Integer(), nullable=True
        ),
        sa.Column("recdrug_cannabis_prevheavy", sa.Boolean(), nullable=True),
        sa.Column("recdrug_alcohol_frequency", sa.Integer(), nullable=True),
        sa.Column("recdrug_alcohol_unitsperweek", sa.Integer(), nullable=True),
        sa.Column("recdrug_alcohol_prevheavy", sa.Boolean(), nullable=True),
        sa.Column("recdrug_mdma_frequency", sa.Integer(), nullable=True),
        sa.Column("recdrug_mdma_prevheavy", sa.Boolean(), nullable=True),
        sa.Column("recdrug_cocaine_frequency", sa.Integer(), nullable=True),
        sa.Column("recdrug_cocaine_prevheavy", sa.Boolean(), nullable=True),
        sa.Column("recdrug_crack_frequency", sa.Integer(), nullable=True),
        sa.Column("recdrug_crack_prevheavy", sa.Boolean(), nullable=True),
        sa.Column("recdrug_heroin_frequency", sa.Integer(), nullable=True),
        sa.Column("recdrug_heroin_prevheavy", sa.Boolean(), nullable=True),
        sa.Column("recdrug_methadone_frequency", sa.Integer(), nullable=True),
        sa.Column("recdrug_methadone_prevheavy", sa.Boolean(), nullable=True),
        sa.Column(
            "recdrug_amphetamines_frequency", sa.Integer(), nullable=True
        ),
        sa.Column(
            "recdrug_amphetamines_prevheavy",
            sa.Boolean(name=op.f("ck_deakin_1_healthreview_amphetprevheavy")),
            nullable=True,
        ),
        sa.Column(
            "recdrug_benzodiazepines_frequency", sa.Integer(), nullable=True
        ),
        sa.Column(
            "recdrug_benzodiazepines_prevheavy",
            sa.Boolean(name=op.f("ck_deakin_1_healthreview_benzoprevheavy")),
            nullable=True,
        ),
        sa.Column("recdrug_ketamine_frequency", sa.Integer(), nullable=True),
        sa.Column("recdrug_ketamine_prevheavy", sa.Boolean(), nullable=True),
        sa.Column("recdrug_legalhighs_frequency", sa.Integer(), nullable=True),
        sa.Column("recdrug_legalhighs_prevheavy", sa.Boolean(), nullable=True),
        sa.Column("recdrug_inhalants_frequency", sa.Integer(), nullable=True),
        sa.Column("recdrug_inhalants_prevheavy", sa.Boolean(), nullable=True),
        sa.Column(
            "recdrug_hallucinogens_frequency", sa.Integer(), nullable=True
        ),
        sa.Column(
            "recdrug_hallucinogens_prevheavy",
            sa.Boolean(
                name=op.f("ck_deakin_1_healthreview_hallucinogenprevheavy")
            ),
            nullable=True,
        ),
        sa.Column("recdrug_details", sa.UnicodeText(), nullable=True),
        sa.Column("recdrug_prevheavy", sa.Boolean(), nullable=True),
        sa.Column(
            "recdrug_prevheavy_details", sa.UnicodeText(), nullable=True
        ),
        sa.Column("mri_claustrophobic", sa.Boolean(), nullable=True),
        sa.Column("mri_difficulty_lying_1_hour", sa.Boolean(), nullable=True),
        sa.Column("mri_nonremovable_metal", sa.Boolean(), nullable=True),
        sa.Column("mri_metal_from_operations", sa.Boolean(), nullable=True),
        sa.Column("mri_tattoos_nicotine_patches", sa.Boolean(), nullable=True),
        sa.Column("mri_worked_with_metal", sa.Boolean(), nullable=True),
        sa.Column("mri_previous_brain_scan", sa.Boolean(), nullable=True),
        sa.Column(
            "mri_previous_brain_scan_details", sa.UnicodeText(), nullable=True
        ),
        sa.Column("other_relevant_things", sa.Boolean(), nullable=True),
        sa.Column(
            "other_relevant_things_details", sa.UnicodeText(), nullable=True
        ),
        sa.Column(
            "willing_to_participate_in_further_studies",
            sa.Boolean(name=op.f("ck_deakin_1_healthreview_wtpifs")),
            nullable=True,
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_deakin_1_healthreview__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_deakin_1_healthreview__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_deakin_1_healthreview__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_deakin_1_healthreview__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_deakin_1_healthreview__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_deakin_1_healthreview__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_deakin_1_healthreview")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table(
        "deakin_1_healthreview", schema=None
    ) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_deakin_1_healthreview__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_deakin_1_healthreview__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_deakin_1_healthreview__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_deakin_1_healthreview__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_deakin_1_healthreview__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_deakin_1_healthreview_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_deakin_1_healthreview_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_deakin_1_healthreview_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "demoquestionnaire",
        sa.Column("mcqtext_1a", sa.UnicodeText(), nullable=True),
        sa.Column("mcqtext_1b", sa.UnicodeText(), nullable=True),
        sa.Column("mcqtext_2a", sa.UnicodeText(), nullable=True),
        sa.Column("mcqtext_2b", sa.UnicodeText(), nullable=True),
        sa.Column("mcqtext_3a", sa.UnicodeText(), nullable=True),
        sa.Column("mcqtext_3b", sa.UnicodeText(), nullable=True),
        sa.Column("typedvar_text", sa.UnicodeText(), nullable=True),
        sa.Column("typedvar_text_multiline", sa.UnicodeText(), nullable=True),
        sa.Column("typedvar_text_rich", sa.UnicodeText(), nullable=True),
        sa.Column("typedvar_int", sa.Integer(), nullable=True),
        sa.Column("typedvar_real", sa.Float(), nullable=True),
        sa.Column("date_only", sa.Date(), nullable=True),
        sa.Column(
            "date_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("thermometer", sa.Integer(), nullable=True),
        sa.Column("diagnosticcode_code", sa.String(length=7), nullable=True),
        sa.Column(
            "diagnosticcode_description", sa.UnicodeText(), nullable=True
        ),
        sa.Column("diagnosticcode2_code", sa.String(length=7), nullable=True),
        sa.Column(
            "diagnosticcode2_description", sa.UnicodeText(), nullable=True
        ),
        sa.Column("photo_blobid", sa.Integer(), nullable=True),
        sa.Column("photo_rotation", sa.Integer(), nullable=True),
        sa.Column("canvas_blobid", sa.Integer(), nullable=True),
        sa.Column("canvas2_blobid", sa.Integer(), nullable=True),
        sa.Column("spinbox_int", sa.Integer(), nullable=True),
        sa.Column("spinbox_real", sa.Float(), nullable=True),
        sa.Column("time_only", sa.Time(), nullable=True),
        sa.Column("mcq1", sa.Integer(), nullable=True),
        sa.Column("mcq2", sa.Integer(), nullable=True),
        sa.Column("mcq3", sa.Integer(), nullable=True),
        sa.Column("mcq4", sa.Integer(), nullable=True),
        sa.Column("mcq5", sa.Integer(), nullable=True),
        sa.Column("mcq6", sa.Integer(), nullable=True),
        sa.Column("mcq7", sa.Integer(), nullable=True),
        sa.Column("mcq8", sa.Integer(), nullable=True),
        sa.Column("mcq9", sa.Integer(), nullable=True),
        sa.Column("mcq10", sa.Integer(), nullable=True),
        sa.Column("mcqbool1", sa.Integer(), nullable=True),
        sa.Column("mcqbool2", sa.Integer(), nullable=True),
        sa.Column("mcqbool3", sa.Integer(), nullable=True),
        sa.Column("multipleresponse1", sa.Integer(), nullable=True),
        sa.Column("multipleresponse2", sa.Integer(), nullable=True),
        sa.Column("multipleresponse3", sa.Integer(), nullable=True),
        sa.Column("multipleresponse4", sa.Integer(), nullable=True),
        sa.Column("multipleresponse5", sa.Integer(), nullable=True),
        sa.Column("multipleresponse6", sa.Integer(), nullable=True),
        sa.Column("booltext1", sa.Integer(), nullable=True),
        sa.Column("booltext2", sa.Integer(), nullable=True),
        sa.Column("booltext3", sa.Integer(), nullable=True),
        sa.Column("booltext4", sa.Integer(), nullable=True),
        sa.Column("booltext5", sa.Integer(), nullable=True),
        sa.Column("booltext6", sa.Integer(), nullable=True),
        sa.Column("booltext7", sa.Integer(), nullable=True),
        sa.Column("booltext8", sa.Integer(), nullable=True),
        sa.Column("booltext9", sa.Integer(), nullable=True),
        sa.Column("booltext10", sa.Integer(), nullable=True),
        sa.Column("booltext11", sa.Integer(), nullable=True),
        sa.Column("booltext12", sa.Integer(), nullable=True),
        sa.Column("booltext13", sa.Integer(), nullable=True),
        sa.Column("booltext14", sa.Integer(), nullable=True),
        sa.Column("booltext15", sa.Integer(), nullable=True),
        sa.Column("booltext16", sa.Integer(), nullable=True),
        sa.Column("booltext17", sa.Integer(), nullable=True),
        sa.Column("booltext18", sa.Integer(), nullable=True),
        sa.Column("booltext19", sa.Integer(), nullable=True),
        sa.Column("booltext20", sa.Integer(), nullable=True),
        sa.Column("booltext21", sa.Integer(), nullable=True),
        sa.Column("booltext22", sa.Integer(), nullable=True),
        sa.Column("boolimage1", sa.Integer(), nullable=True),
        sa.Column("boolimage2", sa.Integer(), nullable=True),
        sa.Column("picker1", sa.Integer(), nullable=True),
        sa.Column("picker2", sa.Integer(), nullable=True),
        sa.Column("slider1", sa.Float(), nullable=True),
        sa.Column("slider2", sa.Float(), nullable=True),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_demoquestionnaire__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_demoquestionnaire__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_demoquestionnaire__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_demoquestionnaire__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_demoquestionnaire__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_demoquestionnaire__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_demoquestionnaire")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("demoquestionnaire", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_demoquestionnaire__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_demoquestionnaire__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_demoquestionnaire__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_demoquestionnaire__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_demoquestionnaire__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_demoquestionnaire_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_demoquestionnaire_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "demqol",
        sa.Column("q29", sa.Integer(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("q18", sa.Integer(), nullable=True),
        sa.Column("q19", sa.Integer(), nullable=True),
        sa.Column("q20", sa.Integer(), nullable=True),
        sa.Column("q21", sa.Integer(), nullable=True),
        sa.Column("q22", sa.Integer(), nullable=True),
        sa.Column("q23", sa.Integer(), nullable=True),
        sa.Column("q24", sa.Integer(), nullable=True),
        sa.Column("q25", sa.Integer(), nullable=True),
        sa.Column("q26", sa.Integer(), nullable=True),
        sa.Column("q27", sa.Integer(), nullable=True),
        sa.Column("q28", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_demqol__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_demqol__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_demqol__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_demqol__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_demqol__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_demqol__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_demqol")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("demqol", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_demqol__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_demqol__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_demqol__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_demqol__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_demqol__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_demqol_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_demqol_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_demqol_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "demqolproxy",
        sa.Column("q32", sa.Integer(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("q18", sa.Integer(), nullable=True),
        sa.Column("q19", sa.Integer(), nullable=True),
        sa.Column("q20", sa.Integer(), nullable=True),
        sa.Column("q21", sa.Integer(), nullable=True),
        sa.Column("q22", sa.Integer(), nullable=True),
        sa.Column("q23", sa.Integer(), nullable=True),
        sa.Column("q24", sa.Integer(), nullable=True),
        sa.Column("q25", sa.Integer(), nullable=True),
        sa.Column("q26", sa.Integer(), nullable=True),
        sa.Column("q27", sa.Integer(), nullable=True),
        sa.Column("q28", sa.Integer(), nullable=True),
        sa.Column("q29", sa.Integer(), nullable=True),
        sa.Column("q30", sa.Integer(), nullable=True),
        sa.Column("q31", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("respondent_relationship", sa.Text(), nullable=True),
        sa.Column("respondent_name", sa.Text(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_demqolproxy__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_demqolproxy__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_demqolproxy__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_demqolproxy__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_demqolproxy__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_demqolproxy__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_demqolproxy")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("demqolproxy", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_demqolproxy__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_demqolproxy__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_demqolproxy__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_demqolproxy__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_demqolproxy__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_demqolproxy_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_demqolproxy_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_demqolproxy_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "diagnosis_icd10",
        sa.Column("relates_to_date", sa.Date(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd10__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_diagnosis_icd10__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_diagnosis_icd10__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd10__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd10__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd10__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_diagnosis_icd10")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("diagnosis_icd10", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd10__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd10__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd10__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd10__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd10__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd10_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd10_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd10_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "diagnosis_icd10_item",
        sa.Column("diagnosis_icd10_id", sa.Integer(), nullable=False),
        sa.Column("comment", sa.UnicodeText(), nullable=True),
        sa.Column("description", sa.UnicodeText(), nullable=True),
        sa.Column("code", sa.String(length=7), nullable=True),
        sa.Column("seqnum", sa.Integer(), nullable=False),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd10_item__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_diagnosis_icd10_item__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_diagnosis_icd10_item__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd10_item__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd10_item__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd10_item__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_diagnosis_icd10_item")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("diagnosis_icd10_item", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd10_item__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd10_item__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd10_item__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd10_item__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd10_item__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd10_item_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd10_item_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "diagnosis_icd9cm",
        sa.Column("relates_to_date", sa.Date(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd9cm__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_diagnosis_icd9cm__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_diagnosis_icd9cm__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd9cm__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd9cm__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd9cm__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_diagnosis_icd9cm")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("diagnosis_icd9cm", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd9cm__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd9cm__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd9cm__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd9cm__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd9cm__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd9cm_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd9cm_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd9cm_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "diagnosis_icd9cm_item",
        sa.Column("diagnosis_icd9cm_id", sa.Integer(), nullable=False),
        sa.Column("comment", sa.UnicodeText(), nullable=True),
        sa.Column("description", sa.UnicodeText(), nullable=True),
        sa.Column("code", sa.String(length=7), nullable=True),
        sa.Column("seqnum", sa.Integer(), nullable=False),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd9cm_item__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_diagnosis_icd9cm_item__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_diagnosis_icd9cm_item__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd9cm_item__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd9cm_item__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_diagnosis_icd9cm_item__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_diagnosis_icd9cm_item")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table(
        "diagnosis_icd9cm_item", schema=None
    ) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd9cm_item__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd9cm_item__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd9cm_item__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd9cm_item__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd9cm_item__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd9cm_item_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_diagnosis_icd9cm_item_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "distressthermometer",
        sa.Column("distress", sa.Integer(), nullable=True),
        sa.Column("other", sa.UnicodeText(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("q18", sa.Integer(), nullable=True),
        sa.Column("q19", sa.Integer(), nullable=True),
        sa.Column("q20", sa.Integer(), nullable=True),
        sa.Column("q21", sa.Integer(), nullable=True),
        sa.Column("q22", sa.Integer(), nullable=True),
        sa.Column("q23", sa.Integer(), nullable=True),
        sa.Column("q24", sa.Integer(), nullable=True),
        sa.Column("q25", sa.Integer(), nullable=True),
        sa.Column("q26", sa.Integer(), nullable=True),
        sa.Column("q27", sa.Integer(), nullable=True),
        sa.Column("q28", sa.Integer(), nullable=True),
        sa.Column("q29", sa.Integer(), nullable=True),
        sa.Column("q30", sa.Integer(), nullable=True),
        sa.Column("q31", sa.Integer(), nullable=True),
        sa.Column("q32", sa.Integer(), nullable=True),
        sa.Column("q33", sa.Integer(), nullable=True),
        sa.Column("q34", sa.Integer(), nullable=True),
        sa.Column("q35", sa.Integer(), nullable=True),
        sa.Column("q36", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_distressthermometer__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_distressthermometer__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_distressthermometer__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_distressthermometer__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_distressthermometer__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_distressthermometer__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_distressthermometer")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("distressthermometer", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_distressthermometer__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_distressthermometer__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_distressthermometer__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_distressthermometer__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_distressthermometer__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_distressthermometer_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_distressthermometer_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_distressthermometer_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "fast",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_fast__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_fast__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_fast__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_fast__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_fast__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_fast__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_fast")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("fast", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_fast__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_fast__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_fast__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_fast__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_fast__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_fast_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_fast_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_fast_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "fft",
        sa.Column("service", sa.UnicodeText(), nullable=True),
        sa.Column("rating", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_fft__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_fft__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_fft__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_fft__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_fft__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_fft__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_fft")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("fft", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_fft__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_fft__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_fft__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_fft__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_fft__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_fft_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_fft_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_fft_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "frs",
        sa.Column("comments", sa.UnicodeText(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("q18", sa.Integer(), nullable=True),
        sa.Column("q19", sa.Integer(), nullable=True),
        sa.Column("q20", sa.Integer(), nullable=True),
        sa.Column("q21", sa.Integer(), nullable=True),
        sa.Column("q22", sa.Integer(), nullable=True),
        sa.Column("q23", sa.Integer(), nullable=True),
        sa.Column("q24", sa.Integer(), nullable=True),
        sa.Column("q25", sa.Integer(), nullable=True),
        sa.Column("q26", sa.Integer(), nullable=True),
        sa.Column("q27", sa.Integer(), nullable=True),
        sa.Column("q28", sa.Integer(), nullable=True),
        sa.Column("q29", sa.Integer(), nullable=True),
        sa.Column("q30", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("respondent_relationship", sa.Text(), nullable=True),
        sa.Column("respondent_name", sa.Text(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_frs__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_frs__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_frs__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_frs__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_frs__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_frs__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_frs")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("frs", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_frs__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_frs__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_frs__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_frs__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_frs__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_frs_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_frs_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_frs_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "gad7",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gad7__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_gad7__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_gad7__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gad7__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gad7__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gad7__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_gad7")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("gad7", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_gad7__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gad7__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gad7__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gad7__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_gad7__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_gad7_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_gad7_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gad7_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "gaf",
        sa.Column("score", sa.Integer(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gaf__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_gaf__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_gaf__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gaf__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gaf__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gaf__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_gaf")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("gaf", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_gaf__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gaf__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gaf__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gaf__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_gaf__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_gaf_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_gaf_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gaf_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "gds15",
        sa.Column("q1", sa.String(length=1), nullable=True),
        sa.Column("q2", sa.String(length=1), nullable=True),
        sa.Column("q3", sa.String(length=1), nullable=True),
        sa.Column("q4", sa.String(length=1), nullable=True),
        sa.Column("q5", sa.String(length=1), nullable=True),
        sa.Column("q6", sa.String(length=1), nullable=True),
        sa.Column("q7", sa.String(length=1), nullable=True),
        sa.Column("q8", sa.String(length=1), nullable=True),
        sa.Column("q9", sa.String(length=1), nullable=True),
        sa.Column("q10", sa.String(length=1), nullable=True),
        sa.Column("q11", sa.String(length=1), nullable=True),
        sa.Column("q12", sa.String(length=1), nullable=True),
        sa.Column("q13", sa.String(length=1), nullable=True),
        sa.Column("q14", sa.String(length=1), nullable=True),
        sa.Column("q15", sa.String(length=1), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gds15__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_gds15__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_gds15__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gds15__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gds15__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gds15__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_gds15")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("gds15", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_gds15__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gds15__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gds15__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gds15__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gds15__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_gds15_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_gds15_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gds15_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "gmcpq",
        sa.Column("doctor", sa.UnicodeText(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2a", sa.Integer(), nullable=True),
        sa.Column("q2b", sa.Integer(), nullable=True),
        sa.Column("q2c", sa.Integer(), nullable=True),
        sa.Column("q2d", sa.Integer(), nullable=True),
        sa.Column("q2e", sa.Integer(), nullable=True),
        sa.Column("q2f", sa.Integer(), nullable=True),
        sa.Column("q2f_details", sa.UnicodeText(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4a", sa.Integer(), nullable=True),
        sa.Column("q4b", sa.Integer(), nullable=True),
        sa.Column("q4c", sa.Integer(), nullable=True),
        sa.Column("q4d", sa.Integer(), nullable=True),
        sa.Column("q4e", sa.Integer(), nullable=True),
        sa.Column("q4f", sa.Integer(), nullable=True),
        sa.Column("q4g", sa.Integer(), nullable=True),
        sa.Column("q5a", sa.Integer(), nullable=True),
        sa.Column("q5b", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.UnicodeText(), nullable=True),
        sa.Column("q10", sa.String(length=1), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q12_details", sa.UnicodeText(), nullable=True),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gmcpq__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_gmcpq__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_gmcpq__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gmcpq__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gmcpq__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_gmcpq__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_gmcpq")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("gmcpq", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_gmcpq__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gmcpq__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gmcpq__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gmcpq__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_gmcpq__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_gmcpq_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_gmcpq_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "hads",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hads__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_hads__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_hads__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hads__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hads__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hads__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_hads")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("hads", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_hads__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hads__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hads__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hads__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_hads__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_hads_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_hads_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hads_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "hads_respondent",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("respondent_relationship", sa.Text(), nullable=True),
        sa.Column("respondent_name", sa.Text(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hads_respondent__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_hads_respondent__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_hads_respondent__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hads_respondent__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hads_respondent__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hads_respondent__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_hads_respondent")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("hads_respondent", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_hads_respondent__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_hads_respondent__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_hads_respondent__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hads_respondent__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_hads_respondent__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hads_respondent_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hads_respondent_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_hads_respondent_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "hama",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hama__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_hama__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_hama__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hama__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hama__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hama__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_hama")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("hama", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_hama__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hama__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hama__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hama__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_hama__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_hama_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_hama_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hama_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "hamd",
        sa.Column("whichq16", sa.Integer(), nullable=True),
        sa.Column("q16a", sa.Integer(), nullable=True),
        sa.Column("q16b", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("q18a", sa.Integer(), nullable=True),
        sa.Column("q18b", sa.Integer(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q19", sa.Integer(), nullable=True),
        sa.Column("q20", sa.Integer(), nullable=True),
        sa.Column("q21", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hamd__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_hamd__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_hamd__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hamd__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hamd__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hamd__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_hamd")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("hamd", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_hamd__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hamd__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hamd__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hamd__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_hamd__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_hamd_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_hamd_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hamd_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "hamd7",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hamd7__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_hamd7__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_hamd7__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hamd7__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hamd7__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_hamd7__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_hamd7")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("hamd7", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_hamd7__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hamd7__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hamd7__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hamd7__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hamd7__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_hamd7_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_hamd7_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_hamd7_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "honos",
        sa.Column("period_rated", sa.UnicodeText(), nullable=True),
        sa.Column("q8problemtype", sa.String(length=1), nullable=True),
        sa.Column("q8otherproblem", sa.UnicodeText(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_honos__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_honos__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_honos__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_honos__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_honos__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_honos__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_honos")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("honos", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_honos__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honos__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honos__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honos__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honos__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_honos_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_honos_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honos_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "honos65",
        sa.Column("period_rated", sa.UnicodeText(), nullable=True),
        sa.Column("q8problemtype", sa.String(length=1), nullable=True),
        sa.Column("q8otherproblem", sa.UnicodeText(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_honos65__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_honos65__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_honos65__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_honos65__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_honos65__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_honos65__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_honos65")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("honos65", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_honos65__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honos65__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honos65__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honos65__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honos65__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honos65_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honos65_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honos65_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "honosca",
        sa.Column("period_rated", sa.UnicodeText(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_honosca__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_honosca__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_honosca__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_honosca__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_honosca__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_honosca__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_honosca")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("honosca", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_honosca__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honosca__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honosca__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honosca__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honosca__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honosca_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honosca_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_honosca_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "icd10depressive",
        sa.Column("mood", sa.Boolean(), nullable=True),
        sa.Column("anhedonia", sa.Boolean(), nullable=True),
        sa.Column("energy", sa.Boolean(), nullable=True),
        sa.Column("sleep", sa.Boolean(), nullable=True),
        sa.Column("worth", sa.Boolean(), nullable=True),
        sa.Column("appetite", sa.Boolean(), nullable=True),
        sa.Column("guilt", sa.Boolean(), nullable=True),
        sa.Column("concentration", sa.Boolean(), nullable=True),
        sa.Column("activity", sa.Boolean(), nullable=True),
        sa.Column("death", sa.Boolean(), nullable=True),
        sa.Column("somatic_anhedonia", sa.Boolean(), nullable=True),
        sa.Column(
            "somatic_emotional_unreactivity", sa.Boolean(), nullable=True
        ),
        sa.Column("somatic_early_morning_waking", sa.Boolean(), nullable=True),
        sa.Column("somatic_mood_worse_morning", sa.Boolean(), nullable=True),
        sa.Column("somatic_psychomotor", sa.Boolean(), nullable=True),
        sa.Column("somatic_appetite", sa.Boolean(), nullable=True),
        sa.Column("somatic_weight", sa.Boolean(), nullable=True),
        sa.Column("somatic_libido", sa.Boolean(), nullable=True),
        sa.Column("hallucinations_schizophrenic", sa.Boolean(), nullable=True),
        sa.Column("hallucinations_other", sa.Boolean(), nullable=True),
        sa.Column("delusions_schizophrenic", sa.Boolean(), nullable=True),
        sa.Column("delusions_other", sa.Boolean(), nullable=True),
        sa.Column("stupor", sa.Boolean(), nullable=True),
        sa.Column("date_pertains_to", sa.Date(), nullable=True),
        sa.Column("comments", sa.UnicodeText(), nullable=True),
        sa.Column("duration_at_least_2_weeks", sa.Boolean(), nullable=True),
        sa.Column("severe_clinically", sa.Boolean(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10depressive__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_icd10depressive__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_icd10depressive__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10depressive__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10depressive__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10depressive__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_icd10depressive")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("icd10depressive", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_icd10depressive__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10depressive__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10depressive__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10depressive__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10depressive__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10depressive_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10depressive_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10depressive_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "icd10manic",
        sa.Column("mood_elevated", sa.Boolean(), nullable=True),
        sa.Column("mood_irritable", sa.Boolean(), nullable=True),
        sa.Column("distractible", sa.Boolean(), nullable=True),
        sa.Column("activity", sa.Boolean(), nullable=True),
        sa.Column("sleep", sa.Boolean(), nullable=True),
        sa.Column("talkativeness", sa.Boolean(), nullable=True),
        sa.Column("recklessness", sa.Boolean(), nullable=True),
        sa.Column("social_disinhibition", sa.Boolean(), nullable=True),
        sa.Column("sexual", sa.Boolean(), nullable=True),
        sa.Column("grandiosity", sa.Boolean(), nullable=True),
        sa.Column("flight_of_ideas", sa.Boolean(), nullable=True),
        sa.Column("sustained4days", sa.Boolean(), nullable=True),
        sa.Column("sustained7days", sa.Boolean(), nullable=True),
        sa.Column("admission_required", sa.Boolean(), nullable=True),
        sa.Column(
            "some_interference_functioning", sa.Boolean(), nullable=True
        ),
        sa.Column(
            "severe_interference_functioning", sa.Boolean(), nullable=True
        ),
        sa.Column("perceptual_alterations", sa.Boolean(), nullable=True),
        sa.Column("hallucinations_schizophrenic", sa.Boolean(), nullable=True),
        sa.Column("hallucinations_other", sa.Boolean(), nullable=True),
        sa.Column("delusions_schizophrenic", sa.Boolean(), nullable=True),
        sa.Column("delusions_other", sa.Boolean(), nullable=True),
        sa.Column("date_pertains_to", sa.Date(), nullable=True),
        sa.Column("comments", sa.UnicodeText(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10manic__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_icd10manic__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_icd10manic__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10manic__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10manic__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10manic__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_icd10manic")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("icd10manic", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_icd10manic__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10manic__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10manic__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10manic__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10manic__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10manic_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10manic_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10manic_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "icd10mixed",
        sa.Column("date_pertains_to", sa.Date(), nullable=True),
        sa.Column("comments", sa.UnicodeText(), nullable=True),
        sa.Column("mixture_or_rapid_alternation", sa.Boolean(), nullable=True),
        sa.Column("duration_at_least_2_weeks", sa.Boolean(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10mixed__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_icd10mixed__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_icd10mixed__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10mixed__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10mixed__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10mixed__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_icd10mixed")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("icd10mixed", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_icd10mixed__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10mixed__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10mixed__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10mixed__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10mixed__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10mixed_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10mixed_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10mixed_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "icd10schizophrenia",
        sa.Column("passivity_bodily", sa.Boolean(), nullable=True),
        sa.Column("passivity_mental", sa.Boolean(), nullable=True),
        sa.Column("hv_commentary", sa.Boolean(), nullable=True),
        sa.Column("hv_discussing", sa.Boolean(), nullable=True),
        sa.Column("hv_from_body", sa.Boolean(), nullable=True),
        sa.Column("delusions", sa.Boolean(), nullable=True),
        sa.Column("delusional_perception", sa.Boolean(), nullable=True),
        sa.Column("thought_echo", sa.Boolean(), nullable=True),
        sa.Column("thought_withdrawal", sa.Boolean(), nullable=True),
        sa.Column("thought_insertion", sa.Boolean(), nullable=True),
        sa.Column("thought_broadcasting", sa.Boolean(), nullable=True),
        sa.Column("hallucinations_other", sa.Boolean(), nullable=True),
        sa.Column("thought_disorder", sa.Boolean(), nullable=True),
        sa.Column("catatonia", sa.Boolean(), nullable=True),
        sa.Column("negative", sa.Boolean(), nullable=True),
        sa.Column("present_one_month", sa.Boolean(), nullable=True),
        sa.Column("also_manic", sa.Boolean(), nullable=True),
        sa.Column("also_depressive", sa.Boolean(), nullable=True),
        sa.Column("if_mood_psychosis_first", sa.Boolean(), nullable=True),
        sa.Column("not_organic_or_substance", sa.Boolean(), nullable=True),
        sa.Column("behaviour_change", sa.Boolean(), nullable=True),
        sa.Column("performance_decline", sa.Boolean(), nullable=True),
        sa.Column("subtype_paranoid", sa.Boolean(), nullable=True),
        sa.Column("subtype_hebephrenic", sa.Boolean(), nullable=True),
        sa.Column("subtype_catatonic", sa.Boolean(), nullable=True),
        sa.Column("subtype_undifferentiated", sa.Boolean(), nullable=True),
        sa.Column(
            "subtype_postschizophrenic_depression", sa.Boolean(), nullable=True
        ),
        sa.Column("subtype_residual", sa.Boolean(), nullable=True),
        sa.Column("subtype_simple", sa.Boolean(), nullable=True),
        sa.Column("subtype_cenesthopathic", sa.Boolean(), nullable=True),
        sa.Column("date_pertains_to", sa.Date(), nullable=True),
        sa.Column("comments", sa.UnicodeText(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10schizophrenia__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_icd10schizophrenia__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_icd10schizophrenia__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10schizophrenia__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10schizophrenia__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10schizophrenia__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_icd10schizophrenia")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("icd10schizophrenia", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_icd10schizophrenia__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10schizophrenia__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10schizophrenia__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10schizophrenia__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10schizophrenia__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10schizophrenia_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10schizophrenia_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10schizophrenia_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "icd10schizotypal",
        sa.Column("date_pertains_to", sa.Date(), nullable=True),
        sa.Column("comments", sa.UnicodeText(), nullable=True),
        sa.Column("b", sa.Boolean(), nullable=True),
        sa.Column("a1", sa.Boolean(), nullable=True),
        sa.Column("a2", sa.Boolean(), nullable=True),
        sa.Column("a3", sa.Boolean(), nullable=True),
        sa.Column("a4", sa.Boolean(), nullable=True),
        sa.Column("a5", sa.Boolean(), nullable=True),
        sa.Column("a6", sa.Boolean(), nullable=True),
        sa.Column("a7", sa.Boolean(), nullable=True),
        sa.Column("a8", sa.Boolean(), nullable=True),
        sa.Column("a9", sa.Boolean(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10schizotypal__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_icd10schizotypal__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_icd10schizotypal__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10schizotypal__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10schizotypal__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10schizotypal__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_icd10schizotypal")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("icd10schizotypal", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_icd10schizotypal__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10schizotypal__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10schizotypal__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10schizotypal__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10schizotypal__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10schizotypal_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10schizotypal_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10schizotypal_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "icd10specpd",
        sa.Column("date_pertains_to", sa.Date(), nullable=True),
        sa.Column("comments", sa.UnicodeText(), nullable=True),
        sa.Column("skip_paranoid", sa.Boolean(), nullable=True),
        sa.Column("skip_schizoid", sa.Boolean(), nullable=True),
        sa.Column("skip_dissocial", sa.Boolean(), nullable=True),
        sa.Column("skip_eu", sa.Boolean(), nullable=True),
        sa.Column("skip_histrionic", sa.Boolean(), nullable=True),
        sa.Column("skip_anankastic", sa.Boolean(), nullable=True),
        sa.Column("skip_anxious", sa.Boolean(), nullable=True),
        sa.Column("skip_dependent", sa.Boolean(), nullable=True),
        sa.Column("other_pd_present", sa.Boolean(), nullable=True),
        sa.Column("vignette", sa.UnicodeText(), nullable=True),
        sa.Column("g1", sa.Boolean(), nullable=True),
        sa.Column("g2", sa.Boolean(), nullable=True),
        sa.Column("g3", sa.Boolean(), nullable=True),
        sa.Column("g4", sa.Boolean(), nullable=True),
        sa.Column("g5", sa.Boolean(), nullable=True),
        sa.Column("g6", sa.Boolean(), nullable=True),
        sa.Column("g1_1", sa.Boolean(), nullable=True),
        sa.Column("g1_2", sa.Boolean(), nullable=True),
        sa.Column("g1_3", sa.Boolean(), nullable=True),
        sa.Column("g1_4", sa.Boolean(), nullable=True),
        sa.Column("paranoid1", sa.Boolean(), nullable=True),
        sa.Column("paranoid2", sa.Boolean(), nullable=True),
        sa.Column("paranoid3", sa.Boolean(), nullable=True),
        sa.Column("paranoid4", sa.Boolean(), nullable=True),
        sa.Column("paranoid5", sa.Boolean(), nullable=True),
        sa.Column("paranoid6", sa.Boolean(), nullable=True),
        sa.Column("paranoid7", sa.Boolean(), nullable=True),
        sa.Column("schizoid1", sa.Boolean(), nullable=True),
        sa.Column("schizoid2", sa.Boolean(), nullable=True),
        sa.Column("schizoid3", sa.Boolean(), nullable=True),
        sa.Column("schizoid4", sa.Boolean(), nullable=True),
        sa.Column("schizoid5", sa.Boolean(), nullable=True),
        sa.Column("schizoid6", sa.Boolean(), nullable=True),
        sa.Column("schizoid7", sa.Boolean(), nullable=True),
        sa.Column("schizoid8", sa.Boolean(), nullable=True),
        sa.Column("schizoid9", sa.Boolean(), nullable=True),
        sa.Column("dissocial1", sa.Boolean(), nullable=True),
        sa.Column("dissocial2", sa.Boolean(), nullable=True),
        sa.Column("dissocial3", sa.Boolean(), nullable=True),
        sa.Column("dissocial4", sa.Boolean(), nullable=True),
        sa.Column("dissocial5", sa.Boolean(), nullable=True),
        sa.Column("dissocial6", sa.Boolean(), nullable=True),
        sa.Column("eu1", sa.Boolean(), nullable=True),
        sa.Column("eu2", sa.Boolean(), nullable=True),
        sa.Column("eu3", sa.Boolean(), nullable=True),
        sa.Column("eu4", sa.Boolean(), nullable=True),
        sa.Column("eu5", sa.Boolean(), nullable=True),
        sa.Column("eu6", sa.Boolean(), nullable=True),
        sa.Column("eu7", sa.Boolean(), nullable=True),
        sa.Column("eu8", sa.Boolean(), nullable=True),
        sa.Column("eu9", sa.Boolean(), nullable=True),
        sa.Column("eu10", sa.Boolean(), nullable=True),
        sa.Column("histrionic1", sa.Boolean(), nullable=True),
        sa.Column("histrionic2", sa.Boolean(), nullable=True),
        sa.Column("histrionic3", sa.Boolean(), nullable=True),
        sa.Column("histrionic4", sa.Boolean(), nullable=True),
        sa.Column("histrionic5", sa.Boolean(), nullable=True),
        sa.Column("histrionic6", sa.Boolean(), nullable=True),
        sa.Column("anankastic1", sa.Boolean(), nullable=True),
        sa.Column("anankastic2", sa.Boolean(), nullable=True),
        sa.Column("anankastic3", sa.Boolean(), nullable=True),
        sa.Column("anankastic4", sa.Boolean(), nullable=True),
        sa.Column("anankastic5", sa.Boolean(), nullable=True),
        sa.Column("anankastic6", sa.Boolean(), nullable=True),
        sa.Column("anankastic7", sa.Boolean(), nullable=True),
        sa.Column("anankastic8", sa.Boolean(), nullable=True),
        sa.Column("anxious1", sa.Boolean(), nullable=True),
        sa.Column("anxious2", sa.Boolean(), nullable=True),
        sa.Column("anxious3", sa.Boolean(), nullable=True),
        sa.Column("anxious4", sa.Boolean(), nullable=True),
        sa.Column("anxious5", sa.Boolean(), nullable=True),
        sa.Column("dependent1", sa.Boolean(), nullable=True),
        sa.Column("dependent2", sa.Boolean(), nullable=True),
        sa.Column("dependent3", sa.Boolean(), nullable=True),
        sa.Column("dependent4", sa.Boolean(), nullable=True),
        sa.Column("dependent5", sa.Boolean(), nullable=True),
        sa.Column("dependent6", sa.Boolean(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10specpd__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_icd10specpd__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_icd10specpd__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10specpd__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10specpd__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_icd10specpd__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_icd10specpd")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("icd10specpd", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_icd10specpd__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10specpd__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10specpd__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10specpd__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10specpd__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10specpd_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_icd10specpd_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_icd10specpd_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "ided3d",
        sa.Column("last_stage", sa.Integer(), nullable=True),
        sa.Column("max_trials_per_stage", sa.Integer(), nullable=True),
        sa.Column("progress_criterion_x", sa.Integer(), nullable=True),
        sa.Column("progress_criterion_y", sa.Integer(), nullable=True),
        sa.Column("min_number", sa.Integer(), nullable=True),
        sa.Column("max_number", sa.Integer(), nullable=True),
        sa.Column("pause_after_beep_ms", sa.Integer(), nullable=True),
        sa.Column("iti_ms", sa.Integer(), nullable=True),
        sa.Column("counterbalance_dimensions", sa.Integer(), nullable=True),
        sa.Column("volume", sa.Float(), nullable=True),
        sa.Column("offer_abort", sa.Boolean(), nullable=True),
        sa.Column("debug_display_stimuli_only", sa.Boolean(), nullable=True),
        sa.Column("shape_definitions_svg", sa.Text(), nullable=True),
        sa.Column("colour_definitions_rgb", sa.Text(), nullable=True),
        sa.Column("aborted", sa.Integer(), nullable=True),
        sa.Column("finished", sa.Integer(), nullable=True),
        sa.Column("last_trial_completed", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ided3d__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_ided3d__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_ided3d__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ided3d__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ided3d__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ided3d__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_ided3d")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("ided3d", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_ided3d__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_ided3d_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_ided3d_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "ided3d_stages",
        sa.Column("ided3d_id", sa.Integer(), nullable=False),
        sa.Column("stage", sa.Integer(), nullable=False),
        sa.Column("stage_name", sa.Text(), nullable=True),
        sa.Column("relevant_dimension", sa.Text(), nullable=True),
        sa.Column("correct_exemplar", sa.Text(), nullable=True),
        sa.Column("incorrect_exemplar", sa.Text(), nullable=True),
        sa.Column("correct_stimulus_shapes", sa.Text(), nullable=True),
        sa.Column("correct_stimulus_colours", sa.Text(), nullable=True),
        sa.Column("correct_stimulus_numbers", sa.Text(), nullable=True),
        sa.Column("incorrect_stimulus_shapes", sa.Text(), nullable=True),
        sa.Column("incorrect_stimulus_colours", sa.Text(), nullable=True),
        sa.Column("incorrect_stimulus_numbers", sa.Text(), nullable=True),
        sa.Column("first_trial_num", sa.Integer(), nullable=True),
        sa.Column("n_completed_trials", sa.Integer(), nullable=True),
        sa.Column("n_correct", sa.Integer(), nullable=True),
        sa.Column("n_incorrect", sa.Integer(), nullable=True),
        sa.Column("stage_passed", sa.Boolean(), nullable=True),
        sa.Column("stage_failed", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ided3d_stages__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_ided3d_stages__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_ided3d_stages__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ided3d_stages__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ided3d_stages__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ided3d_stages__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_ided3d_stages")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("ided3d_stages", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_ided3d_stages__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d_stages__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d_stages__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d_stages__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d_stages__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d_stages_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d_stages_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "ided3d_trials",
        sa.Column("ided3d_id", sa.Integer(), nullable=False),
        sa.Column("trial", sa.Integer(), nullable=False),
        sa.Column("stage", sa.Integer(), nullable=True),
        sa.Column("correct_location", sa.Integer(), nullable=True),
        sa.Column("incorrect_location", sa.Integer(), nullable=True),
        sa.Column("correct_shape", sa.Integer(), nullable=True),
        sa.Column("correct_colour", sa.Text(), nullable=True),
        sa.Column("correct_number", sa.Integer(), nullable=True),
        sa.Column("incorrect_shape", sa.Integer(), nullable=True),
        sa.Column("incorrect_colour", sa.Text(), nullable=True),
        sa.Column("incorrect_number", sa.Integer(), nullable=True),
        sa.Column(
            "trial_start_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("responded", sa.Boolean(), nullable=True),
        sa.Column(
            "response_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("response_latency_ms", sa.Integer(), nullable=True),
        sa.Column("correct", sa.Boolean(), nullable=True),
        sa.Column("incorrect", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ided3d_trials__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_ided3d_trials__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_ided3d_trials__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ided3d_trials__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ided3d_trials__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ided3d_trials__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_ided3d_trials")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("ided3d_trials", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_ided3d_trials__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d_trials__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d_trials__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d_trials__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d_trials__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d_trials_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ided3d_trials_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "iesr",
        sa.Column("event", sa.UnicodeText(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("q18", sa.Integer(), nullable=True),
        sa.Column("q19", sa.Integer(), nullable=True),
        sa.Column("q20", sa.Integer(), nullable=True),
        sa.Column("q21", sa.Integer(), nullable=True),
        sa.Column("q22", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_iesr__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_iesr__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_iesr__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_iesr__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_iesr__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_iesr__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_iesr")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("iesr", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_iesr__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_iesr__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_iesr__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_iesr__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_iesr__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_iesr_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_iesr_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_iesr_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "ifs",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q4_len2_1", sa.Boolean(), nullable=True),
        sa.Column("q4_len2_2", sa.Boolean(), nullable=True),
        sa.Column("q4_len3_1", sa.Boolean(), nullable=True),
        sa.Column("q4_len3_2", sa.Boolean(), nullable=True),
        sa.Column("q4_len4_1", sa.Boolean(), nullable=True),
        sa.Column("q4_len4_2", sa.Boolean(), nullable=True),
        sa.Column("q4_len5_1", sa.Boolean(), nullable=True),
        sa.Column("q4_len5_2", sa.Boolean(), nullable=True),
        sa.Column("q4_len6_1", sa.Boolean(), nullable=True),
        sa.Column("q4_len6_2", sa.Boolean(), nullable=True),
        sa.Column("q4_len7_1", sa.Boolean(), nullable=True),
        sa.Column("q4_len7_2", sa.Boolean(), nullable=True),
        sa.Column("q6_seq1", sa.Integer(), nullable=True),
        sa.Column("q6_seq2", sa.Integer(), nullable=True),
        sa.Column("q6_seq3", sa.Integer(), nullable=True),
        sa.Column("q6_seq4", sa.Integer(), nullable=True),
        sa.Column("q7_proverb1", sa.Float(), nullable=True),
        sa.Column("q7_proverb2", sa.Float(), nullable=True),
        sa.Column("q7_proverb3", sa.Float(), nullable=True),
        sa.Column("q8_sentence1", sa.Integer(), nullable=True),
        sa.Column("q8_sentence2", sa.Integer(), nullable=True),
        sa.Column("q8_sentence3", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ifs__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_ifs__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_ifs__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ifs__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ifs__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ifs__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_ifs")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("ifs", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_ifs__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ifs__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ifs__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ifs__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_ifs__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_ifs_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_ifs_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ifs_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "irac",
        sa.Column("aim", sa.UnicodeText(), nullable=True),
        sa.Column("achieved", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_irac__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_irac__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_irac__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_irac__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_irac__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_irac__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_irac")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("irac", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_irac__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_irac__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_irac__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_irac__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_irac__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_irac_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_irac_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_irac_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "mast",
        sa.Column("q1", sa.String(length=1), nullable=True),
        sa.Column("q2", sa.String(length=1), nullable=True),
        sa.Column("q3", sa.String(length=1), nullable=True),
        sa.Column("q4", sa.String(length=1), nullable=True),
        sa.Column("q5", sa.String(length=1), nullable=True),
        sa.Column("q6", sa.String(length=1), nullable=True),
        sa.Column("q7", sa.String(length=1), nullable=True),
        sa.Column("q8", sa.String(length=1), nullable=True),
        sa.Column("q9", sa.String(length=1), nullable=True),
        sa.Column("q10", sa.String(length=1), nullable=True),
        sa.Column("q11", sa.String(length=1), nullable=True),
        sa.Column("q12", sa.String(length=1), nullable=True),
        sa.Column("q13", sa.String(length=1), nullable=True),
        sa.Column("q14", sa.String(length=1), nullable=True),
        sa.Column("q15", sa.String(length=1), nullable=True),
        sa.Column("q16", sa.String(length=1), nullable=True),
        sa.Column("q17", sa.String(length=1), nullable=True),
        sa.Column("q18", sa.String(length=1), nullable=True),
        sa.Column("q19", sa.String(length=1), nullable=True),
        sa.Column("q20", sa.String(length=1), nullable=True),
        sa.Column("q21", sa.String(length=1), nullable=True),
        sa.Column("q22", sa.String(length=1), nullable=True),
        sa.Column("q23", sa.String(length=1), nullable=True),
        sa.Column("q24", sa.String(length=1), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_mast__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_mast__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_mast__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_mast__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_mast__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_mast__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_mast")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("mast", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_mast__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_mast__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_mast__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_mast__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_mast__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_mast_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_mast_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_mast_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "mds_updrs",
        sa.Column("q1a", sa.Integer(), nullable=True),
        sa.Column("q1_1", sa.Integer(), nullable=True),
        sa.Column("q1_2", sa.Integer(), nullable=True),
        sa.Column("q1_3", sa.Integer(), nullable=True),
        sa.Column("q1_4", sa.Integer(), nullable=True),
        sa.Column("q1_5", sa.Integer(), nullable=True),
        sa.Column("q1_6", sa.Integer(), nullable=True),
        sa.Column("q1_6a", sa.Integer(), nullable=True),
        sa.Column("q1_7", sa.Integer(), nullable=True),
        sa.Column("q1_8", sa.Integer(), nullable=True),
        sa.Column("q1_9", sa.Integer(), nullable=True),
        sa.Column("q1_10", sa.Integer(), nullable=True),
        sa.Column("q1_11", sa.Integer(), nullable=True),
        sa.Column("q1_12", sa.Integer(), nullable=True),
        sa.Column("q1_13", sa.Integer(), nullable=True),
        sa.Column("q2_1", sa.Integer(), nullable=True),
        sa.Column("q2_2", sa.Integer(), nullable=True),
        sa.Column("q2_3", sa.Integer(), nullable=True),
        sa.Column("q2_4", sa.Integer(), nullable=True),
        sa.Column("q2_5", sa.Integer(), nullable=True),
        sa.Column("q2_6", sa.Integer(), nullable=True),
        sa.Column("q2_7", sa.Integer(), nullable=True),
        sa.Column("q2_8", sa.Integer(), nullable=True),
        sa.Column("q2_9", sa.Integer(), nullable=True),
        sa.Column("q2_10", sa.Integer(), nullable=True),
        sa.Column("q2_11", sa.Integer(), nullable=True),
        sa.Column("q2_12", sa.Integer(), nullable=True),
        sa.Column("q2_13", sa.Integer(), nullable=True),
        sa.Column("q3a", sa.Boolean(), nullable=True),
        sa.Column("q3b", sa.Boolean(), nullable=True),
        sa.Column("q3c", sa.Boolean(), nullable=True),
        sa.Column("q3c1", sa.Float(), nullable=True),
        sa.Column("q3_1", sa.Integer(), nullable=True),
        sa.Column("q3_2", sa.Integer(), nullable=True),
        sa.Column("q3_3a", sa.Integer(), nullable=True),
        sa.Column("q3_3b", sa.Integer(), nullable=True),
        sa.Column("q3_3c", sa.Integer(), nullable=True),
        sa.Column("q3_3d", sa.Integer(), nullable=True),
        sa.Column("q3_3e", sa.Integer(), nullable=True),
        sa.Column("q3_4a", sa.Integer(), nullable=True),
        sa.Column("q3_4b", sa.Integer(), nullable=True),
        sa.Column("q3_5a", sa.Integer(), nullable=True),
        sa.Column("q3_5b", sa.Integer(), nullable=True),
        sa.Column("q3_6a", sa.Integer(), nullable=True),
        sa.Column("q3_6b", sa.Integer(), nullable=True),
        sa.Column("q3_7a", sa.Integer(), nullable=True),
        sa.Column("q3_7b", sa.Integer(), nullable=True),
        sa.Column("q3_8a", sa.Integer(), nullable=True),
        sa.Column("q3_8b", sa.Integer(), nullable=True),
        sa.Column("q3_9", sa.Integer(), nullable=True),
        sa.Column("q3_10", sa.Integer(), nullable=True),
        sa.Column("q3_11", sa.Integer(), nullable=True),
        sa.Column("q3_12", sa.Integer(), nullable=True),
        sa.Column("q3_13", sa.Integer(), nullable=True),
        sa.Column("q3_14", sa.Integer(), nullable=True),
        sa.Column("q3_15a", sa.Integer(), nullable=True),
        sa.Column("q3_15b", sa.Integer(), nullable=True),
        sa.Column("q3_16a", sa.Integer(), nullable=True),
        sa.Column("q3_16b", sa.Integer(), nullable=True),
        sa.Column("q3_17a", sa.Integer(), nullable=True),
        sa.Column("q3_17b", sa.Integer(), nullable=True),
        sa.Column("q3_17c", sa.Integer(), nullable=True),
        sa.Column("q3_17d", sa.Integer(), nullable=True),
        sa.Column("q3_17e", sa.Integer(), nullable=True),
        sa.Column("q3_18", sa.Integer(), nullable=True),
        sa.Column("q3_dyskinesia_present", sa.Boolean(), nullable=True),
        sa.Column("q3_dyskinesia_interfered", sa.Boolean(), nullable=True),
        sa.Column("q3_hy_stage", sa.Integer(), nullable=True),
        sa.Column("q4_1", sa.Integer(), nullable=True),
        sa.Column("q4_2", sa.Integer(), nullable=True),
        sa.Column("q4_3", sa.Integer(), nullable=True),
        sa.Column("q4_4", sa.Integer(), nullable=True),
        sa.Column("q4_5", sa.Integer(), nullable=True),
        sa.Column("q4_6", sa.Integer(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_mds_updrs__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_mds_updrs__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_mds_updrs__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_mds_updrs__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_mds_updrs__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_mds_updrs__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_mds_updrs")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("mds_updrs", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_mds_updrs__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_mds_updrs__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_mds_updrs__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_mds_updrs__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_mds_updrs__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_mds_updrs_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_mds_updrs_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_mds_updrs_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "moca",
        sa.Column("education12y_or_less", sa.Integer(), nullable=True),
        sa.Column("trailpicture_blobid", sa.Integer(), nullable=True),
        sa.Column("cubepicture_blobid", sa.Integer(), nullable=True),
        sa.Column("clockpicture_blobid", sa.Integer(), nullable=True),
        sa.Column("comments", sa.UnicodeText(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("q18", sa.Integer(), nullable=True),
        sa.Column("q19", sa.Integer(), nullable=True),
        sa.Column("q20", sa.Integer(), nullable=True),
        sa.Column("q21", sa.Integer(), nullable=True),
        sa.Column("q22", sa.Integer(), nullable=True),
        sa.Column("q23", sa.Integer(), nullable=True),
        sa.Column("q24", sa.Integer(), nullable=True),
        sa.Column("q25", sa.Integer(), nullable=True),
        sa.Column("q26", sa.Integer(), nullable=True),
        sa.Column("q27", sa.Integer(), nullable=True),
        sa.Column("q28", sa.Integer(), nullable=True),
        sa.Column("register_trial1_1", sa.Integer(), nullable=True),
        sa.Column("register_trial1_2", sa.Integer(), nullable=True),
        sa.Column("register_trial1_3", sa.Integer(), nullable=True),
        sa.Column("register_trial1_4", sa.Integer(), nullable=True),
        sa.Column("register_trial1_5", sa.Integer(), nullable=True),
        sa.Column("register_trial2_1", sa.Integer(), nullable=True),
        sa.Column("register_trial2_2", sa.Integer(), nullable=True),
        sa.Column("register_trial2_3", sa.Integer(), nullable=True),
        sa.Column("register_trial2_4", sa.Integer(), nullable=True),
        sa.Column("register_trial2_5", sa.Integer(), nullable=True),
        sa.Column("recall_category_cue_1", sa.Integer(), nullable=True),
        sa.Column("recall_category_cue_2", sa.Integer(), nullable=True),
        sa.Column("recall_category_cue_3", sa.Integer(), nullable=True),
        sa.Column("recall_category_cue_4", sa.Integer(), nullable=True),
        sa.Column("recall_category_cue_5", sa.Integer(), nullable=True),
        sa.Column("recall_mc_cue_1", sa.Integer(), nullable=True),
        sa.Column("recall_mc_cue_2", sa.Integer(), nullable=True),
        sa.Column("recall_mc_cue_3", sa.Integer(), nullable=True),
        sa.Column("recall_mc_cue_4", sa.Integer(), nullable=True),
        sa.Column("recall_mc_cue_5", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_moca__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_moca__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_moca__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_moca__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_moca__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_moca__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_moca")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("moca", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_moca__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_moca__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_moca__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_moca__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_moca__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_moca_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_moca_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_moca_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "nart",
        sa.Column("chord", sa.Boolean(), nullable=True),
        sa.Column("ache", sa.Boolean(), nullable=True),
        sa.Column("depot", sa.Boolean(), nullable=True),
        sa.Column("aisle", sa.Boolean(), nullable=True),
        sa.Column("bouquet", sa.Boolean(), nullable=True),
        sa.Column("psalm", sa.Boolean(), nullable=True),
        sa.Column("capon", sa.Boolean(), nullable=True),
        sa.Column("deny", sa.Boolean(), nullable=True),
        sa.Column("nausea", sa.Boolean(), nullable=True),
        sa.Column("debt", sa.Boolean(), nullable=True),
        sa.Column("courteous", sa.Boolean(), nullable=True),
        sa.Column("rarefy", sa.Boolean(), nullable=True),
        sa.Column("equivocal", sa.Boolean(), nullable=True),
        sa.Column("naive", sa.Boolean(), nullable=True),
        sa.Column("catacomb", sa.Boolean(), nullable=True),
        sa.Column("gaoled", sa.Boolean(), nullable=True),
        sa.Column("thyme", sa.Boolean(), nullable=True),
        sa.Column("heir", sa.Boolean(), nullable=True),
        sa.Column("radix", sa.Boolean(), nullable=True),
        sa.Column("assignate", sa.Boolean(), nullable=True),
        sa.Column("hiatus", sa.Boolean(), nullable=True),
        sa.Column("subtle", sa.Boolean(), nullable=True),
        sa.Column("procreate", sa.Boolean(), nullable=True),
        sa.Column("gist", sa.Boolean(), nullable=True),
        sa.Column("gouge", sa.Boolean(), nullable=True),
        sa.Column("superfluous", sa.Boolean(), nullable=True),
        sa.Column("simile", sa.Boolean(), nullable=True),
        sa.Column("banal", sa.Boolean(), nullable=True),
        sa.Column("quadruped", sa.Boolean(), nullable=True),
        sa.Column("cellist", sa.Boolean(), nullable=True),
        sa.Column("facade", sa.Boolean(), nullable=True),
        sa.Column("zealot", sa.Boolean(), nullable=True),
        sa.Column("drachm", sa.Boolean(), nullable=True),
        sa.Column("aeon", sa.Boolean(), nullable=True),
        sa.Column("placebo", sa.Boolean(), nullable=True),
        sa.Column("abstemious", sa.Boolean(), nullable=True),
        sa.Column("detente", sa.Boolean(), nullable=True),
        sa.Column("idyll", sa.Boolean(), nullable=True),
        sa.Column("puerperal", sa.Boolean(), nullable=True),
        sa.Column("aver", sa.Boolean(), nullable=True),
        sa.Column("gauche", sa.Boolean(), nullable=True),
        sa.Column("topiary", sa.Boolean(), nullable=True),
        sa.Column("leviathan", sa.Boolean(), nullable=True),
        sa.Column("beatify", sa.Boolean(), nullable=True),
        sa.Column("prelate", sa.Boolean(), nullable=True),
        sa.Column("sidereal", sa.Boolean(), nullable=True),
        sa.Column("demesne", sa.Boolean(), nullable=True),
        sa.Column("syncope", sa.Boolean(), nullable=True),
        sa.Column("labile", sa.Boolean(), nullable=True),
        sa.Column("campanile", sa.Boolean(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_nart__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_nart__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_nart__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_nart__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_nart__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_nart__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_nart")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("nart", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_nart__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_nart__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_nart__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_nart__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_nart__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_nart_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_nart_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_nart_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "npiq",
        sa.Column("endorsed1", sa.Boolean(), nullable=True),
        sa.Column("endorsed2", sa.Boolean(), nullable=True),
        sa.Column("endorsed3", sa.Boolean(), nullable=True),
        sa.Column("endorsed4", sa.Boolean(), nullable=True),
        sa.Column("endorsed5", sa.Boolean(), nullable=True),
        sa.Column("endorsed6", sa.Boolean(), nullable=True),
        sa.Column("endorsed7", sa.Boolean(), nullable=True),
        sa.Column("endorsed8", sa.Boolean(), nullable=True),
        sa.Column("endorsed9", sa.Boolean(), nullable=True),
        sa.Column("endorsed10", sa.Boolean(), nullable=True),
        sa.Column("endorsed11", sa.Boolean(), nullable=True),
        sa.Column("endorsed12", sa.Boolean(), nullable=True),
        sa.Column("severity1", sa.Integer(), nullable=True),
        sa.Column("severity2", sa.Integer(), nullable=True),
        sa.Column("severity3", sa.Integer(), nullable=True),
        sa.Column("severity4", sa.Integer(), nullable=True),
        sa.Column("severity5", sa.Integer(), nullable=True),
        sa.Column("severity6", sa.Integer(), nullable=True),
        sa.Column("severity7", sa.Integer(), nullable=True),
        sa.Column("severity8", sa.Integer(), nullable=True),
        sa.Column("severity9", sa.Integer(), nullable=True),
        sa.Column("severity10", sa.Integer(), nullable=True),
        sa.Column("severity11", sa.Integer(), nullable=True),
        sa.Column("severity12", sa.Integer(), nullable=True),
        sa.Column("distress1", sa.Integer(), nullable=True),
        sa.Column("distress2", sa.Integer(), nullable=True),
        sa.Column("distress3", sa.Integer(), nullable=True),
        sa.Column("distress4", sa.Integer(), nullable=True),
        sa.Column("distress5", sa.Integer(), nullable=True),
        sa.Column("distress6", sa.Integer(), nullable=True),
        sa.Column("distress7", sa.Integer(), nullable=True),
        sa.Column("distress8", sa.Integer(), nullable=True),
        sa.Column("distress9", sa.Integer(), nullable=True),
        sa.Column("distress10", sa.Integer(), nullable=True),
        sa.Column("distress11", sa.Integer(), nullable=True),
        sa.Column("distress12", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("respondent_relationship", sa.Text(), nullable=True),
        sa.Column("respondent_name", sa.Text(), nullable=True),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_npiq__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_npiq__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_npiq__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_npiq__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_npiq__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_npiq__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_npiq")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("npiq", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_npiq__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_npiq__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_npiq__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_npiq__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_npiq__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_npiq_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_npiq_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_npiq_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "panss",
        sa.Column("p1", sa.Integer(), nullable=True),
        sa.Column("p2", sa.Integer(), nullable=True),
        sa.Column("p3", sa.Integer(), nullable=True),
        sa.Column("p4", sa.Integer(), nullable=True),
        sa.Column("p5", sa.Integer(), nullable=True),
        sa.Column("p6", sa.Integer(), nullable=True),
        sa.Column("p7", sa.Integer(), nullable=True),
        sa.Column("n1", sa.Integer(), nullable=True),
        sa.Column("n2", sa.Integer(), nullable=True),
        sa.Column("n3", sa.Integer(), nullable=True),
        sa.Column("n4", sa.Integer(), nullable=True),
        sa.Column("n5", sa.Integer(), nullable=True),
        sa.Column("n6", sa.Integer(), nullable=True),
        sa.Column("n7", sa.Integer(), nullable=True),
        sa.Column("g1", sa.Integer(), nullable=True),
        sa.Column("g2", sa.Integer(), nullable=True),
        sa.Column("g3", sa.Integer(), nullable=True),
        sa.Column("g4", sa.Integer(), nullable=True),
        sa.Column("g5", sa.Integer(), nullable=True),
        sa.Column("g6", sa.Integer(), nullable=True),
        sa.Column("g7", sa.Integer(), nullable=True),
        sa.Column("g8", sa.Integer(), nullable=True),
        sa.Column("g9", sa.Integer(), nullable=True),
        sa.Column("g10", sa.Integer(), nullable=True),
        sa.Column("g11", sa.Integer(), nullable=True),
        sa.Column("g12", sa.Integer(), nullable=True),
        sa.Column("g13", sa.Integer(), nullable=True),
        sa.Column("g14", sa.Integer(), nullable=True),
        sa.Column("g15", sa.Integer(), nullable=True),
        sa.Column("g16", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_panss__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_panss__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_panss__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_panss__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_panss__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_panss__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_panss")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("panss", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_panss__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_panss__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_panss__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_panss__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_panss__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_panss_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_panss_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_panss_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "patient",
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column("forename", sa.Unicode(length=255), nullable=True),
        sa.Column("surname", sa.Unicode(length=255), nullable=True),
        sa.Column("dob", sa.Date(), nullable=True),
        sa.Column("sex", sa.String(length=1), nullable=True),
        sa.Column("address", sa.UnicodeText(), nullable=True),
        sa.Column("gp", sa.UnicodeText(), nullable=True),
        sa.Column("other", sa.UnicodeText(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_patient__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_patient__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_patient__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_patient__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_patient__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_patient__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_patient")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("patient", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_patient__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_patient__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_patient__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_patient__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_patient__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_patient_dob"), ["dob"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_patient_forename"), ["forename"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_patient_sex"), ["sex"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_patient_surname"), ["surname"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_patient_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "patient_idnum",
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("which_idnum", sa.Integer(), nullable=False),
        sa.Column("idnum_value", sa.BigInteger(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_patient_idnum__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_patient_idnum__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_patient_idnum__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_patient_idnum__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_patient_idnum__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_patient_idnum__removing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["which_idnum"],
            ["_idnum_definitions.which_idnum"],
            name=op.f("fk_patient_idnum_which_idnum"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_patient_idnum")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("patient_idnum", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_patient_idnum__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_patient_idnum__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_patient_idnum__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_patient_idnum__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_patient_idnum__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_patient_idnum_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "pclc",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pclc__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_pclc__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_pclc__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pclc__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pclc__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pclc__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_pclc")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("pclc", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_pclc__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pclc__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pclc__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pclc__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_pclc__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_pclc_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_pclc_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pclc_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "pclm",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pclm__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_pclm__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_pclm__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pclm__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pclm__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pclm__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_pclm")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("pclm", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_pclm__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pclm__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pclm__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pclm__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_pclm__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_pclm_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_pclm_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pclm_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "pcls",
        sa.Column("event", sa.UnicodeText(), nullable=True),
        sa.Column("eventdate", sa.UnicodeText(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pcls__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_pcls__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_pcls__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pcls__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pcls__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pcls__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_pcls")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("pcls", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_pcls__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pcls__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pcls__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pcls__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_pcls__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_pcls_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_pcls_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pcls_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "pdss",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pdss__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_pdss__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_pdss__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pdss__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pdss__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pdss__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_pdss")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("pdss", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_pdss__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pdss__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pdss__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pdss__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_pdss__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_pdss_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_pdss_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pdss_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "photo",
        sa.Column("description", sa.UnicodeText(), nullable=True),
        sa.Column("photo_blobid", sa.Integer(), nullable=True),
        sa.Column("rotation", sa.Integer(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_photo__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_photo__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_photo__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_photo__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_photo__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_photo__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_photo")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("photo", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_photo__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_photo__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_photo__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_photo__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_photo__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_photo_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_photo_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_photo_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "photosequence",
        sa.Column("sequence_description", sa.UnicodeText(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_photosequence__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_photosequence__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_photosequence__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_photosequence__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_photosequence__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_photosequence__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_photosequence")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("photosequence", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_photosequence__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_photosequence__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_photosequence__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_photosequence__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_photosequence__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_photosequence_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_photosequence_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_photosequence_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "photosequence_photos",
        sa.Column("photosequence_id", sa.Integer(), nullable=False),
        sa.Column("seqnum", sa.Integer(), nullable=False),
        sa.Column("description", sa.UnicodeText(), nullable=True),
        sa.Column("photo_blobid", sa.Integer(), nullable=True),
        sa.Column("rotation", sa.Integer(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_photosequence_photos__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_photosequence_photos__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_photosequence_photos__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_photosequence_photos__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_photosequence_photos__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_photosequence_photos__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_photosequence_photos")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("photosequence_photos", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_photosequence_photos__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_photosequence_photos__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_photosequence_photos__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_photosequence_photos__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_photosequence_photos__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_photosequence_photos_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_photosequence_photos_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "phq15",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_phq15__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_phq15__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_phq15__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_phq15__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_phq15__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_phq15__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_phq15")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("phq15", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_phq15__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_phq15__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_phq15__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_phq15__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_phq15__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_phq15_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_phq15_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_phq15_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "phq9",
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_phq9__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_phq9__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_phq9__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_phq9__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_phq9__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_phq9__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_phq9")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("phq9", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_phq9__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_phq9__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_phq9__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_phq9__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_phq9__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_phq9_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_phq9_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_phq9_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "progressnote",
        sa.Column("location", sa.UnicodeText(), nullable=True),
        sa.Column("note", sa.UnicodeText(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_progressnote__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_progressnote__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_progressnote__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_progressnote__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_progressnote__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_progressnote__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_progressnote")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("progressnote", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_progressnote__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_progressnote__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_progressnote__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_progressnote__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_progressnote__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_progressnote_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_progressnote_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_progressnote_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "pswq",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pswq__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_pswq__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_pswq__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pswq__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pswq__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pswq__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_pswq")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("pswq", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_pswq__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pswq__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pswq__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pswq__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_pswq__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_pswq_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_pswq_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pswq_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "psychiatricclerking",
        sa.Column("location", sa.UnicodeText(), nullable=True),
        sa.Column("contact_type", sa.UnicodeText(), nullable=True),
        sa.Column("reason_for_contact", sa.UnicodeText(), nullable=True),
        sa.Column("presenting_issue", sa.UnicodeText(), nullable=True),
        sa.Column("systems_review", sa.UnicodeText(), nullable=True),
        sa.Column("collateral_history", sa.UnicodeText(), nullable=True),
        sa.Column("diagnoses_psychiatric", sa.UnicodeText(), nullable=True),
        sa.Column("diagnoses_medical", sa.UnicodeText(), nullable=True),
        sa.Column("operations_procedures", sa.UnicodeText(), nullable=True),
        sa.Column(
            "allergies_adverse_reactions", sa.UnicodeText(), nullable=True
        ),
        sa.Column("medications", sa.UnicodeText(), nullable=True),
        sa.Column("recreational_drug_use", sa.UnicodeText(), nullable=True),
        sa.Column("family_history", sa.UnicodeText(), nullable=True),
        sa.Column("developmental_history", sa.UnicodeText(), nullable=True),
        sa.Column("personal_history", sa.UnicodeText(), nullable=True),
        sa.Column("premorbid_personality", sa.UnicodeText(), nullable=True),
        sa.Column("forensic_history", sa.UnicodeText(), nullable=True),
        sa.Column("current_social_situation", sa.UnicodeText(), nullable=True),
        sa.Column("mse_appearance_behaviour", sa.UnicodeText(), nullable=True),
        sa.Column("mse_speech", sa.UnicodeText(), nullable=True),
        sa.Column("mse_mood_subjective", sa.UnicodeText(), nullable=True),
        sa.Column("mse_mood_objective", sa.UnicodeText(), nullable=True),
        sa.Column("mse_thought_form", sa.UnicodeText(), nullable=True),
        sa.Column("mse_thought_content", sa.UnicodeText(), nullable=True),
        sa.Column("mse_perception", sa.UnicodeText(), nullable=True),
        sa.Column("mse_cognition", sa.UnicodeText(), nullable=True),
        sa.Column("mse_insight", sa.UnicodeText(), nullable=True),
        sa.Column(
            "physical_examination_general", sa.UnicodeText(), nullable=True
        ),
        sa.Column(
            "physical_examination_cardiovascular",
            sa.UnicodeText(),
            nullable=True,
        ),
        sa.Column(
            "physical_examination_respiratory", sa.UnicodeText(), nullable=True
        ),
        sa.Column(
            "physical_examination_abdominal", sa.UnicodeText(), nullable=True
        ),
        sa.Column(
            "physical_examination_neurological",
            sa.UnicodeText(),
            nullable=True,
        ),
        sa.Column("assessment_scales", sa.UnicodeText(), nullable=True),
        sa.Column("investigations_results", sa.UnicodeText(), nullable=True),
        sa.Column("safety_alerts", sa.UnicodeText(), nullable=True),
        sa.Column("risk_assessment", sa.UnicodeText(), nullable=True),
        sa.Column(
            "relevant_legal_information", sa.UnicodeText(), nullable=True
        ),
        sa.Column("current_problems", sa.UnicodeText(), nullable=True),
        sa.Column("patient_carer_concerns", sa.UnicodeText(), nullable=True),
        sa.Column("impression", sa.UnicodeText(), nullable=True),
        sa.Column("management_plan", sa.UnicodeText(), nullable=True),
        sa.Column("information_given", sa.UnicodeText(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_psychiatricclerking__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_psychiatricclerking__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_psychiatricclerking__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_psychiatricclerking__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_psychiatricclerking__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_psychiatricclerking__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_psychiatricclerking")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("psychiatricclerking", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_psychiatricclerking__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_psychiatricclerking__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_psychiatricclerking__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_psychiatricclerking__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_psychiatricclerking__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_psychiatricclerking_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_psychiatricclerking_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_psychiatricclerking_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "pt_satis",
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("bad", sa.UnicodeText(), nullable=True),
        sa.Column("good", sa.UnicodeText(), nullable=True),
        sa.Column("rating", sa.Integer(), nullable=True),
        sa.Column("service", sa.UnicodeText(), nullable=True),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pt_satis__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_pt_satis__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_pt_satis__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pt_satis__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pt_satis__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_pt_satis__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_pt_satis")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("pt_satis", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_pt_satis__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pt_satis__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pt_satis__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pt_satis__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pt_satis__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pt_satis_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pt_satis_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_pt_satis_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "qolbasic",
        sa.Column("tto", sa.Float(), nullable=True),
        sa.Column("rs", sa.Float(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_qolbasic__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_qolbasic__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_qolbasic__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_qolbasic__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_qolbasic__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_qolbasic__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_qolbasic")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("qolbasic", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_qolbasic__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_qolbasic__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_qolbasic__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_qolbasic__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_qolbasic__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_qolbasic_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_qolbasic_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_qolbasic_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "qolsg",
        sa.Column(
            "category_start_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("category_responded", sa.Integer(), nullable=True),
        sa.Column(
            "category_response_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("category_chosen", sa.String(length=6), nullable=True),
        sa.Column("gamble_fixed_option", sa.String(length=7), nullable=True),
        sa.Column(
            "gamble_lottery_option_p", sa.String(length=7), nullable=True
        ),
        sa.Column(
            "gamble_lottery_option_q", sa.String(length=7), nullable=True
        ),
        sa.Column("gamble_lottery_on_left", sa.Integer(), nullable=True),
        sa.Column("gamble_starting_p", sa.Float(), nullable=True),
        sa.Column(
            "gamble_start_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("gamble_responded", sa.Integer(), nullable=True),
        sa.Column(
            "gamble_response_time",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("gamble_p", sa.Float(), nullable=True),
        sa.Column("utility", sa.Float(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_qolsg__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_qolsg__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_qolsg__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_qolsg__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_qolsg__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_qolsg__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_qolsg")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("qolsg", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_qolsg__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_qolsg__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_qolsg__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_qolsg__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_qolsg__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_qolsg_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_qolsg_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_qolsg_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "rand36",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q20", sa.Integer(), nullable=True),
        sa.Column("q21", sa.Integer(), nullable=True),
        sa.Column("q22", sa.Integer(), nullable=True),
        sa.Column("q32", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("q18", sa.Integer(), nullable=True),
        sa.Column("q19", sa.Integer(), nullable=True),
        sa.Column("q23", sa.Integer(), nullable=True),
        sa.Column("q24", sa.Integer(), nullable=True),
        sa.Column("q25", sa.Integer(), nullable=True),
        sa.Column("q26", sa.Integer(), nullable=True),
        sa.Column("q27", sa.Integer(), nullable=True),
        sa.Column("q28", sa.Integer(), nullable=True),
        sa.Column("q29", sa.Integer(), nullable=True),
        sa.Column("q30", sa.Integer(), nullable=True),
        sa.Column("q31", sa.Integer(), nullable=True),
        sa.Column("q33", sa.Integer(), nullable=True),
        sa.Column("q34", sa.Integer(), nullable=True),
        sa.Column("q35", sa.Integer(), nullable=True),
        sa.Column("q36", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_rand36__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_rand36__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_rand36__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_rand36__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_rand36__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_rand36__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_rand36")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("rand36", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_rand36__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_rand36__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_rand36__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_rand36__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_rand36__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_rand36_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_rand36_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_rand36_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "ref_satis_gen",
        sa.Column("bad", sa.UnicodeText(), nullable=True),
        sa.Column("good", sa.UnicodeText(), nullable=True),
        sa.Column("rating", sa.Integer(), nullable=True),
        sa.Column("service", sa.UnicodeText(), nullable=True),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ref_satis_gen__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_ref_satis_gen__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_ref_satis_gen__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ref_satis_gen__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ref_satis_gen__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ref_satis_gen__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_ref_satis_gen")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("ref_satis_gen", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_ref_satis_gen__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ref_satis_gen__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_ref_satis_gen__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ref_satis_gen__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_ref_satis_gen__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ref_satis_gen_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ref_satis_gen_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "ref_satis_spec",
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("bad", sa.UnicodeText(), nullable=True),
        sa.Column("good", sa.UnicodeText(), nullable=True),
        sa.Column("rating", sa.Integer(), nullable=True),
        sa.Column("service", sa.UnicodeText(), nullable=True),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ref_satis_spec__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_ref_satis_spec__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_ref_satis_spec__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ref_satis_spec__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ref_satis_spec__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ref_satis_spec__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_ref_satis_spec")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("ref_satis_spec", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_ref_satis_spec__current"),
            ["_current"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_ref_satis_spec__device_id"),
            ["_device_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_ref_satis_spec__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ref_satis_spec__group_id"),
            ["_group_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_ref_satis_spec__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ref_satis_spec_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ref_satis_spec_patient_id"),
            ["patient_id"],
            unique=False,
        )
        batch_op.create_index(
            batch_op.f("ix_ref_satis_spec_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "slums",
        sa.Column("alert", sa.Integer(), nullable=True),
        sa.Column("highschooleducation", sa.Integer(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q5a", sa.Integer(), nullable=True),
        sa.Column("q5b", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7a", sa.Integer(), nullable=True),
        sa.Column("q7b", sa.Integer(), nullable=True),
        sa.Column("q7c", sa.Integer(), nullable=True),
        sa.Column("q7d", sa.Integer(), nullable=True),
        sa.Column("q7e", sa.Integer(), nullable=True),
        sa.Column("q8b", sa.Integer(), nullable=True),
        sa.Column("q8c", sa.Integer(), nullable=True),
        sa.Column("q9a", sa.Integer(), nullable=True),
        sa.Column("q9b", sa.Integer(), nullable=True),
        sa.Column("q10a", sa.Integer(), nullable=True),
        sa.Column("q10b", sa.Integer(), nullable=True),
        sa.Column("q11a", sa.Integer(), nullable=True),
        sa.Column("q11b", sa.Integer(), nullable=True),
        sa.Column("q11c", sa.Integer(), nullable=True),
        sa.Column("q11d", sa.Integer(), nullable=True),
        sa.Column("clockpicture_blobid", sa.Integer(), nullable=True),
        sa.Column("shapespicture_blobid", sa.Integer(), nullable=True),
        sa.Column("comments", sa.UnicodeText(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_slums__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_slums__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_slums__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_slums__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_slums__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_slums__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_slums")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("slums", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_slums__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_slums__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_slums__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_slums__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_slums__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_slums_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_slums_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_slums_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "smast",
        sa.Column("q1", sa.String(length=1), nullable=True),
        sa.Column("q2", sa.String(length=1), nullable=True),
        sa.Column("q3", sa.String(length=1), nullable=True),
        sa.Column("q4", sa.String(length=1), nullable=True),
        sa.Column("q5", sa.String(length=1), nullable=True),
        sa.Column("q6", sa.String(length=1), nullable=True),
        sa.Column("q7", sa.String(length=1), nullable=True),
        sa.Column("q8", sa.String(length=1), nullable=True),
        sa.Column("q9", sa.String(length=1), nullable=True),
        sa.Column("q10", sa.String(length=1), nullable=True),
        sa.Column("q11", sa.String(length=1), nullable=True),
        sa.Column("q12", sa.String(length=1), nullable=True),
        sa.Column("q13", sa.String(length=1), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_smast__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_smast__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_smast__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_smast__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_smast__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_smast__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_smast")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("smast", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_smast__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_smast__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_smast__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_smast__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_smast__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_smast_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_smast_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_smast_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "swemwbs",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_swemwbs__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_swemwbs__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_swemwbs__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_swemwbs__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_swemwbs__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_swemwbs__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_swemwbs")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("swemwbs", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_swemwbs__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_swemwbs__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_swemwbs__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_swemwbs__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_swemwbs__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_swemwbs_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_swemwbs_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_swemwbs_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "wemwbs",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_wemwbs__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_wemwbs__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_wemwbs__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_wemwbs__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_wemwbs__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_wemwbs__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_wemwbs")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("wemwbs", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_wemwbs__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_wemwbs__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_wemwbs__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_wemwbs__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_wemwbs__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_wemwbs_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_wemwbs_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_wemwbs_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "wsas",
        sa.Column("retired_etc", sa.Boolean(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_wsas__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_wsas__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_wsas__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_wsas__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_wsas__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_wsas__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_wsas")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("wsas", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_wsas__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_wsas__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_wsas__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_wsas__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_wsas__pk"), ["_pk"], unique=False)
        batch_op.create_index(batch_op.f("ix_wsas_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_wsas_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_wsas_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "ybocs",
        sa.Column("target_obsession_1", sa.UnicodeText(), nullable=True),
        sa.Column("target_obsession_2", sa.UnicodeText(), nullable=True),
        sa.Column("target_obsession_3", sa.UnicodeText(), nullable=True),
        sa.Column("target_compulsion_1", sa.UnicodeText(), nullable=True),
        sa.Column("target_compulsion_2", sa.UnicodeText(), nullable=True),
        sa.Column("target_compulsion_3", sa.UnicodeText(), nullable=True),
        sa.Column("target_avoidance_1", sa.UnicodeText(), nullable=True),
        sa.Column("target_avoidance_2", sa.UnicodeText(), nullable=True),
        sa.Column("target_avoidance_3", sa.UnicodeText(), nullable=True),
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q1b", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q6b", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("q13", sa.Integer(), nullable=True),
        sa.Column("q14", sa.Integer(), nullable=True),
        sa.Column("q15", sa.Integer(), nullable=True),
        sa.Column("q16", sa.Integer(), nullable=True),
        sa.Column("q17", sa.Integer(), nullable=True),
        sa.Column("q18", sa.Integer(), nullable=True),
        sa.Column("q19", sa.Integer(), nullable=True),
        sa.Column("clinician_specialty", sa.Text(), nullable=True),
        sa.Column("clinician_service", sa.Text(), nullable=True),
        sa.Column("clinician_contact_details", sa.Text(), nullable=True),
        sa.Column("clinician_post", sa.Text(), nullable=True),
        sa.Column("clinician_name", sa.Text(), nullable=True),
        sa.Column(
            "clinician_professional_registration", sa.Text(), nullable=True
        ),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ybocs__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_ybocs__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_ybocs__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ybocs__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ybocs__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_ybocs__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_ybocs")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("ybocs", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_ybocs__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ybocs__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ybocs__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ybocs__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ybocs__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_ybocs_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_ybocs_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ybocs_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "ybocssc",
        *[
            sa.Column(
                "obs_aggressive_harm_self_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_harm_self_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_harm_self_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_aggressive_harm_others_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_aggressive_harm_others_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_harm_others_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_aggressive_imagery_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_imagery_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_imagery_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_obscenities_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_aggressive_obscenities_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_obscenities_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_aggressive_embarrassing_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_aggressive_embarrassing_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_embarrassing_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_aggressive_impulses_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_impulses_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_impulses_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_aggressive_steal_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_steal_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_steal_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_accident_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_accident_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_accident_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_aggressive_responsible_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_aggressive_responsible_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_responsible_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_aggressive_other_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_other_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_other_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_aggressive_other_detail", sa.UnicodeText(), nullable=True
            ),
            sa.Column(
                "obs_contamination_bodily_waste_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_bodily_waste_past",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_bodily_waste_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_dirt_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_contamination_dirt_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_contamination_dirt_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_contamination_environmental_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_environmental_past",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_environmental_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_household_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_household_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_contamination_household_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_animals_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_animals_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_contamination_animals_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_sticky_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_contamination_sticky_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_contamination_sticky_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_ill_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_contamination_ill_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_contamination_ill_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_contamination_others_ill_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_others_ill_past",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_others_ill_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_feeling_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_feeling_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_contamination_feeling_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_other_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_contamination_other_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_contamination_other_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_contamination_other_detail",
                sa.UnicodeText(),
                nullable=True,
            ),
            sa.Column(
                "obs_sexual_forbidden_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_sexual_forbidden_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_sexual_forbidden_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_sexual_children_incest_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_sexual_children_incest_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_sexual_children_incest_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_sexual_homosexuality_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_sexual_homosexuality_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_sexual_homosexuality_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_sexual_to_others_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_sexual_to_others_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_sexual_to_others_principal", sa.Boolean(), nullable=True
            ),
            sa.Column("obs_sexual_other_current", sa.Boolean(), nullable=True),
            sa.Column("obs_sexual_other_past", sa.Boolean(), nullable=True),
            sa.Column(
                "obs_sexual_other_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_sexual_other_detail", sa.UnicodeText(), nullable=True
            ),
            sa.Column(
                "obs_hoarding_other_current", sa.Boolean(), nullable=True
            ),
            sa.Column("obs_hoarding_other_past", sa.Boolean(), nullable=True),
            sa.Column(
                "obs_hoarding_other_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_hoarding_other_detail", sa.UnicodeText(), nullable=True
            ),
            sa.Column(
                "obs_religious_sacrilege_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_religious_sacrilege_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_religious_sacrilege_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_religious_morality_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_religious_morality_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_religious_morality_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_religious_other_current", sa.Boolean(), nullable=True
            ),
            sa.Column("obs_religious_other_past", sa.Boolean(), nullable=True),
            sa.Column(
                "obs_religious_other_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_religious_other_detail", sa.UnicodeText(), nullable=True
            ),
            sa.Column(
                "obs_symmetry_with_magical_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_symmetry_with_magical_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_symmetry_with_magical_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_symmetry_without_magical_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_symmetry_without_magical_past",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_symmetry_without_magical_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_misc_know_remember_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_know_remember_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_know_remember_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_fear_saying_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_fear_saying_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_fear_saying_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_fear_not_saying_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_fear_not_saying_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_fear_not_saying_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_misc_fear_losing_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_fear_losing_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_fear_losing_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_intrusive_nonviolent_images_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_misc_intrusive_nonviolent_images_past",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_misc_intrusive_nonviolent_images_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_misc_intrusive_sounds_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_misc_intrusive_sounds_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_intrusive_sounds_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "obs_misc_bothered_sounds_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_bothered_sounds_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_bothered_sounds_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column("obs_misc_numbers_current", sa.Boolean(), nullable=True),
            sa.Column("obs_misc_numbers_past", sa.Boolean(), nullable=True),
            sa.Column(
                "obs_misc_numbers_principal", sa.Boolean(), nullable=True
            ),
            sa.Column("obs_misc_colours_current", sa.Boolean(), nullable=True),
            sa.Column("obs_misc_colours_past", sa.Boolean(), nullable=True),
            sa.Column(
                "obs_misc_colours_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_superstitious_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_superstitious_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_misc_superstitious_principal", sa.Boolean(), nullable=True
            ),
            sa.Column("obs_misc_other_current", sa.Boolean(), nullable=True),
            sa.Column("obs_misc_other_past", sa.Boolean(), nullable=True),
            sa.Column("obs_misc_other_principal", sa.Boolean(), nullable=True),
            sa.Column(
                "obs_misc_other_detail", sa.UnicodeText(), nullable=True
            ),
            sa.Column(
                "obs_somatic_illness_current", sa.Boolean(), nullable=True
            ),
            sa.Column("obs_somatic_illness_past", sa.Boolean(), nullable=True),
            sa.Column(
                "obs_somatic_illness_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_somatic_appearance_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_somatic_appearance_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_somatic_appearance_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_somatic_other_current", sa.Boolean(), nullable=True
            ),
            sa.Column("obs_somatic_other_past", sa.Boolean(), nullable=True),
            sa.Column(
                "obs_somatic_other_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "obs_somatic_other_detail", sa.UnicodeText(), nullable=True
            ),
            sa.Column(
                "com_cleaning_handwashing_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_cleaning_handwashing_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_cleaning_handwashing_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_cleaning_toileting_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_cleaning_toileting_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_cleaning_toileting_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_cleaning_cleaning_items_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_cleaning_cleaning_items_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_cleaning_cleaning_items_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_cleaning_other_contaminant_avoidance_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_cleaning_other_contaminant_avoidance_past",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_cleaning_other_contaminant_avoidance_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_cleaning_other_current", sa.Boolean(), nullable=True
            ),
            sa.Column("com_cleaning_other_past", sa.Boolean(), nullable=True),
            sa.Column(
                "com_cleaning_other_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_cleaning_other_detail", sa.UnicodeText(), nullable=True
            ),
            sa.Column(
                "com_checking_locks_appliances_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_checking_locks_appliances_past",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_checking_locks_appliances_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_checking_not_harm_others_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_checking_not_harm_others_past",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_checking_not_harm_others_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_checking_not_harm_self_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_checking_not_harm_self_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_checking_not_harm_self_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_checking_nothing_bad_happens_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_checking_nothing_bad_happens_past",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_checking_nothing_bad_happens_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_checking_no_mistake_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_checking_no_mistake_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_checking_no_mistake_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_checking_somatic_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_checking_somatic_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_checking_somatic_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_checking_other_current", sa.Boolean(), nullable=True
            ),
            sa.Column("com_checking_other_past", sa.Boolean(), nullable=True),
            sa.Column(
                "com_checking_other_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_checking_other_detail", sa.UnicodeText(), nullable=True
            ),
            sa.Column(
                "com_repeat_reread_rewrite_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_repeat_reread_rewrite_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_repeat_reread_rewrite_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_repeat_routine_current", sa.Boolean(), nullable=True
            ),
            sa.Column("com_repeat_routine_past", sa.Boolean(), nullable=True),
            sa.Column(
                "com_repeat_routine_principal", sa.Boolean(), nullable=True
            ),
            sa.Column("com_repeat_other_current", sa.Boolean(), nullable=True),
            sa.Column("com_repeat_other_past", sa.Boolean(), nullable=True),
            sa.Column(
                "com_repeat_other_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_repeat_other_detail", sa.UnicodeText(), nullable=True
            ),
            sa.Column(
                "com_counting_other_current", sa.Boolean(), nullable=True
            ),
            sa.Column("com_counting_other_past", sa.Boolean(), nullable=True),
            sa.Column(
                "com_counting_other_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_counting_other_detail", sa.UnicodeText(), nullable=True
            ),
            sa.Column(
                "com_arranging_other_current", sa.Boolean(), nullable=True
            ),
            sa.Column("com_arranging_other_past", sa.Boolean(), nullable=True),
            sa.Column(
                "com_arranging_other_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_arranging_other_detail", sa.UnicodeText(), nullable=True
            ),
            sa.Column(
                "com_hoarding_other_current", sa.Boolean(), nullable=True
            ),
            sa.Column("com_hoarding_other_past", sa.Boolean(), nullable=True),
            sa.Column(
                "com_hoarding_other_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_hoarding_other_detail", sa.UnicodeText(), nullable=True
            ),
            sa.Column(
                "com_misc_mental_rituals_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_misc_mental_rituals_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_misc_mental_rituals_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column("com_misc_lists_current", sa.Boolean(), nullable=True),
            sa.Column("com_misc_lists_past", sa.Boolean(), nullable=True),
            sa.Column("com_misc_lists_principal", sa.Boolean(), nullable=True),
            sa.Column(
                "com_misc_tell_ask_current", sa.Boolean(), nullable=True
            ),
            sa.Column("com_misc_tell_ask_past", sa.Boolean(), nullable=True),
            sa.Column(
                "com_misc_tell_ask_principal", sa.Boolean(), nullable=True
            ),
            sa.Column("com_misc_touch_current", sa.Boolean(), nullable=True),
            sa.Column("com_misc_touch_past", sa.Boolean(), nullable=True),
            sa.Column("com_misc_touch_principal", sa.Boolean(), nullable=True),
            sa.Column(
                "com_misc_blink_stare_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_misc_blink_stare_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_misc_blink_stare_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_misc_prevent_harm_self_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_misc_prevent_harm_self_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_misc_prevent_harm_self_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_misc_prevent_harm_others_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_misc_prevent_harm_others_past",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_misc_prevent_harm_others_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_misc_prevent_terrible_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_misc_prevent_terrible_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_misc_prevent_terrible_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_misc_eating_ritual_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_misc_eating_ritual_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_misc_eating_ritual_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_misc_superstitious_current", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_misc_superstitious_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_misc_superstitious_principal", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_misc_trichotillomania_current",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_misc_trichotillomania_past", sa.Boolean(), nullable=True
            ),
            sa.Column(
                "com_misc_trichotillomania_principal",
                sa.Boolean(),
                nullable=True,
            ),
            sa.Column(
                "com_misc_self_harm_current", sa.Boolean(), nullable=True
            ),
            sa.Column("com_misc_self_harm_past", sa.Boolean(), nullable=True),
            sa.Column(
                "com_misc_self_harm_principal", sa.Boolean(), nullable=True
            ),
            sa.Column("com_misc_other_current", sa.Boolean(), nullable=True),
            sa.Column("com_misc_other_past", sa.Boolean(), nullable=True),
            sa.Column("com_misc_other_principal", sa.Boolean(), nullable=True),
            sa.Column(
                "com_misc_other_detail", sa.UnicodeText(), nullable=True
            ),
            sa.Column("clinician_specialty", sa.Text(), nullable=True),
            sa.Column("clinician_service", sa.Text(), nullable=True),
            sa.Column("clinician_contact_details", sa.Text(), nullable=True),
            sa.Column("clinician_post", sa.Text(), nullable=True),
            sa.Column("clinician_name", sa.Text(), nullable=True),
            sa.Column(
                "clinician_professional_registration", sa.Text(), nullable=True
            ),
            sa.Column("patient_id", sa.Integer(), nullable=False),
            sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
            sa.Column(
                "when_created",
                PendulumDateTimeAsIsoTextColType(length=32),
                nullable=False,
            ),
            sa.Column(
                "when_firstexit",
                PendulumDateTimeAsIsoTextColType(length=32),
                nullable=True,
            ),
            sa.Column("editing_time_s", sa.Float(), nullable=True),
            sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
            sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
            sa.Column("_manually_erased", sa.Boolean(), nullable=True),
            sa.Column("_removal_pending", sa.Boolean(), nullable=True),
            sa.Column("_group_id", sa.Integer(), nullable=False),
            sa.Column(
                "_manually_erasing_user_id", sa.Integer(), nullable=True
            ),
            sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
            sa.Column("_adding_user_id", sa.Integer(), nullable=True),
            sa.Column(
                "when_last_modified",
                PendulumDateTimeAsIsoTextColType(length=32),
                nullable=True,
            ),
            sa.Column("_removing_user_id", sa.Integer(), nullable=True),
            sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
            sa.Column("_device_id", sa.Integer(), nullable=False),
            sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
            sa.Column("_pk", sa.Integer(), nullable=False),
            sa.Column("_current", sa.Boolean(), nullable=False),
            sa.Column("_addition_pending", sa.Boolean(), nullable=False),
            sa.Column(
                "_manually_erased_at",
                PendulumDateTimeAsIsoTextColType(length=32),
                nullable=True,
            ),
            sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
            sa.Column("_era", sa.String(length=32), nullable=False),
            sa.Column(
                "_when_added_exact",
                PendulumDateTimeAsIsoTextColType(length=32),
                nullable=True,
            ),
            sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
            sa.Column("id", sa.Integer(), nullable=False),
            sa.Column(
                "_camcops_version",
                SemanticVersionColType(length=147),
                nullable=True,
            ),
            sa.Column(
                "_when_removed_exact",
                PendulumDateTimeAsIsoTextColType(length=32),
                nullable=True,
            ),
            sa.Column("_successor_pk", sa.Integer(), nullable=True),
            sa.ForeignKeyConstraint(
                ["_adding_user_id"],
                ["_security_users.id"],
                name=op.f("fk_ybocssc__adding_user_id"),
            ),
            sa.ForeignKeyConstraint(
                ["_device_id"],
                ["_security_devices.id"],
                name=op.f("fk_ybocssc__device_id"),
            ),
            sa.ForeignKeyConstraint(
                ["_group_id"],
                ["_security_groups.id"],
                name=op.f("fk_ybocssc__group_id"),
            ),
            sa.ForeignKeyConstraint(
                ["_manually_erasing_user_id"],
                ["_security_users.id"],
                name=op.f("fk_ybocssc__manually_erasing_user_id"),
            ),
            sa.ForeignKeyConstraint(
                ["_preserving_user_id"],
                ["_security_users.id"],
                name=op.f("fk_ybocssc__preserving_user_id"),
            ),
            sa.ForeignKeyConstraint(
                ["_removing_user_id"],
                ["_security_users.id"],
                name=op.f("fk_ybocssc__removing_user_id"),
            ),
            sa.PrimaryKeyConstraint("_pk", name=op.f("pk_ybocssc")),
        ],
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC"
    )
    with op.batch_alter_table("ybocssc", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_ybocssc__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ybocssc__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ybocssc__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ybocssc__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ybocssc__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ybocssc_id"), ["id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ybocssc_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_ybocssc_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    op.create_table(
        "zbi12",
        sa.Column("q1", sa.Integer(), nullable=True),
        sa.Column("q2", sa.Integer(), nullable=True),
        sa.Column("q3", sa.Integer(), nullable=True),
        sa.Column("q4", sa.Integer(), nullable=True),
        sa.Column("q5", sa.Integer(), nullable=True),
        sa.Column("q6", sa.Integer(), nullable=True),
        sa.Column("q7", sa.Integer(), nullable=True),
        sa.Column("q8", sa.Integer(), nullable=True),
        sa.Column("q9", sa.Integer(), nullable=True),
        sa.Column("q10", sa.Integer(), nullable=True),
        sa.Column("q11", sa.Integer(), nullable=True),
        sa.Column("q12", sa.Integer(), nullable=True),
        sa.Column("respondent_relationship", sa.Text(), nullable=True),
        sa.Column("respondent_name", sa.Text(), nullable=True),
        sa.Column("patient_id", sa.Integer(), nullable=False),
        sa.Column("firstexit_is_finish", sa.Boolean(), nullable=True),
        sa.Column(
            "when_created",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=False,
        ),
        sa.Column(
            "when_firstexit",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("editing_time_s", sa.Float(), nullable=True),
        sa.Column("firstexit_is_abort", sa.Boolean(), nullable=True),
        sa.Column("_preserving_user_id", sa.Integer(), nullable=True),
        sa.Column("_manually_erased", sa.Boolean(), nullable=True),
        sa.Column("_removal_pending", sa.Boolean(), nullable=True),
        sa.Column("_group_id", sa.Integer(), nullable=False),
        sa.Column("_manually_erasing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_removed_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_adding_user_id", sa.Integer(), nullable=True),
        sa.Column(
            "when_last_modified",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_removing_user_id", sa.Integer(), nullable=True),
        sa.Column("_when_added_batch_utc", sa.DateTime(), nullable=True),
        sa.Column("_device_id", sa.Integer(), nullable=False),
        sa.Column("_move_off_tablet", sa.Boolean(), nullable=True),
        sa.Column("_pk", sa.Integer(), nullable=False),
        sa.Column("_current", sa.Boolean(), nullable=False),
        sa.Column("_addition_pending", sa.Boolean(), nullable=False),
        sa.Column(
            "_manually_erased_at",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_predecessor_pk", sa.Integer(), nullable=True),
        sa.Column("_era", sa.String(length=32), nullable=False),
        sa.Column(
            "_when_added_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_forcibly_preserved", sa.Boolean(), nullable=True),
        sa.Column("id", sa.Integer(), nullable=False),
        sa.Column(
            "_camcops_version",
            SemanticVersionColType(length=147),
            nullable=True,
        ),
        sa.Column(
            "_when_removed_exact",
            PendulumDateTimeAsIsoTextColType(length=32),
            nullable=True,
        ),
        sa.Column("_successor_pk", sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(
            ["_adding_user_id"],
            ["_security_users.id"],
            name=op.f("fk_zbi12__adding_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_device_id"],
            ["_security_devices.id"],
            name=op.f("fk_zbi12__device_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_group_id"],
            ["_security_groups.id"],
            name=op.f("fk_zbi12__group_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_manually_erasing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_zbi12__manually_erasing_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_preserving_user_id"],
            ["_security_users.id"],
            name=op.f("fk_zbi12__preserving_user_id"),
        ),
        sa.ForeignKeyConstraint(
            ["_removing_user_id"],
            ["_security_users.id"],
            name=op.f("fk_zbi12__removing_user_id"),
        ),
        sa.PrimaryKeyConstraint("_pk", name=op.f("pk_zbi12")),
        # mysql_charset='utf8mb4',
        # mysql_collate='utf8mb4_unicode_ci',
        mysql_charset="utf8mb4 COLLATE utf8mb4_unicode_ci",
        mysql_engine="InnoDB",
        mysql_row_format="DYNAMIC",
    )
    with op.batch_alter_table("zbi12", schema=None) as batch_op:
        batch_op.create_index(
            batch_op.f("ix_zbi12__current"), ["_current"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_zbi12__device_id"), ["_device_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_zbi12__era"), ["_era"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_zbi12__group_id"), ["_group_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_zbi12__pk"), ["_pk"], unique=False
        )
        batch_op.create_index(batch_op.f("ix_zbi12_id"), ["id"], unique=False)
        batch_op.create_index(
            batch_op.f("ix_zbi12_patient_id"), ["patient_id"], unique=False
        )
        batch_op.create_index(
            batch_op.f("ix_zbi12_when_last_modified"),
            ["when_last_modified"],
            unique=False,
        )

    # ### end Alembic commands ###


# noinspection PyPep8
def downgrade():
    op.drop_table("zbi12")
    op.drop_table("ybocssc")
    op.drop_table("ybocs")
    op.drop_table("wsas")
    op.drop_table("wemwbs")
    op.drop_table("swemwbs")
    op.drop_table("smast")
    op.drop_table("slums")
    op.drop_table("ref_satis_spec")
    op.drop_table("ref_satis_gen")
    op.drop_table("rand36")
    op.drop_table("qolsg")
    op.drop_table("qolbasic")
    op.drop_table("pt_satis")
    op.drop_table("psychiatricclerking")
    op.drop_table("pswq")
    op.drop_table("progressnote")
    op.drop_table("phq9")
    op.drop_table("phq15")
    op.drop_table("photosequence_photos")
    op.drop_table("photosequence")
    op.drop_table("photo")
    op.drop_table("pdss")
    op.drop_table("pcls")
    op.drop_table("pclm")
    op.drop_table("pclc")
    op.drop_table("patient_idnum")
    op.drop_table("patient")
    op.drop_table("panss")
    op.drop_table("npiq")
    op.drop_table("nart")
    op.drop_table("moca")
    op.drop_table("mds_updrs")
    op.drop_table("mast")
    op.drop_table("irac")
    op.drop_table("ifs")
    op.drop_table("iesr")
    op.drop_table("ided3d_trials")
    op.drop_table("ided3d_stages")
    op.drop_table("ided3d")
    op.drop_table("icd10specpd")
    op.drop_table("icd10schizotypal")
    op.drop_table("icd10schizophrenia")
    op.drop_table("icd10mixed")
    op.drop_table("icd10manic")
    op.drop_table("icd10depressive")
    op.drop_table("honosca")
    op.drop_table("honos65")
    op.drop_table("honos")
    op.drop_table("hamd7")
    op.drop_table("hamd")
    op.drop_table("hama")
    op.drop_table("hads_respondent")
    op.drop_table("hads")
    op.drop_table("gmcpq")
    op.drop_table("gds15")
    op.drop_table("gaf")
    op.drop_table("gad7")
    op.drop_table("frs")
    op.drop_table("fft")
    op.drop_table("fast")
    op.drop_table("distressthermometer")
    op.drop_table("diagnosis_icd9cm_item")
    op.drop_table("diagnosis_icd9cm")
    op.drop_table("diagnosis_icd10_item")
    op.drop_table("diagnosis_icd10")
    op.drop_table("demqolproxy")
    op.drop_table("demqol")
    op.drop_table("demoquestionnaire")
    op.drop_table("deakin_1_healthreview")
    op.drop_table("dast")
    op.drop_table("dad")
    op.drop_table("cpft_lps_resetresponseclock")
    op.drop_table("cpft_lps_referral")
    op.drop_table("cpft_lps_discharge")
    op.drop_table("cope_brief")
    op.drop_table("contactlog")
    op.drop_table("ciwa")
    op.drop_table("cgisch")
    op.drop_table("cgi_i")
    op.drop_table("cgi")
    op.drop_table("cecaq3")
    op.drop_table("cbir")
    op.drop_table("cardinal_expdetthreshold_trials")
    op.drop_table("cardinal_expdetthreshold")
    op.drop_table("cardinal_expdet_trials")
    op.drop_table("cardinal_expdet_trialgroupspec")
    op.drop_table("cardinal_expdet")
    op.drop_table("caps")
    op.drop_table("cape42")
    op.drop_table("cage")
    op.drop_table("bprse")
    op.drop_table("bprs")
    op.drop_table("bmi")
    op.drop_table("blobs")
    op.drop_table("bdi")
    op.drop_table("badls")
    op.drop_table("audit_c")
    op.drop_table("audit")
    op.drop_table("aims")
    op.drop_table("ace3")
    op.drop_table("_security_audit")
    op.drop_table("_dirty_tables")
    op.drop_table("_special_notes")
    op.drop_table("_security_webviewer_sessions")
    op.drop_table("_security_user_group")
    op.drop_table("_security_devices")
    op.drop_table("_hl7_message_log")
    op.drop_table("_security_users")
    op.drop_table("_security_group_group")
    op.drop_table("_hl7_run_log")
    op.drop_table("_task_filters")
    op.drop_table("_server_settings")
    op.drop_table("_security_login_failures")
    op.drop_table("_security_groups")
    op.drop_table("_security_account_lockouts")
    op.drop_table("_idnum_definitions")
