#
# rbfly - a library for RabbitMQ Streams using Python asyncio
#
# Copyright (C) 2021-2022 by Artur Wroblewski <wrobell@riseup.net>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

from rbfly.amqp._message import Message, encode_amqp, decode_amqp

import pytest

# decoding and encoding of AMQP types in the context of Python objects
DATA = (
    # binary, opaque message (first with nulls in the message)
    (Message(b'\x01\x00\x00\x00\x02'), b'\x00Su\xa0\x05\x01\x00\x00\x00\x02'),
    (Message(b'abcde'), b'\x00Su\xa0\x05abcde'),
    (Message(b'a' * 256), b'\x00Su\xb0\x00\x00\x01\x00' + b'a' * 256),

    # string
    (Message('abcde'), b'\x00Sw\xa1\x05abcde'),
    (Message('a' * 256), b'\x00Sw\xb1\x00\x00\x01\x00' + b'a' * 256),

    # boolean
    (Message(True), b'\x00SwA'),
    (Message(False), b'\x00SwB'),

    # int
    (Message(-2 ** 31), b'\x00Sw\x71\x80\x00\x00\x00'),
    (Message(2 ** 31 - 1), b'\x00Sw\x71\x7f\xff\xff\xff'),

    # long
    (Message(-2 ** 63), b'\x00Sw\x81\x80\x00\x00\x00\x00\x00\x00\x00'),
    (Message(2 ** 63 - 1), b'\x00Sw\x81\x7f\xff\xff\xff\xff\xff\xff\xff'),

    # ulong
    (Message(2 ** 64 - 1), b'\x00Sw\x80\xff\xff\xff\xff\xff\xff\xff\xff'),

    # double
    (Message(201.102), b'\x00Sw\x82@i#C\x95\x81\x06%'),

    # map
    (
        Message({'a': 1, 'b': 2}),
        b'\x00Sw\xd1\x00\x00\x00\x18\x00\x00\x00\x04\xa1\x01a\x71\x00\x00\x00\x01\xa1\x01b\x71\x00\x00\x00\x02',
    ),
    # nested map
    (
        Message({'a': 1, 'b': {'a': 1, 'b': 2}}),  # type: ignore
        b'\x00Sw\xd1\x00\x00\x00\x2c\x00\x00\x00\x04\xa1\x01a\x71\x00\x00\x00\x01\xa1\x01b\xd1\x00\x00\x00\x18\x00\x00\x00\x04\xa1\x01a\x71\x00\x00\x00\x01\xa1\x01b\x71\x00\x00\x00\x02',
    ),
    # list
    (
        Message(['a', 1, 'b', 2]),
        b'\x00Sw\xd0\x00\x00\x00\x18\x00\x00\x00\x04\xa1\x01a\x71\x00\x00\x00\x01\xa1\x01b\x71\x00\x00\x00\x02',
    ),
    # nested list
    (
        Message(['a', 1, 'b', ['a', 1, 'b', 2]]),  # type: ignore
        b'\x00Sw\xd0\x00\x00\x00\x2c\x00\x00\x00\x04\xa1\x01a\x71\x00\x00\x00\x01\xa1\x01b\xd0\x00\x00\x00\x18\x00\x00\x00\x04\xa1\x01a\x71\x00\x00\x00\x01\xa1\x01b\x71\x00\x00\x00\x02',
    ),
)

# decoding of AMQP types, which are not encoded by rbfly
DATA_PARSED = (
    # ubyte
    (Message(255), b'\x00Sw\x50\xff'),

    # ushort
    (Message(2 ** 16 - 1), b'\x00Sw\x60\xff\xff'),

    # uint, smalluint, uint0
    (Message(2 ** 32 - 1), b'\x00Sw\x70\xff\xff\xff\xff'),
    (Message(255), b'\x00Sw\x52\xff'),
    (Message(0), b'\x00Sw\x43'),

    # ulong, smallulong, ulong0
    (Message(2 ** 64 - 1), b'\x00Sw\x80\xff\xff\xff\xff\xff\xff\xff\xff'),
    (Message(255), b'\x00Sw\x53\xff'),
    (Message(0), b'\x00Sw\x44'),

    # byte
    (Message(-1), b'\x00Sw\x51\xff'),

    # short
    (Message(-1), b'\x00Sw\x61\xff\xff'),

    # int, smallint
    (Message(-1), b'\x00Sw\x71\xff\xff\xff\xff'),
    (Message(-1), b'\x00Sw\x54\xff'),

    # long
    (Message(-1), b'\x00Sw\x81\xff\xff\xff\xff\xff\xff\xff\xff'),
    (Message(-1), b'\x00Sw\x55\xff'),

    # float
    (Message(201.1020050048828), b'\x00Sw\x72CI\x1a\x1d'),

    # map8
    (
        Message({'a': 1, 'b': 2}),
        b'\x00Sw\xc1\x12\x04\xa1\x01a\x71\x00\x00\x00\x01\xa1\x01b\x71\x00\x00\x00\x02',
    ),
    # list8
    (
        Message(['a', 1, 'b', 2]),
        b'\x00Sw\xc0\x12\x04\xa1\x01a\x71\x00\x00\x00\x01\xa1\x01b\x71\x00\x00\x00\x02',
    ),
    # list0
    (
        Message([]),
        b'\x00Sw\x45',
    ),
)

MESSAGE_REPR = (
    (
        Message('a-string'),
        'Message(body=\'a-string\', stream_offset=0, stream_timestamp=0.0)',
    ),
    (
        Message('a-string-and-more', stream_timestamp=2.0),
        'Message(body=\'a-string-a...\', stream_offset=0, stream_timestamp=2.0)',
    ),
    (
        Message(b'binary-data-and-more'),
        'Message(body=b\'binary-dat...\', stream_offset=0, stream_timestamp=0.0)',
    ),
    (
        Message(15),
        'Message(body=15, stream_offset=0, stream_timestamp=0.0)',
    ),
    (
        Message({'a': 15}),
        'Message(body={\'a\': 15}, stream_offset=0, stream_timestamp=0.0)',
    ),
)

@pytest.mark.parametrize('message, expected', DATA)
def test_encode(message: Message, expected: bytes) -> None:
    """
    Test encoding AMQP messages.
    """
    msg_buffer = bytearray(1024)
    size = encode_amqp(msg_buffer, message)
    assert bytes(msg_buffer[:size]) == expected

def test_encode_binary_invalid() -> None:
    """
    Test error on encoding too long binary AMQP messages.
    """
    msg_buffer = bytearray(1024)
    with pytest.raises(ValueError):
        encode_amqp(msg_buffer, Message(b'a' * 2 ** 32))

@pytest.mark.parametrize('message, data', DATA + DATA_PARSED)
def test_decode(message: Message, data: bytes) -> None:
    """
    Test decoding AMQP messages.
    """
    result = decode_amqp(data)
    assert result == message

@pytest.mark.parametrize('message, expected', MESSAGE_REPR)
def test_message_repr(message: Message, expected: str) -> None:
    """
    Test AMQP message representation.
    """
    assert repr(message) == expected

def test_decode_map_invalid_len() -> None:
    """
    Test decoding AMQP map of invalid length.
    """
    # odd number of elements (3 to be exact) instead of even
    data = b'\x00Sw\xd1\x00\x00\x00\x18\x00\x00\x00\x03\xa1\x01a\x71\x00\x00\x00\x01\xa1\x01b\x71\x00\x00\x00\x02'
    msg = decode_amqp(data)
    assert msg.body is None

# vim: sw=4:et:ai
