"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssociationManager = void 0;
const cdk = require("aws-cdk-lib");
const aws_servicecatalog_1 = require("aws-cdk-lib/aws-servicecatalog");
const util_1 = require("./util");
const validation_1 = require("./validation");
class AssociationManager {
    static associateProductWithPortfolio(portfolio, product, options) {
        validation_1.InputValidator.validateLength(this.prettyPrintAssociation(portfolio, product), 'description', 0, 2000, options === null || options === void 0 ? void 0 : options.description);
        const associationKey = util_1.hashValues(portfolio.node.addr, product.node.addr, product.stack.node.addr);
        const constructId = `PortfolioProductAssociation${associationKey}`;
        const existingAssociation = portfolio.node.tryFindChild(constructId);
        const cfnAssociation = existingAssociation
            ? existingAssociation
            : new aws_servicecatalog_1.CfnPortfolioProductAssociation(portfolio, constructId, {
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
            });
        return {
            associationKey: associationKey,
            cfnPortfolioProductAssociation: cfnAssociation,
        };
    }
    static constrainTagUpdates(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `ResourceUpdateConstraint${association.associationKey}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new aws_servicecatalog_1.CfnResourceUpdateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                tagUpdateOnProvisionedProduct: options.allow === false ? 'NOT_ALLOWED' : 'ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot have multiple tag update constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static notifyOnStackEvents(portfolio, product, topic, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchNotificationConstraint${util_1.hashValues(topic.node.addr, topic.stack.node.addr, association.associationKey)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new aws_servicecatalog_1.CfnLaunchNotificationConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                notificationArns: [topic.topicArn],
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Topic ${topic.node.path} is already subscribed to association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static constrainCloudFormationParameters(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchTemplateConstraint${util_1.hashValues(association.associationKey, options.rule.ruleName)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new aws_servicecatalog_1.CfnLaunchTemplateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                rules: this.formatTemplateRule(portfolio.stack, options.rule),
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Provisioning rule ${options.rule.ruleName} already configured on association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static setLaunchRole(portfolio, product, launchRole, options) {
        this.setLaunchRoleConstraint(portfolio, product, options, {
            roleArn: launchRole.roleArn,
        });
    }
    static setLocalLaunchRoleName(portfolio, product, launchRoleName, options) {
        this.setLaunchRoleConstraint(portfolio, product, options, {
            localRoleName: launchRoleName,
        });
    }
    static deployWithStackSets(portfolio, product, options) {
        var _a;
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a launch role has already been set.
        if (portfolio.node.tryFindChild(this.launchRoleConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot configure StackSet deployment when a launch role is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.stackSetConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new aws_servicecatalog_1.CfnStackSetConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: (_a = options.description) !== null && _a !== void 0 ? _a : '',
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                accountList: options.accounts,
                regionList: options.regions,
                adminRole: options.adminRole.roleArn,
                executionRole: options.executionRoleName,
                stackInstanceControl: options.allowStackSetInstanceOperations ? 'ALLOWED' : 'NOT_ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot configure multiple StackSet deployment constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static associateTagOptions(portfolio, tagOptions) {
        const portfolioStack = cdk.Stack.of(portfolio);
        for (const [key, tagOptionsList] of Object.entries(tagOptions.tagOptionsMap)) {
            validation_1.InputValidator.validateLength(portfolio.node.addr, 'TagOption key', 1, 128, key);
            tagOptionsList.forEach((value) => {
                validation_1.InputValidator.validateLength(portfolio.node.addr, 'TagOption value', 1, 256, value);
                const tagOptionKey = util_1.hashValues(key, value, portfolioStack.node.addr);
                const tagOptionConstructId = `TagOption${tagOptionKey}`;
                let cfnTagOption = portfolioStack.node.tryFindChild(tagOptionConstructId);
                if (!cfnTagOption) {
                    cfnTagOption = new aws_servicecatalog_1.CfnTagOption(portfolioStack, tagOptionConstructId, {
                        key: key,
                        value: value,
                        active: true,
                    });
                }
                const tagAssocationKey = util_1.hashValues(key, value, portfolio.node.addr);
                const tagAssocationConstructId = `TagOptionAssociation${tagAssocationKey}`;
                if (!portfolio.node.tryFindChild(tagAssocationConstructId)) {
                    new aws_servicecatalog_1.CfnTagOptionAssociation(portfolio, tagAssocationConstructId, {
                        resourceId: portfolio.portfolioId,
                        tagOptionId: cfnTagOption.ref,
                    });
                }
            });
        }
        ;
    }
    static setLaunchRoleConstraint(portfolio, product, options, roleOptions) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a stackset deployment constraint has already been configured.
        if (portfolio.node.tryFindChild(this.stackSetConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot set launch role when a StackSet rule is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.launchRoleConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new aws_servicecatalog_1.CfnLaunchRoleConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                roleArn: roleOptions.roleArn,
                localRoleName: roleOptions.localRoleName,
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot set multiple launch roles for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static stackSetConstraintLogicalId(associationKey) {
        return `StackSetConstraint${associationKey}`;
    }
    static launchRoleConstraintLogicalId(associationKey) {
        return `LaunchRoleConstraint${associationKey}`;
    }
    static prettyPrintAssociation(portfolio, product) {
        return `- Portfolio: ${portfolio.node.path} | Product: ${product.node.path}`;
    }
    static formatTemplateRule(stack, rule) {
        return JSON.stringify({
            [rule.ruleName]: {
                Assertions: this.formatAssertions(stack, rule.assertions),
                RuleCondition: rule.condition ? stack.resolve(rule.condition) : undefined,
            },
        });
    }
    static formatAssertions(stack, assertions) {
        return assertions.reduce((formattedAssertions, assertion) => {
            formattedAssertions.push({
                Assert: stack.resolve(assertion.assert),
                AssertDescription: assertion.description,
            });
            return formattedAssertions;
        }, new Array());
    }
    ;
}
exports.AssociationManager = AssociationManager;
//# sourceMappingURL=data:application/json;base64,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