"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 */
class OriginBase {
    constructor(domainName, props = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cloudfront_OriginProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, OriginBase);
            }
            throw error;
        }
        validateIntInRangeOrUndefined('connectionTimeout', 1, 10, props.connectionTimeout?.toSeconds());
        validateIntInRangeOrUndefined('connectionAttempts', 1, 3, props.connectionAttempts, false);
        validateCustomHeaders(props.customHeaders);
        this.domainName = domainName;
        this.originPath = this.validateOriginPath(props.originPath);
        this.connectionTimeout = props.connectionTimeout;
        this.connectionAttempts = props.connectionAttempts;
        this.customHeaders = props.customHeaders;
        this.originShieldRegion = props.originShieldRegion;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     */
    bind(_scope, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cloudfront_OriginBindOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return {
            originProperty: {
                domainName: this.domainName,
                id: options.originId,
                originPath: this.originPath,
                connectionAttempts: this.connectionAttempts,
                connectionTimeout: this.connectionTimeout?.toSeconds(),
                originCustomHeaders: this.renderCustomHeaders(),
                s3OriginConfig,
                customOriginConfig,
                originShield: this.renderOriginShield(this.originShieldRegion),
            },
        };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
    renderCustomHeaders() {
        if (!this.customHeaders || Object.entries(this.customHeaders).length === 0) {
            return undefined;
        }
        return Object.entries(this.customHeaders).map(([headerName, headerValue]) => {
            return { headerName, headerValue };
        });
    }
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    validateOriginPath(originPath) {
        if (core_1.Token.isUnresolved(originPath)) {
            return originPath;
        }
        if (originPath === undefined) {
            return undefined;
        }
        let path = originPath;
        if (!path.startsWith('/')) {
            path = '/' + path;
        }
        if (path.endsWith('/')) {
            path = path.slice(0, -1);
        }
        return path;
    }
    /**
     * Takes origin shield region and converts to CfnDistribution.OriginShieldProperty
     */
    renderOriginShield(originShieldRegion) {
        return originShieldRegion
            ? { enabled: true, originShieldRegion }
            : undefined;
    }
}
exports.OriginBase = OriginBase;
_a = JSII_RTTI_SYMBOL_1;
OriginBase[_a] = { fqn: "@aws-cdk/aws-cloudfront.OriginBase", version: "1.175.0" };
/**
 * Throws an error if a value is defined and not an integer or not in a range.
 */
function validateIntInRangeOrUndefined(name, min, max, value, isDuration = true) {
    if (value === undefined) {
        return;
    }
    if (!Number.isInteger(value) || value < min || value > max) {
        const seconds = isDuration ? ' seconds' : '';
        throw new Error(`${name}: Must be an int between ${min} and ${max}${seconds} (inclusive); received ${value}.`);
    }
}
/**
 * Throws an error if custom header assignment is prohibited by CloudFront.
 * @link: https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/add-origin-custom-headers.html#add-origin-custom-headers-denylist
 */
function validateCustomHeaders(customHeaders) {
    if (!customHeaders || Object.entries(customHeaders).length === 0) {
        return;
    }
    const customHeaderKeys = Object.keys(customHeaders);
    const prohibitedHeaderKeys = [
        'Cache-Control', 'Connection', 'Content-Length', 'Cookie', 'Host', 'If-Match', 'If-Modified-Since', 'If-None-Match', 'If-Range', 'If-Unmodified-Since',
        'Max-Forwards', 'Pragma', 'Proxy-Authorization', 'Proxy-Connection', 'Range', 'Request-Range', 'TE', 'Trailer', 'Transfer-Encoding', 'Upgrade', 'Via',
        'X-Real-Ip',
    ];
    const prohibitedHeaderKeyPrefixes = [
        'X-Amz-', 'X-Edge-',
    ];
    const prohibitedHeadersKeysMatches = customHeaderKeys.filter(customKey => {
        return prohibitedHeaderKeys.map((prohibitedKey) => prohibitedKey.toLowerCase()).includes(customKey.toLowerCase());
    });
    const prohibitedHeaderPrefixMatches = customHeaderKeys.filter(customKey => {
        return prohibitedHeaderKeyPrefixes.some(prohibitedKeyPrefix => customKey.toLowerCase().startsWith(prohibitedKeyPrefix.toLowerCase()));
    });
    if (prohibitedHeadersKeysMatches.length !== 0) {
        throw new Error(`The following headers cannot be configured as custom origin headers: ${prohibitedHeadersKeysMatches.join(', ')}`);
    }
    if (prohibitedHeaderPrefixMatches.length !== 0) {
        throw new Error(`The following headers cannot be used as prefixes for custom origin headers: ${prohibitedHeaderPrefixMatches.join(', ')}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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