"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const dns_1 = require("./dns");
const CUSTOM_HTTP_HEADER = 'X_Request_From_CloudFront';
class Application extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id);
        const stack = aws_cdk_lib_1.Stack.of(this);
        this.domainName = props.domainName;
        this.domainZone = props.domainZone;
        this.cloudFrontHashHeader = (_b = props.cloudFrontHashHeader) !== null && _b !== void 0 ? _b : Buffer.from(`${stack.stackName}.${this.domainName}`).toString('base64');
        this.removalPolicy = props.removalPolicy;
        const cluster = new aws_cdk_lib_1.aws_ecs.Cluster(this, 'Cluster', {
            containerInsights: true,
            vpc: props.vpc,
        });
        this.targetGroup = new aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationTargetGroup(this, 'TargetGroup', {
            vpc: props.vpc,
            port: 80,
            targetType: aws_cdk_lib_1.aws_elasticloadbalancingv2.TargetType.IP,
            stickinessCookieDuration: aws_cdk_lib_1.Duration.days(7),
        });
        this.targetGroup.configureHealthCheck({
            path: '/wp-includes/images/blank.gif',
            interval: aws_cdk_lib_1.Duration.minutes(1),
        });
        const loadBalancer = new aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationLoadBalancer(this, 'Loadbalancer', {
            vpc: props.vpc,
            internetFacing: true,
            http2Enabled: true,
        });
        this.listener = loadBalancer.addListener('Listener', {
            port: 443,
            protocol: aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTPS,
            certificates: [
                aws_cdk_lib_1.aws_elasticloadbalancingv2.ListenerCertificate.fromCertificateManager(new aws_cdk_lib_1.aws_certificatemanager.Certificate(this, 'LBCertificate', {
                    domainName: this.domainName,
                    validation: aws_cdk_lib_1.aws_certificatemanager.CertificateValidation.fromDns(this.domainZone),
                })),
            ],
            sslPolicy: aws_cdk_lib_1.aws_elasticloadbalancingv2.SslPolicy.FORWARD_SECRECY_TLS12,
            defaultAction: aws_cdk_lib_1.aws_elasticloadbalancingv2.ListenerAction.fixedResponse(403, {
                contentType: 'text/plain',
                messageBody: 'Access denied',
            }),
        });
        this.listener.addAction('Cloudfornt', {
            action: aws_cdk_lib_1.aws_elasticloadbalancingv2.ListenerAction.forward([this.targetGroup]),
            conditions: [aws_cdk_lib_1.aws_elasticloadbalancingv2.ListenerCondition.httpHeader(CUSTOM_HTTP_HEADER, [this.cloudFrontHashHeader])],
            priority: 100,
        });
        const taskDefinition = new aws_cdk_lib_1.aws_ecs.FargateTaskDefinition(this, 'TaskDefinition', {
            memoryLimitMiB: (_c = props.memoryLimitMiB) !== null && _c !== void 0 ? _c : 512,
            cpu: 256,
            volumes: [props.volume],
        });
        const container = taskDefinition.addContainer('Wordpress', {
            image: (_d = props.image) !== null && _d !== void 0 ? _d : aws_cdk_lib_1.aws_ecs.ContainerImage.fromRegistry('wordpress:5.8-apache'),
            environment: {
                ...props.environment,
                ...props.database.environment,
            },
            secrets: {
                ...props.secrets,
                ...props.database.secrets,
            },
            logging: (_e = props.logDriver) !== null && _e !== void 0 ? _e : aws_cdk_lib_1.aws_ecs.LogDriver.awsLogs({
                streamPrefix: `${stack.stackName}WordpressContainerLog`,
                logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
            }),
        });
        container.addPortMappings({
            containerPort: 80,
        });
        container.addMountPoints({
            containerPath: '/var/www/html',
            readOnly: false,
            sourceVolume: props.volume.name,
        });
        this.service = new aws_cdk_lib_1.aws_ecs.FargateService(this, 'Service', {
            cluster,
            serviceName: props.serviceName,
            taskDefinition,
            desiredCount: 2,
        });
        this.service.connections.allowFrom(loadBalancer, aws_cdk_lib_1.aws_ec2.Port.tcp(80));
        this.targetGroup.addTarget(this.service);
        const origin = new aws_cdk_lib_1.aws_cloudfront_origins.LoadBalancerV2Origin(loadBalancer, {
            originSslProtocols: [aws_cdk_lib_1.aws_cloudfront.OriginSslPolicy.TLS_V1_2],
            customHeaders: {
                [CUSTOM_HTTP_HEADER]: this.cloudFrontHashHeader,
            },
            readTimeout: aws_cdk_lib_1.Duration.seconds(60),
            protocolPolicy: aws_cdk_lib_1.aws_cloudfront.OriginProtocolPolicy.HTTPS_ONLY,
        });
        this.distribution = new aws_cdk_lib_1.aws_cloudfront.Distribution(this, 'WorpdressDistribution', {
            comment: 'wordpress cdn',
            defaultBehavior: {
                origin,
                originRequestPolicy: new aws_cdk_lib_1.aws_cloudfront.OriginRequestPolicy(this, 'OriginRequestPolicy', {
                    originRequestPolicyName: 'WordpressDefaultBehavior',
                    cookieBehavior: aws_cdk_lib_1.aws_cloudfront.OriginRequestCookieBehavior.allowList('comment_*', 'wordpress_*', 'wp-settings-*'),
                    headerBehavior: aws_cdk_lib_1.aws_cloudfront.OriginRequestHeaderBehavior.allowList('Host', 'CloudFront-Forwarded-Proto', 'CloudFront-Is-Mobile-Viewer', 'CloudFront-Is-Tablet-Viewer', 'CloudFront-Is-Desktop-Viewer'),
                    queryStringBehavior: aws_cdk_lib_1.aws_cloudfront.OriginRequestQueryStringBehavior.all(),
                }),
                allowedMethods: aws_cdk_lib_1.aws_cloudfront.AllowedMethods.ALLOW_ALL,
                cachedMethods: aws_cdk_lib_1.aws_cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS,
                viewerProtocolPolicy: aws_cdk_lib_1.aws_cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            additionalBehaviors: {
                'wp-admin/*': {
                    origin,
                    originRequestPolicy: aws_cdk_lib_1.aws_cloudfront.OriginRequestPolicy.ALL_VIEWER,
                    allowedMethods: aws_cdk_lib_1.aws_cloudfront.AllowedMethods.ALLOW_ALL,
                    cachedMethods: aws_cdk_lib_1.aws_cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS,
                    viewerProtocolPolicy: aws_cdk_lib_1.aws_cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                },
                'wp-login.php': {
                    origin,
                    originRequestPolicy: aws_cdk_lib_1.aws_cloudfront.OriginRequestPolicy.ALL_VIEWER,
                    allowedMethods: aws_cdk_lib_1.aws_cloudfront.AllowedMethods.ALLOW_ALL,
                    cachedMethods: aws_cdk_lib_1.aws_cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS,
                    viewerProtocolPolicy: aws_cdk_lib_1.aws_cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                },
            },
            enableIpv6: true,
            httpVersion: aws_cdk_lib_1.aws_cloudfront.HttpVersion.HTTP2,
            certificate: props.certificate,
            domainNames: [this.domainName],
            priceClass: aws_cdk_lib_1.aws_cloudfront.PriceClass.PRICE_CLASS_100,
        });
        new dns_1.Dns(this, 'WordpressDns', {
            domainName: this.domainName,
            domainZone: this.domainZone,
            distribution: this.distribution,
        });
    }
    enableStaticContentOffload(domainName, certificate) {
        const bucket = new aws_cdk_lib_1.aws_s3.Bucket(this, 'Bucket', {
            encryption: aws_cdk_lib_1.aws_s3.BucketEncryption.S3_MANAGED,
            versioned: true,
            removalPolicy: this.removalPolicy,
            autoDeleteObjects: this.removalPolicy === aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        bucket.grantReadWrite(this.service.taskDefinition.taskRole);
        this.service.taskDefinition.taskRole.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: ['s3:GetBucketLocation'],
            resources: [bucket.bucketArn],
        }));
        const distribution = new aws_cdk_lib_1.aws_cloudfront.Distribution(this, 'StaticContentDistribution', {
            comment: 'static content cdn',
            defaultBehavior: {
                origin: new aws_cdk_lib_1.aws_cloudfront_origins.S3Origin(bucket),
                allowedMethods: aws_cdk_lib_1.aws_cloudfront.AllowedMethods.ALLOW_GET_HEAD,
                cachedMethods: aws_cdk_lib_1.aws_cloudfront.CachedMethods.CACHE_GET_HEAD,
                viewerProtocolPolicy: aws_cdk_lib_1.aws_cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            enableIpv6: true,
            httpVersion: aws_cdk_lib_1.aws_cloudfront.HttpVersion.HTTP2,
            certificate,
            domainNames: [domainName],
            priceClass: aws_cdk_lib_1.aws_cloudfront.PriceClass.PRICE_CLASS_100,
        });
        new dns_1.Dns(this, 'StaticContentDns', {
            domainName: domainName,
            domainZone: this.domainZone,
            distribution,
        });
        return {
            domainName,
            distribution,
        };
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@cloudcomponents/cdk-wordpress.Application", version: "2.0.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvYXBwbGljYXRpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw2Q0FjcUI7QUFDckIsMkNBQXVDO0FBR3ZDLCtCQUE0QjtBQW9CNUIsTUFBTSxrQkFBa0IsR0FBRywyQkFBMkIsQ0FBQztBQU92RCxNQUFhLFdBQVksU0FBUSxzQkFBUztJQVd4QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXVCOztRQUMvRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sS0FBSyxHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTdCLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztRQUNuQyxJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7UUFFbkMsSUFBSSxDQUFDLG9CQUFvQixTQUFHLEtBQUssQ0FBQyxvQkFBb0IsbUNBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLEtBQUssQ0FBQyxTQUFTLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ2xJLElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQztRQUV6QyxNQUFNLE9BQU8sR0FBRyxJQUFJLHFCQUFPLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDbkQsaUJBQWlCLEVBQUUsSUFBSTtZQUN2QixHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7U0FDZixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksd0NBQTBCLENBQUMsc0JBQXNCLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtZQUM1RixHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7WUFDZCxJQUFJLEVBQUUsRUFBRTtZQUNSLFVBQVUsRUFBRSx3Q0FBMEIsQ0FBQyxVQUFVLENBQUMsRUFBRTtZQUNwRCx3QkFBd0IsRUFBRSxzQkFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7U0FDM0MsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFdBQVcsQ0FBQyxvQkFBb0IsQ0FBQztZQUNwQyxJQUFJLEVBQUUsK0JBQStCO1lBQ3JDLFFBQVEsRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7U0FDOUIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxZQUFZLEdBQUcsSUFBSSx3Q0FBMEIsQ0FBQyx1QkFBdUIsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQ2hHLEdBQUcsRUFBRSxLQUFLLENBQUMsR0FBRztZQUNkLGNBQWMsRUFBRSxJQUFJO1lBQ3BCLFlBQVksRUFBRSxJQUFJO1NBQ25CLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxRQUFRLEdBQUcsWUFBWSxDQUFDLFdBQVcsQ0FBQyxVQUFVLEVBQUU7WUFDbkQsSUFBSSxFQUFFLEdBQUc7WUFDVCxRQUFRLEVBQUUsd0NBQTBCLENBQUMsbUJBQW1CLENBQUMsS0FBSztZQUM5RCxZQUFZLEVBQUU7Z0JBQ1osd0NBQTBCLENBQUMsbUJBQW1CLENBQUMsc0JBQXNCLENBQ25FLElBQUksb0NBQXNCLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUU7b0JBQzVELFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTtvQkFDM0IsVUFBVSxFQUFFLG9DQUFzQixDQUFDLHFCQUFxQixDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDO2lCQUNsRixDQUFDLENBQ0g7YUFDRjtZQUNELFNBQVMsRUFBRSx3Q0FBMEIsQ0FBQyxTQUFTLENBQUMscUJBQXFCO1lBQ3JFLGFBQWEsRUFBRSx3Q0FBMEIsQ0FBQyxjQUFjLENBQUMsYUFBYSxDQUFDLEdBQUcsRUFBRTtnQkFDMUUsV0FBVyxFQUFFLFlBQVk7Z0JBQ3pCLFdBQVcsRUFBRSxlQUFlO2FBQzdCLENBQUM7U0FDSCxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxZQUFZLEVBQUU7WUFDcEMsTUFBTSxFQUFFLHdDQUEwQixDQUFDLGNBQWMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7WUFDN0UsVUFBVSxFQUFFLENBQUMsd0NBQTBCLENBQUMsaUJBQWlCLENBQUMsVUFBVSxDQUFDLGtCQUFrQixFQUFFLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUMsQ0FBQztZQUN0SCxRQUFRLEVBQUUsR0FBRztTQUNkLENBQUMsQ0FBQztRQUVILE1BQU0sY0FBYyxHQUFHLElBQUkscUJBQU8sQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUU7WUFDL0UsY0FBYyxRQUFFLEtBQUssQ0FBQyxjQUFjLG1DQUFJLEdBQUc7WUFDM0MsR0FBRyxFQUFFLEdBQUc7WUFDUixPQUFPLEVBQUUsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDO1NBQ3hCLENBQUMsQ0FBQztRQUVILE1BQU0sU0FBUyxHQUFHLGNBQWMsQ0FBQyxZQUFZLENBQUMsV0FBVyxFQUFFO1lBQ3pELEtBQUssUUFBRSxLQUFLLENBQUMsS0FBSyxtQ0FBSSxxQkFBTyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsc0JBQXNCLENBQUM7WUFDakYsV0FBVyxFQUFFO2dCQUNYLEdBQUcsS0FBSyxDQUFDLFdBQVc7Z0JBQ3BCLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQyxXQUFXO2FBQzlCO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLEdBQUcsS0FBSyxDQUFDLE9BQU87Z0JBQ2hCLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQyxPQUFPO2FBQzFCO1lBQ0QsT0FBTyxRQUNMLEtBQUssQ0FBQyxTQUFTLG1DQUNmLHFCQUFPLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQztnQkFDeEIsWUFBWSxFQUFFLEdBQUcsS0FBSyxDQUFDLFNBQVMsdUJBQXVCO2dCQUN2RCxZQUFZLEVBQUUsc0JBQVEsQ0FBQyxhQUFhLENBQUMsU0FBUzthQUMvQyxDQUFDO1NBQ0wsQ0FBQyxDQUFDO1FBRUgsU0FBUyxDQUFDLGVBQWUsQ0FBQztZQUN4QixhQUFhLEVBQUUsRUFBRTtTQUNsQixDQUFDLENBQUM7UUFFSCxTQUFTLENBQUMsY0FBYyxDQUFDO1lBQ3ZCLGFBQWEsRUFBRSxlQUFlO1lBQzlCLFFBQVEsRUFBRSxLQUFLO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsSUFBSTtTQUNoQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUkscUJBQU8sQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUN6RCxPQUFPO1lBQ1AsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzlCLGNBQWM7WUFDZCxZQUFZLEVBQUUsQ0FBQztTQUNoQixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsWUFBWSxFQUFFLHFCQUFPLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ3ZFLElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUV6QyxNQUFNLE1BQU0sR0FBRyxJQUFJLG9DQUFzQixDQUFDLG9CQUFvQixDQUFDLFlBQVksRUFBRTtZQUMzRSxrQkFBa0IsRUFBRSxDQUFDLDRCQUFjLENBQUMsZUFBZSxDQUFDLFFBQVEsQ0FBQztZQUM3RCxhQUFhLEVBQUU7Z0JBQ2IsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFLElBQUksQ0FBQyxvQkFBb0I7YUFDaEQ7WUFDRCxXQUFXLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1lBQ2pDLGNBQWMsRUFBRSw0QkFBYyxDQUFDLG9CQUFvQixDQUFDLFVBQVU7U0FDL0QsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLDRCQUFjLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRTtZQUNqRixPQUFPLEVBQUUsZUFBZTtZQUN4QixlQUFlLEVBQUU7Z0JBQ2YsTUFBTTtnQkFDTixtQkFBbUIsRUFBRSxJQUFJLDRCQUFjLENBQUMsbUJBQW1CLENBQUMsSUFBSSxFQUFFLHFCQUFxQixFQUFFO29CQUN2Rix1QkFBdUIsRUFBRSwwQkFBMEI7b0JBQ25ELGNBQWMsRUFBRSw0QkFBYyxDQUFDLDJCQUEyQixDQUFDLFNBQVMsQ0FBQyxXQUFXLEVBQUUsYUFBYSxFQUFFLGVBQWUsQ0FBQztvQkFDakgsY0FBYyxFQUFFLDRCQUFjLENBQUMsMkJBQTJCLENBQUMsU0FBUyxDQUNsRSxNQUFNLEVBQ04sNEJBQTRCLEVBQzVCLDZCQUE2QixFQUM3Qiw2QkFBNkIsRUFDN0IsOEJBQThCLENBQy9CO29CQUNELG1CQUFtQixFQUFFLDRCQUFjLENBQUMsZ0NBQWdDLENBQUMsR0FBRyxFQUFFO2lCQUMzRSxDQUFDO2dCQUNGLGNBQWMsRUFBRSw0QkFBYyxDQUFDLGNBQWMsQ0FBQyxTQUFTO2dCQUN2RCxhQUFhLEVBQUUsNEJBQWMsQ0FBQyxhQUFhLENBQUMsc0JBQXNCO2dCQUNsRSxvQkFBb0IsRUFBRSw0QkFBYyxDQUFDLG9CQUFvQixDQUFDLGlCQUFpQjthQUM1RTtZQUNELG1CQUFtQixFQUFFO2dCQUNuQixZQUFZLEVBQUU7b0JBQ1osTUFBTTtvQkFDTixtQkFBbUIsRUFBRSw0QkFBYyxDQUFDLG1CQUFtQixDQUFDLFVBQVU7b0JBQ2xFLGNBQWMsRUFBRSw0QkFBYyxDQUFDLGNBQWMsQ0FBQyxTQUFTO29CQUN2RCxhQUFhLEVBQUUsNEJBQWMsQ0FBQyxhQUFhLENBQUMsc0JBQXNCO29CQUNsRSxvQkFBb0IsRUFBRSw0QkFBYyxDQUFDLG9CQUFvQixDQUFDLGlCQUFpQjtpQkFDNUU7Z0JBQ0QsY0FBYyxFQUFFO29CQUNkLE1BQU07b0JBQ04sbUJBQW1CLEVBQUUsNEJBQWMsQ0FBQyxtQkFBbUIsQ0FBQyxVQUFVO29CQUNsRSxjQUFjLEVBQUUsNEJBQWMsQ0FBQyxjQUFjLENBQUMsU0FBUztvQkFDdkQsYUFBYSxFQUFFLDRCQUFjLENBQUMsYUFBYSxDQUFDLHNCQUFzQjtvQkFDbEUsb0JBQW9CLEVBQUUsNEJBQWMsQ0FBQyxvQkFBb0IsQ0FBQyxpQkFBaUI7aUJBQzVFO2FBQ0Y7WUFDRCxVQUFVLEVBQUUsSUFBSTtZQUNoQixXQUFXLEVBQUUsNEJBQWMsQ0FBQyxXQUFXLENBQUMsS0FBSztZQUM3QyxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsV0FBVyxFQUFFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQztZQUM5QixVQUFVLEVBQUUsNEJBQWMsQ0FBQyxVQUFVLENBQUMsZUFBZTtTQUN0RCxDQUFDLENBQUM7UUFFSCxJQUFJLFNBQUcsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQzVCLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTtZQUMzQixVQUFVLEVBQUUsSUFBSSxDQUFDLFVBQVU7WUFDM0IsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO1NBQ2hDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTSwwQkFBMEIsQ0FBQyxVQUFrQixFQUFFLFdBQWdEO1FBQ3BHLE1BQU0sTUFBTSxHQUFHLElBQUksb0JBQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRTtZQUMvQyxVQUFVLEVBQUUsb0JBQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVO1lBQzlDLFNBQVMsRUFBRSxJQUFJO1lBQ2YsYUFBYSxFQUFFLElBQUksQ0FBQyxhQUFhO1lBQ2pDLGlCQUFpQixFQUFFLElBQUksQ0FBQyxhQUFhLEtBQUssMkJBQWEsQ0FBQyxPQUFPO1NBQ2hFLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFNUQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDLG9CQUFvQixDQUN2RCxJQUFJLHFCQUFPLENBQUMsZUFBZSxDQUFDO1lBQzFCLE1BQU0sRUFBRSxxQkFBTyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQzVCLE9BQU8sRUFBRSxDQUFDLHNCQUFzQixDQUFDO1lBQ2pDLFNBQVMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUM7U0FDOUIsQ0FBQyxDQUNILENBQUM7UUFFRixNQUFNLFlBQVksR0FBRyxJQUFJLDRCQUFjLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSwyQkFBMkIsRUFBRTtZQUN0RixPQUFPLEVBQUUsb0JBQW9CO1lBQzdCLGVBQWUsRUFBRTtnQkFDZixNQUFNLEVBQUUsSUFBSSxvQ0FBc0IsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDO2dCQUNuRCxjQUFjLEVBQUUsNEJBQWMsQ0FBQyxjQUFjLENBQUMsY0FBYztnQkFDNUQsYUFBYSxFQUFFLDRCQUFjLENBQUMsYUFBYSxDQUFDLGNBQWM7Z0JBQzFELG9CQUFvQixFQUFFLDRCQUFjLENBQUMsb0JBQW9CLENBQUMsaUJBQWlCO2FBQzVFO1lBQ0QsVUFBVSxFQUFFLElBQUk7WUFDaEIsV0FBVyxFQUFFLDRCQUFjLENBQUMsV0FBVyxDQUFDLEtBQUs7WUFDN0MsV0FBVztZQUNYLFdBQVcsRUFBRSxDQUFDLFVBQVUsQ0FBQztZQUN6QixVQUFVLEVBQUUsNEJBQWMsQ0FBQyxVQUFVLENBQUMsZUFBZTtTQUN0RCxDQUFDLENBQUM7UUFFSCxJQUFJLFNBQUcsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7WUFDaEMsVUFBVSxFQUFFLFVBQVU7WUFDdEIsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVO1lBQzNCLFlBQVk7U0FDYixDQUFDLENBQUM7UUFFSCxPQUFPO1lBQ0wsVUFBVTtZQUNWLFlBQVk7U0FDYixDQUFDO0lBQ0osQ0FBQzs7QUF2Tkgsa0NBd05DIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHtcbiAgRHVyYXRpb24sXG4gIFJlbW92YWxQb2xpY3ksXG4gIFN0YWNrLFxuICBhd3NfY2VydGlmaWNhdGVtYW5hZ2VyLFxuICBhd3NfY2xvdWRmcm9udCxcbiAgYXdzX2Nsb3VkZnJvbnRfb3JpZ2lucyxcbiAgYXdzX2VjMixcbiAgYXdzX2VjcyxcbiAgYXdzX2VsYXN0aWNsb2FkYmFsYW5jaW5ndjIsXG4gIGF3c19pYW0sXG4gIGF3c19sb2dzLFxuICBhd3Nfcm91dGU1MyxcbiAgYXdzX3MzLFxufSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuaW1wb3J0IHsgRGF0YWJhc2UgfSBmcm9tICcuL2RhdGFiYXNlJztcbmltcG9ydCB7IERucyB9IGZyb20gJy4vZG5zJztcbmltcG9ydCB7IEVmc1ZvbHVtZSB9IGZyb20gJy4vZWZzLXZvbHVtZSc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgQXBwbGljYXRpb25Qcm9wcyB7XG4gIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcbiAgcmVhZG9ubHkgZG9tYWluWm9uZTogYXdzX3JvdXRlNTMuSUhvc3RlZFpvbmU7XG4gIHJlYWRvbmx5IGNlcnRpZmljYXRlOiBhd3NfY2VydGlmaWNhdGVtYW5hZ2VyLklDZXJ0aWZpY2F0ZTtcbiAgcmVhZG9ubHkgdnBjOiBhd3NfZWMyLklWcGM7XG4gIHJlYWRvbmx5IHZvbHVtZTogRWZzVm9sdW1lO1xuICByZWFkb25seSBkYXRhYmFzZTogRGF0YWJhc2U7XG4gIHJlYWRvbmx5IGltYWdlPzogYXdzX2Vjcy5Db250YWluZXJJbWFnZTtcbiAgcmVhZG9ubHkgc2VydmljZU5hbWU/OiBzdHJpbmc7XG4gIHJlYWRvbmx5IG1lbW9yeUxpbWl0TWlCPzogbnVtYmVyO1xuICByZWFkb25seSBlbnZpcm9ubWVudD86IFJlY29yZDxzdHJpbmcsIHN0cmluZz47XG4gIHJlYWRvbmx5IHNlY3JldHM/OiBSZWNvcmQ8c3RyaW5nLCBhd3NfZWNzLlNlY3JldD47XG4gIHJlYWRvbmx5IGxvZ0RyaXZlcj86IGF3c19lY3MuTG9nRHJpdmVyO1xuICByZWFkb25seSBjbG91ZEZyb250SGFzaEhlYWRlcj86IHN0cmluZztcbiAgcmVhZG9ubHkgcmVtb3ZhbFBvbGljeT86IFJlbW92YWxQb2xpY3k7XG59XG5cbmNvbnN0IENVU1RPTV9IVFRQX0hFQURFUiA9ICdYX1JlcXVlc3RfRnJvbV9DbG91ZEZyb250JztcblxuZXhwb3J0IGludGVyZmFjZSBTdGF0aWNDb250ZW50T2ZmbG9hZCB7XG4gIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcbiAgcmVhZG9ubHkgZGlzdHJpYnV0aW9uOiBhd3NfY2xvdWRmcm9udC5JRGlzdHJpYnV0aW9uO1xufVxuXG5leHBvcnQgY2xhc3MgQXBwbGljYXRpb24gZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICBwdWJsaWMgcmVhZG9ubHkgZG9tYWluTmFtZTogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgZG9tYWluWm9uZTogYXdzX3JvdXRlNTMuSUhvc3RlZFpvbmU7XG4gIHB1YmxpYyByZWFkb25seSBkaXN0cmlidXRpb246IGF3c19jbG91ZGZyb250LklEaXN0cmlidXRpb247XG4gIHB1YmxpYyByZWFkb25seSBzZXJ2aWNlOiBhd3NfZWNzLkZhcmdhdGVTZXJ2aWNlO1xuICBwdWJsaWMgcmVhZG9ubHkgdGFyZ2V0R3JvdXA6IGF3c19lbGFzdGljbG9hZGJhbGFuY2luZ3YyLkFwcGxpY2F0aW9uVGFyZ2V0R3JvdXA7XG4gIHB1YmxpYyByZWFkb25seSBsaXN0ZW5lcjogYXdzX2VsYXN0aWNsb2FkYmFsYW5jaW5ndjIuQXBwbGljYXRpb25MaXN0ZW5lcjtcblxuICBwcml2YXRlIHJlYWRvbmx5IGNsb3VkRnJvbnRIYXNoSGVhZGVyOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgcmVtb3ZhbFBvbGljeT86IFJlbW92YWxQb2xpY3k7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEFwcGxpY2F0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZih0aGlzKTtcblxuICAgIHRoaXMuZG9tYWluTmFtZSA9IHByb3BzLmRvbWFpbk5hbWU7XG4gICAgdGhpcy5kb21haW5ab25lID0gcHJvcHMuZG9tYWluWm9uZTtcblxuICAgIHRoaXMuY2xvdWRGcm9udEhhc2hIZWFkZXIgPSBwcm9wcy5jbG91ZEZyb250SGFzaEhlYWRlciA/PyBCdWZmZXIuZnJvbShgJHtzdGFjay5zdGFja05hbWV9LiR7dGhpcy5kb21haW5OYW1lfWApLnRvU3RyaW5nKCdiYXNlNjQnKTtcbiAgICB0aGlzLnJlbW92YWxQb2xpY3kgPSBwcm9wcy5yZW1vdmFsUG9saWN5O1xuXG4gICAgY29uc3QgY2x1c3RlciA9IG5ldyBhd3NfZWNzLkNsdXN0ZXIodGhpcywgJ0NsdXN0ZXInLCB7XG4gICAgICBjb250YWluZXJJbnNpZ2h0czogdHJ1ZSxcbiAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgIH0pO1xuXG4gICAgdGhpcy50YXJnZXRHcm91cCA9IG5ldyBhd3NfZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mi5BcHBsaWNhdGlvblRhcmdldEdyb3VwKHRoaXMsICdUYXJnZXRHcm91cCcsIHtcbiAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgICAgcG9ydDogODAsXG4gICAgICB0YXJnZXRUeXBlOiBhd3NfZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mi5UYXJnZXRUeXBlLklQLFxuICAgICAgc3RpY2tpbmVzc0Nvb2tpZUR1cmF0aW9uOiBEdXJhdGlvbi5kYXlzKDcpLFxuICAgIH0pO1xuXG4gICAgdGhpcy50YXJnZXRHcm91cC5jb25maWd1cmVIZWFsdGhDaGVjayh7XG4gICAgICBwYXRoOiAnL3dwLWluY2x1ZGVzL2ltYWdlcy9ibGFuay5naWYnLFxuICAgICAgaW50ZXJ2YWw6IER1cmF0aW9uLm1pbnV0ZXMoMSksXG4gICAgfSk7XG5cbiAgICBjb25zdCBsb2FkQmFsYW5jZXIgPSBuZXcgYXdzX2VsYXN0aWNsb2FkYmFsYW5jaW5ndjIuQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIodGhpcywgJ0xvYWRiYWxhbmNlcicsIHtcbiAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgICAgaW50ZXJuZXRGYWNpbmc6IHRydWUsXG4gICAgICBodHRwMkVuYWJsZWQ6IHRydWUsXG4gICAgfSk7XG5cbiAgICB0aGlzLmxpc3RlbmVyID0gbG9hZEJhbGFuY2VyLmFkZExpc3RlbmVyKCdMaXN0ZW5lcicsIHtcbiAgICAgIHBvcnQ6IDQ0MyxcbiAgICAgIHByb3RvY29sOiBhd3NfZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mi5BcHBsaWNhdGlvblByb3RvY29sLkhUVFBTLFxuICAgICAgY2VydGlmaWNhdGVzOiBbXG4gICAgICAgIGF3c19lbGFzdGljbG9hZGJhbGFuY2luZ3YyLkxpc3RlbmVyQ2VydGlmaWNhdGUuZnJvbUNlcnRpZmljYXRlTWFuYWdlcihcbiAgICAgICAgICBuZXcgYXdzX2NlcnRpZmljYXRlbWFuYWdlci5DZXJ0aWZpY2F0ZSh0aGlzLCAnTEJDZXJ0aWZpY2F0ZScsIHtcbiAgICAgICAgICAgIGRvbWFpbk5hbWU6IHRoaXMuZG9tYWluTmFtZSxcbiAgICAgICAgICAgIHZhbGlkYXRpb246IGF3c19jZXJ0aWZpY2F0ZW1hbmFnZXIuQ2VydGlmaWNhdGVWYWxpZGF0aW9uLmZyb21EbnModGhpcy5kb21haW5ab25lKSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgKSxcbiAgICAgIF0sXG4gICAgICBzc2xQb2xpY3k6IGF3c19lbGFzdGljbG9hZGJhbGFuY2luZ3YyLlNzbFBvbGljeS5GT1JXQVJEX1NFQ1JFQ1lfVExTMTIsXG4gICAgICBkZWZhdWx0QWN0aW9uOiBhd3NfZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mi5MaXN0ZW5lckFjdGlvbi5maXhlZFJlc3BvbnNlKDQwMywge1xuICAgICAgICBjb250ZW50VHlwZTogJ3RleHQvcGxhaW4nLFxuICAgICAgICBtZXNzYWdlQm9keTogJ0FjY2VzcyBkZW5pZWQnLFxuICAgICAgfSksXG4gICAgfSk7XG5cbiAgICB0aGlzLmxpc3RlbmVyLmFkZEFjdGlvbignQ2xvdWRmb3JudCcsIHtcbiAgICAgIGFjdGlvbjogYXdzX2VsYXN0aWNsb2FkYmFsYW5jaW5ndjIuTGlzdGVuZXJBY3Rpb24uZm9yd2FyZChbdGhpcy50YXJnZXRHcm91cF0pLFxuICAgICAgY29uZGl0aW9uczogW2F3c19lbGFzdGljbG9hZGJhbGFuY2luZ3YyLkxpc3RlbmVyQ29uZGl0aW9uLmh0dHBIZWFkZXIoQ1VTVE9NX0hUVFBfSEVBREVSLCBbdGhpcy5jbG91ZEZyb250SGFzaEhlYWRlcl0pXSxcbiAgICAgIHByaW9yaXR5OiAxMDAsXG4gICAgfSk7XG5cbiAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBhd3NfZWNzLkZhcmdhdGVUYXNrRGVmaW5pdGlvbih0aGlzLCAnVGFza0RlZmluaXRpb24nLCB7XG4gICAgICBtZW1vcnlMaW1pdE1pQjogcHJvcHMubWVtb3J5TGltaXRNaUIgPz8gNTEyLFxuICAgICAgY3B1OiAyNTYsXG4gICAgICB2b2x1bWVzOiBbcHJvcHMudm9sdW1lXSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGNvbnRhaW5lciA9IHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcignV29yZHByZXNzJywge1xuICAgICAgaW1hZ2U6IHByb3BzLmltYWdlID8/IGF3c19lY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KCd3b3JkcHJlc3M6NS44LWFwYWNoZScpLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgLi4ucHJvcHMuZW52aXJvbm1lbnQsXG4gICAgICAgIC4uLnByb3BzLmRhdGFiYXNlLmVudmlyb25tZW50LFxuICAgICAgfSxcbiAgICAgIHNlY3JldHM6IHtcbiAgICAgICAgLi4ucHJvcHMuc2VjcmV0cyxcbiAgICAgICAgLi4ucHJvcHMuZGF0YWJhc2Uuc2VjcmV0cyxcbiAgICAgIH0sXG4gICAgICBsb2dnaW5nOlxuICAgICAgICBwcm9wcy5sb2dEcml2ZXIgPz9cbiAgICAgICAgYXdzX2Vjcy5Mb2dEcml2ZXIuYXdzTG9ncyh7XG4gICAgICAgICAgc3RyZWFtUHJlZml4OiBgJHtzdGFjay5zdGFja05hbWV9V29yZHByZXNzQ29udGFpbmVyTG9nYCxcbiAgICAgICAgICBsb2dSZXRlbnRpb246IGF3c19sb2dzLlJldGVudGlvbkRheXMuT05FX01PTlRILFxuICAgICAgICB9KSxcbiAgICB9KTtcblxuICAgIGNvbnRhaW5lci5hZGRQb3J0TWFwcGluZ3Moe1xuICAgICAgY29udGFpbmVyUG9ydDogODAsXG4gICAgfSk7XG5cbiAgICBjb250YWluZXIuYWRkTW91bnRQb2ludHMoe1xuICAgICAgY29udGFpbmVyUGF0aDogJy92YXIvd3d3L2h0bWwnLFxuICAgICAgcmVhZE9ubHk6IGZhbHNlLFxuICAgICAgc291cmNlVm9sdW1lOiBwcm9wcy52b2x1bWUubmFtZSxcbiAgICB9KTtcblxuICAgIHRoaXMuc2VydmljZSA9IG5ldyBhd3NfZWNzLkZhcmdhdGVTZXJ2aWNlKHRoaXMsICdTZXJ2aWNlJywge1xuICAgICAgY2x1c3RlcixcbiAgICAgIHNlcnZpY2VOYW1lOiBwcm9wcy5zZXJ2aWNlTmFtZSxcbiAgICAgIHRhc2tEZWZpbml0aW9uLFxuICAgICAgZGVzaXJlZENvdW50OiAyLFxuICAgIH0pO1xuXG4gICAgdGhpcy5zZXJ2aWNlLmNvbm5lY3Rpb25zLmFsbG93RnJvbShsb2FkQmFsYW5jZXIsIGF3c19lYzIuUG9ydC50Y3AoODApKTtcbiAgICB0aGlzLnRhcmdldEdyb3VwLmFkZFRhcmdldCh0aGlzLnNlcnZpY2UpO1xuXG4gICAgY29uc3Qgb3JpZ2luID0gbmV3IGF3c19jbG91ZGZyb250X29yaWdpbnMuTG9hZEJhbGFuY2VyVjJPcmlnaW4obG9hZEJhbGFuY2VyLCB7XG4gICAgICBvcmlnaW5Tc2xQcm90b2NvbHM6IFthd3NfY2xvdWRmcm9udC5PcmlnaW5Tc2xQb2xpY3kuVExTX1YxXzJdLFxuICAgICAgY3VzdG9tSGVhZGVyczoge1xuICAgICAgICBbQ1VTVE9NX0hUVFBfSEVBREVSXTogdGhpcy5jbG91ZEZyb250SGFzaEhlYWRlcixcbiAgICAgIH0sXG4gICAgICByZWFkVGltZW91dDogRHVyYXRpb24uc2Vjb25kcyg2MCksXG4gICAgICBwcm90b2NvbFBvbGljeTogYXdzX2Nsb3VkZnJvbnQuT3JpZ2luUHJvdG9jb2xQb2xpY3kuSFRUUFNfT05MWSxcbiAgICB9KTtcblxuICAgIHRoaXMuZGlzdHJpYnV0aW9uID0gbmV3IGF3c19jbG91ZGZyb250LkRpc3RyaWJ1dGlvbih0aGlzLCAnV29ycGRyZXNzRGlzdHJpYnV0aW9uJywge1xuICAgICAgY29tbWVudDogJ3dvcmRwcmVzcyBjZG4nLFxuICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7XG4gICAgICAgIG9yaWdpbixcbiAgICAgICAgb3JpZ2luUmVxdWVzdFBvbGljeTogbmV3IGF3c19jbG91ZGZyb250Lk9yaWdpblJlcXVlc3RQb2xpY3kodGhpcywgJ09yaWdpblJlcXVlc3RQb2xpY3knLCB7XG4gICAgICAgICAgb3JpZ2luUmVxdWVzdFBvbGljeU5hbWU6ICdXb3JkcHJlc3NEZWZhdWx0QmVoYXZpb3InLFxuICAgICAgICAgIGNvb2tpZUJlaGF2aW9yOiBhd3NfY2xvdWRmcm9udC5PcmlnaW5SZXF1ZXN0Q29va2llQmVoYXZpb3IuYWxsb3dMaXN0KCdjb21tZW50XyonLCAnd29yZHByZXNzXyonLCAnd3Atc2V0dGluZ3MtKicpLFxuICAgICAgICAgIGhlYWRlckJlaGF2aW9yOiBhd3NfY2xvdWRmcm9udC5PcmlnaW5SZXF1ZXN0SGVhZGVyQmVoYXZpb3IuYWxsb3dMaXN0KFxuICAgICAgICAgICAgJ0hvc3QnLFxuICAgICAgICAgICAgJ0Nsb3VkRnJvbnQtRm9yd2FyZGVkLVByb3RvJyxcbiAgICAgICAgICAgICdDbG91ZEZyb250LUlzLU1vYmlsZS1WaWV3ZXInLFxuICAgICAgICAgICAgJ0Nsb3VkRnJvbnQtSXMtVGFibGV0LVZpZXdlcicsXG4gICAgICAgICAgICAnQ2xvdWRGcm9udC1Jcy1EZXNrdG9wLVZpZXdlcicsXG4gICAgICAgICAgKSxcbiAgICAgICAgICBxdWVyeVN0cmluZ0JlaGF2aW9yOiBhd3NfY2xvdWRmcm9udC5PcmlnaW5SZXF1ZXN0UXVlcnlTdHJpbmdCZWhhdmlvci5hbGwoKSxcbiAgICAgICAgfSksXG4gICAgICAgIGFsbG93ZWRNZXRob2RzOiBhd3NfY2xvdWRmcm9udC5BbGxvd2VkTWV0aG9kcy5BTExPV19BTEwsXG4gICAgICAgIGNhY2hlZE1ldGhvZHM6IGF3c19jbG91ZGZyb250LkNhY2hlZE1ldGhvZHMuQ0FDSEVfR0VUX0hFQURfT1BUSU9OUyxcbiAgICAgICAgdmlld2VyUHJvdG9jb2xQb2xpY3k6IGF3c19jbG91ZGZyb250LlZpZXdlclByb3RvY29sUG9saWN5LlJFRElSRUNUX1RPX0hUVFBTLFxuICAgICAgfSxcbiAgICAgIGFkZGl0aW9uYWxCZWhhdmlvcnM6IHtcbiAgICAgICAgJ3dwLWFkbWluLyonOiB7XG4gICAgICAgICAgb3JpZ2luLFxuICAgICAgICAgIG9yaWdpblJlcXVlc3RQb2xpY3k6IGF3c19jbG91ZGZyb250Lk9yaWdpblJlcXVlc3RQb2xpY3kuQUxMX1ZJRVdFUixcbiAgICAgICAgICBhbGxvd2VkTWV0aG9kczogYXdzX2Nsb3VkZnJvbnQuQWxsb3dlZE1ldGhvZHMuQUxMT1dfQUxMLFxuICAgICAgICAgIGNhY2hlZE1ldGhvZHM6IGF3c19jbG91ZGZyb250LkNhY2hlZE1ldGhvZHMuQ0FDSEVfR0VUX0hFQURfT1BUSU9OUyxcbiAgICAgICAgICB2aWV3ZXJQcm90b2NvbFBvbGljeTogYXdzX2Nsb3VkZnJvbnQuVmlld2VyUHJvdG9jb2xQb2xpY3kuUkVESVJFQ1RfVE9fSFRUUFMsXG4gICAgICAgIH0sXG4gICAgICAgICd3cC1sb2dpbi5waHAnOiB7XG4gICAgICAgICAgb3JpZ2luLFxuICAgICAgICAgIG9yaWdpblJlcXVlc3RQb2xpY3k6IGF3c19jbG91ZGZyb250Lk9yaWdpblJlcXVlc3RQb2xpY3kuQUxMX1ZJRVdFUixcbiAgICAgICAgICBhbGxvd2VkTWV0aG9kczogYXdzX2Nsb3VkZnJvbnQuQWxsb3dlZE1ldGhvZHMuQUxMT1dfQUxMLFxuICAgICAgICAgIGNhY2hlZE1ldGhvZHM6IGF3c19jbG91ZGZyb250LkNhY2hlZE1ldGhvZHMuQ0FDSEVfR0VUX0hFQURfT1BUSU9OUyxcbiAgICAgICAgICB2aWV3ZXJQcm90b2NvbFBvbGljeTogYXdzX2Nsb3VkZnJvbnQuVmlld2VyUHJvdG9jb2xQb2xpY3kuUkVESVJFQ1RfVE9fSFRUUFMsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgZW5hYmxlSXB2NjogdHJ1ZSxcbiAgICAgIGh0dHBWZXJzaW9uOiBhd3NfY2xvdWRmcm9udC5IdHRwVmVyc2lvbi5IVFRQMixcbiAgICAgIGNlcnRpZmljYXRlOiBwcm9wcy5jZXJ0aWZpY2F0ZSxcbiAgICAgIGRvbWFpbk5hbWVzOiBbdGhpcy5kb21haW5OYW1lXSxcbiAgICAgIHByaWNlQ2xhc3M6IGF3c19jbG91ZGZyb250LlByaWNlQ2xhc3MuUFJJQ0VfQ0xBU1NfMTAwLFxuICAgIH0pO1xuXG4gICAgbmV3IERucyh0aGlzLCAnV29yZHByZXNzRG5zJywge1xuICAgICAgZG9tYWluTmFtZTogdGhpcy5kb21haW5OYW1lLFxuICAgICAgZG9tYWluWm9uZTogdGhpcy5kb21haW5ab25lLFxuICAgICAgZGlzdHJpYnV0aW9uOiB0aGlzLmRpc3RyaWJ1dGlvbixcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBlbmFibGVTdGF0aWNDb250ZW50T2ZmbG9hZChkb21haW5OYW1lOiBzdHJpbmcsIGNlcnRpZmljYXRlOiBhd3NfY2VydGlmaWNhdGVtYW5hZ2VyLklDZXJ0aWZpY2F0ZSk6IFN0YXRpY0NvbnRlbnRPZmZsb2FkIHtcbiAgICBjb25zdCBidWNrZXQgPSBuZXcgYXdzX3MzLkJ1Y2tldCh0aGlzLCAnQnVja2V0Jywge1xuICAgICAgZW5jcnlwdGlvbjogYXdzX3MzLkJ1Y2tldEVuY3J5cHRpb24uUzNfTUFOQUdFRCxcbiAgICAgIHZlcnNpb25lZDogdHJ1ZSxcbiAgICAgIHJlbW92YWxQb2xpY3k6IHRoaXMucmVtb3ZhbFBvbGljeSxcbiAgICAgIGF1dG9EZWxldGVPYmplY3RzOiB0aGlzLnJlbW92YWxQb2xpY3kgPT09IFJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICB9KTtcblxuICAgIGJ1Y2tldC5ncmFudFJlYWRXcml0ZSh0aGlzLnNlcnZpY2UudGFza0RlZmluaXRpb24udGFza1JvbGUpO1xuXG4gICAgdGhpcy5zZXJ2aWNlLnRhc2tEZWZpbml0aW9uLnRhc2tSb2xlLmFkZFRvUHJpbmNpcGFsUG9saWN5KFxuICAgICAgbmV3IGF3c19pYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgZWZmZWN0OiBhd3NfaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydzMzpHZXRCdWNrZXRMb2NhdGlvbiddLFxuICAgICAgICByZXNvdXJjZXM6IFtidWNrZXQuYnVja2V0QXJuXSxcbiAgICAgIH0pLFxuICAgICk7XG5cbiAgICBjb25zdCBkaXN0cmlidXRpb24gPSBuZXcgYXdzX2Nsb3VkZnJvbnQuRGlzdHJpYnV0aW9uKHRoaXMsICdTdGF0aWNDb250ZW50RGlzdHJpYnV0aW9uJywge1xuICAgICAgY29tbWVudDogJ3N0YXRpYyBjb250ZW50IGNkbicsXG4gICAgICBkZWZhdWx0QmVoYXZpb3I6IHtcbiAgICAgICAgb3JpZ2luOiBuZXcgYXdzX2Nsb3VkZnJvbnRfb3JpZ2lucy5TM09yaWdpbihidWNrZXQpLFxuICAgICAgICBhbGxvd2VkTWV0aG9kczogYXdzX2Nsb3VkZnJvbnQuQWxsb3dlZE1ldGhvZHMuQUxMT1dfR0VUX0hFQUQsXG4gICAgICAgIGNhY2hlZE1ldGhvZHM6IGF3c19jbG91ZGZyb250LkNhY2hlZE1ldGhvZHMuQ0FDSEVfR0VUX0hFQUQsXG4gICAgICAgIHZpZXdlclByb3RvY29sUG9saWN5OiBhd3NfY2xvdWRmcm9udC5WaWV3ZXJQcm90b2NvbFBvbGljeS5SRURJUkVDVF9UT19IVFRQUyxcbiAgICAgIH0sXG4gICAgICBlbmFibGVJcHY2OiB0cnVlLFxuICAgICAgaHR0cFZlcnNpb246IGF3c19jbG91ZGZyb250Lkh0dHBWZXJzaW9uLkhUVFAyLFxuICAgICAgY2VydGlmaWNhdGUsXG4gICAgICBkb21haW5OYW1lczogW2RvbWFpbk5hbWVdLFxuICAgICAgcHJpY2VDbGFzczogYXdzX2Nsb3VkZnJvbnQuUHJpY2VDbGFzcy5QUklDRV9DTEFTU18xMDAsXG4gICAgfSk7XG5cbiAgICBuZXcgRG5zKHRoaXMsICdTdGF0aWNDb250ZW50RG5zJywge1xuICAgICAgZG9tYWluTmFtZTogZG9tYWluTmFtZSxcbiAgICAgIGRvbWFpblpvbmU6IHRoaXMuZG9tYWluWm9uZSxcbiAgICAgIGRpc3RyaWJ1dGlvbixcbiAgICB9KTtcblxuICAgIHJldHVybiB7XG4gICAgICBkb21haW5OYW1lLFxuICAgICAgZGlzdHJpYnV0aW9uLFxuICAgIH07XG4gIH1cbn1cbiJdfQ==