[![tests](https://github.com/dayyass/rllib/actions/workflows/tests.yml/badge.svg)](https://github.com/dayyass/rllib/actions/workflows/tests.yml)
[![linter](https://github.com/dayyass/rllib/actions/workflows/linter.yml/badge.svg)](https://github.com/dayyass/rllib/actions/workflows/linter.yml)
[![codecov](https://codecov.io/gh/dayyass/rllib/branch/main/graph/badge.svg?token=45O5NRAD8G)](https://codecov.io/gh/dayyass/rllib)

[![python 3.7](https://img.shields.io/badge/python-3.7-blue.svg)](https://github.com/dayyass/rllib#requirements)
[![release (latest by date)](https://img.shields.io/github/v/release/dayyass/rllib)](https://github.com/dayyass/rllib/releases/latest)
[![license](https://img.shields.io/github/license/dayyass/rllib?color=blue)](https://github.com/dayyass/rllib/blob/main/LICENSE)

[![pre-commit](https://img.shields.io/badge/pre--commit-enabled-black)](https://github.com/dayyass/rllib/blob/main/.pre-commit-config.yaml)
[![code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)

[![pypi version](https://img.shields.io/pypi/v/pytorch-rllib)](https://pypi.org/project/pytorch-rllib)
[![pypi downloads](https://img.shields.io/pypi/dm/pytorch-rllib)](https://pypi.org/project/pytorch-rllib)

# rllib
Reinforcement Learning Library

## Installation
```
pip install pytorch-rllib
```

## Usage
Implemented agents:
- [ ] CrossEntropy
- [ ] Value / Policy Iteration
- [x] Q-Learning
- [x] Expected Value SARSA
- [x] Approximate Q-Learning
- [x] DQN
- [ ] Rainbow
- [ ] REINFORCE
- [ ] A2C

```python3
import gym
import numpy as np
import torch

from rllib.qlearning import ApproximateQLearningAgent
from rllib.trainer import TrainerTorch as Trainer
from rllib.utils import set_global_seed

device = torch.device("cuda" if torch.cuda.is_available() else "cpu")

# init environment
env = gym.make("CartPole-v0")
set_global_seed(seed=42, env=env)

n_actions = env.action_space.n
n_state = env.observation_space.shape[0]

# init torch model
model = torch.nn.Sequential()
model.add_module("layer1", torch.nn.Linear(n_state, 128))
model.add_module("relu1", torch.nn.ReLU())
model.add_module("layer2", torch.nn.Linear(128, 64))
model.add_module("relu2", torch.nn.ReLU())
model.add_module("values", torch.nn.Linear(64, n_actions))
model = model.to(device)

# init agent
agent = ApproximateQLearningAgent(
    model=model,
    alpha=0.5,
    epsilon=0.5,
    discount=0.99,
    n_actions=n_actions,
)

# train
optimizer = torch.optim.Adam(model.parameters(), lr=1e-4)

trainer = Trainer(env=env)

train_rewards = trainer.train(
    agent=agent,
    optimizer=optimizer,
    n_epochs=20,
    n_sessions=100,
)

# train results
print(f"Mean train reward: {np.mean(train_rewards[-10:])}")  # reward: 120.318

# inference
inference_reward = trainer.play_session(
    agent=agent,
    t_max=10**4,
)

# inference results
print(f"Inference reward: {inference_reward}")  # reward: 171.0
```

More examples you can find [here](https://github.com/dayyass/rllib/tree/main/examples).

## Requirements
Python >= 3.7

## Citation
If you use **rllib** in a scientific publication, we would appreciate references to the following BibTex entry:
```bibtex
@misc{dayyass2022rllib,
    author       = {El-Ayyass, Dani},
    title        = {Reinforcement Learning Library},
    howpublished = {\url{https://github.com/dayyass/rllib}},
    year         = {2022}
}
```
