# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['torcheck', 'torcheck.utils']

package_data = \
{'': ['*']}

install_requires = \
['torch>=1.8,<2.0']

setup_kwargs = {
    'name': 'torcheck',
    'version': '1.0.1',
    'description': 'A machine learning sanity check toolkit for PyTorch',
    'long_description': '# torcheck\n[![Build Status](https://travis-ci.com/pengyan510/torcheck.svg?branch=master)](https://travis-ci.com/pengyan510/torcheck)\n[![License](https://img.shields.io/badge/License-MIT-green.svg)](https://opensource.org/licenses/MIT)\n[![codecov](https://codecov.io/gh/pengyan510/torcheck/branch/master/graph/badge.svg?token=Q8ADT16N8A)](https://codecov.io/gh/pengyan510/torcheck)\n[![PyPI version](https://badge.fury.io/py/torcheck.svg)](https://badge.fury.io/py/torcheck)\n\nTorcheck is a machine learning sanity check toolkit for PyTorch.\n\nFor a general introduction, please check this out: [Testing Your PyTorch Models with\xa0Torcheck](https://towardsdatascience.com/testing-your-pytorch-models-with-torcheck-cb689ecbc08c)\n\n## About\nThe creation of torcheck is inspired by Chase Roberts\' [Medium post](https://thenerdstation.medium.com/mltest-automatically-test-neural-network-models-in-one-function-call-eb6f1fa5019d). The innovation and major benefit is that you no longer\nneed to write additional testing code for your model training. Just add a few\nlines of code specifying the checks before training, torcheck will then take over and\nperform the checks simultaneouly while the training happens.\n\nAnother benefit is that torcheck allows you to check your model on different levels.\nInstead of checking the whole model, you can specify checks for a submodule, a linear\nlayer, or even the weight tensor! This enables more customization around the sanity\nchecks.\n\n## Installation\n```\npip install torcheck\n```\n\n## Torcheck in 5 minutes\nOK, suppose you have coded up a standard PyTorch training routine like this:\n```\nmodel = Model()\noptimizer = torch.optim.Adam(\n    model.parameters(),\n    lr=0.001,\n)\n\n# torcheck code goes here\n\nfor epoch in range(num_epochs):\n    for x, y in dataloader:\n        # calculate loss and backward propagation\n```\n\nBy simply adding a few lines of code right before the for loop, you can be more confident\nabout whether your model is training as expected!\n\n### Step 1: Registering your optimizer(s)\nFirst, register the optimizer(s) with torcheck:\n```\ntorcheck.register(optimizer)\n```\n\n### Step 2: Adding sanity checks\nTorcheck enables you to perform a wide range of checks, on both module level and tensor\nlevel.\n\nA rule of thumb is that use APIs with `add_module` prefix when checking something that\nsubclasses from `nn.Module`, use APIs with `add_tensor` prefix when checking tensors.\n\n#### Parameters change/not change\nYou can check whether model parameters actually get updated during the training.\nOr you can check whether they remain constant if you want them to be frozen.\n\nFor our example, some of the possible checks are:\n\n```\n# check all the model parameters will change\n# module_name is optional, but it makes error messages more informative when checks fail\ntorcheck.add_module_changing_check(model, module_name="my_model")\n```\n\n```\n# check the linear layer\'s parameters won\'t change\ntorcheck.add_module_unchanging_check(model.linear_0, module_name="linear_layer_0")\n```\n\n```\n# check the linear layer\'s weight parameters will change\ntorcheck.add_tensor_changing_check(\n    model.linear_0.weight, tensor_name="linear_0.weight", module_name="my_model"\n)\n```\n\n```\n# check the linear layer\'s bias parameters won\'t change\ntorcheck.add_tensor_unchanging_check(\n    model.linear_0.bias, tensor_name="linear_0.bias", module_name="my_model"\n)\n```\n\n#### Output range check\nThe basic use case is that you can check whether model outputs are all within a range,\nsay (-1, 1).\n\nYou can also check that model outputs are not all within a range. This is useful when\nyou want softmax to behave correctly. It enables you to check model ouputs are not all\nwithin (0, 1).\n\nYou can check the final model output or intermediate output of a submodule.\n```\n# check model outputs are within (-1, 1)\ntorcheck.add_module_output_range_check(\n    model, output_range=(-1, 1), module_name="my_model"\n)\n```\n\n```\n# check outputs from the linear layer are within (-5, 5)\ntorcheck.add_module_output_range_check(\n    model.linear_0, output_range=(-5, 5), module_name="linear_layer_0"\n)\n\n```\n\n```\n# check model outputs are not all within (0, 1)\n# aka softmax hasn\'t been applied before loss calculation\ntorcheck.add_module_output_range_check(\n    model,\n    output_range=(0, 1),\n    negate_range=True,\n    module_name="my_model",\n)\n```\n\n#### NaN check\nCheck whether parameters become NaN during training, or model outputs contain NaN.\n\n```\n# check whether model parameters become NaN or outputs contain NaN\ntorcheck.add_module_nan_check(model, module_name="my_model")\n```\n\n```\n# check whether linear layer\'s weight parameters become NaN\ntorcheck.add_tensor_nan_check(\n    model.linear_0.weight, tensor_name="linear_0.weight", module_name="my_model"\n)\n```\n\n#### Inf check\nCheck whether parameters become infinite (positive or negative infinity) during training,\nor model outputs contain infinite value.\n\n```\n# check whether model parameters become infinite or outputs contain infinite value\ntorcheck.add_module_inf_check(model, module_name="my_model")\n```\n\n```\n# check whether linear layer\'s weight parameters become infinite\ntorcheck.add_tensor_inf_check(\n    model.linear_0.weight, tensor_name="linear_0.weight", module_name="my_model"\n)\n```\n\n#### Adding multiple checks in one call\nYou can add all checks for a module/tensor in one call:\n```\n# add all checks for model together\ntorcheck.add_module(\n    model,\n    module_name="my_model",\n    changing=True,\n    output_range=(-1, 1),\n    check_nan=True,\n    check_inf=True,\n)\n```\n\n```\n# add all checks for linear layer\'s weight together\ntorcheck.add_tensor(\n    model.linear_0.weight,\n    tensor_name="linear_0.weight",\n    module_name="my_model",\n    changing=True,\n    check_nan=True,\n    check_inf=True,\n)\n```\n\n### Step 3: Training and fixing\nAfter adding all the checks, run the training as usual and fix errors if any.\n\nBy default torcheck\'s error messages don\'t include tensor value information. If you\nthink it would be helpful, you can add the following line inside your torcheck code:\n```\ntorcheck.verbose_on()\n```\n\nYou can turn it off again by calling\n```\ntorcheck.verbose_off()\n```\n\n### (Optional) Step 4: Turning off checks\nWhen your model has passed all the checks, you can easily turn them off by calling\n```\ntorcheck.disable()\n```\nThis is useful when you want to run your model on a validation set, or you just want to\nremove the checking overhead from training.\n\nIf you want to turn on the checks again, just call\n```\ntorcheck.enable()\n```\n',
    'author': 'Peng Yan',
    'author_email': 'pengyan510@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/pengyan510/torcheck',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
