import json

import requests


class BaseEndpointHandler:
    def __init__(self, **kwargs):
        self.result = {}
        self.path_white_list = kwargs.get('path_white_list', [])
        self.path_black_list = kwargs.get('path_black_list', [])
        self.name_white_list = kwargs.get('name_white_list', [])
        self.name_black_list = kwargs.get('name_black_list', [])
        self.services = kwargs.get('services', {})

        if any(w_point in self.path_black_list for w_point in self.path_white_list):
            raise Exception('Black and white path lists cannot have the same values')
        if any(w_point in self.name_black_list for w_point in self.name_white_list):
            raise Exception('Black and white name lists cannot have the same values')

        self.start()

    def start(self):
        pass

    def is_service(self, name) -> bool:
        if name in self.services.keys():
            response = requests.get(f'{self.services[name]}/api/endpoints/')
            if response.status_code == 401:
                self.result[name] = None
                return True

            self.result[name] = json.loads(response.content)
            return True
        return False

    def endpoint_valid(self, name, path) -> bool:
        print(vars(self))
        print(path)
        if len(self.name_white_list) > 0 and not any(point in name for point in self.name_white_list):
            return False
        elif len(self.name_black_list) > 0 and any(point in name for point in self.name_black_list):
            return False
        elif len(self.path_white_list) > 0 and not any(point in path for point in self.path_white_list):
            return False
        elif len(self.path_black_list) > 0 and any(point in path for point in self.path_black_list):
            return False

        return True


class DjangoEndpointHandler(BaseEndpointHandler):
    def __init__(self, **kwargs):
        super().__init__(**kwargs)

    def start(self):
        from django.urls import get_resolver

        urlpattern = get_resolver().url_patterns

        for i, item in enumerate(urlpattern):
            urls_list = self.get_urls(item)
            if not urls_list:
                continue

            for key, value in urls_list.items():
                self.result[key] = value

    def get_urls(self, item, parent_url=''):
        result = {}

        endpoint = str(item.pattern)

        if hasattr(item, 'url_patterns'):
            for pattern in item.url_patterns:
                urls_list = self.get_urls(pattern, parent_url + endpoint)
                if not urls_list:
                    continue

                for key, value in urls_list.items():
                    if value is None:
                        del result[key]
                    else:
                        result[key] = value
        else:
            while True:
                path = parent_url + endpoint

                if not item.name:
                    print(f'No url patterns found for {path}')
                    break

                if self.is_service(item.name):
                    break

                if not self.endpoint_valid(item.name, path):
                    break

                result[item.name] = path
                break

        return result


class FastAPIEndpointHandler(BaseEndpointHandler):
    def __init__(self, fastapi_app,  **kwargs):
        super().__init__(**kwargs)
        self.get_urls(fastapi_app.routes)

    def get_urls(self, routes):
        for route in routes:
            name = getattr(route, 'name')
            if not name:
                continue

            while True:
                path = getattr(route, 'path')

                if self.is_service(name):
                    break

                if not self.endpoint_valid(name, path):
                    break

                self.result[name] = path
                break
