"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const path = require("path");
/**
 * A stack that will make a Lambda that will launch a lambda to glue
 * together all the DynamoDB tables into a global table
 */
class GlobalTableCoordinator extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        const lambdaFunction = new lambda.SingletonFunction(this, 'SingletonLambda', {
            code: lambda.Code.fromAsset(path.resolve(__dirname, '../', 'lambda-packages', 'aws-global-table-coordinator', 'lib')),
            description: 'Lambda to make DynamoDB a global table',
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
            timeout: cdk.Duration.minutes(5),
            uuid: 'D38B65A6-6B54-4FB6-9BAD-9CD40A6DAC12',
        });
        grantCreateGlobalTableLambda(lambdaFunction.role);
        new cdk.CustomResource(this, 'CfnCustomResource', {
            serviceToken: lambdaFunction.functionArn,
            pascalCaseProperties: true,
            properties: {
                regions: props.regions,
                resourceType: 'Custom::DynamoGlobalTableCoordinator',
                tableName: props.tableName,
            },
            removalPolicy: props.removalPolicy,
        });
    }
}
exports.GlobalTableCoordinator = GlobalTableCoordinator;
/**
 * Permits an IAM Principal to create a global dynamodb table.
 * @param principal The principal (no-op if undefined)
 */
function grantCreateGlobalTableLambda(principal) {
    if (principal) {
        principal.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'iam:CreateServiceLinkedRole',
                'application-autoscaling:DeleteScalingPolicy',
                'application-autoscaling:DeregisterScalableTarget',
                'dynamodb:CreateGlobalTable', 'dynamodb:DescribeLimits',
                'dynamodb:DeleteTable', 'dynamodb:DescribeGlobalTable',
                'dynamodb:UpdateGlobalTable',
            ],
        }));
    }
}
//# sourceMappingURL=data:application/json;base64,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