# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dollar_lambda']

package_data = \
{'': ['*']}

install_requires = \
['pytypeclass>=0.1.1,<0.2.0']

setup_kwargs = {
    'name': 'dollar-lambda',
    'version': '0.4.1',
    'description': 'An argument parser for Python built from functional first principles',
    'long_description': '<p align="center">\n  <img height="300" src="https://dollar-lambda.readthedocs.io/en/latest/_static/logo.png">\n</p>\n\n[$λ](https://dollar-lambda.readthedocs.io/) provides an alternative to [`argparse`](https://docs.python.org/3/library/argparse.html)\nbased on parser combinators and functional first principles. Arguably, `$λ` is way more expressive than any reasonable\nperson would ever need... but even if it\'s not the parser that we need, it\'s the parser we deserve.\n\n# Installation\n```\npip install dollar-lambda\n```\n\n# [Documentation](https://dollar-lambda.readthedocs.io/)\n\n# Highlights\n`$λ` comes with syntactic sugar that can make building parsers completely boilerplate-free.\nFor complex parsing situations that exceed the expressive capacity of this syntax,\nthe user can also drop down to the lower-level syntax that lies behind the sugar, which can\nhandle any reasonable amount of logical complexity.\n\n## The [`@command`](https://dollar-lambda.readthedocs.io/en/latest/api.html?highlight=command#dollar_lambda.decorators.command)\ndecorator\nFor the vast majority of parsing patterns, `@command` is the most concise way to\ndefine a parser:\n\n\n```python\nfrom dollar_lambda import command\n\n\n@command()\ndef main(x: int, dev: bool = False, prod: bool = False):\n    print(dict(x=x, dev=dev, prod=prod))\n```\n\nHere is the help text generated by this parser:\n\n\n```python\nmain("-h")\n```\n\n    usage: -x X --dev --prod\n\n\nOrdinarily you provide no arguments to `main` and it would get them from the command line.\nThe explicit arguments in this Readme are for demonstration purposes only.\nHere is how the main function handles input:\n\n\n```python\nmain("-x", "1", "--dev")\n```\n\n    {\'x\': 1, \'dev\': True, \'prod\': False}\n\n\nUse the `parsers` argument to add custom logic using the lower-level syntax:\n\n\n```python\nfrom dollar_lambda import flag\n\n\n@command(parsers=dict(kwargs=flag("dev") | flag("prod")))\ndef main(x: int, **kwargs):\n    print(dict(x=x, **kwargs))\n```\n\nThis parser requires either a `--dev` or `--prod` flag and maps it to the `kwargs` argument:\n\n\n```python\nmain("-h")\n```\n\n    usage: -x X [--dev | --prod]\n\n\nThis assigns `{\'dev\': True}` to the `kwargs` argument:\n\n\n```python\nmain("-x", "1", "--dev")\n```\n\n    {\'x\': 1, \'dev\': True}\n\n\nThis assigns `{\'prod\': True}` to the `kwargs` argument:\n\n\n```python\nmain("-x", "1", "--prod")\n```\n\n    {\'x\': 1, \'prod\': True}\n\n\nThis fails because the parser requires one or the other:\n\n\n```python\nmain("-x", "1")\n```\n\n    usage: -x X [--dev | --prod]\n    The following arguments are required: --dev\n\n\n## [`CommandTree`](https://dollar-lambda.readthedocs.io/en/latest/api.html#dollar_lambda.decorators.CommandTree) for dynamic dispatch\nFor many programs, a user will want to use one entrypoint for one set of\narguments, and another for another set of arguments. Returning to our example,\nlet\'s say we wanted to execute `prod_function` when the user provides the\n`--prod` flag, and `dev_function` when the user provides the `--dev` flag:\n\n\n```python\nfrom dollar_lambda import CommandTree\n\ntree = CommandTree()\n\n\n@tree.command()\ndef base_function(x: int):\n    print("Ran base_function with arguments:", dict(x=x))\n\n\n@base_function.command()\ndef prod_function(x: int, prod: bool):\n    print("Ran prod_function with arguments:", dict(x=x, prod=prod))\n\n\n@base_function.command()\ndef dev_function(x: int, dev: bool):\n    print("Ran dev_function with arguments:", dict(x=x, dev=dev))\n```\n\nLet\'s see how this parser handles different inputs.\nIf we provide the `--prod` flag, `$λ` automatically invokes\n `prod_function` with the parsed arguments:\n\n\n```python\ntree(\n    "-x", "1", "--prod"\n)  # usually you provide no arguments and tree gets them from sys.argv\n```\n\n    Ran prod_function with arguments: {\'x\': 1, \'prod\': True}\n\n\nIf we provide the `--dev` flag, `$λ` invokes `dev_function`:\n\n\n```python\ntree("-x", "1", "--dev")\n```\n\n    Ran dev_function with arguments: {\'x\': 1, \'dev\': True}\n\n\nWith this configuration, the parser will run `base_function` if neither\n`--prod` nor `--dev` are given:\n\n\n```python\ntree("-x", "1")\n```\n\n    Ran base_function with arguments: {\'x\': 1}\n\n\nThere are many other ways to use `CommandTree`,\nincluding some that make use of the `base_function`.\nTo learn more, we recommend the [`CommandTree` tutorial](https://dollar-lambda.readthedocs.io/en/latest/command_tree.html).\n\n## Lower-level syntax\n[`@command`](https://dollar-lambda.readthedocs.io/en/latest/api.html?highlight=command#dollar_lambda.decorators.command)\nand [`CommandTree`](https://dollar-lambda.readthedocs.io/en/latest/api.html#dollar_lambda.decorators.CommandTree)\ncover many use cases,\nbut they are both syntactic sugar for a lower-level interface that is far\nmore expressive.\n\nSuppose you want to implement a parser that first tries to parse an option\n(a flag that takes an argument),\n`-x X` and if that fails, tries to parse the input as a variadic sequence of\nfloats:\n\n\n```python\nfrom dollar_lambda import argument, option\n\np = option("x", type=int) | argument("y", type=float).many()\n```\n\nWe go over this syntax in greater detail in the [tutorial](https://dollar-lambda.readthedocs.io/en/latest/tutorial.html).\nFor now, suffice to say that [`argument`](https://dollar-lambda.readthedocs.io/en/latest/api.html?highlight=argument#dollar_lambda.parsers.argument)\n defines a positional argument,\n[`many`](https://dollar-lambda.readthedocs.io/en/latest/variations.html?highlight=many#many) allows parsers to be applied\nzero or more times, and [`|`](https://dollar-lambda.readthedocs.io/en/latest/api.html?highlight=__or__#dollar_lambda.parsers.Parser.__or__) expresses alternatives.\n\nHere is the help text:\n\n\n```python\np.parse_args(\n    "-h"\n)  # usually you provide no arguments and parse_args gets them from sys.argv\n```\n\n    usage: [-x X | [Y ...]]\n\n\nAs promised, this succeeds:\n\n\n```python\np.parse_args("-x", "1")\n```\n\n\n\n\n    {\'x\': 1}\n\n\n\nAnd this succeeds:\n\n\n```python\np.parse_args("1", "2", "3")\n```\n\n\n\n\n    {\'y\': [1.0, 2.0, 3.0]}\n\n\n\n### Thanks\nSpecial thanks to ["Functional Pearls"](https://www.cs.nott.ac.uk/~pszgmh/pearl.pdf) by Graham Hutton and Erik Meijer for bringing these topics to life.\n',
    'author': 'Ethan Brooks',
    'author_email': 'ethanabrooks@gmail.com',
    'maintainer': 'Ethan Brooks',
    'maintainer_email': 'ethanabrooks@gmail.com',
    'url': 'https://dollar-lambda.readthedocs.io',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
