# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['django_app_parameter',
 'django_app_parameter.management',
 'django_app_parameter.management.commands',
 'django_app_parameter.migrations']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'django-app-parameter',
    'version': '1.0.0',
    'description': "App-Parameter is a very simple Django app to save application's parameter in the database.",
    'long_description': '# Django-app-parameter\n\n![Python](https://img.shields.io/badge/python-3.9-yellow)\n![coverage](https://img.shields.io/badge/coverage-100%25-green)\n![version](https://img.shields.io/badge/version-1.0.0-blue)\n\nApp-Parameter is a very simple Django app to save some application\'s parameters in the database. Those parameters can be updated by users at running (no need to new deployment or any restart). It can be used to store the website\'s title or default e-mail expeditor...\n\n## Install\n\n    pip install django-app-parameter\n\n## Settings\n\n1. Add "django_app_parameter" to your INSTALLED_APPS setting like this:\n\n```python\nINSTALLED_APPS = [\n    ...\n    "django_app_parameter",\n]\n```\n\nIf you want global parameters to be available in templates, set provided context processor:\n\n```python\nTEMPLATES = [\n    ...\n    "OPTIONS": {\n        "context_processors": [\n            ...\n            "django_app_parameter.context_processors.add_global_parameter_context",\n        ],\n    },\n]\n```\n\n2. Run `python manage.py migrate` to create the django_app_parameter\'s table.\n\n3. Start development server and visit http://127.0.0.1:8000/admin/ to create parameters (you\'ll need the Admin app enabled).\n\n## Usage\n\n### Add new parameters\n\nUse admin interface to add parameters. You can access a parameter in your code use the "slug" field. Slug is built at first save with: `slugify(self.name).upper().replace("-", "_")`.\n\nExamples:\n\n    self.name     ==> self.slug\n    blog title    ==> BLOG_TITLE\n    sender e-mail ==> SENDER_E_MAIL\n    ##weird@Na_me ==> WERIDNA_ME\n\nSee [Django\'s slugify function](https://docs.djangoproject.com/fr/4.0/ref/utils/#django.utils.text.slugify) for more informations.\n\n### Access parameter in python code\n\nYou can read parameter anywhere in your code:\n\n```python\nfrom django.views.generic import TemplateView\nfrom django_app_parameter import app_parameter\n\nclass RandomView(TemplateView):\n    def get_context_data(self, **kwargs):\n        kwargs.update({"blog_title": app_parameter.BLOG_TITLE})\n        return super().get_context_data(**kwargs)\n```\n\nIn case you try to read a non existent parameter, an ImproperlyConfigured exception is raised.\n\n### Access parameter in templates\n\nYou can also access "global" parameters from every templates:\n\n```html\n<head>\n    <title>{{ BLOG_TITLE }}</title>\n</head>\n```\n\nA to make a parameter global, you only need to check is_global in admin.\n\n### Bulk load parameter with management command\n\nA management command is provided to let you easily load new parameters: `load_param`.\n\nIt will create or update, the key for matching is the SLUG.\n\nIt accepts 3 parameters: file, json and no-update.\n\n#### Option --file\n\nAdd all parameters listed in the provided file.\n\n`load_param --file /path/to/json.file`\n\nExample of file content:\n\n```json\n[\n    {"name": "hello ze world", "value": "yes", "description": "123", "is_global": true},\n    {"slug": "A8B8C", "name": "back on test", "value": "yes", "value_type": "INT" }\n]\n```\n\nHere all available property you can add to the json:\n* name\n* slug\n* value_type\n* value\n* description\n* is_global\n\nIf slug is not provided it will be built. Default value_type is STR (string) and default is_global is False. Name is always required, others properties are optionnals.\n\n#### Option --json\n\nAdd parameters in one shot.\n\n`load_param --json "[{\'name\': \'param1\'}, {\'name\': \'param2\'},]"`\n\nThe provided json needs to match same rules as for --file option above.\n\nYou can\'t use --json and --file together.\n\n#### Option --no-update\n\nThis option is provided to disable \'update\' if parameter with same SLUG already exists. It can be used with --json and --file. It\'s useful to ensure all parameters are created in all environments and can be executed altogether with migrate. It avoid replacing already existing parameters\' values which could lead to breaking environments.\n\n`load_param --no-update --file required_parameters.json`\n\nI use it in my starting container script:\n```bash\n#!/bin/bash\n\n# Execute migrations\npython manage.py migrate\n\n# load new parameters if any\npython manage.py load_param --no-update --file required_parameters.json\n\n# launch webserver\ngunicorn config.wsgi\n```\n\nEnjoy.\n\n## Ideas which could come later (or not)\n\n* [x] A migration process to keep a list of your parameters in a file and automatically add them in each environment\n* [x] Shortcut to use Parameter.str(slug) (skip \'objects\' key word)\n* [x] Management command to add a new parameter\n* Check correctness of value type on save\n* modifications history\n* Boolean and Datetime types\n\nIf you have new idea you would like to see, feel free to open a new issue in this repo.\n\n## Help developping\n\nIf you want to participate to the development, there are (only) 2 constraints:\n* Format all your code with black\n* All unit test must pass and new code must be covered\n\nBecause tests require a whole django environment, to run them I use https://github.com/Swannbm/runtest_on_dj_packages ; if you know a better way to do it I am all ears :D\n\n## Why Django-App-Parameter\n\nBecause I wanted to try packaging a Django app and I used this one in most of my projects so it seemed a good idea.',
    'author': 'Swann',
    'author_email': 'swann.bouviermuller@gmail.com',
    'maintainer': 'Swann',
    'maintainer_email': 'swann.bouviermuller@gmail.com',
    'url': 'https://github.com/Swannbm/django-app-parameter',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
