import zmq
import logging
from functools import wraps
from zmq.asyncio import Context
from .message import decode_msg, encode_msg

logger = logging.getLogger(__package__)


class SimRpcClient:
    def __init__(self, server_address="tcp://localhost:5559",
                 is_async=False, timeout=3000):
        self.is_async = is_async
        self.server_address = server_address
        self.poll = zmq.Poller()
        self.async_poll = zmq.asyncio.Poller()
        self.timeout = timeout

    def get_socket(self):
        if self.is_async:
            context = Context()
        else:
            context = zmq.Context()
        socket = context.socket(zmq.REQ)
        socket.connect(self.server_address)
        self.poll.register(socket, zmq.POLLIN)
        return socket

    def task(self, response_only: bool = False, func=False):

        def decorate(func):

            @wraps(func)
            def wrapper(*args, **kwargs):
                cls = args[0]
                if hasattr(cls, "__module__") and \
                        cls.__class__.__name__ != "function":
                    cls = args[0]
                    args = args[1:]
                else:
                    cls = func
                data = decode_msg(
                    service=cls.__class__.__name__ if cls else "",
                    entry=func.__name__,
                    args=args,
                    kwargs=kwargs
                )
                if hasattr(cls, "socket"):
                    socket = getattr(cls, "socket")
                else:
                    socket = self.get_socket()
                    setattr(cls, "socket", socket)
                socket.send(data)
                socks = dict(self.poll.poll(self.timeout))
                if socks.get(socket) == zmq.POLLIN:
                    res = socket.recv()
                else:
                    res = b'\x82\xa8response\xc0\xa3msg\xa7timeout'
                return encode_msg(res, response_only=response_only)

            @wraps(func)
            async def async_wrapper(*args, **kwargs):
                cls = args[0]
                if hasattr(cls, "__module__") and \
                        cls.__class__.__name__ != "function":
                    cls = args[0]
                    args = args[1:]
                else:
                    cls = func
                data = decode_msg(
                    service=cls.__class__.__name__ if cls else "",
                    entry=func.__name__,
                    args=args,
                    kwargs=kwargs
                )
                if hasattr(cls, "socket"):
                    socket = getattr(cls, "socket")
                else:
                    socket = self.get_socket()
                    setattr(cls, "socket", socket)
                await socket.send(data)
                socks = await self.async_poll.poll(self.timeout)
                socks = dict(socks)
                if socks.get(socket) == zmq.POLLIN:
                    res = await socket.recv()
                else:
                    res = b'\x82\xa8response\xc0\xa3msg\xa7timeout'
                return encode_msg(res, response_only=response_only)

            if self.is_async:
                return async_wrapper
            else:
                return wrapper
        return decorate
