"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrAddStep = exports.ActionOnFailure = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * The action to take when the cluster step fails.
 * @see https://docs.aws.amazon.com/emr/latest/APIReference/API_StepConfig.html
 *
 * Here, they are named as TERMINATE_JOB_FLOW, TERMINATE_CLUSTER, CANCEL_AND_WAIT, and CONTINUE respectively.
 *
 * @default CONTINUE
 *
 */
var ActionOnFailure;
(function (ActionOnFailure) {
    /**
     * Terminate the Cluster on Step Failure
     */
    ActionOnFailure["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    /**
     * Cancel Step execution and enter WAITING state
     */
    ActionOnFailure["CANCEL_AND_WAIT"] = "CANCEL_AND_WAIT";
    /**
     * Continue to the next Step
     */
    ActionOnFailure["CONTINUE"] = "CONTINUE";
})(ActionOnFailure = exports.ActionOnFailure || (exports.ActionOnFailure = {}));
/**
 * A Step Functions Task to add a Step to an EMR Cluster
 *
 * The StepConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the StepId
 *
 */
class EmrAddStep extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, props);
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EmrAddStepProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.actionOnFailure = (_b = props.actionOnFailure) !== null && _b !== void 0 ? _b : ActionOnFailure.CONTINUE;
        this.integrationPattern = (_c = props.integrationPattern) !== null && _c !== void 0 ? _c : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrAddStep.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskPolicies = this.createPolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('elasticmapreduce', 'addStep', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                ClusterId: this.props.clusterId,
                Step: {
                    Name: this.props.name,
                    ActionOnFailure: this.actionOnFailure.valueOf(),
                    HadoopJarStep: {
                        Jar: this.props.jar,
                        MainClass: this.props.mainClass,
                        Args: this.props.args,
                        Properties: (this.props.properties === undefined) ?
                            undefined :
                            Object.entries(this.props.properties).map(kv => ({
                                Key: kv[0],
                                Value: kv[1],
                            })),
                    },
                },
            }),
        };
    }
    /**
     * This generates the PolicyStatements required by the Task to call AddStep.
     */
    createPolicyStatements() {
        const stack = core_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: [
                    'elasticmapreduce:AddJobFlowSteps',
                    'elasticmapreduce:DescribeStep',
                    'elasticmapreduce:CancelSteps',
                ],
                resources: [
                    stack.formatArn({
                        service: 'elasticmapreduce',
                        resource: 'cluster',
                        resourceName: '*',
                    }),
                ],
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRAddJobFlowStepsRule',
                    })],
            }));
        }
        return policyStatements;
    }
}
exports.EmrAddStep = EmrAddStep;
_a = JSII_RTTI_SYMBOL_1;
EmrAddStep[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EmrAddStep", version: "1.153.0" };
EmrAddStep.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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