"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BatchSubmitJob = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Task to submits an AWS Batch job from a job definition.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html
 * @stability stable
 */
class BatchSubmitJob extends sfn.TaskStateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_b = props.integrationPattern) !== null && _b !== void 0 ? _b : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS);
        // validate arraySize limits
        core_1.withResolved(props.arraySize, (arraySize) => {
            if (arraySize !== undefined && (arraySize < 2 || arraySize > 10000)) {
                throw new Error(`arraySize must be between 2 and 10,000. Received ${arraySize}.`);
            }
        });
        // validate dependency size
        if (props.dependsOn && props.dependsOn.length > 20) {
            throw new Error(`dependencies must be 20 or less. Received ${props.dependsOn.length}.`);
        }
        // validate attempts
        core_1.withResolved(props.attempts, (attempts) => {
            if (attempts !== undefined && (attempts < 1 || attempts > 10)) {
                throw new Error(`attempts must be between 1 and 10. Received ${attempts}.`);
            }
        });
        // validate timeout
        props.timeout !== undefined && core_1.withResolved(props.timeout.toSeconds(), (timeout) => {
            if (timeout < 60) {
                throw new Error(`attempt duration must be greater than 60 seconds. Received ${timeout} seconds.`);
            }
        });
        // This is required since environment variables must not start with AWS_BATCH;
        // this naming convention is reserved for variables that are set by the AWS Batch service.
        if ((_c = props.containerOverrides) === null || _c === void 0 ? void 0 : _c.environment) {
            Object.keys(props.containerOverrides.environment).forEach(key => {
                if (key.match(/^AWS_BATCH/)) {
                    throw new Error(`Invalid environment variable name: ${key}. Environment variable names starting with 'AWS_BATCH' are reserved.`);
                }
            });
        }
        this.taskPolicies = this.configurePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        var _b;
        return {
            Resource: task_utils_1.integrationResourceArn('batch', 'submitJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                JobDefinition: this.props.jobDefinition.jobDefinitionArn,
                JobName: this.props.jobName,
                JobQueue: this.props.jobQueue.jobQueueArn,
                Parameters: (_b = this.props.payload) === null || _b === void 0 ? void 0 : _b.value,
                ArrayProperties: this.props.arraySize !== undefined
                    ? { Size: this.props.arraySize }
                    : undefined,
                ContainerOverrides: this.props.containerOverrides
                    ? this.configureContainerOverrides(this.props.containerOverrides)
                    : undefined,
                DependsOn: this.props.dependsOn
                    ? this.props.dependsOn.map(jobDependency => ({
                        JobId: jobDependency.jobId,
                        Type: jobDependency.type,
                    }))
                    : undefined,
                RetryStrategy: this.props.attempts !== undefined
                    ? { Attempts: this.props.attempts }
                    : undefined,
                Timeout: this.props.timeout
                    ? { AttemptDurationSeconds: this.props.timeout.toSeconds() }
                    : undefined,
            }),
            TimeoutSeconds: undefined,
        };
    }
    configurePolicyStatements() {
        return [
            // Resource level access control for job-definition requires revision which batch does not support yet
            // Using the alternative permissions as mentioned here:
            // https://docs.aws.amazon.com/batch/latest/userguide/batch-supported-iam-actions-resources.html
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'batch',
                        resource: 'job-definition',
                        resourceName: '*',
                    }),
                    this.props.jobQueue.jobQueueArn,
                ],
                actions: ['batch:SubmitJob'],
            }),
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventsForBatchJobsRule',
                    }),
                ],
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
            }),
        ];
    }
    configureContainerOverrides(containerOverrides) {
        var _b, _c;
        let environment;
        if (containerOverrides.environment) {
            environment = Object.entries(containerOverrides.environment).map(([key, value]) => ({
                Name: key,
                Value: value,
            }));
        }
        let resources;
        if (containerOverrides.gpuCount) {
            resources = [
                {
                    Type: 'GPU',
                    Value: `${containerOverrides.gpuCount}`,
                },
            ];
        }
        return {
            Command: containerOverrides.command,
            Environment: environment,
            InstanceType: (_b = containerOverrides.instanceType) === null || _b === void 0 ? void 0 : _b.toString(),
            Memory: (_c = containerOverrides.memory) === null || _c === void 0 ? void 0 : _c.toMebibytes(),
            ResourceRequirements: resources,
            Vcpus: containerOverrides.vcpus,
        };
    }
}
exports.BatchSubmitJob = BatchSubmitJob;
_a = JSII_RTTI_SYMBOL_1;
BatchSubmitJob[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.BatchSubmitJob", version: "1.97.0" };
BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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