# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['avr', 'avr.mqtt', 'avr.serial', 'avr.utils']

package_data = \
{'': ['*']}

install_requires = \
['loguru>=0.6.0,<0.7.0']

extras_require = \
{'mqtt': ['paho-mqtt>=1.6.1,<2.0.0'], 'serial': ['pyserial>=3.5,<4.0']}

setup_kwargs = {
    'name': 'bell-avr-libraries',
    'version': '0.1.9',
    'description': 'Common Python libraries used by parts of Bell AVR',
    'long_description': '# AVR-Python-Libraries\n\n## Install\n\nTo install the base package, run:\n\n```bash\npip install bell-avr-libraries\n```\n\nAdditionally, the `mqtt` and `serial` extras are available if you want to use\nthe MQTT or PCC functionality.\n\n```bash\npip install bell-avr-libraries[mqtt,serial]\n```\n\n## Usage\n\n### MQTT\n\n```python\nfrom bell.avr import mqtt\n```\n\nThese are MQTT utilities that are used to have a consistent messaging protocol\nthroughout all the AVR software modules.\n\nThe first part of this are the payloads for the MQTT messages themselves. As AVR\nexclusively uses JSON, these are all\n[`TypedDict`](https://docs.python.org/3/library/typing.html#typing.TypedDict)s\nthat have all of the required fields for a message.\n\nExample:\n\n```python\nfrom bell.avr.mqtt.payloads import AvrPcmSetBaseColorPayload\n\npayload = AvrPcmSetBaseColorPayload((128, 232, 142, 0))\n```\n\nThe second part of the MQTT libraries, is the `MQTTModule` class.\nThis is a boilerplate module for AVR that makes it very easy to send\nand recieve MQTT messages and do something with them.\n\nExample:\n\n```python\nfrom bell.avr.mqtt.client import MQTTModule\nfrom bell.avr.mqtt.payloads import AvrFcmVelocityPayload, AvrPcmSetServoOpenClosePayload\n\n\nclass Sandbox(MQTTModule):\n    def __init__(self) -> None:\n        super().__init__()\n        self.topic_map = {"avr/fcm/velocity": self.show_velocity}\n\n    def show_velocity(self, payload: AvrFcmVelocityPayload) -> None:\n        vx = payload["vX"]\n        vy = payload["vY"]\n        vz = payload["vZ"]\n        v_ms = (vx, vy, vz)\n        print(f"Velocity information: {v_ms} m/s")\n\n    def open_servo(self) -> None:\n        payload = AvrPcmSetServoOpenClosePayload(servo=0, action="open")\n        self.send_message("avr/pcm/set_servo_open_close", payload)\n\n\nif __name__ == "__main__":\n    box = Sandbox()\n    box.run()\n```\n\nThe `topic_map` dictionary is a class attribute that maps topics to subscribe to\nand a function that will handle an incoming payload. The `payload` argument\nshould match the appropriate `Payload` class for that topic.\n\nAdditionally, the `message_cache` attribute is a dictionary that holds\na copy of the last payload sent by that module on a given topic. The keys are the\ntopic strings, and the values are the topic payloads.\n\n### Utils\n\n```python\nfrom bell.avr import utils\n```\n\nThese are general purpose utilities.\n\n#### Decorators\n\n```python\nfrom bell.avr.utils import decorators\n```\n\nThere are 3 different function decorators available, which are helpful for MQTT\nmessage callbacks. First is the `try_except` decorator, which wraps the\nfunction in a `try: except:` statement and will log any exceptions to the console:\n\n```python\n    @decorators.try_except\n    def assemble_hil_gps_message(self) -> None:\n        ...\n```\n\nAdditionally, there is the `reraise` argument, which can be set to `True` to raise\nany exceptions that are encountered. This is helpful if you still want exceptions\nto propagate up, but log them.\n\nThere is an async version of this decorator as well with an `async_` prefix.\n\n```python\n    @decorators.async_try_except()\n    async def connected_status_telemetry(self) -> None:\n        ...\n```\n\nThe last decorator is `run_forever` which will run the wrapped function forever,\nwith a given `period` or `frequency`.\n\n```python\n    @decorators.run_forever(frequency=5)\n    def read_data(self) -> None:\n        ...\n```\n\n#### Timing\n\n```python\nfrom bell.avr.utils import timing\n```\n\nHere is a `rate_limit` function which take a callable and a\nperiod or frequency, and only run the callable at that given rate.\n\n```python\nfor _ in range(10):\n    # add() will only run twice\n    timing.rate_limit(add, period=5)\n    time.sleep(1)\n```\n\nThis works tracking calls to the `rate_limit` function from a line number\nwithin a file, so multiple calls to `rate_limit` say within a loop\nwith the same callable and period will be treated seperately. This allows\nfor dynamic frequency manipulation.\n\n### Serial\n\n```python\nfrom bell.avr import serial\n```\n\nThese are serial utilities that help facilitate finding and communicating\nwith the AVR peripherial control computer.\n\n#### Client\n\n```python\nfrom bell.avr.serial import client\n```\n\nThe `SerialLoop` class is a small wrapper around the `pyserial` `serial.Serial`\nclass which adds a `run` method that will try to read data from the serial device\nas fast as possible.\n\n```python\nser = client.SerialLoop()\n```\n\n#### PCC\n\n```python\nfrom bell.avr.serial import client\n```\n\nThe `PeripheralControlComputer` class sends serial messages\nto the AVR peripherial control computer, via easy-to-use class methods.\n\n```python\nimport bell.avr.serial.client\nimport bell.avr.serial.pcc\nimport threading\n\nclient = bell.avr.serial.client.SerialLoop()\nclient.port = port\nclient.baudrate = baudrate\nclient.open()\n\npcc = bell.avr.serial.pcc.PeripheralControlComputer(client)\n\nclient_thread = threading.Thread(target=client.run)\nclient_thread.start()\n\npcc.set_servo_max(0, 100)\n```\n\n#### Ports\n\n```python\nfrom bell.avr.serial import ports\n```\n\nHere is a `list_serial_ports` function which returns a list of detected serial\nports connected to the system.\n\n```python\nserial_ports = ports.list_serial_ports()\n# ["COM1", "COM5", ...]\n```\n\n## Development\n\nInstall [`poetry`](https://python-poetry.org/) and run\n`poetry install --extras mqtt --extras serial` to install of the dependencies\ninside a virtual environment.\n\nBuild the auto-generated code with `poetry run python build.py`. From here,\nyou can now produce a package with `poetry build`.\n\nTo add new message definitions, add entries to the `bell/avr/mqtt/messages.jsonc` file.\nThe 3 parts of a new message are as follows:\n\n1. "topic": This is the full topic path for the message. This must be all lower case and\n   start with "avr/".\n2. "payload": These are the keys of the payload for the message.\n   This is a list of key entries (see below).\n3. "docs": This is an optional list of Markdown strings that explains what this\n   message does. Each list item is a new line.\n\nThe key entries for a message have the following elements:\n\n1. "key": This is the name of the key. Must be a valid Python variable name.\n2. "type": This is the data type of the key such as `Tuple[int, int, int, int]`.\n   Must be a valid Python type hint. Otherwise, this can be a list of more\n   key entries, effectively creating a nested dictionary.\n3. "docs": This is an optional list of Markdown strings that explains what the\n   key is. Each list item is a new line.\n\nThe `bell/avr/mqtt/schema.json` file will help ensure the correct schema is maintained,\nassuming you are using VS Code.\n\n## MQTT Documentation\n\n### Data Types\n\n#### AvrApriltagsSelectedPos\n\nThe position of the vehicle in world frame in centimeters\n\n- `"n"` (`float`):\n    The +north position of the vehicle relative to the world origin in world frame\n- `"e"` (`float`):\n    The +east position of the vehicle relative to the world origin in world frame\n- `"d"` (`float`):\n    The +down position of the vehicle relative to the world origin in world frame\n\n#### AvrApriltagsRawTags\n\n- `"id"` (`int`):\n    The ID of the AprilTag\n- `"pos"` (`AvrApriltagsRawTagsPos`)\n- `"rotation"` (`Tuple[Tuple[float, float, float], Tuple[float, float, float], Tuple[float, float, float]]`):\n    The 3x3 rotation matrix\n\n#### AvrApriltagsRawTagsPos\n\n- `"x"` (`float`):\n    The position in meters of the camera relative to the AprilTag\'s X frame\n- `"y"` (`float`):\n    The position in meters of the camera relative to the AprilTag\'s Y frame\n- `"z"` (`float`):\n    The position in meters of the camera relative to the AprilTag\'s Z frame\n\n#### AvrApriltagsVisibleTags\n\n- `"id"` (`int`):\n    The ID of the AprilTag\n- `"horizontal_dist"` (`float`):\n    The horizontal scalar distance from vehicle to AprilTag, in centimeters\n- `"vertical_dist"` (`float`):\n    The vertical scalar distance from vehicle to AprilTag, in centimeters\n- `"angle_to_tag"` (`float`):\n    The angle formed by the vector pointing from the vehicles body to the AprilTag in world frame relative to world-north\n- `"heading"` (`float`):\n    The heading of the vehicle in world frame\n- `"pos_rel"` (`AvrApriltagsVisibleTagsPosRel`):\n    The relative position of the vehicle to the tag in world frame in centimeters\n- `"pos_world"` (`AvrApriltagsVisibleTagsPosWorld`):\n    The position of the vehicle in world frame in centimeters (if the tag has no truth data, this will not be present in the output)\n\n#### AvrApriltagsVisibleTagsPosRel\n\nThe relative position of the vehicle to the tag in world frame in centimeters\n\n- `"x"` (`float`):\n    The x (+north/-south) position of the vehicle relative to the AprilTag in world frame\n- `"y"` (`float`):\n    The y (+east/-west) position of the vehicle relative to the AprilTag in world frame\n- `"z"` (`float`):\n    The z (+down/-up) position of the vehicle relative to the AprilTag in world frame\n\n#### AvrApriltagsVisibleTagsPosWorld\n\nThe position of the vehicle in world frame in centimeters (if the tag has no truth data, this will not be present in the output)\n\n- `"x"` (`Optional[float]`):\n    The x position of the vehicle relative to the world origin (this is the ship) in world frame (for reference the mountain is **north** of the beach)\n- `"y"` (`Optional[float]`):\n    The y position of the vehicle relative to the world origin in world frame\n- `"z"` (`Optional[float]`):\n    The z position of the vehicle relative to the world origin in world frame\n\n### Message Payloads\n\n#### AvrPcmSetBaseColorPayload\n\nTopic: `avr/pcm/set_base_color`\n\nThis sets the color of the LED strip on the PCC\n\n- `"wrgb"` (`Tuple[int, int, int, int]`):\n    A list of 4 `int`s between 0 and 255 to set the base color of the LEDs. This is in order of Red, Green, Blue, and Alpha. Example: [255, 0, 128, 255].\n\n#### AvrPcmSetTempColorPayload\n\nTopic: `avr/pcm/set_temp_color`\n\nThis sets the color of the LED strip on the PCC temporarily\n\n- `"wrgb"` (`Tuple[int, int, int, int]`):\n    A list of 4 `int`s between 0 and 255 to set the base color of the LEDs. This is in order of Red, Green, Blue, and Alpha. Example: [255, 0, 128, 255].\n- `"time"` (`float`):\n    Optional `float` for the number of seconds the color should be set for. Default is 0.5.\n\n#### AvrPcmFireLaserPayload\n\nTopic: `avr/pcm/fire_laser`\n\nFires the laser for a 0.25 sec pulse. Has a cooldown of 0.5 sec.\n\nThere is no content for this class\n\n#### AvrPcmSetLaserOnPayload\n\nTopic: `avr/pcm/set_laser_on`\n\nTurns on laser (in blip mode - 0.1 second on every 0.5. sec)\n\nThere is no content for this class\n\n#### AvrPcmSetLaserOffPayload\n\nTopic: `avr/pcm/set_laser_off`\n\nTurns off laser (laser off from blip mode - but doesn\'t prevent fire_laser)\n\nThere is no content for this class\n\n#### AvrPcmSetServoOpenClosePayload\n\nTopic: `avr/pcm/set_servo_open_close`\n\nThis opens or closes a specific servo.\n\n- `"servo"` (`int`):\n    ID of the servo to open or close as an `int`. This is 0-indexed.\n- `"action"` (`Literal["open", "close"]`):\n    Either the literal string "open" or "close".\n\n#### AvrPcmSetServoMinPayload\n\nTopic: `avr/pcm/set_servo_min`\n\nThis sets the minimum pulse width of a specific servo.\n\n- `"servo"` (`int`):\n    ID of the servo to set the minimum pulse width as an `int`. This is 0-indexed.\n- `"min_pulse"` (`int`):\n    A `int` between 0 and 1000.\n\n#### AvrPcmSetServoMaxPayload\n\nTopic: `avr/pcm/set_servo_max`\n\nThis sets the maximum pulse width of a specific servo.\n\n- `"servo"` (`int`):\n    ID of the servo to set the maximum pulse width as an `int`. This is 0-indexed.\n- `"max_pulse"` (`int`):\n    A `int` between 0 and 1000.\n\n#### AvrPcmSetServoPctPayload\n\nTopic: `avr/pcm/set_servo_pct`\n\nThis sets the percentage of a specific servo. 0 is closed, and 100 is open.\n\n- `"servo"` (`int`):\n    ID of the servo to set the percent as an `int`. This is 0-indexed.\n- `"percent"` (`int`):\n    A `int` between 0 and 100.\n\n#### AvrPcmSetServoAbsPayload\n\nTopic: `avr/pcm/set_servo_abs`\n\nThis sets the absolute position of a specific servo. SERVOMIN 150 is closed, and SERVOMAX 425 is open. We need to send a High and Low byte due to limitations of the API\n\n- `"servo"` (`int`):\n    ID of the servo to set the percent as an `int`. This is 0-indexed.\n- `"absolute"` (`int`):\n    Absolute position between SERVOMIN 150 and SERVOMAX 425\n\n#### AvrFcmHilGpsStatsPayload\n\nTopic: `avr/fcm/hil_gps_stats`\n\nThis reports statistics on the HIL GPS data that is fed into the flight controller.\n\n- `"num_frames"` (`int`):\n    This is the number of messages that have been sent to the flight controller since the software has started.\n\n#### AvrFcmEventsPayload\n\nTopic: `avr/fcm/events`\n\nThis reports events from the flight controller such as flight mode changes.\n\n- `"name"` (`str`):\n    The name of the event.\n- `"payload"` (`str`):\n    The payload of the event.\n\n#### AvrFcmBatteryPayload\n\nTopic: `avr/fcm/battery`\n\nThis reports battery information from the flight controller.\n\n- `"voltage"` (`float`):\n    Battery voltage\n- `"soc"` (`float`):\n    State of charge (0 - 100)\n\n#### AvrFcmStatusPayload\n\nTopic: `avr/fcm/status`\n\nThis reports general status of the flight controller.\n\n- `"armed"` (`bool`):\n    Boolean of if the drone is currently armed\n- `"mode"` (`str`):\n    Current flight mode, which is one of the following:\n    - \'UNKNOWN\'\n    - \'READY\'\n    - \'TAKEOFF\'\n    - \'HOLD\'\n    - \'MISSION\'\n    - \'RETURN_TO_LAUNCH\'\n    - \'LAND\'\n    - \'OFFBOARD\'\n    - \'FOLLOW_ME\'\n    - \'MANUAL\'\n    - \'ALTCTL\'\n    - \'POSCTL\'\n    - \'ACRO\'\n    - \'STABILIZED\'\n    - \'RATTITUDE\'\n\n#### AvrFcmLocationLocalPayload\n\nTopic: `avr/fcm/location/local`\n\nThis reports the current position of the drone in local coordinates from the flight controller.\n\n- `"dX"` (`float`):\n    X position in a local North/East/Down coordinate system in meters\n- `"dY"` (`float`):\n    Y position in a local North/East/Down coordinate system in meters\n- `"dZ"` (`float`):\n    Z position in a local North/East/Down coordinate system in meters\n\n#### AvrFcmLocationGlobalPayload\n\nTopic: `avr/fcm/location/global`\n\nThis reports the current position of the drone in global coordinates from the flight controller.\n\n- `"lat"` (`float`):\n    Latitude in degrees\n- `"lon"` (`float`):\n    Longitude in degrees\n- `"alt"` (`float`):\n    Altitude relative to takeoff altitude in meters\n- `"hdg"` (`float`):\n    Heading in degrees.\n\n#### AvrFcmLocationHomePayload\n\nTopic: `avr/fcm/location/home`\n\nThis reports the current position of the drone\'s home position in global coordinates.\n\n- `"lat"` (`float`):\n    Latitude in degrees of the home position\n- `"lon"` (`float`):\n    Longitude in degrees of the home position\n- `"alt"` (`float`):\n    Altitude relative to the home position in meters\n\n#### AvrFcmAttitudeEulerPayload\n\nTopic: `avr/fcm/attitude/euler`\n\nThis reports the current attitude of the drone from the flight controller.\n\n- `"roll"` (`float`):\n    Roll in degrees\n- `"pitch"` (`float`):\n    Pitch in degrees\n- `"yaw"` (`float`):\n    Yaw in degrees\n\n#### AvrFcmVelocityPayload\n\nTopic: `avr/fcm/velocity`\n\nThis reports the current velocity vectors of the drone from the flight controller.\n\n- `"vX"` (`float`):\n    X velocity in a local North/East/Down coordinate system in meters per second\n- `"vY"` (`float`):\n    Y velocity in a local North/East/Down coordinate system in meters per second\n- `"vZ"` (`float`):\n    Z velocity in a local North/East/Down coordinate system in meters per second\n\n#### AvrFcmGpsInfoPayload\n\nTopic: `avr/fcm/gps_info`\n\nThis reports the current status of the flight controller\'s GPS connection.\n\n- `"num_satellites"` (`int`):\n    Number of visible satellites in use. HIL GPS will appear as 13.\n- `"fix_type"` (`str`):\n    GPS fix type\n\n#### AvrFusionPositionNedPayload\n\nTopic: `avr/fusion/position/ned`\n\nThis reports the computed position of the drone in local coordinates from our sensor fusion.\n\n- `"n"` (`float`):\n    X position in a local North/East/Down coordinate system in meters\n- `"e"` (`float`):\n    Y position in a local North/East/Down coordinate system in meters\n- `"d"` (`float`):\n    Z position in a local North/East/Down coordinate system in meters\n\n#### AvrFusionVelocityNedPayload\n\nTopic: `avr/fusion/velocity/ned`\n\nThis reports the computed velocity vectors of the drone from our sensor fusion.\n\n- `"Vn"` (`float`):\n    X velocity in a local North/East/Down coordinate system in meters per second\n- `"Ve"` (`float`):\n    Y velocity in a local North/East/Down coordinate system in meters per second\n- `"Vd"` (`float`):\n    Z velocity in a local North/East/Down coordinate system in meters per second\n\n#### AvrFusionGeoPayload\n\nTopic: `avr/fusion/geo`\n\nThis reports the computed position of the drone in global coordinates from our sensor fusion.\n\n- `"lat"` (`float`):\n    Latitude in degrees\n- `"lon"` (`float`):\n    Longitude in degrees\n- `"alt"` (`float`):\n    Altitude relative to takeoff altitude in meters\n\n#### AvrFusionGroundspeedPayload\n\nTopic: `avr/fusion/groundspeed`\n\nThis reports the computed groundspeed of the drone from our sensor fusion.\n\n- `"groundspeed"` (`float`):\n    Groundspeed of the drone in meters per second. This is a normal vector of the N and E velocities.\n\n#### AvrFusionCoursePayload\n\nTopic: `avr/fusion/course`\n\nThis reports the computed course of the drone from our sensor fusion.\n\n- `"course"` (`float`):\n    Course in degrees\n\n#### AvrFusionClimbratePayload\n\nTopic: `avr/fusion/climbrate`\n\nThis reports the computed rate of climb of the drone from our sensor fusion.\n\n- `"climb_rate_fps"` (`float`):\n    Rate of climb in feet per second\n\n#### AvrFusionAttitudeQuatPayload\n\nTopic: `avr/fusion/attitude/quat`\n\nThis reports the computed attitude of the drone from our sensor fusion.\n\n- `"w"` (`float`):\n    Quaternion w value\n- `"x"` (`float`):\n    Quaternion x value\n- `"y"` (`float`):\n    Quaternion y value\n- `"z"` (`float`):\n    Quaternion z value\n\n#### AvrFusionAttitudeEulerPayload\n\nTopic: `avr/fusion/attitude/euler`\n\nThis reports the computed attitude of the drone from our sensor fusion.\n\n- `"psi"` (`float`):\n    Roll in radians\n- `"theta"` (`float`):\n    Pitch in radians\n- `"phi"` (`float`):\n    Yaw in radians\n\n#### AvrFusionAttitudeHeadingPayload\n\nTopic: `avr/fusion/attitude/heading`\n\nThis reports the computed heading of the drone from our sensor fusion.\n\n- `"heading"` (`float`):\n    Heading in degrees\n\n#### AvrFusionHilGpsPayload\n\nTopic: `avr/fusion/hil_gps`\n\nThis is the raw data that will get converted to a MAVLink message and sent to the flight controller for HIL GPS. <https://mavlink.io/en/messages/common.html#HIL_GPS>\n\n- `"time_usec"` (`int`):\n    UNIX epoch timestamp in microseconds\n- `"fix_type"` (`int`):\n    0-1: no fix, 2: 2D fix, 3: 3D fix.\n- `"lat"` (`int`):\n    WGS84 Latitude * 10000000\n- `"lon"` (`int`):\n    WGS84 Longitude * 10000000\n- `"alt"` (`int`):\n    Altitude from sea level in mm. Positive for up.\n- `"eph"` (`int`):\n    GPS HDOP horizontal dilution of position\n- `"epv"` (`int`):\n    GPS VDOP vertical dilution of position\n- `"vel"` (`int`):\n    GPS ground speed in centimeters per second\n- `"vn"` (`int`):\n    GPS velocity in north direction in centimeters per second\n- `"ve"` (`int`):\n    GPS velocity in east direction in centimeters per second\n- `"vd"` (`int`):\n    GPS velocity in down direction in centimeters per second\n- `"cog"` (`int`):\n    Course over ground in degrees\n- `"satellites_visible"` (`int`):\n    Number of satellites visible. This is hardcoded to 13 for our HIL GPS.\n- `"heading"` (`int`):\n    Custom heading field. This is the heading in degrees * 100.\n\n#### AvrVioResyncPayload\n\nTopic: `avr/vio/resync`\n\nThis reports significant position differences from the tracking camera, and detected AprilTags at known positions.\n\n- `"n"` (`float`):\n    X position difference in a local North/East/Down coordinate system in meters\n- `"e"` (`float`):\n    Y position difference in a local North/East/Down coordinate system in meters\n- `"d"` (`float`):\n    Z position difference in a local North/East/Down coordinate system in meters\n- `"heading"` (`float`):\n    Heading difference in degrees\n\n#### AvrVioPositionNedPayload\n\nTopic: `avr/vio/position/ned`\n\nThis reports the measured position of the drone in local coordinates from the tracking camera.\n\n- `"n"` (`float`):\n    X position in a local North/East/Down coordinate system in meters\n- `"e"` (`float`):\n    Y position in a local North/East/Down coordinate system in meters\n- `"d"` (`float`):\n    Z position in a local North/East/Down coordinate system in meters\n\n#### AvrVioVelocityNedPayload\n\nTopic: `avr/vio/velocity/ned`\n\nThis reports the measued velocity vectors of the drone from the tracking camera.\n\n- `"n"` (`float`):\n    X velocity in a local North/East/Down coordinate system in meters per second\n- `"e"` (`float`):\n    Y velocity in a local North/East/Down coordinate system in meters per second\n- `"d"` (`float`):\n    Z velocity in a local North/East/Down coordinate system in meters per second\n\n#### AvrVioOrientationEulPayload\n\nTopic: `avr/vio/orientation/eul`\n\nThis reports the measued attitude of the drone from the tracking camera.\n\n- `"psi"` (`float`):\n    Roll in radians\n- `"theta"` (`float`):\n    Pitch in radians\n- `"phi"` (`float`):\n    Yaw in radians\n\n#### AvrVioOrientationQuatPayload\n\nTopic: `avr/vio/orientation/quat`\n\nThis reports the measued attitude of the drone from the tracking camera.\n\n- `"w"` (`float`):\n    Quaternion w value\n- `"x"` (`float`):\n    Quaternion x value\n- `"y"` (`float`):\n    Quaternion y value\n- `"z"` (`float`):\n    Quaternion z value\n\n#### AvrVioHeadingPayload\n\nTopic: `avr/vio/heading`\n\nThis reports the measued heading of the drone from the tracking camera.\n\n- `"degrees"` (`float`):\n    Heading in degrees\n\n#### AvrVioConfidencePayload\n\nTopic: `avr/vio/confidence`\n\nThis reports the tracking camera\'s confidence\n\n- `"tracker"` (`float`):\n    Number between 0 and 100 of tracking confidence\n\n#### AvrApriltagsSelectedPayload\n\nTopic: `avr/apriltags/selected`\n\nThis topic publishes its best candidate for position feedback\n\n- `"tag_id"` (`int`):\n    The id of the tag\n- `"pos"` (`AvrApriltagsSelectedPos`):\n    The position of the vehicle in world frame in centimeters\n- `"heading"` (`float`)\n\n#### AvrApriltagsRawPayload\n\nTopic: `avr/apriltags/raw`\n\nThis topic publishes the raw AprilTag data\n\n- `"tags"` (`List[AvrApriltagsRawTags]`)\n\n#### AvrApriltagsVisiblePayload\n\nTopic: `avr/apriltags/visible`\n\nThis topic publishes the transformed AprilTag data\n\n- `"tags"` (`List[AvrApriltagsVisibleTags]`)\n\n#### AvrApriltagsFpsPayload\n\nTopic: `avr/apriltags/fps`\n\nThis reports the framerate of AprilTag processing\n\n- `"fps"` (`int`):\n    Number of frames of video data processed in the last second\n\n#### AvrThermalReadingPayload\n\nTopic: `avr/thermal/reading`\n\nThis publishes data from the thermal camera\n\n- `"data"` (`str`):\n    The raw data from the thermal camera are integer values from an 8x8 grid of pixels. This data is then converted into a bytearray and base64 encoded. Any example of how to unpack this data:\n    \n    ```python\n    import base64\n    import json\n    \n    data = json.loads(payload)["data"]\n    base64_decoded = data.encode("utf-8")\n    as_bytes = base64.b64decode(base64_decoded)\n    pixel_ints = list(bytearray(as_bytes))\n    ```\n\n#### AvrStatusLightPcmPayload\n\nTopic: `avr/status/light/pcm`\n\nThere is no content for this class\n\n#### AvrStatusLightVioPayload\n\nTopic: `avr/status/light/vio`\n\nThere is no content for this class\n\n#### AvrStatusLightApriltagsPayload\n\nTopic: `avr/status/light/apriltags`\n\nThere is no content for this class\n\n#### AvrStatusLightFcmPayload\n\nTopic: `avr/status/light/fcm`\n\nThere is no content for this class\n\n#### AvrStatusLightThermalPayload\n\nTopic: `avr/status/light/thermal`\n\nThere is no content for this class\n\n#### AvrAutonomousEnablePayload\n\nTopic: `avr/autonomous/enable`\n\nThis enables or disables autonomous mode. This is not used by any Bell code, but available to students to.\n\n- `"enabled"` (`bool`)\n\n#### AvrAutonomousBuildingDropPayload\n\nTopic: `avr/autonomous/building/drop`\n\nThis enables or disables a building payload drop. This is not used by any Bell code, but available to students to listen to.\n\n- `"id"` (`int`):\n    0-index ID of the relevant building\n- `"enabled"` (`bool`):\n    Boolean of whether the building should have drop enabled\n',
    'author': 'Chris Padilla',
    'author_email': 'cpadilla@bellflight.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/bellflight/AVR-Python-Libraries',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<3.11',
}


setup(**setup_kwargs)
