################################################################################
# Copyright (c) 2021 ContinualAI.                                              #
# Copyrights licensed under the MIT License.                                   #
# See the accompanying LICENSE file for terms.                                 #
#                                                                              #
# Date: 20-05-2020                                                             #
# Author: Vincenzo Lomonaco                                                    #
# E-mail: contact@continualai.org                                              #
# Website: continualai.org                                                     #
################################################################################
from pathlib import Path
from typing import Union, Optional, Any

from torchvision.transforms import Compose, ToTensor, Resize

from avalanche.benchmarks.classic.classic_benchmarks_utils import \
    check_vision_benchmark
from avalanche.benchmarks.datasets import CUB200
from avalanche.benchmarks import nc_benchmark

from torchvision import transforms


_default_train_transform = transforms.Compose([
        transforms.RandomHorizontalFlip(),
        transforms.ToTensor(),
        transforms.Normalize((0.4914, 0.4822, 0.4465),
                             (0.2023, 0.1994, 0.2010))
])

_default_eval_transform = transforms.Compose([
    transforms.ToTensor(),
    transforms.Normalize((0.4914, 0.4822, 0.4465),
                         (0.2023, 0.1994, 0.2010))
])


def SplitCUB200(
        n_experiences=11,
        *,
        classes_first_batch=100,
        return_task_id=False,
        seed=0,
        fixed_class_order=None,
        shuffle=False,
        train_transform: Optional[Any] = _default_train_transform,
        eval_transform: Optional[Any] = _default_eval_transform,
        dataset_root: Union[str, Path] = None):
    """
    Creates a CL benchmark using the Cub-200 dataset.

    If the dataset is not present in the computer, **this method will NOT be
    able automatically download** and store it.

    The returned benchmark will return experiences containing all patterns of a
    subset of classes, which means that each class is only seen "once".
    This is one of the most common scenarios in the Continual Learning
    literature. Common names used in literature to describe this kind of
    scenario are "Class Incremental", "New Classes", etc. By default,
    an equal amount of classes will be assigned to each experience.

    This generator doesn't force a choice on the availability of task labels,
    a choice that is left to the user (see the `return_task_id` parameter for
    more info on task labels).

    The benchmark instance returned by this method will have two fields,
    `train_stream` and `test_stream`, which can be iterated to obtain
    training and test :class:`Experience`. Each Experience contains the
    `dataset` and the associated task label.

    The benchmark API is quite simple and is uniform across all benchmark
    generators. It is recommended to check the tutorial of the "benchmark" API,
    which contains usage examples ranging from "basic" to "advanced".

    :param n_experiences: The number of experiences in the current benchmark.
        Defaults to 11.
    :param classes_first_batch: Number of classes in the first batch.
        Usually this is set to 500. Defaults to 100.
    :param return_task_id: if True, a progressive task id is returned for every
        experience. If False, all experiences will have a task ID of 0.
    :param seed: A valid int used to initialize the random number generator.
        Can be None.
    :param fixed_class_order: A list of class IDs used to define the class
        order. If None, value of ``seed`` will be used to define the class
        order. If non-None, ``seed`` parameter will be ignored.
        Defaults to None.
    :param shuffle: If true, the class order in the incremental experiences is
        randomly shuffled. Default to false.
    :param train_transform: The transformation to apply to the training data,
        e.g. a random crop, a normalization or a concatenation of different
        transformations (see torchvision.transform documentation for a
        comprehensive list of possible transformations).
        If no transformation is passed, the default train transformation
        will be used.
    :param eval_transform: The transformation to apply to the test data,
        e.g. a random crop, a normalization or a concatenation of different
        transformations (see torchvision.transform documentation for a
        comprehensive list of possible transformations).
        If no transformation is passed, the default test transformation
        will be used.
    :param dataset_root: The root path of the dataset.
        Defaults to None, which means that the default location for
        'CUB_200_2011' will be used.

    :returns: A properly initialized :class:`NCScenario` instance.
    """

    train_set, test_set = _get_cub200_dataset(dataset_root)

    if classes_first_batch is not None:
        per_exp_classes = {0: classes_first_batch}
    else:
        per_exp_classes = None

    if return_task_id:
        return nc_benchmark(
            train_dataset=train_set,
            test_dataset=test_set,
            n_experiences=n_experiences,
            task_labels=True,
            per_exp_classes=per_exp_classes,
            seed=seed,
            fixed_class_order=fixed_class_order,
            shuffle=shuffle,
            one_dataset_per_exp=True,
            train_transform=train_transform,
            eval_transform=eval_transform)
    else:
        return nc_benchmark(
            train_dataset=train_set,
            test_dataset=test_set,
            n_experiences=n_experiences,
            task_labels=False,
            per_exp_classes=per_exp_classes,
            seed=seed,
            fixed_class_order=fixed_class_order,
            shuffle=shuffle,
            train_transform=train_transform,
            eval_transform=eval_transform)


def _get_cub200_dataset(root):
    train_set = CUB200(root, train=True)
    test_set = CUB200(root, train=False)

    return train_set, test_set


__all__ = [
    'SplitCUB200'
]

if __name__ == "__main__":
    import sys

    benchmark_instance = SplitCUB200(
        5,
        train_transform=Compose([
            ToTensor(), Resize((128, 128))
        ]))
    check_vision_benchmark(benchmark_instance, show_without_transforms=False)
    sys.exit(0)
