"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findUpMultiple = exports.callsites = exports.LockFile = exports.NodejsEdgeFunction = void 0;
const fs = require("fs");
const path = require("path");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const lambda = require("aws-cdk-lib/aws-lambda");
const bundling_1 = require("./lib/bundling");
class NodejsEdgeFunction extends cloudfront.experimental.EdgeFunction {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        const handler = (_a = props.handler) !== null && _a !== void 0 ? _a : 'handler';
        const runtime = (_b = props.runtime) !== null && _b !== void 0 ? _b : lambda.Runtime.NODEJS_14_X;
        const entry = path.resolve(findEntry(id, props.entry));
        const architecture = (_c = props.architecture) !== null && _c !== void 0 ? _c : lambda.Architecture.X86_64;
        const depsLockFilePath = findLockFile(props.depsLockFilePath);
        const projectRoot = (_d = props.projectRoot) !== null && _d !== void 0 ? _d : path.dirname(depsLockFilePath);
        super(scope, id, {
            ...props,
            runtime,
            stackId: props.stackId,
            code: bundling_1.Bundling.bundle({
                ...(_e = props.bundling) !== null && _e !== void 0 ? _e : {},
                architecture,
                runtime,
                depsLockFilePath,
                entry,
                projectRoot,
            }),
            handler: `index.${handler}`,
        });
    }
}
exports.NodejsEdgeFunction = NodejsEdgeFunction;
/**
 * Checks given lock file or searches for a lock file
 */
function findLockFile(depsLockFilePath) {
    if (depsLockFilePath) {
        if (!fs.existsSync(depsLockFilePath)) {
            throw new Error(`Lock file at ${depsLockFilePath} doesn't exist`);
        }
        if (!fs.statSync(depsLockFilePath).isFile()) {
            throw new Error('`depsLockFilePath` should point to a file');
        }
        return path.resolve(depsLockFilePath);
    }
    const lockFiles = findUpMultiple([
        LockFile.PNPM,
        LockFile.YARN,
        LockFile.NPM,
    ]);
    if (lockFiles.length === 0) {
        throw new Error('Cannot find a package lock file (`pnpm-lock.yaml`, `yarn.lock` or `package-lock.json`). Please specify it with `depsFileLockPath`.');
    }
    if (lockFiles.length > 1) {
        throw new Error(`Multiple package lock files found: ${lockFiles.join(', ')}. Please specify the desired one with \`depsFileLockPath\`.`);
    }
    return lockFiles[0];
}
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
    LockFile["PNPM"] = "pnpm-lock.yaml";
})(LockFile = exports.LockFile || (exports.LockFile = {}));
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 * 4. A .mjs file name as the defining file with id as suffix (defining-file.id.mjs)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(jsx?|tsx?|mjs)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    const mjsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.mjs`);
    if (fs.existsSync(mjsHandlerFile)) {
        return mjsHandlerFile;
    }
    throw new Error(`Cannot find handler file ${tsHandlerFile}, ${jsHandlerFile} or ${mjsHandlerFile}`);
}
/**
* Finds the name of the file where the `NodejsFunction` is defined
*/
function findDefiningFile() {
    let definingIndex;
    const sites = callsites();
    for (const [index, site] of sites.entries()) {
        if (site.getFunctionName() === 'NodejsEdgeFunction') {
            // The next site is the site where the NodejsFunction was created
            definingIndex = index + 1;
            break;
        }
    }
    if (!definingIndex || !sites[definingIndex]) {
        throw new Error('Cannot find defining file.');
    }
    return sites[definingIndex].getFileName();
}
/**
 * Get callsites from the V8 stack trace API
 *
 * https://github.com/sindresorhus/callsites
 */
function callsites() {
    var _a;
    const _prepareStackTrace = Error.prepareStackTrace;
    Error.prepareStackTrace = (_, stack) => stack;
    const stack = (_a = new Error().stack) === null || _a === void 0 ? void 0 : _a.slice(1);
    Error.prepareStackTrace = _prepareStackTrace;
    return stack;
}
exports.callsites = callsites;
/**
 * Find the lowest of multiple files by walking up parent directories. If
 * multiple files exist at the same level, they will all be returned.
 */
function findUpMultiple(names, directory = process.cwd()) {
    const absoluteDirectory = path.resolve(directory);
    const files = [];
    for (const name of names) {
        const file = path.join(directory, name);
        if (fs.existsSync(file)) {
            files.push(file);
        }
    }
    if (files.length > 0) {
        return files;
    }
    const { root } = path.parse(absoluteDirectory);
    if (absoluteDirectory === root) {
        return [];
    }
    return findUpMultiple(names, path.dirname(absoluteDirectory));
}
exports.findUpMultiple = findUpMultiple;
//# sourceMappingURL=data:application/json;base64,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