/*
 * Copyright (c) Jupyter Development Team.
 * Distributed under the terms of the Modified BSD License.
 */
import React, { useState } from 'react';
import { caretDownIcon, caretUpIcon } from '../icon';
export const TABLE_CLASS = 'jp-sortable-table';
/**
 * Sortable table component for small datasets.
 *
 * For large datasets use `DataGrid` from `@lumino/datagrid`.
 */
export function Table(props) {
    const [sortState, setSortState] = useState({
        sortKey: props.sortKey,
        sortDirection: props.sortDirection || 1
    });
    const sort = (key) => {
        if (key === sortState.sortKey) {
            setSortState({
                sortKey: key,
                sortDirection: (sortState.sortDirection * -1)
            });
        }
        else {
            setSortState({ sortKey: key, sortDirection: 1 });
        }
    };
    let rows = props.rows;
    const sortedColumn = props.columns.filter(column => column.id === sortState.sortKey)[0];
    if (sortedColumn) {
        const sorter = sortedColumn.sort.bind(sortedColumn);
        rows = props.rows.sort((a, b) => sorter(a.data, b.data) * sortState.sortDirection);
    }
    const visibleColumns = props.columns.filter(column => (column.isAvailable ? column.isAvailable() : true) && !column.isHidden);
    const elements = rows.map(row => {
        const cells = visibleColumns.map(column => (React.createElement("td", { key: column.id + '-' + row.key }, column.renderCell(row.data))));
        return (React.createElement("tr", { key: row.key, "data-key": row.key, onClick: props.onRowClick, className: 'jp-sortable-table-tr' }, cells));
    });
    const columnsHeaders = visibleColumns.map(column => (React.createElement(SortableTH, { label: column.label, id: column.id, state: sortState, key: column.id, onSort: () => {
            sort(column.id);
        } })));
    return (React.createElement("table", { className: TABLE_CLASS },
        React.createElement("thead", null,
            React.createElement("tr", { className: 'jp-sortable-table-tr' }, columnsHeaders)),
        React.createElement("tbody", null, elements)));
}
function SortableTH(props) {
    const isSortKey = props.id === props.state.sortKey;
    const sortIcon = !isSortKey || props.state.sortDirection === 1 ? caretUpIcon : caretDownIcon;
    return (React.createElement("th", { key: props.id, onClick: () => props.onSort(), className: isSortKey ? 'jp-sorted-header' : undefined, "data-id": props.id },
        React.createElement("div", { className: "jp-sortable-table-th-wrapper" },
            React.createElement("label", null, props.label),
            React.createElement(sortIcon.react, { tag: "span", className: "jp-sort-icon" }))));
}
//# sourceMappingURL=table.js.map