# constants_test.py
# Copyright 2012 Roger Marsh
# Licence: See LICENCE (BSD licence)

"""constants tests"""

import unittest
import re

from .. import constants


class PGNParserConstants(unittest.TestCase):

    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_values(self):
        """"""
        self.assertEqual(constants.TAG_EVENT, 'Event')
        self.assertEqual(constants.TAG_SITE, 'Site')
        self.assertEqual(constants.TAG_DATE, 'Date')
        self.assertEqual(constants.TAG_ROUND, 'Round')
        self.assertEqual(constants.TAG_WHITE, 'White')
        self.assertEqual(constants.TAG_BLACK, 'Black')
        self.assertEqual(constants.TAG_RESULT, 'Result')
        self.assertEqual(constants.TAG_FEN, 'FEN')
        self.assertEqual(constants.SEVEN_TAG_ROSTER,
                         {'Event': '?',
                          'Result': '*',
                          'Site': '?',
                          'Black': '?',
                          'Date': '????.??.??',
                          'Round': '?',
                          'White': '?'})
        self.assertEqual(
            constants.SEVEN_TAG_ROSTER_DISPLAY_ORDER,
            ('Site', 'Round', 'Event', 'Date', 'White', 'Black', 'Result'))
        self.assertEqual(
            constants.SEVEN_TAG_ROSTER_EXPORT_ORDER,
            ('Event', 'Site', 'Date', 'Round', 'White', 'Black', 'Result'))
        self.assertEqual(constants.SPECIAL_TAG_DATE, ('?', '0'))
        self.assertEqual(constants.SPECIAL_TAG_ROUND, {'-': 2, '?': 1})
        self.assertEqual(constants.NORMAL_TAG_ROUND, 3)
        self.assertEqual(constants.SEVEN_TAG_ROSTER_ARCHIVE_SORT1,
                         ('Event', 'Site', 'Date'))
        self.assertEqual(constants.SEVEN_TAG_ROSTER_ARCHIVE_SORT2,
                         ('Round', 'White', 'Black', 'Result'))
        self.assertEqual(constants.TAG_OPENING, 'Opening')
        self.assertEqual(constants.REPERTOIRE_TAG_ORDER, ('Opening', 'Result'))
        self.assertEqual(constants.REPERTOIRE_GAME_TAGS,
                         {'Result': '*', 'Opening': '?'})
        self.assertEqual(constants.WHITE_WIN, '1-0')
        self.assertEqual(constants.BLACK_WIN, '0-1')
        self.assertEqual(constants.DRAW, '1/2-1/2')
        self.assertEqual(constants.UNKNOWN_RESULT, '*')
        self.assertEqual(constants.RESULT_SET, {'1/2-1/2', '1-0', '*', '0-1'})
        self.assertEqual(constants.PGN_PAWN, '')
        self.assertEqual(constants.PGN_KING, 'K')
        self.assertEqual(constants.PGN_QUEEN, 'Q')
        self.assertEqual(constants.PGN_ROOK, 'R')
        self.assertEqual(constants.PGN_BISHOP, 'B')
        self.assertEqual(constants.PGN_KNIGHT, 'N')
        self.assertEqual(constants.PGN_FROM_SQUARE_DISAMBIGUATION,
                         frozenset({'Q', 'N', 'B'}))
        self.assertEqual(constants.O_O_O, 'O-O-O')
        self.assertEqual(constants.O_O, 'O-O')
        self.assertEqual(constants.PLAIN_MOVE, '')
        self.assertEqual(constants.CAPTURE_MOVE, 'x')
        self.assertEqual(constants.LINEFEED, '\n')
        self.assertEqual(constants.CARRIAGE_RETURN, '\r')
        self.assertEqual(constants.NEWLINE, '\n\r')
        self.assertEqual(constants.SPACE, ' ')
        self.assertEqual(constants.HORIZONTAL_TAB, '\t')
        self.assertEqual(constants.FORMFEED, '\f')
        self.assertEqual(constants.VERTICAL_TAB, '\v')
        self.assertEqual(constants.ANYTHING_ELSE, '.')
        self.assertEqual(constants.WHITESPACE, '\s+')
        self.assertEqual(constants.FULLSTOP, '.')
        self.assertEqual(constants.PERIOD, '\.')
        self.assertEqual(constants.INTEGER, '[1-9][0-9]*')
        self.assertEqual(constants.TERMINATION, '1-0|0-1|1/2-1/2|\*')
        self.assertEqual(constants.START_TAG, '[')
        self.assertEqual(constants.END_TAG, ']')
        self.assertEqual(constants.SYMBOL, '([A-Za-z0-9][A-Za-z0-9_+#=:-]*)')
        self.assertEqual(constants.STRING, r'"((?:[^\\"]|\\.)*)"')
        self.assertEqual(constants.TAG_PAIR,
                         ''.join(('(\[)\s*([A-Za-z0-9][A-Za-z0-9_+#=:-]*)\s*',
                                  r'"((?:[^\\"]|\\.)*)"',
                                  '\s*(\])')))
        self.assertEqual(constants.START_COMMENT, '{')
        self.assertEqual(constants.END_COMMENT, '}')
        self.assertEqual(constants.COMMENT, '\{[^}]*\}')
        self.assertEqual(constants.LEFT_ANGLE_BRACE, '<')
        self.assertEqual(constants.RIGHT_ANGLE_BRACE, '>')
        self.assertEqual(constants.RESERVED, '<[^>]*>')
        self.assertEqual(constants.COMMENT_TO_EOL, ';(?:[^\n]*)\n')
        self.assertEqual(constants.ESCAPE_LINE, '(?:\A|(?<=\n))%(?:[^\n]*)\n')
        self.assertEqual(constants.NAG, '\$[0-9]+(?!/|-)')
        self.assertEqual(constants.START_RAV, '(')
        self.assertEqual(constants.END_RAV, ')')
        self.assertEqual(constants.PAWN_PROMOTE,
                         '(?:([a-h])(x))?([a-h][18])(=[BNQR])')
        self.assertEqual(constants.PAWN_CAPTURE, '([a-h])(x)([a-h][2-7])')
        self.assertEqual(constants.PIECE_CAPTURE,
                         '(?:(K)|(?:([BNQR])([a-h]?[1-8]?)))(x)([a-h][1-8])')
        self.assertEqual(constants.PIECE_CHOICE_MOVE,
                         '([BNQR])([a-h1-8])([a-h][1-8])')
        self.assertEqual(constants.PIECE_MOVE, '([KBNQR])([a-h][1-8])')
        self.assertEqual(constants.PAWN_MOVE, '([a-h][1-8])')
        self.assertEqual(constants.CASTLES, '(O-O(?:-O)?)')
        self.assertEqual(constants.CHECK, '([+#]?)')
        self.assertEqual(constants.ANNOTATION, '([!?][!?]?)?')
        self.assertEqual(constants.IMPORT_FORMAT,
                         ''.join(('(?:',
                                  ''.join((
                                      '(\[)\s*([A-Za-z0-9][A-Za-z0-9_+#=:-]*)',
                                      '\s*',
                                      r'"((?:[^\\"]|\\.)*)"',
                                      '\s*(\])')),
                                  ')|',
                                  '(?:',
                                  '(?:',
                                  '(?:(?:([a-h])(x))?([a-h][18])(=[BNQR]))|',
                                  '(?:([a-h])(x)([a-h][2-7]))|',
                                  '(?:',
                                  ''.join(('(?:',
                                           '(K)|(?:([BNQR])([a-h]?[1-8]?)))',
                                           '(x)([a-h][1-8])', ')|')),
                                  '(?:([BNQR])([a-h1-8])([a-h][1-8]))|',
                                  '(?:([KBNQR])([a-h][1-8]))|',
                                  '(?:([a-h][1-8]))|',
                                  '(?:(O-O(?:-O)?))',
                                  ')',
                                  '(?:([+#]?))',
                                  '(?:([!?][!?]?)?)',
                                  ')|',
                                  '(\{[^}]*\}', ')|',
                                  '(\$[0-9]+(?!/|-)', ')|',
                                  '(;(?:[^\n]*)\n)|',
                                  '(', '1-0|0-1|1/2-1/2|\*)|',
                                  '[1-9][0-9]*|',
                                  '\.|',
                                  '\s+|',
                                  '(\()|',
                                  '(\))|',
                                  '<[^>]*>|',
                                  '(?:\A|(?<=\n))%(?:[^\n]*)\n|',
                                  '(.)',
                                  )))
        self.assertEqual(constants.DISAMBIGUATE_FORMAT,
                         '[BNQ][a-h][1-8][a-h][1-8]')
        self.assertEqual(constants.UNAMBIGUOUS_FORMAT, '.*')
        self.assertEqual(constants.POSSIBLE_MOVE,
                         '[OKBNRQa-h][-Oa-h1-8+#?!=]* *')
        self.assertEqual(constants.IFG_START_TAG, 1)
        self.assertEqual(constants.IFG_TAG_SYMBOL, 2)
        #self.assertEqual(constants.IFG_TAG_STRING, 3)
        self.assertEqual(constants.IFG_TAG_STRING_VALUE, 3)
        #self.assertEqual(constants.IFG_TAG_END, 5)
        self.assertEqual(constants.IFG_PAWN_PROMOTE_FROM_FILE, 5)
        self.assertEqual(constants.IFG_PAWN_TAKES_PROMOTE, 6)
        self.assertEqual(constants.IFG_PAWN_PROMOTE_SQUARE, 7)
        self.assertEqual(constants.IFG_PAWN_PROMOTE_PIECE, 8)
        self.assertEqual(constants.IFG_PAWN_CAPTURE_FROM_FILE, 9)
        self.assertEqual(constants.IFG_PAWN_TAKES, 10)
        self.assertEqual(constants.IFG_PAWN_CAPTURE_SQUARE, 11)
        self.assertEqual(constants.IFG_KING_CAPTURE, 12)
        self.assertEqual(constants.IFG_PIECE_CAPTURE, 13)
        self.assertEqual(constants.IFG_PIECE_CAPTURE_FROM, 14)
        self.assertEqual(constants.IFG_PIECE_TAKES, 15)
        self.assertEqual(constants.IFG_PIECE_CAPTURE_SQUARE, 16)
        self.assertEqual(constants.IFG_PIECE_CHOICE, 17)
        self.assertEqual(constants.IFG_PIECE_CHOICE_FILE_OR_RANK, 18)
        self.assertEqual(constants.IFG_PIECE_CHOICE_SQUARE, 19)
        self.assertEqual(constants.IFG_PIECE_MOVE, 20)
        self.assertEqual(constants.IFG_PIECE_SQUARE, 21)
        self.assertEqual(constants.IFG_PAWN_SQUARE, 22)
        self.assertEqual(constants.IFG_CASTLES, 23)
        self.assertEqual(constants.IFG_CHECK, 24)
        self.assertEqual(constants.IFG_ANNOTATION, 25)
        self.assertEqual(constants.IFG_COMMENT, 26)
        self.assertEqual(constants.IFG_NAG, 27)
        self.assertEqual(constants.IFG_COMMENT_TO_EOL, 28)
        self.assertEqual(constants.IFG_TERMINATION, 29)
        self.assertEqual(constants.IFG_START_RAV, 30)
        self.assertEqual(constants.IFG_END_RAV, 31)
        self.assertEqual(constants.IFG_ANYTHING_ELSE, 32)
        self.assertEqual(constants.PGN_SEARCHING, 0)
        self.assertEqual(constants.PGN_SEARCHING_AFTER_ERROR_IN_RAV, 1)
        self.assertEqual(constants.PGN_SEARCHING_AFTER_ERROR_IN_GAME, 2)
        self.assertEqual(constants.PGN_COLLECTING_TAG_PAIRS, 3)
        self.assertEqual(constants.PGN_COLLECTING_MOVETEXT, 4)
        self.assertEqual(
            constants.PGN_COLLECTING_NON_WHITESPACE_WHILE_SEARCHING, 5)
        self.assertEqual(constants.PGN_DISAMBIGUATE_MOVE, 6)
        self.assertEqual(
            [p[1:] for p in sorted([(int(k[1:]), k, v)
                             for k, v in constants.NAG_TRANSLATION.items()])],
            [('$1', None),
             ('$2', None),
             ('$3', None),
             ('$4', None),
             ('$5', None),
             ('$6', None),
             ('$7', None),
             ('$8', None),
             ('$9', None),
             ('$10', None),
             ('$11', None),
             ('$12', None),
             ('$13', None),
             ('$14', None),
             ('$15', None),
             ('$16', None),
             ('$17', None),
             ('$18', None),
             ('$19', None),
             ('$20', None),
             ('$21', None),
             ('$22', None),
             ('$23', None),
             ('$24', None),
             ('$25', None),
             ('$26', None),
             ('$27', None),
             ('$28', None),
             ('$29', None),
             ('$30', None),
             ('$31', None),
             ('$32', None),
             ('$33', None),
             ('$34', None),
             ('$35', None),
             ('$36', None),
             ('$37', None),
             ('$38', None),
             ('$39', None),
             ('$40', None),
             ('$41', None),
             ('$42', None),
             ('$43', None),
             ('$44', None),
             ('$45', None),
             ('$46', None),
             ('$47', None),
             ('$48', None),
             ('$49', None),
             ('$50', None),
             ('$51', None),
             ('$52', None),
             ('$53', None),
             ('$54', None),
             ('$55', None),
             ('$56', None),
             ('$57', None),
             ('$58', None),
             ('$59', None),
             ('$60', None),
             ('$61', None),
             ('$62', None),
             ('$63', None),
             ('$64', None),
             ('$65', None),
             ('$66', None),
             ('$67', None),
             ('$68', None),
             ('$69', None),
             ('$70', None),
             ('$71', None),
             ('$72', None),
             ('$73', None),
             ('$74', None),
             ('$75', None),
             ('$76', None),
             ('$77', None),
             ('$78', None),
             ('$79', None),
             ('$80', None),
             ('$81', None),
             ('$82', None),
             ('$83', None),
             ('$84', None),
             ('$85', None),
             ('$86', None),
             ('$87', None),
             ('$88', None),
             ('$89', None),
             ('$90', None),
             ('$91', None),
             ('$92', None),
             ('$93', None),
             ('$94', None),
             ('$95', None),
             ('$96', None),
             ('$97', None),
             ('$98', None),
             ('$99', None),
             ('$100', None),
             ('$101', None),
             ('$102', None),
             ('$103', None),
             ('$104', None),
             ('$105', None),
             ('$106', None),
             ('$107', None),
             ('$108', None),
             ('$109', None),
             ('$110', None),
             ('$111', None),
             ('$112', None),
             ('$113', None),
             ('$114', None),
             ('$115', None),
             ('$116', None),
             ('$117', None),
             ('$118', None),
             ('$119', None),
             ('$120', None),
             ('$121', None),
             ('$122', None),
             ('$123', None),
             ('$124', None),
             ('$125', None),
             ('$126', None),
             ('$127', None),
             ('$128', None),
             ('$129', None),
             ('$130', None),
             ('$131', None),
             ('$132', None),
             ('$133', None),
             ('$134', None),
             ('$135', None),
             ('$136', None),
             ('$137', None),
             ('$138', None),
             ('$139', None),
             ('$140', None),
             ('$141', None),
             ('$142', None),
             ('$143', None),
             ('$144', None),
             ('$145', None),
             ('$146', None),
             ('$147', None),
             ('$148', None),
             ('$149', None),
             ('$150', None),
             ('$151', None),
             ('$152', None),
             ('$153', None),
             ('$154', None),
             ('$155', None),
             ('$156', None),
             ('$157', None),
             ('$158', None),
             ('$159', None),
             ('$160', None),
             ('$161', None),
             ('$162', None),
             ('$163', None),
             ('$164', None),
             ('$165', None),
             ('$166', None),
             ('$167', None),
             ('$168', None),
             ('$169', None),
             ('$170', None),
             ('$171', None),
             ('$172', None),
             ('$173', None),
             ('$174', None),
             ('$175', None),
             ('$176', None),
             ('$177', None),
             ('$178', None),
             ('$179', None),
             ('$180', None),
             ('$181', None),
             ('$182', None),
             ('$183', None),
             ('$184', None),
             ('$185', None),
             ('$186', None),
             ('$187', None),
             ('$188', None),
             ('$189', None),
             ('$190', None),
             ('$191', None),
             ('$192', None),
             ('$193', None),
             ('$194', None),
             ('$195', None),
             ('$196', None),
             ('$197', None),
             ('$198', None),
             ('$199', None),
             ('$200', None),
             ('$201', None),
             ('$202', None),
             ('$203', None),
             ('$204', None),
             ('$205', None),
             ('$206', None),
             ('$207', None),
             ('$208', None),
             ('$209', None),
             ('$210', None),
             ('$211', None),
             ('$212', None),
             ('$213', None),
             ('$214', None),
             ('$215', None),
             ('$216', None),
             ('$217', None),
             ('$218', None),
             ('$219', None),
             ('$220', None),
             ('$221', None),
             ('$222', None),
             ('$223', None),
             ('$224', None),
             ('$225', None),
             ('$226', None),
             ('$227', None),
             ('$228', None),
             ('$229', None),
             ('$230', None),
             ('$231', None),
             ('$232', None),
             ('$233', None),
             ('$234', None),
             ('$235', None),
             ('$236', None),
             ('$237', None),
             ('$238', None),
             ('$239', None),
             ('$240', None),
             ('$241', None),
             ('$242', None),
             ('$243', None),
             ('$244', None),
             ('$245', None),
             ('$246', None),
             ('$247', None),
             ('$248', None),
             ('$249', None),
             ('$250', None),
             ('$251', None),
             ('$252', None),
             ('$253', None),
             ('$254', None),
             ('$255', None),
             ('$256', None),
             ('$257', None),
             ('$258', None),
             ('$259', None),
             ('$260', None),
             ('$261', None),
             ('$262', None),
             ('$263', None),
             ('$264', None),
             ('$265', None),
             ('$266', None),
             ('$267', None),
             ('$268', None),
             ('$269', None),
             ('$270', None),
             ('$271', None),
             ('$272', None),
             ('$273', None),
             ('$274', None),
             ('$275', None),
             ('$276', None),
             ('$277', None),
             ('$278', None),
             ('$279', None),
             ('$280', None),
             ('$281', None),
             ('$282', None),
             ('$283', None),
             ('$284', None),
             ('$285', None),
             ('$286', None),
             ('$287', None),
             ('$288', None),
             ('$289', None),
             ('$290', None),
             ('$291', None),
             ('$292', None),
             ('$293', None),
             ('$294', None),
             ('$295', None),
             ('$296', None),
             ('$297', None),
             ('$298', None),
             ('$299', None),
             ('$300', None),
             ('$301', None),
             ('$302', None),
             ('$303', None),
             ('$304', None),
             ('$305', None),
             ('$306', None),
             ('$307', None),
             ('$308', None),
             ('$309', None),
             ('$310', None),
             ('$311', None),
             ('$312', None),
             ('$313', None),
             ('$314', None),
             ('$315', None),
             ('$316', None),
             ('$317', None),
             ('$318', None),
             ('$319', None),
             ('$320', None),
             ('$321', None),
             ('$322', None),
             ('$323', None),
             ('$324', None),
             ('$325', None),
             ('$326', None),
             ('$327', None),
             ('$328', None),
             ('$329', None),
             ('$330', None),
             ('$331', None),
             ('$332', None),
             ('$333', None),
             ('$334', None),
             ('$335', None),
             ('$336', None),
             ('$337', None),
             ('$338', None),
             ('$339', None),
             ('$340', None),
             ('$341', None),
             ('$342', None),
             ('$343', None),
             ('$344', None),
             ('$345', None),
             ('$346', None),
             ('$347', None),
             ('$348', None),
             ('$349', None),
             ('$350', None),
             ('$351', None),
             ('$352', None),
             ('$353', None),
             ('$354', None),
             ('$355', None),
             ('$356', None),
             ('$357', None),
             ('$358', None),
             ('$359', None),
             ('$360', None),
             ('$361', None),
             ('$362', None),
             ('$363', None),
             ('$364', None),
             ('$365', None),
             ('$366', None),
             ('$367', None),
             ('$368', None),
             ('$369', None),
             ('$370', None),
             ('$371', None),
             ('$372', None),
             ('$373', None),
             ('$374', None),
             ('$375', None),
             ('$376', None),
             ('$377', None),
             ('$378', None),
             ('$379', None),
             ('$380', None),
             ('$381', None),
             ('$382', None),
             ('$383', None),
             ('$384', None),
             ('$385', None),
             ('$386', None),
             ('$387', None),
             ('$388', None),
             ('$389', None),
             ('$390', None),
             ('$391', None),
             ('$392', None),
             ('$393', None),
             ('$394', None),
             ('$395', None),
             ('$396', None),
             ('$397', None),
             ('$398', None),
             ('$399', None),
             ('$400', None),
             ('$401', None),
             ('$402', None),
             ('$403', None),
             ('$404', None),
             ('$405', None),
             ('$406', None),
             ('$407', None),
             ('$408', None),
             ('$409', None),
             ('$410', None),
             ('$411', None),
             ('$412', None),
             ('$413', None),
             ('$414', None),
             ('$415', None),
             ('$416', None),
             ('$417', None),
             ('$418', None),
             ('$419', None),
             ('$420', None),
             ('$421', None),
             ('$422', None),
             ('$423', None),
             ('$424', None),
             ('$425', None),
             ('$426', None),
             ('$427', None),
             ('$428', None),
             ('$429', None),
             ('$430', None),
             ('$431', None),
             ('$432', None),
             ('$433', None),
             ('$434', None),
             ('$435', None),
             ('$436', None),
             ('$437', None),
             ('$438', None),
             ('$439', None),
             ('$440', None),
             ('$441', None),
             ('$442', None),
             ('$443', None),
             ('$444', None),
             ('$445', None),
             ('$446', None),
             ('$447', None),
             ('$448', None),
             ('$449', None),
             ('$450', None),
             ('$451', None),
             ('$452', None),
             ('$453', None),
             ('$454', None),
             ('$455', None),
             ('$456', None),
             ('$457', None),
             ('$458', None),
             ('$459', None),
             ('$460', None),
             ('$461', None),
             ('$462', None),
             ('$463', None),
             ('$464', None),
             ('$465', None),
             ('$466', None),
             ('$467', None),
             ('$468', None),
             ('$469', None),
             ('$470', None),
             ('$471', None),
             ('$472', None),
             ('$473', None),
             ('$474', None),
             ('$475', None),
             ('$476', None),
             ('$477', None),
             ('$478', None),
             ('$479', None),
             ('$480', None),
             ('$481', None),
             ('$482', None),
             ('$483', None),
             ('$484', None),
             ('$485', None),
             ('$486', None),
             ('$487', None),
             ('$488', None),
             ('$489', None),
             ('$490', None),
             ('$491', None),
             ('$492', None),
             ('$493', None),
             ('$494', None),
             ('$495', None),
             ('$496', None),
             ('$497', None),
             ('$498', None),
             ])

    def test_import_format_re(self):
        # The re compiles.
        self.assertEqual(bool(re.compile(constants.IMPORT_FORMAT)), True)


class FENConstants(unittest.TestCase):

    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_values(self):
        """"""
        self.assertEqual(constants.WHITE_SIDE, 0)
        self.assertEqual(constants.BLACK_SIDE, 1)
        self.assertEqual(constants.OTHER_SIDE, (1, 0))
        self.assertEqual(constants.SIDE_KING, ('K', 'k'))
        self.assertEqual(constants.FEN_NULL, '-')
        self.assertEqual(constants.FEN_FIELD_DELIM, ' ')
        self.assertEqual(constants.FEN_FIELD_COUNT, 6)
        self.assertEqual(constants.FEN_SIDES, {'w':0, 'b':1})
        self.assertEqual(constants.FEN_TOMOVE, ('w','b'))
        self.assertEqual(constants.FEN_WHITE_MOVE_TO_EN_PASSANT,
                         {'a6': 16,
                          'b6': 17,
                          'c6': 18,
                          'd6': 19,
                          'e6': 20,
                          'f6': 21,
                          'g6': 22,
                          'h6': 23,
                          })
        self.assertEqual(constants.FEN_BLACK_MOVE_TO_EN_PASSANT,
                         {'a3': 40,
                          'b3': 41,
                          'c3': 42,
                          'd3': 43,
                          'e3': 44,
                          'f3': 45,
                          'g3': 46,
                          'h3': 47,
                          })
        self.assertEqual(constants.FEN_WHITE_CAPTURE_EN_PASSANT,
                         {'a6': 24,
                          'b6': 25,
                          'c6': 26,
                          'd6': 27,
                          'e6': 28,
                          'f6': 29,
                          'g6': 30,
                          'h6': 31,
                          })
        self.assertEqual(constants.FEN_BLACK_CAPTURE_EN_PASSANT,
                         {'a3': 32,
                          'b3': 33,
                          'c3': 34,
                          'd3': 35,
                          'e3': 36,
                          'f3': 37,
                          'g3': 38,
                          'h3': 39,
                          })
        self.assertEqual(constants.FEN_EN_PASSANT_TARGET_RANK,
                         {'5':'6', '4':'3'})
        self.assertEqual(constants.FEN_RANK_DELIM, '/')
        self.assertEqual(constants.FEN_CASTLING_OPTION_REPEAT_MAX, 1)
        self.assertEqual(constants.FEN_PIECE_COUNT_PER_SIDE_MAX, 16)
        self.assertEqual(constants.FEN_KING_COUNT, 1)
        self.assertEqual(constants.FEN_PAWN_COUNT_MAX, 8)
        self.assertEqual(constants.FEN_QUEEN_COUNT_INITIAL, 1)
        self.assertEqual(constants.FEN_ROOK_COUNT_INITIAL, 2)
        self.assertEqual(constants.FEN_BISHOP_COUNT_INITIAL, 2)
        self.assertEqual(constants.FEN_KNIGHT_COUNT_INITIAL, 2)
        self.assertEqual(constants.FEN_MAXIMUM_PIECES_GIVING_CHECK, 2)
        self.assertEqual(constants.INITIAL_BOARD,
                         ('r', 'n', 'b', 'q', 'k', 'b', 'n', 'r',
                          'p', 'p', 'p', 'p', 'p', 'p', 'p', 'p',
                          '', '', '', '', '', '', '', '',
                          '', '', '', '', '', '', '', '',
                          '', '', '', '', '', '', '', '',
                          '', '', '', '', '', '', '', '',
                          'P', 'P', 'P', 'P', 'P', 'P', 'P', 'P',
                          'R', 'N', 'B', 'Q', 'K', 'B', 'N', 'R',
                          ))
        self.assertEqual(constants.INITIAL_BOARD_BITMAP,
                         18446462598732906495)
        self.assertEqual(
            constants.INITIAL_OCCUPIED_SQUARES,
            (frozenset((48, 49, 50, 51, 52, 53, 54, 55,
                        56, 57, 58, 59, 60, 61, 62, 63)),
             frozenset((0, 1, 2, 3, 4, 5, 6, 7,
                        8, 9, 10, 11, 12, 13, 14, 15))))
        self.assertEqual(
            constants.INITIAL_PIECE_LOCATIONS,
            {'P':(48, 49, 50, 51, 52, 53, 54, 55),
             'p':(8, 9, 10, 11, 12, 13, 14, 15),
             'K':(60,),
             'k':(4,),
             'Q':(59,),
             'R':(56, 63),
             'B':(58, 61),
             'N':(57, 62),
             'q':(3,),
             'r':(0, 7),
             'b':(2, 5),
             'n':(1, 6),
             })
        self.assertEqual(constants.FEN_INITIAL_CASTLING, 'KQkq')
        self.assertEqual(constants.FEN_INITIAL_HALFMOVE_COUNT, 0)
        self.assertEqual(constants.FEN_INITIAL_FULLMOVE_NUMBER, 1)

    def test_squares(self):
        """"""
        self.assertEqual(constants.BOARDSIDE, 8)
        self.assertEqual(constants.BOARDSQUARES, 64)
        self.assertEqual(constants.ALL_SQUARES, 18446744073709551615)
        self.assertEqual(sum(constants.SQUARE_BITS), 18446744073709551615)
        self.assertEqual(constants.SQUARE_BITS, [1 << i for i in range(64)])
        self.assertEqual(constants.SQUARE_BITS[0], 1)
        self.assertEqual(constants.SQUARE_BITS[-1], 9223372036854775808)
        self.assertEqual(
            constants.MAPFILE,
            {'a':0, 'b':1, 'c':2, 'd':3, 'e':4, 'f':5, 'g':6, 'h':7})
        self.assertEqual(
            constants.MAPRANK,
            {'8':0, '7':8, '6':16, '5':24, '4':32, '3':40, '2':48, '1':56})
        self.assertEqual(
            constants.MAPROW,
            {'8':0, '7':1, '6':2, '5':3, '4':4, '3':5, '2':6, '1':7})
        self.assertEqual(
            constants.MAP_PGN_SQUARE_NAME_TO_FEN_ORDER,
            {'d5': 27, 'g8': 6, 'e8': 4, 'a7': 8, 'h4': 39, 'd4': 35,
             'd3': 43, 'f1': 61, 'c1': 58, 'a1': 56, 'e6': 20, 'd8': 3,
             'h8': 7, 'g4': 38, 'e2': 52, 'f4': 37, 'b8': 1, 'h1': 63,
             'a5': 24, 'h3': 47, 'c5': 26, 'e3': 44, 'a4': 32, 'b1': 57,
             'g1': 62, 'b2': 49, 'b4': 33, 'h7': 15, 'a8': 0, 'f6': 21,
             'e1': 60, 'h2': 55, 'g3': 46, 'a6': 16, 'f5': 29, 'd7': 11,
             'g6': 22, 'a3': 40, 'd2': 51, 'c7': 10, 'c2': 50, 'e4': 36,
             'a2': 48, 'f3': 45, 'f2': 53, 'e5': 28, 'h6': 23, 'b7': 9,
             'b5': 25, 'g5': 30, 'f8': 5, 'f7': 13, 'g7': 14, 'g2': 54,
             'c6': 18, 'c3': 42, 'b3': 41, 'd1': 59, 'c8': 2, 'e7': 12,
             'c4': 34, 'b6': 17, 'h5': 31, 'd6': 19})
        self.assertEqual(
            constants.MAPPIECE,
            ({'K':'K', 'Q':'Q', 'R':'R', 'B':'B', 'N':'N', '':'P'},
             {'K':'k', 'Q':'q', 'R':'r', 'B':'b', 'N':'n', '':'p'}))
        self.assertEqual(constants.CASTLING_W, 60)
        self.assertEqual(constants.CASTLING_WK, 63)
        self.assertEqual(constants.CASTLING_WQ, 56)
        self.assertEqual(constants.CASTLING_B, 4)
        self.assertEqual(constants.CASTLING_BK, 7)
        self.assertEqual(constants.CASTLING_BQ, 0)
        self.assertEqual(constants.CASTLING_AVAILABITY_SQUARES,
                         10448351135499550865)
        self.assertEqual(constants.CASTLING_SQUARES,
                         {'k': (4, 7, (5, 6), (), 'r', 'k'),
                          'q': (4, 0, (3, 2), (1,), 'r', 'k'),
                          'K': (60, 63, (61, 62), (), 'R', 'K'),
                          'Q': (60, 56, (59, 58), (57,), 'R', 'K'),
                          })
        self.assertEqual(constants.EN_PASSANT_TO_SQUARES,
                         1099494850560)
        self.assertEqual(constants.EN_PASSANT_FROM_SQUARES,
                         71776119061282560)
        self.assertEqual(constants.PIECE_CAPTURE_MAP,
                         {'P':constants.WHITE_PAWN_CAPTURES_TO_SQUARE,
                          'p':constants.BLACK_PAWN_CAPTURES_TO_SQUARE,
                          'K':constants.KING_MOVES,
                          'k':constants.KING_MOVES,
                          'Q':constants.QUEEN_MOVES,
                          'R':constants.ROOK_MOVES,
                          'B':constants.BISHOP_MOVES,
                          'N':constants.KNIGHT_MOVES,
                          'q':constants.QUEEN_MOVES,
                          'r':constants.ROOK_MOVES,
                          'b':constants.BISHOP_MOVES,
                          'n':constants.KNIGHT_MOVES})
        self.assertEqual(constants.PIECE_MOVE_MAP,
                         {'P':constants.WHITE_PAWN_MOVES_TO_SQUARE,
                          'p':constants.BLACK_PAWN_MOVES_TO_SQUARE,
                          'K':constants.KING_MOVES,
                          'k':constants.KING_MOVES,
                          'Q':constants.QUEEN_MOVES,
                          'R':constants.ROOK_MOVES,
                          'B':constants.BISHOP_MOVES,
                          'N':constants.KNIGHT_MOVES,
                          'q':constants.QUEEN_MOVES,
                          'r':constants.ROOK_MOVES,
                          'b':constants.BISHOP_MOVES,
                          'n':constants.KNIGHT_MOVES})
        self.assertEqual(constants.PAWN_MOVE_DESITINATION,
                         [281474976710400, 72057594037862400])
                          

    def test_RANKS(self):
        """"""
        self.assertEqual(constants.RANKS,
                         [255,
                          65280,
                          16711680,
                          4278190080,
                          1095216660480,
                          280375465082880,
                          71776119061217280,
                          18374686479671623680])

    def test_FILES(self):
        """"""
        self.assertEqual(constants.FILES,
                         [72340172838076673,
                          144680345676153346,
                          289360691352306692,
                          578721382704613384,
                          1157442765409226768,
                          2314885530818453536,
                          4629771061636907072,
                          9259542123273814144])


class RookMovesConstants(unittest.TestCase):

    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_ROOK_MOVES(self):
        """"""
        self.assertEqual(
            [(e,s) for e, s in enumerate(constants.ROOK_MOVES)],
            [(0, 72340172838076926),
             (1, 144680345676153597),
             (2, 289360691352306939),
             (3, 578721382704613623),
             (4, 1157442765409226991),
             (5, 2314885530818453727),
             (6, 4629771061636907199),
             (7, 9259542123273814143),
             (8, 72340172838141441),
             (9, 144680345676217602),
             (10, 289360691352369924),
             (11, 578721382704674568),
             (12, 1157442765409283856),
             (13, 2314885530818502432),
             (14, 4629771061636939584),
             (15, 9259542123273813888),
             (16, 72340172854657281),
             (17, 144680345692602882),
             (18, 289360691368494084),
             (19, 578721382720276488),
             (20, 1157442765423841296),
             (21, 2314885530830970912),
             (22, 4629771061645230144),
             (23, 9259542123273748608),
             (24, 72340177082712321),
             (25, 144680349887234562),
             (26, 289360695496279044),
             (27, 578721386714368008),
             (28, 1157442769150545936),
             (29, 2314885534022901792),
             (30, 4629771063767613504),
             (31, 9259542123257036928),
             (32, 72341259464802561),
             (33, 144681423712944642),
             (34, 289361752209228804),
             (35, 578722409201797128),
             (36, 1157443723186933776),
             (37, 2314886351157207072),
             (38, 4629771607097753664),
             (39, 9259542118978846848),
             (40, 72618349279904001),
             (41, 144956323094725122),
             (42, 289632270724367364),
             (43, 578984165983651848),
             (44, 1157687956502220816),
             (45, 2315095537539358752),
             (46, 4629910699613634624),
             (47, 9259541023762186368),
             (48, 143553341945872641),
             (49, 215330564830528002),
             (50, 358885010599838724),
             (51, 645993902138460168),
             (52, 1220211685215703056),
             (53, 2368647251370188832),
             (54, 4665518383679160384),
             (55, 9259260648297103488),
             (56, 18302911464433844481),
             (57, 18231136449196065282),
             (58, 18087586418720506884),
             (59, 17800486357769390088),
             (60, 17226286235867156496),
             (61, 16077885992062689312),
             (62, 13781085504453754944),
             (63, 9187484529235886208)
             ,])


class BishopMovesConstants(unittest.TestCase):

    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_BISHOP_MOVES(self):
        """"""
        self.assertEqual(
            [(e,s) for e, s in enumerate(constants.BISHOP_MOVES)],
            [(0, 9241421688590303744),
             (1, 36099303471056128),
             (2, 141012904249856),
             (3, 550848566272),
             (4, 6480472064),
             (5, 1108177604608),
             (6, 283691315142656),
             (7, 72624976668147712),
             (8, 4620710844295151618),
             (9, 9241421688590368773),
             (10, 36099303487963146),
             (11, 141017232965652),
             (12, 1659000848424),
             (13, 283693466779728),
             (14, 72624976676520096),
             (15, 145249953336262720),
             (16, 2310355422147510788),
             (17, 4620710844311799048),
             (18, 9241421692918565393),
             (19, 36100411639206946),
             (20, 424704217196612),
             (21, 72625527495610504),
             (22, 145249955479592976),
             (23, 290499906664153120),
             (24, 1155177711057110024),
             (25, 2310355426409252880),
             (26, 4620711952330133792),
             (27, 9241705379636978241),
             (28, 108724279602332802),
             (29, 145390965166737412),
             (30, 290500455356698632),
             (31, 580999811184992272),
             (32, 577588851267340304),
             (33, 1155178802063085600),
             (34, 2310639079102947392),
             (35, 4693335752243822976),
             (36, 9386671504487645697),
             (37, 326598935265674242),
             (38, 581140276476643332),
             (39, 1161999073681608712),
             (40, 288793334762704928),
             (41, 577868148797087808),
             (42, 1227793891648880768),
             (43, 2455587783297826816),
             (44, 4911175566595588352),
             (45, 9822351133174399489),
             (46, 1197958188344280066),
             (47, 2323857683139004420),
             (48, 144117404414255168),
             (49, 360293502378066048),
             (50, 720587009051099136),
             (51, 1441174018118909952),
             (52, 2882348036221108224),
             (53, 5764696068147249408),
             (54, 11529391036782871041),
             (55, 4611756524879479810),
             (56, 567382630219904),
             (57, 1416240237150208),
             (58, 2833579985862656),
             (59, 5667164249915392),
             (60, 11334324221640704),
             (61, 22667548931719168),
             (62, 45053622886727936),
             (63, 18049651735527937),
             ])


class QueenMovesConstants(unittest.TestCase):

    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_QUEEN_MOVES(self):
        """"""
        self.assertEqual(
            [(e,s) for e, s in enumerate(constants.QUEEN_MOVES)],
            [(0, 9313761861428380670),
             (1, 180779649147209725),
             (2, 289501704256556795),
             (3, 578721933553179895),
             (4, 1157442771889699055),
             (5, 2314886638996058335),
             (6, 4630054752952049855),
             (7, 9332167099941961855),
             (8, 4693051017133293059),
             (9, 9386102034266586375),
             (10, 325459994840333070),
             (11, 578862399937640220),
             (12, 1157444424410132280),
             (13, 2315169224285282160),
             (14, 4702396038313459680),
             (15, 9404792076610076608),
             (16, 2382695595002168069),
             (17, 4765391190004401930),
             (18, 9530782384287059477),
             (19, 614821794359483434),
             (20, 1157867469641037908),
             (21, 2387511058326581416),
             (22, 4775021017124823120),
             (23, 9550042029937901728),
             (24, 1227517888139822345),
             (25, 2455035776296487442),
             (26, 4910072647826412836),
             (27, 9820426766351346249),
             (28, 1266167048752878738),
             (29, 2460276499189639204),
             (30, 4920271519124312136),
             (31, 9840541934442029200),
             (32, 649930110732142865),
             (33, 1299860225776030242),
             (34, 2600000831312176196),
             (35, 5272058161445620104),
             (36, 10544115227674579473),
             (37, 2641485286422881314),
             (38, 5210911883574396996),
             (39, 10421541192660455560),
             (40, 361411684042608929),
             (41, 722824471891812930),
             (42, 1517426162373248132),
             (43, 3034571949281478664),
             (44, 6068863523097809168),
             (45, 12137446670713758241),
             (46, 5827868887957914690),
             (47, 11583398706901190788),
             (48, 287670746360127809),
             (49, 575624067208594050),
             (50, 1079472019650937860),
             (51, 2087167920257370120),
             (52, 4102559721436811280),
             (53, 8133343319517438240),
             (54, 16194909420462031425),
             (55, 13871017173176583298),
             (56, 18303478847064064385),
             (57, 18232552689433215490),
             (58, 18090419998706369540),
             (59, 17806153522019305480),
             (60, 17237620560088797200),
             (61, 16100553540994408480),
             (62, 13826139127340482880),
             (63, 9205534180971414145),
             ])


class KingMovesConstants(unittest.TestCase):

    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_KING_MOVES(self):
        """"""
        self.assertEqual(
            [(e,s) for e, s in enumerate(constants.KING_MOVES)],
            [(0, 770),
             (1, 1797),
             (2, 3594),
             (3, 7188),
             (4, 14376),
             (5, 28752),
             (6, 57504),
             (7, 49216),
             (8, 197123),
             (9, 460039),
             (10, 920078),
             (11, 1840156),
             (12, 3680312),
             (13, 7360624),
             (14, 14721248),
             (15, 12599488),
             (16, 50463488),
             (17, 117769984),
             (18, 235539968),
             (19, 471079936),
             (20, 942159872),
             (21, 1884319744),
             (22, 3768639488),
             (23, 3225468928),
             (24, 12918652928),
             (25, 30149115904),
             (26, 60298231808),
             (27, 120596463616),
             (28, 241192927232),
             (29, 482385854464),
             (30, 964771708928),
             (31, 825720045568),
             (32, 3307175149568),
             (33, 7718173671424),
             (34, 15436347342848),
             (35, 30872694685696),
             (36, 61745389371392),
             (37, 123490778742784),
             (38, 246981557485568),
             (39, 211384331665408),
             (40, 846636838289408),
             (41, 1975852459884544),
             (42, 3951704919769088),
             (43, 7903409839538176),
             (44, 15806819679076352),
             (45, 31613639358152704),
             (46, 63227278716305408),
             (47, 54114388906344448),
             (48, 216739030602088448),
             (49, 505818229730443264),
             (50, 1011636459460886528),
             (51, 2023272918921773056),
             (52, 4046545837843546112),
             (53, 8093091675687092224),
             (54, 16186183351374184448),
             (55, 13853283560024178688),
             (56, 144959613005987840),
             (57, 362258295026614272),
             (58, 724516590053228544),
             (59, 1449033180106457088),
             (60, 2898066360212914176),
             (61, 5796132720425828352),
             (62, 11592265440851656704),
             (63, 4665729213955833856),
             ])


class KnightMovesConstants(unittest.TestCase):

    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_KNIGHT_MOVES(self):
        """"""
        self.assertEqual(
            [(e,s) for e, s in enumerate(constants.KNIGHT_MOVES)],
            [(0, 132096),
             (1, 329728),
             (2, 659712),
             (3, 1319424),
             (4, 2638848),
             (5, 5277696),
             (6, 10489856),
             (7, 4202496),
             (8, 33816580),
             (9, 84410376),
             (10, 168886289),
             (11, 337772578),
             (12, 675545156),
             (13, 1351090312),
             (14, 2685403152),
             (15, 1075839008),
             (16, 8657044482),
             (17, 21609056261),
             (18, 43234889994),
             (19, 86469779988),
             (20, 172939559976),
             (21, 345879119952),
             (22, 687463207072),
             (23, 275414786112),
             (24, 2216203387392),
             (25, 5531918402816),
             (26, 11068131838464),
             (27, 22136263676928),
             (28, 44272527353856),
             (29, 88545054707712),
             (30, 175990581010432),
             (31, 70506185244672),
             (32, 567348067172352),
             (33, 1416171111120896),
             (34, 2833441750646784),
             (35, 5666883501293568),
             (36, 11333767002587136),
             (37, 22667534005174272),
             (38, 45053588738670592),
             (39, 18049583422636032),
             (40, 145241105196122112),
             (41, 362539804446949376),
             (42, 725361088165576704),
             (43, 1450722176331153408),
             (44, 2901444352662306816),
             (45, 5802888705324613632),
             (46, 11533718717099671552),
             (47, 4620693356194824192),
             (48, 288234782788157440),
             (49, 576469569871282176),
             (50, 1224997833292120064),
             (51, 2449995666584240128),
             (52, 4899991333168480256),
             (53, 9799982666336960512),
             (54, 1152939783987658752),
             (55, 2305878468463689728),
             (56, 1128098930098176),
             (57, 2257297371824128),
             (58, 4796069720358912),
             (59, 9592139440717824),
             (60, 19184278881435648),
             (61, 38368557762871296),
             (62, 4679521487814656),
             (63, 9077567998918656),
             ])


class WhitePawnMovesToSquareConstants(unittest.TestCase):

    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_WHITE_PAWN_MOVES_TO_SQUARE(self):
        """"""
        self.assertEqual(
            [(e,s) for e, s
             in enumerate(constants.WHITE_PAWN_MOVES_TO_SQUARE)],
            [(0, 256),
             (1, 512),
             (2, 1024),
             (3, 2048),
             (4, 4096),
             (5, 8192),
             (6, 16384),
             (7, 32768),
             (8, 65536),
             (9, 131072),
             (10, 262144),
             (11, 524288),
             (12, 1048576),
             (13, 2097152),
             (14, 4194304),
             (15, 8388608),
             (16, 16777216),
             (17, 33554432),
             (18, 67108864),
             (19, 134217728),
             (20, 268435456),
             (21, 536870912),
             (22, 1073741824),
             (23, 2147483648),
             (24, 4294967296),
             (25, 8589934592),
             (26, 17179869184),
             (27, 34359738368),
             (28, 68719476736),
             (29, 137438953472),
             (30, 274877906944),
             (31, 549755813888),
             (32, 282574488338432),
             (33, 565148976676864),
             (34, 1130297953353728),
             (35, 2260595906707456),
             (36, 4521191813414912),
             (37, 9042383626829824),
             (38, 18084767253659648),
             (39, 36169534507319296),
             (40, 281474976710656),
             (41, 562949953421312),
             (42, 1125899906842624),
             (43, 2251799813685248),
             (44, 4503599627370496),
             (45, 9007199254740992),
             (46, 18014398509481984),
             (47, 36028797018963968),
             (48, 0),
             (49, 0),
             (50, 0),
             (51, 0),
             (52, 0),
             (53, 0),
             (54, 0),
             (55, 0),
             (56, 0),
             (57, 0),
             (58, 0),
             (59, 0),
             (60, 0),
             (61, 0),
             (62, 0),
             (63, 0),
             ])


class WhitePawnCapturesToSquareConstants(unittest.TestCase):

    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_WHITE_PAWN_CAPTURES_TO_SQUARE(self):
        """"""
        self.assertEqual(
            [(e,s) for e, s
             in enumerate(constants.WHITE_PAWN_CAPTURES_TO_SQUARE)],
            [(0, 512),
             (1, 1280),
             (2, 2560),
             (3, 5120),
             (4, 10240),
             (5, 20480),
             (6, 40960),
             (7, 16384),
             (8, 131072),
             (9, 327680),
             (10, 655360),
             (11, 1310720),
             (12, 2621440),
             (13, 5242880),
             (14, 10485760),
             (15, 4194304),
             (16, 33554432),
             (17, 83886080),
             (18, 167772160),
             (19, 335544320),
             (20, 671088640),
             (21, 1342177280),
             (22, 2684354560),
             (23, 1073741824),
             (24, 8589934592),
             (25, 21474836480),
             (26, 42949672960),
             (27, 85899345920),
             (28, 171798691840),
             (29, 343597383680),
             (30, 687194767360),
             (31, 274877906944),
             (32, 2199023255552),
             (33, 5497558138880),
             (34, 10995116277760),
             (35, 21990232555520),
             (36, 43980465111040),
             (37, 87960930222080),
             (38, 175921860444160),
             (39, 70368744177664),
             (40, 562949953421312),
             (41, 1407374883553280),
             (42, 2814749767106560),
             (43, 5629499534213120),
             (44, 11258999068426240),
             (45, 22517998136852480),
             (46, 45035996273704960),
             (47, 18014398509481984),
             (48, 0),
             (49, 0),
             (50, 0),
             (51, 0),
             (52, 0),
             (53, 0),
             (54, 0),
             (55, 0),
             (56, 0),
             (57, 0),
             (58, 0),
             (59, 0),
             (60, 0),
             (61, 0),
             (62, 0),
             (63, 0),
             ])


class BlackPawnMovesToSquareConstants(unittest.TestCase):

    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_BLACK_PAWN_MOVES_TO_SQUARE(self):
        """"""
        self.assertEqual(
            [(e,s) for e, s
             in enumerate(constants.BLACK_PAWN_MOVES_TO_SQUARE)],
            [(0, 0),
             (1, 0),
             (2, 0),
             (3, 0),
             (4, 0),
             (5, 0),
             (6, 0),
             (7, 0),
             (8, 0),
             (9, 0),
             (10, 0),
             (11, 0),
             (12, 0),
             (13, 0),
             (14, 0),
             (15, 0),
             (16, 256),
             (17, 512),
             (18, 1024),
             (19, 2048),
             (20, 4096),
             (21, 8192),
             (22, 16384),
             (23, 32768),
             (24, 65792),
             (25, 131584),
             (26, 263168),
             (27, 526336),
             (28, 1052672),
             (29, 2105344),
             (30, 4210688),
             (31, 8421376),
             (32, 16777216),
             (33, 33554432),
             (34, 67108864),
             (35, 134217728),
             (36, 268435456),
             (37, 536870912),
             (38, 1073741824),
             (39, 2147483648),
             (40, 4294967296),
             (41, 8589934592),
             (42, 17179869184),
             (43, 34359738368),
             (44, 68719476736),
             (45, 137438953472),
             (46, 274877906944),
             (47, 549755813888),
             (48, 1099511627776),
             (49, 2199023255552),
             (50, 4398046511104),
             (51, 8796093022208),
             (52, 17592186044416),
             (53, 35184372088832),
             (54, 70368744177664),
             (55, 140737488355328),
             (56, 281474976710656),
             (57, 562949953421312),
             (58, 1125899906842624),
             (59, 2251799813685248),
             (60, 4503599627370496),
             (61, 9007199254740992),
             (62, 18014398509481984),
             (63, 36028797018963968),
             ])


class BlackPawnCapturesToSquareConstants(unittest.TestCase):

    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_BLACK_PAWN_CAPTURES_TO_SQUARE(self):
        """"""
        self.assertEqual(
            [(e,s) for e, s
             in enumerate(constants.BLACK_PAWN_CAPTURES_TO_SQUARE)],
            [(0, 0),
             (1, 0),
             (2, 0),
             (3, 0),
             (4, 0),
             (5, 0),
             (6, 0),
             (7, 0),
             (8, 0),
             (9, 0),
             (10, 0),
             (11, 0),
             (12, 0),
             (13, 0),
             (14, 0),
             (15, 0),
             (16, 512),
             (17, 1280),
             (18, 2560),
             (19, 5120),
             (20, 10240),
             (21, 20480),
             (22, 40960),
             (23, 16384),
             (24, 131072),
             (25, 327680),
             (26, 655360),
             (27, 1310720),
             (28, 2621440),
             (29, 5242880),
             (30, 10485760),
             (31, 4194304),
             (32, 33554432),
             (33, 83886080),
             (34, 167772160),
             (35, 335544320),
             (36, 671088640),
             (37, 1342177280),
             (38, 2684354560),
             (39, 1073741824),
             (40, 8589934592),
             (41, 21474836480),
             (42, 42949672960),
             (43, 85899345920),
             (44, 171798691840),
             (45, 343597383680),
             (46, 687194767360),
             (47, 274877906944),
             (48, 2199023255552),
             (49, 5497558138880),
             (50, 10995116277760),
             (51, 21990232555520),
             (52, 43980465111040),
             (53, 87960930222080),
             (54, 175921860444160),
             (55, 70368744177664),
             (56, 562949953421312),
             (57, 1407374883553280),
             (58, 2814749767106560),
             (59, 5629499534213120),
             (60, 11258999068426240),
             (61, 22517998136852480),
             (62, 45035996273704960),
             (63, 18014398509481984),
             ])


class GAPSConstants(unittest.TestCase):

    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_GAPS(self):
        """"""
        # GAPS[x][y] == GAPS[y][x]
        # Except for those listed, GAPS[x][y] == all_squares
        gaps = constants.GAPS
        self.assertEqual(len(gaps), 64)
        for g in gaps:
            self.assertEqual(len(g), 64)
        for x in range(64):
            for y in range(64):
                self.assertEqual(gaps[x][y], gaps[y][x])
        all_squares = constants.ALL_SQUARES
        self.assertEqual(
            [((f,t), gaps[f][t]) for f in range(64) for t in range(64)
             if gaps[f][t] != all_squares],
            [((0, 1), 0),
             ((0, 2), 2),
             ((0, 3), 6),
             ((0, 4), 14),
             ((0, 5), 30),
             ((0, 6), 62),
             ((0, 7), 126),
             ((0, 8), 0),
             ((0, 9), 0),
             ((0, 10), 0),
             ((0, 16), 256),
             ((0, 17), 0),
             ((0, 18), 512),
             ((0, 24), 65792),
             ((0, 27), 262656),
             ((0, 32), 16843008),
             ((0, 36), 134480384),
             ((0, 40), 4311810304),
             ((0, 45), 68853957120),
             ((0, 48), 1103823438080),
             ((0, 54), 35253226045952),
             ((0, 56), 282578800148736),
             ((0, 63), 18049651735527936),
             ((1, 0), 0),
             ((1, 2), 0),
             ((1, 3), 4),
             ((1, 4), 12),
             ((1, 5), 28),
             ((1, 6), 60),
             ((1, 7), 124),
             ((1, 8), 0),
             ((1, 9), 0),
             ((1, 10), 0),
             ((1, 11), 0),
             ((1, 16), 0),
             ((1, 17), 512),
             ((1, 18), 0),
             ((1, 19), 1024),
             ((1, 25), 131584),
             ((1, 28), 525312),
             ((1, 33), 33686016),
             ((1, 37), 268960768),
             ((1, 41), 8623620608),
             ((1, 46), 137707914240),
             ((1, 49), 2207646876160),
             ((1, 55), 70506452091904),
             ((1, 57), 565157600297472),
             ((2, 0), 2),
             ((2, 1), 0),
             ((2, 3), 0),
             ((2, 4), 8),
             ((2, 5), 24),
             ((2, 6), 56),
             ((2, 7), 120),
             ((2, 8), 0),
             ((2, 9), 0),
             ((2, 10), 0),
             ((2, 11), 0),
             ((2, 12), 0),
             ((2, 16), 512),
             ((2, 17), 0),
             ((2, 18), 1024),
             ((2, 19), 0),
             ((2, 20), 2048),
             ((2, 26), 263168),
             ((2, 29), 1050624),
             ((2, 34), 67372032),
             ((2, 38), 537921536),
             ((2, 42), 17247241216),
             ((2, 47), 275415828480),
             ((2, 50), 4415293752320),
             ((2, 58), 1130315200594944),
             ((3, 0), 6),
             ((3, 1), 4),
             ((3, 2), 0),
             ((3, 4), 0),
             ((3, 5), 16),
             ((3, 6), 48),
             ((3, 7), 112),
             ((3, 9), 0),
             ((3, 10), 0),
             ((3, 11), 0),
             ((3, 12), 0),
             ((3, 13), 0),
             ((3, 17), 1024),
             ((3, 18), 0),
             ((3, 19), 2048),
             ((3, 20), 0),
             ((3, 21), 4096),
             ((3, 24), 132096),
             ((3, 27), 526336),
             ((3, 30), 2101248),
             ((3, 35), 134744064),
             ((3, 39), 1075843072),
             ((3, 43), 34494482432),
             ((3, 51), 8830587504640),
             ((3, 59), 2260630401189888),
             ((4, 0), 14),
             ((4, 1), 12),
             ((4, 2), 8),
             ((4, 3), 0),
             ((4, 5), 0),
             ((4, 6), 32),
             ((4, 7), 96),
             ((4, 10), 0),
             ((4, 11), 0),
             ((4, 12), 0),
             ((4, 13), 0),
             ((4, 14), 0),
             ((4, 18), 2048),
             ((4, 19), 0),
             ((4, 20), 4096),
             ((4, 21), 0),
             ((4, 22), 8192),
             ((4, 25), 264192),
             ((4, 28), 1052672),
             ((4, 31), 4202496),
             ((4, 32), 33818624),
             ((4, 36), 269488128),
             ((4, 44), 68988964864),
             ((4, 52), 17661175009280),
             ((4, 60), 4521260802379776),
             ((5, 0), 30),
             ((5, 1), 28),
             ((5, 2), 24),
             ((5, 3), 16),
             ((5, 4), 0),
             ((5, 6), 0),
             ((5, 7), 64),
             ((5, 11), 0),
             ((5, 12), 0),
             ((5, 13), 0),
             ((5, 14), 0),
             ((5, 15), 0),
             ((5, 19), 4096),
             ((5, 20), 0),
             ((5, 21), 8192),
             ((5, 22), 0),
             ((5, 23), 16384),
             ((5, 26), 528384),
             ((5, 29), 2105344),
             ((5, 33), 67637248),
             ((5, 37), 538976256),
             ((5, 40), 8657571840),
             ((5, 45), 137977929728),
             ((5, 53), 35322350018560),
             ((5, 61), 9042521604759552),
             ((6, 0), 62),
             ((6, 1), 60),
             ((6, 2), 56),
             ((6, 3), 48),
             ((6, 4), 32),
             ((6, 5), 0),
             ((6, 7), 0),
             ((6, 12), 0),
             ((6, 13), 0),
             ((6, 14), 0),
             ((6, 15), 0),
             ((6, 20), 8192),
             ((6, 21), 0),
             ((6, 22), 16384),
             ((6, 23), 0),
             ((6, 27), 1056768),
             ((6, 30), 4210688),
             ((6, 34), 135274496),
             ((6, 38), 1077952512),
             ((6, 41), 17315143680),
             ((6, 46), 275955859456),
             ((6, 48), 2216338399232),
             ((6, 54), 70644700037120),
             ((6, 62), 18085043209519104),
             ((7, 0), 126),
             ((7, 1), 124),
             ((7, 2), 120),
             ((7, 3), 112),
             ((7, 4), 96),
             ((7, 5), 64),
             ((7, 6), 0),
             ((7, 13), 0),
             ((7, 14), 0),
             ((7, 15), 0),
             ((7, 21), 16384),
             ((7, 22), 0),
             ((7, 23), 32768),
             ((7, 28), 2113536),
             ((7, 31), 8421376),
             ((7, 35), 270548992),
             ((7, 39), 2155905024),
             ((7, 42), 34630287360),
             ((7, 47), 551911718912),
             ((7, 49), 4432676798464),
             ((7, 55), 141289400074240),
             ((7, 56), 567382630219776),
             ((7, 63), 36170086419038208),
             ((8, 0), 0),
             ((8, 1), 0),
             ((8, 2), 0),
             ((8, 9), 0),
             ((8, 10), 512),
             ((8, 11), 1536),
             ((8, 12), 3584),
             ((8, 13), 7680),
             ((8, 14), 15872),
             ((8, 15), 32256),
             ((8, 16), 0),
             ((8, 17), 0),
             ((8, 18), 0),
             ((8, 24), 65536),
             ((8, 25), 0),
             ((8, 26), 131072),
             ((8, 32), 16842752),
             ((8, 35), 67239936),
             ((8, 40), 4311810048),
             ((8, 44), 34426978304),
             ((8, 48), 1103823437824),
             ((8, 53), 17626613022720),
             ((8, 56), 282578800148480),
             ((8, 62), 9024825867763712),
             ((9, 0), 0),
             ((9, 1), 0),
             ((9, 2), 0),
             ((9, 3), 0),
             ((9, 8), 0),
             ((9, 10), 0),
             ((9, 11), 1024),
             ((9, 12), 3072),
             ((9, 13), 7168),
             ((9, 14), 15360),
             ((9, 15), 31744),
             ((9, 16), 0),
             ((9, 17), 0),
             ((9, 18), 0),
             ((9, 19), 0),
             ((9, 24), 0),
             ((9, 25), 131072),
             ((9, 26), 0),
             ((9, 27), 262144),
             ((9, 33), 33685504),
             ((9, 36), 134479872),
             ((9, 41), 8623620096),
             ((9, 45), 68853956608),
             ((9, 49), 2207646875648),
             ((9, 54), 35253226045440),
             ((9, 57), 565157600296960),
             ((9, 63), 18049651735527424),
             ((10, 0), 0),
             ((10, 1), 0),
             ((10, 2), 0),
             ((10, 3), 0),
             ((10, 4), 0),
             ((10, 8), 512),
             ((10, 9), 0),
             ((10, 11), 0),
             ((10, 12), 2048),
             ((10, 13), 6144),
             ((10, 14), 14336),
             ((10, 15), 30720),
             ((10, 16), 0),
             ((10, 17), 0),
             ((10, 18), 0),
             ((10, 19), 0),
             ((10, 20), 0),
             ((10, 24), 131072),
             ((10, 25), 0),
             ((10, 26), 262144),
             ((10, 27), 0),
             ((10, 28), 524288),
             ((10, 34), 67371008),
             ((10, 37), 268959744),
             ((10, 42), 17247240192),
             ((10, 46), 137707913216),
             ((10, 50), 4415293751296),
             ((10, 55), 70506452090880),
             ((10, 58), 1130315200593920),
             ((11, 1), 0),
             ((11, 2), 0),
             ((11, 3), 0),
             ((11, 4), 0),
             ((11, 5), 0),
             ((11, 8), 1536),
             ((11, 9), 1024),
             ((11, 10), 0),
             ((11, 12), 0),
             ((11, 13), 4096),
             ((11, 14), 12288),
             ((11, 15), 28672),
             ((11, 17), 0),
             ((11, 18), 0),
             ((11, 19), 0),
             ((11, 20), 0),
             ((11, 21), 0),
             ((11, 25), 262144),
             ((11, 26), 0),
             ((11, 27), 524288),
             ((11, 28), 0),
             ((11, 29), 1048576),
             ((11, 32), 33816576),
             ((11, 35), 134742016),
             ((11, 38), 537919488),
             ((11, 43), 34494480384),
             ((11, 47), 275415826432),
             ((11, 51), 8830587502592),
             ((11, 59), 2260630401187840),
             ((12, 2), 0),
             ((12, 3), 0),
             ((12, 4), 0),
             ((12, 5), 0),
             ((12, 6), 0),
             ((12, 8), 3584),
             ((12, 9), 3072),
             ((12, 10), 2048),
             ((12, 11), 0),
             ((12, 13), 0),
             ((12, 14), 8192),
             ((12, 15), 24576),
             ((12, 18), 0),
             ((12, 19), 0),
             ((12, 20), 0),
             ((12, 21), 0),
             ((12, 22), 0),
             ((12, 26), 524288),
             ((12, 27), 0),
             ((12, 28), 1048576),
             ((12, 29), 0),
             ((12, 30), 2097152),
             ((12, 33), 67633152),
             ((12, 36), 269484032),
             ((12, 39), 1075838976),
             ((12, 40), 8657567744),
             ((12, 44), 68988960768),
             ((12, 52), 17661175005184),
             ((12, 60), 4521260802375680),
             ((13, 3), 0),
             ((13, 4), 0),
             ((13, 5), 0),
             ((13, 6), 0),
             ((13, 7), 0),
             ((13, 8), 7680),
             ((13, 9), 7168),
             ((13, 10), 6144),
             ((13, 11), 4096),
             ((13, 12), 0),
             ((13, 14), 0),
             ((13, 15), 16384),
             ((13, 19), 0),
             ((13, 20), 0),
             ((13, 21), 0),
             ((13, 22), 0),
             ((13, 23), 0),
             ((13, 27), 1048576),
             ((13, 28), 0),
             ((13, 29), 2097152),
             ((13, 30), 0),
             ((13, 31), 4194304),
             ((13, 34), 135266304),
             ((13, 37), 538968064),
             ((13, 41), 17315135488),
             ((13, 45), 137977921536),
             ((13, 48), 2216338391040),
             ((13, 53), 35322350010368),
             ((13, 61), 9042521604751360),
             ((14, 4), 0),
             ((14, 5), 0),
             ((14, 6), 0),
             ((14, 7), 0),
             ((14, 8), 15872),
             ((14, 9), 15360),
             ((14, 10), 14336),
             ((14, 11), 12288),
             ((14, 12), 8192),
             ((14, 13), 0),
             ((14, 15), 0),
             ((14, 20), 0),
             ((14, 21), 0),
             ((14, 22), 0),
             ((14, 23), 0),
             ((14, 28), 2097152),
             ((14, 29), 0),
             ((14, 30), 4194304),
             ((14, 31), 0),
             ((14, 35), 270532608),
             ((14, 38), 1077936128),
             ((14, 42), 34630270976),
             ((14, 46), 275955843072),
             ((14, 49), 4432676782080),
             ((14, 54), 70644700020736),
             ((14, 56), 567382630203392),
             ((14, 62), 18085043209502720),
             ((15, 5), 0),
             ((15, 6), 0),
             ((15, 7), 0),
             ((15, 8), 32256),
             ((15, 9), 31744),
             ((15, 10), 30720),
             ((15, 11), 28672),
             ((15, 12), 24576),
             ((15, 13), 16384),
             ((15, 14), 0),
             ((15, 21), 0),
             ((15, 22), 0),
             ((15, 23), 0),
             ((15, 29), 4194304),
             ((15, 30), 0),
             ((15, 31), 8388608),
             ((15, 36), 541065216),
             ((15, 39), 2155872256),
             ((15, 43), 69260541952),
             ((15, 47), 551911686144),
             ((15, 50), 8865353564160),
             ((15, 55), 141289400041472),
             ((15, 57), 1134765260406784),
             ((15, 63), 36170086419005440),
             ((16, 0), 256),
             ((16, 1), 0),
             ((16, 2), 512),
             ((16, 8), 0),
             ((16, 9), 0),
             ((16, 10), 0),
             ((16, 17), 0),
             ((16, 18), 131072),
             ((16, 19), 393216),
             ((16, 20), 917504),
             ((16, 21), 1966080),
             ((16, 22), 4063232),
             ((16, 23), 8257536),
             ((16, 24), 0),
             ((16, 25), 0),
             ((16, 26), 0),
             ((16, 32), 16777216),
             ((16, 33), 0),
             ((16, 34), 33554432),
             ((16, 40), 4311744512),
             ((16, 43), 17213423616),
             ((16, 48), 1103823372288),
             ((16, 52), 8813306445824),
             ((16, 56), 282578800082944),
             ((16, 61), 4512412933816320),
             ((17, 0), 0),
             ((17, 1), 512),
             ((17, 2), 0),
             ((17, 3), 1024),
             ((17, 8), 0),
             ((17, 9), 0),
             ((17, 10), 0),
             ((17, 11), 0),
             ((17, 16), 0),
             ((17, 18), 0),
             ((17, 19), 262144),
             ((17, 20), 786432),
             ((17, 21), 1835008),
             ((17, 22), 3932160),
             ((17, 23), 8126464),
             ((17, 24), 0),
             ((17, 25), 0),
             ((17, 26), 0),
             ((17, 27), 0),
             ((17, 32), 0),
             ((17, 33), 33554432),
             ((17, 34), 0),
             ((17, 35), 67108864),
             ((17, 41), 8623489024),
             ((17, 44), 34426847232),
             ((17, 49), 2207646744576),
             ((17, 53), 17626612891648),
             ((17, 57), 565157600165888),
             ((17, 62), 9024825867632640),
             ((18, 0), 512),
             ((18, 1), 0),
             ((18, 2), 1024),
             ((18, 3), 0),
             ((18, 4), 2048),
             ((18, 8), 0),
             ((18, 9), 0),
             ((18, 10), 0),
             ((18, 11), 0),
             ((18, 12), 0),
             ((18, 16), 131072),
             ((18, 17), 0),
             ((18, 19), 0),
             ((18, 20), 524288),
             ((18, 21), 1572864),
             ((18, 22), 3670016),
             ((18, 23), 7864320),
             ((18, 24), 0),
             ((18, 25), 0),
             ((18, 26), 0),
             ((18, 27), 0),
             ((18, 28), 0),
             ((18, 32), 33554432),
             ((18, 33), 0),
             ((18, 34), 67108864),
             ((18, 35), 0),
             ((18, 36), 134217728),
             ((18, 42), 17246978048),
             ((18, 45), 68853694464),
             ((18, 50), 4415293489152),
             ((18, 54), 35253225783296),
             ((18, 58), 1130315200331776),
             ((18, 63), 18049651735265280),
             ((19, 1), 1024),
             ((19, 2), 0),
             ((19, 3), 2048),
             ((19, 4), 0),
             ((19, 5), 4096),
             ((19, 9), 0),
             ((19, 10), 0),
             ((19, 11), 0),
             ((19, 12), 0),
             ((19, 13), 0),
             ((19, 16), 393216),
             ((19, 17), 262144),
             ((19, 18), 0),
             ((19, 20), 0),
             ((19, 21), 1048576),
             ((19, 22), 3145728),
             ((19, 23), 7340032),
             ((19, 25), 0),
             ((19, 26), 0),
             ((19, 27), 0),
             ((19, 28), 0),
             ((19, 29), 0),
             ((19, 33), 67108864),
             ((19, 34), 0),
             ((19, 35), 134217728),
             ((19, 36), 0),
             ((19, 37), 268435456),
             ((19, 40), 8657043456),
             ((19, 43), 34493956096),
             ((19, 46), 137707388928),
             ((19, 51), 8830586978304),
             ((19, 55), 70506451566592),
             ((19, 59), 2260630400663552),
             ((20, 2), 2048),
             ((20, 3), 0),
             ((20, 4), 4096),
             ((20, 5), 0),
             ((20, 6), 8192),
             ((20, 10), 0),
             ((20, 11), 0),
             ((20, 12), 0),
             ((20, 13), 0),
             ((20, 14), 0),
             ((20, 16), 917504),
             ((20, 17), 786432),
             ((20, 18), 524288),
             ((20, 19), 0),
             ((20, 21), 0),
             ((20, 22), 2097152),
             ((20, 23), 6291456),
             ((20, 26), 0),
             ((20, 27), 0),
             ((20, 28), 0),
             ((20, 29), 0),
             ((20, 30), 0),
             ((20, 34), 134217728),
             ((20, 35), 0),
             ((20, 36), 268435456),
             ((20, 37), 0),
             ((20, 38), 536870912),
             ((20, 41), 17314086912),
             ((20, 44), 68987912192),
             ((20, 47), 275414777856),
             ((20, 48), 2216337342464),
             ((20, 52), 17661173956608),
             ((20, 60), 4521260801327104),
             ((21, 3), 4096),
             ((21, 4), 0),
             ((21, 5), 8192),
             ((21, 6), 0),
             ((21, 7), 16384),
             ((21, 11), 0),
             ((21, 12), 0),
             ((21, 13), 0),
             ((21, 14), 0),
             ((21, 15), 0),
             ((21, 16), 1966080),
             ((21, 17), 1835008),
             ((21, 18), 1572864),
             ((21, 19), 1048576),
             ((21, 20), 0),
             ((21, 22), 0),
             ((21, 23), 4194304),
             ((21, 27), 0),
             ((21, 28), 0),
             ((21, 29), 0),
             ((21, 30), 0),
             ((21, 31), 0),
             ((21, 35), 268435456),
             ((21, 36), 0),
             ((21, 37), 536870912),
             ((21, 38), 0),
             ((21, 39), 1073741824),
             ((21, 42), 34628173824),
             ((21, 45), 137975824384),
             ((21, 49), 4432674684928),
             ((21, 53), 35322347913216),
             ((21, 56), 567382628106240),
             ((21, 61), 9042521602654208),
             ((22, 4), 8192),
             ((22, 5), 0),
             ((22, 6), 16384),
             ((22, 7), 0),
             ((22, 12), 0),
             ((22, 13), 0),
             ((22, 14), 0),
             ((22, 15), 0),
             ((22, 16), 4063232),
             ((22, 17), 3932160),
             ((22, 18), 3670016),
             ((22, 19), 3145728),
             ((22, 20), 2097152),
             ((22, 21), 0),
             ((22, 23), 0),
             ((22, 28), 0),
             ((22, 29), 0),
             ((22, 30), 0),
             ((22, 31), 0),
             ((22, 36), 536870912),
             ((22, 37), 0),
             ((22, 38), 1073741824),
             ((22, 39), 0),
             ((22, 43), 69256347648),
             ((22, 46), 275951648768),
             ((22, 50), 8865349369856),
             ((22, 54), 70644695826432),
             ((22, 57), 1134765256212480),
             ((22, 62), 18085043205308416),
             ((23, 5), 16384),
             ((23, 6), 0),
             ((23, 7), 32768),
             ((23, 13), 0),
             ((23, 14), 0),
             ((23, 15), 0),
             ((23, 16), 8257536),
             ((23, 17), 8126464),
             ((23, 18), 7864320),
             ((23, 19), 7340032),
             ((23, 20), 6291456),
             ((23, 21), 4194304),
             ((23, 22), 0),
             ((23, 29), 0),
             ((23, 30), 0),
             ((23, 31), 0),
             ((23, 37), 1073741824),
             ((23, 38), 0),
             ((23, 39), 2147483648),
             ((23, 44), 138512695296),
             ((23, 47), 551903297536),
             ((23, 51), 17730698739712),
             ((23, 55), 141289391652864),
             ((23, 58), 2269530512424960),
             ((23, 63), 36170086410616832),
             ((24, 0), 65792),
             ((24, 3), 132096),
             ((24, 8), 65536),
             ((24, 9), 0),
             ((24, 10), 131072),
             ((24, 16), 0),
             ((24, 17), 0),
             ((24, 18), 0),
             ((24, 25), 0),
             ((24, 26), 33554432),
             ((24, 27), 100663296),
             ((24, 28), 234881024),
             ((24, 29), 503316480),
             ((24, 30), 1040187392),
             ((24, 31), 2113929216),
             ((24, 32), 0),
             ((24, 33), 0),
             ((24, 34), 0),
             ((24, 40), 4294967296),
             ((24, 41), 0),
             ((24, 42), 8589934592),
             ((24, 48), 1103806595072),
             ((24, 51), 4406636445696),
             ((24, 56), 282578783305728),
             ((24, 60), 2256206450130944),
             ((25, 1), 131584),
             ((25, 4), 264192),
             ((25, 8), 0),
             ((25, 9), 131072),
             ((25, 10), 0),
             ((25, 11), 262144),
             ((25, 16), 0),
             ((25, 17), 0),
             ((25, 18), 0),
             ((25, 19), 0),
             ((25, 24), 0),
             ((25, 26), 0),
             ((25, 27), 67108864),
             ((25, 28), 201326592),
             ((25, 29), 469762048),
             ((25, 30), 1006632960),
             ((25, 31), 2080374784),
             ((25, 32), 0),
             ((25, 33), 0),
             ((25, 34), 0),
             ((25, 35), 0),
             ((25, 40), 0),
             ((25, 41), 8589934592),
             ((25, 42), 0),
             ((25, 43), 17179869184),
             ((25, 49), 2207613190144),
             ((25, 52), 8813272891392),
             ((25, 57), 565157566611456),
             ((25, 61), 4512412900261888),
             ((26, 2), 263168),
             ((26, 5), 528384),
             ((26, 8), 131072),
             ((26, 9), 0),
             ((26, 10), 262144),
             ((26, 11), 0),
             ((26, 12), 524288),
             ((26, 16), 0),
             ((26, 17), 0),
             ((26, 18), 0),
             ((26, 19), 0),
             ((26, 20), 0),
             ((26, 24), 33554432),
             ((26, 25), 0),
             ((26, 27), 0),
             ((26, 28), 134217728),
             ((26, 29), 402653184),
             ((26, 30), 939524096),
             ((26, 31), 2013265920),
             ((26, 32), 0),
             ((26, 33), 0),
             ((26, 34), 0),
             ((26, 35), 0),
             ((26, 36), 0),
             ((26, 40), 8589934592),
             ((26, 41), 0),
             ((26, 42), 17179869184),
             ((26, 43), 0),
             ((26, 44), 34359738368),
             ((26, 50), 4415226380288),
             ((26, 53), 17626545782784),
             ((26, 58), 1130315133222912),
             ((26, 62), 9024825800523776),
             ((27, 0), 262656),
             ((27, 3), 526336),
             ((27, 6), 1056768),
             ((27, 9), 262144),
             ((27, 10), 0),
             ((27, 11), 524288),
             ((27, 12), 0),
             ((27, 13), 1048576),
             ((27, 17), 0),
             ((27, 18), 0),
             ((27, 19), 0),
             ((27, 20), 0),
             ((27, 21), 0),
             ((27, 24), 100663296),
             ((27, 25), 67108864),
             ((27, 26), 0),
             ((27, 28), 0),
             ((27, 29), 268435456),
             ((27, 30), 805306368),
             ((27, 31), 1879048192),
             ((27, 33), 0),
             ((27, 34), 0),
             ((27, 35), 0),
             ((27, 36), 0),
             ((27, 37), 0),
             ((27, 41), 17179869184),
             ((27, 42), 0),
             ((27, 43), 34359738368),
             ((27, 44), 0),
             ((27, 45), 68719476736),
             ((27, 48), 2216203124736),
             ((27, 51), 8830452760576),
             ((27, 54), 35253091565568),
             ((27, 59), 2260630266445824),
             ((27, 63), 18049651601047552),
             ((28, 1), 525312),
             ((28, 4), 1052672),
             ((28, 7), 2113536),
             ((28, 10), 524288),
             ((28, 11), 0),
             ((28, 12), 1048576),
             ((28, 13), 0),
             ((28, 14), 2097152),
             ((28, 18), 0),
             ((28, 19), 0),
             ((28, 20), 0),
             ((28, 21), 0),
             ((28, 22), 0),
             ((28, 24), 234881024),
             ((28, 25), 201326592),
             ((28, 26), 134217728),
             ((28, 27), 0),
             ((28, 29), 0),
             ((28, 30), 536870912),
             ((28, 31), 1610612736),
             ((28, 34), 0),
             ((28, 35), 0),
             ((28, 36), 0),
             ((28, 37), 0),
             ((28, 38), 0),
             ((28, 42), 34359738368),
             ((28, 43), 0),
             ((28, 44), 68719476736),
             ((28, 45), 0),
             ((28, 46), 137438953472),
             ((28, 49), 4432406249472),
             ((28, 52), 17660905521152),
             ((28, 55), 70506183131136),
             ((28, 56), 567382359670784),
             ((28, 60), 4521260532891648),
             ((29, 2), 1050624),
             ((29, 5), 2105344),
             ((29, 11), 1048576),
             ((29, 12), 0),
             ((29, 13), 2097152),
             ((29, 14), 0),
             ((29, 15), 4194304),
             ((29, 19), 0),
             ((29, 20), 0),
             ((29, 21), 0),
             ((29, 22), 0),
             ((29, 23), 0),
             ((29, 24), 503316480),
             ((29, 25), 469762048),
             ((29, 26), 402653184),
             ((29, 27), 268435456),
             ((29, 28), 0),
             ((29, 30), 0),
             ((29, 31), 1073741824),
             ((29, 35), 0),
             ((29, 36), 0),
             ((29, 37), 0),
             ((29, 38), 0),
             ((29, 39), 0),
             ((29, 43), 68719476736),
             ((29, 44), 0),
             ((29, 45), 137438953472),
             ((29, 46), 0),
             ((29, 47), 274877906944),
             ((29, 50), 8864812498944),
             ((29, 53), 35321811042304),
             ((29, 57), 1134764719341568),
             ((29, 61), 9042521065783296),
             ((30, 3), 2101248),
             ((30, 6), 4210688),
             ((30, 12), 2097152),
             ((30, 13), 0),
             ((30, 14), 4194304),
             ((30, 15), 0),
             ((30, 20), 0),
             ((30, 21), 0),
             ((30, 22), 0),
             ((30, 23), 0),
             ((30, 24), 1040187392),
             ((30, 25), 1006632960),
             ((30, 26), 939524096),
             ((30, 27), 805306368),
             ((30, 28), 536870912),
             ((30, 29), 0),
             ((30, 31), 0),
             ((30, 36), 0),
             ((30, 37), 0),
             ((30, 38), 0),
             ((30, 39), 0),
             ((30, 44), 137438953472),
             ((30, 45), 0),
             ((30, 46), 274877906944),
             ((30, 47), 0),
             ((30, 51), 17729624997888),
             ((30, 54), 70643622084608),
             ((30, 58), 2269529438683136),
             ((30, 62), 18085042131566592),
             ((31, 4), 4202496),
             ((31, 7), 8421376),
             ((31, 13), 4194304),
             ((31, 14), 0),
             ((31, 15), 8388608),
             ((31, 21), 0),
             ((31, 22), 0),
             ((31, 23), 0),
             ((31, 24), 2113929216),
             ((31, 25), 2080374784),
             ((31, 26), 2013265920),
             ((31, 27), 1879048192),
             ((31, 28), 1610612736),
             ((31, 29), 1073741824),
             ((31, 30), 0),
             ((31, 37), 0),
             ((31, 38), 0),
             ((31, 39), 0),
             ((31, 45), 274877906944),
             ((31, 46), 0),
             ((31, 47), 549755813888),
             ((31, 52), 35459249995776),
             ((31, 55), 141287244169216),
             ((31, 59), 4539058877366272),
             ((31, 63), 36170084263133184),
             ((32, 0), 16843008),
             ((32, 4), 33818624),
             ((32, 8), 16842752),
             ((32, 11), 33816576),
             ((32, 16), 16777216),
             ((32, 17), 0),
             ((32, 18), 33554432),
             ((32, 24), 0),
             ((32, 25), 0),
             ((32, 26), 0),
             ((32, 33), 0),
             ((32, 34), 8589934592),
             ((32, 35), 25769803776),
             ((32, 36), 60129542144),
             ((32, 37), 128849018880),
             ((32, 38), 266287972352),
             ((32, 39), 541165879296),
             ((32, 40), 0),
             ((32, 41), 0),
             ((32, 42), 0),
             ((32, 48), 1099511627776),
             ((32, 49), 0),
             ((32, 50), 2199023255552),
             ((32, 56), 282574488338432),
             ((32, 59), 1128098930098176),
             ((33, 1), 33686016),
             ((33, 5), 67637248),
             ((33, 9), 33685504),
             ((33, 12), 67633152),
             ((33, 16), 0),
             ((33, 17), 33554432),
             ((33, 18), 0),
             ((33, 19), 67108864),
             ((33, 24), 0),
             ((33, 25), 0),
             ((33, 26), 0),
             ((33, 27), 0),
             ((33, 32), 0),
             ((33, 34), 0),
             ((33, 35), 17179869184),
             ((33, 36), 51539607552),
             ((33, 37), 120259084288),
             ((33, 38), 257698037760),
             ((33, 39), 532575944704),
             ((33, 40), 0),
             ((33, 41), 0),
             ((33, 42), 0),
             ((33, 43), 0),
             ((33, 48), 0),
             ((33, 49), 2199023255552),
             ((33, 50), 0),
             ((33, 51), 4398046511104),
             ((33, 57), 565148976676864),
             ((33, 60), 2256197860196352),
             ((34, 2), 67372032),
             ((34, 6), 135274496),
             ((34, 10), 67371008),
             ((34, 13), 135266304),
             ((34, 16), 33554432),
             ((34, 17), 0),
             ((34, 18), 67108864),
             ((34, 19), 0),
             ((34, 20), 134217728),
             ((34, 24), 0),
             ((34, 25), 0),
             ((34, 26), 0),
             ((34, 27), 0),
             ((34, 28), 0),
             ((34, 32), 8589934592),
             ((34, 33), 0),
             ((34, 35), 0),
             ((34, 36), 34359738368),
             ((34, 37), 103079215104),
             ((34, 38), 240518168576),
             ((34, 39), 515396075520),
             ((34, 40), 0),
             ((34, 41), 0),
             ((34, 42), 0),
             ((34, 43), 0),
             ((34, 44), 0),
             ((34, 48), 2199023255552),
             ((34, 49), 0),
             ((34, 50), 4398046511104),
             ((34, 51), 0),
             ((34, 52), 8796093022208),
             ((34, 58), 1130297953353728),
             ((34, 61), 4512395720392704),
             ((35, 3), 134744064),
             ((35, 7), 270548992),
             ((35, 8), 67239936),
             ((35, 11), 134742016),
             ((35, 14), 270532608),
             ((35, 17), 67108864),
             ((35, 18), 0),
             ((35, 19), 134217728),
             ((35, 20), 0),
             ((35, 21), 268435456),
             ((35, 25), 0),
             ((35, 26), 0),
             ((35, 27), 0),
             ((35, 28), 0),
             ((35, 29), 0),
             ((35, 32), 25769803776),
             ((35, 33), 17179869184),
             ((35, 34), 0),
             ((35, 36), 0),
             ((35, 37), 68719476736),
             ((35, 38), 206158430208),
             ((35, 39), 481036337152),
             ((35, 41), 0),
             ((35, 42), 0),
             ((35, 43), 0),
             ((35, 44), 0),
             ((35, 45), 0),
             ((35, 49), 4398046511104),
             ((35, 50), 0),
             ((35, 51), 8796093022208),
             ((35, 52), 0),
             ((35, 53), 17592186044416),
             ((35, 56), 567347999932416),
             ((35, 59), 2260595906707456),
             ((35, 62), 9024791440785408),
             ((36, 0), 134480384),
             ((36, 4), 269488128),
             ((36, 9), 134479872),
             ((36, 12), 269484032),
             ((36, 15), 541065216),
             ((36, 18), 134217728),
             ((36, 19), 0),
             ((36, 20), 268435456),
             ((36, 21), 0),
             ((36, 22), 536870912),
             ((36, 26), 0),
             ((36, 27), 0),
             ((36, 28), 0),
             ((36, 29), 0),
             ((36, 30), 0),
             ((36, 32), 60129542144),
             ((36, 33), 51539607552),
             ((36, 34), 34359738368),
             ((36, 35), 0),
             ((36, 37), 0),
             ((36, 38), 137438953472),
             ((36, 39), 412316860416),
             ((36, 42), 0),
             ((36, 43), 0),
             ((36, 44), 0),
             ((36, 45), 0),
             ((36, 46), 0),
             ((36, 50), 8796093022208),
             ((36, 51), 0),
             ((36, 52), 17592186044416),
             ((36, 53), 0),
             ((36, 54), 35184372088832),
             ((36, 57), 1134695999864832),
             ((36, 60), 4521191813414912),
             ((36, 63), 18049582881570816),
             ((37, 1), 268960768),
             ((37, 5), 538976256),
             ((37, 10), 268959744),
             ((37, 13), 538968064),
             ((37, 19), 268435456),
             ((37, 20), 0),
             ((37, 21), 536870912),
             ((37, 22), 0),
             ((37, 23), 1073741824),
             ((37, 27), 0),
             ((37, 28), 0),
             ((37, 29), 0),
             ((37, 30), 0),
             ((37, 31), 0),
             ((37, 32), 128849018880),
             ((37, 33), 120259084288),
             ((37, 34), 103079215104),
             ((37, 35), 68719476736),
             ((37, 36), 0),
             ((37, 38), 0),
             ((37, 39), 274877906944),
             ((37, 43), 0),
             ((37, 44), 0),
             ((37, 45), 0),
             ((37, 46), 0),
             ((37, 47), 0),
             ((37, 51), 17592186044416),
             ((37, 52), 0),
             ((37, 53), 35184372088832),
             ((37, 54), 0),
             ((37, 55), 70368744177664),
             ((37, 58), 2269391999729664),
             ((37, 61), 9042383626829824),
             ((38, 2), 537921536),
             ((38, 6), 1077952512),
             ((38, 11), 537919488),
             ((38, 14), 1077936128),
             ((38, 20), 536870912),
             ((38, 21), 0),
             ((38, 22), 1073741824),
             ((38, 23), 0),
             ((38, 28), 0),
             ((38, 29), 0),
             ((38, 30), 0),
             ((38, 31), 0),
             ((38, 32), 266287972352),
             ((38, 33), 257698037760),
             ((38, 34), 240518168576),
             ((38, 35), 206158430208),
             ((38, 36), 137438953472),
             ((38, 37), 0),
             ((38, 39), 0),
             ((38, 44), 0),
             ((38, 45), 0),
             ((38, 46), 0),
             ((38, 47), 0),
             ((38, 52), 35184372088832),
             ((38, 53), 0),
             ((38, 54), 70368744177664),
             ((38, 55), 0),
             ((38, 59), 4538783999459328),
             ((38, 62), 18084767253659648),
             ((39, 3), 1075843072),
             ((39, 7), 2155905024),
             ((39, 12), 1075838976),
             ((39, 15), 2155872256),
             ((39, 21), 1073741824),
             ((39, 22), 0),
             ((39, 23), 2147483648),
             ((39, 29), 0),
             ((39, 30), 0),
             ((39, 31), 0),
             ((39, 32), 541165879296),
             ((39, 33), 532575944704),
             ((39, 34), 515396075520),
             ((39, 35), 481036337152),
             ((39, 36), 412316860416),
             ((39, 37), 274877906944),
             ((39, 38), 0),
             ((39, 45), 0),
             ((39, 46), 0),
             ((39, 47), 0),
             ((39, 53), 70368744177664),
             ((39, 54), 0),
             ((39, 55), 140737488355328),
             ((39, 60), 9077567998918656),
             ((39, 63), 36169534507319296),
             ((40, 0), 4311810304),
             ((40, 5), 8657571840),
             ((40, 8), 4311810048),
             ((40, 12), 8657567744),
             ((40, 16), 4311744512),
             ((40, 19), 8657043456),
             ((40, 24), 4294967296),
             ((40, 25), 0),
             ((40, 26), 8589934592),
             ((40, 32), 0),
             ((40, 33), 0),
             ((40, 34), 0),
             ((40, 41), 0),
             ((40, 42), 2199023255552),
             ((40, 43), 6597069766656),
             ((40, 44), 15393162788864),
             ((40, 45), 32985348833280),
             ((40, 46), 68169720922112),
             ((40, 47), 138538465099776),
             ((40, 48), 0),
             ((40, 49), 0),
             ((40, 50), 0),
             ((40, 56), 281474976710656),
             ((40, 57), 0),
             ((40, 58), 562949953421312),
             ((41, 1), 8623620608),
             ((41, 6), 17315143680),
             ((41, 9), 8623620096),
             ((41, 13), 17315135488),
             ((41, 17), 8623489024),
             ((41, 20), 17314086912),
             ((41, 24), 0),
             ((41, 25), 8589934592),
             ((41, 26), 0),
             ((41, 27), 17179869184),
             ((41, 32), 0),
             ((41, 33), 0),
             ((41, 34), 0),
             ((41, 35), 0),
             ((41, 40), 0),
             ((41, 42), 0),
             ((41, 43), 4398046511104),
             ((41, 44), 13194139533312),
             ((41, 45), 30786325577728),
             ((41, 46), 65970697666560),
             ((41, 47), 136339441844224),
             ((41, 48), 0),
             ((41, 49), 0),
             ((41, 50), 0),
             ((41, 51), 0),
             ((41, 56), 0),
             ((41, 57), 562949953421312),
             ((41, 58), 0),
             ((41, 59), 1125899906842624),
             ((42, 2), 17247241216),
             ((42, 7), 34630287360),
             ((42, 10), 17247240192),
             ((42, 14), 34630270976),
             ((42, 18), 17246978048),
             ((42, 21), 34628173824),
             ((42, 24), 8589934592),
             ((42, 25), 0),
             ((42, 26), 17179869184),
             ((42, 27), 0),
             ((42, 28), 34359738368),
             ((42, 32), 0),
             ((42, 33), 0),
             ((42, 34), 0),
             ((42, 35), 0),
             ((42, 36), 0),
             ((42, 40), 2199023255552),
             ((42, 41), 0),
             ((42, 43), 0),
             ((42, 44), 8796093022208),
             ((42, 45), 26388279066624),
             ((42, 46), 61572651155456),
             ((42, 47), 131941395333120),
             ((42, 48), 0),
             ((42, 49), 0),
             ((42, 50), 0),
             ((42, 51), 0),
             ((42, 52), 0),
             ((42, 56), 562949953421312),
             ((42, 57), 0),
             ((42, 58), 1125899906842624),
             ((42, 59), 0),
             ((42, 60), 2251799813685248),
             ((43, 3), 34494482432),
             ((43, 11), 34494480384),
             ((43, 15), 69260541952),
             ((43, 16), 17213423616),
             ((43, 19), 34493956096),
             ((43, 22), 69256347648),
             ((43, 25), 17179869184),
             ((43, 26), 0),
             ((43, 27), 34359738368),
             ((43, 28), 0),
             ((43, 29), 68719476736),
             ((43, 33), 0),
             ((43, 34), 0),
             ((43, 35), 0),
             ((43, 36), 0),
             ((43, 37), 0),
             ((43, 40), 6597069766656),
             ((43, 41), 4398046511104),
             ((43, 42), 0),
             ((43, 44), 0),
             ((43, 45), 17592186044416),
             ((43, 46), 52776558133248),
             ((43, 47), 123145302310912),
             ((43, 49), 0),
             ((43, 50), 0),
             ((43, 51), 0),
             ((43, 52), 0),
             ((43, 53), 0),
             ((43, 57), 1125899906842624),
             ((43, 58), 0),
             ((43, 59), 2251799813685248),
             ((43, 60), 0),
             ((43, 61), 4503599627370496),
             ((44, 4), 68988964864),
             ((44, 8), 34426978304),
             ((44, 12), 68988960768),
             ((44, 17), 34426847232),
             ((44, 20), 68987912192),
             ((44, 23), 138512695296),
             ((44, 26), 34359738368),
             ((44, 27), 0),
             ((44, 28), 68719476736),
             ((44, 29), 0),
             ((44, 30), 137438953472),
             ((44, 34), 0),
             ((44, 35), 0),
             ((44, 36), 0),
             ((44, 37), 0),
             ((44, 38), 0),
             ((44, 40), 15393162788864),
             ((44, 41), 13194139533312),
             ((44, 42), 8796093022208),
             ((44, 43), 0),
             ((44, 45), 0),
             ((44, 46), 35184372088832),
             ((44, 47), 105553116266496),
             ((44, 50), 0),
             ((44, 51), 0),
             ((44, 52), 0),
             ((44, 53), 0),
             ((44, 54), 0),
             ((44, 58), 2251799813685248),
             ((44, 59), 0),
             ((44, 60), 4503599627370496),
             ((44, 61), 0),
             ((44, 62), 9007199254740992),
             ((45, 0), 68853957120),
             ((45, 5), 137977929728),
             ((45, 9), 68853956608),
             ((45, 13), 137977921536),
             ((45, 18), 68853694464),
             ((45, 21), 137975824384),
             ((45, 27), 68719476736),
             ((45, 28), 0),
             ((45, 29), 137438953472),
             ((45, 30), 0),
             ((45, 31), 274877906944),
             ((45, 35), 0),
             ((45, 36), 0),
             ((45, 37), 0),
             ((45, 38), 0),
             ((45, 39), 0),
             ((45, 40), 32985348833280),
             ((45, 41), 30786325577728),
             ((45, 42), 26388279066624),
             ((45, 43), 17592186044416),
             ((45, 44), 0),
             ((45, 46), 0),
             ((45, 47), 70368744177664),
             ((45, 51), 0),
             ((45, 52), 0),
             ((45, 53), 0),
             ((45, 54), 0),
             ((45, 55), 0),
             ((45, 59), 4503599627370496),
             ((45, 60), 0),
             ((45, 61), 9007199254740992),
             ((45, 62), 0),
             ((45, 63), 18014398509481984),
             ((46, 1), 137707914240),
             ((46, 6), 275955859456),
             ((46, 10), 137707913216),
             ((46, 14), 275955843072),
             ((46, 19), 137707388928),
             ((46, 22), 275951648768),
             ((46, 28), 137438953472),
             ((46, 29), 0),
             ((46, 30), 274877906944),
             ((46, 31), 0),
             ((46, 36), 0),
             ((46, 37), 0),
             ((46, 38), 0),
             ((46, 39), 0),
             ((46, 40), 68169720922112),
             ((46, 41), 65970697666560),
             ((46, 42), 61572651155456),
             ((46, 43), 52776558133248),
             ((46, 44), 35184372088832),
             ((46, 45), 0),
             ((46, 47), 0),
             ((46, 52), 0),
             ((46, 53), 0),
             ((46, 54), 0),
             ((46, 55), 0),
             ((46, 60), 9007199254740992),
             ((46, 61), 0),
             ((46, 62), 18014398509481984),
             ((46, 63), 0),
             ((47, 2), 275415828480),
             ((47, 7), 551911718912),
             ((47, 11), 275415826432),
             ((47, 15), 551911686144),
             ((47, 20), 275414777856),
             ((47, 23), 551903297536),
             ((47, 29), 274877906944),
             ((47, 30), 0),
             ((47, 31), 549755813888),
             ((47, 37), 0),
             ((47, 38), 0),
             ((47, 39), 0),
             ((47, 40), 138538465099776),
             ((47, 41), 136339441844224),
             ((47, 42), 131941395333120),
             ((47, 43), 123145302310912),
             ((47, 44), 105553116266496),
             ((47, 45), 70368744177664),
             ((47, 46), 0),
             ((47, 53), 0),
             ((47, 54), 0),
             ((47, 55), 0),
             ((47, 61), 18014398509481984),
             ((47, 62), 0),
             ((47, 63), 36028797018963968),
             ((48, 0), 1103823438080),
             ((48, 6), 2216338399232),
             ((48, 8), 1103823437824),
             ((48, 13), 2216338391040),
             ((48, 16), 1103823372288),
             ((48, 20), 2216337342464),
             ((48, 24), 1103806595072),
             ((48, 27), 2216203124736),
             ((48, 32), 1099511627776),
             ((48, 33), 0),
             ((48, 34), 2199023255552),
             ((48, 40), 0),
             ((48, 41), 0),
             ((48, 42), 0),
             ((48, 49), 0),
             ((48, 50), 562949953421312),
             ((48, 51), 1688849860263936),
             ((48, 52), 3940649673949184),
             ((48, 53), 8444249301319680),
             ((48, 54), 17451448556060672),
             ((48, 55), 35465847065542656),
             ((48, 56), 0),
             ((48, 57), 0),
             ((48, 58), 0),
             ((49, 1), 2207646876160),
             ((49, 7), 4432676798464),
             ((49, 9), 2207646875648),
             ((49, 14), 4432676782080),
             ((49, 17), 2207646744576),
             ((49, 21), 4432674684928),
             ((49, 25), 2207613190144),
             ((49, 28), 4432406249472),
             ((49, 32), 0),
             ((49, 33), 2199023255552),
             ((49, 34), 0),
             ((49, 35), 4398046511104),
             ((49, 40), 0),
             ((49, 41), 0),
             ((49, 42), 0),
             ((49, 43), 0),
             ((49, 48), 0),
             ((49, 50), 0),
             ((49, 51), 1125899906842624),
             ((49, 52), 3377699720527872),
             ((49, 53), 7881299347898368),
             ((49, 54), 16888498602639360),
             ((49, 55), 34902897112121344),
             ((49, 56), 0),
             ((49, 57), 0),
             ((49, 58), 0),
             ((49, 59), 0),
             ((50, 2), 4415293752320),
             ((50, 10), 4415293751296),
             ((50, 15), 8865353564160),
             ((50, 18), 4415293489152),
             ((50, 22), 8865349369856),
             ((50, 26), 4415226380288),
             ((50, 29), 8864812498944),
             ((50, 32), 2199023255552),
             ((50, 33), 0),
             ((50, 34), 4398046511104),
             ((50, 35), 0),
             ((50, 36), 8796093022208),
             ((50, 40), 0),
             ((50, 41), 0),
             ((50, 42), 0),
             ((50, 43), 0),
             ((50, 44), 0),
             ((50, 48), 562949953421312),
             ((50, 49), 0),
             ((50, 51), 0),
             ((50, 52), 2251799813685248),
             ((50, 53), 6755399441055744),
             ((50, 54), 15762598695796736),
             ((50, 55), 33776997205278720),
             ((50, 56), 0),
             ((50, 57), 0),
             ((50, 58), 0),
             ((50, 59), 0),
             ((50, 60), 0),
             ((51, 3), 8830587504640),
             ((51, 11), 8830587502592),
             ((51, 19), 8830586978304),
             ((51, 23), 17730698739712),
             ((51, 24), 4406636445696),
             ((51, 27), 8830452760576),
             ((51, 30), 17729624997888),
             ((51, 33), 4398046511104),
             ((51, 34), 0),
             ((51, 35), 8796093022208),
             ((51, 36), 0),
             ((51, 37), 17592186044416),
             ((51, 41), 0),
             ((51, 42), 0),
             ((51, 43), 0),
             ((51, 44), 0),
             ((51, 45), 0),
             ((51, 48), 1688849860263936),
             ((51, 49), 1125899906842624),
             ((51, 50), 0),
             ((51, 52), 0),
             ((51, 53), 4503599627370496),
             ((51, 54), 13510798882111488),
             ((51, 55), 31525197391593472),
             ((51, 57), 0),
             ((51, 58), 0),
             ((51, 59), 0),
             ((51, 60), 0),
             ((51, 61), 0),
             ((52, 4), 17661175009280),
             ((52, 12), 17661175005184),
             ((52, 16), 8813306445824),
             ((52, 20), 17661173956608),
             ((52, 25), 8813272891392),
             ((52, 28), 17660905521152),
             ((52, 31), 35459249995776),
             ((52, 34), 8796093022208),
             ((52, 35), 0),
             ((52, 36), 17592186044416),
             ((52, 37), 0),
             ((52, 38), 35184372088832),
             ((52, 42), 0),
             ((52, 43), 0),
             ((52, 44), 0),
             ((52, 45), 0),
             ((52, 46), 0),
             ((52, 48), 3940649673949184),
             ((52, 49), 3377699720527872),
             ((52, 50), 2251799813685248),
             ((52, 51), 0),
             ((52, 53), 0),
             ((52, 54), 9007199254740992),
             ((52, 55), 27021597764222976),
             ((52, 58), 0),
             ((52, 59), 0),
             ((52, 60), 0),
             ((52, 61), 0),
             ((52, 62), 0),
             ((53, 5), 35322350018560),
             ((53, 8), 17626613022720),
             ((53, 13), 35322350010368),
             ((53, 17), 17626612891648),
             ((53, 21), 35322347913216),
             ((53, 26), 17626545782784),
             ((53, 29), 35321811042304),
             ((53, 35), 17592186044416),
             ((53, 36), 0),
             ((53, 37), 35184372088832),
             ((53, 38), 0),
             ((53, 39), 70368744177664),
             ((53, 43), 0),
             ((53, 44), 0),
             ((53, 45), 0),
             ((53, 46), 0),
             ((53, 47), 0),
             ((53, 48), 8444249301319680),
             ((53, 49), 7881299347898368),
             ((53, 50), 6755399441055744),
             ((53, 51), 4503599627370496),
             ((53, 52), 0),
             ((53, 54), 0),
             ((53, 55), 18014398509481984),
             ((53, 59), 0),
             ((53, 60), 0),
             ((53, 61), 0),
             ((53, 62), 0),
             ((53, 63), 0),
             ((54, 0), 35253226045952),
             ((54, 6), 70644700037120),
             ((54, 9), 35253226045440),
             ((54, 14), 70644700020736),
             ((54, 18), 35253225783296),
             ((54, 22), 70644695826432),
             ((54, 27), 35253091565568),
             ((54, 30), 70643622084608),
             ((54, 36), 35184372088832),
             ((54, 37), 0),
             ((54, 38), 70368744177664),
             ((54, 39), 0),
             ((54, 44), 0),
             ((54, 45), 0),
             ((54, 46), 0),
             ((54, 47), 0),
             ((54, 48), 17451448556060672),
             ((54, 49), 16888498602639360),
             ((54, 50), 15762598695796736),
             ((54, 51), 13510798882111488),
             ((54, 52), 9007199254740992),
             ((54, 53), 0),
             ((54, 55), 0),
             ((54, 60), 0),
             ((54, 61), 0),
             ((54, 62), 0),
             ((54, 63), 0),
             ((55, 1), 70506452091904),
             ((55, 7), 141289400074240),
             ((55, 10), 70506452090880),
             ((55, 15), 141289400041472),
             ((55, 19), 70506451566592),
             ((55, 23), 141289391652864),
             ((55, 28), 70506183131136),
             ((55, 31), 141287244169216),
             ((55, 37), 70368744177664),
             ((55, 38), 0),
             ((55, 39), 140737488355328),
             ((55, 45), 0),
             ((55, 46), 0),
             ((55, 47), 0),
             ((55, 48), 35465847065542656),
             ((55, 49), 34902897112121344),
             ((55, 50), 33776997205278720),
             ((55, 51), 31525197391593472),
             ((55, 52), 27021597764222976),
             ((55, 53), 18014398509481984),
             ((55, 54), 0),
             ((55, 61), 0),
             ((55, 62), 0),
             ((55, 63), 0),
             ((56, 0), 282578800148736),
             ((56, 7), 567382630219776),
             ((56, 8), 282578800148480),
             ((56, 14), 567382630203392),
             ((56, 16), 282578800082944),
             ((56, 21), 567382628106240),
             ((56, 24), 282578783305728),
             ((56, 28), 567382359670784),
             ((56, 32), 282574488338432),
             ((56, 35), 567347999932416),
             ((56, 40), 281474976710656),
             ((56, 41), 0),
             ((56, 42), 562949953421312),
             ((56, 48), 0),
             ((56, 49), 0),
             ((56, 50), 0),
             ((56, 57), 0),
             ((56, 58), 144115188075855872),
             ((56, 59), 432345564227567616),
             ((56, 60), 1008806316530991104),
             ((56, 61), 2161727821137838080),
             ((56, 62), 4467570830351532032),
             ((56, 63), 9079256848778919936),
             ((57, 1), 565157600297472),
             ((57, 9), 565157600296960),
             ((57, 15), 1134765260406784),
             ((57, 17), 565157600165888),
             ((57, 22), 1134765256212480),
             ((57, 25), 565157566611456),
             ((57, 29), 1134764719341568),
             ((57, 33), 565148976676864),
             ((57, 36), 1134695999864832),
             ((57, 40), 0),
             ((57, 41), 562949953421312),
             ((57, 42), 0),
             ((57, 43), 1125899906842624),
             ((57, 48), 0),
             ((57, 49), 0),
             ((57, 50), 0),
             ((57, 51), 0),
             ((57, 56), 0),
             ((57, 58), 0),
             ((57, 59), 288230376151711744),
             ((57, 60), 864691128455135232),
             ((57, 61), 2017612633061982208),
             ((57, 62), 4323455642275676160),
             ((57, 63), 8935141660703064064),
             ((58, 2), 1130315200594944),
             ((58, 10), 1130315200593920),
             ((58, 18), 1130315200331776),
             ((58, 23), 2269530512424960),
             ((58, 26), 1130315133222912),
             ((58, 30), 2269529438683136),
             ((58, 34), 1130297953353728),
             ((58, 37), 2269391999729664),
             ((58, 40), 562949953421312),
             ((58, 41), 0),
             ((58, 42), 1125899906842624),
             ((58, 43), 0),
             ((58, 44), 2251799813685248),
             ((58, 48), 0),
             ((58, 49), 0),
             ((58, 50), 0),
             ((58, 51), 0),
             ((58, 52), 0),
             ((58, 56), 144115188075855872),
             ((58, 57), 0),
             ((58, 59), 0),
             ((58, 60), 576460752303423488),
             ((58, 61), 1729382256910270464),
             ((58, 62), 4035225266123964416),
             ((58, 63), 8646911284551352320),
             ((59, 3), 2260630401189888),
             ((59, 11), 2260630401187840),
             ((59, 19), 2260630400663552),
             ((59, 27), 2260630266445824),
             ((59, 31), 4539058877366272),
             ((59, 32), 1128098930098176),
             ((59, 35), 2260595906707456),
             ((59, 38), 4538783999459328),
             ((59, 41), 1125899906842624),
             ((59, 42), 0),
             ((59, 43), 2251799813685248),
             ((59, 44), 0),
             ((59, 45), 4503599627370496),
             ((59, 49), 0),
             ((59, 50), 0),
             ((59, 51), 0),
             ((59, 52), 0),
             ((59, 53), 0),
             ((59, 56), 432345564227567616),
             ((59, 57), 288230376151711744),
             ((59, 58), 0),
             ((59, 60), 0),
             ((59, 61), 1152921504606846976),
             ((59, 62), 3458764513820540928),
             ((59, 63), 8070450532247928832),
             ((60, 4), 4521260802379776),
             ((60, 12), 4521260802375680),
             ((60, 20), 4521260801327104),
             ((60, 24), 2256206450130944),
             ((60, 28), 4521260532891648),
             ((60, 33), 2256197860196352),
             ((60, 36), 4521191813414912),
             ((60, 39), 9077567998918656),
             ((60, 42), 2251799813685248),
             ((60, 43), 0),
             ((60, 44), 4503599627370496),
             ((60, 45), 0),
             ((60, 46), 9007199254740992),
             ((60, 50), 0),
             ((60, 51), 0),
             ((60, 52), 0),
             ((60, 53), 0),
             ((60, 54), 0),
             ((60, 56), 1008806316530991104),
             ((60, 57), 864691128455135232),
             ((60, 58), 576460752303423488),
             ((60, 59), 0),
             ((60, 61), 0),
             ((60, 62), 2305843009213693952),
             ((60, 63), 6917529027641081856),
             ((61, 5), 9042521604759552),
             ((61, 13), 9042521604751360),
             ((61, 16), 4512412933816320),
             ((61, 21), 9042521602654208),
             ((61, 25), 4512412900261888),
             ((61, 29), 9042521065783296),
             ((61, 34), 4512395720392704),
             ((61, 37), 9042383626829824),
             ((61, 43), 4503599627370496),
             ((61, 44), 0),
             ((61, 45), 9007199254740992),
             ((61, 46), 0),
             ((61, 47), 18014398509481984),
             ((61, 51), 0),
             ((61, 52), 0),
             ((61, 53), 0),
             ((61, 54), 0),
             ((61, 55), 0),
             ((61, 56), 2161727821137838080),
             ((61, 57), 2017612633061982208),
             ((61, 58), 1729382256910270464),
             ((61, 59), 1152921504606846976),
             ((61, 60), 0),
             ((61, 62), 0),
             ((61, 63), 4611686018427387904),
             ((62, 6), 18085043209519104),
             ((62, 8), 9024825867763712),
             ((62, 14), 18085043209502720),
             ((62, 17), 9024825867632640),
             ((62, 22), 18085043205308416),
             ((62, 26), 9024825800523776),
             ((62, 30), 18085042131566592),
             ((62, 35), 9024791440785408),
             ((62, 38), 18084767253659648),
             ((62, 44), 9007199254740992),
             ((62, 45), 0),
             ((62, 46), 18014398509481984),
             ((62, 47), 0),
             ((62, 52), 0),
             ((62, 53), 0),
             ((62, 54), 0),
             ((62, 55), 0),
             ((62, 56), 4467570830351532032),
             ((62, 57), 4323455642275676160),
             ((62, 58), 4035225266123964416),
             ((62, 59), 3458764513820540928),
             ((62, 60), 2305843009213693952),
             ((62, 61), 0),
             ((62, 63), 0),
             ((63, 0), 18049651735527936),
             ((63, 7), 36170086419038208),
             ((63, 9), 18049651735527424),
             ((63, 15), 36170086419005440),
             ((63, 18), 18049651735265280),
             ((63, 23), 36170086410616832),
             ((63, 27), 18049651601047552),
             ((63, 31), 36170084263133184),
             ((63, 36), 18049582881570816),
             ((63, 39), 36169534507319296),
             ((63, 45), 18014398509481984),
             ((63, 46), 0),
             ((63, 47), 36028797018963968),
             ((63, 53), 0),
             ((63, 54), 0),
             ((63, 55), 0),
             ((63, 56), 9079256848778919936),
             ((63, 57), 8935141660703064064),
             ((63, 58), 8646911284551352320),
             ((63, 59), 8070450532247928832),
             ((63, 60), 6917529027641081856),
             ((63, 61), 4611686018427387904),
             ((63, 62), 0),
             ])


class EncodingConstants(unittest.TestCase):

    def setUp(self):
        pass

    def tearDown(self):
        pass

    def test_encodings(self):
        self.assertEqual(
            [(e, v) for e, v in enumerate(constants.MOVE_NUMBER_KEYS)],
            [(0, '0'),
             (1, '11'),
             (2, '12'),
             (3, '13'),
             (4, '14'),
             (5, '15'),
             (6, '16'),
             (7, '17'),
             (8, '18'),
             (9, '19'),
             (10, '1a'),
             (11, '1b'),
             (12, '1c'),
             (13, '1d'),
             (14, '1e'),
             (15, '1f'),
             (16, '210'),
             (17, '211'),
             (18, '212'),
             (19, '213'),
             (20, '214'),
             (21, '215'),
             (22, '216'),
             (23, '217'),
             (24, '218'),
             (25, '219'),
             (26, '21a'),
             (27, '21b'),
             (28, '21c'),
             (29, '21d'),
             (30, '21e'),
             (31, '21f'),
             (32, '220'),
             (33, '221'),
             (34, '222'),
             (35, '223'),
             (36, '224'),
             (37, '225'),
             (38, '226'),
             (39, "227"),
             (40, '228'),
             (41, '229'),
             (42, '22a'),
             (43, '22b'),
             (44, '22c'),
             (45, '22d'),
             (46, '22e'),
             (47, '22f'),
             (48, '230'),
             (49, '231'),
             (50, '232'),
             (51, '233'),
             (52, '234'),
             (53, '235'),
             (54, '236'),
             (55, '237'),
             (56, '238'),
             (57, '239'),
             (58, '23a'),
             (59, '23b'),
             (60, '23c'),
             (61, '23d'),
             (62, '23e'),
             (63, '23f'),
             (64, '240'),
             (65, '241'),
             (66, '242'),
             (67, '243'),
             (68, '244'),
             (69, '245'),
             (70, '246'),
             (71, '247'),
             (72, '248'),
             (73, '249'),
             (74, '24a'),
             (75, '24b'),
             (76, '24c'),
             (77, '24d'),
             (78, '24e'),
             (79, '24f'),
             (80, '250'),
             (81, '251'),
             (82, '252'),
             (83, '253'),
             (84, '254'),
             (85, '255'),
             (86, '256'),
             (87, '257'),
             (88, '258'),
             (89, '259'),
             (90, '25a'),
             (91, '25b'),
             (92, '25c'),
             (93, '25d'),
             (94, '25e'),
             (95, '25f'),
             (96, '260'),
             (97, '261'),
             (98, '262'),
             (99, '263'),
             (100, '264'),
             (101, '265'),
             (102, '266'),
             (103, '267'),
             (104, '268'),
             (105, '269'),
             (106, '26a'),
             (107, '26b'),
             (108, '26c'),
             (109, '26d'),
             (110, '26e'),
             (111, '26f'),
             (112, '270'),
             (113, '271'),
             (114, '272'),
             (115, '273'),
             (116, '274'),
             (117, '275'),
             (118, '276'),
             (119, '277'),
             (120, '278'),
             (121, '279'),
             (122, '27a'),
             (123, '27b'),
             (124, '27c'),
             (125, '27d'),
             (126, '27e'),
             (127, '27f'),
             (128, '280'),
             (129, '281'),
             (130, '282'),
             (131, '283'),
             (132, '284'),
             (133, '285'),
             (134, '286'),
             (135, '287'),
             (136, '288'),
             (137, '289'),
             (138, '28a'),
             (139, '28b'),
             (140, '28c'),
             (141, '28d'),
             (142, '28e'),
             (143, '28f'),
             (144, '290'),
             (145, '291'),
             (146, '292'),
             (147, '293'),
             (148, '294'),
             (149, '295'),
             (150, '296'),
             (151, '297'),
             (152, '298'),
             (153, '299'),
             (154, '29a'),
             (155, '29b'),
             (156, '29c'),
             (157, '29d'),
             (158, '29e'),
             (159, '29f'),
             (160, '2a0'),
             (161, '2a1'),
             (162, '2a2'),
             (163, '2a3'),
             (164, '2a4'),
             (165, '2a5'),
             (166, '2a6'),
             (167, '2a7'),
             (168, '2a8'),
             (169, '2a9'),
             (170, '2aa'),
             (171, '2ab'),
             (172, '2ac'),
             (173, '2ad'),
             (174, '2ae'),
             (175, '2af'),
             (176, '2b0'),
             (177, '2b1'),
             (178, '2b2'),
             (179, '2b3'),
             (180, '2b4'),
             (181, '2b5'),
             (182, '2b6'),
             (183, '2b7'),
             (184, '2b8'),
             (185, '2b9'),
             (186, '2ba'),
             (187, '2bb'),
             (188, '2bc'),
             (189, '2bd'),
             (190, '2be'),
             (191, '2bf'),
             (192, '2c0'),
             (193, '2c1'),
             (194, '2c2'),
             (195, '2c3'),
             (196, '2c4'),
             (197, '2c5'),
             (198, '2c6'),
             (199, '2c7'),
             (200, '2c8'),
             (201, '2c9'),
             (202, '2ca'),
             (203, '2cb'),
             (204, '2cc'),
             (205, '2cd'),
             (206, '2ce'),
             (207, '2cf'),
             (208, '2d0'),
             (209, '2d1'),
             (210, '2d2'),
             (211, '2d3'),
             (212, '2d4'),
             (213, '2d5'),
             (214, '2d6'),
             (215, '2d7'),
             (216, '2d8'),
             (217, '2d9'),
             (218, '2da'),
             (219, '2db'),
             (220, '2dc'),
             (221, '2dd'),
             (222, '2de'),
             (223, '2df'),
             (224, '2e0'),
             (225, '2e1'),
             (226, '2e2'),
             (227, '2e3'),
             (228, '2e4'),
             (229, '2e5'),
             (230, '2e6'),
             (231, '2e7'),
             (232, '2e8'),
             (233, '2e9'),
             (234, '2ea'),
             (235, '2eb'),
             (236, '2ec'),
             (237, '2ed'),
             (238, '2ee'),
             (239, '2ef'),
             (240, '2f0'),
             (241, '2f1'),
             (242, '2f2'),
             (243, '2f3'),
             (244, '2f4'),
             (245, '2f5'),
             (246, '2f6'),
             (247, '2f7'),
             (248, '2f8'),
             (249, '2f9'),
             (250, '2fa'),
             (251, '2fb'),
             (252, '2fc'),
             (253, '2fd'),
             (254, '2fe'),
             (255, '2ff'),
             ])


if __name__ == '__main__':
    runner = unittest.TextTestRunner
    loader = unittest.defaultTestLoader.loadTestsFromTestCase
    runner().run(loader(PGNParserConstants))
    runner().run(loader(FENConstants))
    runner().run(loader(BishopMovesConstants))
    runner().run(loader(QueenMovesConstants))
    runner().run(loader(KnightMovesConstants))
    runner().run(loader(KingMovesConstants))
    runner().run(loader(RookMovesConstants))
    runner().run(loader(WhitePawnMovesToSquareConstants))
    runner().run(loader(WhitePawnCapturesToSquareConstants))
    runner().run(loader(BlackPawnMovesToSquareConstants))
    runner().run(loader(BlackPawnCapturesToSquareConstants))
    runner().run(loader(GAPSConstants))
    runner().run(loader(EncodingConstants))
