# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['sql_athame']

package_data = \
{'': ['*']}

extras_require = \
{'asyncpg': ['asyncpg']}

setup_kwargs = {
    'name': 'sql-athame',
    'version': '0.3.10',
    'description': 'Python tool for slicing and dicing SQL',
    'long_description': '# sql-athame\n\nPython tool for slicing and dicing SQL.  Its intended target is\nPostgres with _asyncpg_.\n\n## API Reference\n\n```python\nfrom sql_athame import sql\n```\n\n### sql(fmt: str, \\*args, \\*\\*kwargs) -> Fragment\n\nCreates a SQL `Fragment` from the `fmt` string.  The `fmt` string\ncontains literal SQL and may contain positional references, marked by\n`{}`, and named references, marked by `{name}`.  Positional references\n_must_ have a matching argument in `*args`.  Named references _may_\nhave a matching argument in `**kwargs`; if a named reference is not\nfullfilled by `**kwargs` it remains as a named _slot_ to be filled\nlater.\n\nIf a referenced argument is a `Fragment`, it is substituted into the\nSQL along with all of its embedded placeholders if any.  Otherwise, it\nis treated as a placeholder value and substituted in place as a\nplaceholder.\n\n### Fragment.query(self) -> Tuple[str, List[Any]]\n\nRenders a SQL `Fragment` into a query string and list of placeholder\nparameters.\n\n```python\n>>> q = sql("SELECT * FROM tbl WHERE qty > {qty}", qty=10)\n>>> q.query()\n(\'SELECT * FROM tbl WHERE qty > $1\', [10])\n```\n\nIf there are any unfilled _slots_ `ValueError` will be raised.\n\n```python\n>>> q = sql("SELECT * FROM tbl WHERE qty > {qty}")\n>>> q.query()\nValueError: Unfilled slot: \'qty\'\n>>> q.fill(qty=10).query()\n(\'SELECT * FROM tbl WHERE qty > $1\', [10])\n```\n\n### Fragment.\\_\\_iter\\_\\_(self) -> Iterator[Any]:\n\nA `Fragment` is an iterable which will return the query string\nfollowed by the placeholder parameters as returned by\n`Fragment.query(self)`.  This matches the `(query, *args)` argument\npattern of the _asyncpg_ API:\n\n```python\nq = sql("SELECT * FROM tbl WHERE qty > {}", 10)\nawait conn.fetch(*q)\n```\n\n### sql.list(parts: Iterable[Fragment]) -> Fragment:\n\nCreates a SQL `Fragment` joining the fragments in `parts` together\nwith commas.\n\n```python\n>>> cols = [sql("a"), sql("b"), sql("c")]\n>>> list(sql("SELECT {cols} FROM tbl", cols=sql.list(cols)))\n[\'SELECT a, b, c FROM tbl\']\n```\n\n### sql.all(parts: Iterable[Fragment]) -> Fragment:\n\nCreates a SQL `Fragment` joining the fragments in `parts` together\nwith `AND`.  If `parts` is empty, returns `TRUE`.\n\n```python\n>>> where = [sql("a = {}", 42), sql("x <> {}", "foo")]\n>>> list(sql("SELECT * FROM tbl WHERE {}", sql.all(where)))\n[\'SELECT * FROM tbl WHERE (a = $1) AND (x <> $2)\', 42, \'foo\']\n>>> list(sql("SELECT * FROM tbl WHERE {}", sql.all([])))\n[\'SELECT * FROM tbl WHERE TRUE\']\n```\n\n### sql.any(parts: Iterable[Fragment]) -> Fragment:\n\nCreates a SQL `Fragment` joining the fragments in `parts` together\nwith `OR`.  If `parts` is empty, returns `FALSE`.\n\n```python\n>>> where = [sql("a = {}", 42), sql("x <> {}", "foo")]\n>>> list(sql("SELECT * FROM tbl WHERE {}", sql.any(where)))\n[\'SELECT * FROM tbl WHERE (a = $1) OR (x <> $2)\', 42, \'foo\']\n>>> list(sql("SELECT * FROM tbl WHERE {}", sql.any([])))\n[\'SELECT * FROM tbl WHERE FALSE\']\n```\n\n### Fragment.join(self, parts: Iterable[Fragment]) -> Fragment:\n\nCreates a SQL `Fragment` by joining the fragments in `parts` together\nwith `self`.\n\n```python\n>>> clauses = [sql("WHEN {} THEN {}", a, b) for a, b in ((sql("a"), 1), (sql("b"), 2))]\n>>> case = sql("CASE {clauses} END", clauses=sql(" ").join(clauses))\n>>> list(case)\n[\'CASE WHEN a THEN $1 WHEN b THEN $2 END\', 1, 2]\n```\n\n### sql.literal(text: str) -> Fragment\n\nCreates a SQL `Fragment` with the literal SQL `text`.  No substitution\nof any kind is performed.  **Be very careful of SQL injection.**\n\n### sql.identifier(name: str, prefix: Optional[str] = None) -> Fragment\n\nCreates a SQL `Fragment` with a quoted identifier name, optionally\nwith a dotted prefix.\n\n```python\n>>> list(sql("SELECT {a} FROM tbl", a=sql.identifier("a", prefix="tbl")))\n[\'SELECT "tbl"."a" FROM tbl\']\n```\n\n### sql.value(value: Any) -> Fragment\n\nCreates a SQL `Fragment` with a single placeholder to `value`.\nEquivalent to:\n\n```python\nsql("{}", value)\n```\n\n### sql.escape(value: Any) -> Fragment\n\nCreates a SQL `Fragment` with `value` escaped and embedded into the\nSQL.  Types currently supported are strings, floats, ints, UUIDs, and\nsequences of the above.\n\n```python\n>>> list(sql("SELECT * FROM tbl WHERE qty = ANY({})", sql.escape([1, 3, 5])))\n[\'SELECT * FROM tbl WHERE qty = ANY(ARRAY[1, 3, 5])\']\n```\n\nCompare to with a placeholder:\n\n```python\n>>> list(sql("SELECT * FROM tbl WHERE qty = ANY({})", [1, 3, 5]))\n[\'SELECT * FROM tbl WHERE qty = ANY($1)\', [1, 3, 5]]\n```\n\n"Burning" an invariant value into the query can potentially help the\nquery optimizer.\n\n### sql.slot(name: str) -> Fragment\n\nCreates a SQL `Fragment` with a single empty _slot_ named `name`.\nEquivalent to:\n\n```python\nsql("{name}")\n```\n\n### Fragment.fill(self, \\*\\*kwargs) -> Fragment\n\nCreates a SQL `Fragment` by filling any empty _slots_ in `self` with\n`kwargs`.  Similar to `sql` subtitution, if a value is a `Fragment` it\nis substituted in-place, otherwise it is substituted as a placeholder.\n\n### Fragment.compile(self) -> Callable[..., Fragment]\n\nCreates a function that when called with `**kwargs` will create a SQL\n`Fragment` equivalent to calling `self.fill(**kwargs)`.  This is\noptimized to do as much work as possible up front and can be\nconsiderably faster if repeated often.\n\n### Fragment.prepare(self) -> Tuple[str, Callable[..., List[Any]]]\n\nRenders `self` into a SQL query string; returns that string and a\nfunction that when called with `**kwargs` containing the unfilled\nslots of `self` will return a list containing the placeholder values\nfor `self` as filled with `**kwargs`.\n\n```python\n>>> query, query_args = sql("UPDATE tbl SET foo={foo}, bar={bar} WHERE baz < {baz}", baz=10).prepare()\n>>> query\n\'UPDATE tbl SET foo=$1, bar=$2 WHERE baz < $3\'\n>>> query_args(foo=1, bar=2)\n[1, 2, 10]\n>>> query_args(bar=42, foo=3)\n[3, 42, 10]\n```\n\nAs the name implies this is intended to be used in prepared\nstatements:\n\n```python\nquery, query_args = sql("UPDATE tbl SET foo={foo}, bar={bar} WHERE baz < {baz}", baz=10).prepare()\nstmt = await conn.prepare(query)\nawait stmt.execute(*query_args(foo=1, bar=2))\nawait stmt.execute(*query_args(bar=42, foo=3))\n```\n\n## Example\n\n```python\nfrom sql_athame import sql\n\n\ndef get_orders(query):\n    where = []\n\n    if "id" in query:\n        where.append(sql("id = {}", query["id"]))\n    if "eventId" in query:\n        where.append(sql("event_id = {}", query["eventId"]))\n    if "startTime" in query:\n        where.append(sql("start_time = {}", query["startTime"]))\n    if "from" in query:\n        where.append(sql("start_time >= {}", query["from"]))\n    if "until" in query:\n        where.append(sql("start_time < {}", query["until"]))\n\n    return sql("SELECT * FROM orders WHERE {}", sql.all(where))\n\n\nprint(get_orders({}).query())\n# (\'SELECT * FROM orders WHERE TRUE\', [])\n\nprint(list(get_orders({})))\n# [\'SELECT * FROM orders WHERE TRUE\']\n\nprint(get_orders({"id": "xyzzy"}).query())\n# (\'SELECT * FROM orders WHERE TRUE AND id = $1\', [\'xyzzy\'])\n\nprint(list(get_orders({"id": "xyzzy"})))\n# [\'SELECT * FROM orders WHERE TRUE AND id = $1\', \'xyzzy\']\n\nprint(\n    *get_orders(\n        {"eventId": "plugh", "from": "2019-05-01", "until": "2019-08-26"}\n    )\n)\n# SELECT * FROM orders WHERE TRUE AND event_id = $1 AND start_time >= $2 AND start_time < $3 [\'plugh\', \'2019-05-01\', \'2019-08-26\']\n\n\nsuperquery = sql(\n    """\n    SELECT *\n      FROM ({subquery}) sq\n      JOIN other_table ot ON (ot.id = sq.id)\n      WHERE ot.foo = {foo}\n      LIMIT {limit}\n    """,\n    subquery=get_orders({"id": "xyzzy"}),\n    foo="bork",\n    limit=50,\n)\nprint(superquery.query())\n# ("""\n#     SELECT *\n#       FROM (SELECT * FROM orders WHERE TRUE AND id = $1) sq\n#       JOIN other_table ot ON (ot.id = sq.id)\n#       WHERE ot.foo = $2\n#       LIMIT $3\n#     """, [\'xyzzy\', \'bork\', 50])\n```\n\n## License\n\nMIT.\n\n---\nCopyright (c) 2019, 2020 Brian Downing\n',
    'author': 'Brian Downing',
    'author_email': 'bdowning@lavos.net',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/bdowning/sql-athame',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
