import ast
import filecmp
import json
import time

import pytest
from click.testing import CliRunner

from commands import report
from commands.logout import cli as logout
from commands.workspaces import cli as workspaces
from helpers.test_utils import assert_success
from neoload_cli_lib import rest_crud

import os

@pytest.mark.report
@pytest.mark.usefixtures("neoload_login")  # it's like @Before on the neoload_login function
class TestTrendsJsonOutput:

    @pytest.mark.datafiles('tests/resources/report/expected_trends.json')
    def test_parse_source_data_spec(self, monkeypatch, datafiles):
        runner = CliRunner()
        result_ws = runner.invoke(workspaces, ['use', '5f689c3f0860270001606902'])
        assert_success(result_ws)

        monkeypatch.setattr(rest_crud, 'get',
                            lambda actual_endpoint, params=None: ast.literal_eval(self.__return_json(actual_endpoint)))
        monkeypatch.setattr(time, 'localtime', lambda timestamp_utc: time.gmtime(timestamp_utc))
        model = report.initialize_model("result=-3", "")
        json_data = report.parse_source_data_spec(None, model, "trends", "8725e0cd-92a4-4105-803b-86433851fcfc")

        result_logout = runner.invoke(logout)
        assert_success(result_logout)

        with open('tests/resources/report/actual_trends.json', 'w', newline='\n') as f:
            f.write(json.dumps(json_data, indent=2))

        msg = ""
        statinfo = os.stat(datafiles.listdir()[0])
        msg += "{}".format({ 'statinfo': statinfo, 'st_size': statinfo.st_size })
        statinfo = os.stat('tests/resources/report/actual_trends.json')
        msg += "{}".format({ 'statinfo': statinfo, 'st_size': statinfo.st_size })

        assert filecmp.cmp(datafiles.listdir()[0],
                           'tests/resources/report/actual_trends.json') is True, "Json output for the report (file tests/resources/report/actual_trends.json) is not the one expected (file tests/resources/report/expected_trends.json)" + msg

    def __return_json(self, endpoint):
        if endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/8725e0cd-92a4-4105-803b-86433851fcfc':
            return '{"id": "8725e0cd-92a4-4105-803b-86433851fcfc", "name": "7.7.0RC2 (#26)", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1606141751490, "endDate": 1606753761965, "duration": 612010475, "testId": "e0e629ce-80a4-4ad2-b70a-0281488c270a"}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results':
            return '[{"id": "e54b3913-aa9d-409c-9126-0518012929e1", "name": "7.8.0RC3 (#30)", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1611680125889, "endDate": 1612292136289, "duration": 612010400, "testId": "e0e629ce-80a4-4ad2-b70a-0281488c270a"}, {"id": "f2b1a37f-f104-433a-ba47-f7d30675c504", "name": "Test_Endurance_7.8.0_rc3_NLGui", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1611679182524, "endDate": 1612291193156, "duration": 612010632}, {"id": "56b0e975-167d-4b55-8e81-9463e279a4fe", "name": "7.8.0RC1 (#29)", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1610986430453, "endDate": 1611598440650, "duration": 612010197, "testId": "e0e629ce-80a4-4ad2-b70a-0281488c270a"}, {"id": "b3d5daf0-d981-419f-962d-b8b6d3683eb5", "name": "Test_Endurance_7.8.0_rc1_NLGui", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1610968820857, "endDate": 1611580831425, "duration": 612010568}, {"id": "17e5c12b-a5cb-4df4-8fa6-d9cc94d2474a", "name": "Test_Endurance_7.7.0_rc2_NLGui", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1606141822564, "endDate": 1606753833210, "duration": 612010646}, {"id": "8725e0cd-92a4-4105-803b-86433851fcfc", "name": "7.7.0RC2 (#26)", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1606141751490, "endDate": 1606753761965, "duration": 612010475, "testId": "e0e629ce-80a4-4ad2-b70a-0281488c270a"}, {"id": "ba912bd5-d4ee-425c-ae82-b56a75c0e6ef", "name": "7.7.0RC1 (#25)", "description": "", "author": "Luke Skywalker", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1605280469001, "endDate": 1605892479406, "duration": 612010405, "testId": "e0e629ce-80a4-4ad2-b70a-0281488c270a"}, {"id": "71e88176-3353-4bcd-941a-284533099b43", "name": "Test_Endurance_7.7.0_rc1_NLGui", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1605280217712, "endDate": 1605892228403, "duration": 612010691}, {"id": "5d54299f-b4df-4a66-bec6-ee8751888eb1", "name": "7.6.0RC2 (#23)", "description": ".", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "PASSED", "startDate": 1600961628426, "endDate": 1601573638750, "duration": 612010324, "testId": "e0e629ce-80a4-4ad2-b70a-0281488c270a"}, {"id": "00756a20-2637-4dae-ab29-1aeb431471cf", "name": "Test_Endurance_7.6.0_rc2_NLGui", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1600785545919, "endDate": 1601397556395, "duration": 612010476}, {"id": "37115da4-7bda-4592-9c55-46605b07dda2", "name": "7.6.0RC2 (#22)", "description": "", "author": "Guillaume Bert", "terminationReason": "UNKNOWN", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "UNKNOWN", "startDate": 1600781199189, "endDate": 1601034996748, "duration": 253797559, "testId": "e0e629ce-80a4-4ad2-b70a-0281488c270a"}, {"id": "a4f45be3-8846-4fb9-83d2-813eae68a239", "name": "7.6.0RC1 (#21)", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1600264018560, "endDate": 1600876028882, "duration": 612010322, "testId": "e0e629ce-80a4-4ad2-b70a-0281488c270a"}, {"id": "40aa8991-be92-493f-920e-0c5d15df0686", "name": "7.5.0RC1 (#12)", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1594989440865, "endDate": 1595601451194, "duration": 612010329, "testId": "e0e629ce-80a4-4ad2-b70a-0281488c270a"}, {"id": "4c51a375-a8c8-4a47-8f51-8649c25451a0", "name": "Test_Endurance_7.5.0_rc1_NLGui", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1594989435711, "endDate": 1595601446327, "duration": 612010616}, {"id": "ad48efd0-efdf-473e-ad5e-9a352e223981", "name": "7.4.0RC1 (#11)", "description": "endurance", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1589559624354, "endDate": 1590171634707, "duration": 612010353, "testId": "e0e629ce-80a4-4ad2-b70a-0281488c270a"}, {"id": "afda72a9-4b67-4cbd-93fb-fe44d3594e62", "name": "Test_Endurance_7.4.0_rc1_NLGui", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1589559526722, "endDate": 1590171537233, "duration": 612010511}, {"id": "39256ce6-8676-45cc-a720-7dcba4ebcad2", "name": "Test_Endurance_7.3.0_rc2_NLGui", "description": "", "author": "Gregory Sanchez", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1584112025593, "endDate": 1584724036199, "duration": 612010606}, {"id": "113e8fc7-87ba-40b6-a90a-dc065cb3cc6c", "name": "7.3.0RC2 (#10)", "description": "", "author": "Guillaume Bert", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1584112004727, "endDate": 1584724015327, "duration": 612010600, "testId": "e0e629ce-80a4-4ad2-b70a-0281488c270a"}, {"id": "b6e48f1b-a28e-4b34-aada-5bbaa5e74055", "name": "7.3.0-RC1 (#9)", "description": "", "author": "Guillaume Bert", "terminationReason": "MANUAL", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1583848845359, "endDate": 1584111394779, "duration": 262549420, "testId": "e0e629ce-80a4-4ad2-b70a-0281488c270a"}, {"id": "787afa8f-6256-4e98-9e2a-bc711056b2af", "name": "Test_Endurance_7.3.0_rc1_NLGui", "description": "Endurance", "author": "Gregory Sanchez", "terminationReason": "LG_AVAILABILITY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "COMPUTING", "startDate": 1583848580136, "endDate": 1584111686073, "duration": 263105937}, {"id": "43874b4c-85dd-4cf5-b8e0-b01179582ec9", "name": "RC2 (#8)", "description": "", "author": "Benoit Reverbel", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1578927010369, "endDate": 1579539020803, "duration": 612010434, "testId": "e0e629ce-80a4-4ad2-b70a-0281488c270a"}, {"id": "f0ee85cf-3ba9-4a2c-ad24-90fb022a73ab", "name": "Test_Endurance_7.2.0_rc2_NLGui", "description": "", "author": "Gregory Sanchez", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario1", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1578910693621, "endDate": 1579522704104, "duration": 612010483}, {"id": "f5b61706-3df5-4b0a-a3de-7b375dd4931f", "name": "Test_Endurance_7.1.0_rc2_NLGui", "description": "", "author": "Gregory Sanchez", "terminationReason": "POLICY", "lgCount": 1, "project": "Endurance2", "scenario": "scenario endurance", "status": "TERMINATED", "qualityStatus": "FAILED", "startDate": 1573811263870, "endDate": 1574423274350, "duration": 612010480}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/71e88176-3353-4bcd-941a-284533099b43/statistics':
            return '{"totalRequestCountSuccess": 1851781, "totalRequestCountFailure": 2306, "totalRequestDurationAverage": 10.1926365, "totalRequestCountPerSecond": 3.029501, "totalTransactionCountSuccess": 174619, "totalTransactionCountFailure": 515, "totalTransactionDurationAverage": 88.658615, "totalTransactionCountPerSecond": 0.28616166, "totalIterationCountSuccess": 41935, "totalIterationCountFailure": 191, "totalGlobalDownloadedBytes": 2968827076, "totalGlobalDownloadedBytesPerSecond": 4850.94, "totalGlobalCountFailure": 2306}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/71e88176-3353-4bcd-941a-284533099b43/elements?category=TRANSACTION':
            return '[{"id": "all-transactions", "name": "<all transactions>", "type": "ALL_TRANSACTIONS"}, {"id": "b3e1f7c6-a4ef-438e-a3bb-8ffa1cb4a85e", "name": "Other", "path": ["ParcoursUtilisateur", "Actions", "Other"], "type": "TRANSACTION"}, {"id": "0ca05bf2-4ba9-483e-9041-c9ac38ecb8b8", "name": "See cats and reptiles", "path": ["ParcoursUtilisateur", "Actions", "See cats and reptiles"], "type": "TRANSACTION"}, {"id": "062b066f-e7c0-4e67-9a08-65d6494fecc9", "name": "Buy a fish", "path": ["ParcoursUtilisateur", "Actions", "Buy a fish"], "type": "TRANSACTION"}, {"id": "50ddd53f-4a4b-4bf8-a6b3-7a262d3d6cef", "name": "Home", "path": ["ParcoursUtilisateur", "Actions", "Home"], "type": "TRANSACTION"}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/ba912bd5-d4ee-425c-ae82-b56a75c0e6ef/statistics':
            return '{"totalRequestCountSuccess": 1796184, "totalRequestCountFailure": 2330, "totalRequestDurationAverage": 17.706757, "totalRequestCountPerSecond": 2.9386983, "totalTransactionCountSuccess": 169294, "totalTransactionCountFailure": 530, "totalTransactionDurationAverage": 154.04784, "totalTransactionCountPerSecond": 0.2774855, "totalIterationCountSuccess": 40611, "totalIterationCountFailure": 199, "totalGlobalDownloadedBytes": 2894978924, "totalGlobalDownloadedBytesPerSecond": 4730.2773, "totalGlobalCountFailure": 2330}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/71e88176-3353-4bcd-941a-284533099b43/elements/b3e1f7c6-a4ef-438e-a3bb-8ffa1cb4a85e/values':
            return '{"count": 43769, "elementPerSecond": 0.07151672, "minDuration": 42, "maxDuration": 57, "sumDuration": 1930564, "avgDuration": 44.10802, "minTTFB": 42, "maxTTFB": 57, "sumTTFB": 1930564, "avgTTFB": 44.10802, "sumDownloadedBytes": 0, "downloadedBytesPerSecond": 0.0, "successCount": 43769, "successPerSecond": 0.07151672, "successRate": 100.0, "failureCount": 0, "failurePerSecond": 0.0, "failureRate": 0.0, "percentile50": 44.0, "percentile90": 45.0, "percentile95": 46.0, "percentile99": 47.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/ba912bd5-d4ee-425c-ae82-b56a75c0e6ef/elements?category=TRANSACTION':
            return '[{"id": "all-transactions", "name": "<all transactions>", "type": "ALL_TRANSACTIONS"}, {"id": "b3e1f7c6-a4ef-438e-a3bb-8ffa1cb4a85e", "name": "Other", "path": ["ParcoursUtilisateur", "Actions", "Other"], "type": "TRANSACTION"}, {"id": "50ddd53f-4a4b-4bf8-a6b3-7a262d3d6cef", "name": "Home", "path": ["ParcoursUtilisateur", "Actions", "Home"], "type": "TRANSACTION"}, {"id": "062b066f-e7c0-4e67-9a08-65d6494fecc9", "name": "Buy a fish", "path": ["ParcoursUtilisateur", "Actions", "Buy a fish"], "type": "TRANSACTION"}, {"id": "0ca05bf2-4ba9-483e-9041-c9ac38ecb8b8", "name": "See cats and reptiles", "path": ["ParcoursUtilisateur", "Actions", "See cats and reptiles"], "type": "TRANSACTION"}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/71e88176-3353-4bcd-941a-284533099b43/elements/b3e1f7c6-a4ef-438e-a3bb-8ffa1cb4a85e/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 5120000, "AVG_TTFB": 44.96978, "ERRORS": 0, "AVG_DURATION": 44.96978, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 57, "MAX_DURATION": 57, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 5120000, "to": 10240000, "AVG_TTFB": 44.51366, "ERRORS": 0, "AVG_DURATION": 44.51366, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 57, "MAX_DURATION": 57, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 10240000, "to": 15360000, "AVG_TTFB": 44.248634, "ERRORS": 0, "AVG_DURATION": 44.248634, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 50, "MAX_DURATION": 50, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 15360000, "to": 20480000, "AVG_TTFB": 44.300545, "ERRORS": 0, "AVG_DURATION": 44.300545, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 20480000, "to": 25600000, "AVG_TTFB": 44.26703, "ERRORS": 0, "AVG_DURATION": 44.26703, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 52, "MAX_DURATION": 52, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 25600000, "to": 30720000, "AVG_TTFB": 44.275955, "ERRORS": 0, "AVG_DURATION": 44.275955, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 30720000, "to": 35840000, "AVG_TTFB": 44.153004, "ERRORS": 0, "AVG_DURATION": 44.153004, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 35840000, "to": 40960000, "AVG_TTFB": 44.17213, "ERRORS": 0, "AVG_DURATION": 44.17213, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 40960000, "to": 46080000, "AVG_TTFB": 44.082417, "ERRORS": 0, "AVG_DURATION": 44.082417, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 46080000, "to": 51200000, "AVG_TTFB": 44.222828, "ERRORS": 0, "AVG_DURATION": 44.222828, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 50, "MAX_DURATION": 50, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 51200000, "to": 56320000, "AVG_TTFB": 44.06011, "ERRORS": 0, "AVG_DURATION": 44.06011, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 56320000, "to": 61440000, "AVG_TTFB": 44.532787, "ERRORS": 0, "AVG_DURATION": 44.532787, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 52, "MAX_DURATION": 52, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 61440000, "to": 66560000, "AVG_TTFB": 44.285713, "ERRORS": 0, "AVG_DURATION": 44.285713, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 51, "MAX_DURATION": 51, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 66560000, "to": 71680000, "AVG_TTFB": 44.144848, "ERRORS": 0, "AVG_DURATION": 44.144848, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07011719, "THROUGHPUT": 0.0, "COUNT": 359}, {"from": 71680000, "to": 76800000, "AVG_TTFB": 44.153423, "ERRORS": 0, "AVG_DURATION": 44.153423, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 53, "MAX_DURATION": 53, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 76800000, "to": 81920000, "AVG_TTFB": 44.204918, "ERRORS": 0, "AVG_DURATION": 44.204918, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 53, "MAX_DURATION": 53, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 81920000, "to": 87040000, "AVG_TTFB": 43.959015, "ERRORS": 0, "AVG_DURATION": 43.959015, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 87040000, "to": 92160000, "AVG_TTFB": 44.121952, "ERRORS": 0, "AVG_DURATION": 44.121952, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 50, "MAX_DURATION": 50, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 0.0, "COUNT": 369}, {"from": 92160000, "to": 97280000, "AVG_TTFB": 44.071037, "ERRORS": 0, "AVG_DURATION": 44.071037, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 51, "MAX_DURATION": 51, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 97280000, "to": 102400000, "AVG_TTFB": 44.15531, "ERRORS": 0, "AVG_DURATION": 44.15531, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 51, "MAX_DURATION": 51, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 102400000, "to": 107520000, "AVG_TTFB": 44.01635, "ERRORS": 0, "AVG_DURATION": 44.01635, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 54, "MAX_DURATION": 54, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 107520000, "to": 112640000, "AVG_TTFB": 43.920765, "ERRORS": 0, "AVG_DURATION": 43.920765, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 112640000, "to": 117760000, "AVG_TTFB": 44.119892, "ERRORS": 0, "AVG_DURATION": 44.119892, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 50, "MAX_DURATION": 50, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 117760000, "to": 122880000, "AVG_TTFB": 44.092896, "ERRORS": 0, "AVG_DURATION": 44.092896, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 122880000, "to": 128000000, "AVG_TTFB": 44.10959, "ERRORS": 0, "AVG_DURATION": 44.10959, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 128000000, "to": 133120000, "AVG_TTFB": 44.08011, "ERRORS": 0, "AVG_DURATION": 44.08011, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 53, "MAX_DURATION": 53, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 0.0, "COUNT": 362}, {"from": 133120000, "to": 138240000, "AVG_TTFB": 44.119564, "ERRORS": 0, "AVG_DURATION": 44.119564, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 138240000, "to": 143360000, "AVG_TTFB": 43.947945, "ERRORS": 0, "AVG_DURATION": 43.947945, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 143360000, "to": 148480000, "AVG_TTFB": 44.0436, "ERRORS": 0, "AVG_DURATION": 44.0436, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 148480000, "to": 153600000, "AVG_TTFB": 44.02459, "ERRORS": 0, "AVG_DURATION": 44.02459, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 153600000, "to": 158720000, "AVG_TTFB": 44.032608, "ERRORS": 0, "AVG_DURATION": 44.032608, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 158720000, "to": 163840000, "AVG_TTFB": 44.03825, "ERRORS": 0, "AVG_DURATION": 44.03825, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 163840000, "to": 168960000, "AVG_TTFB": 44.06011, "ERRORS": 0, "AVG_DURATION": 44.06011, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 168960000, "to": 174080000, "AVG_TTFB": 44.092644, "ERRORS": 0, "AVG_DURATION": 44.092644, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 52, "MAX_DURATION": 52, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 174080000, "to": 179200000, "AVG_TTFB": 43.942467, "ERRORS": 0, "AVG_DURATION": 43.942467, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 50, "MAX_DURATION": 50, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 179200000, "to": 184320000, "AVG_TTFB": 44.10082, "ERRORS": 0, "AVG_DURATION": 44.10082, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 184320000, "to": 189440000, "AVG_TTFB": 44.024525, "ERRORS": 0, "AVG_DURATION": 44.024525, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 189440000, "to": 194560000, "AVG_TTFB": 44.057377, "ERRORS": 0, "AVG_DURATION": 44.057377, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 194560000, "to": 199680000, "AVG_TTFB": 44.12295, "ERRORS": 0, "AVG_DURATION": 44.12295, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 199680000, "to": 204800000, "AVG_TTFB": 43.99178, "ERRORS": 0, "AVG_DURATION": 43.99178, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 204800000, "to": 209920000, "AVG_TTFB": 44.086956, "ERRORS": 0, "AVG_DURATION": 44.086956, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 52, "MAX_DURATION": 52, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 209920000, "to": 215040000, "AVG_TTFB": 44.046448, "ERRORS": 0, "AVG_DURATION": 44.046448, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 215040000, "to": 220160000, "AVG_TTFB": 44.0436, "ERRORS": 0, "AVG_DURATION": 44.0436, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 52, "MAX_DURATION": 52, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 220160000, "to": 225280000, "AVG_TTFB": 44.23546, "ERRORS": 0, "AVG_DURATION": 44.23546, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 54, "MAX_DURATION": 54, "ELEMENTS_PER_SECOND": 0.07050781, "THROUGHPUT": 0.0, "COUNT": 361}, {"from": 225280000, "to": 230400000, "AVG_TTFB": 43.975475, "ERRORS": 0, "AVG_DURATION": 43.975475, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 230400000, "to": 235520000, "AVG_TTFB": 44.0109, "ERRORS": 0, "AVG_DURATION": 44.0109, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 235520000, "to": 240640000, "AVG_TTFB": 44.05177, "ERRORS": 0, "AVG_DURATION": 44.05177, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 240640000, "to": 245760000, "AVG_TTFB": 44.12055, "ERRORS": 0, "AVG_DURATION": 44.12055, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 245760000, "to": 250880000, "AVG_TTFB": 44.027397, "ERRORS": 0, "AVG_DURATION": 44.027397, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 250880000, "to": 256000000, "AVG_TTFB": 44.067936, "ERRORS": 0, "AVG_DURATION": 44.067936, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 256000000, "to": 261120000, "AVG_TTFB": 44.128414, "ERRORS": 0, "AVG_DURATION": 44.128414, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 261120000, "to": 266240000, "AVG_TTFB": 44.17213, "ERRORS": 0, "AVG_DURATION": 44.17213, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 51, "MAX_DURATION": 51, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 266240000, "to": 271360000, "AVG_TTFB": 43.934425, "ERRORS": 0, "AVG_DURATION": 43.934425, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 271360000, "to": 276480000, "AVG_TTFB": 43.997284, "ERRORS": 0, "AVG_DURATION": 43.997284, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 50, "MAX_DURATION": 50, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 276480000, "to": 281600000, "AVG_TTFB": 44.14208, "ERRORS": 0, "AVG_DURATION": 44.14208, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 50, "MAX_DURATION": 50, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 281600000, "to": 286720000, "AVG_TTFB": 44.183563, "ERRORS": 0, "AVG_DURATION": 44.183563, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 286720000, "to": 291840000, "AVG_TTFB": 44.14674, "ERRORS": 0, "AVG_DURATION": 44.14674, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 291840000, "to": 296960000, "AVG_TTFB": 44.057377, "ERRORS": 0, "AVG_DURATION": 44.057377, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 53, "MAX_DURATION": 53, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 296960000, "to": 302080000, "AVG_TTFB": 43.989132, "ERRORS": 0, "AVG_DURATION": 43.989132, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 302080000, "to": 307200000, "AVG_TTFB": 44.104397, "ERRORS": 0, "AVG_DURATION": 44.104397, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 307200000, "to": 312320000, "AVG_TTFB": 44.174862, "ERRORS": 0, "AVG_DURATION": 44.174862, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 312320000, "to": 317440000, "AVG_TTFB": 44.035618, "ERRORS": 0, "AVG_DURATION": 44.035618, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 317440000, "to": 322560000, "AVG_TTFB": 44.127716, "ERRORS": 0, "AVG_DURATION": 44.127716, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 322560000, "to": 327680000, "AVG_TTFB": 44.035423, "ERRORS": 0, "AVG_DURATION": 44.035423, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 327680000, "to": 332800000, "AVG_TTFB": 44.07989, "ERRORS": 0, "AVG_DURATION": 44.07989, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 0.0, "COUNT": 363}, {"from": 332800000, "to": 337920000, "AVG_TTFB": 44.117165, "ERRORS": 0, "AVG_DURATION": 44.117165, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 337920000, "to": 343040000, "AVG_TTFB": 44.116848, "ERRORS": 0, "AVG_DURATION": 44.116848, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 343040000, "to": 348160000, "AVG_TTFB": 44.133514, "ERRORS": 0, "AVG_DURATION": 44.133514, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 54, "MAX_DURATION": 54, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 348160000, "to": 353280000, "AVG_TTFB": 44.098362, "ERRORS": 0, "AVG_DURATION": 44.098362, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 353280000, "to": 358400000, "AVG_TTFB": 44.138966, "ERRORS": 0, "AVG_DURATION": 44.138966, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 358400000, "to": 363520000, "AVG_TTFB": 44.079235, "ERRORS": 0, "AVG_DURATION": 44.079235, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 363520000, "to": 368640000, "AVG_TTFB": 44.122616, "ERRORS": 0, "AVG_DURATION": 44.122616, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 52, "MAX_DURATION": 52, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 368640000, "to": 373760000, "AVG_TTFB": 44.02466, "ERRORS": 0, "AVG_DURATION": 44.02466, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 373760000, "to": 378880000, "AVG_TTFB": 44.095627, "ERRORS": 0, "AVG_DURATION": 44.095627, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 55, "MAX_DURATION": 55, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 378880000, "to": 384000000, "AVG_TTFB": 44.10326, "ERRORS": 0, "AVG_DURATION": 44.10326, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 51, "MAX_DURATION": 51, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 384000000, "to": 389120000, "AVG_TTFB": 44.019127, "ERRORS": 0, "AVG_DURATION": 44.019127, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 389120000, "to": 394240000, "AVG_TTFB": 44.052055, "ERRORS": 0, "AVG_DURATION": 44.052055, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 394240000, "to": 399360000, "AVG_TTFB": 44.157024, "ERRORS": 0, "AVG_DURATION": 44.157024, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 0.0, "COUNT": 363}, {"from": 399360000, "to": 404480000, "AVG_TTFB": 44.10929, "ERRORS": 0, "AVG_DURATION": 44.10929, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 50, "MAX_DURATION": 50, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 404480000, "to": 409600000, "AVG_TTFB": 44.05722, "ERRORS": 0, "AVG_DURATION": 44.05722, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 409600000, "to": 414720000, "AVG_TTFB": 44.094852, "ERRORS": 0, "AVG_DURATION": 44.094852, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 0.0, "COUNT": 369}, {"from": 414720000, "to": 419840000, "AVG_TTFB": 44.13115, "ERRORS": 0, "AVG_DURATION": 44.13115, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 51, "MAX_DURATION": 51, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 419840000, "to": 424960000, "AVG_TTFB": 44.04121, "ERRORS": 0, "AVG_DURATION": 44.04121, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 424960000, "to": 430080000, "AVG_TTFB": 44.115383, "ERRORS": 0, "AVG_DURATION": 44.115383, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 430080000, "to": 435200000, "AVG_TTFB": 44.08152, "ERRORS": 0, "AVG_DURATION": 44.08152, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 55, "MAX_DURATION": 55, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 435200000, "to": 440320000, "AVG_TTFB": 44.02725, "ERRORS": 0, "AVG_DURATION": 44.02725, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 440320000, "to": 445440000, "AVG_TTFB": 44.14754, "ERRORS": 0, "AVG_DURATION": 44.14754, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 50, "MAX_DURATION": 50, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 445440000, "to": 450560000, "AVG_TTFB": 44.084698, "ERRORS": 0, "AVG_DURATION": 44.084698, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 450560000, "to": 455680000, "AVG_TTFB": 44.07377, "ERRORS": 0, "AVG_DURATION": 44.07377, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 455680000, "to": 460800000, "AVG_TTFB": 44.173912, "ERRORS": 0, "AVG_DURATION": 44.173912, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 50, "MAX_DURATION": 50, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 460800000, "to": 465920000, "AVG_TTFB": 44.211384, "ERRORS": 0, "AVG_DURATION": 44.211384, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 0.0, "COUNT": 369}, {"from": 465920000, "to": 471040000, "AVG_TTFB": 44.10137, "ERRORS": 0, "AVG_DURATION": 44.10137, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 471040000, "to": 476160000, "AVG_TTFB": 44.185287, "ERRORS": 0, "AVG_DURATION": 44.185287, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 476160000, "to": 481280000, "AVG_TTFB": 44.17166, "ERRORS": 0, "AVG_DURATION": 44.17166, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 481280000, "to": 486400000, "AVG_TTFB": 44.067936, "ERRORS": 0, "AVG_DURATION": 44.067936, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 52, "MAX_DURATION": 52, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 486400000, "to": 491520000, "AVG_TTFB": 44.106556, "ERRORS": 0, "AVG_DURATION": 44.106556, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 491520000, "to": 496640000, "AVG_TTFB": 43.97808, "ERRORS": 0, "AVG_DURATION": 43.97808, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 496640000, "to": 501760000, "AVG_TTFB": 44.18033, "ERRORS": 0, "AVG_DURATION": 44.18033, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 52, "MAX_DURATION": 52, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 501760000, "to": 506880000, "AVG_TTFB": 44.032787, "ERRORS": 0, "AVG_DURATION": 44.032787, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 506880000, "to": 512000000, "AVG_TTFB": 44.14754, "ERRORS": 0, "AVG_DURATION": 44.14754, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 512000000, "to": 517120000, "AVG_TTFB": 44.14481, "ERRORS": 0, "AVG_DURATION": 44.14481, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 517120000, "to": 522240000, "AVG_TTFB": 44.005478, "ERRORS": 0, "AVG_DURATION": 44.005478, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 522240000, "to": 527360000, "AVG_TTFB": 44.065395, "ERRORS": 0, "AVG_DURATION": 44.065395, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 527360000, "to": 532480000, "AVG_TTFB": 44.160763, "ERRORS": 0, "AVG_DURATION": 44.160763, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 532480000, "to": 537600000, "AVG_TTFB": 44.11444, "ERRORS": 0, "AVG_DURATION": 44.11444, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 537600000, "to": 542720000, "AVG_TTFB": 44.04918, "ERRORS": 0, "AVG_DURATION": 44.04918, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 542720000, "to": 547840000, "AVG_TTFB": 44.00822, "ERRORS": 0, "AVG_DURATION": 44.00822, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 547840000, "to": 552960000, "AVG_TTFB": 44.092644, "ERRORS": 0, "AVG_DURATION": 44.092644, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 552960000, "to": 558080000, "AVG_TTFB": 44.13624, "ERRORS": 0, "AVG_DURATION": 44.13624, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 558080000, "to": 563200000, "AVG_TTFB": 44.142857, "ERRORS": 0, "AVG_DURATION": 44.142857, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 563200000, "to": 568320000, "AVG_TTFB": 43.98634, "ERRORS": 0, "AVG_DURATION": 43.98634, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 568320000, "to": 573440000, "AVG_TTFB": 44.136612, "ERRORS": 0, "AVG_DURATION": 44.136612, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 573440000, "to": 578560000, "AVG_TTFB": 44.03794, "ERRORS": 0, "AVG_DURATION": 44.03794, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 0.0, "COUNT": 369}, {"from": 578560000, "to": 583680000, "AVG_TTFB": 44.065754, "ERRORS": 0, "AVG_DURATION": 44.065754, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 583680000, "to": 588800000, "AVG_TTFB": 44.14481, "ERRORS": 0, "AVG_DURATION": 44.14481, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 50, "MAX_DURATION": 50, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 588800000, "to": 593920000, "AVG_TTFB": 44.095627, "ERRORS": 0, "AVG_DURATION": 44.095627, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 593920000, "to": 599040000, "AVG_TTFB": 44.019073, "ERRORS": 0, "AVG_DURATION": 44.019073, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 599040000, "to": 604160000, "AVG_TTFB": 44.138588, "ERRORS": 0, "AVG_DURATION": 44.138588, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 604160000, "to": 609280000, "AVG_TTFB": 44.06812, "ERRORS": 0, "AVG_DURATION": 44.06812, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 609280000, "to": 614400000, "AVG_TTFB": 43.847717, "ERRORS": 0, "AVG_DURATION": 43.847717, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.038476564, "THROUGHPUT": 0.0, "COUNT": 197}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/71e88176-3353-4bcd-941a-284533099b43/elements/all-transactions/values':
            return '{"count": 175134, "elementPerSecond": 0.28616166, "minDuration": 4, "maxDuration": 39397, "sumDuration": 15527138, "avgDuration": 88.658615, "minTTFB": 0, "maxTTFB": 38709, "sumTTFB": 3670245, "avgTTFB": 20.956781, "sumDownloadedBytes": 2959355021, "downloadedBytesPerSecond": 4835.463, "successCount": 174619, "successPerSecond": 0.2853202, "successRate": 99.70594, "failureCount": 515, "failurePerSecond": 0.0008414886, "failureRate": 0.29406056, "percentile50": 28.0, "percentile90": 106.0, "percentile95": 147.0, "percentile99": 1255.5}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/ba912bd5-d4ee-425c-ae82-b56a75c0e6ef/elements/all-transactions/values':
            return '{"count": 169824, "elementPerSecond": 0.2774855, "minDuration": 4, "maxDuration": 349267, "sumDuration": 26161019, "avgDuration": 154.04784, "minTTFB": 1, "maxTTFB": 29896, "sumTTFB": 3681557, "avgTTFB": 21.678661, "sumDownloadedBytes": 2885290642, "downloadedBytesPerSecond": 4714.4473, "successCount": 169294, "successPerSecond": 0.2766195, "successRate": 99.68791, "failureCount": 530, "failurePerSecond": 0.0008659983, "failureRate": 0.3120878, "percentile50": 45.0, "percentile90": 46.0, "percentile95": 46.0, "percentile99": 53.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/71e88176-3353-4bcd-941a-284533099b43/elements/50ddd53f-4a4b-4bf8-a6b3-7a262d3d6cef/values':
            return '{"count": 43807, "elementPerSecond": 0.071578816, "minDuration": 4, "maxDuration": 32618, "sumDuration": 2459573, "avgDuration": 56.14566, "minTTFB": 1, "maxTTFB": 21955, "sumTTFB": 528533, "avgTTFB": 12.065035, "sumDownloadedBytes": 872920326, "downloadedBytesPerSecond": 1426.3154, "successCount": 43643, "successPerSecond": 0.07131085, "successRate": 99.62563, "failureCount": 164, "failurePerSecond": 0.00026796918, "failureRate": 0.37436938, "percentile50": 20.0, "percentile90": 77.0, "percentile95": 79.0, "percentile99": 436.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/ba912bd5-d4ee-425c-ae82-b56a75c0e6ef/elements/all-transactions/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 5120000, "AVG_TTFB": 20.147743, "ERRORS": 0, "AVG_DURATION": 81.55746, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1288, "MAX_DURATION": 3506, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 5075.008, "COUNT": 1462}, {"from": 5120000, "to": 10240000, "AVG_TTFB": 20.73634, "ERRORS": 0, "AVG_DURATION": 87.51776, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1267, "MAX_DURATION": 2301, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4831.919, "COUNT": 1464}, {"from": 10240000, "to": 15360000, "AVG_TTFB": 14.272355, "ERRORS": 0, "AVG_DURATION": 76.275085, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 330, "MAX_DURATION": 8294, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4832.119, "COUNT": 1465}, {"from": 15360000, "to": 20480000, "AVG_TTFB": 17.74455, "ERRORS": 0, "AVG_DURATION": 99.60286, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1625, "MAX_DURATION": 9102, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4910.1494, "COUNT": 1468}, {"from": 20480000, "to": 25600000, "AVG_TTFB": 34.788357, "ERRORS": 0, "AVG_DURATION": 131.637, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 6270, "MAX_DURATION": 10825, "ELEMENTS_PER_SECOND": 0.28515625, "THROUGHPUT": 4825.0547, "COUNT": 1460}, {"from": 25600000, "to": 30720000, "AVG_TTFB": 22.922762, "ERRORS": 0, "AVG_DURATION": 100.287766, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4824, "MAX_DURATION": 5919, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4829.8853, "COUNT": 1463}, {"from": 30720000, "to": 35840000, "AVG_TTFB": 27.193989, "ERRORS": 0, "AVG_DURATION": 106.946724, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 7021, "MAX_DURATION": 11902, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4835.695, "COUNT": 1464}, {"from": 35840000, "to": 40960000, "AVG_TTFB": 15.12731, "ERRORS": 0, "AVG_DURATION": 74.502396, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1377, "MAX_DURATION": 2882, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4888.1724, "COUNT": 1461}, {"from": 40960000, "to": 46080000, "AVG_TTFB": 24.919563, "ERRORS": 0, "AVG_DURATION": 126.12952, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2430, "MAX_DURATION": 9202, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4838.2227, "COUNT": 1467}, {"from": 46080000, "to": 51200000, "AVG_TTFB": 22.598495, "ERRORS": 0, "AVG_DURATION": 97.766075, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5259, "MAX_DURATION": 5473, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4821.7534, "COUNT": 1462}, {"from": 51200000, "to": 56320000, "AVG_TTFB": 14.701158, "ERRORS": 13, "AVG_DURATION": 60.709328, "ERRORS_PER_SECOND": 0.0025390624, "MIN_DURATION": 4, "ERROR_RATE": 0.88495576, "MIN_TTFB": 2, "MAX_TTFB": 267, "MAX_DURATION": 1350, "ELEMENTS_PER_SECOND": 0.28691405, "THROUGHPUT": 4804.0405, "COUNT": 1469}, {"from": 56320000, "to": 61440000, "AVG_TTFB": 15.344498, "ERRORS": 0, "AVG_DURATION": 60.91866, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1822, "MAX_DURATION": 2276, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4829.385, "COUNT": 1463}, {"from": 61440000, "to": 66560000, "AVG_TTFB": 31.873114, "ERRORS": 0, "AVG_DURATION": 222.2572, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3782, "MAX_DURATION": 24726, "ELEMENTS_PER_SECOND": 0.28476563, "THROUGHPUT": 4880.78, "COUNT": 1458}, {"from": 66560000, "to": 71680000, "AVG_TTFB": 104.30189, "ERRORS": 74, "AVG_DURATION": 489.8232, "ERRORS_PER_SECOND": 0.014453125, "MIN_DURATION": 4, "ERROR_RATE": 5.171209, "MIN_TTFB": 2, "MAX_TTFB": 29896, "MAX_DURATION": 36404, "ELEMENTS_PER_SECOND": 0.2794922, "THROUGHPUT": 4521.906, "COUNT": 1431}, {"from": 71680000, "to": 76800000, "AVG_TTFB": 27.88175, "ERRORS": 0, "AVG_DURATION": 117.18934, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2439, "MAX_DURATION": 3442, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4824.3633, "COUNT": 1463}, {"from": 76800000, "to": 81920000, "AVG_TTFB": 26.537304, "ERRORS": 0, "AVG_DURATION": 95.72964, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2257, "MAX_DURATION": 3651, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4821.685, "COUNT": 1461}, {"from": 81920000, "to": 87040000, "AVG_TTFB": 15.7234335, "ERRORS": 0, "AVG_DURATION": 70.95368, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1370, "MAX_DURATION": 2318, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4906.827, "COUNT": 1468}, {"from": 87040000, "to": 92160000, "AVG_TTFB": 18.03603, "ERRORS": 0, "AVG_DURATION": 81.98844, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1557, "MAX_DURATION": 4518, "ELEMENTS_PER_SECOND": 0.2873047, "THROUGHPUT": 4848.617, "COUNT": 1471}, {"from": 92160000, "to": 97280000, "AVG_TTFB": 19.962988, "ERRORS": 0, "AVG_DURATION": 97.09047, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1769, "MAX_DURATION": 4406, "ELEMENTS_PER_SECOND": 0.28496093, "THROUGHPUT": 4821.9185, "COUNT": 1459}, {"from": 97280000, "to": 102400000, "AVG_TTFB": 16.2944, "ERRORS": 0, "AVG_DURATION": 89.97063, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2337, "MAX_DURATION": 10871, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4829.215, "COUNT": 1464}, {"from": 102400000, "to": 107520000, "AVG_TTFB": 16.027267, "ERRORS": 0, "AVG_DURATION": 69.254944, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 795, "MAX_DURATION": 4599, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4834.0747, "COUNT": 1467}, {"from": 107520000, "to": 112640000, "AVG_TTFB": 21.629251, "ERRORS": 0, "AVG_DURATION": 69.80272, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4324, "MAX_DURATION": 4536, "ELEMENTS_PER_SECOND": 0.28710938, "THROUGHPUT": 4914.7256, "COUNT": 1470}, {"from": 112640000, "to": 117760000, "AVG_TTFB": 16.707085, "ERRORS": 0, "AVG_DURATION": 61.824932, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3223, "MAX_DURATION": 3292, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4836.872, "COUNT": 1468}, {"from": 117760000, "to": 122880000, "AVG_TTFB": 14.581522, "ERRORS": 0, "AVG_DURATION": 57.617527, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 979, "MAX_DURATION": 1043, "ELEMENTS_PER_SECOND": 0.2875, "THROUGHPUT": 4845.6445, "COUNT": 1472}, {"from": 122880000, "to": 128000000, "AVG_TTFB": 15.784689, "ERRORS": 0, "AVG_DURATION": 66.895424, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1527, "MAX_DURATION": 1954, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4833.0977, "COUNT": 1463}, {"from": 128000000, "to": 133120000, "AVG_TTFB": 51.940205, "ERRORS": 0, "AVG_DURATION": 252.19794, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5494, "MAX_DURATION": 17302, "ELEMENTS_PER_SECOND": 0.2841797, "THROUGHPUT": 4871.878, "COUNT": 1455}, {"from": 133120000, "to": 138240000, "AVG_TTFB": 22.182438, "ERRORS": 0, "AVG_DURATION": 77.78352, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2503, "MAX_DURATION": 2858, "ELEMENTS_PER_SECOND": 0.28691405, "THROUGHPUT": 4841.341, "COUNT": 1469}, {"from": 138240000, "to": 143360000, "AVG_TTFB": 20.733927, "ERRORS": 29, "AVG_DURATION": 99.83858, "ERRORS_PER_SECOND": 0.0056640627, "MIN_DURATION": 5, "ERROR_RATE": 1.9835842, "MIN_TTFB": 2, "MAX_TTFB": 2788, "MAX_DURATION": 12945, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4745.198, "COUNT": 1462}, {"from": 143360000, "to": 148480000, "AVG_TTFB": 17.100817, "ERRORS": 0, "AVG_DURATION": 67.12739, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1470, "MAX_DURATION": 2980, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4835.953, "COUNT": 1468}, {"from": 148480000, "to": 153600000, "AVG_TTFB": 15.350649, "ERRORS": 18, "AVG_DURATION": 63.004784, "ERRORS_PER_SECOND": 0.003515625, "MIN_DURATION": 4, "ERROR_RATE": 1.2303486, "MIN_TTFB": 1, "MAX_TTFB": 1008, "MAX_DURATION": 3949, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4785.994, "COUNT": 1463}, {"from": 153600000, "to": 158720000, "AVG_TTFB": 21.77217, "ERRORS": 0, "AVG_DURATION": 80.99932, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4513, "MAX_DURATION": 6292, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4902.6543, "COUNT": 1466}, {"from": 158720000, "to": 163840000, "AVG_TTFB": 17.455412, "ERRORS": 0, "AVG_DURATION": 86.58611, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1064, "MAX_DURATION": 3206, "ELEMENTS_PER_SECOND": 0.28691405, "THROUGHPUT": 4835.805, "COUNT": 1469}, {"from": 163840000, "to": 168960000, "AVG_TTFB": 14.427105, "ERRORS": 0, "AVG_DURATION": 61.238876, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 350, "MAX_DURATION": 1217, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4833.3335, "COUNT": 1461}, {"from": 168960000, "to": 174080000, "AVG_TTFB": 14.650307, "ERRORS": 0, "AVG_DURATION": 63.65099, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 837, "MAX_DURATION": 2227, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4838.5176, "COUNT": 1467}, {"from": 174080000, "to": 179200000, "AVG_TTFB": 15.812798, "ERRORS": 0, "AVG_DURATION": 73.40504, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2564, "MAX_DURATION": 3575, "ELEMENTS_PER_SECOND": 0.28691405, "THROUGHPUT": 4896.8467, "COUNT": 1469}, {"from": 179200000, "to": 184320000, "AVG_TTFB": 28.11058, "ERRORS": 0, "AVG_DURATION": 111.519455, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 6870, "MAX_DURATION": 7801, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4837.0425, "COUNT": 1465}, {"from": 184320000, "to": 189440000, "AVG_TTFB": 14.169619, "ERRORS": 0, "AVG_DURATION": 64.55245, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 601, "MAX_DURATION": 4673, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4843.0273, "COUNT": 1468}, {"from": 189440000, "to": 194560000, "AVG_TTFB": 13.917462, "ERRORS": 0, "AVG_DURATION": 56.738064, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 123, "MAX_DURATION": 578, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4832.885, "COUNT": 1466}, {"from": 194560000, "to": 199680000, "AVG_TTFB": 15.389078, "ERRORS": 0, "AVG_DURATION": 66.917404, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1104, "MAX_DURATION": 4870, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4833.2275, "COUNT": 1465}, {"from": 199680000, "to": 204800000, "AVG_TTFB": 27.37517, "ERRORS": 0, "AVG_DURATION": 138.87108, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4928, "MAX_DURATION": 7582, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4887.3154, "COUNT": 1466}, {"from": 204800000, "to": 209920000, "AVG_TTFB": 19.216087, "ERRORS": 0, "AVG_DURATION": 86.12884, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3658, "MAX_DURATION": 3870, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4846.009, "COUNT": 1467}, {"from": 209920000, "to": 215040000, "AVG_TTFB": 13.937968, "ERRORS": 0, "AVG_DURATION": 55.41854, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 157, "MAX_DURATION": 259, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4829.4727, "COUNT": 1467}, {"from": 215040000, "to": 220160000, "AVG_TTFB": 17.785276, "ERRORS": 0, "AVG_DURATION": 81.41513, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 514, "MAX_DURATION": 1922, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4837.0903, "COUNT": 1467}, {"from": 220160000, "to": 225280000, "AVG_TTFB": 34.744156, "ERRORS": 0, "AVG_DURATION": 204.38995, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 13329, "MAX_DURATION": 41637, "ELEMENTS_PER_SECOND": 0.28398436, "THROUGHPUT": 4858.8516, "COUNT": 1454}, {"from": 225280000, "to": 230400000, "AVG_TTFB": 29.157787, "ERRORS": 76, "AVG_DURATION": 153.38661, "ERRORS_PER_SECOND": 0.01484375, "MIN_DURATION": 4, "ERROR_RATE": 5.191257, "MIN_TTFB": 2, "MAX_TTFB": 3390, "MAX_DURATION": 7307, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4613.5327, "COUNT": 1464}, {"from": 230400000, "to": 235520000, "AVG_TTFB": 14.544959, "ERRORS": 0, "AVG_DURATION": 60.44278, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 469, "MAX_DURATION": 1552, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4837.676, "COUNT": 1468}, {"from": 235520000, "to": 240640000, "AVG_TTFB": 15.206967, "ERRORS": 14, "AVG_DURATION": 60.4556, "ERRORS_PER_SECOND": 0.002734375, "MIN_DURATION": 6, "ERROR_RATE": 0.95628417, "MIN_TTFB": 1, "MAX_TTFB": 199, "MAX_DURATION": 816, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4789.479, "COUNT": 1464}, {"from": 240640000, "to": 245760000, "AVG_TTFB": 34.83769, "ERRORS": 0, "AVG_DURATION": 147.90578, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6218, "MAX_DURATION": 14355, "ELEMENTS_PER_SECOND": 0.28398436, "THROUGHPUT": 4815.904, "COUNT": 1454}, {"from": 245760000, "to": 250880000, "AVG_TTFB": 17.362644, "ERRORS": 0, "AVG_DURATION": 80.12543, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1697, "MAX_DURATION": 7160, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4901.9565, "COUNT": 1467}, {"from": 250880000, "to": 256000000, "AVG_TTFB": 15.455662, "ERRORS": 0, "AVG_DURATION": 64.53888, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1083, "MAX_DURATION": 2640, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4840.051, "COUNT": 1466}, {"from": 256000000, "to": 261120000, "AVG_TTFB": 15.144126, "ERRORS": 0, "AVG_DURATION": 72.93579, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 716, "MAX_DURATION": 3680, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4829.773, "COUNT": 1464}, {"from": 261120000, "to": 266240000, "AVG_TTFB": 17.867577, "ERRORS": 0, "AVG_DURATION": 71.37202, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1726, "MAX_DURATION": 2865, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4835.563, "COUNT": 1465}, {"from": 266240000, "to": 271360000, "AVG_TTFB": 16.103212, "ERRORS": 0, "AVG_DURATION": 61.8797, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1398, "MAX_DURATION": 1423, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4885.0977, "COUNT": 1463}, {"from": 271360000, "to": 276480000, "AVG_TTFB": 23.881275, "ERRORS": 0, "AVG_DURATION": 103.11533, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4975, "MAX_DURATION": 11914, "ELEMENTS_PER_SECOND": 0.2878906, "THROUGHPUT": 4858.3647, "COUNT": 1474}, {"from": 276480000, "to": 281600000, "AVG_TTFB": 27.172626, "ERRORS": 0, "AVG_DURATION": 198.12448, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4161, "MAX_DURATION": 21375, "ELEMENTS_PER_SECOND": 0.28398436, "THROUGHPUT": 4804.776, "COUNT": 1454}, {"from": 281600000, "to": 286720000, "AVG_TTFB": 17.089542, "ERRORS": 0, "AVG_DURATION": 92.55913, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2041, "MAX_DURATION": 15596, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4827.3574, "COUNT": 1463}, {"from": 286720000, "to": 291840000, "AVG_TTFB": 16.250511, "ERRORS": 0, "AVG_DURATION": 66.21035, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2981, "MAX_DURATION": 3806, "ELEMENTS_PER_SECOND": 0.28691405, "THROUGHPUT": 4840.0312, "COUNT": 1469}, {"from": 291840000, "to": 296960000, "AVG_TTFB": 15.578839, "ERRORS": 0, "AVG_DURATION": 169.32286, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1107, "MAX_DURATION": 26922, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4894.8086, "COUNT": 1465}, {"from": 296960000, "to": 302080000, "AVG_TTFB": 14.406675, "ERRORS": 0, "AVG_DURATION": 63.93188, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 381, "MAX_DURATION": 2565, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4843.588, "COUNT": 1468}, {"from": 302080000, "to": 307200000, "AVG_TTFB": 22.676933, "ERRORS": 0, "AVG_DURATION": 118.098564, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2919, "MAX_DURATION": 9405, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4822.74, "COUNT": 1461}, {"from": 307200000, "to": 312320000, "AVG_TTFB": 15.848049, "ERRORS": 0, "AVG_DURATION": 76.1759, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 871, "MAX_DURATION": 4294, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4818.3843, "COUNT": 1461}, {"from": 312320000, "to": 317440000, "AVG_TTFB": 22.231924, "ERRORS": 32, "AVG_DURATION": 79.225784, "ERRORS_PER_SECOND": 0.00625, "MIN_DURATION": 4, "ERROR_RATE": 2.1828103, "MIN_TTFB": 2, "MAX_TTFB": 4161, "MAX_DURATION": 6340, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4745.5317, "COUNT": 1466}, {"from": 317440000, "to": 322560000, "AVG_TTFB": 19.02171, "ERRORS": 0, "AVG_DURATION": 70.274086, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5467, "MAX_DURATION": 5679, "ELEMENTS_PER_SECOND": 0.2878906, "THROUGHPUT": 4855.7715, "COUNT": 1474}, {"from": 322560000, "to": 327680000, "AVG_TTFB": 14.358606, "ERRORS": 17, "AVG_DURATION": 58.191257, "ERRORS_PER_SECOND": 0.0033203126, "MIN_DURATION": 4, "ERROR_RATE": 1.1612022, "MIN_TTFB": 2, "MAX_TTFB": 139, "MAX_DURATION": 1251, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4789.373, "COUNT": 1464}, {"from": 327680000, "to": 332800000, "AVG_TTFB": 31.206991, "ERRORS": 0, "AVG_DURATION": 147.62097, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4533, "MAX_DURATION": 5303, "ELEMENTS_PER_SECOND": 0.28496093, "THROUGHPUT": 4813.0557, "COUNT": 1459}, {"from": 332800000, "to": 337920000, "AVG_TTFB": 17.838224, "ERRORS": 0, "AVG_DURATION": 77.904434, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1078, "MAX_DURATION": 3024, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4824.2793, "COUNT": 1465}, {"from": 337920000, "to": 343040000, "AVG_TTFB": 18.355585, "ERRORS": 0, "AVG_DURATION": 84.43733, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 970, "MAX_DURATION": 2060, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4909.639, "COUNT": 1468}, {"from": 343040000, "to": 348160000, "AVG_TTFB": 14.608726, "ERRORS": 0, "AVG_DURATION": 60.623722, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 123, "MAX_DURATION": 340, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4839.8066, "COUNT": 1467}, {"from": 348160000, "to": 353280000, "AVG_TTFB": 14.653951, "ERRORS": 0, "AVG_DURATION": 62.075615, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 224, "MAX_DURATION": 1584, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4836.8047, "COUNT": 1468}, {"from": 353280000, "to": 358400000, "AVG_TTFB": 18.20725, "ERRORS": 0, "AVG_DURATION": 86.803696, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 815, "MAX_DURATION": 2038, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4827.3784, "COUNT": 1462}, {"from": 358400000, "to": 363520000, "AVG_TTFB": 16.91337, "ERRORS": 0, "AVG_DURATION": 128.01978, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1601, "MAX_DURATION": 26411, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4893.659, "COUNT": 1466}, {"from": 363520000, "to": 368640000, "AVG_TTFB": 17.29809, "ERRORS": 0, "AVG_DURATION": 121.66508, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1465, "MAX_DURATION": 19228, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4839.7236, "COUNT": 1466}, {"from": 368640000, "to": 373760000, "AVG_TTFB": 18.984983, "ERRORS": 0, "AVG_DURATION": 85.57952, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3718, "MAX_DURATION": 7248, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4838.902, "COUNT": 1465}, {"from": 373760000, "to": 378880000, "AVG_TTFB": 22.859589, "ERRORS": 0, "AVG_DURATION": 96.70548, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3741, "MAX_DURATION": 8199, "ELEMENTS_PER_SECOND": 0.28515625, "THROUGHPUT": 4817.939, "COUNT": 1460}, {"from": 378880000, "to": 384000000, "AVG_TTFB": 19.874146, "ERRORS": 0, "AVG_DURATION": 143.20041, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4964, "MAX_DURATION": 29394, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4827.9404, "COUNT": 1462}, {"from": 384000000, "to": 389120000, "AVG_TTFB": 28.76687, "ERRORS": 0, "AVG_DURATION": 113.0743, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4713, "MAX_DURATION": 9040, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4897.4346, "COUNT": 1467}, {"from": 389120000, "to": 394240000, "AVG_TTFB": 24.146091, "ERRORS": 0, "AVG_DURATION": 135.17078, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4245, "MAX_DURATION": 14917, "ELEMENTS_PER_SECOND": 0.28476563, "THROUGHPUT": 4824.958, "COUNT": 1458}, {"from": 394240000, "to": 399360000, "AVG_TTFB": 44.520332, "ERRORS": 89, "AVG_DURATION": 220.78842, "ERRORS_PER_SECOND": 0.017382812, "MIN_DURATION": 4, "ERROR_RATE": 6.133701, "MIN_TTFB": 2, "MAX_TTFB": 7359, "MAX_DURATION": 12889, "ELEMENTS_PER_SECOND": 0.28339845, "THROUGHPUT": 4465.4736, "COUNT": 1451}, {"from": 399360000, "to": 404480000, "AVG_TTFB": 16.6603, "ERRORS": 0, "AVG_DURATION": 92.36153, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 954, "MAX_DURATION": 7057, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4831.624, "COUNT": 1466}, {"from": 404480000, "to": 409600000, "AVG_TTFB": 14.996592, "ERRORS": 0, "AVG_DURATION": 62.63804, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 552, "MAX_DURATION": 1281, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4904.7964, "COUNT": 1467}, {"from": 409600000, "to": 414720000, "AVG_TTFB": 13.900613, "ERRORS": 17, "AVG_DURATION": 64.371, "ERRORS_PER_SECOND": 0.0033203126, "MIN_DURATION": 4, "ERROR_RATE": 1.1572498, "MIN_TTFB": 2, "MAX_TTFB": 129, "MAX_DURATION": 5019, "ELEMENTS_PER_SECOND": 0.28691405, "THROUGHPUT": 4793.1055, "COUNT": 1469}, {"from": 414720000, "to": 419840000, "AVG_TTFB": 20.72529, "ERRORS": 0, "AVG_DURATION": 75.70075, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4137, "MAX_DURATION": 5779, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4836.657, "COUNT": 1467}, {"from": 419840000, "to": 424960000, "AVG_TTFB": 24.710958, "ERRORS": 0, "AVG_DURATION": 163.20891, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3790, "MAX_DURATION": 24606, "ELEMENTS_PER_SECOND": 0.28515625, "THROUGHPUT": 4811.2686, "COUNT": 1460}, {"from": 424960000, "to": 430080000, "AVG_TTFB": 26.802605, "ERRORS": 0, "AVG_DURATION": 131.62645, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 8043, "MAX_DURATION": 11015, "ELEMENTS_PER_SECOND": 0.28496093, "THROUGHPUT": 4819.648, "COUNT": 1459}, {"from": 430080000, "to": 435200000, "AVG_TTFB": 16.949015, "ERRORS": 0, "AVG_DURATION": 65.93338, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3216, "MAX_DURATION": 4248, "ELEMENTS_PER_SECOND": 0.2873047, "THROUGHPUT": 4908.877, "COUNT": 1471}, {"from": 435200000, "to": 440320000, "AVG_TTFB": 15.477489, "ERRORS": 0, "AVG_DURATION": 77.17121, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 747, "MAX_DURATION": 2024, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4837.0845, "COUNT": 1466}, {"from": 440320000, "to": 445440000, "AVG_TTFB": 15.414452, "ERRORS": 0, "AVG_DURATION": 82.23858, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 977, "MAX_DURATION": 4403, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4832.538, "COUNT": 1467}, {"from": 445440000, "to": 450560000, "AVG_TTFB": 18.946001, "ERRORS": 0, "AVG_DURATION": 99.37321, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4465, "MAX_DURATION": 7854, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4827.445, "COUNT": 1463}, {"from": 450560000, "to": 455680000, "AVG_TTFB": 16.49112, "ERRORS": 0, "AVG_DURATION": 106.0123, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1791, "MAX_DURATION": 10725, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4899.4424, "COUNT": 1464}, {"from": 455680000, "to": 460800000, "AVG_TTFB": 14.305235, "ERRORS": 0, "AVG_DURATION": 90.091774, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 306, "MAX_DURATION": 21895, "ELEMENTS_PER_SECOND": 0.2873047, "THROUGHPUT": 4854.6416, "COUNT": 1471}, {"from": 460800000, "to": 465920000, "AVG_TTFB": 17.699932, "ERRORS": 0, "AVG_DURATION": 74.30144, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3256, "MAX_DURATION": 3424, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4823.368, "COUNT": 1463}, {"from": 465920000, "to": 471040000, "AVG_TTFB": 23.39166, "ERRORS": 0, "AVG_DURATION": 102.047165, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 9875, "MAX_DURATION": 27662, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4826.558, "COUNT": 1463}, {"from": 471040000, "to": 476160000, "AVG_TTFB": 30.246761, "ERRORS": 0, "AVG_DURATION": 88.81936, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 9614, "MAX_DURATION": 9678, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4837.0176, "COUNT": 1467}, {"from": 476160000, "to": 481280000, "AVG_TTFB": 14.40723, "ERRORS": 0, "AVG_DURATION": 59.664394, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 619, "MAX_DURATION": 1404, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4892.914, "COUNT": 1466}, {"from": 481280000, "to": 486400000, "AVG_TTFB": 28.646858, "ERRORS": 42, "AVG_DURATION": 111.508194, "ERRORS_PER_SECOND": 0.008203125, "MIN_DURATION": 5, "ERROR_RATE": 2.8688524, "MIN_TTFB": 1, "MAX_TTFB": 6982, "MAX_DURATION": 8059, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4651.2495, "COUNT": 1464}, {"from": 486400000, "to": 491520000, "AVG_TTFB": 14.050477, "ERRORS": 0, "AVG_DURATION": 62.828785, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 216, "MAX_DURATION": 4369, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4833.1797, "COUNT": 1466}, {"from": 491520000, "to": 496640000, "AVG_TTFB": 17.458704, "ERRORS": 0, "AVG_DURATION": 74.4826, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4179, "MAX_DURATION": 7594, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4837.3545, "COUNT": 1465}, {"from": 496640000, "to": 501760000, "AVG_TTFB": 17.639835, "ERRORS": 17, "AVG_DURATION": 72.17326, "ERRORS_PER_SECOND": 0.0033203126, "MIN_DURATION": 4, "ERROR_RATE": 1.159618, "MIN_TTFB": 2, "MAX_TTFB": 1918, "MAX_DURATION": 3121, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4847.5957, "COUNT": 1466}, {"from": 501760000, "to": 506880000, "AVG_TTFB": 16.904568, "ERRORS": 0, "AVG_DURATION": 85.13293, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2194, "MAX_DURATION": 7747, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4840.069, "COUNT": 1467}, {"from": 506880000, "to": 512000000, "AVG_TTFB": 20.510536, "ERRORS": 0, "AVG_DURATION": 90.74099, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4101, "MAX_DURATION": 4791, "ELEMENTS_PER_SECOND": 0.2873047, "THROUGHPUT": 4852.423, "COUNT": 1471}, {"from": 512000000, "to": 517120000, "AVG_TTFB": 18.537304, "ERRORS": 0, "AVG_DURATION": 83.144424, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1925, "MAX_DURATION": 3589, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4820.047, "COUNT": 1461}, {"from": 517120000, "to": 522240000, "AVG_TTFB": 15.456849, "ERRORS": 0, "AVG_DURATION": 67.347946, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 616, "MAX_DURATION": 1713, "ELEMENTS_PER_SECOND": 0.28515625, "THROUGHPUT": 4832.166, "COUNT": 1460}, {"from": 522240000, "to": 527360000, "AVG_TTFB": 29.979536, "ERRORS": 0, "AVG_DURATION": 126.10914, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5741, "MAX_DURATION": 17571, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4895.1123, "COUNT": 1466}, {"from": 527360000, "to": 532480000, "AVG_TTFB": 27.202637, "ERRORS": 0, "AVG_DURATION": 219.69188, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2438, "MAX_DURATION": 31330, "ELEMENTS_PER_SECOND": 0.28144532, "THROUGHPUT": 4772.3433, "COUNT": 1441}, {"from": 532480000, "to": 537600000, "AVG_TTFB": 39.15368, "ERRORS": 1, "AVG_DURATION": 1068.2208, "ERRORS_PER_SECOND": 0.0001953125, "MIN_DURATION": 14, "ERROR_RATE": 0.072150074, "MIN_TTFB": 2, "MAX_TTFB": 26125, "MAX_DURATION": 300191, "ELEMENTS_PER_SECOND": 0.27070314, "THROUGHPUT": 4615.507, "COUNT": 1386}, {"from": 537600000, "to": 542720000, "AVG_TTFB": 53.66898, "ERRORS": 2, "AVG_DURATION": 1768.6891, "ERRORS_PER_SECOND": 0.000390625, "MIN_DURATION": 13, "ERROR_RATE": 0.15432099, "MIN_TTFB": 2, "MAX_TTFB": 25288, "MAX_DURATION": 300103, "ELEMENTS_PER_SECOND": 0.253125, "THROUGHPUT": 4373.33, "COUNT": 1296}, {"from": 542720000, "to": 547840000, "AVG_TTFB": 18.453056, "ERRORS": 4, "AVG_DURATION": 3549.5732, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 13, "ERROR_RATE": 0.4366812, "MIN_TTFB": 2, "MAX_TTFB": 1060, "MAX_DURATION": 349267, "ELEMENTS_PER_SECOND": 0.17890625, "THROUGHPUT": 3341.9338, "COUNT": 916}, {"from": 547840000, "to": 552960000, "AVG_TTFB": 89.988686, "ERRORS": 4, "AVG_DURATION": 2692.926, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 14, "ERROR_RATE": 0.41152263, "MIN_TTFB": 2, "MAX_TTFB": 27974, "MAX_DURATION": 347335, "ELEMENTS_PER_SECOND": 0.18984374, "THROUGHPUT": 3506.419, "COUNT": 972}, {"from": 552960000, "to": 558080000, "AVG_TTFB": 17.342318, "ERRORS": 0, "AVG_DURATION": 274.42947, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1007, "MAX_DURATION": 27176, "ELEMENTS_PER_SECOND": 0.21738282, "THROUGHPUT": 3882.6392, "COUNT": 1113}, {"from": 558080000, "to": 563200000, "AVG_TTFB": 24.866964, "ERRORS": 0, "AVG_DURATION": 169.37321, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1433, "MAX_DURATION": 8575, "ELEMENTS_PER_SECOND": 0.21875, "THROUGHPUT": 3898.0583, "COUNT": 1120}, {"from": 563200000, "to": 568320000, "AVG_TTFB": 19.91119, "ERRORS": 0, "AVG_DURATION": 87.36856, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1981, "MAX_DURATION": 3682, "ELEMENTS_PER_SECOND": 0.21992187, "THROUGHPUT": 3921.1187, "COUNT": 1126}, {"from": 568320000, "to": 573440000, "AVG_TTFB": 38.523132, "ERRORS": 67, "AVG_DURATION": 180.34253, "ERRORS_PER_SECOND": 0.013085937, "MIN_DURATION": 5, "ERROR_RATE": 5.960854, "MIN_TTFB": 2, "MAX_TTFB": 7821, "MAX_DURATION": 15599, "ELEMENTS_PER_SECOND": 0.21953125, "THROUGHPUT": 3709.1409, "COUNT": 1124}, {"from": 573440000, "to": 578560000, "AVG_TTFB": 15.770594, "ERRORS": 0, "AVG_DURATION": 68.11337, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 787, "MAX_DURATION": 1357, "ELEMENTS_PER_SECOND": 0.22050782, "THROUGHPUT": 3927.21, "COUNT": 1129}, {"from": 578560000, "to": 583680000, "AVG_TTFB": 14.414742, "ERRORS": 0, "AVG_DURATION": 62.24245, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 813, "MAX_DURATION": 940, "ELEMENTS_PER_SECOND": 0.21992187, "THROUGHPUT": 3919.1428, "COUNT": 1126}, {"from": 583680000, "to": 588800000, "AVG_TTFB": 20.695923, "ERRORS": 14, "AVG_DURATION": 84.20479, "ERRORS_PER_SECOND": 0.002734375, "MIN_DURATION": 4, "ERROR_RATE": 1.2411348, "MIN_TTFB": 2, "MAX_TTFB": 1765, "MAX_DURATION": 2850, "ELEMENTS_PER_SECOND": 0.2203125, "THROUGHPUT": 3885.9102, "COUNT": 1128}, {"from": 588800000, "to": 593920000, "AVG_TTFB": 14.182624, "ERRORS": 0, "AVG_DURATION": 64.07979, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 299, "MAX_DURATION": 1014, "ELEMENTS_PER_SECOND": 0.2203125, "THROUGHPUT": 3980.9788, "COUNT": 1128}, {"from": 593920000, "to": 599040000, "AVG_TTFB": 15.859292, "ERRORS": 0, "AVG_DURATION": 78.22124, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1445, "MAX_DURATION": 2163, "ELEMENTS_PER_SECOND": 0.22070312, "THROUGHPUT": 3932.5696, "COUNT": 1130}, {"from": 599040000, "to": 604160000, "AVG_TTFB": 17.51773, "ERRORS": 0, "AVG_DURATION": 71.06383, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3196, "MAX_DURATION": 3217, "ELEMENTS_PER_SECOND": 0.2203125, "THROUGHPUT": 3921.5566, "COUNT": 1128}, {"from": 604160000, "to": 609280000, "AVG_TTFB": 14.144504, "ERRORS": 0, "AVG_DURATION": 70.31826, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 218, "MAX_DURATION": 1891, "ELEMENTS_PER_SECOND": 0.2203125, "THROUGHPUT": 3924.5186, "COUNT": 1128}, {"from": 609280000, "to": 614400000, "AVG_TTFB": 14.703151, "ERRORS": 0, "AVG_DURATION": 63.804314, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 547, "MAX_DURATION": 1156, "ELEMENTS_PER_SECOND": 0.117773436, "THROUGHPUT": 2134.4448, "COUNT": 603}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/71e88176-3353-4bcd-941a-284533099b43/elements/50ddd53f-4a4b-4bf8-a6b3-7a262d3d6cef/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 5120000, "AVG_TTFB": 7.6103544, "ERRORS": 0, "AVG_DURATION": 59.75477, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 529, "MAX_DURATION": 3116, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1660.3157, "COUNT": 367}, {"from": 5120000, "to": 10240000, "AVG_TTFB": 11.587432, "ERRORS": 0, "AVG_DURATION": 60.76776, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1023, "MAX_DURATION": 1417, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 10240000, "to": 15360000, "AVG_TTFB": 6.079235, "ERRORS": 0, "AVG_DURATION": 41.254097, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 313, "MAX_DURATION": 623, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 15360000, "to": 20480000, "AVG_TTFB": 7.7934785, "ERRORS": 0, "AVG_DURATION": 39.866848, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 733, "MAX_DURATION": 807, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1478.0605, "COUNT": 368}, {"from": 20480000, "to": 25600000, "AVG_TTFB": 19.947945, "ERRORS": 0, "AVG_DURATION": 67.49589, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1985, "MAX_DURATION": 2256, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 25600000, "to": 30720000, "AVG_TTFB": 7.6775956, "ERRORS": 0, "AVG_DURATION": 45.677597, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1142, "MAX_DURATION": 1423, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 30720000, "to": 35840000, "AVG_TTFB": 16.688524, "ERRORS": 0, "AVG_DURATION": 78.05738, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2059, "MAX_DURATION": 4227, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 35840000, "to": 40960000, "AVG_TTFB": 7.3542233, "ERRORS": 0, "AVG_DURATION": 43.550407, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1436, "MAX_DURATION": 2203, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 40960000, "to": 46080000, "AVG_TTFB": 13.994551, "ERRORS": 0, "AVG_DURATION": 61.400543, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1863, "MAX_DURATION": 1889, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 46080000, "to": 51200000, "AVG_TTFB": 4.4617486, "ERRORS": 0, "AVG_DURATION": 40.934425, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 537, "MAX_DURATION": 1499, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 51200000, "to": 56320000, "AVG_TTFB": 2.63388, "ERRORS": 4, "AVG_DURATION": 48.13388, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 4, "ERROR_RATE": 1.0928962, "MIN_TTFB": 2, "MAX_TTFB": 10, "MAX_DURATION": 4841, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1408.5739, "COUNT": 366}, {"from": 56320000, "to": 61440000, "AVG_TTFB": 11.904633, "ERRORS": 0, "AVG_DURATION": 41.80109, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1856, "MAX_DURATION": 1933, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 61440000, "to": 66560000, "AVG_TTFB": 69.75956, "ERRORS": 0, "AVG_DURATION": 188.57924, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 21955, "MAX_DURATION": 24709, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1475.0215, "COUNT": 366}, {"from": 66560000, "to": 71680000, "AVG_TTFB": 162.85994, "ERRORS": 26, "AVG_DURATION": 412.10083, "ERRORS_PER_SECOND": 0.005078125, "MIN_DURATION": 5, "ERROR_RATE": 7.282913, "MIN_TTFB": 2, "MAX_TTFB": 17837, "MAX_DURATION": 32089, "ELEMENTS_PER_SECOND": 0.069726564, "THROUGHPUT": 1305.07, "COUNT": 357}, {"from": 71680000, "to": 76800000, "AVG_TTFB": 12.030137, "ERRORS": 0, "AVG_DURATION": 52.424656, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 926, "MAX_DURATION": 1493, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 76800000, "to": 81920000, "AVG_TTFB": 18.446867, "ERRORS": 0, "AVG_DURATION": 61.986378, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1448, "MAX_DURATION": 1827, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 81920000, "to": 87040000, "AVG_TTFB": 2.6866486, "ERRORS": 0, "AVG_DURATION": 34.52861, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 45, "MAX_DURATION": 492, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 87040000, "to": 92160000, "AVG_TTFB": 9.970027, "ERRORS": 0, "AVG_DURATION": 49.53406, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1080, "MAX_DURATION": 2647, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 92160000, "to": 97280000, "AVG_TTFB": 7.708447, "ERRORS": 0, "AVG_DURATION": 48.89918, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 893, "MAX_DURATION": 1756, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 97280000, "to": 102400000, "AVG_TTFB": 2.920981, "ERRORS": 0, "AVG_DURATION": 32.871933, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 90, "MAX_DURATION": 110, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 102400000, "to": 107520000, "AVG_TTFB": 7.7342467, "ERRORS": 0, "AVG_DURATION": 39.328766, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1300, "MAX_DURATION": 1377, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 107520000, "to": 112640000, "AVG_TTFB": 8.590786, "ERRORS": 0, "AVG_DURATION": 39.317074, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1874, "MAX_DURATION": 1894, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1479.5801, "COUNT": 369}, {"from": 112640000, "to": 117760000, "AVG_TTFB": 2.669399, "ERRORS": 0, "AVG_DURATION": 33.008198, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 23, "MAX_DURATION": 126, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 117760000, "to": 122880000, "AVG_TTFB": 2.5753424, "ERRORS": 0, "AVG_DURATION": 34.20548, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 7, "MAX_DURATION": 393, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 122880000, "to": 128000000, "AVG_TTFB": 3.953552, "ERRORS": 0, "AVG_DURATION": 34.48907, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 197, "MAX_DURATION": 218, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 128000000, "to": 133120000, "AVG_TTFB": 49.578514, "ERRORS": 0, "AVG_DURATION": 143.92561, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4774, "MAX_DURATION": 5785, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1470.4629, "COUNT": 363}, {"from": 133120000, "to": 138240000, "AVG_TTFB": 6.258152, "ERRORS": 0, "AVG_DURATION": 40.274456, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 987, "MAX_DURATION": 1272, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 138240000, "to": 143360000, "AVG_TTFB": 11.309589, "ERRORS": 8, "AVG_DURATION": 55.38904, "ERRORS_PER_SECOND": 0.0015625, "MIN_DURATION": 5, "ERROR_RATE": 2.1917808, "MIN_TTFB": 2, "MAX_TTFB": 734, "MAX_DURATION": 1830, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1401.875, "COUNT": 365}, {"from": 143360000, "to": 148480000, "AVG_TTFB": 6.47139, "ERRORS": 0, "AVG_DURATION": 36.724796, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1190, "MAX_DURATION": 1215, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 148480000, "to": 153600000, "AVG_TTFB": 3.730245, "ERRORS": 4, "AVG_DURATION": 36.3297, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 4, "ERROR_RATE": 1.0899183, "MIN_TTFB": 1, "MAX_TTFB": 451, "MAX_DURATION": 1025, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1410.0891, "COUNT": 367}, {"from": 153600000, "to": 158720000, "AVG_TTFB": 9.122616, "ERRORS": 0, "AVG_DURATION": 40.179836, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 892, "MAX_DURATION": 924, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.5227, "COUNT": 367}, {"from": 158720000, "to": 163840000, "AVG_TTFB": 5.6702995, "ERRORS": 0, "AVG_DURATION": 46.93733, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 521, "MAX_DURATION": 1694, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 163840000, "to": 168960000, "AVG_TTFB": 3.7643836, "ERRORS": 0, "AVG_DURATION": 33.969864, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 319, "MAX_DURATION": 394, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 168960000, "to": 174080000, "AVG_TTFB": 4.3722825, "ERRORS": 0, "AVG_DURATION": 35.01902, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 610, "MAX_DURATION": 624, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 174080000, "to": 179200000, "AVG_TTFB": 12.190736, "ERRORS": 0, "AVG_DURATION": 42.223434, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3539, "MAX_DURATION": 3557, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 179200000, "to": 184320000, "AVG_TTFB": 8.483696, "ERRORS": 0, "AVG_DURATION": 45.701088, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 830, "MAX_DURATION": 847, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 184320000, "to": 189440000, "AVG_TTFB": 4.0027323, "ERRORS": 0, "AVG_DURATION": 34.26776, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 526, "MAX_DURATION": 539, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 189440000, "to": 194560000, "AVG_TTFB": 3.3770492, "ERRORS": 0, "AVG_DURATION": 34.53825, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 150, "MAX_DURATION": 165, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 194560000, "to": 199680000, "AVG_TTFB": 7.6164384, "ERRORS": 0, "AVG_DURATION": 37.515068, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 960, "MAX_DURATION": 977, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 199680000, "to": 204800000, "AVG_TTFB": 14.146739, "ERRORS": 0, "AVG_DURATION": 60.035328, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1838, "MAX_DURATION": 1853, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1478.0605, "COUNT": 368}, {"from": 204800000, "to": 209920000, "AVG_TTFB": 23.567934, "ERRORS": 0, "AVG_DURATION": 62.138588, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3831, "MAX_DURATION": 5218, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 209920000, "to": 215040000, "AVG_TTFB": 2.5983605, "ERRORS": 0, "AVG_DURATION": 32.30601, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 21, "MAX_DURATION": 84, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 215040000, "to": 220160000, "AVG_TTFB": 4.7540984, "ERRORS": 0, "AVG_DURATION": 46.02732, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 72, "MAX_DURATION": 394, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 220160000, "to": 225280000, "AVG_TTFB": 74.92055, "ERRORS": 0, "AVG_DURATION": 240.91232, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 13385, "MAX_DURATION": 32618, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1473.502, "COUNT": 365}, {"from": 225280000, "to": 230400000, "AVG_TTFB": 29.527323, "ERRORS": 23, "AVG_DURATION": 117.1612, "ERRORS_PER_SECOND": 0.0044921874, "MIN_DURATION": 4, "ERROR_RATE": 6.284153, "MIN_TTFB": 1, "MAX_TTFB": 3638, "MAX_DURATION": 4975, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1322.7336, "COUNT": 366}, {"from": 230400000, "to": 235520000, "AVG_TTFB": 8.729508, "ERRORS": 0, "AVG_DURATION": 39.994537, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1692, "MAX_DURATION": 1769, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 235520000, "to": 240640000, "AVG_TTFB": 2.5191257, "ERRORS": 5, "AVG_DURATION": 31.773224, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 4, "ERROR_RATE": 1.3661202, "MIN_TTFB": 1, "MAX_TTFB": 6, "MAX_DURATION": 82, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1407.2754, "COUNT": 366}, {"from": 240640000, "to": 245760000, "AVG_TTFB": 28.230137, "ERRORS": 0, "AVG_DURATION": 143.2685, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4386, "MAX_DURATION": 12515, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 245760000, "to": 250880000, "AVG_TTFB": 3.1444142, "ERRORS": 0, "AVG_DURATION": 34.237057, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 79, "MAX_DURATION": 206, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 250880000, "to": 256000000, "AVG_TTFB": 5.0901637, "ERRORS": 0, "AVG_DURATION": 39.289616, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 573, "MAX_DURATION": 1616, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 256000000, "to": 261120000, "AVG_TTFB": 7.381471, "ERRORS": 0, "AVG_DURATION": 40.29973, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1278, "MAX_DURATION": 1302, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 261120000, "to": 266240000, "AVG_TTFB": 6.2868853, "ERRORS": 0, "AVG_DURATION": 36.68306, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1301, "MAX_DURATION": 1372, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 266240000, "to": 271360000, "AVG_TTFB": 2.704607, "ERRORS": 0, "AVG_DURATION": 35.159893, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 52, "MAX_DURATION": 594, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1479.5801, "COUNT": 369}, {"from": 271360000, "to": 276480000, "AVG_TTFB": 24.923498, "ERRORS": 0, "AVG_DURATION": 81.773224, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5533, "MAX_DURATION": 6199, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 276480000, "to": 281600000, "AVG_TTFB": 3.5122616, "ERRORS": 0, "AVG_DURATION": 41.12534, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 221, "MAX_DURATION": 1655, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 281600000, "to": 286720000, "AVG_TTFB": 15.008152, "ERRORS": 0, "AVG_DURATION": 46.309784, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2821, "MAX_DURATION": 2837, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 286720000, "to": 291840000, "AVG_TTFB": 8.420765, "ERRORS": 0, "AVG_DURATION": 41.95082, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 958, "MAX_DURATION": 1033, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 291840000, "to": 296960000, "AVG_TTFB": 10.536785, "ERRORS": 0, "AVG_DURATION": 41.93733, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1181, "MAX_DURATION": 1209, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 296960000, "to": 302080000, "AVG_TTFB": 9.441417, "ERRORS": 0, "AVG_DURATION": 39.986378, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1322, "MAX_DURATION": 1337, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 302080000, "to": 307200000, "AVG_TTFB": 30.144808, "ERRORS": 0, "AVG_DURATION": 97.57377, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4099, "MAX_DURATION": 7568, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 307200000, "to": 312320000, "AVG_TTFB": 16.565575, "ERRORS": 0, "AVG_DURATION": 52.404373, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4911, "MAX_DURATION": 4929, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 312320000, "to": 317440000, "AVG_TTFB": 4.1389647, "ERRORS": 7, "AVG_DURATION": 54.354225, "ERRORS_PER_SECOND": 0.0013671875, "MIN_DURATION": 4, "ERROR_RATE": 1.907357, "MIN_TTFB": 1, "MAX_TTFB": 222, "MAX_DURATION": 6514, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1467.4769, "COUNT": 367}, {"from": 317440000, "to": 322560000, "AVG_TTFB": 20.959349, "ERRORS": 0, "AVG_DURATION": 51.471546, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4447, "MAX_DURATION": 4468, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1418.3219, "COUNT": 369}, {"from": 322560000, "to": 327680000, "AVG_TTFB": 2.7041097, "ERRORS": 5, "AVG_DURATION": 32.358906, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 4, "ERROR_RATE": 1.369863, "MIN_TTFB": 2, "MAX_TTFB": 27, "MAX_DURATION": 83, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1405.7415, "COUNT": 365}, {"from": 327680000, "to": 332800000, "AVG_TTFB": 23.871935, "ERRORS": 0, "AVG_DURATION": 90.75477, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1532, "MAX_DURATION": 3160, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 332800000, "to": 337920000, "AVG_TTFB": 5.103825, "ERRORS": 0, "AVG_DURATION": 48.35792, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 315, "MAX_DURATION": 2683, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 337920000, "to": 343040000, "AVG_TTFB": 13.002725, "ERRORS": 0, "AVG_DURATION": 55.81199, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 998, "MAX_DURATION": 1418, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 343040000, "to": 348160000, "AVG_TTFB": 2.691057, "ERRORS": 0, "AVG_DURATION": 33.623306, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 26, "MAX_DURATION": 169, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1418.3219, "COUNT": 369}, {"from": 348160000, "to": 353280000, "AVG_TTFB": 5.724796, "ERRORS": 0, "AVG_DURATION": 35.697548, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1125, "MAX_DURATION": 1197, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 353280000, "to": 358400000, "AVG_TTFB": 4.8767123, "ERRORS": 0, "AVG_DURATION": 40.78356, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 445, "MAX_DURATION": 497, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 358400000, "to": 363520000, "AVG_TTFB": 5.2119565, "ERRORS": 0, "AVG_DURATION": 41.154892, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 869, "MAX_DURATION": 1414, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1478.0605, "COUNT": 368}, {"from": 363520000, "to": 368640000, "AVG_TTFB": 5.2255435, "ERRORS": 0, "AVG_DURATION": 44.225544, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 474, "MAX_DURATION": 2548, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 368640000, "to": 373760000, "AVG_TTFB": 9.114754, "ERRORS": 0, "AVG_DURATION": 48.636612, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2129, "MAX_DURATION": 2252, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 373760000, "to": 378880000, "AVG_TTFB": 8.961853, "ERRORS": 0, "AVG_DURATION": 55.51226, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 772, "MAX_DURATION": 3267, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 378880000, "to": 384000000, "AVG_TTFB": 6.2739725, "ERRORS": 0, "AVG_DURATION": 38.361645, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1012, "MAX_DURATION": 1029, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 384000000, "to": 389120000, "AVG_TTFB": 3.4945354, "ERRORS": 0, "AVG_DURATION": 39.551914, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 123, "MAX_DURATION": 420, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1475.0215, "COUNT": 366}, {"from": 389120000, "to": 394240000, "AVG_TTFB": 9.571038, "ERRORS": 0, "AVG_DURATION": 69.38525, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1010, "MAX_DURATION": 2397, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 394240000, "to": 399360000, "AVG_TTFB": 33.923496, "ERRORS": 26, "AVG_DURATION": 155.95355, "ERRORS_PER_SECOND": 0.005078125, "MIN_DURATION": 4, "ERROR_RATE": 7.103825, "MIN_TTFB": 1, "MAX_TTFB": 4742, "MAX_DURATION": 8685, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1318.7457, "COUNT": 366}, {"from": 399360000, "to": 404480000, "AVG_TTFB": 25.334246, "ERRORS": 3, "AVG_DURATION": 101.51233, "ERRORS_PER_SECOND": 0.0005859375, "MIN_DURATION": 14, "ERROR_RATE": 0.82191783, "MIN_TTFB": 1, "MAX_TTFB": 5528, "MAX_DURATION": 9887, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1408.3547, "COUNT": 365}, {"from": 404480000, "to": 409600000, "AVG_TTFB": 2.8157182, "ERRORS": 0, "AVG_DURATION": 38.457996, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 41, "MAX_DURATION": 1567, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1479.5801, "COUNT": 369}, {"from": 409600000, "to": 414720000, "AVG_TTFB": 2.693989, "ERRORS": 4, "AVG_DURATION": 33.79235, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 4, "ERROR_RATE": 1.0928962, "MIN_TTFB": 2, "MAX_TTFB": 27, "MAX_DURATION": 589, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1408.5883, "COUNT": 366}, {"from": 414720000, "to": 419840000, "AVG_TTFB": 5.0245233, "ERRORS": 0, "AVG_DURATION": 46.190735, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 771, "MAX_DURATION": 3443, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 419840000, "to": 424960000, "AVG_TTFB": 15.368132, "ERRORS": 0, "AVG_DURATION": 71.93132, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2932, "MAX_DURATION": 9950, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 424960000, "to": 430080000, "AVG_TTFB": 28.270493, "ERRORS": 0, "AVG_DURATION": 79.510925, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6712, "MAX_DURATION": 7990, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 430080000, "to": 435200000, "AVG_TTFB": 6.910569, "ERRORS": 0, "AVG_DURATION": 45.214092, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 854, "MAX_DURATION": 1875, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1479.5801, "COUNT": 369}, {"from": 435200000, "to": 440320000, "AVG_TTFB": 5.0054646, "ERRORS": 0, "AVG_DURATION": 36.48907, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 599, "MAX_DURATION": 619, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 440320000, "to": 445440000, "AVG_TTFB": 10.185792, "ERRORS": 0, "AVG_DURATION": 42.595627, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1665, "MAX_DURATION": 1775, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 445440000, "to": 450560000, "AVG_TTFB": 25.26703, "ERRORS": 0, "AVG_DURATION": 55.93188, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4732, "MAX_DURATION": 4838, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 450560000, "to": 455680000, "AVG_TTFB": 5.8233695, "ERRORS": 0, "AVG_DURATION": 35.940216, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 450, "MAX_DURATION": 521, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1478.0605, "COUNT": 368}, {"from": 455680000, "to": 460800000, "AVG_TTFB": 2.9836957, "ERRORS": 0, "AVG_DURATION": 34.75, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 16, "MAX_DURATION": 120, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 460800000, "to": 465920000, "AVG_TTFB": 14.887979, "ERRORS": 0, "AVG_DURATION": 49.934425, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 16, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2039, "MAX_DURATION": 2065, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 465920000, "to": 471040000, "AVG_TTFB": 2.5951087, "ERRORS": 0, "AVG_DURATION": 32.64402, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6, "MAX_DURATION": 82, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 471040000, "to": 476160000, "AVG_TTFB": 4.5901637, "ERRORS": 0, "AVG_DURATION": 47.54918, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 181, "MAX_DURATION": 1775, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 476160000, "to": 481280000, "AVG_TTFB": 3.0923913, "ERRORS": 0, "AVG_DURATION": 34.008152, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 161, "MAX_DURATION": 180, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1478.0605, "COUNT": 368}, {"from": 481280000, "to": 486400000, "AVG_TTFB": 17.653006, "ERRORS": 12, "AVG_DURATION": 58.396175, "ERRORS_PER_SECOND": 0.00234375, "MIN_DURATION": 4, "ERROR_RATE": 3.2786884, "MIN_TTFB": 2, "MAX_TTFB": 2323, "MAX_DURATION": 2585, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1398.1781, "COUNT": 366}, {"from": 486400000, "to": 491520000, "AVG_TTFB": 2.9291553, "ERRORS": 0, "AVG_DURATION": 45.752045, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 110, "MAX_DURATION": 4730, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 491520000, "to": 496640000, "AVG_TTFB": 7.09863, "ERRORS": 0, "AVG_DURATION": 46.61644, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1080, "MAX_DURATION": 1765, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 496640000, "to": 501760000, "AVG_TTFB": 8.081744, "ERRORS": 4, "AVG_DURATION": 40.79019, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 4, "ERROR_RATE": 1.0899183, "MIN_TTFB": 1, "MAX_TTFB": 2003, "MAX_DURATION": 2569, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1471.3372, "COUNT": 367}, {"from": 501760000, "to": 506880000, "AVG_TTFB": 16.160763, "ERRORS": 0, "AVG_DURATION": 66.531334, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2818, "MAX_DURATION": 8056, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 506880000, "to": 512000000, "AVG_TTFB": 8.601093, "ERRORS": 0, "AVG_DURATION": 42.453552, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1157, "MAX_DURATION": 1191, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 512000000, "to": 517120000, "AVG_TTFB": 7.057377, "ERRORS": 0, "AVG_DURATION": 42.46448, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1084, "MAX_DURATION": 1100, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 517120000, "to": 522240000, "AVG_TTFB": 3.520436, "ERRORS": 0, "AVG_DURATION": 40.746593, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 152, "MAX_DURATION": 1750, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 522240000, "to": 527360000, "AVG_TTFB": 20.453552, "ERRORS": 0, "AVG_DURATION": 85.37158, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4840, "MAX_DURATION": 6508, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1475.0215, "COUNT": 366}, {"from": 527360000, "to": 532480000, "AVG_TTFB": 6.951087, "ERRORS": 0, "AVG_DURATION": 51.891304, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 382, "MAX_DURATION": 1200, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 532480000, "to": 537600000, "AVG_TTFB": 5.8825135, "ERRORS": 0, "AVG_DURATION": 50.346996, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 542, "MAX_DURATION": 2496, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 537600000, "to": 542720000, "AVG_TTFB": 10.346994, "ERRORS": 0, "AVG_DURATION": 48.005463, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2596, "MAX_DURATION": 4102, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 542720000, "to": 547840000, "AVG_TTFB": 5.8501363, "ERRORS": 0, "AVG_DURATION": 36.85286, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 750, "MAX_DURATION": 774, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 547840000, "to": 552960000, "AVG_TTFB": 2.86612, "ERRORS": 0, "AVG_DURATION": 40.07377, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 99, "MAX_DURATION": 1481, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 552960000, "to": 558080000, "AVG_TTFB": 6.3841963, "ERRORS": 0, "AVG_DURATION": 39.049046, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 740, "MAX_DURATION": 758, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 558080000, "to": 563200000, "AVG_TTFB": 33.578083, "ERRORS": 0, "AVG_DURATION": 107.287674, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 7389, "MAX_DURATION": 8233, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 563200000, "to": 568320000, "AVG_TTFB": 5.185286, "ERRORS": 0, "AVG_DURATION": 44.880108, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 390, "MAX_DURATION": 811, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 568320000, "to": 573440000, "AVG_TTFB": 18.687672, "ERRORS": 29, "AVG_DURATION": 83.369865, "ERRORS_PER_SECOND": 0.0056640627, "MIN_DURATION": 4, "ERROR_RATE": 7.9452057, "MIN_TTFB": 2, "MAX_TTFB": 1605, "MAX_DURATION": 7301, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1374.6016, "COUNT": 365}, {"from": 573440000, "to": 578560000, "AVG_TTFB": 2.6983695, "ERRORS": 0, "AVG_DURATION": 32.442936, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 52, "MAX_DURATION": 82, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 578560000, "to": 583680000, "AVG_TTFB": 2.4755435, "ERRORS": 0, "AVG_DURATION": 31.97826, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 5, "MAX_DURATION": 80, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 583680000, "to": 588800000, "AVG_TTFB": 3.578082, "ERRORS": 4, "AVG_DURATION": 39.134247, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 4, "ERROR_RATE": 1.0958904, "MIN_TTFB": 1, "MAX_TTFB": 151, "MAX_DURATION": 926, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1407.0543, "COUNT": 365}, {"from": 588800000, "to": 593920000, "AVG_TTFB": 2.7820163, "ERRORS": 0, "AVG_DURATION": 35.607628, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 45, "MAX_DURATION": 714, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 593920000, "to": 599040000, "AVG_TTFB": 10.210811, "ERRORS": 0, "AVG_DURATION": 39.92973, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1534, "MAX_DURATION": 1552, "ELEMENTS_PER_SECOND": 0.072265625, "THROUGHPUT": 1419.8414, "COUNT": 370}, {"from": 599040000, "to": 604160000, "AVG_TTFB": 5.5150685, "ERRORS": 0, "AVG_DURATION": 37.824657, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 773, "MAX_DURATION": 846, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 604160000, "to": 609280000, "AVG_TTFB": 3.9266305, "ERRORS": 0, "AVG_DURATION": 38.16576, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 460, "MAX_DURATION": 1324, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 609280000, "to": 614400000, "AVG_TTFB": 10.28718, "ERRORS": 0, "AVG_DURATION": 39.210255, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1270, "MAX_DURATION": 1287, "ELEMENTS_PER_SECOND": 0.038085938, "THROUGHPUT": 786.3742, "COUNT": 195}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/ba912bd5-d4ee-425c-ae82-b56a75c0e6ef/elements/50ddd53f-4a4b-4bf8-a6b3-7a262d3d6cef/values':
            return '{"count": 42488, "elementPerSecond": 0.06942365, "minDuration": 4, "maxDuration": 347335, "sumDuration": 6375181, "avgDuration": 150.04663, "minTTFB": 1, "maxTTFB": 26125, "sumTTFB": 572596, "avgTTFB": 13.476652, "sumDownloadedBytes": 862332257, "downloadedBytesPerSecond": 1409.0156, "successCount": 42318, "successPerSecond": 0.06914588, "successRate": 99.599884, "failureCount": 170, "failurePerSecond": 0.00027777307, "failureRate": 0.400113, "percentile50": 19.0, "percentile90": 76.0, "percentile95": 82.0, "percentile99": 1003.5}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/71e88176-3353-4bcd-941a-284533099b43/elements/0ca05bf2-4ba9-483e-9041-c9ac38ecb8b8/values':
            return '{"count": 43769, "elementPerSecond": 0.07151672, "minDuration": 13, "maxDuration": 25501, "sumDuration": 3875062, "avgDuration": 88.53439, "minTTFB": 2, "maxTTFB": 25208, "sumTTFB": 613668, "avgTTFB": 14.020608, "sumDownloadedBytes": 1046435509, "downloadedBytesPerSecond": 1709.832, "successCount": 43592, "successPerSecond": 0.07122751, "successRate": 99.595604, "failureCount": 177, "failurePerSecond": 0.00028921064, "failureRate": 0.40439582, "percentile50": 28.0, "percentile90": 106.0, "percentile95": 147.0, "percentile99": 1255.5}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/71e88176-3353-4bcd-941a-284533099b43/elements/062b066f-e7c0-4e67-9a08-65d6494fecc9/values':
            return '{"count": 43789, "elementPerSecond": 0.0715494, "minDuration": 12, "maxDuration": 39397, "sumDuration": 7261939, "avgDuration": 165.83934, "minTTFB": 0, "maxTTFB": 38709, "sumTTFB": 597480, "avgTTFB": 13.644523, "sumDownloadedBytes": 1039999186, "downloadedBytesPerSecond": 1699.3154, "successCount": 43615, "successPerSecond": 0.071265094, "successRate": 99.60264, "failureCount": 174, "failurePerSecond": 0.00028430877, "failureRate": 0.39736006, "percentile50": 70.0, "percentile90": 218.0, "percentile95": 255.0, "percentile99": 1628.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/ba912bd5-d4ee-425c-ae82-b56a75c0e6ef/elements/0ca05bf2-4ba9-483e-9041-c9ac38ecb8b8/values':
            return '{"count": 42438, "elementPerSecond": 0.06934196, "minDuration": 14, "maxDuration": 300103, "sumDuration": 6675153, "avgDuration": 157.29189, "minTTFB": 2, "maxTTFB": 27974, "sumTTFB": 605443, "avgTTFB": 14.26653, "sumDownloadedBytes": 1014341342, "downloadedBytesPerSecond": 1657.3923, "successCount": 42265, "successPerSecond": 0.06905928, "successRate": 99.59235, "failureCount": 173, "failurePerSecond": 0.00028267494, "failureRate": 0.4076535, "percentile50": 30.0, "percentile90": 109.0, "percentile95": 164.0, "percentile99": 1487.5}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/71e88176-3353-4bcd-941a-284533099b43/elements/all-transactions/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 5120000, "AVG_TTFB": 20.777018, "ERRORS": 0, "AVG_DURATION": 88.40287, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1941, "MAX_DURATION": 3116, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 5074.6216, "COUNT": 1462}, {"from": 5120000, "to": 10240000, "AVG_TTFB": 19.699932, "ERRORS": 0, "AVG_DURATION": 89.74778, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1124, "MAX_DURATION": 2634, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4831.117, "COUNT": 1463}, {"from": 10240000, "to": 15360000, "AVG_TTFB": 15.615174, "ERRORS": 0, "AVG_DURATION": 70.27, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 317, "MAX_DURATION": 4255, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4833.4517, "COUNT": 1463}, {"from": 15360000, "to": 20480000, "AVG_TTFB": 16.445503, "ERRORS": 0, "AVG_DURATION": 71.64101, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1173, "MAX_DURATION": 2588, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4902.4097, "COUNT": 1468}, {"from": 20480000, "to": 25600000, "AVG_TTFB": 24.9194, "ERRORS": 0, "AVG_DURATION": 113.66462, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4449, "MAX_DURATION": 10159, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4834.401, "COUNT": 1464}, {"from": 25600000, "to": 30720000, "AVG_TTFB": 21.150854, "ERRORS": 0, "AVG_DURATION": 91.057335, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3042, "MAX_DURATION": 5173, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4834.575, "COUNT": 1465}, {"from": 30720000, "to": 35840000, "AVG_TTFB": 25.73274, "ERRORS": 0, "AVG_DURATION": 111.73958, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4457, "MAX_DURATION": 6413, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4823.765, "COUNT": 1463}, {"from": 35840000, "to": 40960000, "AVG_TTFB": 16.478172, "ERRORS": 0, "AVG_DURATION": 65.79945, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1436, "MAX_DURATION": 2203, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4904.95, "COUNT": 1466}, {"from": 40960000, "to": 46080000, "AVG_TTFB": 21.526352, "ERRORS": 0, "AVG_DURATION": 138.09924, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1863, "MAX_DURATION": 15555, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4821.683, "COUNT": 1461}, {"from": 46080000, "to": 51200000, "AVG_TTFB": 27.429447, "ERRORS": 0, "AVG_DURATION": 77.092705, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4649, "MAX_DURATION": 4673, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4829.245, "COUNT": 1467}, {"from": 51200000, "to": 56320000, "AVG_TTFB": 17.99249, "ERRORS": 14, "AVG_DURATION": 65.535835, "ERRORS_PER_SECOND": 0.002734375, "MIN_DURATION": 4, "ERROR_RATE": 0.9556314, "MIN_TTFB": 2, "MAX_TTFB": 5256, "MAX_DURATION": 5323, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4800.1045, "COUNT": 1465}, {"from": 56320000, "to": 61440000, "AVG_TTFB": 16.847885, "ERRORS": 0, "AVG_DURATION": 64.081856, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1856, "MAX_DURATION": 2690, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4837.223, "COUNT": 1466}, {"from": 61440000, "to": 66560000, "AVG_TTFB": 41.54427, "ERRORS": 0, "AVG_DURATION": 177.73439, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 21955, "MAX_DURATION": 24709, "ELEMENTS_PER_SECOND": 0.2845703, "THROUGHPUT": 4871.27, "COUNT": 1457}, {"from": 66560000, "to": 71680000, "AVG_TTFB": 96.5272, "ERRORS": 76, "AVG_DURATION": 469.2162, "ERRORS_PER_SECOND": 0.01484375, "MIN_DURATION": 5, "ERROR_RATE": 5.2998605, "MIN_TTFB": 1, "MAX_TTFB": 17837, "MAX_DURATION": 37863, "ELEMENTS_PER_SECOND": 0.2800781, "THROUGHPUT": 4471.0, "COUNT": 1434}, {"from": 71680000, "to": 76800000, "AVG_TTFB": 22.59343, "ERRORS": 0, "AVG_DURATION": 102.58727, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1236, "MAX_DURATION": 3268, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4824.1455, "COUNT": 1461}, {"from": 76800000, "to": 81920000, "AVG_TTFB": 21.854607, "ERRORS": 0, "AVG_DURATION": 94.98772, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1504, "MAX_DURATION": 2750, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4827.8174, "COUNT": 1465}, {"from": 81920000, "to": 87040000, "AVG_TTFB": 16.041582, "ERRORS": 0, "AVG_DURATION": 70.92024, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 940, "MAX_DURATION": 3190, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4902.8325, "COUNT": 1467}, {"from": 87040000, "to": 92160000, "AVG_TTFB": 24.258152, "ERRORS": 0, "AVG_DURATION": 86.80095, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4714, "MAX_DURATION": 4978, "ELEMENTS_PER_SECOND": 0.2875, "THROUGHPUT": 4855.702, "COUNT": 1472}, {"from": 92160000, "to": 97280000, "AVG_TTFB": 17.785666, "ERRORS": 0, "AVG_DURATION": 83.80546, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1274, "MAX_DURATION": 4183, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4829.443, "COUNT": 1465}, {"from": 97280000, "to": 102400000, "AVG_TTFB": 14.702997, "ERRORS": 0, "AVG_DURATION": 69.17915, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 649, "MAX_DURATION": 9702, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4838.0103, "COUNT": 1468}, {"from": 102400000, "to": 107520000, "AVG_TTFB": 16.6157, "ERRORS": 0, "AVG_DURATION": 64.59181, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1300, "MAX_DURATION": 3212, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4833.805, "COUNT": 1465}, {"from": 107520000, "to": 112640000, "AVG_TTFB": 15.242672, "ERRORS": 0, "AVG_DURATION": 66.80368, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1874, "MAX_DURATION": 3908, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4909.47, "COUNT": 1467}, {"from": 112640000, "to": 117760000, "AVG_TTFB": 15.991138, "ERRORS": 0, "AVG_DURATION": 62.074303, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3715, "MAX_DURATION": 3734, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4845.639, "COUNT": 1467}, {"from": 117760000, "to": 122880000, "AVG_TTFB": 14.17635, "ERRORS": 0, "AVG_DURATION": 56.944633, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 926, "MAX_DURATION": 1050, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4836.9375, "COUNT": 1463}, {"from": 122880000, "to": 128000000, "AVG_TTFB": 17.446342, "ERRORS": 0, "AVG_DURATION": 70.68763, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3784, "MAX_DURATION": 3805, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4833.6367, "COUNT": 1463}, {"from": 128000000, "to": 133120000, "AVG_TTFB": 48.267082, "ERRORS": 0, "AVG_DURATION": 245.9855, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4774, "MAX_DURATION": 17518, "ELEMENTS_PER_SECOND": 0.2830078, "THROUGHPUT": 4857.262, "COUNT": 1449}, {"from": 133120000, "to": 138240000, "AVG_TTFB": 18.223505, "ERRORS": 0, "AVG_DURATION": 73.446335, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1963, "MAX_DURATION": 3660, "ELEMENTS_PER_SECOND": 0.2875, "THROUGHPUT": 4855.7036, "COUNT": 1472}, {"from": 138240000, "to": 143360000, "AVG_TTFB": 31.657534, "ERRORS": 25, "AVG_DURATION": 116.70274, "ERRORS_PER_SECOND": 0.0048828125, "MIN_DURATION": 5, "ERROR_RATE": 1.7123288, "MIN_TTFB": 2, "MAX_TTFB": 7871, "MAX_DURATION": 12874, "ELEMENTS_PER_SECOND": 0.28515625, "THROUGHPUT": 4746.133, "COUNT": 1460}, {"from": 143360000, "to": 148480000, "AVG_TTFB": 15.637168, "ERRORS": 0, "AVG_DURATION": 62.625595, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1190, "MAX_DURATION": 2300, "ELEMENTS_PER_SECOND": 0.28691405, "THROUGHPUT": 4837.149, "COUNT": 1469}, {"from": 148480000, "to": 153600000, "AVG_TTFB": 14.748295, "ERRORS": 12, "AVG_DURATION": 62.311733, "ERRORS_PER_SECOND": 0.00234375, "MIN_DURATION": 4, "ERROR_RATE": 0.81855386, "MIN_TTFB": 1, "MAX_TTFB": 1211, "MAX_DURATION": 3455, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4803.9253, "COUNT": 1466}, {"from": 153600000, "to": 158720000, "AVG_TTFB": 20.153847, "ERRORS": 3, "AVG_DURATION": 98.339005, "ERRORS_PER_SECOND": 0.0005859375, "MIN_DURATION": 14, "ERROR_RATE": 0.20422056, "MIN_TTFB": 2, "MAX_TTFB": 2165, "MAX_DURATION": 9199, "ELEMENTS_PER_SECOND": 0.28691405, "THROUGHPUT": 4894.3643, "COUNT": 1469}, {"from": 158720000, "to": 163840000, "AVG_TTFB": 19.535788, "ERRORS": 0, "AVG_DURATION": 86.27608, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1369, "MAX_DURATION": 5731, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4846.5703, "COUNT": 1467}, {"from": 163840000, "to": 168960000, "AVG_TTFB": 14.469542, "ERRORS": 0, "AVG_DURATION": 64.53251, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 412, "MAX_DURATION": 1789, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4828.1826, "COUNT": 1461}, {"from": 168960000, "to": 174080000, "AVG_TTFB": 14.203678, "ERRORS": 0, "AVG_DURATION": 67.80382, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 610, "MAX_DURATION": 3949, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4840.786, "COUNT": 1468}, {"from": 174080000, "to": 179200000, "AVG_TTFB": 18.950102, "ERRORS": 0, "AVG_DURATION": 70.97403, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4363, "MAX_DURATION": 4616, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4892.9663, "COUNT": 1463}, {"from": 179200000, "to": 184320000, "AVG_TTFB": 21.476871, "ERRORS": 0, "AVG_DURATION": 89.556465, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2501, "MAX_DURATION": 5299, "ELEMENTS_PER_SECOND": 0.28710938, "THROUGHPUT": 4852.4414, "COUNT": 1470}, {"from": 184320000, "to": 189440000, "AVG_TTFB": 19.281229, "ERRORS": 0, "AVG_DURATION": 70.90649, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3087, "MAX_DURATION": 4101, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4836.545, "COUNT": 1465}, {"from": 189440000, "to": 194560000, "AVG_TTFB": 13.819796, "ERRORS": 0, "AVG_DURATION": 56.47099, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 150, "MAX_DURATION": 272, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4837.619, "COUNT": 1465}, {"from": 194560000, "to": 199680000, "AVG_TTFB": 17.688782, "ERRORS": 0, "AVG_DURATION": 72.2565, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1645, "MAX_DURATION": 3975, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4834.3096, "COUNT": 1462}, {"from": 199680000, "to": 204800000, "AVG_TTFB": 24.529694, "ERRORS": 0, "AVG_DURATION": 106.48601, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2220, "MAX_DURATION": 8276, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4898.1265, "COUNT": 1465}, {"from": 204800000, "to": 209920000, "AVG_TTFB": 21.42789, "ERRORS": 0, "AVG_DURATION": 85.7, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3831, "MAX_DURATION": 6232, "ELEMENTS_PER_SECOND": 0.28710938, "THROUGHPUT": 4844.6567, "COUNT": 1470}, {"from": 209920000, "to": 215040000, "AVG_TTFB": 13.449829, "ERRORS": 0, "AVG_DURATION": 54.118088, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 80, "MAX_DURATION": 293, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4832.6934, "COUNT": 1465}, {"from": 215040000, "to": 220160000, "AVG_TTFB": 17.782402, "ERRORS": 0, "AVG_DURATION": 82.27967, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 502, "MAX_DURATION": 1481, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4834.398, "COUNT": 1466}, {"from": 220160000, "to": 225280000, "AVG_TTFB": 87.10283, "ERRORS": 0, "AVG_DURATION": 224.1049, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 38709, "MAX_DURATION": 39397, "ELEMENTS_PER_SECOND": 0.2830078, "THROUGHPUT": 4851.9814, "COUNT": 1449}, {"from": 225280000, "to": 230400000, "AVG_TTFB": 38.58515, "ERRORS": 72, "AVG_DURATION": 138.82903, "ERRORS_PER_SECOND": 0.0140625, "MIN_DURATION": 4, "ERROR_RATE": 4.904632, "MIN_TTFB": 1, "MAX_TTFB": 6733, "MAX_DURATION": 9095, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4561.451, "COUNT": 1468}, {"from": 230400000, "to": 235520000, "AVG_TTFB": 15.178718, "ERRORS": 0, "AVG_DURATION": 60.86835, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1692, "MAX_DURATION": 2182, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4834.542, "COUNT": 1466}, {"from": 235520000, "to": 240640000, "AVG_TTFB": 13.839018, "ERRORS": 18, "AVG_DURATION": 56.20805, "ERRORS_PER_SECOND": 0.003515625, "MIN_DURATION": 4, "ERROR_RATE": 1.2278309, "MIN_TTFB": 1, "MAX_TTFB": 413, "MAX_DURATION": 1161, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4791.5703, "COUNT": 1466}, {"from": 240640000, "to": 245760000, "AVG_TTFB": 32.889725, "ERRORS": 0, "AVG_DURATION": 126.00479, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6287, "MAX_DURATION": 12515, "ELEMENTS_PER_SECOND": 0.28515625, "THROUGHPUT": 4816.7827, "COUNT": 1460}, {"from": 245760000, "to": 250880000, "AVG_TTFB": 26.667578, "ERRORS": 0, "AVG_DURATION": 98.99795, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6702, "MAX_DURATION": 7037, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4889.466, "COUNT": 1462}, {"from": 250880000, "to": 256000000, "AVG_TTFB": 15.033991, "ERRORS": 0, "AVG_DURATION": 66.03535, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 573, "MAX_DURATION": 2030, "ELEMENTS_PER_SECOND": 0.2873047, "THROUGHPUT": 4858.2744, "COUNT": 1471}, {"from": 256000000, "to": 261120000, "AVG_TTFB": 16.046448, "ERRORS": 0, "AVG_DURATION": 73.215164, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1278, "MAX_DURATION": 3553, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4821.874, "COUNT": 1464}, {"from": 261120000, "to": 266240000, "AVG_TTFB": 18.466211, "ERRORS": 0, "AVG_DURATION": 78.2901, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2920, "MAX_DURATION": 3436, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4836.827, "COUNT": 1465}, {"from": 266240000, "to": 271360000, "AVG_TTFB": 14.466939, "ERRORS": 0, "AVG_DURATION": 58.100887, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 754, "MAX_DURATION": 1036, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4895.029, "COUNT": 1467}, {"from": 271360000, "to": 276480000, "AVG_TTFB": 19.35486, "ERRORS": 0, "AVG_DURATION": 87.1482, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5533, "MAX_DURATION": 6199, "ELEMENTS_PER_SECOND": 0.2873047, "THROUGHPUT": 4851.295, "COUNT": 1471}, {"from": 276480000, "to": 281600000, "AVG_TTFB": 20.67213, "ERRORS": 0, "AVG_DURATION": 96.01776, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4050, "MAX_DURATION": 6388, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4824.3467, "COUNT": 1464}, {"from": 281600000, "to": 286720000, "AVG_TTFB": 16.997269, "ERRORS": 0, "AVG_DURATION": 66.415985, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2821, "MAX_DURATION": 2837, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4833.435, "COUNT": 1464}, {"from": 286720000, "to": 291840000, "AVG_TTFB": 16.77347, "ERRORS": 0, "AVG_DURATION": 66.00884, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1193, "MAX_DURATION": 3127, "ELEMENTS_PER_SECOND": 0.28710938, "THROUGHPUT": 4844.983, "COUNT": 1470}, {"from": 291840000, "to": 296960000, "AVG_TTFB": 29.145393, "ERRORS": 0, "AVG_DURATION": 96.90785, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 13451, "MAX_DURATION": 13511, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4894.143, "COUNT": 1465}, {"from": 296960000, "to": 302080000, "AVG_TTFB": 15.446332, "ERRORS": 0, "AVG_DURATION": 60.267662, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1322, "MAX_DURATION": 1337, "ELEMENTS_PER_SECOND": 0.2875, "THROUGHPUT": 4854.921, "COUNT": 1472}, {"from": 302080000, "to": 307200000, "AVG_TTFB": 25.489376, "ERRORS": 0, "AVG_DURATION": 105.22002, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4099, "MAX_DURATION": 9613, "ELEMENTS_PER_SECOND": 0.28496093, "THROUGHPUT": 4820.052, "COUNT": 1459}, {"from": 307200000, "to": 312320000, "AVG_TTFB": 18.640465, "ERRORS": 0, "AVG_DURATION": 80.76282, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4911, "MAX_DURATION": 7279, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4832.239, "COUNT": 1463}, {"from": 312320000, "to": 317440000, "AVG_TTFB": 19.315575, "ERRORS": 23, "AVG_DURATION": 73.6291, "ERRORS_PER_SECOND": 0.0044921874, "MIN_DURATION": 4, "ERROR_RATE": 1.5710382, "MIN_TTFB": 0, "MAX_TTFB": 5724, "MAX_DURATION": 6514, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4825.5347, "COUNT": 1464}, {"from": 317440000, "to": 322560000, "AVG_TTFB": 18.28581, "ERRORS": 0, "AVG_DURATION": 70.465034, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4447, "MAX_DURATION": 4468, "ELEMENTS_PER_SECOND": 0.28769532, "THROUGHPUT": 4849.021, "COUNT": 1473}, {"from": 322560000, "to": 327680000, "AVG_TTFB": 13.871078, "ERRORS": 17, "AVG_DURATION": 56.31446, "ERRORS_PER_SECOND": 0.0033203126, "MIN_DURATION": 4, "ERROR_RATE": 1.159618, "MIN_TTFB": 2, "MAX_TTFB": 360, "MAX_DURATION": 399, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4798.8657, "COUNT": 1466}, {"from": 327680000, "to": 332800000, "AVG_TTFB": 38.35827, "ERRORS": 0, "AVG_DURATION": 143.05833, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5332, "MAX_DURATION": 5874, "ELEMENTS_PER_SECOND": 0.2845703, "THROUGHPUT": 4807.993, "COUNT": 1457}, {"from": 332800000, "to": 337920000, "AVG_TTFB": 18.56587, "ERRORS": 0, "AVG_DURATION": 85.07372, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1772, "MAX_DURATION": 2683, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4835.676, "COUNT": 1465}, {"from": 337920000, "to": 343040000, "AVG_TTFB": 17.893343, "ERRORS": 0, "AVG_DURATION": 80.152855, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 998, "MAX_DURATION": 2193, "ELEMENTS_PER_SECOND": 0.2875, "THROUGHPUT": 4908.7573, "COUNT": 1472}, {"from": 343040000, "to": 348160000, "AVG_TTFB": 13.50068, "ERRORS": 0, "AVG_DURATION": 55.42653, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 63, "MAX_DURATION": 293, "ELEMENTS_PER_SECOND": 0.28710938, "THROUGHPUT": 4841.963, "COUNT": 1470}, {"from": 348160000, "to": 353280000, "AVG_TTFB": 14.30764, "ERRORS": 0, "AVG_DURATION": 56.223057, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1125, "MAX_DURATION": 1197, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4832.64, "COUNT": 1466}, {"from": 353280000, "to": 358400000, "AVG_TTFB": 17.770805, "ERRORS": 0, "AVG_DURATION": 80.30082, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2001, "MAX_DURATION": 2178, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4846.252, "COUNT": 1466}, {"from": 358400000, "to": 363520000, "AVG_TTFB": 16.095497, "ERRORS": 0, "AVG_DURATION": 67.90109, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1553, "MAX_DURATION": 1722, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4894.9277, "COUNT": 1466}, {"from": 363520000, "to": 368640000, "AVG_TTFB": 19.030613, "ERRORS": 0, "AVG_DURATION": 81.838776, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3017, "MAX_DURATION": 3228, "ELEMENTS_PER_SECOND": 0.28710938, "THROUGHPUT": 4846.6235, "COUNT": 1470}, {"from": 368640000, "to": 373760000, "AVG_TTFB": 16.32216, "ERRORS": 0, "AVG_DURATION": 74.066345, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2129, "MAX_DURATION": 2990, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4830.3564, "COUNT": 1462}, {"from": 373760000, "to": 378880000, "AVG_TTFB": 16.66985, "ERRORS": 0, "AVG_DURATION": 88.40518, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 919, "MAX_DURATION": 6189, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4831.707, "COUNT": 1466}, {"from": 378880000, "to": 384000000, "AVG_TTFB": 21.616486, "ERRORS": 0, "AVG_DURATION": 89.36989, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5189, "MAX_DURATION": 6064, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4839.1143, "COUNT": 1468}, {"from": 384000000, "to": 389120000, "AVG_TTFB": 18.965847, "ERRORS": 0, "AVG_DURATION": 102.65505, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1733, "MAX_DURATION": 8880, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4900.4014, "COUNT": 1464}, {"from": 389120000, "to": 394240000, "AVG_TTFB": 25.825462, "ERRORS": 0, "AVG_DURATION": 146.57906, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 10722, "MAX_DURATION": 17862, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4819.0996, "COUNT": 1461}, {"from": 394240000, "to": 399360000, "AVG_TTFB": 37.800274, "ERRORS": 78, "AVG_DURATION": 175.1709, "ERRORS_PER_SECOND": 0.015234375, "MIN_DURATION": 4, "ERROR_RATE": 5.353466, "MIN_TTFB": 1, "MAX_TTFB": 4742, "MAX_DURATION": 12223, "ELEMENTS_PER_SECOND": 0.2845703, "THROUGHPUT": 4517.378, "COUNT": 1457}, {"from": 399360000, "to": 404480000, "AVG_TTFB": 28.843472, "ERRORS": 11, "AVG_DURATION": 113.528366, "ERRORS_PER_SECOND": 0.0021484375, "MIN_DURATION": 14, "ERROR_RATE": 0.7518797, "MIN_TTFB": 1, "MAX_TTFB": 7392, "MAX_DURATION": 9887, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4794.224, "COUNT": 1463}, {"from": 404480000, "to": 409600000, "AVG_TTFB": 14.731791, "ERRORS": 0, "AVG_DURATION": 63.181076, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 765, "MAX_DURATION": 2917, "ELEMENTS_PER_SECOND": 0.28691405, "THROUGHPUT": 4892.078, "COUNT": 1469}, {"from": 409600000, "to": 414720000, "AVG_TTFB": 13.777174, "ERRORS": 12, "AVG_DURATION": 59.528534, "ERRORS_PER_SECOND": 0.00234375, "MIN_DURATION": 4, "ERROR_RATE": 0.8152174, "MIN_TTFB": 2, "MAX_TTFB": 53, "MAX_DURATION": 2452, "ELEMENTS_PER_SECOND": 0.2875, "THROUGHPUT": 4819.4536, "COUNT": 1472}, {"from": 414720000, "to": 419840000, "AVG_TTFB": 21.358799, "ERRORS": 0, "AVG_DURATION": 85.31924, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5259, "MAX_DURATION": 5829, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4836.419, "COUNT": 1466}, {"from": 419840000, "to": 424960000, "AVG_TTFB": 34.405907, "ERRORS": 0, "AVG_DURATION": 164.74313, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 9205, "MAX_DURATION": 23043, "ELEMENTS_PER_SECOND": 0.284375, "THROUGHPUT": 4808.9985, "COUNT": 1456}, {"from": 424960000, "to": 430080000, "AVG_TTFB": 24.215069, "ERRORS": 0, "AVG_DURATION": 112.363014, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6712, "MAX_DURATION": 10144, "ELEMENTS_PER_SECOND": 0.28515625, "THROUGHPUT": 4821.448, "COUNT": 1460}, {"from": 430080000, "to": 435200000, "AVG_TTFB": 19.226223, "ERRORS": 0, "AVG_DURATION": 69.671875, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4258, "MAX_DURATION": 4281, "ELEMENTS_PER_SECOND": 0.2875, "THROUGHPUT": 4903.8193, "COUNT": 1472}, {"from": 435200000, "to": 440320000, "AVG_TTFB": 18.31812, "ERRORS": 0, "AVG_DURATION": 75.57425, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2514, "MAX_DURATION": 3226, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4842.3145, "COUNT": 1468}, {"from": 440320000, "to": 445440000, "AVG_TTFB": 18.546139, "ERRORS": 0, "AVG_DURATION": 83.474365, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1665, "MAX_DURATION": 3605, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4833.596, "COUNT": 1463}, {"from": 445440000, "to": 450560000, "AVG_TTFB": 20.26603, "ERRORS": 0, "AVG_DURATION": 66.51774, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4732, "MAX_DURATION": 4838, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4832.1533, "COUNT": 1466}, {"from": 450560000, "to": 455680000, "AVG_TTFB": 14.552524, "ERRORS": 0, "AVG_DURATION": 71.676674, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 450, "MAX_DURATION": 3893, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4899.1494, "COUNT": 1466}, {"from": 455680000, "to": 460800000, "AVG_TTFB": 14.106513, "ERRORS": 0, "AVG_DURATION": 61.355495, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 193, "MAX_DURATION": 4098, "ELEMENTS_PER_SECOND": 0.2878906, "THROUGHPUT": 4861.3975, "COUNT": 1474}, {"from": 460800000, "to": 465920000, "AVG_TTFB": 18.11217, "ERRORS": 0, "AVG_DURATION": 70.08226, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 16, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2039, "MAX_DURATION": 2065, "ELEMENTS_PER_SECOND": 0.2873047, "THROUGHPUT": 4846.706, "COUNT": 1471}, {"from": 465920000, "to": 471040000, "AVG_TTFB": 14.719262, "ERRORS": 0, "AVG_DURATION": 64.94604, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1682, "MAX_DURATION": 3096, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4826.668, "COUNT": 1464}, {"from": 471040000, "to": 476160000, "AVG_TTFB": 16.225784, "ERRORS": 0, "AVG_DURATION": 80.94475, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 666, "MAX_DURATION": 11000, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4833.4927, "COUNT": 1466}, {"from": 476160000, "to": 481280000, "AVG_TTFB": 13.757988, "ERRORS": 0, "AVG_DURATION": 59.352142, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 161, "MAX_DURATION": 882, "ELEMENTS_PER_SECOND": 0.2873047, "THROUGHPUT": 4911.2524, "COUNT": 1471}, {"from": 481280000, "to": 486400000, "AVG_TTFB": 25.928474, "ERRORS": 37, "AVG_DURATION": 92.08174, "ERRORS_PER_SECOND": 0.0072265626, "MIN_DURATION": 4, "ERROR_RATE": 2.520436, "MIN_TTFB": 2, "MAX_TTFB": 6258, "MAX_DURATION": 7838, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4726.0693, "COUNT": 1468}, {"from": 486400000, "to": 491520000, "AVG_TTFB": 13.844475, "ERRORS": 0, "AVG_DURATION": 63.116646, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 110, "MAX_DURATION": 4730, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4838.047, "COUNT": 1466}, {"from": 491520000, "to": 496640000, "AVG_TTFB": 22.06717, "ERRORS": 0, "AVG_DURATION": 84.39822, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2791, "MAX_DURATION": 6536, "ELEMENTS_PER_SECOND": 0.28496093, "THROUGHPUT": 4822.693, "COUNT": 1459}, {"from": 496640000, "to": 501760000, "AVG_TTFB": 15.556996, "ERRORS": 15, "AVG_DURATION": 63.563824, "ERRORS_PER_SECOND": 0.0029296875, "MIN_DURATION": 4, "ERROR_RATE": 1.0238907, "MIN_TTFB": 1, "MAX_TTFB": 2003, "MAX_DURATION": 3218, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4863.895, "COUNT": 1465}, {"from": 501760000, "to": 506880000, "AVG_TTFB": 19.846731, "ERRORS": 0, "AVG_DURATION": 93.485695, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2818, "MAX_DURATION": 10848, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4847.3027, "COUNT": 1468}, {"from": 506880000, "to": 512000000, "AVG_TTFB": 18.61407, "ERRORS": 0, "AVG_DURATION": 90.66393, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1771, "MAX_DURATION": 4788, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4834.0684, "COUNT": 1464}, {"from": 512000000, "to": 517120000, "AVG_TTFB": 17.910458, "ERRORS": 0, "AVG_DURATION": 83.36159, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1157, "MAX_DURATION": 2354, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4826.2144, "COUNT": 1463}, {"from": 517120000, "to": 522240000, "AVG_TTFB": 15.511278, "ERRORS": 0, "AVG_DURATION": 69.9378, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1394, "MAX_DURATION": 2097, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4831.352, "COUNT": 1463}, {"from": 522240000, "to": 527360000, "AVG_TTFB": 26.494888, "ERRORS": 0, "AVG_DURATION": 100.454666, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5927, "MAX_DURATION": 6681, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4896.354, "COUNT": 1467}, {"from": 527360000, "to": 532480000, "AVG_TTFB": 19.565395, "ERRORS": 0, "AVG_DURATION": 91.03406, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3357, "MAX_DURATION": 8330, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4831.1343, "COUNT": 1468}, {"from": 532480000, "to": 537600000, "AVG_TTFB": 17.244036, "ERRORS": 0, "AVG_DURATION": 79.409, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1643, "MAX_DURATION": 2552, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4831.822, "COUNT": 1467}, {"from": 537600000, "to": 542720000, "AVG_TTFB": 17.92901, "ERRORS": 0, "AVG_DURATION": 80.552216, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2596, "MAX_DURATION": 4815, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4833.0986, "COUNT": 1465}, {"from": 542720000, "to": 547840000, "AVG_TTFB": 17.510595, "ERRORS": 0, "AVG_DURATION": 67.94463, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1537, "MAX_DURATION": 2862, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4897.9634, "COUNT": 1463}, {"from": 547840000, "to": 552960000, "AVG_TTFB": 16.653952, "ERRORS": 0, "AVG_DURATION": 76.10627, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1797, "MAX_DURATION": 3195, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4856.584, "COUNT": 1468}, {"from": 552960000, "to": 558080000, "AVG_TTFB": 15.846625, "ERRORS": 0, "AVG_DURATION": 75.144516, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1333, "MAX_DURATION": 7234, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4838.813, "COUNT": 1467}, {"from": 558080000, "to": 563200000, "AVG_TTFB": 30.027435, "ERRORS": 0, "AVG_DURATION": 134.52469, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 7389, "MAX_DURATION": 8233, "ELEMENTS_PER_SECOND": 0.28476563, "THROUGHPUT": 4817.2065, "COUNT": 1458}, {"from": 563200000, "to": 568320000, "AVG_TTFB": 18.148022, "ERRORS": 0, "AVG_DURATION": 83.652115, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 15, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2781, "MAX_DURATION": 6026, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4834.3154, "COUNT": 1466}, {"from": 568320000, "to": 573440000, "AVG_TTFB": 28.66621, "ERRORS": 88, "AVG_DURATION": 138.85841, "ERRORS_PER_SECOND": 0.0171875, "MIN_DURATION": 4, "ERROR_RATE": 6.0191517, "MIN_TTFB": 2, "MAX_TTFB": 5677, "MAX_DURATION": 14580, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4560.5073, "COUNT": 1462}, {"from": 573440000, "to": 578560000, "AVG_TTFB": 15.160678, "ERRORS": 0, "AVG_DURATION": 63.446102, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1218, "MAX_DURATION": 1402, "ELEMENTS_PER_SECOND": 0.28808594, "THROUGHPUT": 4847.0073, "COUNT": 1475}, {"from": 578560000, "to": 583680000, "AVG_TTFB": 14.207508, "ERRORS": 0, "AVG_DURATION": 56.01843, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 980, "MAX_DURATION": 1133, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4835.253, "COUNT": 1465}, {"from": 583680000, "to": 588800000, "AVG_TTFB": 16.143738, "ERRORS": 14, "AVG_DURATION": 74.83436, "ERRORS_PER_SECOND": 0.002734375, "MIN_DURATION": 4, "ERROR_RATE": 0.9582478, "MIN_TTFB": 1, "MAX_TTFB": 1346, "MAX_DURATION": 1960, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4783.3467, "COUNT": 1461}, {"from": 588800000, "to": 593920000, "AVG_TTFB": 15.642127, "ERRORS": 0, "AVG_DURATION": 62.828903, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1601, "MAX_DURATION": 2091, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4907.3105, "COUNT": 1467}, {"from": 593920000, "to": 599040000, "AVG_TTFB": 16.32359, "ERRORS": 0, "AVG_DURATION": 62.979607, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1534, "MAX_DURATION": 1552, "ELEMENTS_PER_SECOND": 0.2873047, "THROUGHPUT": 4846.01, "COUNT": 1471}, {"from": 599040000, "to": 604160000, "AVG_TTFB": 14.7726345, "ERRORS": 0, "AVG_DURATION": 66.415245, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 773, "MAX_DURATION": 4000, "ELEMENTS_PER_SECOND": 0.28691405, "THROUGHPUT": 4846.059, "COUNT": 1469}, {"from": 604160000, "to": 609280000, "AVG_TTFB": 15.877384, "ERRORS": 0, "AVG_DURATION": 62.780655, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1182, "MAX_DURATION": 2007, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4838.114, "COUNT": 1468}, {"from": 609280000, "to": 614400000, "AVG_TTFB": 15.536896, "ERRORS": 0, "AVG_DURATION": 70.346054, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1270, "MAX_DURATION": 3402, "ELEMENTS_PER_SECOND": 0.15351562, "THROUGHPUT": 2624.7065, "COUNT": 786}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/ba912bd5-d4ee-425c-ae82-b56a75c0e6ef/elements/50ddd53f-4a4b-4bf8-a6b3-7a262d3d6cef/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 5120000, "AVG_TTFB": 6.0708447, "ERRORS": 0, "AVG_DURATION": 60.72752, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 189, "MAX_DURATION": 3506, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1660.3157, "COUNT": 367}, {"from": 5120000, "to": 10240000, "AVG_TTFB": 8.068306, "ERRORS": 0, "AVG_DURATION": 59.901638, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 530, "MAX_DURATION": 1737, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 10240000, "to": 15360000, "AVG_TTFB": 3.8555858, "ERRORS": 0, "AVG_DURATION": 77.239784, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 330, "MAX_DURATION": 8294, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 15360000, "to": 20480000, "AVG_TTFB": 11.395096, "ERRORS": 0, "AVG_DURATION": 134.57494, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1625, "MAX_DURATION": 9102, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 20480000, "to": 25600000, "AVG_TTFB": 43.595108, "ERRORS": 0, "AVG_DURATION": 196.55707, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6270, "MAX_DURATION": 10825, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 25600000, "to": 30720000, "AVG_TTFB": 7.6565933, "ERRORS": 0, "AVG_DURATION": 57.953297, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 978, "MAX_DURATION": 5818, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 30720000, "to": 35840000, "AVG_TTFB": 34.923286, "ERRORS": 0, "AVG_DURATION": 106.55068, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 7021, "MAX_DURATION": 11902, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 35840000, "to": 40960000, "AVG_TTFB": 4.1409216, "ERRORS": 0, "AVG_DURATION": 35.596207, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 341, "MAX_DURATION": 476, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1479.5801, "COUNT": 369}, {"from": 40960000, "to": 46080000, "AVG_TTFB": 18.553425, "ERRORS": 0, "AVG_DURATION": 72.8274, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1886, "MAX_DURATION": 2938, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 46080000, "to": 51200000, "AVG_TTFB": 8.494535, "ERRORS": 0, "AVG_DURATION": 41.628414, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1673, "MAX_DURATION": 1688, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 51200000, "to": 56320000, "AVG_TTFB": 3.396175, "ERRORS": 4, "AVG_DURATION": 33.188526, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 4, "ERROR_RATE": 1.0928962, "MIN_TTFB": 2, "MAX_TTFB": 132, "MAX_DURATION": 273, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1408.5883, "COUNT": 366}, {"from": 56320000, "to": 61440000, "AVG_TTFB": 8.027174, "ERRORS": 0, "AVG_DURATION": 45.201088, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1822, "MAX_DURATION": 2276, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 61440000, "to": 66560000, "AVG_TTFB": 10.554945, "ERRORS": 0, "AVG_DURATION": 137.16484, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 589, "MAX_DURATION": 24726, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1471.9824, "COUNT": 364}, {"from": 66560000, "to": 71680000, "AVG_TTFB": 119.20448, "ERRORS": 25, "AVG_DURATION": 325.451, "ERRORS_PER_SECOND": 0.0048828125, "MIN_DURATION": 4, "ERROR_RATE": 7.002801, "MIN_TTFB": 2, "MAX_TTFB": 19981, "MAX_DURATION": 23991, "ELEMENTS_PER_SECOND": 0.069726564, "THROUGHPUT": 1367.6328, "COUNT": 357}, {"from": 71680000, "to": 76800000, "AVG_TTFB": 25.247957, "ERRORS": 0, "AVG_DURATION": 72.64033, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2439, "MAX_DURATION": 2560, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 76800000, "to": 81920000, "AVG_TTFB": 14.669399, "ERRORS": 0, "AVG_DURATION": 53.393444, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1453, "MAX_DURATION": 2053, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 81920000, "to": 87040000, "AVG_TTFB": 3.3451087, "ERRORS": 0, "AVG_DURATION": 37.497284, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 104, "MAX_DURATION": 731, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1478.0605, "COUNT": 368}, {"from": 87040000, "to": 92160000, "AVG_TTFB": 12.15847, "ERRORS": 0, "AVG_DURATION": 52.251366, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1557, "MAX_DURATION": 2482, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 92160000, "to": 97280000, "AVG_TTFB": 8.739726, "ERRORS": 0, "AVG_DURATION": 56.8, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1189, "MAX_DURATION": 2650, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 97280000, "to": 102400000, "AVG_TTFB": 9.350543, "ERRORS": 0, "AVG_DURATION": 40.345108, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2337, "MAX_DURATION": 2410, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 102400000, "to": 107520000, "AVG_TTFB": 5.7786884, "ERRORS": 0, "AVG_DURATION": 38.52186, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 795, "MAX_DURATION": 1183, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 107520000, "to": 112640000, "AVG_TTFB": 10.956404, "ERRORS": 0, "AVG_DURATION": 43.33515, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 976, "MAX_DURATION": 1049, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 112640000, "to": 117760000, "AVG_TTFB": 6.2173915, "ERRORS": 0, "AVG_DURATION": 35.059784, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1066, "MAX_DURATION": 1079, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 117760000, "to": 122880000, "AVG_TTFB": 2.7282608, "ERRORS": 0, "AVG_DURATION": 31.413044, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 7, "MAX_DURATION": 92, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 122880000, "to": 128000000, "AVG_TTFB": 4.6191783, "ERRORS": 0, "AVG_DURATION": 45.32055, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 481, "MAX_DURATION": 1888, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 128000000, "to": 133120000, "AVG_TTFB": 41.417583, "ERRORS": 0, "AVG_DURATION": 178.18132, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4206, "MAX_DURATION": 6173, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1471.9824, "COUNT": 364}, {"from": 133120000, "to": 138240000, "AVG_TTFB": 6.929539, "ERRORS": 0, "AVG_DURATION": 41.571815, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1020, "MAX_DURATION": 1285, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1418.3219, "COUNT": 369}, {"from": 138240000, "to": 143360000, "AVG_TTFB": 13.465754, "ERRORS": 8, "AVG_DURATION": 79.69315, "ERRORS_PER_SECOND": 0.0015625, "MIN_DURATION": 5, "ERROR_RATE": 2.1917808, "MIN_TTFB": 2, "MAX_TTFB": 1945, "MAX_DURATION": 9930, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1401.9808, "COUNT": 365}, {"from": 143360000, "to": 148480000, "AVG_TTFB": 3.0163488, "ERRORS": 0, "AVG_DURATION": 32.378746, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 63, "MAX_DURATION": 161, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 148480000, "to": 153600000, "AVG_TTFB": 4.6757493, "ERRORS": 5, "AVG_DURATION": 34.47139, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 4, "ERROR_RATE": 1.3623978, "MIN_TTFB": 2, "MAX_TTFB": 561, "MAX_DURATION": 631, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1408.7887, "COUNT": 367}, {"from": 153600000, "to": 158720000, "AVG_TTFB": 17.820164, "ERRORS": 0, "AVG_DURATION": 49.468666, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4513, "MAX_DURATION": 4530, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 158720000, "to": 163840000, "AVG_TTFB": 8.553134, "ERRORS": 0, "AVG_DURATION": 47.059944, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1054, "MAX_DURATION": 3206, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 163840000, "to": 168960000, "AVG_TTFB": 4.10137, "ERRORS": 0, "AVG_DURATION": 34.235615, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 276, "MAX_DURATION": 386, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 168960000, "to": 174080000, "AVG_TTFB": 3.6059783, "ERRORS": 0, "AVG_DURATION": 33.752716, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 252, "MAX_DURATION": 618, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 174080000, "to": 179200000, "AVG_TTFB": 11.380435, "ERRORS": 0, "AVG_DURATION": 41.690216, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2564, "MAX_DURATION": 3015, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1478.0605, "COUNT": 368}, {"from": 179200000, "to": 184320000, "AVG_TTFB": 6.215847, "ERRORS": 0, "AVG_DURATION": 45.849728, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 411, "MAX_DURATION": 1834, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 184320000, "to": 189440000, "AVG_TTFB": 2.668493, "ERRORS": 0, "AVG_DURATION": 34.046574, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 8, "MAX_DURATION": 1131, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 189440000, "to": 194560000, "AVG_TTFB": 3.3206522, "ERRORS": 0, "AVG_DURATION": 32.785328, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 123, "MAX_DURATION": 196, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 194560000, "to": 199680000, "AVG_TTFB": 8.861035, "ERRORS": 0, "AVG_DURATION": 50.697548, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1104, "MAX_DURATION": 4870, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 199680000, "to": 204800000, "AVG_TTFB": 17.530054, "ERRORS": 0, "AVG_DURATION": 93.9235, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2637, "MAX_DURATION": 7582, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1475.0215, "COUNT": 366}, {"from": 204800000, "to": 209920000, "AVG_TTFB": 5.2479563, "ERRORS": 0, "AVG_DURATION": 51.67575, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 491, "MAX_DURATION": 3723, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 209920000, "to": 215040000, "AVG_TTFB": 3.2309783, "ERRORS": 0, "AVG_DURATION": 32.538044, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 156, "MAX_DURATION": 170, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 215040000, "to": 220160000, "AVG_TTFB": 4.2561307, "ERRORS": 0, "AVG_DURATION": 45.713898, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 60, "MAX_DURATION": 745, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 220160000, "to": 225280000, "AVG_TTFB": 41.123627, "ERRORS": 0, "AVG_DURATION": 116.92033, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 13329, "MAX_DURATION": 20823, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1471.9824, "COUNT": 364}, {"from": 225280000, "to": 230400000, "AVG_TTFB": 19.183561, "ERRORS": 25, "AVG_DURATION": 85.1863, "ERRORS_PER_SECOND": 0.0048828125, "MIN_DURATION": 4, "ERROR_RATE": 6.849315, "MIN_TTFB": 2, "MAX_TTFB": 1719, "MAX_DURATION": 4217, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1379.7727, "COUNT": 365}, {"from": 230400000, "to": 235520000, "AVG_TTFB": 2.7656677, "ERRORS": 0, "AVG_DURATION": 35.566757, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 8, "MAX_DURATION": 1552, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 235520000, "to": 240640000, "AVG_TTFB": 3.2970028, "ERRORS": 4, "AVG_DURATION": 35.896458, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 6, "ERROR_RATE": 1.0899183, "MIN_TTFB": 1, "MAX_TTFB": 40, "MAX_DURATION": 291, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1410.0934, "COUNT": 367}, {"from": 240640000, "to": 245760000, "AVG_TTFB": 37.97514, "ERRORS": 0, "AVG_DURATION": 161.14365, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4470, "MAX_DURATION": 14355, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1407.6852, "COUNT": 362}, {"from": 245760000, "to": 250880000, "AVG_TTFB": 10.572207, "ERRORS": 0, "AVG_DURATION": 44.14169, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1697, "MAX_DURATION": 2425, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 250880000, "to": 256000000, "AVG_TTFB": 4.160763, "ERRORS": 0, "AVG_DURATION": 35.776566, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 536, "MAX_DURATION": 701, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 256000000, "to": 261120000, "AVG_TTFB": 4.6775956, "ERRORS": 0, "AVG_DURATION": 45.28142, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 716, "MAX_DURATION": 3680, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 261120000, "to": 266240000, "AVG_TTFB": 11.030055, "ERRORS": 0, "AVG_DURATION": 42.598362, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1726, "MAX_DURATION": 1742, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 266240000, "to": 271360000, "AVG_TTFB": 4.9483695, "ERRORS": 0, "AVG_DURATION": 36.203804, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 627, "MAX_DURATION": 642, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1478.0605, "COUNT": 368}, {"from": 271360000, "to": 276480000, "AVG_TTFB": 26.427794, "ERRORS": 0, "AVG_DURATION": 137.7793, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4779, "MAX_DURATION": 11914, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 276480000, "to": 281600000, "AVG_TTFB": 27.711538, "ERRORS": 0, "AVG_DURATION": 319.3901, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4161, "MAX_DURATION": 21375, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 281600000, "to": 286720000, "AVG_TTFB": 2.6967213, "ERRORS": 0, "AVG_DURATION": 121.38525, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 6, "MAX_DURATION": 15596, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 286720000, "to": 291840000, "AVG_TTFB": 4.0, "ERRORS": 0, "AVG_DURATION": 33.89918, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 358, "MAX_DURATION": 431, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 291840000, "to": 296960000, "AVG_TTFB": 4.18306, "ERRORS": 0, "AVG_DURATION": 59.898907, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 210, "MAX_DURATION": 7446, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1475.0215, "COUNT": 366}, {"from": 296960000, "to": 302080000, "AVG_TTFB": 4.978202, "ERRORS": 0, "AVG_DURATION": 43.35695, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 381, "MAX_DURATION": 2313, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 302080000, "to": 307200000, "AVG_TTFB": 13.724044, "ERRORS": 0, "AVG_DURATION": 76.830605, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1572, "MAX_DURATION": 6977, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 307200000, "to": 312320000, "AVG_TTFB": 5.365123, "ERRORS": 0, "AVG_DURATION": 34.564034, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 404, "MAX_DURATION": 417, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 312320000, "to": 317440000, "AVG_TTFB": 13.896458, "ERRORS": 10, "AVG_DURATION": 43.3733, "ERRORS_PER_SECOND": 0.001953125, "MIN_DURATION": 4, "ERROR_RATE": 2.7247956, "MIN_TTFB": 2, "MAX_TTFB": 2911, "MAX_DURATION": 2927, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1402.3027, "COUNT": 367}, {"from": 317440000, "to": 322560000, "AVG_TTFB": 4.826087, "ERRORS": 0, "AVG_DURATION": 38.002716, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 458, "MAX_DURATION": 1590, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 322560000, "to": 327680000, "AVG_TTFB": 3.0684931, "ERRORS": 4, "AVG_DURATION": 32.857533, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 4, "ERROR_RATE": 1.0958904, "MIN_TTFB": 2, "MAX_TTFB": 49, "MAX_DURATION": 97, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1407.0398, "COUNT": 365}, {"from": 327680000, "to": 332800000, "AVG_TTFB": 13.824657, "ERRORS": 0, "AVG_DURATION": 69.838356, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 800, "MAX_DURATION": 1920, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 332800000, "to": 337920000, "AVG_TTFB": 5.617886, "ERRORS": 0, "AVG_DURATION": 48.92954, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 470, "MAX_DURATION": 1436, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1418.3219, "COUNT": 369}, {"from": 337920000, "to": 343040000, "AVG_TTFB": 8.827397, "ERRORS": 0, "AVG_DURATION": 60.72603, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 970, "MAX_DURATION": 1870, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1473.502, "COUNT": 365}, {"from": 343040000, "to": 348160000, "AVG_TTFB": 3.4130435, "ERRORS": 0, "AVG_DURATION": 36.434784, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 24, "MAX_DURATION": 166, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 348160000, "to": 353280000, "AVG_TTFB": 3.746594, "ERRORS": 0, "AVG_DURATION": 39.04087, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 112, "MAX_DURATION": 1194, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 353280000, "to": 358400000, "AVG_TTFB": 5.808219, "ERRORS": 0, "AVG_DURATION": 54.38904, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 407, "MAX_DURATION": 1371, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 358400000, "to": 363520000, "AVG_TTFB": 7.1662126, "ERRORS": 0, "AVG_DURATION": 133.31335, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1112, "MAX_DURATION": 14105, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 363520000, "to": 368640000, "AVG_TTFB": 7.521739, "ERRORS": 0, "AVG_DURATION": 234.73642, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1113, "MAX_DURATION": 19228, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 368640000, "to": 373760000, "AVG_TTFB": 17.673077, "ERRORS": 0, "AVG_DURATION": 83.21978, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3718, "MAX_DURATION": 7248, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 373760000, "to": 378880000, "AVG_TTFB": 17.171661, "ERRORS": 0, "AVG_DURATION": 83.9891, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3156, "MAX_DURATION": 8199, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 378880000, "to": 384000000, "AVG_TTFB": 5.2362638, "ERRORS": 0, "AVG_DURATION": 64.49176, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 271, "MAX_DURATION": 8022, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 384000000, "to": 389120000, "AVG_TTFB": 19.299728, "ERRORS": 0, "AVG_DURATION": 78.313354, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2205, "MAX_DURATION": 9040, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 389120000, "to": 394240000, "AVG_TTFB": 22.50137, "ERRORS": 0, "AVG_DURATION": 81.89589, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4245, "MAX_DURATION": 6178, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 394240000, "to": 399360000, "AVG_TTFB": 27.881868, "ERRORS": 30, "AVG_DURATION": 108.13187, "ERRORS_PER_SECOND": 0.005859375, "MIN_DURATION": 4, "ERROR_RATE": 8.241758, "MIN_TTFB": 2, "MAX_TTFB": 2855, "MAX_DURATION": 4005, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1310.5192, "COUNT": 364}, {"from": 399360000, "to": 404480000, "AVG_TTFB": 6.844262, "ERRORS": 0, "AVG_DURATION": 51.674862, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 478, "MAX_DURATION": 1959, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 404480000, "to": 409600000, "AVG_TTFB": 3.6239781, "ERRORS": 0, "AVG_DURATION": 34.294277, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 130, "MAX_DURATION": 263, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 409600000, "to": 414720000, "AVG_TTFB": 2.826087, "ERRORS": 5, "AVG_DURATION": 32.220108, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 4, "ERROR_RATE": 1.3586956, "MIN_TTFB": 2, "MAX_TTFB": 9, "MAX_DURATION": 292, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1410.3226, "COUNT": 368}, {"from": 414720000, "to": 419840000, "AVG_TTFB": 7.479564, "ERRORS": 0, "AVG_DURATION": 38.106266, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 933, "MAX_DURATION": 954, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 419840000, "to": 424960000, "AVG_TTFB": 11.605479, "ERRORS": 0, "AVG_DURATION": 73.43014, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1853, "MAX_DURATION": 4938, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 424960000, "to": 430080000, "AVG_TTFB": 37.480873, "ERRORS": 0, "AVG_DURATION": 84.31421, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 8043, "MAX_DURATION": 9323, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 430080000, "to": 435200000, "AVG_TTFB": 3.6448088, "ERRORS": 0, "AVG_DURATION": 36.046448, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 269, "MAX_DURATION": 1071, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1475.0215, "COUNT": 366}, {"from": 435200000, "to": 440320000, "AVG_TTFB": 3.1032608, "ERRORS": 0, "AVG_DURATION": 47.51902, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 48, "MAX_DURATION": 2024, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 440320000, "to": 445440000, "AVG_TTFB": 3.4153006, "ERRORS": 0, "AVG_DURATION": 37.699455, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 88, "MAX_DURATION": 1025, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 445440000, "to": 450560000, "AVG_TTFB": 10.585831, "ERRORS": 0, "AVG_DURATION": 142.70027, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2840, "MAX_DURATION": 7854, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 450560000, "to": 455680000, "AVG_TTFB": 8.789618, "ERRORS": 0, "AVG_DURATION": 162.53279, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1791, "MAX_DURATION": 10725, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1475.0215, "COUNT": 366}, {"from": 455680000, "to": 460800000, "AVG_TTFB": 3.6212535, "ERRORS": 0, "AVG_DURATION": 148.64578, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 202, "MAX_DURATION": 21895, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 460800000, "to": 465920000, "AVG_TTFB": 3.8387978, "ERRORS": 0, "AVG_DURATION": 37.797813, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 149, "MAX_DURATION": 646, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 465920000, "to": 471040000, "AVG_TTFB": 29.618528, "ERRORS": 0, "AVG_DURATION": 151.89919, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 9875, "MAX_DURATION": 27662, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 471040000, "to": 476160000, "AVG_TTFB": 36.07357, "ERRORS": 0, "AVG_DURATION": 76.43597, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 7048, "MAX_DURATION": 7064, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 476160000, "to": 481280000, "AVG_TTFB": 3.9755435, "ERRORS": 0, "AVG_DURATION": 33.470108, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 343, "MAX_DURATION": 358, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1478.0605, "COUNT": 368}, {"from": 481280000, "to": 486400000, "AVG_TTFB": 5.0684934, "ERRORS": 14, "AVG_DURATION": 39.649315, "ERRORS_PER_SECOND": 0.002734375, "MIN_DURATION": 5, "ERROR_RATE": 3.8356164, "MIN_TTFB": 1, "MAX_TTFB": 150, "MAX_DURATION": 433, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1332.818, "COUNT": 365}, {"from": 486400000, "to": 491520000, "AVG_TTFB": 2.8777175, "ERRORS": 0, "AVG_DURATION": 33.798912, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 33, "MAX_DURATION": 479, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 491520000, "to": 496640000, "AVG_TTFB": 3.9313188, "ERRORS": 0, "AVG_DURATION": 47.31044, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 337, "MAX_DURATION": 3901, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 496640000, "to": 501760000, "AVG_TTFB": 5.4579945, "ERRORS": 5, "AVG_DURATION": 40.688347, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 4, "ERROR_RATE": 1.3550136, "MIN_TTFB": 2, "MAX_TTFB": 356, "MAX_DURATION": 2073, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1473.168, "COUNT": 369}, {"from": 501760000, "to": 506880000, "AVG_TTFB": 5.7520437, "ERRORS": 0, "AVG_DURATION": 42.803814, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 294, "MAX_DURATION": 712, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 506880000, "to": 512000000, "AVG_TTFB": 19.156164, "ERRORS": 0, "AVG_DURATION": 68.652054, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4101, "MAX_DURATION": 4172, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 512000000, "to": 517120000, "AVG_TTFB": 5.9289618, "ERRORS": 0, "AVG_DURATION": 55.204918, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 466, "MAX_DURATION": 1838, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 517120000, "to": 522240000, "AVG_TTFB": 5.4332423, "ERRORS": 0, "AVG_DURATION": 44.065395, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 616, "MAX_DURATION": 1385, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 522240000, "to": 527360000, "AVG_TTFB": 31.769863, "ERRORS": 0, "AVG_DURATION": 166.3452, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5741, "MAX_DURATION": 17571, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1473.502, "COUNT": 365}, {"from": 527360000, "to": 532480000, "AVG_TTFB": 14.373961, "ERRORS": 0, "AVG_DURATION": 383.723, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1003, "MAX_DURATION": 31330, "ELEMENTS_PER_SECOND": 0.07050781, "THROUGHPUT": 1406.1656, "COUNT": 361}, {"from": 532480000, "to": 537600000, "AVG_TTFB": 89.68588, "ERRORS": 0, "AVG_DURATION": 1223.9625, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 26125, "MAX_DURATION": 61205, "ELEMENTS_PER_SECOND": 0.06777344, "THROUGHPUT": 1384.8922, "COUNT": 347}, {"from": 537600000, "to": 542720000, "AVG_TTFB": 76.47853, "ERRORS": 1, "AVG_DURATION": 2443.0183, "ERRORS_PER_SECOND": 0.0001953125, "MIN_DURATION": 13, "ERROR_RATE": 0.30674848, "MIN_TTFB": 2, "MAX_TTFB": 21694, "MAX_DURATION": 300018, "ELEMENTS_PER_SECOND": 0.06367187, "THROUGHPUT": 1352.9489, "COUNT": 326}, {"from": 542720000, "to": 547840000, "AVG_TTFB": 8.845493, "ERRORS": 0, "AVG_DURATION": 2780.1887, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1004, "MAX_DURATION": 55156, "ELEMENTS_PER_SECOND": 0.04550781, "THROUGHPUT": 1211.6656, "COUNT": 233}, {"from": 547840000, "to": 552960000, "AVG_TTFB": 83.15164, "ERRORS": 3, "AVG_DURATION": 5723.086, "ERRORS_PER_SECOND": 0.0005859375, "MIN_DURATION": 14, "ERROR_RATE": 1.2295082, "MIN_TTFB": 2, "MAX_TTFB": 19497, "MAX_DURATION": 347335, "ELEMENTS_PER_SECOND": 0.04765625, "THROUGHPUT": 1228.3307, "COUNT": 244}, {"from": 552960000, "to": 558080000, "AVG_TTFB": 11.622302, "ERRORS": 0, "AVG_DURATION": 366.54675, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1003, "MAX_DURATION": 21078, "ELEMENTS_PER_SECOND": 0.054296874, "THROUGHPUT": 1280.0446, "COUNT": 278}, {"from": 558080000, "to": 563200000, "AVG_TTFB": 15.485714, "ERRORS": 0, "AVG_DURATION": 113.582146, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 552, "MAX_DURATION": 3122, "ELEMENTS_PER_SECOND": 0.0546875, "THROUGHPUT": 1283.0836, "COUNT": 280}, {"from": 563200000, "to": 568320000, "AVG_TTFB": 11.295374, "ERRORS": 0, "AVG_DURATION": 55.049824, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1981, "MAX_DURATION": 1994, "ELEMENTS_PER_SECOND": 0.054882813, "THROUGHPUT": 1284.6031, "COUNT": 281}, {"from": 568320000, "to": 573440000, "AVG_TTFB": 58.405693, "ERRORS": 23, "AVG_DURATION": 198.71175, "ERRORS_PER_SECOND": 0.0044921874, "MIN_DURATION": 5, "ERROR_RATE": 8.185054, "MIN_TTFB": 2, "MAX_TTFB": 7821, "MAX_DURATION": 15599, "ELEMENTS_PER_SECOND": 0.054882813, "THROUGHPUT": 1254.7308, "COUNT": 281}, {"from": 573440000, "to": 578560000, "AVG_TTFB": 5.671378, "ERRORS": 0, "AVG_DURATION": 43.448765, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 787, "MAX_DURATION": 860, "ELEMENTS_PER_SECOND": 0.055273436, "THROUGHPUT": 1287.6422, "COUNT": 283}, {"from": 578560000, "to": 583680000, "AVG_TTFB": 5.5957446, "ERRORS": 0, "AVG_DURATION": 38.446808, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 813, "MAX_DURATION": 826, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 1286.1227, "COUNT": 282}, {"from": 583680000, "to": 588800000, "AVG_TTFB": 4.390071, "ERRORS": 4, "AVG_DURATION": 39.053192, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 4, "ERROR_RATE": 1.4184397, "MIN_TTFB": 2, "MAX_TTFB": 336, "MAX_DURATION": 350, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 1281.0535, "COUNT": 282}, {"from": 588800000, "to": 593920000, "AVG_TTFB": 3.0070672, "ERRORS": 0, "AVG_DURATION": 36.30742, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 51, "MAX_DURATION": 85, "ELEMENTS_PER_SECOND": 0.055273436, "THROUGHPUT": 1348.9004, "COUNT": 283}, {"from": 593920000, "to": 599040000, "AVG_TTFB": 10.237589, "ERRORS": 0, "AVG_DURATION": 45.20922, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1445, "MAX_DURATION": 1460, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 1286.1227, "COUNT": 282}, {"from": 599040000, "to": 604160000, "AVG_TTFB": 3.35461, "ERRORS": 0, "AVG_DURATION": 36.925533, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 63, "MAX_DURATION": 175, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 1286.1227, "COUNT": 282}, {"from": 604160000, "to": 609280000, "AVG_TTFB": 3.6512456, "ERRORS": 0, "AVG_DURATION": 38.548042, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 141, "MAX_DURATION": 723, "ELEMENTS_PER_SECOND": 0.054882813, "THROUGHPUT": 1284.6031, "COUNT": 281}, {"from": 609280000, "to": 614400000, "AVG_TTFB": 2.7417219, "ERRORS": 0, "AVG_DURATION": 34.9404, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 14, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 7, "MAX_DURATION": 84, "ELEMENTS_PER_SECOND": 0.029492188, "THROUGHPUT": 719.51483, "COUNT": 151}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/ba912bd5-d4ee-425c-ae82-b56a75c0e6ef/elements/b3e1f7c6-a4ef-438e-a3bb-8ffa1cb4a85e/values':
            return '{"count": 42438, "elementPerSecond": 0.06934196, "minDuration": 43, "maxDuration": 532, "sumDuration": 1912308, "avgDuration": 45.06122, "minTTFB": 43, "maxTTFB": 532, "sumTTFB": 1912308, "avgTTFB": 45.06122, "sumDownloadedBytes": 0, "downloadedBytesPerSecond": 0.0, "successCount": 42438, "successPerSecond": 0.06934196, "successRate": 100.0, "failureCount": 0, "failurePerSecond": 0.0, "failureRate": 0.0, "percentile50": 45.0, "percentile90": 46.0, "percentile95": 46.0, "percentile99": 53.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/ba912bd5-d4ee-425c-ae82-b56a75c0e6ef/elements/062b066f-e7c0-4e67-9a08-65d6494fecc9/values':
            return '{"count": 42460, "elementPerSecond": 0.06937791, "minDuration": 12, "maxDuration": 349267, "sumDuration": 11198377, "avgDuration": 263.73944, "minTTFB": 1, "maxTTFB": 29896, "sumTTFB": 591210, "avgTTFB": 13.923928, "sumDownloadedBytes": 1008617043, "downloadedBytesPerSecond": 1648.0391, "successCount": 42273, "successPerSecond": 0.06907236, "successRate": 99.559586, "failureCount": 187, "failurePerSecond": 0.00030555035, "failureRate": 0.44041452, "percentile50": 73.0, "percentile90": 222.0, "percentile95": 284.0, "percentile99": 2191.5}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/ba912bd5-d4ee-425c-ae82-b56a75c0e6ef/elements/0ca05bf2-4ba9-483e-9041-c9ac38ecb8b8/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 5120000, "AVG_TTFB": 14.890411, "ERRORS": 0, "AVG_DURATION": 75.10959, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 25, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1079, "MAX_DURATION": 1508, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1711.9056, "COUNT": 365}, {"from": 5120000, "to": 10240000, "AVG_TTFB": 11.65847, "ERRORS": 0, "AVG_DURATION": 81.28689, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 481, "MAX_DURATION": 1272, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1715.4281, "COUNT": 366}, {"from": 10240000, "to": 15360000, "AVG_TTFB": 5.551913, "ERRORS": 0, "AVG_DURATION": 49.07377, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 234, "MAX_DURATION": 636, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1714.1057, "COUNT": 366}, {"from": 15360000, "to": 20480000, "AVG_TTFB": 9.318801, "ERRORS": 0, "AVG_DURATION": 76.378746, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 585, "MAX_DURATION": 1416, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1724.8818, "COUNT": 367}, {"from": 20480000, "to": 25600000, "AVG_TTFB": 14.688705, "ERRORS": 0, "AVG_DURATION": 87.523415, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1222, "MAX_DURATION": 2294, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1705.5305, "COUNT": 363}, {"from": 25600000, "to": 30720000, "AVG_TTFB": 22.92896, "ERRORS": 0, "AVG_DURATION": 116.99727, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 4824, "MAX_DURATION": 5234, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1711.7649, "COUNT": 366}, {"from": 30720000, "to": 35840000, "AVG_TTFB": 18.790192, "ERRORS": 0, "AVG_DURATION": 94.69755, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2473, "MAX_DURATION": 8128, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1725.183, "COUNT": 367}, {"from": 35840000, "to": 40960000, "AVG_TTFB": 4.8209367, "ERRORS": 0, "AVG_DURATION": 75.002754, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 71, "MAX_DURATION": 2118, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1704.3871, "COUNT": 363}, {"from": 40960000, "to": 46080000, "AVG_TTFB": 21.298914, "ERRORS": 0, "AVG_DURATION": 103.39402, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2430, "MAX_DURATION": 4193, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1723.0785, "COUNT": 368}, {"from": 46080000, "to": 51200000, "AVG_TTFB": 9.553425, "ERRORS": 0, "AVG_DURATION": 114.42466, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1696, "MAX_DURATION": 4551, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1704.9739, "COUNT": 365}, {"from": 51200000, "to": 56320000, "AVG_TTFB": 6.1141305, "ERRORS": 4, "AVG_DURATION": 49.907608, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 17, "ERROR_RATE": 1.0869565, "MIN_TTFB": 2, "MAX_TTFB": 267, "MAX_DURATION": 1100, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1704.2463, "COUNT": 368}, {"from": 56320000, "to": 61440000, "AVG_TTFB": 5.1945205, "ERRORS": 0, "AVG_DURATION": 48.536987, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 74, "MAX_DURATION": 1431, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1714.401, "COUNT": 365}, {"from": 61440000, "to": 66560000, "AVG_TTFB": 35.384617, "ERRORS": 0, "AVG_DURATION": 184.18956, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1898, "MAX_DURATION": 7624, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1704.7615, "COUNT": 364}, {"from": 66560000, "to": 71680000, "AVG_TTFB": 68.466484, "ERRORS": 23, "AVG_DURATION": 460.7849, "ERRORS_PER_SECOND": 0.0044921874, "MIN_DURATION": 23, "ERROR_RATE": 6.424581, "MIN_TTFB": 2, "MAX_TTFB": 1557, "MAX_DURATION": 8864, "ELEMENTS_PER_SECOND": 0.06992187, "THROUGHPUT": 1582.8951, "COUNT": 358}, {"from": 71680000, "to": 76800000, "AVG_TTFB": 15.682192, "ERRORS": 0, "AVG_DURATION": 130.9863, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1061, "MAX_DURATION": 3192, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1706.3557, "COUNT": 365}, {"from": 76800000, "to": 81920000, "AVG_TTFB": 16.619179, "ERRORS": 0, "AVG_DURATION": 94.23836, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1581, "MAX_DURATION": 2891, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1709.816, "COUNT": 365}, {"from": 81920000, "to": 87040000, "AVG_TTFB": 6.557377, "ERRORS": 0, "AVG_DURATION": 69.489075, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 292, "MAX_DURATION": 1342, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1715.4934, "COUNT": 366}, {"from": 87040000, "to": 92160000, "AVG_TTFB": 9.422764, "ERRORS": 0, "AVG_DURATION": 84.097565, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1448, "MAX_DURATION": 4518, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1727.4963, "COUNT": 369}, {"from": 92160000, "to": 97280000, "AVG_TTFB": 10.506849, "ERRORS": 0, "AVG_DURATION": 94.02192, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 386, "MAX_DURATION": 2683, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1716.0547, "COUNT": 365}, {"from": 97280000, "to": 102400000, "AVG_TTFB": 7.9068494, "ERRORS": 0, "AVG_DURATION": 97.819176, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 463, "MAX_DURATION": 10079, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1709.6111, "COUNT": 365}, {"from": 102400000, "to": 107520000, "AVG_TTFB": 10.119891, "ERRORS": 0, "AVG_DURATION": 63.485012, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 751, "MAX_DURATION": 1895, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1712.7129, "COUNT": 367}, {"from": 107520000, "to": 112640000, "AVG_TTFB": 16.653952, "ERRORS": 0, "AVG_DURATION": 54.91553, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2384, "MAX_DURATION": 2413, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1720.1559, "COUNT": 367}, {"from": 112640000, "to": 117760000, "AVG_TTFB": 4.5776567, "ERRORS": 0, "AVG_DURATION": 44.95368, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 45, "MAX_DURATION": 1067, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1717.0215, "COUNT": 367}, {"from": 117760000, "to": 122880000, "AVG_TTFB": 4.4211955, "ERRORS": 0, "AVG_DURATION": 44.97826, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 9, "MAX_DURATION": 784, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1716.6838, "COUNT": 368}, {"from": 122880000, "to": 128000000, "AVG_TTFB": 9.150273, "ERRORS": 0, "AVG_DURATION": 57.434425, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1527, "MAX_DURATION": 1624, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1717.9662, "COUNT": 366}, {"from": 128000000, "to": 133120000, "AVG_TTFB": 75.65014, "ERRORS": 0, "AVG_DURATION": 291.73828, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5494, "MAX_DURATION": 8462, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1700.175, "COUNT": 363}, {"from": 133120000, "to": 138240000, "AVG_TTFB": 11.032698, "ERRORS": 0, "AVG_DURATION": 65.386925, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1270, "MAX_DURATION": 2858, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1719.9534, "COUNT": 367}, {"from": 138240000, "to": 143360000, "AVG_TTFB": 17.087431, "ERRORS": 10, "AVG_DURATION": 143.847, "ERRORS_PER_SECOND": 0.001953125, "MIN_DURATION": 15, "ERROR_RATE": 2.7322404, "MIN_TTFB": 2, "MAX_TTFB": 2788, "MAX_DURATION": 12945, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1680.9414, "COUNT": 366}, {"from": 143360000, "to": 148480000, "AVG_TTFB": 16.653952, "ERRORS": 0, "AVG_DURATION": 63.220707, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1470, "MAX_DURATION": 2570, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1713.3284, "COUNT": 367}, {"from": 148480000, "to": 153600000, "AVG_TTFB": 9.419178, "ERRORS": 6, "AVG_DURATION": 50.778084, "ERRORS_PER_SECOND": 0.001171875, "MIN_DURATION": 16, "ERROR_RATE": 1.6438357, "MIN_TTFB": 2, "MAX_TTFB": 1008, "MAX_DURATION": 1033, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1693.707, "COUNT": 365}, {"from": 153600000, "to": 158720000, "AVG_TTFB": 7.6748633, "ERRORS": 0, "AVG_DURATION": 73.642075, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 24, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 767, "MAX_DURATION": 2607, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1717.8363, "COUNT": 366}, {"from": 158720000, "to": 163840000, "AVG_TTFB": 9.842391, "ERRORS": 0, "AVG_DURATION": 89.222824, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 610, "MAX_DURATION": 1863, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1717.6781, "COUNT": 368}, {"from": 163840000, "to": 168960000, "AVG_TTFB": 5.271233, "ERRORS": 0, "AVG_DURATION": 54.739727, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 228, "MAX_DURATION": 1217, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1718.2256, "COUNT": 365}, {"from": 168960000, "to": 174080000, "AVG_TTFB": 6.8497267, "ERRORS": 0, "AVG_DURATION": 51.150272, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 837, "MAX_DURATION": 2227, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1714.3793, "COUNT": 366}, {"from": 174080000, "to": 179200000, "AVG_TTFB": 4.4741144, "ERRORS": 0, "AVG_DURATION": 59.912807, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 23, "MAX_DURATION": 3230, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1710.1514, "COUNT": 367}, {"from": 179200000, "to": 184320000, "AVG_TTFB": 33.47814, "ERRORS": 0, "AVG_DURATION": 148.59836, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6870, "MAX_DURATION": 7801, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1715.8151, "COUNT": 366}, {"from": 184320000, "to": 189440000, "AVG_TTFB": 4.902174, "ERRORS": 0, "AVG_DURATION": 66.252716, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 125, "MAX_DURATION": 4673, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1723.3066, "COUNT": 368}, {"from": 189440000, "to": 194560000, "AVG_TTFB": 4.7896175, "ERRORS": 0, "AVG_DURATION": 42.77869, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 12, "MAX_DURATION": 186, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1713.0244, "COUNT": 366}, {"from": 194560000, "to": 199680000, "AVG_TTFB": 4.816438, "ERRORS": 0, "AVG_DURATION": 59.50411, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 86, "MAX_DURATION": 1287, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1705.7445, "COUNT": 365}, {"from": 199680000, "to": 204800000, "AVG_TTFB": 26.733696, "ERRORS": 0, "AVG_DURATION": 164.65761, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1694, "MAX_DURATION": 6720, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1717.2549, "COUNT": 368}, {"from": 204800000, "to": 209920000, "AVG_TTFB": 4.989071, "ERRORS": 0, "AVG_DURATION": 55.117485, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 134, "MAX_DURATION": 1288, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1718.5553, "COUNT": 366}, {"from": 209920000, "to": 215040000, "AVG_TTFB": 4.585831, "ERRORS": 0, "AVG_DURATION": 40.762943, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 17, "MAX_DURATION": 145, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1714.293, "COUNT": 367}, {"from": 215040000, "to": 220160000, "AVG_TTFB": 12.139344, "ERRORS": 0, "AVG_DURATION": 72.65574, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 228, "MAX_DURATION": 993, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1709.5638, "COUNT": 366}, {"from": 220160000, "to": 225280000, "AVG_TTFB": 23.59066, "ERRORS": 0, "AVG_DURATION": 333.77472, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2930, "MAX_DURATION": 41637, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1702.281, "COUNT": 364}, {"from": 225280000, "to": 230400000, "AVG_TTFB": 30.745901, "ERRORS": 25, "AVG_DURATION": 216.4918, "ERRORS_PER_SECOND": 0.0048828125, "MIN_DURATION": 15, "ERROR_RATE": 6.830601, "MIN_TTFB": 2, "MAX_TTFB": 3390, "MAX_DURATION": 6085, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1621.6676, "COUNT": 366}, {"from": 230400000, "to": 235520000, "AVG_TTFB": 6.825613, "ERRORS": 0, "AVG_DURATION": 48.861034, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 469, "MAX_DURATION": 840, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1714.9753, "COUNT": 367}, {"from": 235520000, "to": 240640000, "AVG_TTFB": 5.661202, "ERRORS": 5, "AVG_DURATION": 47.22131, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 15, "ERROR_RATE": 1.3661202, "MIN_TTFB": 2, "MAX_TTFB": 94, "MAX_DURATION": 174, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1696.4125, "COUNT": 366}, {"from": 240640000, "to": 245760000, "AVG_TTFB": 13.233517, "ERRORS": 0, "AVG_DURATION": 146.47803, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1665, "MAX_DURATION": 11871, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1714.9508, "COUNT": 364}, {"from": 245760000, "to": 250880000, "AVG_TTFB": 8.693989, "ERRORS": 0, "AVG_DURATION": 107.91803, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 788, "MAX_DURATION": 7160, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1712.0793, "COUNT": 366}, {"from": 250880000, "to": 256000000, "AVG_TTFB": 4.814714, "ERRORS": 0, "AVG_DURATION": 58.463215, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 27, "MAX_DURATION": 2105, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1726.5111, "COUNT": 367}, {"from": 256000000, "to": 261120000, "AVG_TTFB": 7.9836063, "ERRORS": 0, "AVG_DURATION": 74.70492, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 649, "MAX_DURATION": 2954, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1713.1458, "COUNT": 366}, {"from": 261120000, "to": 266240000, "AVG_TTFB": 11.062841, "ERRORS": 0, "AVG_DURATION": 80.31421, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1259, "MAX_DURATION": 2865, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1714.4127, "COUNT": 366}, {"from": 266240000, "to": 271360000, "AVG_TTFB": 8.947946, "ERRORS": 0, "AVG_DURATION": 49.61096, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1398, "MAX_DURATION": 1423, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1708.8323, "COUNT": 365}, {"from": 271360000, "to": 276480000, "AVG_TTFB": 5.6287265, "ERRORS": 0, "AVG_DURATION": 63.1355, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 130, "MAX_DURATION": 6111, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1725.1628, "COUNT": 369}, {"from": 276480000, "to": 281600000, "AVG_TTFB": 21.396694, "ERRORS": 0, "AVG_DURATION": 276.719, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 3582, "MAX_DURATION": 18241, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1700.3516, "COUNT": 363}, {"from": 281600000, "to": 286720000, "AVG_TTFB": 10.631147, "ERRORS": 0, "AVG_DURATION": 77.53278, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2041, "MAX_DURATION": 2275, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1715.1537, "COUNT": 366}, {"from": 286720000, "to": 291840000, "AVG_TTFB": 4.942779, "ERRORS": 0, "AVG_DURATION": 55.509537, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 40, "MAX_DURATION": 1570, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1712.6455, "COUNT": 367}, {"from": 291840000, "to": 296960000, "AVG_TTFB": 10.109289, "ERRORS": 0, "AVG_DURATION": 251.75137, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1107, "MAX_DURATION": 15091, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1711.0319, "COUNT": 366}, {"from": 296960000, "to": 302080000, "AVG_TTFB": 5.53406, "ERRORS": 0, "AVG_DURATION": 43.392372, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 189, "MAX_DURATION": 212, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1720.7185, "COUNT": 367}, {"from": 302080000, "to": 307200000, "AVG_TTFB": 12.816439, "ERRORS": 0, "AVG_DURATION": 126.753426, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 308, "MAX_DURATION": 9315, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1710.4235, "COUNT": 365}, {"from": 307200000, "to": 312320000, "AVG_TTFB": 7.3945208, "ERRORS": 0, "AVG_DURATION": 83.2, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 871, "MAX_DURATION": 4294, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1709.259, "COUNT": 365}, {"from": 312320000, "to": 317440000, "AVG_TTFB": 6.31694, "ERRORS": 11, "AVG_DURATION": 83.49454, "ERRORS_PER_SECOND": 0.0021484375, "MIN_DURATION": 15, "ERROR_RATE": 3.0054646, "MIN_TTFB": 2, "MAX_TTFB": 140, "MAX_DURATION": 6340, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1677.0613, "COUNT": 366}, {"from": 317440000, "to": 322560000, "AVG_TTFB": 6.796748, "ERRORS": 0, "AVG_DURATION": 64.98645, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 692, "MAX_DURATION": 5175, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1725.6764, "COUNT": 369}, {"from": 322560000, "to": 327680000, "AVG_TTFB": 5.2021856, "ERRORS": 7, "AVG_DURATION": 48.532787, "ERRORS_PER_SECOND": 0.0013671875, "MIN_DURATION": 14, "ERROR_RATE": 1.9125683, "MIN_TTFB": 2, "MAX_TTFB": 48, "MAX_DURATION": 1251, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1692.7806, "COUNT": 366}, {"from": 327680000, "to": 332800000, "AVG_TTFB": 23.142466, "ERRORS": 0, "AVG_DURATION": 185.47945, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 25, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 2281, "MAX_DURATION": 5303, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1707.5208, "COUNT": 365}, {"from": 332800000, "to": 337920000, "AVG_TTFB": 12.021917, "ERRORS": 0, "AVG_DURATION": 82.42192, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 24, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1078, "MAX_DURATION": 3024, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1703.1862, "COUNT": 365}, {"from": 337920000, "to": 343040000, "AVG_TTFB": 6.540761, "ERRORS": 0, "AVG_DURATION": 79.33152, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 400, "MAX_DURATION": 2060, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1727.5004, "COUNT": 368}, {"from": 343040000, "to": 348160000, "AVG_TTFB": 5.259563, "ERRORS": 0, "AVG_DURATION": 49.008198, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 74, "MAX_DURATION": 340, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1715.54, "COUNT": 366}, {"from": 348160000, "to": 353280000, "AVG_TTFB": 4.7138963, "ERRORS": 0, "AVG_DURATION": 54.79564, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 25, "MAX_DURATION": 1584, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1714.1443, "COUNT": 367}, {"from": 353280000, "to": 358400000, "AVG_TTFB": 12.172132, "ERRORS": 0, "AVG_DURATION": 83.20765, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 815, "MAX_DURATION": 1280, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1717.0289, "COUNT": 366}, {"from": 358400000, "to": 363520000, "AVG_TTFB": 7.087432, "ERRORS": 0, "AVG_DURATION": 204.59016, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 701, "MAX_DURATION": 26411, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1709.654, "COUNT": 366}, {"from": 363520000, "to": 368640000, "AVG_TTFB": 9.226776, "ERRORS": 0, "AVG_DURATION": 64.13661, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1465, "MAX_DURATION": 1968, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1718.626, "COUNT": 366}, {"from": 368640000, "to": 373760000, "AVG_TTFB": 7.626703, "ERRORS": 0, "AVG_DURATION": 82.65668, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 824, "MAX_DURATION": 4315, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1720.5107, "COUNT": 367}, {"from": 373760000, "to": 378880000, "AVG_TTFB": 8.2774725, "ERRORS": 0, "AVG_DURATION": 86.76923, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 327, "MAX_DURATION": 6078, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1704.1989, "COUNT": 364}, {"from": 378880000, "to": 384000000, "AVG_TTFB": 23.24317, "ERRORS": 0, "AVG_DURATION": 111.991806, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4964, "MAX_DURATION": 6102, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1714.2725, "COUNT": 366}, {"from": 384000000, "to": 389120000, "AVG_TTFB": 36.457767, "ERRORS": 0, "AVG_DURATION": 153.32153, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4713, "MAX_DURATION": 6022, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1716.527, "COUNT": 367}, {"from": 389120000, "to": 394240000, "AVG_TTFB": 16.063187, "ERRORS": 0, "AVG_DURATION": 166.8434, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1477, "MAX_DURATION": 14917, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1714.1621, "COUNT": 364}, {"from": 394240000, "to": 399360000, "AVG_TTFB": 43.436462, "ERRORS": 29, "AVG_DURATION": 195.70442, "ERRORS_PER_SECOND": 0.0056640627, "MIN_DURATION": 22, "ERROR_RATE": 8.011049, "MIN_TTFB": 2, "MAX_TTFB": 2938, "MAX_DURATION": 9069, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1577.0101, "COUNT": 362}, {"from": 399360000, "to": 404480000, "AVG_TTFB": 6.741144, "ERRORS": 0, "AVG_DURATION": 95.87193, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 24, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 191, "MAX_DURATION": 5798, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1715.4603, "COUNT": 367}, {"from": 404480000, "to": 409600000, "AVG_TTFB": 5.678474, "ERRORS": 0, "AVG_DURATION": 49.21526, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 297, "MAX_DURATION": 585, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1725.4089, "COUNT": 367}, {"from": 409600000, "to": 414720000, "AVG_TTFB": 5.0054646, "ERRORS": 5, "AVG_DURATION": 54.691257, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 15, "ERROR_RATE": 1.3661202, "MIN_TTFB": 2, "MAX_TTFB": 11, "MAX_DURATION": 3705, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1684.9408, "COUNT": 366}, {"from": 414720000, "to": 419840000, "AVG_TTFB": 22.779291, "ERRORS": 0, "AVG_DURATION": 78.56131, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 24, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 4137, "MAX_DURATION": 5779, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1718.8024, "COUNT": 367}, {"from": 419840000, "to": 424960000, "AVG_TTFB": 24.484932, "ERRORS": 0, "AVG_DURATION": 203.02466, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3790, "MAX_DURATION": 19609, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1700.8118, "COUNT": 365}, {"from": 424960000, "to": 430080000, "AVG_TTFB": 9.258242, "ERRORS": 0, "AVG_DURATION": 147.82417, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1137, "MAX_DURATION": 10811, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1707.7178, "COUNT": 364}, {"from": 430080000, "to": 435200000, "AVG_TTFB": 14.205962, "ERRORS": 0, "AVG_DURATION": 53.886177, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3216, "MAX_DURATION": 3237, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1725.1517, "COUNT": 369}, {"from": 435200000, "to": 440320000, "AVG_TTFB": 8.032787, "ERRORS": 0, "AVG_DURATION": 81.80601, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 620, "MAX_DURATION": 1649, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1717.5707, "COUNT": 366}, {"from": 440320000, "to": 445440000, "AVG_TTFB": 6.640327, "ERRORS": 0, "AVG_DURATION": 88.986374, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 140, "MAX_DURATION": 4403, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1711.4401, "COUNT": 367}, {"from": 445440000, "to": 450560000, "AVG_TTFB": 17.386301, "ERRORS": 0, "AVG_DURATION": 90.76164, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 4465, "MAX_DURATION": 4567, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1709.2025, "COUNT": 365}, {"from": 450560000, "to": 455680000, "AVG_TTFB": 9.062841, "ERRORS": 0, "AVG_DURATION": 82.44536, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 4, "MAX_TTFB": 1258, "MAX_DURATION": 3705, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1721.4885, "COUNT": 366}, {"from": 455680000, "to": 460800000, "AVG_TTFB": 5.7744565, "ERRORS": 0, "AVG_DURATION": 54.616848, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 306, "MAX_DURATION": 2047, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1725.8425, "COUNT": 368}, {"from": 460800000, "to": 465920000, "AVG_TTFB": 6.6502733, "ERRORS": 0, "AVG_DURATION": 61.765026, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 536, "MAX_DURATION": 2622, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1711.1472, "COUNT": 366}, {"from": 465920000, "to": 471040000, "AVG_TTFB": 13.89863, "ERRORS": 0, "AVG_DURATION": 76.30411, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1242, "MAX_DURATION": 3030, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1708.3656, "COUNT": 365}, {"from": 471040000, "to": 476160000, "AVG_TTFB": 34.893734, "ERRORS": 0, "AVG_DURATION": 100.9782, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 9614, "MAX_DURATION": 9678, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1718.6481, "COUNT": 367}, {"from": 476160000, "to": 481280000, "AVG_TTFB": 4.639344, "ERRORS": 0, "AVG_DURATION": 46.85792, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 54, "MAX_DURATION": 1164, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1710.4264, "COUNT": 366}, {"from": 481280000, "to": 486400000, "AVG_TTFB": 25.377048, "ERRORS": 14, "AVG_DURATION": 118.275955, "ERRORS_PER_SECOND": 0.002734375, "MIN_DURATION": 19, "ERROR_RATE": 3.8251367, "MIN_TTFB": 3, "MAX_TTFB": 5817, "MAX_DURATION": 6599, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1667.5173, "COUNT": 366}, {"from": 486400000, "to": 491520000, "AVG_TTFB": 5.068306, "ERRORS": 0, "AVG_DURATION": 54.18306, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 24, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 32, "MAX_DURATION": 3768, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1713.9084, "COUNT": 366}, {"from": 491520000, "to": 496640000, "AVG_TTFB": 6.06812, "ERRORS": 0, "AVG_DURATION": 65.20164, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 237, "MAX_DURATION": 4966, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1719.232, "COUNT": 367}, {"from": 496640000, "to": 501760000, "AVG_TTFB": 13.252055, "ERRORS": 5, "AVG_DURATION": 67.01918, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 23, "ERROR_RATE": 1.369863, "MIN_TTFB": 2, "MAX_TTFB": 1918, "MAX_DURATION": 3121, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1687.1455, "COUNT": 365}, {"from": 501760000, "to": 506880000, "AVG_TTFB": 5.694823, "ERRORS": 0, "AVG_DURATION": 72.83651, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 24, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 103, "MAX_DURATION": 3324, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1720.7852, "COUNT": 367}, {"from": 506880000, "to": 512000000, "AVG_TTFB": 8.932249, "ERRORS": 0, "AVG_DURATION": 74.230354, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1329, "MAX_DURATION": 2520, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1728.2601, "COUNT": 369}, {"from": 512000000, "to": 517120000, "AVG_TTFB": 8.575342, "ERRORS": 0, "AVG_DURATION": 75.67945, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1129, "MAX_DURATION": 1442, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1708.2037, "COUNT": 365}, {"from": 517120000, "to": 522240000, "AVG_TTFB": 5.4505496, "ERRORS": 0, "AVG_DURATION": 55.953297, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 179, "MAX_DURATION": 1219, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1718.692, "COUNT": 364}, {"from": 522240000, "to": 527360000, "AVG_TTFB": 21.419619, "ERRORS": 0, "AVG_DURATION": 121.57766, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1775, "MAX_DURATION": 6866, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1713.0426, "COUNT": 367}, {"from": 527360000, "to": 532480000, "AVG_TTFB": 26.077778, "ERRORS": 0, "AVG_DURATION": 235.80556, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1068, "MAX_DURATION": 24500, "ELEMENTS_PER_SECOND": 0.0703125, "THROUGHPUT": 1691.0901, "COUNT": 360}, {"from": 532480000, "to": 537600000, "AVG_TTFB": 13.962428, "ERRORS": 0, "AVG_DURATION": 819.1272, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1381, "MAX_DURATION": 30014, "ELEMENTS_PER_SECOND": 0.06757812, "THROUGHPUT": 1615.3519, "COUNT": 346}, {"from": 537600000, "to": 542720000, "AVG_TTFB": 8.839009, "ERRORS": 1, "AVG_DURATION": 2277.5696, "ERRORS_PER_SECOND": 0.0001953125, "MIN_DURATION": 21, "ERROR_RATE": 0.30959752, "MIN_TTFB": 2, "MAX_TTFB": 687, "MAX_DURATION": 300103, "ELEMENTS_PER_SECOND": 0.063085936, "THROUGHPUT": 1510.764, "COUNT": 323}, {"from": 542720000, "to": 547840000, "AVG_TTFB": 9.215859, "ERRORS": 1, "AVG_DURATION": 3338.599, "ERRORS_PER_SECOND": 0.0001953125, "MIN_DURATION": 23, "ERROR_RATE": 0.44052863, "MIN_TTFB": 2, "MAX_TTFB": 1005, "MAX_DURATION": 300028, "ELEMENTS_PER_SECOND": 0.04433594, "THROUGHPUT": 1062.1433, "COUNT": 227}, {"from": 547840000, "to": 552960000, "AVG_TTFB": 224.33472, "ERRORS": 1, "AVG_DURATION": 3172.4792, "ERRORS_PER_SECOND": 0.0001953125, "MIN_DURATION": 22, "ERROR_RATE": 0.41322315, "MIN_TTFB": 3, "MAX_TTFB": 27974, "MAX_DURATION": 300031, "ELEMENTS_PER_SECOND": 0.047265626, "THROUGHPUT": 1135.5621, "COUNT": 242}, {"from": 552960000, "to": 558080000, "AVG_TTFB": 8.730216, "ERRORS": 0, "AVG_DURATION": 324.17267, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1007, "MAX_DURATION": 24201, "ELEMENTS_PER_SECOND": 0.054296874, "THROUGHPUT": 1300.1738, "COUNT": 278}, {"from": 558080000, "to": 563200000, "AVG_TTFB": 18.357143, "ERRORS": 0, "AVG_DURATION": 201.08928, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 680, "MAX_DURATION": 4901, "ELEMENTS_PER_SECOND": 0.0546875, "THROUGHPUT": 1307.9392, "COUNT": 280}, {"from": 563200000, "to": 568320000, "AVG_TTFB": 13.234042, "ERRORS": 0, "AVG_DURATION": 92.03191, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1160, "MAX_DURATION": 3682, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 1324.8197, "COUNT": 282}, {"from": 568320000, "to": 573440000, "AVG_TTFB": 36.772243, "ERRORS": 21, "AVG_DURATION": 224.1993, "ERRORS_PER_SECOND": 0.0041015623, "MIN_DURATION": 21, "ERROR_RATE": 7.4733095, "MIN_TTFB": 2, "MAX_TTFB": 5641, "MAX_DURATION": 12855, "ELEMENTS_PER_SECOND": 0.054882813, "THROUGHPUT": 1231.4827, "COUNT": 281}, {"from": 573440000, "to": 578560000, "AVG_TTFB": 5.712766, "ERRORS": 0, "AVG_DURATION": 55.234043, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 176, "MAX_DURATION": 1016, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 1323.6904, "COUNT": 282}, {"from": 578560000, "to": 583680000, "AVG_TTFB": 4.647687, "ERRORS": 0, "AVG_DURATION": 45.46975, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 8, "MAX_DURATION": 146, "ELEMENTS_PER_SECOND": 0.054882813, "THROUGHPUT": 1316.917, "COUNT": 281}, {"from": 583680000, "to": 588800000, "AVG_TTFB": 15.056738, "ERRORS": 5, "AVG_DURATION": 93.163124, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 15, "ERROR_RATE": 1.7730496, "MIN_TTFB": 2, "MAX_TTFB": 987, "MAX_DURATION": 2153, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 1304.9235, "COUNT": 282}, {"from": 588800000, "to": 593920000, "AVG_TTFB": 4.7588654, "ERRORS": 0, "AVG_DURATION": 49.975178, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 13, "MAX_DURATION": 327, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 1320.6279, "COUNT": 282}, {"from": 593920000, "to": 599040000, "AVG_TTFB": 4.87234, "ERRORS": 0, "AVG_DURATION": 64.358154, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 41, "MAX_DURATION": 1451, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 1319.8715, "COUNT": 282}, {"from": 599040000, "to": 604160000, "AVG_TTFB": 17.411348, "ERRORS": 0, "AVG_DURATION": 80.99645, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3196, "MAX_DURATION": 3217, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 1319.3308, "COUNT": 282}, {"from": 604160000, "to": 609280000, "AVG_TTFB": 5.286219, "ERRORS": 0, "AVG_DURATION": 59.530037, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 218, "MAX_DURATION": 1074, "ELEMENTS_PER_SECOND": 0.055273436, "THROUGHPUT": 1328.4166, "COUNT": 283}, {"from": 609280000, "to": 614400000, "AVG_TTFB": 8.013333, "ERRORS": 0, "AVG_DURATION": 57.593334, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 547, "MAX_DURATION": 1156, "ELEMENTS_PER_SECOND": 0.029296875, "THROUGHPUT": 704.9676, "COUNT": 150}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/71e88176-3353-4bcd-941a-284533099b43/elements/0ca05bf2-4ba9-483e-9041-c9ac38ecb8b8/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 5120000, "AVG_TTFB": 14.953297, "ERRORS": 0, "AVG_DURATION": 88.41209, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 816, "MAX_DURATION": 2984, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1702.3073, "COUNT": 364}, {"from": 5120000, "to": 10240000, "AVG_TTFB": 9.800547, "ERRORS": 0, "AVG_DURATION": 80.37705, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1124, "MAX_DURATION": 1475, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1719.2045, "COUNT": 366}, {"from": 10240000, "to": 15360000, "AVG_TTFB": 7.5737705, "ERRORS": 0, "AVG_DURATION": 68.02732, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 284, "MAX_DURATION": 3394, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1721.4285, "COUNT": 366}, {"from": 15360000, "to": 20480000, "AVG_TTFB": 9.590164, "ERRORS": 0, "AVG_DURATION": 66.62022, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1173, "MAX_DURATION": 2588, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1711.0098, "COUNT": 366}, {"from": 20480000, "to": 25600000, "AVG_TTFB": 26.19891, "ERRORS": 0, "AVG_DURATION": 137.99182, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4449, "MAX_DURATION": 8644, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1724.0149, "COUNT": 367}, {"from": 25600000, "to": 30720000, "AVG_TTFB": 13.355191, "ERRORS": 0, "AVG_DURATION": 94.04372, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2613, "MAX_DURATION": 2752, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1713.3857, "COUNT": 366}, {"from": 30720000, "to": 35840000, "AVG_TTFB": 20.338799, "ERRORS": 0, "AVG_DURATION": 102.31421, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3193, "MAX_DURATION": 3989, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1711.7828, "COUNT": 366}, {"from": 35840000, "to": 40960000, "AVG_TTFB": 7.128415, "ERRORS": 0, "AVG_DURATION": 54.453552, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 380, "MAX_DURATION": 1176, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1719.6055, "COUNT": 366}, {"from": 40960000, "to": 46080000, "AVG_TTFB": 15.793956, "ERRORS": 0, "AVG_DURATION": 128.93956, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1702, "MAX_DURATION": 7278, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1703.3718, "COUNT": 364}, {"from": 46080000, "to": 51200000, "AVG_TTFB": 31.788044, "ERRORS": 0, "AVG_DURATION": 79.92663, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 19, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4649, "MAX_DURATION": 4673, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1717.0247, "COUNT": 368}, {"from": 51200000, "to": 56320000, "AVG_TTFB": 6.02459, "ERRORS": 5, "AVG_DURATION": 49.6612, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 15, "ERROR_RATE": 1.3661202, "MIN_TTFB": 2, "MAX_TTFB": 407, "MAX_DURATION": 767, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1701.125, "COUNT": 366}, {"from": 56320000, "to": 61440000, "AVG_TTFB": 7.855191, "ERRORS": 0, "AVG_DURATION": 54.191257, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1158, "MAX_DURATION": 2690, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1714.6263, "COUNT": 366}, {"from": 61440000, "to": 66560000, "AVG_TTFB": 22.494505, "ERRORS": 0, "AVG_DURATION": 197.28847, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 2574, "MAX_DURATION": 23421, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1705.9304, "COUNT": 364}, {"from": 66560000, "to": 71680000, "AVG_TTFB": 95.62117, "ERRORS": 24, "AVG_DURATION": 433.70752, "ERRORS_PER_SECOND": 0.0046875, "MIN_DURATION": 18, "ERROR_RATE": 6.685237, "MIN_TTFB": 2, "MAX_TTFB": 3022, "MAX_DURATION": 7262, "ELEMENTS_PER_SECOND": 0.07011719, "THROUGHPUT": 1587.9846, "COUNT": 359}, {"from": 71680000, "to": 76800000, "AVG_TTFB": 21.071234, "ERRORS": 0, "AVG_DURATION": 128.38904, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1236, "MAX_DURATION": 3268, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1709.0872, "COUNT": 365}, {"from": 76800000, "to": 81920000, "AVG_TTFB": 10.830601, "ERRORS": 0, "AVG_DURATION": 105.02186, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 873, "MAX_DURATION": 2750, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1709.6707, "COUNT": 366}, {"from": 81920000, "to": 87040000, "AVG_TTFB": 9.087432, "ERRORS": 0, "AVG_DURATION": 62.076504, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 940, "MAX_DURATION": 1488, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1713.0632, "COUNT": 366}, {"from": 87040000, "to": 92160000, "AVG_TTFB": 22.807589, "ERRORS": 0, "AVG_DURATION": 91.47154, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 4168, "MAX_DURATION": 4187, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1733.082, "COUNT": 369}, {"from": 92160000, "to": 97280000, "AVG_TTFB": 8.368853, "ERRORS": 0, "AVG_DURATION": 93.98087, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 360, "MAX_DURATION": 4183, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1711.441, "COUNT": 366}, {"from": 97280000, "to": 102400000, "AVG_TTFB": 8.735695, "ERRORS": 0, "AVG_DURATION": 84.0109, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 19, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 649, "MAX_DURATION": 9702, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1715.2722, "COUNT": 367}, {"from": 102400000, "to": 107520000, "AVG_TTFB": 5.373297, "ERRORS": 0, "AVG_DURATION": 50.18256, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 205, "MAX_DURATION": 955, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1718.5867, "COUNT": 367}, {"from": 107520000, "to": 112640000, "AVG_TTFB": 5.7896175, "ERRORS": 0, "AVG_DURATION": 66.02186, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 563, "MAX_DURATION": 3841, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1725.7346, "COUNT": 366}, {"from": 112640000, "to": 117760000, "AVG_TTFB": 14.411444, "ERRORS": 0, "AVG_DURATION": 61.703, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3715, "MAX_DURATION": 3734, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1724.2395, "COUNT": 367}, {"from": 117760000, "to": 122880000, "AVG_TTFB": 4.360656, "ERRORS": 0, "AVG_DURATION": 41.43989, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 11, "MAX_DURATION": 475, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1721.7494, "COUNT": 366}, {"from": 122880000, "to": 128000000, "AVG_TTFB": 16.460274, "ERRORS": 0, "AVG_DURATION": 72.715065, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 3784, "MAX_DURATION": 3805, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1712.2136, "COUNT": 365}, {"from": 128000000, "to": 133120000, "AVG_TTFB": 63.96409, "ERRORS": 0, "AVG_DURATION": 349.91437, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 4694, "MAX_DURATION": 17518, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1700.7959, "COUNT": 362}, {"from": 133120000, "to": 138240000, "AVG_TTFB": 4.9347825, "ERRORS": 0, "AVG_DURATION": 62.733696, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 222, "MAX_DURATION": 2756, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1726.6508, "COUNT": 368}, {"from": 138240000, "to": 143360000, "AVG_TTFB": 20.153425, "ERRORS": 8, "AVG_DURATION": 140.94246, "ERRORS_PER_SECOND": 0.0015625, "MIN_DURATION": 21, "ERROR_RATE": 2.1917808, "MIN_TTFB": 2, "MAX_TTFB": 3282, "MAX_DURATION": 12874, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1678.0084, "COUNT": 365}, {"from": 143360000, "to": 148480000, "AVG_TTFB": 7.5258856, "ERRORS": 0, "AVG_DURATION": 55.3733, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1008, "MAX_DURATION": 2117, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1709.8811, "COUNT": 367}, {"from": 148480000, "to": 153600000, "AVG_TTFB": 8.590164, "ERRORS": 4, "AVG_DURATION": 48.945354, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 14, "ERROR_RATE": 1.0928962, "MIN_TTFB": 3, "MAX_TTFB": 1211, "MAX_DURATION": 1231, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.279, "COUNT": 366}, {"from": 153600000, "to": 158720000, "AVG_TTFB": 13.024457, "ERRORS": 3, "AVG_DURATION": 77.67663, "ERRORS_PER_SECOND": 0.0005859375, "MIN_DURATION": 24, "ERROR_RATE": 0.8152174, "MIN_TTFB": 2, "MAX_TTFB": 2165, "MAX_DURATION": 5529, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1714.3551, "COUNT": 368}, {"from": 158720000, "to": 163840000, "AVG_TTFB": 16.699453, "ERRORS": 0, "AVG_DURATION": 68.52186, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 895, "MAX_DURATION": 1086, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1719.2527, "COUNT": 366}, {"from": 163840000, "to": 168960000, "AVG_TTFB": 5.7377048, "ERRORS": 0, "AVG_DURATION": 59.786884, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 294, "MAX_DURATION": 1396, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1722.3958, "COUNT": 366}, {"from": 168960000, "to": 174080000, "AVG_TTFB": 4.5122614, "ERRORS": 0, "AVG_DURATION": 63.12534, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 80, "MAX_DURATION": 2973, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1721.2992, "COUNT": 367}, {"from": 174080000, "to": 179200000, "AVG_TTFB": 16.523287, "ERRORS": 0, "AVG_DURATION": 73.44383, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4363, "MAX_DURATION": 4380, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1712.3138, "COUNT": 365}, {"from": 179200000, "to": 184320000, "AVG_TTFB": 15.493188, "ERRORS": 0, "AVG_DURATION": 97.38419, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1000, "MAX_DURATION": 3341, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1723.7053, "COUNT": 367}, {"from": 184320000, "to": 189440000, "AVG_TTFB": 8.888284, "ERRORS": 0, "AVG_DURATION": 61.122616, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 840, "MAX_DURATION": 4101, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1724.585, "COUNT": 367}, {"from": 189440000, "to": 194560000, "AVG_TTFB": 4.7704916, "ERRORS": 0, "AVG_DURATION": 42.265026, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 41, "MAX_DURATION": 189, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1716.2117, "COUNT": 366}, {"from": 194560000, "to": 199680000, "AVG_TTFB": 9.114754, "ERRORS": 0, "AVG_DURATION": 71.65847, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 19, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 860, "MAX_DURATION": 3975, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1723.6785, "COUNT": 366}, {"from": 199680000, "to": 204800000, "AVG_TTFB": 14.715069, "ERRORS": 0, "AVG_DURATION": 104.416435, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1709, "MAX_DURATION": 7918, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1711.2891, "COUNT": 365}, {"from": 204800000, "to": 209920000, "AVG_TTFB": 11.741848, "ERRORS": 0, "AVG_DURATION": 76.652176, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1526, "MAX_DURATION": 6232, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1724.9338, "COUNT": 368}, {"from": 209920000, "to": 215040000, "AVG_TTFB": 4.336066, "ERRORS": 0, "AVG_DURATION": 38.953552, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 16, "MAX_DURATION": 144, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1711.3596, "COUNT": 366}, {"from": 215040000, "to": 220160000, "AVG_TTFB": 13.138965, "ERRORS": 0, "AVG_DURATION": 78.166214, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 502, "MAX_DURATION": 1344, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1717.5641, "COUNT": 367}, {"from": 220160000, "to": 225280000, "AVG_TTFB": 85.789474, "ERRORS": 0, "AVG_DURATION": 214.79224, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 25208, "MAX_DURATION": 25501, "ELEMENTS_PER_SECOND": 0.07050781, "THROUGHPUT": 1693.7932, "COUNT": 361}, {"from": 225280000, "to": 230400000, "AVG_TTFB": 56.14169, "ERRORS": 26, "AVG_DURATION": 147.09264, "ERRORS_PER_SECOND": 0.005078125, "MIN_DURATION": 13, "ERROR_RATE": 7.084469, "MIN_TTFB": 2, "MAX_TTFB": 6733, "MAX_DURATION": 9095, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1617.8031, "COUNT": 367}, {"from": 230400000, "to": 235520000, "AVG_TTFB": 4.904632, "ERRORS": 0, "AVG_DURATION": 44.485012, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 43, "MAX_DURATION": 207, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1718.0918, "COUNT": 367}, {"from": 235520000, "to": 240640000, "AVG_TTFB": 5.7956405, "ERRORS": 6, "AVG_DURATION": 43.06267, "ERRORS_PER_SECOND": 0.001171875, "MIN_DURATION": 13, "ERROR_RATE": 1.6348774, "MIN_TTFB": 2, "MAX_TTFB": 413, "MAX_DURATION": 433, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1703.6455, "COUNT": 367}, {"from": 240640000, "to": 245760000, "AVG_TTFB": 15.926027, "ERRORS": 0, "AVG_DURATION": 95.93973, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1200, "MAX_DURATION": 6217, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1706.694, "COUNT": 365}, {"from": 245760000, "to": 250880000, "AVG_TTFB": 21.824657, "ERRORS": 0, "AVG_DURATION": 137.23288, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6099, "MAX_DURATION": 7037, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1713.4078, "COUNT": 365}, {"from": 250880000, "to": 256000000, "AVG_TTFB": 7.2472825, "ERRORS": 0, "AVG_DURATION": 64.20109, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 512, "MAX_DURATION": 1600, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1727.8375, "COUNT": 368}, {"from": 256000000, "to": 261120000, "AVG_TTFB": 9.773224, "ERRORS": 0, "AVG_DURATION": 75.25683, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1223, "MAX_DURATION": 2949, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1708.3317, "COUNT": 366}, {"from": 261120000, "to": 266240000, "AVG_TTFB": 19.478142, "ERRORS": 0, "AVG_DURATION": 82.483604, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2920, "MAX_DURATION": 3169, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1715.7253, "COUNT": 366}, {"from": 266240000, "to": 271360000, "AVG_TTFB": 7.423497, "ERRORS": 0, "AVG_DURATION": 45.407104, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 754, "MAX_DURATION": 1036, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1712.5957, "COUNT": 366}, {"from": 271360000, "to": 276480000, "AVG_TTFB": 4.4483695, "ERRORS": 0, "AVG_DURATION": 45.008152, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 76, "MAX_DURATION": 1202, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1719.5697, "COUNT": 368}, {"from": 276480000, "to": 281600000, "AVG_TTFB": 19.677595, "ERRORS": 0, "AVG_DURATION": 105.434425, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2738, "MAX_DURATION": 6388, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1710.6119, "COUNT": 366}, {"from": 281600000, "to": 286720000, "AVG_TTFB": 5.6547947, "ERRORS": 0, "AVG_DURATION": 64.28219, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 447, "MAX_DURATION": 2318, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1713.7141, "COUNT": 365}, {"from": 286720000, "to": 291840000, "AVG_TTFB": 9.970109, "ERRORS": 0, "AVG_DURATION": 54.440216, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1193, "MAX_DURATION": 1218, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1719.083, "COUNT": 368}, {"from": 291840000, "to": 296960000, "AVG_TTFB": 12.773224, "ERRORS": 0, "AVG_DURATION": 70.672134, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1628, "MAX_DURATION": 5637, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1713.2828, "COUNT": 366}, {"from": 296960000, "to": 302080000, "AVG_TTFB": 4.339674, "ERRORS": 0, "AVG_DURATION": 44.440216, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 9, "MAX_DURATION": 1007, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1722.8496, "COUNT": 368}, {"from": 302080000, "to": 307200000, "AVG_TTFB": 14.7774725, "ERRORS": 0, "AVG_DURATION": 100.86264, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 789, "MAX_DURATION": 4310, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1707.7441, "COUNT": 364}, {"from": 307200000, "to": 312320000, "AVG_TTFB": 8.434426, "ERRORS": 0, "AVG_DURATION": 61.784153, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1400, "MAX_DURATION": 1869, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1719.99, "COUNT": 366}, {"from": 312320000, "to": 317440000, "AVG_TTFB": 9.2109585, "ERRORS": 8, "AVG_DURATION": 64.01644, "ERRORS_PER_SECOND": 0.0015625, "MIN_DURATION": 14, "ERROR_RATE": 2.1917808, "MIN_TTFB": 2, "MAX_TTFB": 1395, "MAX_DURATION": 1418, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1682.7312, "COUNT": 365}, {"from": 317440000, "to": 322560000, "AVG_TTFB": 4.692935, "ERRORS": 0, "AVG_DURATION": 56.097828, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 37, "MAX_DURATION": 1019, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1717.5515, "COUNT": 368}, {"from": 322560000, "to": 327680000, "AVG_TTFB": 5.599455, "ERRORS": 6, "AVG_DURATION": 44.46049, "ERRORS_PER_SECOND": 0.001171875, "MIN_DURATION": 13, "ERROR_RATE": 1.6348774, "MIN_TTFB": 2, "MAX_TTFB": 360, "MAX_DURATION": 399, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1705.0918, "COUNT": 367}, {"from": 327680000, "to": 332800000, "AVG_TTFB": 24.253443, "ERRORS": 0, "AVG_DURATION": 156.73553, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 2157, "MAX_DURATION": 5511, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1699.4633, "COUNT": 363}, {"from": 332800000, "to": 337920000, "AVG_TTFB": 16.735695, "ERRORS": 0, "AVG_DURATION": 94.84469, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1772, "MAX_DURATION": 1945, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1723.7506, "COUNT": 367}, {"from": 337920000, "to": 343040000, "AVG_TTFB": 6.464674, "ERRORS": 0, "AVG_DURATION": 62.967392, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 207, "MAX_DURATION": 1048, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1714.3951, "COUNT": 368}, {"from": 343040000, "to": 348160000, "AVG_TTFB": 4.504087, "ERRORS": 0, "AVG_DURATION": 41.93188, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 63, "MAX_DURATION": 263, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1716.2799, "COUNT": 367}, {"from": 348160000, "to": 353280000, "AVG_TTFB": 4.4726777, "ERRORS": 0, "AVG_DURATION": 42.967213, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 60, "MAX_DURATION": 519, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1709.9144, "COUNT": 366}, {"from": 353280000, "to": 358400000, "AVG_TTFB": 8.005449, "ERRORS": 0, "AVG_DURATION": 71.18801, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 606, "MAX_DURATION": 2101, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1726.6615, "COUNT": 367}, {"from": 358400000, "to": 363520000, "AVG_TTFB": 8.833333, "ERRORS": 0, "AVG_DURATION": 57.871586, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1553, "MAX_DURATION": 1623, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1713.975, "COUNT": 366}, {"from": 363520000, "to": 368640000, "AVG_TTFB": 15.188011, "ERRORS": 0, "AVG_DURATION": 84.75749, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1965, "MAX_DURATION": 3155, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1716.4069, "COUNT": 367}, {"from": 368640000, "to": 373760000, "AVG_TTFB": 8.367124, "ERRORS": 0, "AVG_DURATION": 82.21918, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 885, "MAX_DURATION": 2990, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1713.552, "COUNT": 365}, {"from": 373760000, "to": 378880000, "AVG_TTFB": 6.8469944, "ERRORS": 0, "AVG_DURATION": 93.63115, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 330, "MAX_DURATION": 5264, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1708.9104, "COUNT": 366}, {"from": 378880000, "to": 384000000, "AVG_TTFB": 14.453804, "ERRORS": 0, "AVG_DURATION": 121.108696, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2481, "MAX_DURATION": 6064, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1719.3386, "COUNT": 368}, {"from": 384000000, "to": 389120000, "AVG_TTFB": 18.155737, "ERRORS": 0, "AVG_DURATION": 134.59016, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1733, "MAX_DURATION": 8880, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1721.0465, "COUNT": 366}, {"from": 389120000, "to": 394240000, "AVG_TTFB": 41.887672, "ERRORS": 0, "AVG_DURATION": 246.37808, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 10722, "MAX_DURATION": 17862, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1706.8812, "COUNT": 365}, {"from": 394240000, "to": 399360000, "AVG_TTFB": 19.06887, "ERRORS": 26, "AVG_DURATION": 100.12948, "ERRORS_PER_SECOND": 0.005078125, "MIN_DURATION": 21, "ERROR_RATE": 7.162534, "MIN_TTFB": 2, "MAX_TTFB": 2451, "MAX_DURATION": 2980, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1597.5186, "COUNT": 363}, {"from": 399360000, "to": 404480000, "AVG_TTFB": 19.32787, "ERRORS": 4, "AVG_DURATION": 89.79235, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 22, "ERROR_RATE": 1.0928962, "MIN_TTFB": 2, "MAX_TTFB": 4169, "MAX_DURATION": 4264, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1693.4769, "COUNT": 366}, {"from": 404480000, "to": 409600000, "AVG_TTFB": 6.455041, "ERRORS": 0, "AVG_DURATION": 62.06267, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 424, "MAX_DURATION": 2917, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1709.8226, "COUNT": 367}, {"from": 409600000, "to": 414720000, "AVG_TTFB": 4.8644986, "ERRORS": 4, "AVG_DURATION": 53.81572, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 14, "ERROR_RATE": 1.0840108, "MIN_TTFB": 2, "MAX_TTFB": 31, "MAX_DURATION": 2452, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1713.2008, "COUNT": 369}, {"from": 414720000, "to": 419840000, "AVG_TTFB": 9.650273, "ERRORS": 0, "AVG_DURATION": 71.62022, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 498, "MAX_DURATION": 4496, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1713.9684, "COUNT": 366}, {"from": 419840000, "to": 424960000, "AVG_TTFB": 30.285715, "ERRORS": 0, "AVG_DURATION": 259.3434, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 3823, "MAX_DURATION": 23043, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1704.7542, "COUNT": 364}, {"from": 424960000, "to": 430080000, "AVG_TTFB": 10.760989, "ERRORS": 0, "AVG_DURATION": 132.67033, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1161, "MAX_DURATION": 8236, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1704.9988, "COUNT": 364}, {"from": 430080000, "to": 435200000, "AVG_TTFB": 19.858696, "ERRORS": 0, "AVG_DURATION": 73.21196, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 4258, "MAX_DURATION": 4281, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1715.5332, "COUNT": 368}, {"from": 435200000, "to": 440320000, "AVG_TTFB": 18.850136, "ERRORS": 0, "AVG_DURATION": 83.7139, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 2514, "MAX_DURATION": 2534, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1716.5619, "COUNT": 367}, {"from": 440320000, "to": 445440000, "AVG_TTFB": 13.101093, "ERRORS": 0, "AVG_DURATION": 120.4153, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1275, "MAX_DURATION": 3605, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1721.6338, "COUNT": 366}, {"from": 445440000, "to": 450560000, "AVG_TTFB": 8.852459, "ERRORS": 0, "AVG_DURATION": 55.31421, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 838, "MAX_DURATION": 3142, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1709.2559, "COUNT": 366}, {"from": 450560000, "to": 455680000, "AVG_TTFB": 4.7021856, "ERRORS": 0, "AVG_DURATION": 78.62295, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 41, "MAX_DURATION": 3893, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1718.2129, "COUNT": 366}, {"from": 455680000, "to": 460800000, "AVG_TTFB": 4.923913, "ERRORS": 0, "AVG_DURATION": 46.008152, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 31, "MAX_DURATION": 1053, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1722.0032, "COUNT": 368}, {"from": 460800000, "to": 465920000, "AVG_TTFB": 7.701897, "ERRORS": 0, "AVG_DURATION": 60.609756, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 756, "MAX_DURATION": 775, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1725.4519, "COUNT": 369}, {"from": 465920000, "to": 471040000, "AVG_TTFB": 4.7780824, "ERRORS": 0, "AVG_DURATION": 61.063015, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 71, "MAX_DURATION": 3096, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1706.8383, "COUNT": 365}, {"from": 471040000, "to": 476160000, "AVG_TTFB": 9.024523, "ERRORS": 0, "AVG_DURATION": 63.079018, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 666, "MAX_DURATION": 2483, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1716.7408, "COUNT": 367}, {"from": 476160000, "to": 481280000, "AVG_TTFB": 4.544959, "ERRORS": 0, "AVG_DURATION": 48.21526, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 31, "MAX_DURATION": 874, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1714.9646, "COUNT": 367}, {"from": 481280000, "to": 486400000, "AVG_TTFB": 24.630434, "ERRORS": 13, "AVG_DURATION": 108.875, "ERRORS_PER_SECOND": 0.0025390624, "MIN_DURATION": 21, "ERROR_RATE": 3.5326087, "MIN_TTFB": 2, "MAX_TTFB": 6258, "MAX_DURATION": 7838, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1672.309, "COUNT": 368}, {"from": 486400000, "to": 491520000, "AVG_TTFB": 5.155738, "ERRORS": 0, "AVG_DURATION": 46.62295, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 4, "MAX_TTFB": 80, "MAX_DURATION": 647, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1715.7408, "COUNT": 366}, {"from": 491520000, "to": 496640000, "AVG_TTFB": 21.39726, "ERRORS": 0, "AVG_DURATION": 81.671234, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 2791, "MAX_DURATION": 4889, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1716.8411, "COUNT": 365}, {"from": 496640000, "to": 501760000, "AVG_TTFB": 5.7786884, "ERRORS": 6, "AVG_DURATION": 51.786884, "ERRORS_PER_SECOND": 0.001171875, "MIN_DURATION": 18, "ERROR_RATE": 1.6393442, "MIN_TTFB": 2, "MAX_TTFB": 208, "MAX_DURATION": 1399, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1706.5808, "COUNT": 366}, {"from": 501760000, "to": 506880000, "AVG_TTFB": 12.166667, "ERRORS": 0, "AVG_DURATION": 94.86612, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1312, "MAX_DURATION": 9465, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1714.3044, "COUNT": 366}, {"from": 506880000, "to": 512000000, "AVG_TTFB": 15.560109, "ERRORS": 0, "AVG_DURATION": 99.47541, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1771, "MAX_DURATION": 4373, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1717.5906, "COUNT": 366}, {"from": 512000000, "to": 517120000, "AVG_TTFB": 7.31694, "ERRORS": 0, "AVG_DURATION": 94.934425, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 501, "MAX_DURATION": 2207, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1714.2029, "COUNT": 366}, {"from": 517120000, "to": 522240000, "AVG_TTFB": 5.0958905, "ERRORS": 0, "AVG_DURATION": 61.20274, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 105, "MAX_DURATION": 1627, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1713.3162, "COUNT": 365}, {"from": 522240000, "to": 527360000, "AVG_TTFB": 24.035423, "ERRORS": 0, "AVG_DURATION": 98.16349, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5927, "MAX_DURATION": 6681, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1712.6106, "COUNT": 367}, {"from": 527360000, "to": 532480000, "AVG_TTFB": 19.904633, "ERRORS": 0, "AVG_DURATION": 99.20436, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 3357, "MAX_DURATION": 8330, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1711.5267, "COUNT": 367}, {"from": 532480000, "to": 537600000, "AVG_TTFB": 10.286103, "ERRORS": 0, "AVG_DURATION": 81.61853, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 950, "MAX_DURATION": 1572, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1710.5299, "COUNT": 367}, {"from": 537600000, "to": 542720000, "AVG_TTFB": 14.480874, "ERRORS": 0, "AVG_DURATION": 86.827866, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1333, "MAX_DURATION": 4815, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1711.6781, "COUNT": 366}, {"from": 542720000, "to": 547840000, "AVG_TTFB": 10.131507, "ERRORS": 0, "AVG_DURATION": 61.909588, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1175, "MAX_DURATION": 2862, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1718.36, "COUNT": 365}, {"from": 547840000, "to": 552960000, "AVG_TTFB": 8.817439, "ERRORS": 0, "AVG_DURATION": 70.79019, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 633, "MAX_DURATION": 2337, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1729.303, "COUNT": 367}, {"from": 552960000, "to": 558080000, "AVG_TTFB": 8.8828335, "ERRORS": 0, "AVG_DURATION": 99.80109, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1333, "MAX_DURATION": 7234, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1720.6449, "COUNT": 367}, {"from": 558080000, "to": 563200000, "AVG_TTFB": 26.642857, "ERRORS": 0, "AVG_DURATION": 193.75275, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1645, "MAX_DURATION": 7633, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1706.5615, "COUNT": 364}, {"from": 563200000, "to": 568320000, "AVG_TTFB": 10.062841, "ERRORS": 0, "AVG_DURATION": 86.54372, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 497, "MAX_DURATION": 5806, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1711.3893, "COUNT": 366}, {"from": 568320000, "to": 573440000, "AVG_TTFB": 20.959017, "ERRORS": 29, "AVG_DURATION": 175.81967, "ERRORS_PER_SECOND": 0.0056640627, "MIN_DURATION": 21, "ERROR_RATE": 7.923497, "MIN_TTFB": 2, "MAX_TTFB": 1853, "MAX_DURATION": 14580, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1603.2664, "COUNT": 366}, {"from": 573440000, "to": 578560000, "AVG_TTFB": 10.612466, "ERRORS": 0, "AVG_DURATION": 59.579945, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1218, "MAX_DURATION": 1402, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1713.9359, "COUNT": 369}, {"from": 578560000, "to": 583680000, "AVG_TTFB": 4.8739724, "ERRORS": 0, "AVG_DURATION": 40.934246, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 143, "MAX_DURATION": 166, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1710.9968, "COUNT": 365}, {"from": 583680000, "to": 588800000, "AVG_TTFB": 10.945355, "ERRORS": 5, "AVG_DURATION": 78.81421, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 16, "ERROR_RATE": 1.3661202, "MIN_TTFB": 2, "MAX_TTFB": 1346, "MAX_DURATION": 1610, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1698.9105, "COUNT": 366}, {"from": 588800000, "to": 593920000, "AVG_TTFB": 10.196721, "ERRORS": 0, "AVG_DURATION": 63.846996, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1601, "MAX_DURATION": 2091, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1718.8767, "COUNT": 366}, {"from": 593920000, "to": 599040000, "AVG_TTFB": 5.550409, "ERRORS": 0, "AVG_DURATION": 49.542233, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 206, "MAX_DURATION": 821, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1717.5808, "COUNT": 367}, {"from": 599040000, "to": 604160000, "AVG_TTFB": 5.8722825, "ERRORS": 0, "AVG_DURATION": 54.622284, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 214, "MAX_DURATION": 1569, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1721.867, "COUNT": 368}, {"from": 604160000, "to": 609280000, "AVG_TTFB": 9.702997, "ERRORS": 0, "AVG_DURATION": 50.05722, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1182, "MAX_DURATION": 2007, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1718.6276, "COUNT": 367}, {"from": 609280000, "to": 614400000, "AVG_TTFB": 4.3654823, "ERRORS": 0, "AVG_DURATION": 62.862946, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 20, "MAX_DURATION": 3402, "ELEMENTS_PER_SECOND": 0.038476564, "THROUGHPUT": 921.2252, "COUNT": 197}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/71e88176-3353-4bcd-941a-284533099b43/elements/062b066f-e7c0-4e67-9a08-65d6494fecc9/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 5120000, "AVG_TTFB": 15.724795, "ERRORS": 0, "AVG_DURATION": 160.11989, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 38, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1941, "MAX_DURATION": 2011, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1711.9988, "COUNT": 367}, {"from": 5120000, "to": 10240000, "AVG_TTFB": 12.879452, "ERRORS": 0, "AVG_DURATION": 173.56165, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 894, "MAX_DURATION": 2634, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.1492, "COUNT": 365}, {"from": 10240000, "to": 15360000, "AVG_TTFB": 4.528767, "ERRORS": 0, "AVG_DURATION": 127.70685, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 28, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 317, "MAX_DURATION": 4255, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.2598, "COUNT": 365}, {"from": 15360000, "to": 20480000, "AVG_TTFB": 4.2119565, "ERRORS": 0, "AVG_DURATION": 135.60054, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 555, "MAX_DURATION": 2312, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1713.3395, "COUNT": 368}, {"from": 20480000, "to": 25600000, "AVG_TTFB": 9.150685, "ERRORS": 0, "AVG_DURATION": 205.15068, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 47, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1704, "MAX_DURATION": 10159, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.1422, "COUNT": 365}, {"from": 25600000, "to": 30720000, "AVG_TTFB": 19.299728, "ERRORS": 0, "AVG_DURATION": 179.9891, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3042, "MAX_DURATION": 5173, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.4261, "COUNT": 367}, {"from": 30720000, "to": 35840000, "AVG_TTFB": 21.739725, "ERRORS": 0, "AVG_DURATION": 222.73698, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 57, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4457, "MAX_DURATION": 6413, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.2191, "COUNT": 365}, {"from": 35840000, "to": 40960000, "AVG_TTFB": 7.307902, "ERRORS": 0, "AVG_DURATION": 120.93188, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 34, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1180, "MAX_DURATION": 1390, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.8035, "COUNT": 367}, {"from": 40960000, "to": 46080000, "AVG_TTFB": 12.346994, "ERRORS": 0, "AVG_DURATION": 317.6202, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 396, "MAX_DURATION": 15555, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.0286, "COUNT": 366}, {"from": 46080000, "to": 51200000, "AVG_TTFB": 29.134247, "ERRORS": 0, "AVG_DURATION": 143.63287, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 36, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2671, "MAX_DURATION": 2733, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.4574, "COUNT": 365}, {"from": 51200000, "to": 56320000, "AVG_TTFB": 19.247957, "ERRORS": 5, "AVG_DURATION": 120.13896, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 13, "ERROR_RATE": 1.3623978, "MIN_TTFB": 2, "MAX_TTFB": 5256, "MAX_DURATION": 5323, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1690.4055, "COUNT": 367}, {"from": 56320000, "to": 61440000, "AVG_TTFB": 3.1498637, "ERRORS": 0, "AVG_DURATION": 115.72207, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6, "MAX_DURATION": 2309, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.3141, "COUNT": 367}, {"from": 61440000, "to": 66560000, "AVG_TTFB": 29.449036, "ERRORS": 0, "AVG_DURATION": 281.00827, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2817, "MAX_DURATION": 10946, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1690.318, "COUNT": 363}, {"from": 66560000, "to": 71680000, "AVG_TTFB": 83.85237, "ERRORS": 26, "AVG_DURATION": 986.5933, "ERRORS_PER_SECOND": 0.005078125, "MIN_DURATION": 27, "ERROR_RATE": 7.2423396, "MIN_TTFB": 1, "MAX_TTFB": 3749, "MAX_DURATION": 37863, "ELEMENTS_PER_SECOND": 0.07011719, "THROUGHPUT": 1577.9453, "COUNT": 359}, {"from": 71680000, "to": 76800000, "AVG_TTFB": 13.144809, "ERRORS": 0, "AVG_DURATION": 185.15573, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1231, "MAX_DURATION": 2656, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.8145, "COUNT": 366}, {"from": 76800000, "to": 81920000, "AVG_TTFB": 13.945355, "ERRORS": 0, "AVG_DURATION": 168.82787, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1504, "MAX_DURATION": 2741, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.864, "COUNT": 366}, {"from": 81920000, "to": 87040000, "AVG_TTFB": 8.51087, "ERRORS": 0, "AVG_DURATION": 142.82336, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 917, "MAX_DURATION": 3190, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1713.2281, "COUNT": 368}, {"from": 87040000, "to": 92160000, "AVG_TTFB": 20.032698, "ERRORS": 0, "AVG_DURATION": 162.28339, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4714, "MAX_DURATION": 4978, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.3375, "COUNT": 367}, {"from": 92160000, "to": 97280000, "AVG_TTFB": 11.021858, "ERRORS": 0, "AVG_DURATION": 148.36612, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 31, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1274, "MAX_DURATION": 1609, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.7191, "COUNT": 366}, {"from": 97280000, "to": 102400000, "AVG_TTFB": 3.0, "ERRORS": 0, "AVG_DURATION": 115.678474, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 57, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 114, "MAX_DURATION": 1432, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.4551, "COUNT": 367}, {"from": 102400000, "to": 107520000, "AVG_TTFB": 9.270492, "ERRORS": 0, "AVG_DURATION": 124.86612, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 40, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 641, "MAX_DURATION": 3212, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.9746, "COUNT": 366}, {"from": 107520000, "to": 112640000, "AVG_TTFB": 2.7240436, "ERRORS": 0, "AVG_DURATION": 118.18033, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 21, "MAX_DURATION": 3908, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1704.1555, "COUNT": 366}, {"from": 112640000, "to": 117760000, "AVG_TTFB": 2.7275205, "ERRORS": 0, "AVG_DURATION": 109.386925, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 36, "MAX_DURATION": 1798, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.6364, "COUNT": 367}, {"from": 117760000, "to": 122880000, "AVG_TTFB": 5.644809, "ERRORS": 0, "AVG_DURATION": 107.97814, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 926, "MAX_DURATION": 1050, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.9446, "COUNT": 366}, {"from": 122880000, "to": 128000000, "AVG_TTFB": 5.365123, "ERRORS": 0, "AVG_DURATION": 131.20436, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 38, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 499, "MAX_DURATION": 3428, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.6598, "COUNT": 367}, {"from": 128000000, "to": 133120000, "AVG_TTFB": 35.44199, "ERRORS": 0, "AVG_DURATION": 446.30386, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 25, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3352, "MAX_DURATION": 17206, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1686.0035, "COUNT": 362}, {"from": 133120000, "to": 138240000, "AVG_TTFB": 17.581522, "ERRORS": 0, "AVG_DURATION": 146.65761, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1963, "MAX_DURATION": 3660, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1712.2504, "COUNT": 368}, {"from": 138240000, "to": 143360000, "AVG_TTFB": 51.219177, "ERRORS": 9, "AVG_DURATION": 226.53151, "ERRORS_PER_SECOND": 0.0017578125, "MIN_DURATION": 28, "ERROR_RATE": 2.4657533, "MIN_TTFB": 2, "MAX_TTFB": 7871, "MAX_DURATION": 11505, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1666.2496, "COUNT": 365}, {"from": 143360000, "to": 148480000, "AVG_TTFB": 4.5380435, "ERRORS": 0, "AVG_DURATION": 114.22011, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 56, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 462, "MAX_DURATION": 2300, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1711.9851, "COUNT": 368}, {"from": 148480000, "to": 153600000, "AVG_TTFB": 2.7111716, "ERRORS": 4, "AVG_DURATION": 119.86104, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 12, "ERROR_RATE": 1.0899183, "MIN_TTFB": 2, "MAX_TTFB": 6, "MAX_DURATION": 3455, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1691.557, "COUNT": 367}, {"from": 153600000, "to": 158720000, "AVG_TTFB": 14.374317, "ERRORS": 0, "AVG_DURATION": 232.03552, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 30, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1568, "MAX_DURATION": 9199, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.4863, "COUNT": 366}, {"from": 158720000, "to": 163840000, "AVG_TTFB": 11.815217, "ERRORS": 0, "AVG_DURATION": 185.17392, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 32, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1369, "MAX_DURATION": 5731, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1712.0348, "COUNT": 368}, {"from": 163840000, "to": 168960000, "AVG_TTFB": 4.230769, "ERRORS": 0, "AVG_DURATION": 120.53571, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 412, "MAX_DURATION": 1789, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.543, "COUNT": 364}, {"from": 168960000, "to": 174080000, "AVG_TTFB": 3.8360655, "ERRORS": 0, "AVG_DURATION": 129.23497, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 393, "MAX_DURATION": 3949, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.6843, "COUNT": 366}, {"from": 174080000, "to": 179200000, "AVG_TTFB": 3.2240436, "ERRORS": 0, "AVG_DURATION": 124.29781, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 188, "MAX_DURATION": 4616, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1704.1113, "COUNT": 366}, {"from": 179200000, "to": 184320000, "AVG_TTFB": 17.875, "ERRORS": 0, "AVG_DURATION": 170.9375, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 25, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2501, "MAX_DURATION": 5299, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1711.934, "COUNT": 368}, {"from": 184320000, "to": 189440000, "AVG_TTFB": 20.172604, "ERRORS": 0, "AVG_DURATION": 144.51233, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 31, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3087, "MAX_DURATION": 3627, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.1965, "COUNT": 365}, {"from": 189440000, "to": 194560000, "AVG_TTFB": 3.1035423, "ERRORS": 0, "AVG_DURATION": 104.89101, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 39, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 48, "MAX_DURATION": 272, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.6442, "COUNT": 367}, {"from": 194560000, "to": 199680000, "AVG_TTFB": 9.852055, "ERRORS": 0, "AVG_DURATION": 135.80821, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1645, "MAX_DURATION": 3293, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.3871, "COUNT": 365}, {"from": 199680000, "to": 204800000, "AVG_TTFB": 25.346048, "ERRORS": 0, "AVG_DURATION": 217.2752, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 49, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2220, "MAX_DURATION": 8276, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.777, "COUNT": 367}, {"from": 204800000, "to": 209920000, "AVG_TTFB": 6.23224, "ERRORS": 0, "AVG_DURATION": 160.32787, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 308, "MAX_DURATION": 5299, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.9207, "COUNT": 366}, {"from": 209920000, "to": 215040000, "AVG_TTFB": 2.8474114, "ERRORS": 0, "AVG_DURATION": 101.03815, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 57, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 80, "MAX_DURATION": 293, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.5703, "COUNT": 367}, {"from": 215040000, "to": 220160000, "AVG_TTFB": 9.13388, "ERRORS": 0, "AVG_DURATION": 160.99727, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 55, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 249, "MAX_DURATION": 1481, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.0707, "COUNT": 366}, {"from": 220160000, "to": 225280000, "AVG_TTFB": 143.44475, "ERRORS": 0, "AVG_DURATION": 395.8177, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 38709, "MAX_DURATION": 39397, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1684.6863, "COUNT": 362}, {"from": 225280000, "to": 230400000, "AVG_TTFB": 24.70924, "ERRORS": 23, "AVG_DURATION": 246.73369, "ERRORS_PER_SECOND": 0.0044921874, "MIN_DURATION": 16, "ERROR_RATE": 6.25, "MIN_TTFB": 2, "MAX_TTFB": 1013, "MAX_DURATION": 5104, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1620.9144, "COUNT": 368}, {"from": 230400000, "to": 235520000, "AVG_TTFB": 3.0191257, "ERRORS": 0, "AVG_DURATION": 115.07377, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5, "MAX_DURATION": 2182, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.6871, "COUNT": 366}, {"from": 235520000, "to": 240640000, "AVG_TTFB": 2.9289618, "ERRORS": 7, "AVG_DURATION": 106.013664, "ERRORS_PER_SECOND": 0.0013671875, "MIN_DURATION": 13, "ERROR_RATE": 1.9125683, "MIN_TTFB": 2, "MAX_TTFB": 17, "MAX_DURATION": 1161, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1680.6492, "COUNT": 366}, {"from": 240640000, "to": 245760000, "AVG_TTFB": 43.282192, "ERRORS": 0, "AVG_DURATION": 220.69041, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 28, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6287, "MAX_DURATION": 8349, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1697.845, "COUNT": 365}, {"from": 245760000, "to": 250880000, "AVG_TTFB": 37.80274, "ERRORS": 0, "AVG_DURATION": 180.84932, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 30, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6702, "MAX_DURATION": 6764, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1699.5172, "COUNT": 365}, {"from": 250880000, "to": 256000000, "AVG_TTFB": 3.707317, "ERRORS": 0, "AVG_DURATION": 116.30081, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 45, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 233, "MAX_DURATION": 2030, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1716.6738, "COUNT": 369}, {"from": 256000000, "to": 261120000, "AVG_TTFB": 2.890411, "ERRORS": 0, "AVG_DURATION": 133.43013, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 44, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 50, "MAX_DURATION": 3553, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.2598, "COUNT": 365}, {"from": 261120000, "to": 266240000, "AVG_TTFB": 3.9673026, "ERRORS": 0, "AVG_DURATION": 149.62671, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 57, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 237, "MAX_DURATION": 3436, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.3386, "COUNT": 367}, {"from": 266240000, "to": 271360000, "AVG_TTFB": 3.9016395, "ERRORS": 0, "AVG_DURATION": 108.090164, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 372, "MAX_DURATION": 701, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.8531, "COUNT": 366}, {"from": 271360000, "to": 276480000, "AVG_TTFB": 4.121951, "ERRORS": 0, "AVG_DURATION": 177.53929, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 412, "MAX_DURATION": 5713, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1717.9617, "COUNT": 369}, {"from": 276480000, "to": 281600000, "AVG_TTFB": 15.389041, "ERRORS": 0, "AVG_DURATION": 193.7863, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 31, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4050, "MAX_DURATION": 6171, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.4519, "COUNT": 365}, {"from": 281600000, "to": 286720000, "AVG_TTFB": 3.1967213, "ERRORS": 0, "AVG_DURATION": 110.931694, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 57, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 159, "MAX_DURATION": 1211, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.9187, "COUNT": 366}, {"from": 286720000, "to": 291840000, "AVG_TTFB": 4.5108695, "ERRORS": 0, "AVG_DURATION": 123.366844, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 209, "MAX_DURATION": 3127, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1712.1367, "COUNT": 368}, {"from": 291840000, "to": 296960000, "AVG_TTFB": 49.265026, "ERRORS": 0, "AVG_DURATION": 231.11476, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 13451, "MAX_DURATION": 13511, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1704.3191, "COUNT": 366}, {"from": 296960000, "to": 302080000, "AVG_TTFB": 4.0298104, "ERRORS": 0, "AVG_DURATION": 112.45799, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 438, "MAX_DURATION": 968, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1716.7883, "COUNT": 369}, {"from": 302080000, "to": 307200000, "AVG_TTFB": 12.939726, "ERRORS": 0, "AVG_DURATION": 178.18082, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 54, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 387, "MAX_DURATION": 9613, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.5442, "COUNT": 365}, {"from": 307200000, "to": 312320000, "AVG_TTFB": 5.350685, "ERRORS": 0, "AVG_DURATION": 164.9178, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 929, "MAX_DURATION": 7279, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.4856, "COUNT": 365}, {"from": 312320000, "to": 317440000, "AVG_TTFB": 19.956404, "ERRORS": 8, "AVG_DURATION": 131.89645, "ERRORS_PER_SECOND": 0.0015625, "MIN_DURATION": 15, "ERROR_RATE": 2.1798365, "MIN_TTFB": 0, "MAX_TTFB": 5724, "MAX_DURATION": 5787, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1675.3265, "COUNT": 367}, {"from": 317440000, "to": 322560000, "AVG_TTFB": 3.3559783, "ERRORS": 0, "AVG_DURATION": 130.21468, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 115, "MAX_DURATION": 4282, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1713.1477, "COUNT": 368}, {"from": 322560000, "to": 327680000, "AVG_TTFB": 3.0844686, "ERRORS": 6, "AVG_DURATION": 104.27248, "ERRORS_PER_SECOND": 0.001171875, "MIN_DURATION": 13, "ERROR_RATE": 1.6348774, "MIN_TTFB": 2, "MAX_TTFB": 12, "MAX_DURATION": 369, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1688.0326, "COUNT": 367}, {"from": 327680000, "to": 332800000, "AVG_TTFB": 61.324177, "ERRORS": 0, "AVG_DURATION": 280.8599, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 32, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5332, "MAX_DURATION": 5874, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.2473, "COUNT": 364}, {"from": 332800000, "to": 337920000, "AVG_TTFB": 8.213698, "ERRORS": 0, "AVG_DURATION": 153.24658, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 39, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 823, "MAX_DURATION": 2529, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.1617, "COUNT": 365}, {"from": 337920000, "to": 343040000, "AVG_TTFB": 8.00271, "ERRORS": 0, "AVG_DURATION": 157.43903, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 923, "MAX_DURATION": 2193, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1717.821, "COUNT": 369}, {"from": 343040000, "to": 348160000, "AVG_TTFB": 2.73297, "ERRORS": 0, "AVG_DURATION": 102.13624, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 57, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 11, "MAX_DURATION": 293, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.361, "COUNT": 367}, {"from": 348160000, "to": 353280000, "AVG_TTFB": 2.989101, "ERRORS": 0, "AVG_DURATION": 102.059944, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 74, "MAX_DURATION": 312, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.443, "COUNT": 367}, {"from": 353280000, "to": 358400000, "AVG_TTFB": 13.991826, "ERRORS": 0, "AVG_DURATION": 164.87738, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 57, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2001, "MAX_DURATION": 2178, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.3469, "COUNT": 367}, {"from": 358400000, "to": 363520000, "AVG_TTFB": 6.31694, "ERRORS": 0, "AVG_DURATION": 128.6448, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 39, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 725, "MAX_DURATION": 1722, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.8922, "COUNT": 366}, {"from": 363520000, "to": 368640000, "AVG_TTFB": 11.644022, "ERRORS": 0, "AVG_DURATION": 154.15489, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3017, "MAX_DURATION": 3228, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1713.4144, "COUNT": 368}, {"from": 368640000, "to": 373760000, "AVG_TTFB": 3.8360655, "ERRORS": 0, "AVG_DURATION": 121.325134, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 96, "MAX_DURATION": 1515, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.0414, "COUNT": 366}, {"from": 373760000, "to": 378880000, "AVG_TTFB": 6.8228884, "ERRORS": 0, "AVG_DURATION": 160.2752, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 54, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 919, "MAX_DURATION": 6189, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.5137, "COUNT": 367}, {"from": 378880000, "to": 384000000, "AVG_TTFB": 21.509537, "ERRORS": 0, "AVG_DURATION": 153.66486, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 50, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5189, "MAX_DURATION": 5256, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.5316, "COUNT": 367}, {"from": 384000000, "to": 389120000, "AVG_TTFB": 10.193989, "ERRORS": 0, "AVG_DURATION": 192.45901, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 30, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1472, "MAX_DURATION": 6163, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1704.3333, "COUNT": 366}, {"from": 389120000, "to": 394240000, "AVG_TTFB": 7.8356166, "ERRORS": 0, "AVG_DURATION": 226.71233, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 398, "MAX_DURATION": 13973, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.4551, "COUNT": 365}, {"from": 394240000, "to": 399360000, "AVG_TTFB": 53.994522, "ERRORS": 26, "AVG_DURATION": 399.36713, "ERRORS_PER_SECOND": 0.005078125, "MIN_DURATION": 16, "ERROR_RATE": 7.1232877, "MIN_TTFB": 2, "MAX_TTFB": 4102, "MAX_DURATION": 12223, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1601.1139, "COUNT": 365}, {"from": 399360000, "to": 404480000, "AVG_TTFB": 26.592896, "ERRORS": 4, "AVG_DURATION": 218.66667, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 58, "ERROR_RATE": 1.0928962, "MIN_TTFB": 2, "MAX_TTFB": 7392, "MAX_DURATION": 7654, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1692.3926, "COUNT": 366}, {"from": 404480000, "to": 409600000, "AVG_TTFB": 5.639344, "ERRORS": 0, "AVG_DURATION": 108.40437, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 24, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 765, "MAX_DURATION": 1269, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.6754, "COUNT": 366}, {"from": 409600000, "to": 414720000, "AVG_TTFB": 3.3369565, "ERRORS": 4, "AVG_DURATION": 106.328804, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 14, "ERROR_RATE": 1.0869565, "MIN_TTFB": 2, "MAX_TTFB": 53, "MAX_DURATION": 1114, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1697.6647, "COUNT": 368}, {"from": 414720000, "to": 419840000, "AVG_TTFB": 26.6594, "ERRORS": 0, "AVG_DURATION": 179.18529, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5259, "MAX_DURATION": 5829, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.1676, "COUNT": 367}, {"from": 419840000, "to": 424960000, "AVG_TTFB": 47.92857, "ERRORS": 0, "AVG_DURATION": 283.6566, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 27, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 9205, "MAX_DURATION": 9473, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.5203, "COUNT": 364}, {"from": 424960000, "to": 430080000, "AVG_TTFB": 13.748634, "ERRORS": 0, "AVG_DURATION": 192.89345, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2834, "MAX_DURATION": 10144, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.6859, "COUNT": 366}, {"from": 430080000, "to": 435200000, "AVG_TTFB": 6.051771, "ERRORS": 0, "AVG_DURATION": 116.3733, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 940, "MAX_DURATION": 3301, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.7058, "COUNT": 367}, {"from": 435200000, "to": 440320000, "AVG_TTFB": 5.388587, "ERRORS": 0, "AVG_DURATION": 137.79076, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 280, "MAX_DURATION": 3226, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1711.9895, "COUNT": 368}, {"from": 440320000, "to": 445440000, "AVG_TTFB": 6.7178082, "ERRORS": 0, "AVG_DURATION": 126.85754, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1014, "MAX_DURATION": 1249, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.1992, "COUNT": 365}, {"from": 445440000, "to": 450560000, "AVG_TTFB": 2.893733, "ERRORS": 0, "AVG_DURATION": 110.6485, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 51, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 66, "MAX_DURATION": 2115, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.6145, "COUNT": 367}, {"from": 450560000, "to": 455680000, "AVG_TTFB": 3.65847, "ERRORS": 0, "AVG_DURATION": 128.26503, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 151, "MAX_DURATION": 3216, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.8757, "COUNT": 366}, {"from": 455680000, "to": 460800000, "AVG_TTFB": 4.3972974, "ERRORS": 0, "AVG_DURATION": 120.17027, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 193, "MAX_DURATION": 4098, "ELEMENTS_PER_SECOND": 0.072265625, "THROUGHPUT": 1722.5922, "COUNT": 370}, {"from": 460800000, "to": 465920000, "AVG_TTFB": 5.5531335, "ERRORS": 0, "AVG_DURATION": 125.711174, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 447, "MAX_DURATION": 1583, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.4906, "COUNT": 367}, {"from": 465920000, "to": 471040000, "AVG_TTFB": 7.5218577, "ERRORS": 0, "AVG_DURATION": 122.0847, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1682, "MAX_DURATION": 1747, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.0273, "COUNT": 366}, {"from": 471040000, "to": 476160000, "AVG_TTFB": 7.046448, "ERRORS": 0, "AVG_DURATION": 169.11476, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 57, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 566, "MAX_DURATION": 11000, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.9886, "COUNT": 366}, {"from": 476160000, "to": 481280000, "AVG_TTFB": 3.308943, "ERRORS": 0, "AVG_DURATION": 110.80217, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 142, "MAX_DURATION": 882, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1718.2273, "COUNT": 369}, {"from": 481280000, "to": 486400000, "AVG_TTFB": 17.270493, "ERRORS": 12, "AVG_DURATION": 157.15846, "ERRORS_PER_SECOND": 0.00234375, "MIN_DURATION": 35, "ERROR_RATE": 3.2786884, "MIN_TTFB": 2, "MAX_TTFB": 4286, "MAX_DURATION": 5785, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1655.582, "COUNT": 366}, {"from": 486400000, "to": 491520000, "AVG_TTFB": 3.2452316, "ERRORS": 0, "AVG_DURATION": 115.88828, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6, "MAX_DURATION": 4681, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.0231, "COUNT": 367}, {"from": 491520000, "to": 496640000, "AVG_TTFB": 15.7774725, "ERRORS": 0, "AVG_DURATION": 165.54945, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2169, "MAX_DURATION": 6536, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.6082, "COUNT": 364}, {"from": 496640000, "to": 501760000, "AVG_TTFB": 4.20765, "ERRORS": 5, "AVG_DURATION": 117.56011, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 14, "ERROR_RATE": 1.3661202, "MIN_TTFB": 2, "MAX_TTFB": 392, "MAX_DURATION": 3218, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1685.9769, "COUNT": 366}, {"from": 501760000, "to": 506880000, "AVG_TTFB": 7.1409216, "ERRORS": 0, "AVG_DURATION": 167.97562, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 44, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 816, "MAX_DURATION": 10848, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1717.7152, "COUNT": 369}, {"from": 506880000, "to": 512000000, "AVG_TTFB": 6.147541, "ERRORS": 0, "AVG_DURATION": 176.57924, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 616, "MAX_DURATION": 4788, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.7145, "COUNT": 366}, {"from": 512000000, "to": 517120000, "AVG_TTFB": 13.109589, "ERRORS": 0, "AVG_DURATION": 152.09041, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1157, "MAX_DURATION": 2354, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.248, "COUNT": 365}, {"from": 517120000, "to": 522240000, "AVG_TTFB": 9.505465, "ERRORS": 0, "AVG_DURATION": 133.78142, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1394, "MAX_DURATION": 2097, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.7532, "COUNT": 366}, {"from": 522240000, "to": 527360000, "AVG_TTFB": 17.40872, "ERRORS": 0, "AVG_DURATION": 174.17711, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3465, "MAX_DURATION": 5409, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.7219, "COUNT": 367}, {"from": 527360000, "to": 532480000, "AVG_TTFB": 7.2459016, "ERRORS": 0, "AVG_DURATION": 169.19945, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 490, "MAX_DURATION": 7237, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.805, "COUNT": 366}, {"from": 532480000, "to": 537600000, "AVG_TTFB": 8.662126, "ERRORS": 0, "AVG_DURATION": 141.47684, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 26, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1643, "MAX_DURATION": 2552, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.5286, "COUNT": 367}, {"from": 537600000, "to": 542720000, "AVG_TTFB": 2.880109, "ERRORS": 0, "AVG_DURATION": 143.15532, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 47, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 13, "MAX_DURATION": 3092, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.6575, "COUNT": 367}, {"from": 542720000, "to": 547840000, "AVG_TTFB": 10.136612, "ERRORS": 0, "AVG_DURATION": 129.01093, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 31, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1537, "MAX_DURATION": 2427, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.0625, "COUNT": 366}, {"from": 547840000, "to": 552960000, "AVG_TTFB": 10.817935, "ERRORS": 0, "AVG_DURATION": 149.17119, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 31, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1797, "MAX_DURATION": 3195, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1713.5176, "COUNT": 368}, {"from": 552960000, "to": 558080000, "AVG_TTFB": 3.9508197, "ERRORS": 0, "AVG_DURATION": 117.70765, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 292, "MAX_DURATION": 1214, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.8851, "COUNT": 366}, {"from": 558080000, "to": 563200000, "AVG_TTFB": 15.775343, "ERRORS": 0, "AVG_DURATION": 192.83014, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 51, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1097, "MAX_DURATION": 5781, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.4011, "COUNT": 365}, {"from": 563200000, "to": 568320000, "AVG_TTFB": 13.405994, "ERRORS": 0, "AVG_DURATION": 159.0981, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2781, "MAX_DURATION": 6026, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.6433, "COUNT": 367}, {"from": 568320000, "to": 573440000, "AVG_TTFB": 30.860273, "ERRORS": 30, "AVG_DURATION": 252.26575, "ERRORS_PER_SECOND": 0.005859375, "MIN_DURATION": 33, "ERROR_RATE": 8.219178, "MIN_TTFB": 2, "MAX_TTFB": 5677, "MAX_DURATION": 13475, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1582.6393, "COUNT": 365}, {"from": 573440000, "to": 578560000, "AVG_TTFB": 3.2601626, "ERRORS": 0, "AVG_DURATION": 117.639565, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 29, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 10, "MAX_DURATION": 1055, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1716.2692, "COUNT": 369}, {"from": 578560000, "to": 583680000, "AVG_TTFB": 5.5585833, "ERRORS": 0, "AVG_DURATION": 107.013626, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 980, "MAX_DURATION": 1133, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.4539, "COUNT": 367}, {"from": 583680000, "to": 588800000, "AVG_TTFB": 5.815934, "ERRORS": 5, "AVG_DURATION": 137.48901, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 14, "ERROR_RATE": 1.3736264, "MIN_TTFB": 2, "MAX_TTFB": 889, "MAX_DURATION": 1960, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1677.3816, "COUNT": 364}, {"from": 588800000, "to": 593920000, "AVG_TTFB": 5.584239, "ERRORS": 0, "AVG_DURATION": 107.59511, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 35, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 596, "MAX_DURATION": 879, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1711.893, "COUNT": 368}, {"from": 593920000, "to": 599040000, "AVG_TTFB": 5.5640326, "ERRORS": 0, "AVG_DURATION": 118.61581, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 57, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 639, "MAX_DURATION": 975, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.5875, "COUNT": 367}, {"from": 599040000, "to": 604160000, "AVG_TTFB": 3.4891305, "ERRORS": 0, "AVG_DURATION": 128.84239, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 27, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 119, "MAX_DURATION": 4000, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1711.9485, "COUNT": 368}, {"from": 604160000, "to": 609280000, "AVG_TTFB": 5.81694, "ERRORS": 0, "AVG_DURATION": 119.05191, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 908, "MAX_DURATION": 1068, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.684, "COUNT": 366}, {"from": 609280000, "to": 614400000, "AVG_TTFB": 3.5939085, "ERRORS": 0, "AVG_DURATION": 135.1472, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 175, "MAX_DURATION": 2609, "ELEMENTS_PER_SECOND": 0.038476564, "THROUGHPUT": 917.10706, "COUNT": 197}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/8725e0cd-92a4-4105-803b-86433851fcfc/statistics':
            return '{"totalRequestCountSuccess": 1829602, "totalRequestCountFailure": 7602, "totalRequestDurationAverage": 18.75853, "totalRequestCountPerSecond": 3.001916, "totalTransactionCountSuccess": 172856, "totalTransactionCountFailure": 1624, "totalTransactionDurationAverage": 156.48524, "totalTransactionCountPerSecond": 0.28509316, "totalIterationCountSuccess": 41421, "totalIterationCountFailure": 542, "totalGlobalDownloadedBytes": 2941165450, "totalGlobalDownloadedBytesPerSecond": 4805.7437, "totalGlobalCountFailure": 7602}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/ba912bd5-d4ee-425c-ae82-b56a75c0e6ef/elements/b3e1f7c6-a4ef-438e-a3bb-8ffa1cb4a85e/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 5120000, "AVG_TTFB": 45.172604, "ERRORS": 0, "AVG_DURATION": 45.172604, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 70, "MAX_DURATION": 70, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 5120000, "to": 10240000, "AVG_TTFB": 45.02732, "ERRORS": 0, "AVG_DURATION": 45.02732, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 10240000, "to": 15360000, "AVG_TTFB": 44.956284, "ERRORS": 0, "AVG_DURATION": 44.956284, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 113, "MAX_DURATION": 113, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 15360000, "to": 20480000, "AVG_TTFB": 44.78474, "ERRORS": 0, "AVG_DURATION": 44.78474, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 20480000, "to": 25600000, "AVG_TTFB": 44.727272, "ERRORS": 0, "AVG_DURATION": 44.727272, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 50, "MAX_DURATION": 50, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 0.0, "COUNT": 363}, {"from": 25600000, "to": 30720000, "AVG_TTFB": 44.759563, "ERRORS": 0, "AVG_DURATION": 44.759563, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 51, "MAX_DURATION": 51, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 30720000, "to": 35840000, "AVG_TTFB": 44.656677, "ERRORS": 0, "AVG_DURATION": 44.656677, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 53, "MAX_DURATION": 53, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 35840000, "to": 40960000, "AVG_TTFB": 44.721764, "ERRORS": 0, "AVG_DURATION": 44.721764, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 0.0, "COUNT": 363}, {"from": 40960000, "to": 46080000, "AVG_TTFB": 44.714672, "ERRORS": 0, "AVG_DURATION": 44.714672, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 46080000, "to": 51200000, "AVG_TTFB": 44.860275, "ERRORS": 0, "AVG_DURATION": 44.860275, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 54, "MAX_DURATION": 54, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 51200000, "to": 56320000, "AVG_TTFB": 45.877716, "ERRORS": 0, "AVG_DURATION": 45.877716, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 168, "MAX_DURATION": 168, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 56320000, "to": 61440000, "AVG_TTFB": 45.09589, "ERRORS": 0, "AVG_DURATION": 45.09589, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 91, "MAX_DURATION": 91, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 61440000, "to": 66560000, "AVG_TTFB": 44.755493, "ERRORS": 0, "AVG_DURATION": 44.755493, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 66560000, "to": 71680000, "AVG_TTFB": 44.723465, "ERRORS": 0, "AVG_DURATION": 44.723465, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.06992187, "THROUGHPUT": 0.0, "COUNT": 358}, {"from": 71680000, "to": 76800000, "AVG_TTFB": 44.720547, "ERRORS": 0, "AVG_DURATION": 44.720547, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 50, "MAX_DURATION": 50, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 76800000, "to": 81920000, "AVG_TTFB": 44.421917, "ERRORS": 0, "AVG_DURATION": 44.421917, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 81920000, "to": 87040000, "AVG_TTFB": 44.53552, "ERRORS": 0, "AVG_DURATION": 44.53552, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 87040000, "to": 92160000, "AVG_TTFB": 44.482384, "ERRORS": 0, "AVG_DURATION": 44.482384, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 54, "MAX_DURATION": 54, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 0.0, "COUNT": 369}, {"from": 92160000, "to": 97280000, "AVG_TTFB": 44.372604, "ERRORS": 0, "AVG_DURATION": 44.372604, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 97280000, "to": 102400000, "AVG_TTFB": 44.29315, "ERRORS": 0, "AVG_DURATION": 44.29315, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 102400000, "to": 107520000, "AVG_TTFB": 44.321526, "ERRORS": 0, "AVG_DURATION": 44.321526, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 107520000, "to": 112640000, "AVG_TTFB": 44.416893, "ERRORS": 0, "AVG_DURATION": 44.416893, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 112640000, "to": 117760000, "AVG_TTFB": 44.452316, "ERRORS": 0, "AVG_DURATION": 44.452316, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 117760000, "to": 122880000, "AVG_TTFB": 45.157608, "ERRORS": 0, "AVG_DURATION": 45.157608, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 139, "MAX_DURATION": 139, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 122880000, "to": 128000000, "AVG_TTFB": 44.68033, "ERRORS": 0, "AVG_DURATION": 44.68033, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 59, "MAX_DURATION": 59, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 128000000, "to": 133120000, "AVG_TTFB": 44.46281, "ERRORS": 0, "AVG_DURATION": 44.46281, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 59, "MAX_DURATION": 59, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 0.0, "COUNT": 363}, {"from": 133120000, "to": 138240000, "AVG_TTFB": 44.65395, "ERRORS": 0, "AVG_DURATION": 44.65395, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 75, "MAX_DURATION": 75, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 138240000, "to": 143360000, "AVG_TTFB": 44.901638, "ERRORS": 0, "AVG_DURATION": 44.901638, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 107, "MAX_DURATION": 107, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 143360000, "to": 148480000, "AVG_TTFB": 44.836514, "ERRORS": 0, "AVG_DURATION": 44.836514, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 103, "MAX_DURATION": 103, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 148480000, "to": 153600000, "AVG_TTFB": 44.53425, "ERRORS": 0, "AVG_DURATION": 44.53425, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 92, "MAX_DURATION": 92, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 153600000, "to": 158720000, "AVG_TTFB": 44.532787, "ERRORS": 0, "AVG_DURATION": 44.532787, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 93, "MAX_DURATION": 93, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 158720000, "to": 163840000, "AVG_TTFB": 44.494564, "ERRORS": 0, "AVG_DURATION": 44.494564, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 163840000, "to": 168960000, "AVG_TTFB": 44.59178, "ERRORS": 0, "AVG_DURATION": 44.59178, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 61, "MAX_DURATION": 61, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 168960000, "to": 174080000, "AVG_TTFB": 44.51366, "ERRORS": 0, "AVG_DURATION": 44.51366, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 55, "MAX_DURATION": 55, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 174080000, "to": 179200000, "AVG_TTFB": 44.705723, "ERRORS": 0, "AVG_DURATION": 44.705723, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 94, "MAX_DURATION": 94, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 179200000, "to": 184320000, "AVG_TTFB": 44.530056, "ERRORS": 0, "AVG_DURATION": 44.530056, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 53, "MAX_DURATION": 53, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 184320000, "to": 189440000, "AVG_TTFB": 44.432064, "ERRORS": 0, "AVG_DURATION": 44.432064, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 189440000, "to": 194560000, "AVG_TTFB": 44.601093, "ERRORS": 0, "AVG_DURATION": 44.601093, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 80, "MAX_DURATION": 80, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 194560000, "to": 199680000, "AVG_TTFB": 44.38356, "ERRORS": 0, "AVG_DURATION": 44.38356, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 199680000, "to": 204800000, "AVG_TTFB": 44.823368, "ERRORS": 0, "AVG_DURATION": 44.823368, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 126, "MAX_DURATION": 126, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 204800000, "to": 209920000, "AVG_TTFB": 45.469944, "ERRORS": 0, "AVG_DURATION": 45.469944, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 139, "MAX_DURATION": 139, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 209920000, "to": 215040000, "AVG_TTFB": 45.242508, "ERRORS": 0, "AVG_DURATION": 45.242508, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 157, "MAX_DURATION": 157, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 215040000, "to": 220160000, "AVG_TTFB": 45.786884, "ERRORS": 0, "AVG_DURATION": 45.786884, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 171, "MAX_DURATION": 171, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 220160000, "to": 225280000, "AVG_TTFB": 45.07967, "ERRORS": 0, "AVG_DURATION": 45.07967, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 78, "MAX_DURATION": 78, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 225280000, "to": 230400000, "AVG_TTFB": 44.467213, "ERRORS": 0, "AVG_DURATION": 44.467213, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 230400000, "to": 235520000, "AVG_TTFB": 44.651226, "ERRORS": 0, "AVG_DURATION": 44.651226, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 58, "MAX_DURATION": 58, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 235520000, "to": 240640000, "AVG_TTFB": 48.336067, "ERRORS": 0, "AVG_DURATION": 48.336067, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 199, "MAX_DURATION": 199, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 240640000, "to": 245760000, "AVG_TTFB": 45.6511, "ERRORS": 0, "AVG_DURATION": 45.6511, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 119, "MAX_DURATION": 119, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 245760000, "to": 250880000, "AVG_TTFB": 44.598362, "ERRORS": 0, "AVG_DURATION": 44.598362, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 58, "MAX_DURATION": 58, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 250880000, "to": 256000000, "AVG_TTFB": 45.520435, "ERRORS": 0, "AVG_DURATION": 45.520435, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 257, "MAX_DURATION": 257, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 256000000, "to": 261120000, "AVG_TTFB": 45.040985, "ERRORS": 0, "AVG_DURATION": 45.040985, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 123, "MAX_DURATION": 123, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 261120000, "to": 266240000, "AVG_TTFB": 46.087433, "ERRORS": 0, "AVG_DURATION": 46.087433, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 226, "MAX_DURATION": 226, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 266240000, "to": 271360000, "AVG_TTFB": 46.50959, "ERRORS": 0, "AVG_DURATION": 46.50959, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 169, "MAX_DURATION": 169, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 271360000, "to": 276480000, "AVG_TTFB": 45.905148, "ERRORS": 0, "AVG_DURATION": 45.905148, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 141, "MAX_DURATION": 141, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 0.0, "COUNT": 369}, {"from": 276480000, "to": 281600000, "AVG_TTFB": 45.8292, "ERRORS": 0, "AVG_DURATION": 45.8292, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 130, "MAX_DURATION": 130, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 0.0, "COUNT": 363}, {"from": 281600000, "to": 286720000, "AVG_TTFB": 44.508198, "ERRORS": 0, "AVG_DURATION": 44.508198, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 286720000, "to": 291840000, "AVG_TTFB": 44.637604, "ERRORS": 0, "AVG_DURATION": 44.637604, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 115, "MAX_DURATION": 115, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 291840000, "to": 296960000, "AVG_TTFB": 44.412567, "ERRORS": 0, "AVG_DURATION": 44.412567, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 296960000, "to": 302080000, "AVG_TTFB": 44.435966, "ERRORS": 0, "AVG_DURATION": 44.435966, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 51, "MAX_DURATION": 51, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 302080000, "to": 307200000, "AVG_TTFB": 44.69589, "ERRORS": 0, "AVG_DURATION": 44.69589, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 133, "MAX_DURATION": 133, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 307200000, "to": 312320000, "AVG_TTFB": 44.471233, "ERRORS": 0, "AVG_DURATION": 44.471233, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 312320000, "to": 317440000, "AVG_TTFB": 44.77869, "ERRORS": 0, "AVG_DURATION": 44.77869, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 178, "MAX_DURATION": 178, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 317440000, "to": 322560000, "AVG_TTFB": 44.918697, "ERRORS": 0, "AVG_DURATION": 44.918697, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 180, "MAX_DURATION": 180, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 0.0, "COUNT": 369}, {"from": 322560000, "to": 327680000, "AVG_TTFB": 45.69399, "ERRORS": 0, "AVG_DURATION": 45.69399, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 106, "MAX_DURATION": 106, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 327680000, "to": 332800000, "AVG_TTFB": 45.328766, "ERRORS": 0, "AVG_DURATION": 45.328766, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 111, "MAX_DURATION": 111, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 332800000, "to": 337920000, "AVG_TTFB": 46.246574, "ERRORS": 0, "AVG_DURATION": 46.246574, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 173, "MAX_DURATION": 173, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 337920000, "to": 343040000, "AVG_TTFB": 46.975544, "ERRORS": 0, "AVG_DURATION": 46.975544, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 219, "MAX_DURATION": 219, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 343040000, "to": 348160000, "AVG_TTFB": 46.56011, "ERRORS": 0, "AVG_DURATION": 46.56011, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 123, "MAX_DURATION": 123, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 348160000, "to": 353280000, "AVG_TTFB": 47.32425, "ERRORS": 0, "AVG_DURATION": 47.32425, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 224, "MAX_DURATION": 224, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 353280000, "to": 358400000, "AVG_TTFB": 46.448086, "ERRORS": 0, "AVG_DURATION": 46.448086, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 137, "MAX_DURATION": 137, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 358400000, "to": 363520000, "AVG_TTFB": 44.863388, "ERRORS": 0, "AVG_DURATION": 44.863388, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 89, "MAX_DURATION": 89, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 363520000, "to": 368640000, "AVG_TTFB": 44.37705, "ERRORS": 0, "AVG_DURATION": 44.37705, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 368640000, "to": 373760000, "AVG_TTFB": 45.280655, "ERRORS": 0, "AVG_DURATION": 45.280655, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 104, "MAX_DURATION": 104, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 373760000, "to": 378880000, "AVG_TTFB": 48.307693, "ERRORS": 0, "AVG_DURATION": 48.307693, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 532, "MAX_DURATION": 532, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 378880000, "to": 384000000, "AVG_TTFB": 44.980873, "ERRORS": 0, "AVG_DURATION": 44.980873, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 84, "MAX_DURATION": 84, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 384000000, "to": 389120000, "AVG_TTFB": 44.929153, "ERRORS": 0, "AVG_DURATION": 44.929153, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 110, "MAX_DURATION": 110, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 389120000, "to": 394240000, "AVG_TTFB": 46.54945, "ERRORS": 0, "AVG_DURATION": 46.54945, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 276, "MAX_DURATION": 276, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 394240000, "to": 399360000, "AVG_TTFB": 45.287292, "ERRORS": 0, "AVG_DURATION": 45.287292, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 121, "MAX_DURATION": 121, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 0.0, "COUNT": 362}, {"from": 399360000, "to": 404480000, "AVG_TTFB": 44.47139, "ERRORS": 0, "AVG_DURATION": 44.47139, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 404480000, "to": 409600000, "AVG_TTFB": 46.190735, "ERRORS": 0, "AVG_DURATION": 46.190735, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 166, "MAX_DURATION": 166, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 409600000, "to": 414720000, "AVG_TTFB": 44.8388, "ERRORS": 0, "AVG_DURATION": 44.8388, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 129, "MAX_DURATION": 129, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 414720000, "to": 419840000, "AVG_TTFB": 45.002724, "ERRORS": 0, "AVG_DURATION": 45.002724, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 158, "MAX_DURATION": 158, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 419840000, "to": 424960000, "AVG_TTFB": 45.12877, "ERRORS": 0, "AVG_DURATION": 45.12877, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 145, "MAX_DURATION": 145, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 424960000, "to": 430080000, "AVG_TTFB": 45.73901, "ERRORS": 0, "AVG_DURATION": 45.73901, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 97, "MAX_DURATION": 97, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 430080000, "to": 435200000, "AVG_TTFB": 46.420055, "ERRORS": 0, "AVG_DURATION": 46.420055, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 309, "MAX_DURATION": 309, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 0.0, "COUNT": 369}, {"from": 435200000, "to": 440320000, "AVG_TTFB": 44.765026, "ERRORS": 0, "AVG_DURATION": 44.765026, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 144, "MAX_DURATION": 144, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 440320000, "to": 445440000, "AVG_TTFB": 44.749317, "ERRORS": 0, "AVG_DURATION": 44.749317, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 143, "MAX_DURATION": 143, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 445440000, "to": 450560000, "AVG_TTFB": 45.019176, "ERRORS": 0, "AVG_DURATION": 45.019176, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 167, "MAX_DURATION": 167, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 450560000, "to": 455680000, "AVG_TTFB": 45.046448, "ERRORS": 0, "AVG_DURATION": 45.046448, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 160, "MAX_DURATION": 160, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 455680000, "to": 460800000, "AVG_TTFB": 44.820652, "ERRORS": 0, "AVG_DURATION": 44.820652, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 110, "MAX_DURATION": 110, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 460800000, "to": 465920000, "AVG_TTFB": 44.68033, "ERRORS": 0, "AVG_DURATION": 44.68033, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 119, "MAX_DURATION": 119, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 465920000, "to": 471040000, "AVG_TTFB": 44.60548, "ERRORS": 0, "AVG_DURATION": 44.60548, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 93, "MAX_DURATION": 93, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 471040000, "to": 476160000, "AVG_TTFB": 45.070847, "ERRORS": 0, "AVG_DURATION": 45.070847, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 219, "MAX_DURATION": 219, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 476160000, "to": 481280000, "AVG_TTFB": 44.587433, "ERRORS": 0, "AVG_DURATION": 44.587433, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 104, "MAX_DURATION": 104, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 481280000, "to": 486400000, "AVG_TTFB": 44.945354, "ERRORS": 0, "AVG_DURATION": 44.945354, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 125, "MAX_DURATION": 125, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 486400000, "to": 491520000, "AVG_TTFB": 45.150272, "ERRORS": 0, "AVG_DURATION": 45.150272, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 216, "MAX_DURATION": 216, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 491520000, "to": 496640000, "AVG_TTFB": 44.56948, "ERRORS": 0, "AVG_DURATION": 44.56948, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 76, "MAX_DURATION": 76, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 496640000, "to": 501760000, "AVG_TTFB": 44.731506, "ERRORS": 0, "AVG_DURATION": 44.731506, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 87, "MAX_DURATION": 87, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 501760000, "to": 506880000, "AVG_TTFB": 44.70027, "ERRORS": 0, "AVG_DURATION": 44.70027, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 82, "MAX_DURATION": 82, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 506880000, "to": 512000000, "AVG_TTFB": 44.487804, "ERRORS": 0, "AVG_DURATION": 44.487804, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 0.0, "COUNT": 369}, {"from": 512000000, "to": 517120000, "AVG_TTFB": 44.780823, "ERRORS": 0, "AVG_DURATION": 44.780823, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 106, "MAX_DURATION": 106, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 517120000, "to": 522240000, "AVG_TTFB": 46.60989, "ERRORS": 0, "AVG_DURATION": 46.60989, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 277, "MAX_DURATION": 277, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 522240000, "to": 527360000, "AVG_TTFB": 45.095367, "ERRORS": 0, "AVG_DURATION": 45.095367, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 100, "MAX_DURATION": 100, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 527360000, "to": 532480000, "AVG_TTFB": 44.677776, "ERRORS": 0, "AVG_DURATION": 44.677776, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 115, "MAX_DURATION": 115, "ELEMENTS_PER_SECOND": 0.0703125, "THROUGHPUT": 0.0, "COUNT": 360}, {"from": 532480000, "to": 537600000, "AVG_TTFB": 45.245667, "ERRORS": 0, "AVG_DURATION": 45.245667, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 146, "MAX_DURATION": 146, "ELEMENTS_PER_SECOND": 0.06757812, "THROUGHPUT": 0.0, "COUNT": 346}, {"from": 537600000, "to": 542720000, "AVG_TTFB": 44.551083, "ERRORS": 0, "AVG_DURATION": 44.551083, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 82, "MAX_DURATION": 82, "ELEMENTS_PER_SECOND": 0.063085936, "THROUGHPUT": 0.0, "COUNT": 323}, {"from": 542720000, "to": 547840000, "AVG_TTFB": 44.48458, "ERRORS": 0, "AVG_DURATION": 44.48458, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.04433594, "THROUGHPUT": 0.0, "COUNT": 227}, {"from": 547840000, "to": 552960000, "AVG_TTFB": 45.48347, "ERRORS": 0, "AVG_DURATION": 45.48347, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 122, "MAX_DURATION": 122, "ELEMENTS_PER_SECOND": 0.047265626, "THROUGHPUT": 0.0, "COUNT": 242}, {"from": 552960000, "to": 558080000, "AVG_TTFB": 44.442448, "ERRORS": 0, "AVG_DURATION": 44.442448, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 51, "MAX_DURATION": 51, "ELEMENTS_PER_SECOND": 0.054296874, "THROUGHPUT": 0.0, "COUNT": 278}, {"from": 558080000, "to": 563200000, "AVG_TTFB": 44.778572, "ERRORS": 0, "AVG_DURATION": 44.778572, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 120, "MAX_DURATION": 120, "ELEMENTS_PER_SECOND": 0.0546875, "THROUGHPUT": 0.0, "COUNT": 280}, {"from": 563200000, "to": 568320000, "AVG_TTFB": 46.00709, "ERRORS": 0, "AVG_DURATION": 46.00709, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 166, "MAX_DURATION": 166, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 0.0, "COUNT": 282}, {"from": 568320000, "to": 573440000, "AVG_TTFB": 44.597866, "ERRORS": 0, "AVG_DURATION": 44.597866, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 53, "MAX_DURATION": 53, "ELEMENTS_PER_SECOND": 0.054882813, "THROUGHPUT": 0.0, "COUNT": 281}, {"from": 573440000, "to": 578560000, "AVG_TTFB": 45.024822, "ERRORS": 0, "AVG_DURATION": 45.024822, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 127, "MAX_DURATION": 127, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 0.0, "COUNT": 282}, {"from": 578560000, "to": 583680000, "AVG_TTFB": 44.654804, "ERRORS": 0, "AVG_DURATION": 44.654804, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 103, "MAX_DURATION": 103, "ELEMENTS_PER_SECOND": 0.054882813, "THROUGHPUT": 0.0, "COUNT": 281}, {"from": 583680000, "to": 588800000, "AVG_TTFB": 44.475178, "ERRORS": 0, "AVG_DURATION": 44.475178, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 51, "MAX_DURATION": 51, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 0.0, "COUNT": 282}, {"from": 588800000, "to": 593920000, "AVG_TTFB": 44.81915, "ERRORS": 0, "AVG_DURATION": 44.81915, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 102, "MAX_DURATION": 102, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 0.0, "COUNT": 282}, {"from": 593920000, "to": 599040000, "AVG_TTFB": 44.932625, "ERRORS": 0, "AVG_DURATION": 44.932625, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 65, "MAX_DURATION": 65, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 0.0, "COUNT": 282}, {"from": 599040000, "to": 604160000, "AVG_TTFB": 44.762413, "ERRORS": 0, "AVG_DURATION": 44.762413, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 70, "MAX_DURATION": 70, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 0.0, "COUNT": 282}, {"from": 604160000, "to": 609280000, "AVG_TTFB": 44.55477, "ERRORS": 0, "AVG_DURATION": 44.55477, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 46, "MAX_DURATION": 46, "ELEMENTS_PER_SECOND": 0.055273436, "THROUGHPUT": 0.0, "COUNT": 283}, {"from": 609280000, "to": 614400000, "AVG_TTFB": 45.02667, "ERRORS": 0, "AVG_DURATION": 45.02667, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 43, "MAX_TTFB": 100, "MAX_DURATION": 100, "ELEMENTS_PER_SECOND": 0.029296875, "THROUGHPUT": 0.0, "COUNT": 150}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/ba912bd5-d4ee-425c-ae82-b56a75c0e6ef/elements/062b066f-e7c0-4e67-9a08-65d6494fecc9/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 5120000, "AVG_TTFB": 14.534246, "ERRORS": 0, "AVG_DURATION": 145.33424, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 37, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1288, "MAX_DURATION": 1775, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1702.7867, "COUNT": 365}, {"from": 5120000, "to": 10240000, "AVG_TTFB": 18.191257, "ERRORS": 0, "AVG_DURATION": 163.8552, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1267, "MAX_DURATION": 2301, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.7273, "COUNT": 366}, {"from": 10240000, "to": 15360000, "AVG_TTFB": 2.7540984, "ERRORS": 0, "AVG_DURATION": 133.82787, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 18, "MAX_DURATION": 2901, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.7308, "COUNT": 366}, {"from": 15360000, "to": 20480000, "AVG_TTFB": 5.479564, "ERRORS": 0, "AVG_DURATION": 142.67302, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 439, "MAX_DURATION": 2395, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.7266, "COUNT": 367}, {"from": 20480000, "to": 25600000, "AVG_TTFB": 36.01093, "ERRORS": 0, "AVG_DURATION": 196.31148, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 37, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 6062, "MAX_DURATION": 6137, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.7219, "COUNT": 366}, {"from": 25600000, "to": 30720000, "AVG_TTFB": 16.280653, "ERRORS": 0, "AVG_DURATION": 180.9891, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2280, "MAX_DURATION": 5919, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.3961, "COUNT": 367}, {"from": 30720000, "to": 35840000, "AVG_TTFB": 10.356164, "ERRORS": 0, "AVG_DURATION": 182.2904, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 988, "MAX_DURATION": 8735, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.268, "COUNT": 365}, {"from": 35840000, "to": 40960000, "AVG_TTFB": 7.0737705, "ERRORS": 0, "AVG_DURATION": 142.76776, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1377, "MAX_DURATION": 2882, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1704.2051, "COUNT": 366}, {"from": 40960000, "to": 46080000, "AVG_TTFB": 15.005465, "ERRORS": 0, "AVG_DURATION": 284.00546, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 35, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 758, "MAX_DURATION": 9202, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.9004, "COUNT": 366}, {"from": 46080000, "to": 51200000, "AVG_TTFB": 27.51093, "ERRORS": 0, "AVG_DURATION": 190.05191, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5259, "MAX_DURATION": 5473, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.0164, "COUNT": 366}, {"from": 51200000, "to": 56320000, "AVG_TTFB": 3.3242507, "ERRORS": 5, "AVG_DURATION": 113.85831, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 17, "ERROR_RATE": 1.3623978, "MIN_TTFB": 2, "MAX_TTFB": 12, "MAX_DURATION": 1350, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1691.2058, "COUNT": 367}, {"from": 56320000, "to": 61440000, "AVG_TTFB": 3.120548, "ERRORS": 0, "AVG_DURATION": 104.969864, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 11, "MAX_DURATION": 298, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.1816, "COUNT": 365}, {"from": 61440000, "to": 66560000, "AVG_TTFB": 36.770493, "ERRORS": 0, "AVG_DURATION": 521.27594, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3782, "MAX_DURATION": 23927, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1704.0359, "COUNT": 366}, {"from": 66560000, "to": 71680000, "AVG_TTFB": 184.85475, "ERRORS": 26, "AVG_DURATION": 1127.8743, "ERRORS_PER_SECOND": 0.005078125, "MIN_DURATION": 16, "ERROR_RATE": 7.26257, "MIN_TTFB": 2, "MAX_TTFB": 29896, "MAX_DURATION": 36404, "ELEMENTS_PER_SECOND": 0.06992187, "THROUGHPUT": 1571.3779, "COUNT": 358}, {"from": 71680000, "to": 76800000, "AVG_TTFB": 25.896175, "ERRORS": 0, "AVG_DURATION": 220.37158, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1375, "MAX_DURATION": 3442, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.725, "COUNT": 366}, {"from": 76800000, "to": 81920000, "AVG_TTFB": 30.471233, "ERRORS": 0, "AVG_DURATION": 190.98082, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 49, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2257, "MAX_DURATION": 3651, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.1058, "COUNT": 365}, {"from": 81920000, "to": 87040000, "AVG_TTFB": 8.5625, "ERRORS": 0, "AVG_DURATION": 132.14131, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1370, "MAX_DURATION": 2318, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1713.2731, "COUNT": 368}, {"from": 87040000, "to": 92160000, "AVG_TTFB": 5.9673023, "ERRORS": 0, "AVG_DURATION": 147.23433, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 923, "MAX_DURATION": 3330, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.3578, "COUNT": 367}, {"from": 92160000, "to": 97280000, "AVG_TTFB": 16.222527, "ERRORS": 0, "AVG_DURATION": 193.43132, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1769, "MAX_DURATION": 4406, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.6199, "COUNT": 364}, {"from": 97280000, "to": 102400000, "AVG_TTFB": 3.7185793, "ERRORS": 0, "AVG_DURATION": 177.5929, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 129, "MAX_DURATION": 10871, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.8015, "COUNT": 366}, {"from": 102400000, "to": 107520000, "AVG_TTFB": 3.8610353, "ERRORS": 0, "AVG_DURATION": 130.60764, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 32, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 134, "MAX_DURATION": 4599, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.5984, "COUNT": 367}, {"from": 107520000, "to": 112640000, "AVG_TTFB": 14.528456, "ERRORS": 0, "AVG_DURATION": 136.18156, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 51, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4324, "MAX_DURATION": 4536, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1718.0289, "COUNT": 369}, {"from": 112640000, "to": 117760000, "AVG_TTFB": 11.595629, "ERRORS": 0, "AVG_DURATION": 123.07377, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3223, "MAX_DURATION": 3292, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.0485, "COUNT": 366}, {"from": 117760000, "to": 122880000, "AVG_TTFB": 6.0190215, "ERRORS": 0, "AVG_DURATION": 108.921196, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 979, "MAX_DURATION": 1043, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1712.1586, "COUNT": 368}, {"from": 122880000, "to": 128000000, "AVG_TTFB": 4.65847, "ERRORS": 0, "AVG_DURATION": 120.08743, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 531, "MAX_DURATION": 1954, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.888, "COUNT": 366}, {"from": 128000000, "to": 133120000, "AVG_TTFB": 46.29041, "ERRORS": 0, "AVG_DURATION": 493.28494, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4527, "MAX_DURATION": 17302, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1699.7207, "COUNT": 365}, {"from": 133120000, "to": 138240000, "AVG_TTFB": 26.207651, "ERRORS": 0, "AVG_DURATION": 159.94263, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 31, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2503, "MAX_DURATION": 2785, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.0657, "COUNT": 366}, {"from": 138240000, "to": 143360000, "AVG_TTFB": 7.4246573, "ERRORS": 11, "AVG_DURATION": 130.94246, "ERRORS_PER_SECOND": 0.0021484375, "MIN_DURATION": 34, "ERROR_RATE": 3.0136986, "MIN_TTFB": 2, "MAX_TTFB": 614, "MAX_DURATION": 1900, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1662.2758, "COUNT": 365}, {"from": 143360000, "to": 148480000, "AVG_TTFB": 3.8964577, "ERRORS": 0, "AVG_DURATION": 128.07356, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 27, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 155, "MAX_DURATION": 2980, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.3422, "COUNT": 367}, {"from": 148480000, "to": 153600000, "AVG_TTFB": 2.86612, "ERRORS": 7, "AVG_DURATION": 122.22951, "ERRORS_PER_SECOND": 0.0013671875, "MIN_DURATION": 15, "ERROR_RATE": 1.9125683, "MIN_TTFB": 1, "MAX_TTFB": 25, "MAX_DURATION": 3949, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1683.4987, "COUNT": 366}, {"from": 153600000, "to": 158720000, "AVG_TTFB": 17.084469, "ERRORS": 0, "AVG_DURATION": 156.23433, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 27, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4392, "MAX_DURATION": 6292, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.277, "COUNT": 367}, {"from": 158720000, "to": 163840000, "AVG_TTFB": 6.8497267, "ERRORS": 0, "AVG_DURATION": 165.89072, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1064, "MAX_DURATION": 1889, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.8441, "COUNT": 366}, {"from": 163840000, "to": 168960000, "AVG_TTFB": 3.773224, "ERRORS": 0, "AVG_DURATION": 111.251366, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 350, "MAX_DURATION": 817, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.864, "COUNT": 366}, {"from": 168960000, "to": 174080000, "AVG_TTFB": 3.722071, "ERRORS": 0, "AVG_DURATION": 125.18256, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 202, "MAX_DURATION": 1818, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.3359, "COUNT": 367}, {"from": 174080000, "to": 179200000, "AVG_TTFB": 2.7029972, "ERRORS": 0, "AVG_DURATION": 147.39783, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 25, "MAX_DURATION": 3575, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.6348, "COUNT": 367}, {"from": 179200000, "to": 184320000, "AVG_TTFB": 28.217983, "ERRORS": 0, "AVG_DURATION": 206.83923, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 24, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3864, "MAX_DURATION": 5644, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.4641, "COUNT": 367}, {"from": 184320000, "to": 189440000, "AVG_TTFB": 4.555858, "ERRORS": 0, "AVG_DURATION": 113.3624, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 601, "MAX_DURATION": 1701, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.4769, "COUNT": 367}, {"from": 189440000, "to": 194560000, "AVG_TTFB": 3.0163934, "ERRORS": 0, "AVG_DURATION": 106.91803, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 60, "MAX_DURATION": 578, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.0579, "COUNT": 366}, {"from": 194560000, "to": 199680000, "AVG_TTFB": 3.6277175, "ERRORS": 0, "AVG_DURATION": 112.796196, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 254, "MAX_DURATION": 3130, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1712.2004, "COUNT": 368}, {"from": 199680000, "to": 204800000, "AVG_TTFB": 20.282967, "ERRORS": 0, "AVG_DURATION": 253.07692, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4928, "MAX_DURATION": 7144, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1695.0391, "COUNT": 364}, {"from": 204800000, "to": 209920000, "AVG_TTFB": 21.184782, "ERRORS": 0, "AVG_DURATION": 191.76903, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 36, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3658, "MAX_DURATION": 3870, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1712.1707, "COUNT": 368}, {"from": 209920000, "to": 215040000, "AVG_TTFB": 2.660274, "ERRORS": 0, "AVG_DURATION": 103.454796, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 7, "MAX_DURATION": 259, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.3773, "COUNT": 365}, {"from": 215040000, "to": 220160000, "AVG_TTFB": 9.043478, "ERRORS": 0, "AVG_DURATION": 161.16576, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 514, "MAX_DURATION": 1922, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1712.2438, "COUNT": 368}, {"from": 220160000, "to": 225280000, "AVG_TTFB": 29.151934, "ERRORS": 0, "AVG_DURATION": 322.43372, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1907, "MAX_DURATION": 35887, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1684.5879, "COUNT": 362}, {"from": 225280000, "to": 230400000, "AVG_TTFB": 22.226158, "ERRORS": 26, "AVG_DURATION": 266.90463, "ERRORS_PER_SECOND": 0.005078125, "MIN_DURATION": 14, "ERROR_RATE": 7.084469, "MIN_TTFB": 2, "MAX_TTFB": 1256, "MAX_DURATION": 7307, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1612.0924, "COUNT": 367}, {"from": 230400000, "to": 235520000, "AVG_TTFB": 3.9373298, "ERRORS": 0, "AVG_DURATION": 112.6921, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 49, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 263, "MAX_DURATION": 1230, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.4176, "COUNT": 367}, {"from": 235520000, "to": 240640000, "AVG_TTFB": 3.5342467, "ERRORS": 5, "AVG_DURATION": 110.5726, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 16, "ERROR_RATE": 1.369863, "MIN_TTFB": 2, "MAX_TTFB": 162, "MAX_DURATION": 816, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1682.973, "COUNT": 365}, {"from": 240640000, "to": 245760000, "AVG_TTFB": 42.508244, "ERRORS": 0, "AVG_DURATION": 238.42308, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 62, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6218, "MAX_DURATION": 13360, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.268, "COUNT": 364}, {"from": 245760000, "to": 250880000, "AVG_TTFB": 5.6684785, "ERRORS": 0, "AVG_DURATION": 123.703804, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 732, "MAX_DURATION": 1380, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1713.3363, "COUNT": 368}, {"from": 250880000, "to": 256000000, "AVG_TTFB": 7.2821918, "ERRORS": 0, "AVG_DURATION": 118.690414, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 43, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1083, "MAX_DURATION": 2640, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.2566, "COUNT": 365}, {"from": 256000000, "to": 261120000, "AVG_TTFB": 2.874317, "ERRORS": 0, "AVG_DURATION": 126.71584, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 31, "MAX_DURATION": 2456, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.864, "COUNT": 366}, {"from": 261120000, "to": 266240000, "AVG_TTFB": 3.3297002, "ERRORS": 0, "AVG_DURATION": 116.36512, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 162, "MAX_DURATION": 1847, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.3871, "COUNT": 367}, {"from": 266240000, "to": 271360000, "AVG_TTFB": 4.09863, "ERRORS": 0, "AVG_DURATION": 115.40548, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 39, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 264, "MAX_DURATION": 847, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.2051, "COUNT": 365}, {"from": 271360000, "to": 276480000, "AVG_TTFB": 17.577236, "ERRORS": 0, "AVG_DURATION": 165.82927, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 24, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4975, "MAX_DURATION": 7551, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1717.9192, "COUNT": 369}, {"from": 276480000, "to": 281600000, "AVG_TTFB": 13.788462, "ERRORS": 0, "AVG_DURATION": 150.35715, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 34, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3413, "MAX_DURATION": 7224, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.7, "COUNT": 364}, {"from": 281600000, "to": 286720000, "AVG_TTFB": 10.504109, "ERRORS": 0, "AVG_DURATION": 126.904106, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1608, "MAX_DURATION": 3869, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.4407, "COUNT": 365}, {"from": 286720000, "to": 291840000, "AVG_TTFB": 11.434783, "ERRORS": 0, "AVG_DURATION": 130.61957, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2981, "MAX_DURATION": 3806, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1712.1028, "COUNT": 368}, {"from": 291840000, "to": 296960000, "AVG_TTFB": 3.6430519, "ERRORS": 0, "AVG_DURATION": 320.81473, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 30, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 141, "MAX_DURATION": 26922, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.7555, "COUNT": 367}, {"from": 296960000, "to": 302080000, "AVG_TTFB": 2.6784742, "ERRORS": 0, "AVG_DURATION": 124.54224, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6, "MAX_DURATION": 2565, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.5863, "COUNT": 367}, {"from": 302080000, "to": 307200000, "AVG_TTFB": 19.49589, "ERRORS": 0, "AVG_DURATION": 224.2274, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2919, "MAX_DURATION": 9405, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.5535, "COUNT": 365}, {"from": 307200000, "to": 312320000, "AVG_TTFB": 6.1923075, "ERRORS": 0, "AVG_DURATION": 142.87912, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 746, "MAX_DURATION": 4055, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.8425, "COUNT": 364}, {"from": 312320000, "to": 317440000, "AVG_TTFB": 23.953678, "ERRORS": 11, "AVG_DURATION": 145.1744, "ERRORS_PER_SECOND": 0.0021484375, "MIN_DURATION": 12, "ERROR_RATE": 2.997275, "MIN_TTFB": 2, "MAX_TTFB": 4161, "MAX_DURATION": 4228, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1666.1677, "COUNT": 367}, {"from": 317440000, "to": 322560000, "AVG_TTFB": 19.508152, "ERRORS": 0, "AVG_DURATION": 133.27174, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5467, "MAX_DURATION": 5679, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1713.293, "COUNT": 368}, {"from": 322560000, "to": 327680000, "AVG_TTFB": 3.468665, "ERRORS": 6, "AVG_DURATION": 105.48229, "ERRORS_PER_SECOND": 0.001171875, "MIN_DURATION": 12, "ERROR_RATE": 1.6348774, "MIN_TTFB": 2, "MAX_TTFB": 139, "MAX_DURATION": 316, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1689.5525, "COUNT": 367}, {"from": 327680000, "to": 332800000, "AVG_TTFB": 42.563187, "ERRORS": 0, "AVG_DURATION": 290.22803, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 27, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4533, "MAX_DURATION": 5058, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.291, "COUNT": 364}, {"from": 332800000, "to": 337920000, "AVG_TTFB": 7.628415, "ERRORS": 0, "AVG_DURATION": 134.18306, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1070, "MAX_DURATION": 1848, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.7711, "COUNT": 366}, {"from": 337920000, "to": 343040000, "AVG_TTFB": 10.9809265, "ERRORS": 0, "AVG_DURATION": 150.703, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 946, "MAX_DURATION": 1439, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.6367, "COUNT": 367}, {"from": 343040000, "to": 348160000, "AVG_TTFB": 3.294278, "ERRORS": 0, "AVG_DURATION": 110.48774, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 83, "MAX_DURATION": 268, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.4641, "COUNT": 367}, {"from": 348160000, "to": 353280000, "AVG_TTFB": 2.8310626, "ERRORS": 0, "AVG_DURATION": 107.14169, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 57, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 42, "MAX_DURATION": 290, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.3773, "COUNT": 367}, {"from": 353280000, "to": 358400000, "AVG_TTFB": 8.339726, "ERRORS": 0, "AVG_DURATION": 163.2904, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 501, "MAX_DURATION": 2038, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.1058, "COUNT": 365}, {"from": 358400000, "to": 363520000, "AVG_TTFB": 8.585831, "ERRORS": 0, "AVG_DURATION": 129.29428, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1601, "MAX_DURATION": 1813, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.4641, "COUNT": 367}, {"from": 363520000, "to": 368640000, "AVG_TTFB": 8.120218, "ERRORS": 0, "AVG_DURATION": 142.79234, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 25, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 827, "MAX_DURATION": 2700, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1704.2953, "COUNT": 366}, {"from": 368640000, "to": 373760000, "AVG_TTFB": 5.348774, "ERRORS": 0, "AVG_DURATION": 131.1417, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 33, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 538, "MAX_DURATION": 4245, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.6667, "COUNT": 367}, {"from": 373760000, "to": 378880000, "AVG_TTFB": 17.742466, "ERRORS": 0, "AVG_DURATION": 167.66576, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 50, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3741, "MAX_DURATION": 5989, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.4574, "COUNT": 365}, {"from": 378880000, "to": 384000000, "AVG_TTFB": 5.956284, "ERRORS": 0, "AVG_DURATION": 350.9071, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1081, "MAX_DURATION": 29394, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.9437, "COUNT": 366}, {"from": 384000000, "to": 389120000, "AVG_TTFB": 14.34153, "ERRORS": 0, "AVG_DURATION": 175.90437, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 27, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2320, "MAX_DURATION": 3560, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1704.3668, "COUNT": 366}, {"from": 389120000, "to": 394240000, "AVG_TTFB": 11.509589, "ERRORS": 0, "AVG_DURATION": 245.23836, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 57, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 549, "MAX_DURATION": 8268, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.552, "COUNT": 365}, {"from": 394240000, "to": 399360000, "AVG_TTFB": 61.52066, "ERRORS": 30, "AVG_DURATION": 533.7879, "ERRORS_PER_SECOND": 0.005859375, "MIN_DURATION": 14, "ERROR_RATE": 8.264462, "MIN_TTFB": 2, "MAX_TTFB": 7359, "MAX_DURATION": 12889, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1577.9441, "COUNT": 363}, {"from": 399360000, "to": 404480000, "AVG_TTFB": 8.53552, "ERRORS": 0, "AVG_DURATION": 177.54918, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 28, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 954, "MAX_DURATION": 7057, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.4004, "COUNT": 366}, {"from": 404480000, "to": 409600000, "AVG_TTFB": 4.464481, "ERRORS": 0, "AVG_DURATION": 121.010925, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 57, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 552, "MAX_DURATION": 1281, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.8464, "COUNT": 366}, {"from": 409600000, "to": 414720000, "AVG_TTFB": 3.0813007, "ERRORS": 7, "AVG_DURATION": 125.40922, "ERRORS_PER_SECOND": 0.0013671875, "MIN_DURATION": 15, "ERROR_RATE": 1.897019, "MIN_TTFB": 2, "MAX_TTFB": 5, "MAX_DURATION": 5019, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1697.8418, "COUNT": 369}, {"from": 414720000, "to": 419840000, "AVG_TTFB": 7.603825, "ERRORS": 0, "AVG_DURATION": 141.31148, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1128, "MAX_DURATION": 2911, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.5719, "COUNT": 366}, {"from": 419840000, "to": 424960000, "AVG_TTFB": 17.624657, "ERRORS": 0, "AVG_DURATION": 331.25204, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 35, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1245, "MAX_DURATION": 24606, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.2129, "COUNT": 365}, {"from": 424960000, "to": 430080000, "AVG_TTFB": 14.706849, "ERRORS": 0, "AVG_DURATION": 248.56712, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 54, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1653, "MAX_DURATION": 11015, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.1667, "COUNT": 365}, {"from": 430080000, "to": 435200000, "AVG_TTFB": 3.3433242, "ERRORS": 0, "AVG_DURATION": 127.47139, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 80, "MAX_DURATION": 4248, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.7039, "COUNT": 367}, {"from": 435200000, "to": 440320000, "AVG_TTFB": 6.076503, "ERRORS": 0, "AVG_DURATION": 134.75684, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 747, "MAX_DURATION": 1689, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.7113, "COUNT": 366}, {"from": 440320000, "to": 445440000, "AVG_TTFB": 6.8201632, "ERRORS": 0, "AVG_DURATION": 157.39783, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 977, "MAX_DURATION": 3119, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.3347, "COUNT": 367}, {"from": 445440000, "to": 450560000, "AVG_TTFB": 2.8825138, "ERRORS": 0, "AVG_DURATION": 118.72131, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 28, "MAX_DURATION": 2118, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.9594, "COUNT": 366}, {"from": 450560000, "to": 455680000, "AVG_TTFB": 3.0655737, "ERRORS": 0, "AVG_DURATION": 134.0246, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 80, "MAX_DURATION": 5276, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.9324, "COUNT": 366}, {"from": 455680000, "to": 460800000, "AVG_TTFB": 2.9755435, "ERRORS": 0, "AVG_DURATION": 112.44293, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 57, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 11, "MAX_DURATION": 1438, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1713.5164, "COUNT": 368}, {"from": 460800000, "to": 465920000, "AVG_TTFB": 15.624658, "ERRORS": 0, "AVG_DURATION": 153.17809, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3256, "MAX_DURATION": 3424, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.4574, "COUNT": 365}, {"from": 465920000, "to": 471040000, "AVG_TTFB": 5.4590163, "ERRORS": 0, "AVG_DURATION": 135.01639, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 38, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 974, "MAX_DURATION": 7081, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.9098, "COUNT": 366}, {"from": 471040000, "to": 476160000, "AVG_TTFB": 4.8797812, "ERRORS": 0, "AVG_DURATION": 132.91257, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 38, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 109, "MAX_DURATION": 2177, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.0867, "COUNT": 366}, {"from": 476160000, "to": 481280000, "AVG_TTFB": 4.4836063, "ERRORS": 0, "AVG_DURATION": 113.88525, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 619, "MAX_DURATION": 1404, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1704.4274, "COUNT": 366}, {"from": 481280000, "to": 486400000, "AVG_TTFB": 39.103542, "ERRORS": 14, "AVG_DURATION": 242.60764, "ERRORS_PER_SECOND": 0.002734375, "MIN_DURATION": 58, "ERROR_RATE": 3.814714, "MIN_TTFB": 2, "MAX_TTFB": 6982, "MAX_DURATION": 8059, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1650.9141, "COUNT": 367}, {"from": 486400000, "to": 491520000, "AVG_TTFB": 3.1666667, "ERRORS": 0, "AVG_DURATION": 118.34153, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6, "MAX_DURATION": 4369, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.4688, "COUNT": 366}, {"from": 491520000, "to": 496640000, "AVG_TTFB": 15.1553135, "ERRORS": 0, "AVG_DURATION": 140.62671, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4179, "MAX_DURATION": 7594, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.3981, "COUNT": 367}, {"from": 496640000, "to": 501760000, "AVG_TTFB": 7.307902, "ERRORS": 7, "AVG_DURATION": 136.24796, "ERRORS_PER_SECOND": 0.0013671875, "MIN_DURATION": 37, "ERROR_RATE": 1.907357, "MIN_TTFB": 2, "MAX_TTFB": 834, "MAX_DURATION": 2887, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1687.2825, "COUNT": 367}, {"from": 501760000, "to": 506880000, "AVG_TTFB": 11.4562845, "ERRORS": 0, "AVG_DURATION": 180.45082, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2194, "MAX_DURATION": 7747, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1704.0007, "COUNT": 366}, {"from": 506880000, "to": 512000000, "AVG_TTFB": 9.421196, "ERRORS": 0, "AVG_DURATION": 175.58424, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 31, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1181, "MAX_DURATION": 4791, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1711.9187, "COUNT": 368}, {"from": 512000000, "to": 517120000, "AVG_TTFB": 14.89863, "ERRORS": 0, "AVG_DURATION": 156.98904, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1925, "MAX_DURATION": 3589, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.0797, "COUNT": 365}, {"from": 517120000, "to": 522240000, "AVG_TTFB": 4.446575, "ERRORS": 0, "AVG_DURATION": 122.80274, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 195, "MAX_DURATION": 1713, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.1914, "COUNT": 365}, {"from": 522240000, "to": 527360000, "AVG_TTFB": 21.643051, "ERRORS": 0, "AVG_DURATION": 171.6376, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 53, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4609, "MAX_DURATION": 4884, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.568, "COUNT": 367}, {"from": 527360000, "to": 532480000, "AVG_TTFB": 23.716667, "ERRORS": 0, "AVG_DURATION": 214.10556, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 27, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2438, "MAX_DURATION": 7703, "ELEMENTS_PER_SECOND": 0.0703125, "THROUGHPUT": 1675.0875, "COUNT": 360}, {"from": 532480000, "to": 537600000, "AVG_TTFB": 7.665706, "ERRORS": 1, "AVG_DURATION": 2180.8818, "ERRORS_PER_SECOND": 0.0001953125, "MIN_DURATION": 32, "ERROR_RATE": 0.28818443, "MIN_TTFB": 2, "MAX_TTFB": 621, "MAX_DURATION": 300191, "ELEMENTS_PER_SECOND": 0.06777344, "THROUGHPUT": 1615.2627, "COUNT": 347}, {"from": 537600000, "to": 542720000, "AVG_TTFB": 84.5, "ERRORS": 0, "AVG_DURATION": 2301.7036, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 52, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 25288, "MAX_DURATION": 111478, "ELEMENTS_PER_SECOND": 0.06328125, "THROUGHPUT": 1509.6172, "COUNT": 324}, {"from": 542720000, "to": 547840000, "AVG_TTFB": 11.580786, "ERRORS": 3, "AVG_DURATION": 8016.0044, "ERRORS_PER_SECOND": 0.0005859375, "MIN_DURATION": 64, "ERROR_RATE": 1.3100437, "MIN_TTFB": 2, "MAX_TTFB": 1060, "MAX_DURATION": 349267, "ELEMENTS_PER_SECOND": 0.04472656, "THROUGHPUT": 1068.1248, "COUNT": 229}, {"from": 547840000, "to": 552960000, "AVG_TTFB": 7.7213116, "ERRORS": 0, "AVG_DURATION": 1812.8853, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 38, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 497, "MAX_DURATION": 54930, "ELEMENTS_PER_SECOND": 0.04765625, "THROUGHPUT": 1142.5261, "COUNT": 244}, {"from": 552960000, "to": 558080000, "AVG_TTFB": 4.620072, "ERRORS": 0, "AVG_DURATION": 362.24014, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 63, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 159, "MAX_DURATION": 27176, "ELEMENTS_PER_SECOND": 0.054492187, "THROUGHPUT": 1302.4207, "COUNT": 279}, {"from": 558080000, "to": 563200000, "AVG_TTFB": 20.846428, "ERRORS": 0, "AVG_DURATION": 318.04285, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 62, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1433, "MAX_DURATION": 8575, "ELEMENTS_PER_SECOND": 0.0546875, "THROUGHPUT": 1307.0355, "COUNT": 280}, {"from": 563200000, "to": 568320000, "AVG_TTFB": 9.039145, "ERRORS": 0, "AVG_DURATION": 156.51602, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 754, "MAX_DURATION": 3599, "ELEMENTS_PER_SECOND": 0.054882813, "THROUGHPUT": 1311.6957, "COUNT": 281}, {"from": 568320000, "to": 573440000, "AVG_TTFB": 14.316726, "ERRORS": 23, "AVG_DURATION": 253.8612, "ERRORS_PER_SECOND": 0.0044921874, "MIN_DURATION": 22, "ERROR_RATE": 8.185054, "MIN_TTFB": 2, "MAX_TTFB": 1041, "MAX_DURATION": 9095, "ELEMENTS_PER_SECOND": 0.054882813, "THROUGHPUT": 1222.9274, "COUNT": 281}, {"from": 573440000, "to": 578560000, "AVG_TTFB": 6.70922, "ERRORS": 0, "AVG_DURATION": 128.83333, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 62, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 760, "MAX_DURATION": 1357, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 1315.8773, "COUNT": 282}, {"from": 578560000, "to": 583680000, "AVG_TTFB": 2.8333333, "ERRORS": 0, "AVG_DURATION": 120.276596, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6, "MAX_DURATION": 940, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 1316.1031, "COUNT": 282}, {"from": 583680000, "to": 588800000, "AVG_TTFB": 18.861702, "ERRORS": 5, "AVG_DURATION": 160.12766, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 17, "ERROR_RATE": 1.7730496, "MIN_TTFB": 2, "MAX_TTFB": 1765, "MAX_DURATION": 2850, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 1299.9332, "COUNT": 282}, {"from": 588800000, "to": 593920000, "AVG_TTFB": 4.149466, "ERRORS": 0, "AVG_DURATION": 125.533806, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 56, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 299, "MAX_DURATION": 1014, "ELEMENTS_PER_SECOND": 0.054882813, "THROUGHPUT": 1311.4504, "COUNT": 281}, {"from": 593920000, "to": 599040000, "AVG_TTFB": 3.4823945, "ERRORS": 0, "AVG_DURATION": 157.82042, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 210, "MAX_DURATION": 2163, "ELEMENTS_PER_SECOND": 0.05546875, "THROUGHPUT": 1326.5754, "COUNT": 284}, {"from": 599040000, "to": 604160000, "AVG_TTFB": 4.5425534, "ERRORS": 0, "AVG_DURATION": 121.57092, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 27, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 410, "MAX_DURATION": 744, "ELEMENTS_PER_SECOND": 0.055078126, "THROUGHPUT": 1316.1031, "COUNT": 282}, {"from": 604160000, "to": 609280000, "AVG_TTFB": 2.9323843, "ERRORS": 0, "AVG_DURATION": 138.90036, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 76, "MAX_DURATION": 1891, "ELEMENTS_PER_SECOND": 0.054882813, "THROUGHPUT": 1311.4988, "COUNT": 281}, {"from": 609280000, "to": 614400000, "AVG_TTFB": 3.2631578, "ERRORS": 0, "AVG_DURATION": 117.13816, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 93, "MAX_DURATION": 288, "ELEMENTS_PER_SECOND": 0.0296875, "THROUGHPUT": 709.9625, "COUNT": 152}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/8725e0cd-92a4-4105-803b-86433851fcfc/elements?category=TRANSACTION':
            return '[{"id": "all-transactions", "name": "<all transactions>", "type": "ALL_TRANSACTIONS"}, {"id": "50ddd53f-4a4b-4bf8-a6b3-7a262d3d6cef", "name": "Home", "path": ["ParcoursUtilisateur", "Actions", "Home"], "type": "TRANSACTION"}, {"id": "0ca05bf2-4ba9-483e-9041-c9ac38ecb8b8", "name": "See cats and reptiles", "path": ["ParcoursUtilisateur", "Actions", "See cats and reptiles"], "type": "TRANSACTION"}, {"id": "b3e1f7c6-a4ef-438e-a3bb-8ffa1cb4a85e", "name": "Other", "path": ["ParcoursUtilisateur", "Actions", "Other"], "type": "TRANSACTION"}, {"id": "062b066f-e7c0-4e67-9a08-65d6494fecc9", "name": "Buy a fish", "path": ["ParcoursUtilisateur", "Actions", "Buy a fish"], "type": "TRANSACTION"}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/8725e0cd-92a4-4105-803b-86433851fcfc/elements/all-transactions/values':
            return '{"count": 174480, "elementPerSecond": 0.28509316, "minDuration": 3, "maxDuration": 52860, "sumDuration": 27303545, "avgDuration": 156.48524, "minTTFB": 1, "maxTTFB": 24906, "sumTTFB": 5662539, "avgTTFB": 32.4538, "sumDownloadedBytes": 2927963658, "downloadedBytesPerSecond": 4784.1724, "successCount": 172856, "successPerSecond": 0.28243962, "successRate": 99.06924, "failureCount": 1624, "failurePerSecond": 0.0026535494, "failureRate": 0.9307657, "percentile50": 31.0, "percentile90": 214.0, "percentile95": 635.0, "percentile99": 2599.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/8725e0cd-92a4-4105-803b-86433851fcfc/elements/0ca05bf2-4ba9-483e-9041-c9ac38ecb8b8/values':
            return '{"count": 43605, "elementPerSecond": 0.07124878, "minDuration": 14, "maxDuration": 52860, "sumDuration": 7772822, "avgDuration": 178.2553, "minTTFB": 2, "maxTTFB": 20923, "sumTTFB": 1288658, "avgTTFB": 29.552986, "sumDownloadedBytes": 1034658589, "downloadedBytesPerSecond": 1690.5897, "successCount": 43069, "successPerSecond": 0.07037298, "successRate": 98.77078, "failureCount": 536, "failurePerSecond": 0.000875802, "failureRate": 1.2292168, "percentile50": 31.0, "percentile90": 214.0, "percentile95": 635.0, "percentile99": 2599.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/8725e0cd-92a4-4105-803b-86433851fcfc/elements/062b066f-e7c0-4e67-9a08-65d6494fecc9/values':
            return '{"count": 43621, "elementPerSecond": 0.07127492, "minDuration": 13, "maxDuration": 42019, "sumDuration": 12234526, "avgDuration": 280.4733, "minTTFB": 1, "maxTTFB": 24906, "sumTTFB": 1459595, "avgTTFB": 33.460835, "sumDownloadedBytes": 1028486703, "downloadedBytesPerSecond": 1680.505, "successCount": 43070, "successPerSecond": 0.070374615, "successRate": 98.73685, "failureCount": 551, "failurePerSecond": 0.0009003114, "failureRate": 1.2631531, "percentile50": 78.0, "percentile90": 360.0, "percentile95": 878.0, "percentile99": 3572.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/8725e0cd-92a4-4105-803b-86433851fcfc/elements/all-transactions/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 5120000, "AVG_TTFB": 18.792465, "ERRORS": 0, "AVG_DURATION": 93.94863, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1307, "MAX_DURATION": 9529, "ELEMENTS_PER_SECOND": 0.28515625, "THROUGHPUT": 5070.547, "COUNT": 1460}, {"from": 5120000, "to": 10240000, "AVG_TTFB": 36.00755, "ERRORS": 0, "AVG_DURATION": 216.47289, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4731, "MAX_DURATION": 17544, "ELEMENTS_PER_SECOND": 0.2845703, "THROUGHPUT": 4800.054, "COUNT": 1457}, {"from": 10240000, "to": 15360000, "AVG_TTFB": 30.880136, "ERRORS": 0, "AVG_DURATION": 135.83562, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4742, "MAX_DURATION": 9271, "ELEMENTS_PER_SECOND": 0.28515625, "THROUGHPUT": 4817.6484, "COUNT": 1460}, {"from": 15360000, "to": 20480000, "AVG_TTFB": 20.534107, "ERRORS": 0, "AVG_DURATION": 102.152794, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1869, "MAX_DURATION": 8345, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4903.9595, "COUNT": 1466}, {"from": 20480000, "to": 25600000, "AVG_TTFB": 28.82204, "ERRORS": 0, "AVG_DURATION": 151.60507, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 5482, "MAX_DURATION": 17162, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4824.918, "COUNT": 1461}, {"from": 25600000, "to": 30720000, "AVG_TTFB": 22.601908, "ERRORS": 0, "AVG_DURATION": 97.944786, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2794, "MAX_DURATION": 5197, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4833.837, "COUNT": 1467}, {"from": 30720000, "to": 35840000, "AVG_TTFB": 15.713798, "ERRORS": 0, "AVG_DURATION": 66.33128, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1223, "MAX_DURATION": 1440, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4832.357, "COUNT": 1464}, {"from": 35840000, "to": 40960000, "AVG_TTFB": 22.360825, "ERRORS": 0, "AVG_DURATION": 209.27835, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2298, "MAX_DURATION": 28220, "ELEMENTS_PER_SECOND": 0.2841797, "THROUGHPUT": 4868.125, "COUNT": 1455}, {"from": 40960000, "to": 46080000, "AVG_TTFB": 21.965305, "ERRORS": 0, "AVG_DURATION": 97.43537, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 10, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 4070, "MAX_DURATION": 8717, "ELEMENTS_PER_SECOND": 0.28710938, "THROUGHPUT": 4846.5195, "COUNT": 1470}, {"from": 46080000, "to": 51200000, "AVG_TTFB": 86.498955, "ERRORS": 0, "AVG_DURATION": 412.34732, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 10385, "MAX_DURATION": 13880, "ELEMENTS_PER_SECOND": 0.2794922, "THROUGHPUT": 4742.8926, "COUNT": 1431}, {"from": 51200000, "to": 56320000, "AVG_TTFB": 54.28522, "ERRORS": 70, "AVG_DURATION": 273.1395, "ERRORS_PER_SECOND": 0.013671875, "MIN_DURATION": 11, "ERROR_RATE": 4.8342543, "MIN_TTFB": 1, "MAX_TTFB": 9447, "MAX_DURATION": 23982, "ELEMENTS_PER_SECOND": 0.2828125, "THROUGHPUT": 4515.2344, "COUNT": 1448}, {"from": 56320000, "to": 61440000, "AVG_TTFB": 88.80139, "ERRORS": 137, "AVG_DURATION": 344.55902, "ERRORS_PER_SECOND": 0.026757812, "MIN_DURATION": 4, "ERROR_RATE": 9.513889, "MIN_TTFB": 1, "MAX_TTFB": 12816, "MAX_DURATION": 16352, "ELEMENTS_PER_SECOND": 0.28125, "THROUGHPUT": 4218.4023, "COUNT": 1440}, {"from": 61440000, "to": 66560000, "AVG_TTFB": 26.461487, "ERRORS": 0, "AVG_DURATION": 109.68303, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 6824, "MAX_DURATION": 16425, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4905.996, "COUNT": 1467}, {"from": 66560000, "to": 71680000, "AVG_TTFB": 25.023256, "ERRORS": 78, "AVG_DURATION": 114.425446, "ERRORS_PER_SECOND": 0.015234375, "MIN_DURATION": 4, "ERROR_RATE": 5.3351574, "MIN_TTFB": 2, "MAX_TTFB": 1957, "MAX_DURATION": 6611, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4587.773, "COUNT": 1462}, {"from": 71680000, "to": 76800000, "AVG_TTFB": 50.111034, "ERRORS": 0, "AVG_DURATION": 244.5069, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6071, "MAX_DURATION": 18217, "ELEMENTS_PER_SECOND": 0.28320312, "THROUGHPUT": 4795.7314, "COUNT": 1450}, {"from": 76800000, "to": 81920000, "AVG_TTFB": 31.90747, "ERRORS": 0, "AVG_DURATION": 178.05003, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3481, "MAX_DURATION": 8585, "ELEMENTS_PER_SECOND": 0.28496093, "THROUGHPUT": 4812.4634, "COUNT": 1459}, {"from": 81920000, "to": 87040000, "AVG_TTFB": 18.347885, "ERRORS": 0, "AVG_DURATION": 78.092766, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 955, "MAX_DURATION": 3836, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4896.2764, "COUNT": 1466}, {"from": 87040000, "to": 92160000, "AVG_TTFB": 20.260544, "ERRORS": 0, "AVG_DURATION": 103.97959, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1158, "MAX_DURATION": 21897, "ELEMENTS_PER_SECOND": 0.28710938, "THROUGHPUT": 4842.032, "COUNT": 1470}, {"from": 92160000, "to": 97280000, "AVG_TTFB": 15.030055, "ERRORS": 0, "AVG_DURATION": 66.068306, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1093, "MAX_DURATION": 1942, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4832.4253, "COUNT": 1464}, {"from": 97280000, "to": 102400000, "AVG_TTFB": 29.262003, "ERRORS": 0, "AVG_DURATION": 147.17558, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3339, "MAX_DURATION": 12790, "ELEMENTS_PER_SECOND": 0.28476563, "THROUGHPUT": 4815.1836, "COUNT": 1458}, {"from": 102400000, "to": 107520000, "AVG_TTFB": 14.768234, "ERRORS": 0, "AVG_DURATION": 87.28289, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1032, "MAX_DURATION": 5240, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4834.0493, "COUNT": 1467}, {"from": 107520000, "to": 112640000, "AVG_TTFB": 16.708559, "ERRORS": 0, "AVG_DURATION": 97.589, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1174, "MAX_DURATION": 15397, "ELEMENTS_PER_SECOND": 0.2875, "THROUGHPUT": 4912.656, "COUNT": 1472}, {"from": 112640000, "to": 117760000, "AVG_TTFB": 13.983005, "ERRORS": 0, "AVG_DURATION": 66.67709, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 310, "MAX_DURATION": 5283, "ELEMENTS_PER_SECOND": 0.2873047, "THROUGHPUT": 4845.1084, "COUNT": 1471}, {"from": 117760000, "to": 122880000, "AVG_TTFB": 16.342428, "ERRORS": 0, "AVG_DURATION": 70.47613, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1340, "MAX_DURATION": 2736, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4831.052, "COUNT": 1466}, {"from": 122880000, "to": 128000000, "AVG_TTFB": 25.046928, "ERRORS": 0, "AVG_DURATION": 226.30367, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1573, "MAX_DURATION": 43991, "ELEMENTS_PER_SECOND": 0.2830078, "THROUGHPUT": 4790.3506, "COUNT": 1449}, {"from": 128000000, "to": 133120000, "AVG_TTFB": 20.407104, "ERRORS": 0, "AVG_DURATION": 113.94809, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 767, "MAX_DURATION": 5530, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4900.702, "COUNT": 1464}, {"from": 133120000, "to": 138240000, "AVG_TTFB": 39.941822, "ERRORS": 0, "AVG_DURATION": 157.4976, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 10959, "MAX_DURATION": 12682, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4821.235, "COUNT": 1461}, {"from": 138240000, "to": 143360000, "AVG_TTFB": 63.052414, "ERRORS": 179, "AVG_DURATION": 274.31586, "ERRORS_PER_SECOND": 0.034960937, "MIN_DURATION": 4, "ERROR_RATE": 12.344828, "MIN_TTFB": 1, "MAX_TTFB": 17386, "MAX_DURATION": 18221, "ELEMENTS_PER_SECOND": 0.28320312, "THROUGHPUT": 4119.5054, "COUNT": 1450}, {"from": 143360000, "to": 148480000, "AVG_TTFB": 40.902473, "ERRORS": 0, "AVG_DURATION": 177.8228, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 8414, "MAX_DURATION": 11868, "ELEMENTS_PER_SECOND": 0.284375, "THROUGHPUT": 4804.1494, "COUNT": 1456}, {"from": 148480000, "to": 153600000, "AVG_TTFB": 22.346102, "ERRORS": 0, "AVG_DURATION": 110.59234, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3320, "MAX_DURATION": 3909, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4828.1895, "COUNT": 1462}, {"from": 153600000, "to": 158720000, "AVG_TTFB": 21.65306, "ERRORS": 43, "AVG_DURATION": 99.71973, "ERRORS_PER_SECOND": 0.008398438, "MIN_DURATION": 4, "ERROR_RATE": 2.9251702, "MIN_TTFB": 1, "MAX_TTFB": 1866, "MAX_DURATION": 8382, "ELEMENTS_PER_SECOND": 0.28710938, "THROUGHPUT": 4706.4844, "COUNT": 1470}, {"from": 158720000, "to": 163840000, "AVG_TTFB": 41.53022, "ERRORS": 0, "AVG_DURATION": 183.48627, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 11142, "MAX_DURATION": 12409, "ELEMENTS_PER_SECOND": 0.284375, "THROUGHPUT": 4808.758, "COUNT": 1456}, {"from": 163840000, "to": 168960000, "AVG_TTFB": 17.32353, "ERRORS": 0, "AVG_DURATION": 78.52394, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 992, "MAX_DURATION": 3542, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4835.8135, "COUNT": 1462}, {"from": 168960000, "to": 174080000, "AVG_TTFB": 21.495895, "ERRORS": 0, "AVG_DURATION": 90.73598, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1431, "MAX_DURATION": 2486, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4826.301, "COUNT": 1462}, {"from": 174080000, "to": 179200000, "AVG_TTFB": 37.910835, "ERRORS": 0, "AVG_DURATION": 156.50891, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 22591, "MAX_DURATION": 25594, "ELEMENTS_PER_SECOND": 0.28476563, "THROUGHPUT": 4873.056, "COUNT": 1458}, {"from": 179200000, "to": 184320000, "AVG_TTFB": 50.98972, "ERRORS": 0, "AVG_DURATION": 255.28787, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3453, "MAX_DURATION": 8512, "ELEMENTS_PER_SECOND": 0.28496093, "THROUGHPUT": 4814.885, "COUNT": 1459}, {"from": 184320000, "to": 189440000, "AVG_TTFB": 31.912027, "ERRORS": 0, "AVG_DURATION": 159.58351, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1791, "MAX_DURATION": 3782, "ELEMENTS_PER_SECOND": 0.2841797, "THROUGHPUT": 4802.6553, "COUNT": 1455}, {"from": 189440000, "to": 194560000, "AVG_TTFB": 16.02316, "ERRORS": 0, "AVG_DURATION": 74.52793, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1454, "MAX_DURATION": 7421, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4836.81, "COUNT": 1468}, {"from": 194560000, "to": 199680000, "AVG_TTFB": 23.393297, "ERRORS": 0, "AVG_DURATION": 170.18468, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1883, "MAX_DURATION": 27852, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4819.699, "COUNT": 1462}, {"from": 199680000, "to": 204800000, "AVG_TTFB": 25.98773, "ERRORS": 0, "AVG_DURATION": 115.69666, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 6737, "MAX_DURATION": 13599, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4902.027, "COUNT": 1467}, {"from": 204800000, "to": 209920000, "AVG_TTFB": 32.3852, "ERRORS": 0, "AVG_DURATION": 340.27454, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3759, "MAX_DURATION": 52860, "ELEMENTS_PER_SECOND": 0.2824219, "THROUGHPUT": 4775.369, "COUNT": 1446}, {"from": 209920000, "to": 215040000, "AVG_TTFB": 35.398212, "ERRORS": 0, "AVG_DURATION": 212.1183, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 20414, "MAX_DURATION": 23846, "ELEMENTS_PER_SECOND": 0.28398436, "THROUGHPUT": 4805.8555, "COUNT": 1454}, {"from": 215040000, "to": 220160000, "AVG_TTFB": 33.779545, "ERRORS": 0, "AVG_DURATION": 278.48584, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3789, "MAX_DURATION": 28459, "ELEMENTS_PER_SECOND": 0.28261718, "THROUGHPUT": 4791.2686, "COUNT": 1447}, {"from": 220160000, "to": 225280000, "AVG_TTFB": 28.98429, "ERRORS": 0, "AVG_DURATION": 157.1127, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1673, "MAX_DURATION": 9228, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4889.3237, "COUNT": 1464}, {"from": 225280000, "to": 230400000, "AVG_TTFB": 155.80421, "ERRORS": 213, "AVG_DURATION": 516.567, "ERRORS_PER_SECOND": 0.04160156, "MIN_DURATION": 4, "ERROR_RATE": 14.947369, "MIN_TTFB": 2, "MAX_TTFB": 24906, "MAX_DURATION": 26337, "ELEMENTS_PER_SECOND": 0.2783203, "THROUGHPUT": 3910.3108, "COUNT": 1425}, {"from": 230400000, "to": 235520000, "AVG_TTFB": 15.618789, "ERRORS": 0, "AVG_DURATION": 66.05854, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 375, "MAX_DURATION": 1382, "ELEMENTS_PER_SECOND": 0.28691405, "THROUGHPUT": 4850.914, "COUNT": 1469}, {"from": 235520000, "to": 240640000, "AVG_TTFB": 15.083844, "ERRORS": 0, "AVG_DURATION": 58.689163, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 717, "MAX_DURATION": 854, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4834.24, "COUNT": 1467}, {"from": 240640000, "to": 245760000, "AVG_TTFB": 41.34452, "ERRORS": 18, "AVG_DURATION": 124.06986, "ERRORS_PER_SECOND": 0.003515625, "MIN_DURATION": 4, "ERROR_RATE": 1.2328767, "MIN_TTFB": 2, "MAX_TTFB": 13238, "MAX_DURATION": 15755, "ELEMENTS_PER_SECOND": 0.28515625, "THROUGHPUT": 4771.6143, "COUNT": 1460}, {"from": 245760000, "to": 250880000, "AVG_TTFB": 86.37042, "ERRORS": 0, "AVG_DURATION": 325.68903, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 15264, "MAX_DURATION": 17409, "ELEMENTS_PER_SECOND": 0.28261718, "THROUGHPUT": 4853.1045, "COUNT": 1447}, {"from": 250880000, "to": 256000000, "AVG_TTFB": 17.491169, "ERRORS": 0, "AVG_DURATION": 80.9015, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 927, "MAX_DURATION": 4439, "ELEMENTS_PER_SECOND": 0.2875, "THROUGHPUT": 4845.287, "COUNT": 1472}, {"from": 256000000, "to": 261120000, "AVG_TTFB": 37.986988, "ERRORS": 0, "AVG_DURATION": 132.87192, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 7001, "MAX_DURATION": 8792, "ELEMENTS_PER_SECOND": 0.28515625, "THROUGHPUT": 4821.9844, "COUNT": 1460}, {"from": 261120000, "to": 266240000, "AVG_TTFB": 14.500341, "ERRORS": 0, "AVG_DURATION": 65.04846, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 503, "MAX_DURATION": 1556, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4831.6455, "COUNT": 1465}, {"from": 266240000, "to": 271360000, "AVG_TTFB": 28.877914, "ERRORS": 0, "AVG_DURATION": 182.4952, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3858, "MAX_DURATION": 13393, "ELEMENTS_PER_SECOND": 0.28476563, "THROUGHPUT": 4883.354, "COUNT": 1458}, {"from": 271360000, "to": 276480000, "AVG_TTFB": 44.949245, "ERRORS": 0, "AVG_DURATION": 241.15843, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3497, "MAX_DURATION": 6735, "ELEMENTS_PER_SECOND": 0.28476563, "THROUGHPUT": 4812.1196, "COUNT": 1458}, {"from": 276480000, "to": 281600000, "AVG_TTFB": 16.546757, "ERRORS": 0, "AVG_DURATION": 120.243004, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1432, "MAX_DURATION": 16258, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4825.885, "COUNT": 1465}, {"from": 281600000, "to": 286720000, "AVG_TTFB": 24.353909, "ERRORS": 0, "AVG_DURATION": 130.23799, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3297, "MAX_DURATION": 10322, "ELEMENTS_PER_SECOND": 0.28476563, "THROUGHPUT": 4820.863, "COUNT": 1458}, {"from": 286720000, "to": 291840000, "AVG_TTFB": 18.056591, "ERRORS": 0, "AVG_DURATION": 290.54108, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1382, "MAX_DURATION": 27016, "ELEMENTS_PER_SECOND": 0.2830078, "THROUGHPUT": 4795.2188, "COUNT": 1449}, {"from": 291840000, "to": 296960000, "AVG_TTFB": 20.734833, "ERRORS": 0, "AVG_DURATION": 115.32516, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5005, "MAX_DURATION": 19404, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4902.565, "COUNT": 1467}, {"from": 296960000, "to": 302080000, "AVG_TTFB": 32.284138, "ERRORS": 0, "AVG_DURATION": 250.89517, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3270, "MAX_DURATION": 32449, "ELEMENTS_PER_SECOND": 0.28320312, "THROUGHPUT": 4790.562, "COUNT": 1450}, {"from": 302080000, "to": 307200000, "AVG_TTFB": 36.849586, "ERRORS": 0, "AVG_DURATION": 171.28297, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3771, "MAX_DURATION": 5704, "ELEMENTS_PER_SECOND": 0.284375, "THROUGHPUT": 4809.876, "COUNT": 1456}, {"from": 307200000, "to": 312320000, "AVG_TTFB": 20.077604, "ERRORS": 0, "AVG_DURATION": 83.55684, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2124, "MAX_DURATION": 4893, "ELEMENTS_PER_SECOND": 0.28691405, "THROUGHPUT": 4841.924, "COUNT": 1469}, {"from": 312320000, "to": 317440000, "AVG_TTFB": 56.546333, "ERRORS": 211, "AVG_DURATION": 281.91977, "ERRORS_PER_SECOND": 0.04121094, "MIN_DURATION": 3, "ERROR_RATE": 14.591978, "MIN_TTFB": 1, "MAX_TTFB": 5528, "MAX_DURATION": 13408, "ELEMENTS_PER_SECOND": 0.2824219, "THROUGHPUT": 4014.428, "COUNT": 1446}, {"from": 317440000, "to": 322560000, "AVG_TTFB": 19.971312, "ERRORS": 0, "AVG_DURATION": 80.10997, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1366, "MAX_DURATION": 1915, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4836.332, "COUNT": 1464}, {"from": 322560000, "to": 327680000, "AVG_TTFB": 16.49557, "ERRORS": 0, "AVG_DURATION": 82.235855, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 10, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1314, "MAX_DURATION": 3538, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4838.237, "COUNT": 1467}, {"from": 327680000, "to": 332800000, "AVG_TTFB": 26.068447, "ERRORS": 12, "AVG_DURATION": 98.04038, "ERRORS_PER_SECOND": 0.00234375, "MIN_DURATION": 4, "ERROR_RATE": 0.8213552, "MIN_TTFB": 2, "MAX_TTFB": 2875, "MAX_DURATION": 4119, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4782.869, "COUNT": 1461}, {"from": 332800000, "to": 337920000, "AVG_TTFB": 29.023384, "ERRORS": 0, "AVG_DURATION": 203.09973, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2024, "MAX_DURATION": 21449, "ELEMENTS_PER_SECOND": 0.28398436, "THROUGHPUT": 4802.3984, "COUNT": 1454}, {"from": 337920000, "to": 343040000, "AVG_TTFB": 22.743519, "ERRORS": 0, "AVG_DURATION": 102.41951, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2581, "MAX_DURATION": 3612, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4901.2144, "COUNT": 1466}, {"from": 343040000, "to": 348160000, "AVG_TTFB": 29.03486, "ERRORS": 0, "AVG_DURATION": 135.23514, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3510, "MAX_DURATION": 11663, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4828.1, "COUNT": 1463}, {"from": 348160000, "to": 353280000, "AVG_TTFB": 14.856753, "ERRORS": 0, "AVG_DURATION": 65.823326, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 469, "MAX_DURATION": 1515, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4831.129, "COUNT": 1466}, {"from": 353280000, "to": 358400000, "AVG_TTFB": 37.30748, "ERRORS": 0, "AVG_DURATION": 165.45985, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 5877, "MAX_DURATION": 6092, "ELEMENTS_PER_SECOND": 0.2845703, "THROUGHPUT": 4815.2017, "COUNT": 1457}, {"from": 358400000, "to": 363520000, "AVG_TTFB": 21.879946, "ERRORS": 0, "AVG_DURATION": 90.93656, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2995, "MAX_DURATION": 3724, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4893.0674, "COUNT": 1466}, {"from": 363520000, "to": 368640000, "AVG_TTFB": 23.370068, "ERRORS": 0, "AVG_DURATION": 122.17823, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5693, "MAX_DURATION": 15643, "ELEMENTS_PER_SECOND": 0.28710938, "THROUGHPUT": 4835.17, "COUNT": 1470}, {"from": 368640000, "to": 373760000, "AVG_TTFB": 29.73069, "ERRORS": 0, "AVG_DURATION": 147.90704, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3536, "MAX_DURATION": 9254, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4818.8765, "COUNT": 1463}, {"from": 373760000, "to": 378880000, "AVG_TTFB": 16.719837, "ERRORS": 0, "AVG_DURATION": 79.40082, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 945, "MAX_DURATION": 6672, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4842.4756, "COUNT": 1467}, {"from": 378880000, "to": 384000000, "AVG_TTFB": 37.190346, "ERRORS": 0, "AVG_DURATION": 225.16414, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3430, "MAX_DURATION": 8389, "ELEMENTS_PER_SECOND": 0.28320312, "THROUGHPUT": 4794.089, "COUNT": 1450}, {"from": 384000000, "to": 389120000, "AVG_TTFB": 19.191547, "ERRORS": 0, "AVG_DURATION": 102.235176, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2183, "MAX_DURATION": 11840, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4900.1475, "COUNT": 1467}, {"from": 389120000, "to": 394240000, "AVG_TTFB": 26.783175, "ERRORS": 0, "AVG_DURATION": 137.09781, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1246, "MAX_DURATION": 3945, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4829.1533, "COUNT": 1462}, {"from": 394240000, "to": 399360000, "AVG_TTFB": 19.592617, "ERRORS": 0, "AVG_DURATION": 106.61108, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1529, "MAX_DURATION": 7649, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4825.9175, "COUNT": 1463}, {"from": 399360000, "to": 404480000, "AVG_TTFB": 43.76305, "ERRORS": 107, "AVG_DURATION": 189.18475, "ERRORS_PER_SECOND": 0.020898437, "MIN_DURATION": 4, "ERROR_RATE": 7.3489013, "MIN_TTFB": 2, "MAX_TTFB": 6095, "MAX_DURATION": 7501, "ELEMENTS_PER_SECOND": 0.284375, "THROUGHPUT": 4484.998, "COUNT": 1456}, {"from": 404480000, "to": 409600000, "AVG_TTFB": 42.562244, "ERRORS": 0, "AVG_DURATION": 116.26402, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 10651, "MAX_DURATION": 14101, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4886.9287, "COUNT": 1462}, {"from": 409600000, "to": 414720000, "AVG_TTFB": 22.406929, "ERRORS": 0, "AVG_DURATION": 91.652176, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4091, "MAX_DURATION": 8604, "ELEMENTS_PER_SECOND": 0.2875, "THROUGHPUT": 4853.6606, "COUNT": 1472}, {"from": 414720000, "to": 419840000, "AVG_TTFB": 34.738, "ERRORS": 29, "AVG_DURATION": 151.39986, "ERRORS_PER_SECOND": 0.0056640627, "MIN_DURATION": 4, "ERROR_RATE": 1.9890261, "MIN_TTFB": 2, "MAX_TTFB": 6970, "MAX_DURATION": 7430, "ELEMENTS_PER_SECOND": 0.28476563, "THROUGHPUT": 4737.073, "COUNT": 1458}, {"from": 419840000, "to": 424960000, "AVG_TTFB": 65.28315, "ERRORS": 0, "AVG_DURATION": 313.43854, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 7558, "MAX_DURATION": 15488, "ELEMENTS_PER_SECOND": 0.2828125, "THROUGHPUT": 4787.003, "COUNT": 1448}, {"from": 424960000, "to": 430080000, "AVG_TTFB": 16.89849, "ERRORS": 0, "AVG_DURATION": 102.954735, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 971, "MAX_DURATION": 7511, "ELEMENTS_PER_SECOND": 0.28476563, "THROUGHPUT": 4816.564, "COUNT": 1458}, {"from": 430080000, "to": 435200000, "AVG_TTFB": 21.637356, "ERRORS": 0, "AVG_DURATION": 103.78119, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1887, "MAX_DURATION": 5125, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4898.6206, "COUNT": 1467}, {"from": 435200000, "to": 440320000, "AVG_TTFB": 23.350817, "ERRORS": 0, "AVG_DURATION": 102.70232, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2149, "MAX_DURATION": 8537, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4835.8643, "COUNT": 1468}, {"from": 440320000, "to": 445440000, "AVG_TTFB": 24.75907, "ERRORS": 0, "AVG_DURATION": 117.26352, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3784, "MAX_DURATION": 5136, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4823.728, "COUNT": 1461}, {"from": 445440000, "to": 450560000, "AVG_TTFB": 18.356802, "ERRORS": 0, "AVG_DURATION": 87.99727, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1285, "MAX_DURATION": 3646, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4823.5146, "COUNT": 1463}, {"from": 450560000, "to": 455680000, "AVG_TTFB": 16.356705, "ERRORS": 0, "AVG_DURATION": 66.08441, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1283, "MAX_DURATION": 1680, "ELEMENTS_PER_SECOND": 0.28691405, "THROUGHPUT": 4905.177, "COUNT": 1469}, {"from": 455680000, "to": 460800000, "AVG_TTFB": 13.674119, "ERRORS": 0, "AVG_DURATION": 62.590786, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 380, "MAX_DURATION": 2002, "ELEMENTS_PER_SECOND": 0.28828126, "THROUGHPUT": 4859.8374, "COUNT": 1476}, {"from": 460800000, "to": 465920000, "AVG_TTFB": 30.055593, "ERRORS": 0, "AVG_DURATION": 140.27042, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 9, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3598, "MAX_DURATION": 5769, "ELEMENTS_PER_SECOND": 0.2845703, "THROUGHPUT": 4809.718, "COUNT": 1457}, {"from": 465920000, "to": 471040000, "AVG_TTFB": 18.14716, "ERRORS": 0, "AVG_DURATION": 103.479805, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1157, "MAX_DURATION": 11373, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4821.8813, "COUNT": 1461}, {"from": 471040000, "to": 476160000, "AVG_TTFB": 15.849455, "ERRORS": 0, "AVG_DURATION": 66.73774, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 10, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 513, "MAX_DURATION": 1299, "ELEMENTS_PER_SECOND": 0.28671876, "THROUGHPUT": 4837.043, "COUNT": 1468}, {"from": 476160000, "to": 481280000, "AVG_TTFB": 120.06788, "ERRORS": 0, "AVG_DURATION": 500.62003, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 12777, "MAX_DURATION": 30984, "ELEMENTS_PER_SECOND": 0.27910155, "THROUGHPUT": 4800.9746, "COUNT": 1429}, {"from": 481280000, "to": 486400000, "AVG_TTFB": 14.914052, "ERRORS": 0, "AVG_DURATION": 66.386086, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1723, "MAX_DURATION": 2559, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4845.995, "COUNT": 1466}, {"from": 486400000, "to": 491520000, "AVG_TTFB": 31.478796, "ERRORS": 64, "AVG_DURATION": 123.354996, "ERRORS_PER_SECOND": 0.0125, "MIN_DURATION": 5, "ERROR_RATE": 4.377565, "MIN_TTFB": 2, "MAX_TTFB": 6537, "MAX_DURATION": 7871, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4633.6025, "COUNT": 1462}, {"from": 491520000, "to": 496640000, "AVG_TTFB": 17.615541, "ERRORS": 0, "AVG_DURATION": 70.49557, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3129, "MAX_DURATION": 3185, "ELEMENTS_PER_SECOND": 0.28652343, "THROUGHPUT": 4835.172, "COUNT": 1467}, {"from": 496640000, "to": 501760000, "AVG_TTFB": 34.54814, "ERRORS": 27, "AVG_DURATION": 216.08734, "ERRORS_PER_SECOND": 0.0052734376, "MIN_DURATION": 11, "ERROR_RATE": 1.8569463, "MIN_TTFB": 2, "MAX_TTFB": 6892, "MAX_DURATION": 15105, "ELEMENTS_PER_SECOND": 0.28398436, "THROUGHPUT": 4721.1543, "COUNT": 1454}, {"from": 501760000, "to": 506880000, "AVG_TTFB": 65.03377, "ERRORS": 106, "AVG_DURATION": 308.78085, "ERRORS_PER_SECOND": 0.020703126, "MIN_DURATION": 4, "ERROR_RATE": 7.305307, "MIN_TTFB": 2, "MAX_TTFB": 12597, "MAX_DURATION": 14332, "ELEMENTS_PER_SECOND": 0.28339845, "THROUGHPUT": 4401.363, "COUNT": 1451}, {"from": 506880000, "to": 512000000, "AVG_TTFB": 29.884642, "ERRORS": 0, "AVG_DURATION": 119.80137, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 10439, "MAX_DURATION": 14159, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4834.9067, "COUNT": 1465}, {"from": 512000000, "to": 517120000, "AVG_TTFB": 16.740767, "ERRORS": 0, "AVG_DURATION": 75.34952, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 926, "MAX_DURATION": 3829, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4831.1274, "COUNT": 1462}, {"from": 517120000, "to": 522240000, "AVG_TTFB": 15.099795, "ERRORS": 0, "AVG_DURATION": 66.72591, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 794, "MAX_DURATION": 1571, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4833.1616, "COUNT": 1463}, {"from": 522240000, "to": 527360000, "AVG_TTFB": 15.701158, "ERRORS": 0, "AVG_DURATION": 74.82913, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1780, "MAX_DURATION": 5222, "ELEMENTS_PER_SECOND": 0.28691405, "THROUGHPUT": 4908.394, "COUNT": 1469}, {"from": 527360000, "to": 532480000, "AVG_TTFB": 34.494198, "ERRORS": 0, "AVG_DURATION": 123.03277, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 6854, "MAX_DURATION": 11098, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4832.767, "COUNT": 1465}, {"from": 532480000, "to": 537600000, "AVG_TTFB": 17.457026, "ERRORS": 0, "AVG_DURATION": 66.24011, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2254, "MAX_DURATION": 2254, "ELEMENTS_PER_SECOND": 0.28632814, "THROUGHPUT": 4837.9062, "COUNT": 1466}, {"from": 537600000, "to": 542720000, "AVG_TTFB": 16.632082, "ERRORS": 0, "AVG_DURATION": 70.69966, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1278, "MAX_DURATION": 3149, "ELEMENTS_PER_SECOND": 0.2861328, "THROUGHPUT": 4829.161, "COUNT": 1465}, {"from": 542720000, "to": 547840000, "AVG_TTFB": 16.103283, "ERRORS": 0, "AVG_DURATION": 85.03967, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1140, "MAX_DURATION": 5063, "ELEMENTS_PER_SECOND": 0.28554687, "THROUGHPUT": 4890.3384, "COUNT": 1462}, {"from": 547840000, "to": 552960000, "AVG_TTFB": 39.18082, "ERRORS": 0, "AVG_DURATION": 206.21645, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 10970, "MAX_DURATION": 20867, "ELEMENTS_PER_SECOND": 0.28515625, "THROUGHPUT": 4838.1094, "COUNT": 1460}, {"from": 552960000, "to": 558080000, "AVG_TTFB": 25.90878, "ERRORS": 0, "AVG_DURATION": 113.60082, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2108, "MAX_DURATION": 4239, "ELEMENTS_PER_SECOND": 0.28476563, "THROUGHPUT": 4812.4995, "COUNT": 1458}, {"from": 558080000, "to": 563200000, "AVG_TTFB": 33.203842, "ERRORS": 0, "AVG_DURATION": 150.09677, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3541, "MAX_DURATION": 6434, "ELEMENTS_PER_SECOND": 0.2845703, "THROUGHPUT": 4817.786, "COUNT": 1457}, {"from": 563200000, "to": 568320000, "AVG_TTFB": 87.936584, "ERRORS": 0, "AVG_DURATION": 456.79932, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 14957, "MAX_DURATION": 15408, "ELEMENTS_PER_SECOND": 0.28027344, "THROUGHPUT": 4758.5215, "COUNT": 1435}, {"from": 568320000, "to": 573440000, "AVG_TTFB": 33.52296, "ERRORS": 0, "AVG_DURATION": 151.72447, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 17701, "MAX_DURATION": 18980, "ELEMENTS_PER_SECOND": 0.28496093, "THROUGHPUT": 4877.2124, "COUNT": 1459}, {"from": 573440000, "to": 578560000, "AVG_TTFB": 126.09747, "ERRORS": 306, "AVG_DURATION": 553.04767, "ERRORS_PER_SECOND": 0.059765626, "MIN_DURATION": 4, "ERROR_RATE": 21.458626, "MIN_TTFB": 1, "MAX_TTFB": 20269, "MAX_DURATION": 20674, "ELEMENTS_PER_SECOND": 0.27851564, "THROUGHPUT": 3542.3882, "COUNT": 1426}, {"from": 578560000, "to": 583680000, "AVG_TTFB": 52.688313, "ERRORS": 0, "AVG_DURATION": 129.17157, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 20923, "MAX_DURATION": 21008, "ELEMENTS_PER_SECOND": 0.2857422, "THROUGHPUT": 4837.8535, "COUNT": 1463}, {"from": 583680000, "to": 588800000, "AVG_TTFB": 15.182377, "ERRORS": 24, "AVG_DURATION": 68.38661, "ERRORS_PER_SECOND": 0.0046875, "MIN_DURATION": 5, "ERROR_RATE": 1.6393442, "MIN_TTFB": 2, "MAX_TTFB": 1005, "MAX_DURATION": 3206, "ELEMENTS_PER_SECOND": 0.2859375, "THROUGHPUT": 4749.2227, "COUNT": 1464}, {"from": 588800000, "to": 593920000, "AVG_TTFB": 80.18763, "ERRORS": 0, "AVG_DURATION": 396.54272, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 19452, "MAX_DURATION": 32052, "ELEMENTS_PER_SECOND": 0.28105468, "THROUGHPUT": 4826.269, "COUNT": 1439}, {"from": 593920000, "to": 599040000, "AVG_TTFB": 44.57123, "ERRORS": 0, "AVG_DURATION": 207.50958, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 10186, "MAX_DURATION": 26057, "ELEMENTS_PER_SECOND": 0.28515625, "THROUGHPUT": 4820.952, "COUNT": 1460}, {"from": 599040000, "to": 604160000, "AVG_TTFB": 39.375, "ERRORS": 0, "AVG_DURATION": 185.27953, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 5995, "MAX_DURATION": 10243, "ELEMENTS_PER_SECOND": 0.284375, "THROUGHPUT": 4810.89, "COUNT": 1456}, {"from": 604160000, "to": 609280000, "AVG_TTFB": 23.138947, "ERRORS": 0, "AVG_DURATION": 110.50719, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4255, "MAX_DURATION": 5121, "ELEMENTS_PER_SECOND": 0.28535157, "THROUGHPUT": 4818.5854, "COUNT": 1461}, {"from": 609280000, "to": 614400000, "AVG_TTFB": 20.254477, "ERRORS": 0, "AVG_DURATION": 162.34015, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1238, "MAX_DURATION": 15647, "ELEMENTS_PER_SECOND": 0.15273437, "THROUGHPUT": 2612.0217, "COUNT": 782}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/8725e0cd-92a4-4105-803b-86433851fcfc/elements/50ddd53f-4a4b-4bf8-a6b3-7a262d3d6cef/values':
            return '{"count": 43649, "elementPerSecond": 0.071320675, "minDuration": 3, "maxDuration": 43991, "sumDuration": 5425624, "avgDuration": 124.30122, "minTTFB": 1, "maxTTFB": 20266, "sumTTFB": 1043713, "avgTTFB": 23.911499, "sumDownloadedBytes": 864818366, "downloadedBytesPerSecond": 1413.0778, "successCount": 43112, "successPerSecond": 0.070443235, "successRate": 98.76973, "failureCount": 537, "failurePerSecond": 0.00087743596, "failureRate": 1.2302687, "percentile50": 18.0, "percentile90": 95.0, "percentile95": 290.0, "percentile99": 1772.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/8725e0cd-92a4-4105-803b-86433851fcfc/elements/b3e1f7c6-a4ef-438e-a3bb-8ffa1cb4a85e/values':
            return '{"count": 43605, "elementPerSecond": 0.07124878, "minDuration": 41, "maxDuration": 2254, "sumDuration": 1870573, "avgDuration": 42.898132, "minTTFB": 41, "maxTTFB": 2254, "sumTTFB": 1870573, "avgTTFB": 42.898132, "sumDownloadedBytes": 0, "downloadedBytesPerSecond": 0.0, "successCount": 43605, "successPerSecond": 0.07124878, "successRate": 100.0, "failureCount": 0, "failurePerSecond": 0.0, "failureRate": 0.0, "percentile50": 42.0, "percentile90": 43.0, "percentile95": 43.0, "percentile99": 44.0}'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/8725e0cd-92a4-4105-803b-86433851fcfc/elements/062b066f-e7c0-4e67-9a08-65d6494fecc9/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 5120000, "AVG_TTFB": 14.583562, "ERRORS": 0, "AVG_DURATION": 157.8548, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 62, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1307, "MAX_DURATION": 6908, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1702.8304, "COUNT": 365}, {"from": 5120000, "to": 10240000, "AVG_TTFB": 29.775343, "ERRORS": 0, "AVG_DURATION": 307.26575, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1928, "MAX_DURATION": 11804, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.1676, "COUNT": 365}, {"from": 10240000, "to": 15360000, "AVG_TTFB": 26.335165, "ERRORS": 0, "AVG_DURATION": 244.0934, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 27, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3496, "MAX_DURATION": 5288, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.5258, "COUNT": 364}, {"from": 15360000, "to": 20480000, "AVG_TTFB": 10.825613, "ERRORS": 0, "AVG_DURATION": 193.45505, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 35, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 957, "MAX_DURATION": 8345, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.6921, "COUNT": 367}, {"from": 20480000, "to": 25600000, "AVG_TTFB": 27.120548, "ERRORS": 0, "AVG_DURATION": 195.12329, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 26, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3295, "MAX_DURATION": 6424, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.1184, "COUNT": 365}, {"from": 25600000, "to": 30720000, "AVG_TTFB": 9.603826, "ERRORS": 0, "AVG_DURATION": 162.94536, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 25, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 906, "MAX_DURATION": 5197, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.8496, "COUNT": 366}, {"from": 30720000, "to": 35840000, "AVG_TTFB": 5.1498637, "ERRORS": 0, "AVG_DURATION": 119.47684, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 56, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 460, "MAX_DURATION": 999, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.5856, "COUNT": 367}, {"from": 35840000, "to": 40960000, "AVG_TTFB": 14.371901, "ERRORS": 0, "AVG_DURATION": 373.50412, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 57, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2298, "MAX_DURATION": 28220, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1690.4066, "COUNT": 363}, {"from": 40960000, "to": 46080000, "AVG_TTFB": 14.842391, "ERRORS": 0, "AVG_DURATION": 177.77989, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 57, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1363, "MAX_DURATION": 3214, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1712.2504, "COUNT": 368}, {"from": 46080000, "to": 51200000, "AVG_TTFB": 132.2521, "ERRORS": 0, "AVG_DURATION": 808.577, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 10385, "MAX_DURATION": 11847, "ELEMENTS_PER_SECOND": 0.069726564, "THROUGHPUT": 1661.5996, "COUNT": 357}, {"from": 51200000, "to": 56320000, "AVG_TTFB": 64.67956, "ERRORS": 23, "AVG_DURATION": 511.67404, "ERRORS_PER_SECOND": 0.0044921874, "MIN_DURATION": 28, "ERROR_RATE": 6.353591, "MIN_TTFB": 2, "MAX_TTFB": 5358, "MAX_DURATION": 23982, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1593.2229, "COUNT": 362}, {"from": 56320000, "to": 61440000, "AVG_TTFB": 162.125, "ERRORS": 46, "AVG_DURATION": 663.48334, "ERRORS_PER_SECOND": 0.008984375, "MIN_DURATION": 27, "ERROR_RATE": 12.777778, "MIN_TTFB": 2, "MAX_TTFB": 12816, "MAX_DURATION": 16352, "ELEMENTS_PER_SECOND": 0.0703125, "THROUGHPUT": 1490.8594, "COUNT": 360}, {"from": 61440000, "to": 66560000, "AVG_TTFB": 41.0109, "ERRORS": 0, "AVG_DURATION": 226.52043, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 29, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6824, "MAX_DURATION": 16425, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.5969, "COUNT": 367}, {"from": 66560000, "to": 71680000, "AVG_TTFB": 20.251366, "ERRORS": 26, "AVG_DURATION": 198.84427, "ERRORS_PER_SECOND": 0.005078125, "MIN_DURATION": 14, "ERROR_RATE": 7.103825, "MIN_TTFB": 2, "MAX_TTFB": 1721, "MAX_DURATION": 3883, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1602.2043, "COUNT": 366}, {"from": 71680000, "to": 76800000, "AVG_TTFB": 57.229282, "ERRORS": 0, "AVG_DURATION": 451.5442, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 36, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3426, "MAX_DURATION": 18217, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1684.0449, "COUNT": 362}, {"from": 76800000, "to": 81920000, "AVG_TTFB": 37.414837, "ERRORS": 0, "AVG_DURATION": 350.3187, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3481, "MAX_DURATION": 8585, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.5637, "COUNT": 364}, {"from": 81920000, "to": 87040000, "AVG_TTFB": 11.256131, "ERRORS": 0, "AVG_DURATION": 140.55042, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 44, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 955, "MAX_DURATION": 1098, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.7351, "COUNT": 367}, {"from": 87040000, "to": 92160000, "AVG_TTFB": 10.6403265, "ERRORS": 0, "AVG_DURATION": 159.13351, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1158, "MAX_DURATION": 1833, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.4652, "COUNT": 367}, {"from": 92160000, "to": 97280000, "AVG_TTFB": 7.160763, "ERRORS": 0, "AVG_DURATION": 132.40327, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 50, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1093, "MAX_DURATION": 1942, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.4684, "COUNT": 367}, {"from": 97280000, "to": 102400000, "AVG_TTFB": 20.137363, "ERRORS": 0, "AVG_DURATION": 284.32144, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 55, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1110, "MAX_DURATION": 12790, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.5449, "COUNT": 364}, {"from": 102400000, "to": 107520000, "AVG_TTFB": 5.773842, "ERRORS": 0, "AVG_DURATION": 124.01635, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 395, "MAX_DURATION": 1291, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.5238, "COUNT": 367}, {"from": 107520000, "to": 112640000, "AVG_TTFB": 7.7384195, "ERRORS": 0, "AVG_DURATION": 144.47139, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 727, "MAX_DURATION": 2095, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.9277, "COUNT": 367}, {"from": 112640000, "to": 117760000, "AVG_TTFB": 3.9322493, "ERRORS": 0, "AVG_DURATION": 117.09214, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 118, "MAX_DURATION": 458, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1716.7301, "COUNT": 369}, {"from": 117760000, "to": 122880000, "AVG_TTFB": 9.8, "ERRORS": 0, "AVG_DURATION": 129.70412, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 50, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1340, "MAX_DURATION": 2736, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.3079, "COUNT": 365}, {"from": 122880000, "to": 128000000, "AVG_TTFB": 19.331491, "ERRORS": 0, "AVG_DURATION": 397.74033, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 38, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1573, "MAX_DURATION": 32750, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1684.5258, "COUNT": 362}, {"from": 128000000, "to": 133120000, "AVG_TTFB": 18.692934, "ERRORS": 0, "AVG_DURATION": 201.32065, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 767, "MAX_DURATION": 5530, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1713.5469, "COUNT": 368}, {"from": 133120000, "to": 138240000, "AVG_TTFB": 40.043957, "ERRORS": 0, "AVG_DURATION": 293.87088, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3715, "MAX_DURATION": 12682, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.7891, "COUNT": 364}, {"from": 138240000, "to": 143360000, "AVG_TTFB": 59.03591, "ERRORS": 61, "AVG_DURATION": 427.98065, "ERRORS_PER_SECOND": 0.011914062, "MIN_DURATION": 15, "ERROR_RATE": 16.850828, "MIN_TTFB": 2, "MAX_TTFB": 3035, "MAX_DURATION": 11457, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1450.3988, "COUNT": 362}, {"from": 143360000, "to": 148480000, "AVG_TTFB": 37.65934, "ERRORS": 0, "AVG_DURATION": 342.25275, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 44, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3989, "MAX_DURATION": 11482, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.1843, "COUNT": 364}, {"from": 148480000, "to": 153600000, "AVG_TTFB": 13.071038, "ERRORS": 0, "AVG_DURATION": 199.08197, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 39, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 783, "MAX_DURATION": 3841, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.7258, "COUNT": 366}, {"from": 153600000, "to": 158720000, "AVG_TTFB": 12.877384, "ERRORS": 15, "AVG_DURATION": 198.9891, "ERRORS_PER_SECOND": 0.0029296875, "MIN_DURATION": 44, "ERROR_RATE": 4.0871935, "MIN_TTFB": 1, "MAX_TTFB": 1126, "MAX_DURATION": 8382, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1652.2874, "COUNT": 367}, {"from": 158720000, "to": 163840000, "AVG_TTFB": 38.774727, "ERRORS": 0, "AVG_DURATION": 334.00824, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 28, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3658, "MAX_DURATION": 7993, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.4766, "COUNT": 364}, {"from": 163840000, "to": 168960000, "AVG_TTFB": 7.8579235, "ERRORS": 0, "AVG_DURATION": 152.89072, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 747, "MAX_DURATION": 3045, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.864, "COUNT": 366}, {"from": 168960000, "to": 174080000, "AVG_TTFB": 14.619178, "ERRORS": 0, "AVG_DURATION": 173.23288, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 33, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1248, "MAX_DURATION": 2486, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.2504, "COUNT": 365}, {"from": 174080000, "to": 179200000, "AVG_TTFB": 74.568306, "ERRORS": 0, "AVG_DURATION": 283.89343, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 22591, "MAX_DURATION": 25249, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.9613, "COUNT": 366}, {"from": 179200000, "to": 184320000, "AVG_TTFB": 71.92055, "ERRORS": 0, "AVG_DURATION": 546.7205, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 30, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2306, "MAX_DURATION": 8243, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.0804, "COUNT": 365}, {"from": 184320000, "to": 189440000, "AVG_TTFB": 25.651098, "ERRORS": 0, "AVG_DURATION": 291.77197, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 32, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 423, "MAX_DURATION": 3782, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.6058, "COUNT": 364}, {"from": 189440000, "to": 194560000, "AVG_TTFB": 7.453552, "ERRORS": 0, "AVG_DURATION": 117.81148, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 33, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 860, "MAX_DURATION": 1338, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.0618, "COUNT": 366}, {"from": 194560000, "to": 199680000, "AVG_TTFB": 14.31694, "ERRORS": 0, "AVG_DURATION": 176.09563, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1883, "MAX_DURATION": 2599, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.0625, "COUNT": 366}, {"from": 199680000, "to": 204800000, "AVG_TTFB": 25.002718, "ERRORS": 0, "AVG_DURATION": 190.78804, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 29, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6737, "MAX_DURATION": 6903, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1713.5378, "COUNT": 368}, {"from": 204800000, "to": 209920000, "AVG_TTFB": 32.038887, "ERRORS": 0, "AVG_DURATION": 596.0333, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 36, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2493, "MAX_DURATION": 42019, "ELEMENTS_PER_SECOND": 0.0703125, "THROUGHPUT": 1675.4297, "COUNT": 360}, {"from": 209920000, "to": 215040000, "AVG_TTFB": 71.73003, "ERRORS": 0, "AVG_DURATION": 440.719, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 20414, "MAX_DURATION": 22773, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1689.0879, "COUNT": 363}, {"from": 215040000, "to": 220160000, "AVG_TTFB": 34.707184, "ERRORS": 0, "AVG_DURATION": 469.11325, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 35, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3789, "MAX_DURATION": 28459, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1684.5859, "COUNT": 362}, {"from": 220160000, "to": 225280000, "AVG_TTFB": 28.11444, "ERRORS": 0, "AVG_DURATION": 294.04904, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1673, "MAX_DURATION": 9228, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.884, "COUNT": 367}, {"from": 225280000, "to": 230400000, "AVG_TTFB": 247.55338, "ERRORS": 73, "AVG_DURATION": 1083.9438, "ERRORS_PER_SECOND": 0.014257813, "MIN_DURATION": 32, "ERROR_RATE": 20.505617, "MIN_TTFB": 2, "MAX_TTFB": 24906, "MAX_DURATION": 26337, "ELEMENTS_PER_SECOND": 0.06953125, "THROUGHPUT": 1383.1487, "COUNT": 356}, {"from": 230400000, "to": 235520000, "AVG_TTFB": 6.160763, "ERRORS": 0, "AVG_DURATION": 124.36512, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 196, "MAX_DURATION": 1382, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.0579, "COUNT": 367}, {"from": 235520000, "to": 240640000, "AVG_TTFB": 5.271739, "ERRORS": 0, "AVG_DURATION": 112.978264, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 30, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 351, "MAX_DURATION": 854, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1711.9437, "COUNT": 368}, {"from": 240640000, "to": 245760000, "AVG_TTFB": 9.192307, "ERRORS": 8, "AVG_DURATION": 177.98627, "ERRORS_PER_SECOND": 0.0015625, "MIN_DURATION": 14, "ERROR_RATE": 2.1978023, "MIN_TTFB": 2, "MAX_TTFB": 1415, "MAX_DURATION": 15755, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1674.0414, "COUNT": 364}, {"from": 245760000, "to": 250880000, "AVG_TTFB": 151.34254, "ERRORS": 0, "AVG_DURATION": 551.6188, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 28, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 15264, "MAX_DURATION": 15425, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1685.5231, "COUNT": 362}, {"from": 250880000, "to": 256000000, "AVG_TTFB": 8.622283, "ERRORS": 0, "AVG_DURATION": 152.10869, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 741, "MAX_DURATION": 2343, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1711.9515, "COUNT": 368}, {"from": 256000000, "to": 261120000, "AVG_TTFB": 41.030136, "ERRORS": 0, "AVG_DURATION": 271.8137, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 62, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 7001, "MAX_DURATION": 8792, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.2246, "COUNT": 365}, {"from": 261120000, "to": 266240000, "AVG_TTFB": 4.773224, "ERRORS": 0, "AVG_DURATION": 121.25956, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 503, "MAX_DURATION": 1105, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.7625, "COUNT": 366}, {"from": 266240000, "to": 271360000, "AVG_TTFB": 25.29589, "ERRORS": 0, "AVG_DURATION": 371.6137, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 47, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1946, "MAX_DURATION": 13066, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1699.4531, "COUNT": 365}, {"from": 271360000, "to": 276480000, "AVG_TTFB": 53.714287, "ERRORS": 0, "AVG_DURATION": 414.8132, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3497, "MAX_DURATION": 6735, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.6355, "COUNT": 364}, {"from": 276480000, "to": 281600000, "AVG_TTFB": 4.2520547, "ERRORS": 0, "AVG_DURATION": 118.68219, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 27, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 282, "MAX_DURATION": 2177, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.4574, "COUNT": 365}, {"from": 281600000, "to": 286720000, "AVG_TTFB": 20.591782, "ERRORS": 0, "AVG_DURATION": 225.1452, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 62, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1197, "MAX_DURATION": 7869, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.4133, "COUNT": 365}, {"from": 286720000, "to": 291840000, "AVG_TTFB": 10.461326, "ERRORS": 0, "AVG_DURATION": 216.60774, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 64, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1382, "MAX_DURATION": 26780, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1684.5621, "COUNT": 362}, {"from": 291840000, "to": 296960000, "AVG_TTFB": 24.105978, "ERRORS": 0, "AVG_DURATION": 176.76358, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 30, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5005, "MAX_DURATION": 5229, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1713.5387, "COUNT": 368}, {"from": 296960000, "to": 302080000, "AVG_TTFB": 43.443214, "ERRORS": 0, "AVG_DURATION": 381.02493, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 28, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3270, "MAX_DURATION": 20452, "ELEMENTS_PER_SECOND": 0.07050781, "THROUGHPUT": 1679.9156, "COUNT": 361}, {"from": 302080000, "to": 307200000, "AVG_TTFB": 41.354397, "ERRORS": 0, "AVG_DURATION": 294.7967, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3267, "MAX_DURATION": 3668, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.8633, "COUNT": 364}, {"from": 307200000, "to": 312320000, "AVG_TTFB": 18.460491, "ERRORS": 0, "AVG_DURATION": 153.27792, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2124, "MAX_DURATION": 2408, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.5879, "COUNT": 367}, {"from": 312320000, "to": 317440000, "AVG_TTFB": 45.41989, "ERRORS": 71, "AVG_DURATION": 505.21823, "ERRORS_PER_SECOND": 0.013867187, "MIN_DURATION": 15, "ERROR_RATE": 19.61326, "MIN_TTFB": 2, "MAX_TTFB": 1935, "MAX_DURATION": 8652, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1407.406, "COUNT": 362}, {"from": 317440000, "to": 322560000, "AVG_TTFB": 16.333334, "ERRORS": 0, "AVG_DURATION": 157.40164, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 30, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1366, "MAX_DURATION": 1915, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.5023, "COUNT": 366}, {"from": 322560000, "to": 327680000, "AVG_TTFB": 8.797814, "ERRORS": 0, "AVG_DURATION": 151.0601, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 29, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1314, "MAX_DURATION": 3538, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.7168, "COUNT": 366}, {"from": 327680000, "to": 332800000, "AVG_TTFB": 25.076502, "ERRORS": 4, "AVG_DURATION": 194.56557, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 14, "ERROR_RATE": 1.0928962, "MIN_TTFB": 2, "MAX_TTFB": 2875, "MAX_DURATION": 4119, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1686.555, "COUNT": 366}, {"from": 332800000, "to": 337920000, "AVG_TTFB": 30.490358, "ERRORS": 0, "AVG_DURATION": 415.62534, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 56, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2024, "MAX_DURATION": 21449, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1688.891, "COUNT": 363}, {"from": 337920000, "to": 343040000, "AVG_TTFB": 25.671196, "ERRORS": 0, "AVG_DURATION": 207.71739, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 32, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2581, "MAX_DURATION": 3612, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1713.3239, "COUNT": 368}, {"from": 343040000, "to": 348160000, "AVG_TTFB": 27.632877, "ERRORS": 0, "AVG_DURATION": 251.74794, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 44, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3510, "MAX_DURATION": 3982, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.2566, "COUNT": 365}, {"from": 348160000, "to": 353280000, "AVG_TTFB": 5.109589, "ERRORS": 0, "AVG_DURATION": 118.89863, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 62, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 374, "MAX_DURATION": 1068, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.2039, "COUNT": 365}, {"from": 353280000, "to": 358400000, "AVG_TTFB": 43.770493, "ERRORS": 0, "AVG_DURATION": 299.55737, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 28, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5877, "MAX_DURATION": 6092, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.7957, "COUNT": 366}, {"from": 358400000, "to": 363520000, "AVG_TTFB": 20.188524, "ERRORS": 0, "AVG_DURATION": 179.32513, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2995, "MAX_DURATION": 3724, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1704.1989, "COUNT": 366}, {"from": 363520000, "to": 368640000, "AVG_TTFB": 4.1389647, "ERRORS": 0, "AVG_DURATION": 147.54495, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 62, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 221, "MAX_DURATION": 4847, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.6637, "COUNT": 367}, {"from": 368640000, "to": 373760000, "AVG_TTFB": 19.07377, "ERRORS": 0, "AVG_DURATION": 246.6448, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 36, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1628, "MAX_DURATION": 2661, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.9504, "COUNT": 366}, {"from": 373760000, "to": 378880000, "AVG_TTFB": 7.636612, "ERRORS": 0, "AVG_DURATION": 130.64207, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 29, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 755, "MAX_DURATION": 1279, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.0496, "COUNT": 366}, {"from": 378880000, "to": 384000000, "AVG_TTFB": 29.933884, "ERRORS": 0, "AVG_DURATION": 407.843, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 52, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2630, "MAX_DURATION": 8099, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1689.2473, "COUNT": 363}, {"from": 384000000, "to": 389120000, "AVG_TTFB": 15.882514, "ERRORS": 0, "AVG_DURATION": 180.28688, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 36, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2183, "MAX_DURATION": 4043, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1704.2308, "COUNT": 366}, {"from": 389120000, "to": 394240000, "AVG_TTFB": 22.349728, "ERRORS": 0, "AVG_DURATION": 257.11203, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 800, "MAX_DURATION": 3945, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.1199, "COUNT": 366}, {"from": 394240000, "to": 399360000, "AVG_TTFB": 11.986339, "ERRORS": 0, "AVG_DURATION": 228.18306, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 29, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 651, "MAX_DURATION": 7649, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.0481, "COUNT": 366}, {"from": 399360000, "to": 404480000, "AVG_TTFB": 47.73901, "ERRORS": 36, "AVG_DURATION": 389.7912, "ERRORS_PER_SECOND": 0.00703125, "MIN_DURATION": 16, "ERROR_RATE": 9.89011, "MIN_TTFB": 2, "MAX_TTFB": 4227, "MAX_DURATION": 7501, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1556.7131, "COUNT": 364}, {"from": 404480000, "to": 409600000, "AVG_TTFB": 14.89863, "ERRORS": 0, "AVG_DURATION": 152.71233, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 60, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 677, "MAX_DURATION": 2091, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1699.2438, "COUNT": 365}, {"from": 409600000, "to": 414720000, "AVG_TTFB": 16.554348, "ERRORS": 0, "AVG_DURATION": 163.23097, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 33, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4091, "MAX_DURATION": 5955, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1711.9016, "COUNT": 368}, {"from": 414720000, "to": 419840000, "AVG_TTFB": 33.575344, "ERRORS": 11, "AVG_DURATION": 284.33972, "ERRORS_PER_SECOND": 0.0021484375, "MIN_DURATION": 46, "ERROR_RATE": 3.0136986, "MIN_TTFB": 2, "MAX_TTFB": 4054, "MAX_DURATION": 7430, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1664.5417, "COUNT": 365}, {"from": 419840000, "to": 424960000, "AVG_TTFB": 109.79558, "ERRORS": 0, "AVG_DURATION": 541.17126, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 35, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 7419, "MAX_DURATION": 8490, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1684.2504, "COUNT": 362}, {"from": 424960000, "to": 430080000, "AVG_TTFB": 3.8520548, "ERRORS": 0, "AVG_DURATION": 190.36986, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 46, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 93, "MAX_DURATION": 7511, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.1797, "COUNT": 365}, {"from": 430080000, "to": 435200000, "AVG_TTFB": 11.997268, "ERRORS": 0, "AVG_DURATION": 207.47815, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1064, "MAX_DURATION": 5125, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1704.1309, "COUNT": 366}, {"from": 435200000, "to": 440320000, "AVG_TTFB": 27.5, "ERRORS": 0, "AVG_DURATION": 214.16032, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 32, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2149, "MAX_DURATION": 8537, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1712.0696, "COUNT": 368}, {"from": 440320000, "to": 445440000, "AVG_TTFB": 32.238358, "ERRORS": 0, "AVG_DURATION": 228.00822, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3784, "MAX_DURATION": 4094, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.2598, "COUNT": 365}, {"from": 445440000, "to": 450560000, "AVG_TTFB": 10.519126, "ERRORS": 0, "AVG_DURATION": 167.8552, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 26, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 894, "MAX_DURATION": 3646, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.8438, "COUNT": 366}, {"from": 450560000, "to": 455680000, "AVG_TTFB": 3.3442624, "ERRORS": 0, "AVG_DURATION": 121.20765, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 25, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 50, "MAX_DURATION": 1537, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.9055, "COUNT": 366}, {"from": 455680000, "to": 460800000, "AVG_TTFB": 3.105691, "ERRORS": 0, "AVG_DURATION": 116.65582, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 57, "MAX_DURATION": 1445, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1718.098, "COUNT": 369}, {"from": 460800000, "to": 465920000, "AVG_TTFB": 24.123627, "ERRORS": 0, "AVG_DURATION": 236.61813, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 62, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2852, "MAX_DURATION": 3172, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.8219, "COUNT": 364}, {"from": 465920000, "to": 471040000, "AVG_TTFB": 12.953424, "ERRORS": 0, "AVG_DURATION": 187.21918, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 32, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1157, "MAX_DURATION": 11058, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.3636, "COUNT": 365}, {"from": 471040000, "to": 476160000, "AVG_TTFB": 7.708447, "ERRORS": 0, "AVG_DURATION": 125.29973, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 62, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 513, "MAX_DURATION": 1299, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.6696, "COUNT": 367}, {"from": 476160000, "to": 481280000, "AVG_TTFB": 141.39944, "ERRORS": 0, "AVG_DURATION": 915.77094, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 62, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 12777, "MAX_DURATION": 30984, "ELEMENTS_PER_SECOND": 0.06992187, "THROUGHPUT": 1667.4407, "COUNT": 358}, {"from": 481280000, "to": 486400000, "AVG_TTFB": 3.4059947, "ERRORS": 0, "AVG_DURATION": 123.88556, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 69, "MAX_DURATION": 1968, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.5847, "COUNT": 367}, {"from": 486400000, "to": 491520000, "AVG_TTFB": 63.38692, "ERRORS": 22, "AVG_DURATION": 256.34604, "ERRORS_PER_SECOND": 0.004296875, "MIN_DURATION": 14, "ERROR_RATE": 5.99455, "MIN_TTFB": 2, "MAX_TTFB": 6537, "MAX_DURATION": 7871, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1626.1252, "COUNT": 367}, {"from": 491520000, "to": 496640000, "AVG_TTFB": 6.7377048, "ERRORS": 0, "AVG_DURATION": 130.82787, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 471, "MAX_DURATION": 3116, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.7969, "COUNT": 366}, {"from": 496640000, "to": 501760000, "AVG_TTFB": 13.655647, "ERRORS": 9, "AVG_DURATION": 361.6336, "ERRORS_PER_SECOND": 0.0017578125, "MIN_DURATION": 53, "ERROR_RATE": 2.479339, "MIN_TTFB": 2, "MAX_TTFB": 621, "MAX_DURATION": 15105, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1655.6588, "COUNT": 363}, {"from": 501760000, "to": 506880000, "AVG_TTFB": 77.67217, "ERRORS": 36, "AVG_DURATION": 508.50412, "ERRORS_PER_SECOND": 0.00703125, "MIN_DURATION": 38, "ERROR_RATE": 9.917356, "MIN_TTFB": 2, "MAX_TTFB": 3960, "MAX_DURATION": 9669, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1547.0072, "COUNT": 363}, {"from": 506880000, "to": 512000000, "AVG_TTFB": 48.032787, "ERRORS": 0, "AVG_DURATION": 195.05737, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 62, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 10439, "MAX_DURATION": 10517, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.6497, "COUNT": 366}, {"from": 512000000, "to": 517120000, "AVG_TTFB": 7.18306, "ERRORS": 0, "AVG_DURATION": 141.99454, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 645, "MAX_DURATION": 2511, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.782, "COUNT": 366}, {"from": 517120000, "to": 522240000, "AVG_TTFB": 6.169399, "ERRORS": 0, "AVG_DURATION": 124.800545, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 59, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 794, "MAX_DURATION": 1571, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.864, "COUNT": 366}, {"from": 522240000, "to": 527360000, "AVG_TTFB": 3.811989, "ERRORS": 0, "AVG_DURATION": 142.02725, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 61, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 156, "MAX_DURATION": 4220, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1708.7351, "COUNT": 367}, {"from": 527360000, "to": 532480000, "AVG_TTFB": 60.21858, "ERRORS": 0, "AVG_DURATION": 225.07924, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 39, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 6854, "MAX_DURATION": 6918, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1702.7972, "COUNT": 366}, {"from": 532480000, "to": 537600000, "AVG_TTFB": 7.4332423, "ERRORS": 0, "AVG_DURATION": 124.03542, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 31, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 521, "MAX_DURATION": 933, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.3903, "COUNT": 367}, {"from": 537600000, "to": 542720000, "AVG_TTFB": 7.644809, "ERRORS": 0, "AVG_DURATION": 129.75137, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 62, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 948, "MAX_DURATION": 1296, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1703.0176, "COUNT": 366}, {"from": 542720000, "to": 547840000, "AVG_TTFB": 9.29863, "ERRORS": 0, "AVG_DURATION": 179.58356, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 28, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1140, "MAX_DURATION": 5063, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.4567, "COUNT": 365}, {"from": 547840000, "to": 552960000, "AVG_TTFB": 39.568306, "ERRORS": 0, "AVG_DURATION": 461.12296, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 62, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4178, "MAX_DURATION": 20867, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1704.263, "COUNT": 366}, {"from": 552960000, "to": 558080000, "AVG_TTFB": 19.90934, "ERRORS": 0, "AVG_DURATION": 212.98901, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 29, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1469, "MAX_DURATION": 3026, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.7898, "COUNT": 364}, {"from": 558080000, "to": 563200000, "AVG_TTFB": 38.046703, "ERRORS": 0, "AVG_DURATION": 305.17307, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 34, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1919, "MAX_DURATION": 5815, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1693.8558, "COUNT": 364}, {"from": 563200000, "to": 568320000, "AVG_TTFB": 144.14206, "ERRORS": 0, "AVG_DURATION": 863.9081, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 58, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 14957, "MAX_DURATION": 15408, "ELEMENTS_PER_SECOND": 0.07011719, "THROUGHPUT": 1670.9246, "COUNT": 359}, {"from": 568320000, "to": 573440000, "AVG_TTFB": 53.90411, "ERRORS": 0, "AVG_DURATION": 272.8685, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 56, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 17701, "MAX_DURATION": 18270, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1699.7953, "COUNT": 365}, {"from": 573440000, "to": 578560000, "AVG_TTFB": 149.07584, "ERRORS": 102, "AVG_DURATION": 1033.1686, "ERRORS_PER_SECOND": 0.019921875, "MIN_DURATION": 13, "ERROR_RATE": 28.651686, "MIN_TTFB": 1, "MAX_TTFB": 10067, "MAX_DURATION": 18530, "ELEMENTS_PER_SECOND": 0.06953125, "THROUGHPUT": 1251.0276, "COUNT": 356}, {"from": 578560000, "to": 583680000, "AVG_TTFB": 6.7929153, "ERRORS": 0, "AVG_DURATION": 145.44687, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 27, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 229, "MAX_DURATION": 1205, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1707.1824, "COUNT": 367}, {"from": 583680000, "to": 588800000, "AVG_TTFB": 3.8005464, "ERRORS": 8, "AVG_DURATION": 121.27049, "ERRORS_PER_SECOND": 0.0015625, "MIN_DURATION": 13, "ERROR_RATE": 2.1857924, "MIN_TTFB": 2, "MAX_TTFB": 74, "MAX_DURATION": 1394, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1671.0234, "COUNT": 366}, {"from": 588800000, "to": 593920000, "AVG_TTFB": 175.14166, "ERRORS": 0, "AVG_DURATION": 787.17224, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 47, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 19452, "MAX_DURATION": 32052, "ELEMENTS_PER_SECOND": 0.0703125, "THROUGHPUT": 1674.7688, "COUNT": 360}, {"from": 593920000, "to": 599040000, "AVG_TTFB": 45.702187, "ERRORS": 0, "AVG_DURATION": 387.44534, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 55, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1875, "MAX_DURATION": 26057, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1704.0227, "COUNT": 366}, {"from": 599040000, "to": 604160000, "AVG_TTFB": 50.217632, "ERRORS": 0, "AVG_DURATION": 377.33334, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 24, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5995, "MAX_DURATION": 10243, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1688.8718, "COUNT": 363}, {"from": 604160000, "to": 609280000, "AVG_TTFB": 12.186301, "ERRORS": 0, "AVG_DURATION": 180.12602, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 38, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 653, "MAX_DURATION": 3744, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1698.166, "COUNT": 365}, {"from": 609280000, "to": 614400000, "AVG_TTFB": 7.479592, "ERRORS": 0, "AVG_DURATION": 188.39796, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 29, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 454, "MAX_DURATION": 4719, "ELEMENTS_PER_SECOND": 0.03828125, "THROUGHPUT": 912.51135, "COUNT": 196}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/8725e0cd-92a4-4105-803b-86433851fcfc/elements/0ca05bf2-4ba9-483e-9041-c9ac38ecb8b8/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 5120000, "AVG_TTFB": 6.6373625, "ERRORS": 0, "AVG_DURATION": 95.73351, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 399, "MAX_DURATION": 9529, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1707.401, "COUNT": 364}, {"from": 5120000, "to": 10240000, "AVG_TTFB": 38.656593, "ERRORS": 0, "AVG_DURATION": 315.9368, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 4731, "MAX_DURATION": 15526, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1691.1625, "COUNT": 364}, {"from": 10240000, "to": 15360000, "AVG_TTFB": 35.60548, "ERRORS": 0, "AVG_DURATION": 166.36165, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 4742, "MAX_DURATION": 9271, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1710.3594, "COUNT": 365}, {"from": 15360000, "to": 20480000, "AVG_TTFB": 6.852459, "ERRORS": 0, "AVG_DURATION": 76.59563, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 238, "MAX_DURATION": 3664, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1718.7262, "COUNT": 366}, {"from": 20480000, "to": 25600000, "AVG_TTFB": 12.131507, "ERRORS": 0, "AVG_DURATION": 165.23836, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 887, "MAX_DURATION": 10410, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1713.0365, "COUNT": 365}, {"from": 25600000, "to": 30720000, "AVG_TTFB": 10.790761, "ERRORS": 0, "AVG_DURATION": 102.58696, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1497, "MAX_DURATION": 2611, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1718.7438, "COUNT": 368}, {"from": 30720000, "to": 35840000, "AVG_TTFB": 5.5068493, "ERRORS": 0, "AVG_DURATION": 56.079453, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 109, "MAX_DURATION": 1440, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1709.4885, "COUNT": 365}, {"from": 35840000, "to": 40960000, "AVG_TTFB": 10.804945, "ERRORS": 0, "AVG_DURATION": 157.01099, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 18, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1970, "MAX_DURATION": 7264, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1705.7361, "COUNT": 364}, {"from": 40960000, "to": 46080000, "AVG_TTFB": 22.79076, "ERRORS": 0, "AVG_DURATION": 118.70652, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 19, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 4070, "MAX_DURATION": 8717, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1720.5059, "COUNT": 368}, {"from": 46080000, "to": 51200000, "AVG_TTFB": 109.12885, "ERRORS": 0, "AVG_DURATION": 531.10364, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3079, "MAX_DURATION": 13880, "ELEMENTS_PER_SECOND": 0.069726564, "THROUGHPUT": 1676.6471, "COUNT": 357}, {"from": 51200000, "to": 56320000, "AVG_TTFB": 44.348064, "ERRORS": 23, "AVG_DURATION": 359.64087, "ERRORS_PER_SECOND": 0.0044921874, "MIN_DURATION": 19, "ERROR_RATE": 6.353591, "MIN_TTFB": 2, "MAX_TTFB": 2540, "MAX_DURATION": 20889, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1606.6794, "COUNT": 362}, {"from": 56320000, "to": 61440000, "AVG_TTFB": 93.925, "ERRORS": 47, "AVG_DURATION": 473.92776, "ERRORS_PER_SECOND": 0.009179687, "MIN_DURATION": 22, "ERROR_RATE": 13.055555, "MIN_TTFB": 2, "MAX_TTFB": 6660, "MAX_DURATION": 12884, "ELEMENTS_PER_SECOND": 0.0703125, "THROUGHPUT": 1502.5198, "COUNT": 360}, {"from": 61440000, "to": 66560000, "AVG_TTFB": 8.444141, "ERRORS": 0, "AVG_DURATION": 61.820164, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 392, "MAX_DURATION": 1255, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1722.3777, "COUNT": 367}, {"from": 66560000, "to": 71680000, "AVG_TTFB": 25.164383, "ERRORS": 26, "AVG_DURATION": 165.47945, "ERRORS_PER_SECOND": 0.005078125, "MIN_DURATION": 14, "ERROR_RATE": 7.1232877, "MIN_TTFB": 2, "MAX_TTFB": 1384, "MAX_DURATION": 6611, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1605.4258, "COUNT": 365}, {"from": 71680000, "to": 76800000, "AVG_TTFB": 62.391186, "ERRORS": 0, "AVG_DURATION": 320.78238, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 6071, "MAX_DURATION": 17797, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1704.0013, "COUNT": 363}, {"from": 76800000, "to": 81920000, "AVG_TTFB": 33.778084, "ERRORS": 0, "AVG_DURATION": 223.71507, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 19, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1372, "MAX_DURATION": 6348, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1706.656, "COUNT": 365}, {"from": 81920000, "to": 87040000, "AVG_TTFB": 11.887979, "ERRORS": 0, "AVG_DURATION": 77.52732, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 19, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 837, "MAX_DURATION": 1519, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1711.0004, "COUNT": 366}, {"from": 87040000, "to": 92160000, "AVG_TTFB": 15.195652, "ERRORS": 0, "AVG_DURATION": 79.328804, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1093, "MAX_DURATION": 1547, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1719.2844, "COUNT": 368}, {"from": 92160000, "to": 97280000, "AVG_TTFB": 5.9945207, "ERRORS": 0, "AVG_DURATION": 56.8, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 296, "MAX_DURATION": 1154, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1709.6741, "COUNT": 365}, {"from": 97280000, "to": 102400000, "AVG_TTFB": 37.28767, "ERRORS": 0, "AVG_DURATION": 158.48492, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 18, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3339, "MAX_DURATION": 4545, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1710.9147, "COUNT": 365}, {"from": 102400000, "to": 107520000, "AVG_TTFB": 8.218579, "ERRORS": 0, "AVG_DURATION": 52.81421, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1032, "MAX_DURATION": 1327, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1709.7233, "COUNT": 366}, {"from": 107520000, "to": 112640000, "AVG_TTFB": 7.189702, "ERRORS": 0, "AVG_DURATION": 72.04065, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 704, "MAX_DURATION": 1177, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1727.1871, "COUNT": 369}, {"from": 112640000, "to": 117760000, "AVG_TTFB": 6.190736, "ERRORS": 0, "AVG_DURATION": 56.269756, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 310, "MAX_DURATION": 907, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1711.5759, "COUNT": 367}, {"from": 117760000, "to": 122880000, "AVG_TTFB": 6.168937, "ERRORS": 0, "AVG_DURATION": 72.30245, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 377, "MAX_DURATION": 1875, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1717.4613, "COUNT": 367}, {"from": 122880000, "to": 128000000, "AVG_TTFB": 18.45304, "ERRORS": 0, "AVG_DURATION": 175.47514, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 19, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1006, "MAX_DURATION": 10938, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1696.6199, "COUNT": 362}, {"from": 128000000, "to": 133120000, "AVG_TTFB": 12.448088, "ERRORS": 0, "AVG_DURATION": 134.11201, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 19, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 599, "MAX_DURATION": 3532, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1715.1729, "COUNT": 366}, {"from": 133120000, "to": 138240000, "AVG_TTFB": 64.71781, "ERRORS": 0, "AVG_DURATION": 207.06302, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 10959, "MAX_DURATION": 10991, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1712.1631, "COUNT": 365}, {"from": 138240000, "to": 143360000, "AVG_TTFB": 106.38292, "ERRORS": 60, "AVG_DURATION": 501.6226, "ERRORS_PER_SECOND": 0.01171875, "MIN_DURATION": 19, "ERROR_RATE": 16.528925, "MIN_TTFB": 2, "MAX_TTFB": 17386, "MAX_DURATION": 18221, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1459.2283, "COUNT": 363}, {"from": 143360000, "to": 148480000, "AVG_TTFB": 74.89561, "ERRORS": 0, "AVG_DURATION": 245.87912, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 8414, "MAX_DURATION": 11868, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1700.2406, "COUNT": 364}, {"from": 148480000, "to": 153600000, "AVG_TTFB": 23.317808, "ERRORS": 0, "AVG_DURATION": 135.32877, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 3320, "MAX_DURATION": 3909, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1711.7006, "COUNT": 365}, {"from": 153600000, "to": 158720000, "AVG_TTFB": 10.456522, "ERRORS": 14, "AVG_DURATION": 99.122284, "ERRORS_PER_SECOND": 0.002734375, "MIN_DURATION": 14, "ERROR_RATE": 3.8043478, "MIN_TTFB": 2, "MAX_TTFB": 517, "MAX_DURATION": 3482, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1657.0798, "COUNT": 368}, {"from": 158720000, "to": 163840000, "AVG_TTFB": 60.396694, "ERRORS": 0, "AVG_DURATION": 255.87604, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 11142, "MAX_DURATION": 12409, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1701.5182, "COUNT": 363}, {"from": 163840000, "to": 168960000, "AVG_TTFB": 13.423497, "ERRORS": 0, "AVG_DURATION": 79.31694, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 992, "MAX_DURATION": 3542, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1722.2253, "COUNT": 366}, {"from": 168960000, "to": 174080000, "AVG_TTFB": 16.068306, "ERRORS": 0, "AVG_DURATION": 89.05191, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 879, "MAX_DURATION": 1693, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1715.8066, "COUNT": 366}, {"from": 174080000, "to": 179200000, "AVG_TTFB": 19.057852, "ERRORS": 0, "AVG_DURATION": 229.32231, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 19, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1419, "MAX_DURATION": 25594, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1694.0735, "COUNT": 363}, {"from": 179200000, "to": 184320000, "AVG_TTFB": 58.909588, "ERRORS": 0, "AVG_DURATION": 296.01645, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 3453, "MAX_DURATION": 8512, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1706.0801, "COUNT": 365}, {"from": 184320000, "to": 189440000, "AVG_TTFB": 41.385674, "ERRORS": 0, "AVG_DURATION": 209.65565, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1791, "MAX_DURATION": 2743, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1696.8054, "COUNT": 363}, {"from": 189440000, "to": 194560000, "AVG_TTFB": 5.334239, "ERRORS": 0, "AVG_DURATION": 46.809784, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 118, "MAX_DURATION": 454, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1719.9851, "COUNT": 368}, {"from": 194560000, "to": 199680000, "AVG_TTFB": 24.147945, "ERRORS": 0, "AVG_DURATION": 146.70412, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1229, "MAX_DURATION": 3859, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1702.8733, "COUNT": 365}, {"from": 199680000, "to": 204800000, "AVG_TTFB": 25.210382, "ERRORS": 0, "AVG_DURATION": 90.669395, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 5687, "MAX_DURATION": 5728, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1711.948, "COUNT": 366}, {"from": 204800000, "to": 209920000, "AVG_TTFB": 37.049862, "ERRORS": 0, "AVG_DURATION": 424.10803, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3759, "MAX_DURATION": 52860, "ELEMENTS_PER_SECOND": 0.07050781, "THROUGHPUT": 1689.2151, "COUNT": 361}, {"from": 209920000, "to": 215040000, "AVG_TTFB": 8.483517, "ERRORS": 0, "AVG_DURATION": 175.67583, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1007, "MAX_DURATION": 15958, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1707.5631, "COUNT": 364}, {"from": 215040000, "to": 220160000, "AVG_TTFB": 39.540165, "ERRORS": 0, "AVG_DURATION": 271.8504, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2809, "MAX_DURATION": 26085, "ELEMENTS_PER_SECOND": 0.07050781, "THROUGHPUT": 1697.4778, "COUNT": 361}, {"from": 220160000, "to": 225280000, "AVG_TTFB": 28.70411, "ERRORS": 0, "AVG_DURATION": 194.28493, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1057, "MAX_DURATION": 7344, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1703.8988, "COUNT": 365}, {"from": 225280000, "to": 230400000, "AVG_TTFB": 118.23876, "ERRORS": 66, "AVG_DURATION": 569.19666, "ERRORS_PER_SECOND": 0.012890625, "MIN_DURATION": 16, "ERROR_RATE": 18.539326, "MIN_TTFB": 2, "MAX_TTFB": 8852, "MAX_DURATION": 20904, "ELEMENTS_PER_SECOND": 0.06953125, "THROUGHPUT": 1407.818, "COUNT": 356}, {"from": 230400000, "to": 235520000, "AVG_TTFB": 8.238482, "ERRORS": 0, "AVG_DURATION": 60.742546, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 332, "MAX_DURATION": 1152, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1733.1322, "COUNT": 369}, {"from": 235520000, "to": 240640000, "AVG_TTFB": 6.8849316, "ERRORS": 0, "AVG_DURATION": 45.564384, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 289, "MAX_DURATION": 645, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1703.9746, "COUNT": 365}, {"from": 240640000, "to": 245760000, "AVG_TTFB": 45.131508, "ERRORS": 5, "AVG_DURATION": 112.89315, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 14, "ERROR_RATE": 1.369863, "MIN_TTFB": 2, "MAX_TTFB": 12214, "MAX_DURATION": 12269, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1690.2894, "COUNT": 365}, {"from": 245760000, "to": 250880000, "AVG_TTFB": 103.64641, "ERRORS": 0, "AVG_DURATION": 487.24863, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 4, "MAX_TTFB": 10306, "MAX_DURATION": 17409, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1700.1576, "COUNT": 362}, {"from": 250880000, "to": 256000000, "AVG_TTFB": 10.524457, "ERRORS": 0, "AVG_DURATION": 82.53804, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 500, "MAX_DURATION": 4439, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1716.533, "COUNT": 368}, {"from": 256000000, "to": 261120000, "AVG_TTFB": 32.61644, "ERRORS": 0, "AVG_DURATION": 135.00275, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 5202, "MAX_DURATION": 7084, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1711.516, "COUNT": 365}, {"from": 261120000, "to": 266240000, "AVG_TTFB": 6.898907, "ERRORS": 0, "AVG_DURATION": 62.245903, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 257, "MAX_DURATION": 1556, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1713.6003, "COUNT": 366}, {"from": 266240000, "to": 271360000, "AVG_TTFB": 25.068682, "ERRORS": 0, "AVG_DURATION": 231.2445, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2107, "MAX_DURATION": 13393, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1710.3988, "COUNT": 364}, {"from": 271360000, "to": 276480000, "AVG_TTFB": 55.065754, "ERRORS": 0, "AVG_DURATION": 344.09314, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1988, "MAX_DURATION": 4880, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1707.76, "COUNT": 365}, {"from": 276480000, "to": 281600000, "AVG_TTFB": 10.775956, "ERRORS": 0, "AVG_DURATION": 97.54645, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1139, "MAX_DURATION": 5240, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1710.625, "COUNT": 366}, {"from": 281600000, "to": 286720000, "AVG_TTFB": 25.19178, "ERRORS": 0, "AVG_DURATION": 175.9233, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 3297, "MAX_DURATION": 10322, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1713.2449, "COUNT": 365}, {"from": 286720000, "to": 291840000, "AVG_TTFB": 12.008287, "ERRORS": 0, "AVG_DURATION": 73.22099, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 979, "MAX_DURATION": 1072, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1701.4518, "COUNT": 362}, {"from": 291840000, "to": 296960000, "AVG_TTFB": 9.513661, "ERRORS": 0, "AVG_DURATION": 72.84699, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 342, "MAX_DURATION": 1367, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1712.4851, "COUNT": 366}, {"from": 296960000, "to": 302080000, "AVG_TTFB": 24.055096, "ERRORS": 0, "AVG_DURATION": 189.40771, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1786, "MAX_DURATION": 17933, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1701.4418, "COUNT": 363}, {"from": 302080000, "to": 307200000, "AVG_TTFB": 39.54945, "ERRORS": 0, "AVG_DURATION": 198.31593, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 3771, "MAX_DURATION": 4886, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1705.2887, "COUNT": 364}, {"from": 307200000, "to": 312320000, "AVG_TTFB": 9.234332, "ERRORS": 0, "AVG_DURATION": 97.059944, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 746, "MAX_DURATION": 4893, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1717.5336, "COUNT": 367}, {"from": 312320000, "to": 317440000, "AVG_TTFB": 92.55125, "ERRORS": 70, "AVG_DURATION": 405.4737, "ERRORS_PER_SECOND": 0.013671875, "MIN_DURATION": 15, "ERROR_RATE": 19.390581, "MIN_TTFB": 2, "MAX_TTFB": 4557, "MAX_DURATION": 13408, "ELEMENTS_PER_SECOND": 0.07050781, "THROUGHPUT": 1412.725, "COUNT": 361}, {"from": 317440000, "to": 322560000, "AVG_TTFB": 13.732241, "ERRORS": 0, "AVG_DURATION": 76.333336, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1337, "MAX_DURATION": 1688, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1720.0662, "COUNT": 366}, {"from": 322560000, "to": 327680000, "AVG_TTFB": 6.4986377, "ERRORS": 0, "AVG_DURATION": 76.89373, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 306, "MAX_DURATION": 3153, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1720.237, "COUNT": 367}, {"from": 327680000, "to": 332800000, "AVG_TTFB": 21.479452, "ERRORS": 4, "AVG_DURATION": 101.2, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 15, "ERROR_RATE": 1.0958904, "MIN_TTFB": 2, "MAX_TTFB": 1647, "MAX_DURATION": 3709, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1689.2451, "COUNT": 365}, {"from": 332800000, "to": 337920000, "AVG_TTFB": 24.126722, "ERRORS": 0, "AVG_DURATION": 232.0854, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1105, "MAX_DURATION": 19710, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1701.2637, "COUNT": 363}, {"from": 337920000, "to": 343040000, "AVG_TTFB": 13.467213, "ERRORS": 0, "AVG_DURATION": 98.97814, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1806, "MAX_DURATION": 3226, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1712.8689, "COUNT": 366}, {"from": 343040000, "to": 348160000, "AVG_TTFB": 27.535519, "ERRORS": 0, "AVG_DURATION": 176.9235, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1688, "MAX_DURATION": 11663, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1716.0801, "COUNT": 366}, {"from": 348160000, "to": 353280000, "AVG_TTFB": 8.174387, "ERRORS": 0, "AVG_DURATION": 63.479565, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 469, "MAX_DURATION": 1515, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1717.642, "COUNT": 367}, {"from": 353280000, "to": 358400000, "AVG_TTFB": 35.914837, "ERRORS": 0, "AVG_DURATION": 204.47253, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 2301, "MAX_DURATION": 4452, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1703.2012, "COUNT": 364}, {"from": 358400000, "to": 363520000, "AVG_TTFB": 15.702186, "ERRORS": 0, "AVG_DURATION": 87.20492, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 681, "MAX_DURATION": 1107, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1710.8079, "COUNT": 366}, {"from": 363520000, "to": 368640000, "AVG_TTFB": 19.807066, "ERRORS": 0, "AVG_DURATION": 90.241844, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 4723, "MAX_DURATION": 7813, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1712.2234, "COUNT": 368}, {"from": 368640000, "to": 373760000, "AVG_TTFB": 40.30959, "ERRORS": 0, "AVG_DURATION": 173.96986, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 3536, "MAX_DURATION": 4340, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1700.6432, "COUNT": 365}, {"from": 373760000, "to": 378880000, "AVG_TTFB": 9.660326, "ERRORS": 0, "AVG_DURATION": 68.733696, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 827, "MAX_DURATION": 1243, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1727.1823, "COUNT": 368}, {"from": 378880000, "to": 384000000, "AVG_TTFB": 43.588398, "ERRORS": 0, "AVG_DURATION": 317.32596, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 3162, "MAX_DURATION": 8389, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1695.6367, "COUNT": 362}, {"from": 384000000, "to": 389120000, "AVG_TTFB": 12.896458, "ERRORS": 0, "AVG_DURATION": 104.3515, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1289, "MAX_DURATION": 4784, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1719.3757, "COUNT": 367}, {"from": 389120000, "to": 394240000, "AVG_TTFB": 29.4, "ERRORS": 0, "AVG_DURATION": 145.92055, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1246, "MAX_DURATION": 1876, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1712.2703, "COUNT": 365}, {"from": 394240000, "to": 399360000, "AVG_TTFB": 10.8579235, "ERRORS": 0, "AVG_DURATION": 81.60656, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1227, "MAX_DURATION": 2106, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1710.6257, "COUNT": 366}, {"from": 399360000, "to": 404480000, "AVG_TTFB": 36.906593, "ERRORS": 37, "AVG_DURATION": 193.06044, "ERRORS_PER_SECOND": 0.0072265626, "MIN_DURATION": 15, "ERROR_RATE": 10.164835, "MIN_TTFB": 2, "MAX_TTFB": 1984, "MAX_DURATION": 5576, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1561.5977, "COUNT": 364}, {"from": 404480000, "to": 409600000, "AVG_TTFB": 14.560439, "ERRORS": 0, "AVG_DURATION": 86.17308, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 929, "MAX_DURATION": 1072, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1708.1051, "COUNT": 364}, {"from": 409600000, "to": 414720000, "AVG_TTFB": 18.666666, "ERRORS": 0, "AVG_DURATION": 89.11924, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 3706, "MAX_DURATION": 8604, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1727.9958, "COUNT": 369}, {"from": 414720000, "to": 419840000, "AVG_TTFB": 54.186813, "ERRORS": 9, "AVG_DURATION": 218.60165, "ERRORS_PER_SECOND": 0.0017578125, "MIN_DURATION": 23, "ERROR_RATE": 2.4725275, "MIN_TTFB": 2, "MAX_TTFB": 6970, "MAX_DURATION": 7005, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1671.9756, "COUNT": 364}, {"from": 419840000, "to": 424960000, "AVG_TTFB": 41.867405, "ERRORS": 0, "AVG_DURATION": 436.06628, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1756, "MAX_DURATION": 15488, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1695.0671, "COUNT": 362}, {"from": 424960000, "to": 430080000, "AVG_TTFB": 12.980769, "ERRORS": 0, "AVG_DURATION": 128.25275, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 773, "MAX_DURATION": 7341, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1706.1404, "COUNT": 364}, {"from": 430080000, "to": 435200000, "AVG_TTFB": 7.455041, "ERRORS": 0, "AVG_DURATION": 102.99183, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 168, "MAX_DURATION": 2119, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1717.9486, "COUNT": 367}, {"from": 435200000, "to": 440320000, "AVG_TTFB": 13.718579, "ERRORS": 0, "AVG_DURATION": 95.99727, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1177, "MAX_DURATION": 2674, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1706.9926, "COUNT": 366}, {"from": 440320000, "to": 445440000, "AVG_TTFB": 16.896175, "ERRORS": 0, "AVG_DURATION": 116.42896, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 571, "MAX_DURATION": 4792, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1714.7441, "COUNT": 366}, {"from": 445440000, "to": 450560000, "AVG_TTFB": 14.309589, "ERRORS": 0, "AVG_DURATION": 86.27671, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1285, "MAX_DURATION": 3552, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1705.3883, "COUNT": 365}, {"from": 450560000, "to": 455680000, "AVG_TTFB": 10.070652, "ERRORS": 0, "AVG_DURATION": 58.296196, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1283, "MAX_DURATION": 1381, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1725.7305, "COUNT": 368}, {"from": 455680000, "to": 460800000, "AVG_TTFB": 6.3766937, "ERRORS": 0, "AVG_DURATION": 55.476963, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 380, "MAX_DURATION": 2002, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1723.4176, "COUNT": 369}, {"from": 460800000, "to": 465920000, "AVG_TTFB": 34.752747, "ERRORS": 0, "AVG_DURATION": 177.58241, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 3598, "MAX_DURATION": 5769, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1703.6521, "COUNT": 364}, {"from": 465920000, "to": 471040000, "AVG_TTFB": 12.5643835, "ERRORS": 0, "AVG_DURATION": 137.53151, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 694, "MAX_DURATION": 11373, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1709.7543, "COUNT": 365}, {"from": 471040000, "to": 476160000, "AVG_TTFB": 8.209809, "ERRORS": 0, "AVG_DURATION": 61.831062, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 490, "MAX_DURATION": 1079, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1714.0906, "COUNT": 367}, {"from": 476160000, "to": 481280000, "AVG_TTFB": 112.146065, "ERRORS": 0, "AVG_DURATION": 542.9972, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 12114, "MAX_DURATION": 23817, "ELEMENTS_PER_SECOND": 0.06953125, "THROUGHPUT": 1669.149, "COUNT": 356}, {"from": 481280000, "to": 486400000, "AVG_TTFB": 11.172132, "ERRORS": 0, "AVG_DURATION": 66.333336, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1723, "MAX_DURATION": 2559, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1723.1276, "COUNT": 366}, {"from": 486400000, "to": 491520000, "AVG_TTFB": 6.767123, "ERRORS": 21, "AVG_DURATION": 110.8274, "ERRORS_PER_SECOND": 0.0041015623, "MIN_DURATION": 16, "ERROR_RATE": 5.7534246, "MIN_TTFB": 2, "MAX_TTFB": 143, "MAX_DURATION": 6591, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1622.3617, "COUNT": 365}, {"from": 491520000, "to": 496640000, "AVG_TTFB": 17.435966, "ERRORS": 0, "AVG_DURATION": 72.70572, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 3129, "MAX_DURATION": 3185, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1717.0924, "COUNT": 367}, {"from": 496640000, "to": 501760000, "AVG_TTFB": 23.68044, "ERRORS": 8, "AVG_DURATION": 269.81543, "ERRORS_PER_SECOND": 0.0015625, "MIN_DURATION": 22, "ERROR_RATE": 2.2038567, "MIN_TTFB": 2, "MAX_TTFB": 2494, "MAX_DURATION": 13450, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1666.236, "COUNT": 363}, {"from": 501760000, "to": 506880000, "AVG_TTFB": 101.22314, "ERRORS": 36, "AVG_DURATION": 471.8044, "ERRORS_PER_SECOND": 0.00703125, "MIN_DURATION": 25, "ERROR_RATE": 9.917356, "MIN_TTFB": 2, "MAX_TTFB": 12597, "MAX_DURATION": 14332, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1552.0857, "COUNT": 363}, {"from": 506880000, "to": 512000000, "AVG_TTFB": 17.346994, "ERRORS": 0, "AVG_DURATION": 171.53552, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 4, "MAX_TTFB": 808, "MAX_DURATION": 14159, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1716.9742, "COUNT": 366}, {"from": 512000000, "to": 517120000, "AVG_TTFB": 9.459017, "ERRORS": 0, "AVG_DURATION": 76.35246, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 926, "MAX_DURATION": 3829, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1717.6211, "COUNT": 366}, {"from": 517120000, "to": 522240000, "AVG_TTFB": 7.5726027, "ERRORS": 0, "AVG_DURATION": 58.139725, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 378, "MAX_DURATION": 1172, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1715.0146, "COUNT": 365}, {"from": 522240000, "to": 527360000, "AVG_TTFB": 5.4211955, "ERRORS": 0, "AVG_DURATION": 62.190216, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 140, "MAX_DURATION": 2327, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1724.6375, "COUNT": 368}, {"from": 527360000, "to": 532480000, "AVG_TTFB": 19.691257, "ERRORS": 0, "AVG_DURATION": 135.87431, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 2125, "MAX_DURATION": 11098, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1714.6871, "COUNT": 366}, {"from": 532480000, "to": 537600000, "AVG_TTFB": 7.163934, "ERRORS": 0, "AVG_DURATION": 53.980873, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 593, "MAX_DURATION": 613, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1715.2332, "COUNT": 366}, {"from": 537600000, "to": 542720000, "AVG_TTFB": 9.349727, "ERRORS": 0, "AVG_DURATION": 66.79781, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1116, "MAX_DURATION": 3149, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1710.8607, "COUNT": 366}, {"from": 542720000, "to": 547840000, "AVG_TTFB": 5.97541, "ERRORS": 0, "AVG_DURATION": 64.94809, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 237, "MAX_DURATION": 1489, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1718.3796, "COUNT": 366}, {"from": 547840000, "to": 552960000, "AVG_TTFB": 58.47671, "ERRORS": 0, "AVG_DURATION": 218.20274, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 20, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 10970, "MAX_DURATION": 15938, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1723.1224, "COUNT": 365}, {"from": 552960000, "to": 558080000, "AVG_TTFB": 24.31044, "ERRORS": 0, "AVG_DURATION": 114.04121, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1954, "MAX_DURATION": 2453, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1704.9465, "COUNT": 364}, {"from": 558080000, "to": 563200000, "AVG_TTFB": 23.692308, "ERRORS": 0, "AVG_DURATION": 163.55495, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2401, "MAX_DURATION": 6434, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1711.6863, "COUNT": 364}, {"from": 563200000, "to": 568320000, "AVG_TTFB": 58.605556, "ERRORS": 0, "AVG_DURATION": 526.44165, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1030, "MAX_DURATION": 10922, "ELEMENTS_PER_SECOND": 0.0703125, "THROUGHPUT": 1689.029, "COUNT": 360}, {"from": 568320000, "to": 573440000, "AVG_TTFB": 9.1126375, "ERRORS": 0, "AVG_DURATION": 173.90935, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 384, "MAX_DURATION": 18980, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1702.3958, "COUNT": 364}, {"from": 573440000, "to": 578560000, "AVG_TTFB": 143.61345, "ERRORS": 102, "AVG_DURATION": 772.65546, "ERRORS_PER_SECOND": 0.019921875, "MIN_DURATION": 16, "ERROR_RATE": 28.571428, "MIN_TTFB": 2, "MAX_TTFB": 20269, "MAX_DURATION": 20674, "ELEMENTS_PER_SECOND": 0.069726564, "THROUGHPUT": 1270.2086, "COUNT": 357}, {"from": 578560000, "to": 583680000, "AVG_TTFB": 123.7863, "ERRORS": 0, "AVG_DURATION": 242.43562, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 20923, "MAX_DURATION": 21008, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1716.9078, "COUNT": 365}, {"from": 583680000, "to": 588800000, "AVG_TTFB": 9.260274, "ERRORS": 8, "AVG_DURATION": 64.95342, "ERRORS_PER_SECOND": 0.0015625, "MIN_DURATION": 14, "ERROR_RATE": 2.1917808, "MIN_TTFB": 2, "MAX_TTFB": 1005, "MAX_DURATION": 1456, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1671.7595, "COUNT": 365}, {"from": 588800000, "to": 593920000, "AVG_TTFB": 62.905556, "ERRORS": 0, "AVG_DURATION": 553.0417, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 23, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 2340, "MAX_DURATION": 22355, "ELEMENTS_PER_SECOND": 0.0703125, "THROUGHPUT": 1687.1156, "COUNT": 360}, {"from": 593920000, "to": 599040000, "AVG_TTFB": 22.563187, "ERRORS": 0, "AVG_DURATION": 223.47528, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1695, "MAX_DURATION": 17302, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1703.1664, "COUNT": 364}, {"from": 599040000, "to": 604160000, "AVG_TTFB": 37.31868, "ERRORS": 0, "AVG_DURATION": 194.28297, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 1709, "MAX_DURATION": 5332, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1709.7747, "COUNT": 364}, {"from": 604160000, "to": 609280000, "AVG_TTFB": 26.967123, "ERRORS": 0, "AVG_DURATION": 146.50137, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 22, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 4255, "MAX_DURATION": 5121, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1706.656, "COUNT": 365}, {"from": 609280000, "to": 614400000, "AVG_TTFB": 17.882051, "ERRORS": 0, "AVG_DURATION": 332.6769, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 21, "ERROR_RATE": 0.0, "MIN_TTFB": 3, "MAX_TTFB": 975, "MAX_DURATION": 15647, "ELEMENTS_PER_SECOND": 0.038085938, "THROUGHPUT": 911.6166, "COUNT": 195}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/8725e0cd-92a4-4105-803b-86433851fcfc/elements/50ddd53f-4a4b-4bf8-a6b3-7a262d3d6cef/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 5120000, "AVG_TTFB": 11.373297, "ERRORS": 0, "AVG_DURATION": 79.501366, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1101, "MAX_DURATION": 3395, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1660.3157, "COUNT": 367}, {"from": 5120000, "to": 10240000, "AVG_TTFB": 32.85989, "ERRORS": 0, "AVG_DURATION": 199.68407, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 3630, "MAX_DURATION": 17544, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 10240000, "to": 15360000, "AVG_TTFB": 18.93989, "ERRORS": 0, "AVG_DURATION": 90.64754, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 916, "MAX_DURATION": 1568, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 15360000, "to": 20480000, "AVG_TTFB": 21.790192, "ERRORS": 0, "AVG_DURATION": 95.6376, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1869, "MAX_DURATION": 5323, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 20480000, "to": 25600000, "AVG_TTFB": 33.387978, "ERRORS": 0, "AVG_DURATION": 203.28142, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 5482, "MAX_DURATION": 17162, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 25600000, "to": 30720000, "AVG_TTFB": 27.284931, "ERRORS": 0, "AVG_DURATION": 83.78904, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2794, "MAX_DURATION": 3765, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 30720000, "to": 35840000, "AVG_TTFB": 8.256131, "ERRORS": 0, "AVG_DURATION": 45.550407, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1223, "MAX_DURATION": 1373, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 35840000, "to": 40960000, "AVG_TTFB": 21.667582, "ERRORS": 0, "AVG_DURATION": 264.47253, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1586, "MAX_DURATION": 23042, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1471.9824, "COUNT": 364}, {"from": 40960000, "to": 46080000, "AVG_TTFB": 7.4508195, "ERRORS": 0, "AVG_DURATION": 50.300545, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 10, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 503, "MAX_DURATION": 1398, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 46080000, "to": 51200000, "AVG_TTFB": 57.60278, "ERRORS": 0, "AVG_DURATION": 263.7028, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3958, "MAX_DURATION": 7747, "ELEMENTS_PER_SECOND": 0.0703125, "THROUGHPUT": 1404.6461, "COUNT": 360}, {"from": 51200000, "to": 56320000, "AVG_TTFB": 65.49448, "ERRORS": 24, "AVG_DURATION": 178.62431, "ERRORS_PER_SECOND": 0.0046875, "MIN_DURATION": 11, "ERROR_RATE": 6.629834, "MIN_TTFB": 1, "MAX_TTFB": 9447, "MAX_DURATION": 13875, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1315.332, "COUNT": 362}, {"from": 56320000, "to": 61440000, "AVG_TTFB": 56.569443, "ERRORS": 44, "AVG_DURATION": 198.23889, "ERRORS_PER_SECOND": 0.00859375, "MIN_DURATION": 4, "ERROR_RATE": 12.222222, "MIN_TTFB": 1, "MAX_TTFB": 2914, "MAX_DURATION": 14152, "ELEMENTS_PER_SECOND": 0.0703125, "THROUGHPUT": 1225.0234, "COUNT": 360}, {"from": 61440000, "to": 66560000, "AVG_TTFB": 13.1912565, "ERRORS": 0, "AVG_DURATION": 107.22131, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1026, "MAX_DURATION": 11005, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1475.0215, "COUNT": 366}, {"from": 66560000, "to": 71680000, "AVG_TTFB": 12.076503, "ERRORS": 26, "AVG_DURATION": 50.67213, "ERRORS_PER_SECOND": 0.005078125, "MIN_DURATION": 4, "ERROR_RATE": 7.103825, "MIN_TTFB": 2, "MAX_TTFB": 1957, "MAX_DURATION": 3156, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1380.143, "COUNT": 366}, {"from": 71680000, "to": 76800000, "AVG_TTFB": 37.803867, "ERRORS": 0, "AVG_DURATION": 163.04144, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1774, "MAX_DURATION": 2833, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1407.6852, "COUNT": 362}, {"from": 76800000, "to": 81920000, "AVG_TTFB": 13.939726, "ERRORS": 0, "AVG_DURATION": 96.12603, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 560, "MAX_DURATION": 3515, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 81920000, "to": 87040000, "AVG_TTFB": 7.667575, "ERRORS": 0, "AVG_DURATION": 51.566757, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 834, "MAX_DURATION": 3836, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 87040000, "to": 92160000, "AVG_TTFB": 12.757493, "ERRORS": 0, "AVG_DURATION": 135.28883, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 643, "MAX_DURATION": 21897, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 92160000, "to": 97280000, "AVG_TTFB": 4.66485, "ERRORS": 0, "AVG_DURATION": 32.490463, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 605, "MAX_DURATION": 620, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 97280000, "to": 102400000, "AVG_TTFB": 15.843407, "ERRORS": 0, "AVG_DURATION": 102.43132, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 723, "MAX_DURATION": 3545, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 102400000, "to": 107520000, "AVG_TTFB": 2.8369565, "ERRORS": 0, "AVG_DURATION": 129.63586, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 60, "MAX_DURATION": 5240, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 107520000, "to": 112640000, "AVG_TTFB": 9.463215, "ERRORS": 0, "AVG_DURATION": 131.92915, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1174, "MAX_DURATION": 15397, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 112640000, "to": 117760000, "AVG_TTFB": 3.4320652, "ERRORS": 0, "AVG_DURATION": 50.654892, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 106, "MAX_DURATION": 5283, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 117760000, "to": 122880000, "AVG_TTFB": 7.016349, "ERRORS": 0, "AVG_DURATION": 37.871933, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 812, "MAX_DURATION": 825, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 122880000, "to": 128000000, "AVG_TTFB": 20.071625, "ERRORS": 0, "AVG_DURATION": 289.47934, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1348, "MAX_DURATION": 43991, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1409.2047, "COUNT": 363}, {"from": 128000000, "to": 133120000, "AVG_TTFB": 8.057693, "ERRORS": 0, "AVG_DURATION": 77.31044, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 449, "MAX_DURATION": 2896, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1471.9824, "COUNT": 364}, {"from": 133120000, "to": 138240000, "AVG_TTFB": 12.441417, "ERRORS": 0, "AVG_DURATION": 87.100815, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 604, "MAX_DURATION": 4384, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 138240000, "to": 143360000, "AVG_TTFB": 44.381214, "ERRORS": 58, "AVG_DURATION": 125.325966, "ERRORS_PER_SECOND": 0.011328125, "MIN_DURATION": 4, "ERROR_RATE": 16.022099, "MIN_TTFB": 1, "MAX_TTFB": 4268, "MAX_DURATION": 5154, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1209.8782, "COUNT": 362}, {"from": 143360000, "to": 148480000, "AVG_TTFB": 8.725275, "ERRORS": 0, "AVG_DURATION": 80.82967, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 657, "MAX_DURATION": 2730, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 148480000, "to": 153600000, "AVG_TTFB": 10.680327, "ERRORS": 0, "AVG_DURATION": 65.46722, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1777, "MAX_DURATION": 3420, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 153600000, "to": 158720000, "AVG_TTFB": 20.891008, "ERRORS": 14, "AVG_DURATION": 58.564034, "ERRORS_PER_SECOND": 0.002734375, "MIN_DURATION": 4, "ERROR_RATE": 3.814714, "MIN_TTFB": 2, "MAX_TTFB": 1866, "MAX_DURATION": 1888, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1397.1172, "COUNT": 367}, {"from": 158720000, "to": 163840000, "AVG_TTFB": 20.382513, "ERRORS": 0, "AVG_DURATION": 97.60656, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2557, "MAX_DURATION": 4428, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 163840000, "to": 168960000, "AVG_TTFB": 5.5054946, "ERRORS": 0, "AVG_DURATION": 39.23077, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 423, "MAX_DURATION": 1062, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 168960000, "to": 174080000, "AVG_TTFB": 12.769863, "ERRORS": 0, "AVG_DURATION": 58.31233, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1431, "MAX_DURATION": 2249, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 174080000, "to": 179200000, "AVG_TTFB": 15.459017, "ERRORS": 0, "AVG_DURATION": 70.040985, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1157, "MAX_DURATION": 7297, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1475.0215, "COUNT": 366}, {"from": 179200000, "to": 184320000, "AVG_TTFB": 30.035715, "ERRORS": 0, "AVG_DURATION": 135.04945, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1835, "MAX_DURATION": 4361, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 184320000, "to": 189440000, "AVG_TTFB": 16.51233, "ERRORS": 0, "AVG_DURATION": 92.70959, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1398, "MAX_DURATION": 3627, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 189440000, "to": 194560000, "AVG_TTFB": 4.8797812, "ERRORS": 0, "AVG_DURATION": 87.47814, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 543, "MAX_DURATION": 7421, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 194560000, "to": 199680000, "AVG_TTFB": 9.972677, "ERRORS": 0, "AVG_DURATION": 312.33606, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 838, "MAX_DURATION": 27852, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 199680000, "to": 204800000, "AVG_TTFB": 11.253406, "ERRORS": 0, "AVG_DURATION": 138.32698, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1244, "MAX_DURATION": 13599, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 204800000, "to": 209920000, "AVG_TTFB": 18.054945, "ERRORS": 0, "AVG_DURATION": 299.489, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1919, "MAX_DURATION": 35904, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 209920000, "to": 215040000, "AVG_TTFB": 18.870523, "ERRORS": 0, "AVG_DURATION": 190.08264, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1005, "MAX_DURATION": 23846, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1409.2047, "COUNT": 363}, {"from": 215040000, "to": 220160000, "AVG_TTFB": 18.228651, "ERRORS": 0, "AVG_DURATION": 329.4435, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 823, "MAX_DURATION": 22004, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1409.2047, "COUNT": 363}, {"from": 220160000, "to": 225280000, "AVG_TTFB": 14.934605, "ERRORS": 0, "AVG_DURATION": 95.43869, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 668, "MAX_DURATION": 4634, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 225280000, "to": 230400000, "AVG_TTFB": 214.77031, "ERRORS": 74, "AVG_DURATION": 371.0476, "ERRORS_PER_SECOND": 0.014453125, "MIN_DURATION": 4, "ERROR_RATE": 20.72829, "MIN_TTFB": 2, "MAX_TTFB": 20266, "MAX_DURATION": 20266, "ELEMENTS_PER_SECOND": 0.069726564, "THROUGHPUT": 1119.3441, "COUNT": 357}, {"from": 230400000, "to": 235520000, "AVG_TTFB": 5.4010987, "ERRORS": 0, "AVG_DURATION": 36.557693, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 375, "MAX_DURATION": 480, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 235520000, "to": 240640000, "AVG_TTFB": 5.8455286, "ERRORS": 0, "AVG_DURATION": 33.528454, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 717, "MAX_DURATION": 751, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1418.3219, "COUNT": 369}, {"from": 240640000, "to": 245760000, "AVG_TTFB": 68.53552, "ERRORS": 5, "AVG_DURATION": 163.0847, "ERRORS_PER_SECOND": 0.0009765625, "MIN_DURATION": 4, "ERROR_RATE": 1.3661202, "MIN_TTFB": 2, "MAX_TTFB": 13238, "MAX_DURATION": 15300, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1407.2836, "COUNT": 366}, {"from": 245760000, "to": 250880000, "AVG_TTFB": 48.116344, "ERRORS": 0, "AVG_DURATION": 221.32964, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2350, "MAX_DURATION": 13280, "ELEMENTS_PER_SECOND": 0.07050781, "THROUGHPUT": 1467.4238, "COUNT": 361}, {"from": 250880000, "to": 256000000, "AVG_TTFB": 8.453804, "ERRORS": 0, "AVG_DURATION": 46.595108, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 927, "MAX_DURATION": 1255, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 256000000, "to": 261120000, "AVG_TTFB": 35.89315, "ERRORS": 0, "AVG_DURATION": 82.263016, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 5723, "MAX_DURATION": 5808, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 261120000, "to": 266240000, "AVG_TTFB": 3.5177112, "ERRORS": 0, "AVG_DURATION": 33.93188, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 143, "MAX_DURATION": 309, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 266240000, "to": 271360000, "AVG_TTFB": 22.753426, "ERRORS": 0, "AVG_DURATION": 84.45206, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3858, "MAX_DURATION": 3973, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1473.502, "COUNT": 365}, {"from": 271360000, "to": 276480000, "AVG_TTFB": 28.483517, "ERRORS": 0, "AVG_DURATION": 163.47803, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1223, "MAX_DURATION": 5157, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 276480000, "to": 281600000, "AVG_TTFB": 8.350543, "ERRORS": 0, "AVG_DURATION": 221.36685, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1432, "MAX_DURATION": 16258, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 281600000, "to": 286720000, "AVG_TTFB": 8.966942, "ERRORS": 0, "AVG_DURATION": 77.01102, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 940, "MAX_DURATION": 3811, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1409.2047, "COUNT": 363}, {"from": 286720000, "to": 291840000, "AVG_TTFB": 7.2396693, "ERRORS": 0, "AVG_DURATION": 828.30304, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1026, "MAX_DURATION": 27016, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1409.2047, "COUNT": 363}, {"from": 291840000, "to": 296960000, "AVG_TTFB": 6.7166214, "ERRORS": 0, "AVG_DURATION": 168.58583, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 768, "MAX_DURATION": 19404, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 296960000, "to": 302080000, "AVG_TTFB": 19.203857, "ERRORS": 0, "AVG_DURATION": 391.36914, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3008, "MAX_DURATION": 32449, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1409.2047, "COUNT": 363}, {"from": 302080000, "to": 307200000, "AVG_TTFB": 23.958792, "ERRORS": 0, "AVG_DURATION": 149.48352, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1748, "MAX_DURATION": 5704, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 307200000, "to": 312320000, "AVG_TTFB": 10.195652, "ERRORS": 0, "AVG_DURATION": 41.557064, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 986, "MAX_DURATION": 1060, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 312320000, "to": 317440000, "AVG_TTFB": 43.24033, "ERRORS": 70, "AVG_DURATION": 171.6326, "ERRORS_PER_SECOND": 0.013671875, "MIN_DURATION": 3, "ERROR_RATE": 19.337017, "MIN_TTFB": 1, "MAX_TTFB": 5528, "MAX_DURATION": 5649, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1194.2969, "COUNT": 362}, {"from": 317440000, "to": 322560000, "AVG_TTFB": 7.2978144, "ERRORS": 0, "AVG_DURATION": 44.18306, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 530, "MAX_DURATION": 645, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 322560000, "to": 327680000, "AVG_TTFB": 8.182561, "ERRORS": 0, "AVG_DURATION": 58.694824, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 10, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 791, "MAX_DURATION": 1992, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 327680000, "to": 332800000, "AVG_TTFB": 15.317808, "ERRORS": 4, "AVG_DURATION": 53.728767, "ERRORS_PER_SECOND": 0.00078125, "MIN_DURATION": 4, "ERROR_RATE": 1.0958904, "MIN_TTFB": 2, "MAX_TTFB": 2363, "MAX_DURATION": 2586, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1407.0687, "COUNT": 365}, {"from": 332800000, "to": 337920000, "AVG_TTFB": 19.10137, "ERRORS": 0, "AVG_DURATION": 122.70137, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1726, "MAX_DURATION": 5924, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 337920000, "to": 343040000, "AVG_TTFB": 9.387979, "ERRORS": 0, "AVG_DURATION": 59.97541, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 558, "MAX_DURATION": 1610, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1475.0215, "COUNT": 366}, {"from": 343040000, "to": 348160000, "AVG_TTFB": 18.568306, "ERRORS": 0, "AVG_DURATION": 70.18852, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2428, "MAX_DURATION": 2443, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 348160000, "to": 353280000, "AVG_TTFB": 3.7029972, "ERRORS": 0, "AVG_DURATION": 38.81744, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 177, "MAX_DURATION": 1029, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 353280000, "to": 358400000, "AVG_TTFB": 26.964188, "ERRORS": 0, "AVG_DURATION": 114.41598, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1238, "MAX_DURATION": 3959, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1409.2047, "COUNT": 363}, {"from": 358400000, "to": 363520000, "AVG_TTFB": 9.173913, "ERRORS": 0, "AVG_DURATION": 54.888588, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1048, "MAX_DURATION": 1059, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1478.0605, "COUNT": 368}, {"from": 363520000, "to": 368640000, "AVG_TTFB": 24.93188, "ERRORS": 0, "AVG_DURATION": 206.6703, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 5693, "MAX_DURATION": 15643, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 368640000, "to": 373760000, "AVG_TTFB": 17.174387, "ERRORS": 0, "AVG_DURATION": 128.38692, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1270, "MAX_DURATION": 9254, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 373760000, "to": 378880000, "AVG_TTFB": 6.6438355, "ERRORS": 0, "AVG_DURATION": 75.668495, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 945, "MAX_DURATION": 6672, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 378880000, "to": 384000000, "AVG_TTFB": 32.7438, "ERRORS": 0, "AVG_DURATION": 132.71074, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3430, "MAX_DURATION": 5739, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 1409.2047, "COUNT": 363}, {"from": 384000000, "to": 389120000, "AVG_TTFB": 5.52861, "ERRORS": 0, "AVG_DURATION": 82.06539, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 628, "MAX_DURATION": 11840, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 389120000, "to": 394240000, "AVG_TTFB": 12.959017, "ERRORS": 0, "AVG_DURATION": 102.650276, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 520, "MAX_DURATION": 3020, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 394240000, "to": 399360000, "AVG_TTFB": 5.4438357, "ERRORS": 0, "AVG_DURATION": 66.50137, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 226, "MAX_DURATION": 1778, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 399360000, "to": 404480000, "AVG_TTFB": 47.983517, "ERRORS": 34, "AVG_DURATION": 131.46428, "ERRORS_PER_SECOND": 0.006640625, "MIN_DURATION": 4, "ERROR_RATE": 9.340659, "MIN_TTFB": 2, "MAX_TTFB": 6095, "MAX_DURATION": 6833, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1366.6875, "COUNT": 364}, {"from": 404480000, "to": 409600000, "AVG_TTFB": 97.64499, "ERRORS": 0, "AVG_DURATION": 182.69377, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 10651, "MAX_DURATION": 14101, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1479.5801, "COUNT": 369}, {"from": 409600000, "to": 414720000, "AVG_TTFB": 10.598361, "ERRORS": 0, "AVG_DURATION": 70.5847, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1428, "MAX_DURATION": 7421, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 414720000, "to": 419840000, "AVG_TTFB": 8.764383, "ERRORS": 9, "AVG_DURATION": 60.043835, "ERRORS_PER_SECOND": 0.0017578125, "MIN_DURATION": 4, "ERROR_RATE": 2.4657533, "MIN_TTFB": 2, "MAX_TTFB": 316, "MAX_DURATION": 1036, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1400.5559, "COUNT": 365}, {"from": 419840000, "to": 424960000, "AVG_TTFB": 66.99724, "ERRORS": 0, "AVG_DURATION": 234.0442, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 7558, "MAX_DURATION": 10335, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1407.6852, "COUNT": 362}, {"from": 424960000, "to": 430080000, "AVG_TTFB": 8.347945, "ERRORS": 0, "AVG_DURATION": 50.627396, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 971, "MAX_DURATION": 1743, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 430080000, "to": 435200000, "AVG_TTFB": 24.54496, "ERRORS": 0, "AVG_DURATION": 62.411446, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1887, "MAX_DURATION": 2073, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 435200000, "to": 440320000, "AVG_TTFB": 9.722826, "ERRORS": 0, "AVG_DURATION": 57.774456, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 854, "MAX_DURATION": 1656, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1416.8024, "COUNT": 368}, {"from": 440320000, "to": 445440000, "AVG_TTFB": 7.357143, "ERRORS": 0, "AVG_DURATION": 82.25824, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 253, "MAX_DURATION": 5136, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 445440000, "to": 450560000, "AVG_TTFB": 6.1880107, "ERRORS": 0, "AVG_DURATION": 55.3188, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 826, "MAX_DURATION": 3149, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 450560000, "to": 455680000, "AVG_TTFB": 9.457766, "ERRORS": 0, "AVG_DURATION": 42.604904, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1087, "MAX_DURATION": 1680, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1476.541, "COUNT": 367}, {"from": 455680000, "to": 460800000, "AVG_TTFB": 2.7371273, "ERRORS": 0, "AVG_DURATION": 35.753387, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 27, "MAX_DURATION": 1042, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 1418.3219, "COUNT": 369}, {"from": 460800000, "to": 465920000, "AVG_TTFB": 18.860273, "ERRORS": 0, "AVG_DURATION": 104.46301, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 9, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 937, "MAX_DURATION": 2823, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 465920000, "to": 471040000, "AVG_TTFB": 4.603825, "ERRORS": 0, "AVG_DURATION": 46.81967, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 182, "MAX_DURATION": 648, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 471040000, "to": 476160000, "AVG_TTFB": 4.201635, "ERRORS": 0, "AVG_DURATION": 36.542233, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 10, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 271, "MAX_DURATION": 748, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 476160000, "to": 481280000, "AVG_TTFB": 183.66017, "ERRORS": 0, "AVG_DURATION": 498.98328, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 11116, "MAX_DURATION": 20869, "ELEMENTS_PER_SECOND": 0.07011719, "THROUGHPUT": 1464.3848, "COUNT": 359}, {"from": 481280000, "to": 486400000, "AVG_TTFB": 2.7002726, "ERRORS": 0, "AVG_DURATION": 32.81744, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 32, "MAX_DURATION": 316, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 486400000, "to": 491520000, "AVG_TTFB": 13.183561, "ERRORS": 21, "AVG_DURATION": 83.11507, "ERRORS_PER_SECOND": 0.0041015623, "MIN_DURATION": 5, "ERROR_RATE": 5.7534246, "MIN_TTFB": 2, "MAX_TTFB": 981, "MAX_DURATION": 6246, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1385.1156, "COUNT": 365}, {"from": 491520000, "to": 496640000, "AVG_TTFB": 3.5258856, "ERRORS": 0, "AVG_DURATION": 35.880108, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 212, "MAX_DURATION": 539, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 496640000, "to": 501760000, "AVG_TTFB": 58.265755, "ERRORS": 10, "AVG_DURATION": 190.58083, "ERRORS_PER_SECOND": 0.001953125, "MIN_DURATION": 11, "ERROR_RATE": 2.739726, "MIN_TTFB": 2, "MAX_TTFB": 6892, "MAX_DURATION": 12578, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1399.2594, "COUNT": 365}, {"from": 501760000, "to": 506880000, "AVG_TTFB": 38.61602, "ERRORS": 34, "AVG_DURATION": 211.99724, "ERRORS_PER_SECOND": 0.006640625, "MIN_DURATION": 4, "ERROR_RATE": 9.392265, "MIN_TTFB": 2, "MAX_TTFB": 2971, "MAX_DURATION": 9037, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 1302.2699, "COUNT": 362}, {"from": 506880000, "to": 512000000, "AVG_TTFB": 11.692098, "ERRORS": 0, "AVG_DURATION": 70.231606, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 955, "MAX_DURATION": 3083, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 512000000, "to": 517120000, "AVG_TTFB": 7.826923, "ERRORS": 0, "AVG_DURATION": 40.414837, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 695, "MAX_DURATION": 715, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 517120000, "to": 522240000, "AVG_TTFB": 4.1662126, "ERRORS": 0, "AVG_DURATION": 41.367847, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 257, "MAX_DURATION": 1419, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 522240000, "to": 527360000, "AVG_TTFB": 10.882514, "ERRORS": 0, "AVG_DURATION": 52.530056, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1780, "MAX_DURATION": 5222, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1475.0215, "COUNT": 366}, {"from": 527360000, "to": 532480000, "AVG_TTFB": 15.509537, "ERRORS": 0, "AVG_DURATION": 88.662125, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1613, "MAX_DURATION": 5707, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 532480000, "to": 537600000, "AVG_TTFB": 6.656676, "ERRORS": 0, "AVG_DURATION": 38.23161, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1309, "MAX_DURATION": 1790, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 537600000, "to": 542720000, "AVG_TTFB": 6.961853, "ERRORS": 0, "AVG_DURATION": 43.724796, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1278, "MAX_DURATION": 1564, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 1415.2828, "COUNT": 367}, {"from": 542720000, "to": 547840000, "AVG_TTFB": 6.5342464, "ERRORS": 0, "AVG_DURATION": 53.238358, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 850, "MAX_DURATION": 1913, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1473.502, "COUNT": 365}, {"from": 547840000, "to": 552960000, "AVG_TTFB": 16.038462, "ERRORS": 0, "AVG_DURATION": 101.98077, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 1031, "MAX_DURATION": 9339, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 1410.7242, "COUNT": 364}, {"from": 552960000, "to": 558080000, "AVG_TTFB": 17.002731, "ERRORS": 0, "AVG_DURATION": 85.068306, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 2108, "MAX_DURATION": 4239, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 558080000, "to": 563200000, "AVG_TTFB": 28.70411, "ERRORS": 0, "AVG_DURATION": 89.44109, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 3541, "MAX_DURATION": 3725, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 563200000, "to": 568320000, "AVG_TTFB": 105.96629, "ERRORS": 0, "AVG_DURATION": 393.89044, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 7801, "MAX_DURATION": 11231, "ELEMENTS_PER_SECOND": 0.06953125, "THROUGHPUT": 1398.568, "COUNT": 356}, {"from": 568320000, "to": 573440000, "AVG_TTFB": 28.565575, "ERRORS": 0, "AVG_DURATION": 117.49454, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 13, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 6125, "MAX_DURATION": 6148, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1475.0215, "COUNT": 366}, {"from": 573440000, "to": 578560000, "AVG_TTFB": 169.39044, "ERRORS": 102, "AVG_DURATION": 364.6882, "ERRORS_PER_SECOND": 0.019921875, "MIN_DURATION": 4, "ERROR_RATE": 28.651686, "MIN_TTFB": 2, "MAX_TTFB": 10781, "MAX_DURATION": 11563, "ELEMENTS_PER_SECOND": 0.06953125, "THROUGHPUT": 1021.152, "COUNT": 356}, {"from": 578560000, "to": 583680000, "AVG_TTFB": 37.836067, "ERRORS": 0, "AVG_DURATION": 86.20219, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 7167, "MAX_DURATION": 7474, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 583680000, "to": 588800000, "AVG_TTFB": 5.3152175, "ERRORS": 8, "AVG_DURATION": 44.904892, "ERRORS_PER_SECOND": 0.0015625, "MIN_DURATION": 5, "ERROR_RATE": 2.173913, "MIN_TTFB": 2, "MAX_TTFB": 762, "MAX_DURATION": 3206, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 1406.4398, "COUNT": 368}, {"from": 588800000, "to": 593920000, "AVG_TTFB": 37.930363, "ERRORS": 0, "AVG_DURATION": 200.75766, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2637, "MAX_DURATION": 12096, "ELEMENTS_PER_SECOND": 0.07011719, "THROUGHPUT": 1464.3848, "COUNT": 359}, {"from": 593920000, "to": 599040000, "AVG_TTFB": 67.30601, "ERRORS": 0, "AVG_DURATION": 175.72131, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 10186, "MAX_DURATION": 11967, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 599040000, "to": 604160000, "AVG_TTFB": 27.471233, "ERRORS": 0, "AVG_DURATION": 127.63287, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 11, "ERROR_RATE": 0.0, "MIN_TTFB": 1, "MAX_TTFB": 2403, "MAX_DURATION": 7497, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 1412.2438, "COUNT": 365}, {"from": 604160000, "to": 609280000, "AVG_TTFB": 11.046448, "ERRORS": 0, "AVG_DURATION": 73.11475, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1149, "MAX_DURATION": 2101, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 1413.7633, "COUNT": 366}, {"from": 609280000, "to": 614400000, "AVG_TTFB": 13.377551, "ERRORS": 0, "AVG_DURATION": 86.16327, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 12, "ERROR_RATE": 0.0, "MIN_TTFB": 2, "MAX_TTFB": 1238, "MAX_DURATION": 3741, "ELEMENTS_PER_SECOND": 0.03828125, "THROUGHPUT": 787.89374, "COUNT": 196}]'

        elif endpoint == 'v3/workspaces/5f689c3f0860270001606902/test-results/8725e0cd-92a4-4105-803b-86433851fcfc/elements/b3e1f7c6-a4ef-438e-a3bb-8ffa1cb4a85e/points?statistics=AVG_DURATION,MIN_DURATION,MAX_DURATION,COUNT,THROUGHPUT,ELEMENTS_PER_SECOND,ERRORS,ERRORS_PER_SECOND,ERROR_RATE,AVG_TTFB,MIN_TTFB,MAX_TTFB':
            return '[{"from": 0, "to": 5120000, "AVG_TTFB": 42.64835, "ERRORS": 0, "AVG_DURATION": 42.64835, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 53, "MAX_DURATION": 53, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 5120000, "to": 10240000, "AVG_TTFB": 42.755493, "ERRORS": 0, "AVG_DURATION": 42.755493, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 52, "MAX_DURATION": 52, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 10240000, "to": 15360000, "AVG_TTFB": 42.660275, "ERRORS": 0, "AVG_DURATION": 42.660275, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 15360000, "to": 20480000, "AVG_TTFB": 42.691257, "ERRORS": 0, "AVG_DURATION": 42.691257, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 20480000, "to": 25600000, "AVG_TTFB": 42.635616, "ERRORS": 0, "AVG_DURATION": 42.635616, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 25600000, "to": 30720000, "AVG_TTFB": 42.695652, "ERRORS": 0, "AVG_DURATION": 42.695652, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 30720000, "to": 35840000, "AVG_TTFB": 44.041096, "ERRORS": 0, "AVG_DURATION": 44.041096, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 552, "MAX_DURATION": 552, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 35840000, "to": 40960000, "AVG_TTFB": 42.576923, "ERRORS": 0, "AVG_DURATION": 42.576923, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 46, "MAX_DURATION": 46, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 40960000, "to": 46080000, "AVG_TTFB": 42.698368, "ERRORS": 0, "AVG_DURATION": 42.698368, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 46080000, "to": 51200000, "AVG_TTFB": 47.2549, "ERRORS": 0, "AVG_DURATION": 47.2549, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 1649, "MAX_DURATION": 1649, "ELEMENTS_PER_SECOND": 0.069726564, "THROUGHPUT": 0.0, "COUNT": 357}, {"from": 51200000, "to": 56320000, "AVG_TTFB": 42.618786, "ERRORS": 0, "AVG_DURATION": 42.618786, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 0.0, "COUNT": 362}, {"from": 56320000, "to": 61440000, "AVG_TTFB": 42.586113, "ERRORS": 0, "AVG_DURATION": 42.586113, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.0703125, "THROUGHPUT": 0.0, "COUNT": 360}, {"from": 61440000, "to": 66560000, "AVG_TTFB": 43.163486, "ERRORS": 0, "AVG_DURATION": 43.163486, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 246, "MAX_DURATION": 246, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 66560000, "to": 71680000, "AVG_TTFB": 42.649315, "ERRORS": 0, "AVG_DURATION": 42.649315, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 71680000, "to": 76800000, "AVG_TTFB": 43.00551, "ERRORS": 0, "AVG_DURATION": 43.00551, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 147, "MAX_DURATION": 147, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 0.0, "COUNT": 363}, {"from": 76800000, "to": 81920000, "AVG_TTFB": 42.51233, "ERRORS": 0, "AVG_DURATION": 42.51233, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 81920000, "to": 87040000, "AVG_TTFB": 42.628414, "ERRORS": 0, "AVG_DURATION": 42.628414, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 55, "MAX_DURATION": 55, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 87040000, "to": 92160000, "AVG_TTFB": 42.402172, "ERRORS": 0, "AVG_DURATION": 42.402172, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 92160000, "to": 97280000, "AVG_TTFB": 42.4, "ERRORS": 0, "AVG_DURATION": 42.4, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 97280000, "to": 102400000, "AVG_TTFB": 43.717808, "ERRORS": 0, "AVG_DURATION": 43.717808, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 543, "MAX_DURATION": 543, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 102400000, "to": 107520000, "AVG_TTFB": 42.333332, "ERRORS": 0, "AVG_DURATION": 42.333332, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 107520000, "to": 112640000, "AVG_TTFB": 42.355015, "ERRORS": 0, "AVG_DURATION": 42.355015, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 0.0, "COUNT": 369}, {"from": 112640000, "to": 117760000, "AVG_TTFB": 42.46049, "ERRORS": 0, "AVG_DURATION": 42.46049, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 102, "MAX_DURATION": 102, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 117760000, "to": 122880000, "AVG_TTFB": 42.348774, "ERRORS": 0, "AVG_DURATION": 42.348774, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 122880000, "to": 128000000, "AVG_TTFB": 42.345303, "ERRORS": 0, "AVG_DURATION": 42.345303, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 0.0, "COUNT": 362}, {"from": 128000000, "to": 133120000, "AVG_TTFB": 42.371586, "ERRORS": 0, "AVG_DURATION": 42.371586, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 133120000, "to": 138240000, "AVG_TTFB": 42.71507, "ERRORS": 0, "AVG_DURATION": 42.71507, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 114, "MAX_DURATION": 114, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 138240000, "to": 143360000, "AVG_TTFB": 42.347107, "ERRORS": 0, "AVG_DURATION": 42.347107, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 0.0, "COUNT": 363}, {"from": 143360000, "to": 148480000, "AVG_TTFB": 42.32967, "ERRORS": 0, "AVG_DURATION": 42.32967, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 148480000, "to": 153600000, "AVG_TTFB": 42.372604, "ERRORS": 0, "AVG_DURATION": 42.372604, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 153600000, "to": 158720000, "AVG_TTFB": 42.361412, "ERRORS": 0, "AVG_DURATION": 42.361412, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 158720000, "to": 163840000, "AVG_TTFB": 46.74931, "ERRORS": 0, "AVG_DURATION": 46.74931, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 1130, "MAX_DURATION": 1130, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 0.0, "COUNT": 363}, {"from": 163840000, "to": 168960000, "AVG_TTFB": 42.442623, "ERRORS": 0, "AVG_DURATION": 42.442623, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 168960000, "to": 174080000, "AVG_TTFB": 42.48361, "ERRORS": 0, "AVG_DURATION": 42.48361, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 55, "MAX_DURATION": 55, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 174080000, "to": 179200000, "AVG_TTFB": 42.440773, "ERRORS": 0, "AVG_DURATION": 42.440773, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 0.0, "COUNT": 363}, {"from": 179200000, "to": 184320000, "AVG_TTFB": 43.035618, "ERRORS": 0, "AVG_DURATION": 43.035618, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 256, "MAX_DURATION": 256, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 184320000, "to": 189440000, "AVG_TTFB": 44.201103, "ERRORS": 0, "AVG_DURATION": 44.201103, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 634, "MAX_DURATION": 634, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 0.0, "COUNT": 363}, {"from": 189440000, "to": 194560000, "AVG_TTFB": 46.317936, "ERRORS": 0, "AVG_DURATION": 46.317936, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 1454, "MAX_DURATION": 1454, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 194560000, "to": 199680000, "AVG_TTFB": 45.19726, "ERRORS": 0, "AVG_DURATION": 45.19726, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 1015, "MAX_DURATION": 1015, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 199680000, "to": 204800000, "AVG_TTFB": 42.530056, "ERRORS": 0, "AVG_DURATION": 42.530056, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 51, "MAX_DURATION": 51, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 204800000, "to": 209920000, "AVG_TTFB": 42.515236, "ERRORS": 0, "AVG_DURATION": 42.515236, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07050781, "THROUGHPUT": 0.0, "COUNT": 361}, {"from": 209920000, "to": 215040000, "AVG_TTFB": 42.563187, "ERRORS": 0, "AVG_DURATION": 42.563187, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 96, "MAX_DURATION": 96, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 215040000, "to": 220160000, "AVG_TTFB": 42.72576, "ERRORS": 0, "AVG_DURATION": 42.72576, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 152, "MAX_DURATION": 152, "ELEMENTS_PER_SECOND": 0.07050781, "THROUGHPUT": 0.0, "COUNT": 361}, {"from": 220160000, "to": 225280000, "AVG_TTFB": 44.265755, "ERRORS": 0, "AVG_DURATION": 44.265755, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 698, "MAX_DURATION": 698, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 225280000, "to": 230400000, "AVG_TTFB": 42.488766, "ERRORS": 0, "AVG_DURATION": 42.488766, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.06953125, "THROUGHPUT": 0.0, "COUNT": 356}, {"from": 230400000, "to": 235520000, "AVG_TTFB": 42.485096, "ERRORS": 0, "AVG_DURATION": 42.485096, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 0.0, "COUNT": 369}, {"from": 235520000, "to": 240640000, "AVG_TTFB": 42.515068, "ERRORS": 0, "AVG_DURATION": 42.515068, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 240640000, "to": 245760000, "AVG_TTFB": 42.356163, "ERRORS": 0, "AVG_DURATION": 42.356163, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 245760000, "to": 250880000, "AVG_TTFB": 42.270718, "ERRORS": 0, "AVG_DURATION": 42.270718, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 0.0, "COUNT": 362}, {"from": 250880000, "to": 256000000, "AVG_TTFB": 42.364132, "ERRORS": 0, "AVG_DURATION": 42.364132, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 67, "MAX_DURATION": 67, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 256000000, "to": 261120000, "AVG_TTFB": 42.40822, "ERRORS": 0, "AVG_DURATION": 42.40822, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 61, "MAX_DURATION": 61, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 261120000, "to": 266240000, "AVG_TTFB": 42.84153, "ERRORS": 0, "AVG_DURATION": 42.84153, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 235, "MAX_DURATION": 235, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 266240000, "to": 271360000, "AVG_TTFB": 42.42033, "ERRORS": 0, "AVG_DURATION": 42.42033, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 46, "MAX_DURATION": 46, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 271360000, "to": 276480000, "AVG_TTFB": 42.51233, "ERRORS": 0, "AVG_DURATION": 42.51233, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 276480000, "to": 281600000, "AVG_TTFB": 42.81967, "ERRORS": 0, "AVG_DURATION": 42.81967, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 145, "MAX_DURATION": 145, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 281600000, "to": 286720000, "AVG_TTFB": 42.580822, "ERRORS": 0, "AVG_DURATION": 42.580822, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 286720000, "to": 291840000, "AVG_TTFB": 42.546963, "ERRORS": 0, "AVG_DURATION": 42.546963, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 46, "MAX_DURATION": 46, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 0.0, "COUNT": 362}, {"from": 291840000, "to": 296960000, "AVG_TTFB": 42.62295, "ERRORS": 0, "AVG_DURATION": 42.62295, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 296960000, "to": 302080000, "AVG_TTFB": 42.49587, "ERRORS": 0, "AVG_DURATION": 42.49587, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 0.0, "COUNT": 363}, {"from": 302080000, "to": 307200000, "AVG_TTFB": 42.535713, "ERRORS": 0, "AVG_DURATION": 42.535713, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 46, "MAX_DURATION": 46, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 307200000, "to": 312320000, "AVG_TTFB": 42.446865, "ERRORS": 0, "AVG_DURATION": 42.446865, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 312320000, "to": 317440000, "AVG_TTFB": 45.04155, "ERRORS": 0, "AVG_DURATION": 45.04155, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 957, "MAX_DURATION": 957, "ELEMENTS_PER_SECOND": 0.07050781, "THROUGHPUT": 0.0, "COUNT": 361}, {"from": 317440000, "to": 322560000, "AVG_TTFB": 42.52186, "ERRORS": 0, "AVG_DURATION": 42.52186, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 322560000, "to": 327680000, "AVG_TTFB": 42.48229, "ERRORS": 0, "AVG_DURATION": 42.48229, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 327680000, "to": 332800000, "AVG_TTFB": 42.40274, "ERRORS": 0, "AVG_DURATION": 42.40274, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 332800000, "to": 337920000, "AVG_TTFB": 42.429752, "ERRORS": 0, "AVG_DURATION": 42.429752, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 0.0, "COUNT": 363}, {"from": 337920000, "to": 343040000, "AVG_TTFB": 42.431694, "ERRORS": 0, "AVG_DURATION": 42.431694, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 343040000, "to": 348160000, "AVG_TTFB": 42.398907, "ERRORS": 0, "AVG_DURATION": 42.398907, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 46, "MAX_DURATION": 46, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 348160000, "to": 353280000, "AVG_TTFB": 42.38692, "ERRORS": 0, "AVG_DURATION": 42.38692, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 353280000, "to": 358400000, "AVG_TTFB": 42.516483, "ERRORS": 0, "AVG_DURATION": 42.516483, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 358400000, "to": 363520000, "AVG_TTFB": 42.52459, "ERRORS": 0, "AVG_DURATION": 42.52459, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 363520000, "to": 368640000, "AVG_TTFB": 44.554348, "ERRORS": 0, "AVG_DURATION": 44.554348, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 819, "MAX_DURATION": 819, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 368640000, "to": 373760000, "AVG_TTFB": 42.463013, "ERRORS": 0, "AVG_DURATION": 42.463013, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 373760000, "to": 378880000, "AVG_TTFB": 42.807064, "ERRORS": 0, "AVG_DURATION": 42.807064, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 182, "MAX_DURATION": 182, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 378880000, "to": 384000000, "AVG_TTFB": 42.527626, "ERRORS": 0, "AVG_DURATION": 42.527626, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 0.0, "COUNT": 362}, {"from": 384000000, "to": 389120000, "AVG_TTFB": 42.449593, "ERRORS": 0, "AVG_DURATION": 42.449593, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 389120000, "to": 394240000, "AVG_TTFB": 42.473972, "ERRORS": 0, "AVG_DURATION": 42.473972, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 394240000, "to": 399360000, "AVG_TTFB": 50.043716, "ERRORS": 0, "AVG_DURATION": 50.043716, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 1529, "MAX_DURATION": 1529, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 399360000, "to": 404480000, "AVG_TTFB": 42.423077, "ERRORS": 0, "AVG_DURATION": 42.423077, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 404480000, "to": 409600000, "AVG_TTFB": 42.464287, "ERRORS": 0, "AVG_DURATION": 42.464287, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 409600000, "to": 414720000, "AVG_TTFB": 43.696476, "ERRORS": 0, "AVG_DURATION": 43.696476, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 497, "MAX_DURATION": 497, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 0.0, "COUNT": 369}, {"from": 414720000, "to": 419840000, "AVG_TTFB": 42.5, "ERRORS": 0, "AVG_DURATION": 42.5, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 419840000, "to": 424960000, "AVG_TTFB": 42.472374, "ERRORS": 0, "AVG_DURATION": 42.472374, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07070313, "THROUGHPUT": 0.0, "COUNT": 362}, {"from": 424960000, "to": 430080000, "AVG_TTFB": 42.472527, "ERRORS": 0, "AVG_DURATION": 42.472527, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 430080000, "to": 435200000, "AVG_TTFB": 42.525887, "ERRORS": 0, "AVG_DURATION": 42.525887, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 435200000, "to": 440320000, "AVG_TTFB": 42.51366, "ERRORS": 0, "AVG_DURATION": 42.51366, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 440320000, "to": 445440000, "AVG_TTFB": 42.469944, "ERRORS": 0, "AVG_DURATION": 42.469944, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 445440000, "to": 450560000, "AVG_TTFB": 42.49863, "ERRORS": 0, "AVG_DURATION": 42.49863, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 450560000, "to": 455680000, "AVG_TTFB": 42.464672, "ERRORS": 0, "AVG_DURATION": 42.464672, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 455680000, "to": 460800000, "AVG_TTFB": 42.476963, "ERRORS": 0, "AVG_DURATION": 42.476963, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.072070315, "THROUGHPUT": 0.0, "COUNT": 369}, {"from": 460800000, "to": 465920000, "AVG_TTFB": 42.516483, "ERRORS": 0, "AVG_DURATION": 42.516483, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 47, "MAX_DURATION": 47, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 465920000, "to": 471040000, "AVG_TTFB": 42.50411, "ERRORS": 0, "AVG_DURATION": 42.50411, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 471040000, "to": 476160000, "AVG_TTFB": 43.277927, "ERRORS": 0, "AVG_DURATION": 43.277927, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 338, "MAX_DURATION": 338, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 476160000, "to": 481280000, "AVG_TTFB": 42.41011, "ERRORS": 0, "AVG_DURATION": 42.41011, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 50, "MAX_DURATION": 50, "ELEMENTS_PER_SECOND": 0.06953125, "THROUGHPUT": 0.0, "COUNT": 356}, {"from": 481280000, "to": 486400000, "AVG_TTFB": 42.442623, "ERRORS": 0, "AVG_DURATION": 42.442623, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 486400000, "to": 491520000, "AVG_TTFB": 42.40274, "ERRORS": 0, "AVG_DURATION": 42.40274, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 491520000, "to": 496640000, "AVG_TTFB": 42.73297, "ERRORS": 0, "AVG_DURATION": 42.73297, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 121, "MAX_DURATION": 121, "ELEMENTS_PER_SECOND": 0.07167969, "THROUGHPUT": 0.0, "COUNT": 367}, {"from": 496640000, "to": 501760000, "AVG_TTFB": 42.460056, "ERRORS": 0, "AVG_DURATION": 42.460056, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 0.0, "COUNT": 363}, {"from": 501760000, "to": 506880000, "AVG_TTFB": 42.550964, "ERRORS": 0, "AVG_DURATION": 42.550964, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.070898436, "THROUGHPUT": 0.0, "COUNT": 363}, {"from": 506880000, "to": 512000000, "AVG_TTFB": 42.51639, "ERRORS": 0, "AVG_DURATION": 42.51639, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 512000000, "to": 517120000, "AVG_TTFB": 42.445354, "ERRORS": 0, "AVG_DURATION": 42.445354, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 517120000, "to": 522240000, "AVG_TTFB": 42.575344, "ERRORS": 0, "AVG_DURATION": 42.575344, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 50, "MAX_DURATION": 50, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 522240000, "to": 527360000, "AVG_TTFB": 42.630436, "ERRORS": 0, "AVG_DURATION": 42.630436, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 55, "MAX_DURATION": 55, "ELEMENTS_PER_SECOND": 0.071875, "THROUGHPUT": 0.0, "COUNT": 368}, {"from": 527360000, "to": 532480000, "AVG_TTFB": 42.60929, "ERRORS": 0, "AVG_DURATION": 42.60929, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 46, "MAX_DURATION": 46, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 532480000, "to": 537600000, "AVG_TTFB": 48.63115, "ERRORS": 0, "AVG_DURATION": 48.63115, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 2254, "MAX_DURATION": 2254, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 537600000, "to": 542720000, "AVG_TTFB": 42.598362, "ERRORS": 0, "AVG_DURATION": 42.598362, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 48, "MAX_DURATION": 48, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 542720000, "to": 547840000, "AVG_TTFB": 42.56011, "ERRORS": 0, "AVG_DURATION": 42.56011, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07148437, "THROUGHPUT": 0.0, "COUNT": 366}, {"from": 547840000, "to": 552960000, "AVG_TTFB": 42.575344, "ERRORS": 0, "AVG_DURATION": 42.575344, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 49, "MAX_DURATION": 49, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 552960000, "to": 558080000, "AVG_TTFB": 42.46154, "ERRORS": 0, "AVG_DURATION": 42.46154, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 46, "MAX_DURATION": 46, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 558080000, "to": 563200000, "AVG_TTFB": 42.384617, "ERRORS": 0, "AVG_DURATION": 42.384617, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 563200000, "to": 568320000, "AVG_TTFB": 43.38889, "ERRORS": 0, "AVG_DURATION": 43.38889, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 380, "MAX_DURATION": 380, "ELEMENTS_PER_SECOND": 0.0703125, "THROUGHPUT": 0.0, "COUNT": 360}, {"from": 568320000, "to": 573440000, "AVG_TTFB": 42.48077, "ERRORS": 0, "AVG_DURATION": 42.48077, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 573440000, "to": 578560000, "AVG_TTFB": 42.4958, "ERRORS": 0, "AVG_DURATION": 42.4958, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.069726564, "THROUGHPUT": 0.0, "COUNT": 357}, {"from": 578560000, "to": 583680000, "AVG_TTFB": 42.63014, "ERRORS": 0, "AVG_DURATION": 42.63014, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 42, "ERROR_RATE": 0.0, "MIN_TTFB": 42, "MAX_TTFB": 46, "MAX_DURATION": 46, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 583680000, "to": 588800000, "AVG_TTFB": 42.46575, "ERRORS": 0, "AVG_DURATION": 42.46575, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 588800000, "to": 593920000, "AVG_TTFB": 44.655556, "ERRORS": 0, "AVG_DURATION": 44.655556, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 600, "MAX_DURATION": 600, "ELEMENTS_PER_SECOND": 0.0703125, "THROUGHPUT": 0.0, "COUNT": 360}, {"from": 593920000, "to": 599040000, "AVG_TTFB": 42.582417, "ERRORS": 0, "AVG_DURATION": 42.582417, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 55, "MAX_DURATION": 55, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 599040000, "to": 604160000, "AVG_TTFB": 42.554947, "ERRORS": 0, "AVG_DURATION": 42.554947, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.07109375, "THROUGHPUT": 0.0, "COUNT": 364}, {"from": 604160000, "to": 609280000, "AVG_TTFB": 42.38904, "ERRORS": 0, "AVG_DURATION": 42.38904, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 45, "MAX_DURATION": 45, "ELEMENTS_PER_SECOND": 0.07128906, "THROUGHPUT": 0.0, "COUNT": 365}, {"from": 609280000, "to": 614400000, "AVG_TTFB": 42.379486, "ERRORS": 0, "AVG_DURATION": 42.379486, "ERRORS_PER_SECOND": 0.0, "MIN_DURATION": 41, "ERROR_RATE": 0.0, "MIN_TTFB": 41, "MAX_TTFB": 44, "MAX_DURATION": 44, "ELEMENTS_PER_SECOND": 0.038085938, "THROUGHPUT": 0.0, "COUNT": 195}]'

        else:
            raise Exception('Endpoint NOT mocked : ' + endpoint)
