# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['odp_gent']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.0.0', 'yarl>=1.6.0']

setup_kwargs = {
    'name': 'odp-gent',
    'version': '0.2.0',
    'description': 'Asynchronous Python client providing Open Data information of Gent',
    'long_description': '<!-- Banner -->\n![alt Banner of the odp gent package](https://raw.githubusercontent.com/klaasnicolaas/python-odp-gent/main/assets/header_odp_gent-min.png)\n\n<!-- PROJECT SHIELDS -->\n[![GitHub Release][releases-shield]][releases]\n[![Python Versions][python-versions-shield]][pypi]\n![Project Stage][project-stage-shield]\n![Project Maintenance][maintenance-shield]\n[![License][license-shield]](LICENSE)\n\n[![GitHub Activity][commits-shield]][commits-url]\n[![PyPi Downloads][downloads-shield]][downloads-url]\n[![GitHub Last Commit][last-commit-shield]][commits-url]\n[![Stargazers][stars-shield]][stars-url]\n[![Issues][issues-shield]][issues-url]\n\n[![Code Quality][code-quality-shield]][code-quality]\n[![Maintainability][maintainability-shield]][maintainability-url]\n[![Code Coverage][codecov-shield]][codecov-url]\n\n[![Build Status][build-shield]][build-url]\n[![Typing Status][typing-shield]][typing-url]\n\nAsynchronous Python client for the open datasets of Gent (Belgium).\n\n## About\n\nA python package with which you can retrieve data from the Open Data Platform of Gent via [their API][api]. This package was initially created to only retrieve parking data from the API, but the code base is made in such a way that it is easy to extend for other datasets from the same platform.\n\n## Installation\n\n```bash\npip install odp-gent\n```\n\n## Datasets\n\nYou can read the following datasets with this package:\n\n- Parking garages occupancy (12 locations)\n- Park and Ride occupancy (5 locations)\n\n<details>\n    <summary>Click here to get more details</summary>\n\n### Parking garages\n\nParameters:\n\n- **limit** (default: 10) - How many results you want to retrieve.\n\n| Variable | Type | Description |\n| :------- | :--- | :---------- |\n| `garage_id` | string | The id of the garage |\n| `name` | string | The name of the garage |\n| `parking_type` | string | The type of parking |\n| `url` | string | The url with more information about the garage |\n| `is_open` | boolean | Whether the garage is open or not |\n| `free_parking` | boolean | Whether there is free parking or not |\n| `temporary_closed` | boolean | Whether the garage is temporarily closed or not |\n| `free_space` | integer | The amount of free parking spaces |\n| `total_capacity` | integer | The total capacity of the garage |\n| `availability_pct` | float | The percentage of free parking spaces |\n| `occupancy_pct` | integer | The percentage of occupied parking spaces |\n| `longitude` | float | The longitude of the garage |\n| `latitude` | float | The latitude of the garage |\n| `updated_at` | datetime | The last time the data was updated |\n\n### Park and Ride\n\nParameters:\n\n- **limit** (default: 10) - How many results you want to retrieve.\n- **gentse_feesten** - Whether a park and ride location is used for the [Gentse Feesten](https://gentsefeesten.stad.gent).\n\n| Variable | Type | Description |\n| :------- | :--- | :---------- |\n| `spot_id` | string | The id of the park and ride |\n| `name` | string | The name of the park and ride |\n| `parking_type` | string | The type of parking |\n| `url` | string | The url with more information about the park and ride |\n| `is_open` | boolean | Whether the park and ride is open or not |\n| `free_parking` | boolean | Whether there is free parking or not |\n| `temporary_closed` | boolean | Whether the park and ride is temporarily closed or not |\n| `gentse_feesten` | boolean | Whether the park and ride is used for the [Gentse Feesten](https://gentsefeesten.stad.gent) |\n| `free_space` | integer | The amount of free parking spaces |\n| `total_capacity` | integer | The total capacity of the park and ride |\n| `availability_pct` | float | The percentage of free parking spaces |\n| `occupancy_pct` | integer | The percentage of occupied parking spaces |\n| `longitude` | float | The longitude of the park and ride |\n| `latitude` | float | The latitude of the park and ride |\n| `updated_at` | datetime | The last time the data was updated |\n</details>\n\n## Example\n\n```python\nimport asyncio\n\nfrom odp_gent import ODPGent\n\n\nasync def main() -> None:\n    """Show example on using the Open Data API client."""\n    async with ODPGent() as client:\n        garages = await client.garages(limit=12)\n        park_and_rides = await client.park_and_rides(limit=5, gentse_feesten="True")\n        print(garages)\n        print(park_and_rides)\n\n\nif __name__ == "__main__":\n    asyncio.run(main())\n```\n\n## Use cases\n\n[NIPKaart.nl][nipkaart]\n\nA website that provides insight into where disabled parking spaces are, based\non data from users and municipalities. Operates mainly in the Netherlands, but\nalso has plans to process data from abroad.\n\n## Contributing\n\nThis is an active open-source project. We are always open to people who want to\nuse the code or contribute to it.\n\nWe\'ve set up a separate document for our\n[contribution guidelines](CONTRIBUTING.md).\n\nThank you for being involved! :heart_eyes:\n\n## Setting up development environment\n\nThis Python project is fully managed using the [Poetry][poetry] dependency\nmanager.\n\nYou need at least:\n\n- Python 3.9+\n- [Poetry][poetry-install]\n\nInstall all packages, including all development requirements:\n\n```bash\npoetry install\n```\n\nPoetry creates by default an virtual environment where it installs all\nnecessary pip packages, to enter or exit the venv run the following commands:\n\n```bash\npoetry shell\nexit\n```\n\nSetup the pre-commit check, you must run this inside the virtual environment:\n\n```bash\npre-commit install\n```\n\n*Now you\'re all set to get started!*\n\nAs this repository uses the [pre-commit][pre-commit] framework, all changes\nare linted and tested with each commit. You can run all checks and tests\nmanually, using the following command:\n\n```bash\npoetry run pre-commit run --all-files\n```\n\nTo run just the Python tests:\n\n```bash\npoetry run pytest\n```\n\n## License\n\nMIT License\n\nCopyright (c) 2022 Klaas Schoute\n\nPermission is hereby granted, free of charge, to any person obtaining a copy\nof this software and associated documentation files (the "Software"), to deal\nin the Software without restriction, including without limitation the rights\nto use, copy, modify, merge, publish, distribute, sublicense, and/or sell\ncopies of the Software, and to permit persons to whom the Software is\nfurnished to do so, subject to the following conditions:\n\nThe above copyright notice and this permission notice shall be included in all\ncopies or substantial portions of the Software.\n\nTHE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR\nIMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,\nFITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE\nAUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER\nLIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,\nOUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE\nSOFTWARE.\n\n[api]: https://data.stad.gent/explore\n[nipkaart]: https://www.nipkaart.nl\n\n<!-- MARKDOWN LINKS & IMAGES -->\n[build-shield]: https://github.com/klaasnicolaas/python-odp-gent/actions/workflows/tests.yaml/badge.svg\n[build-url]: https://github.com/klaasnicolaas/python-odp-gent/actions/workflows/tests.yaml\n[code-quality-shield]: https://img.shields.io/lgtm/grade/python/g/klaasnicolaas/python-odp-gent.svg?logo=lgtm&logoWidth=18\n[code-quality]: https://lgtm.com/projects/g/klaasnicolaas/python-odp-gent/context:python\n[commits-shield]: https://img.shields.io/github/commit-activity/y/klaasnicolaas/python-odp-gent.svg\n[commits-url]: https://github.com/klaasnicolaas/python-odp-gent/commits/main\n[codecov-shield]: https://codecov.io/gh/klaasnicolaas/python-odp-gent/branch/main/graph/badge.svg?token=5JNbz4akUL\n[codecov-url]: https://codecov.io/gh/klaasnicolaas/python-odp-gent\n[downloads-shield]: https://img.shields.io/pypi/dm/odp-gent\n[downloads-url]: https://pypistats.org/packages/odp-gent\n[issues-shield]: https://img.shields.io/github/issues/klaasnicolaas/python-odp-gent.svg\n[issues-url]: https://github.com/klaasnicolaas/python-odp-gent/issues\n[license-shield]: https://img.shields.io/github/license/klaasnicolaas/python-odp-gent.svg\n[last-commit-shield]: https://img.shields.io/github/last-commit/klaasnicolaas/python-odp-gent.svg\n[maintenance-shield]: https://img.shields.io/maintenance/yes/2022.svg\n[maintainability-shield]: https://api.codeclimate.com/v1/badges/ceb27fb15cf0e485dc23/maintainability\n[maintainability-url]: https://codeclimate.com/github/klaasnicolaas/python-odp-gent/maintainability\n[project-stage-shield]: https://img.shields.io/badge/project%20stage-production%20ready-brightgreen.svg\n[pypi]: https://pypi.org/project/odp-gent/\n[python-versions-shield]: https://img.shields.io/pypi/pyversions/odp-gent\n[typing-shield]: https://github.com/klaasnicolaas/python-odp-gent/actions/workflows/typing.yaml/badge.svg\n[typing-url]: https://github.com/klaasnicolaas/python-odp-gent/actions/workflows/typing.yaml\n[releases-shield]: https://img.shields.io/github/release/klaasnicolaas/python-odp-gent.svg\n[releases]: https://github.com/klaasnicolaas/python-odp-gent/releases\n[stars-shield]: https://img.shields.io/github/stars/klaasnicolaas/python-odp-gent.svg\n[stars-url]: https://github.com/klaasnicolaas/python-odp-gent/stargazers\n\n[poetry-install]: https://python-poetry.org/docs/#installation\n[poetry]: https://python-poetry.org\n[pre-commit]: https://pre-commit.com\n',
    'author': 'Klaas Schoute',
    'author_email': 'hello@student-techlife.com',
    'maintainer': 'Klaas Schoute',
    'maintainer_email': 'hello@student-techlife.com',
    'url': 'https://github.com/klaasnicolaas/python-odp-gent',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
