# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['targe']

package_data = \
{'': ['*']}

install_requires = \
['gid>=1.0.1,<2.0.0']

setup_kwargs = {
    'name': 'targe',
    'version': '0.3.0',
    'description': 'Powerful and flexible policy based authorization library.',
    'long_description': '# Targe\nPowerful and flexible authorization library utilizing policy system that reflects your domain.\n\n\n## Installation\n\nWith pip:\n```\npip install targe\n```\n\nor with poetry\n\n```\npoetry add targe\n```\n\n## Quick start\n\n```python\nfrom targe import Auth, ActorProvider, Actor, Policy\nfrom targe.errors import AccessDeniedError\n\n\n# This will provide actor for auth mechanism\nclass MyActorProvider(ActorProvider):\n    def get_actor(self, actor_id: str) -> Actor:\n        return Actor(actor_id)\n\n\n# Initialise auth class\nauth = Auth(MyActorProvider())\n\n# Retrieves and authorizes actor by its id\nauth.authorize("actor_id")\n\n\n# `auth.guard` decorator assigns auth scope to a function and\n# protects it from non-authorized access\n@auth.guard(scope="protected")\ndef protect_this() -> None:\n    ...  # code that should be protected by auth\n\n\ntry:\n    protect_this()\nexcept AccessDeniedError:\n    ...  # this will fail as actor has no access to scope `protected`\n\nauth.actor.policies.append(Policy.allow("protected"))  # add `protected` scope to actor policies\nprotect_this()  # now this works\n```\n\n## Features\n\n### Customisable and flexible policy system\nPolicy system in **Targe** is not limited to specific keywords like `read`, `write`, `create`, etc. \nInstead it uses scopes, scopes can hold any value that makes sense in your application\'s domain \nlike `salads : eat`. To increase flexibility and control over your domain **Targe** allows for defining dynamic\nscopes that can point to specific data in your application.\n\n### Minimal, close to 0 learning curve\nIf you already have some experience with other `acl` or `authorization` libraries there is \nalmost 0 learning curve. In order to start using the library you only need to learn these 5 methods:\n- `Auth.guard`\n- `Auth.guard_after`\n- `Policy.allow`\n- `Policy.deny`\n- `ActorProvider.get_actor`\n\n### Built-in audit log\nEverytime guarded function is executed library creates a log entry. This log entries can be persisted\nand used later on to understand who, when, how and what has changed within your application.\n\n### Elegant and easy to use interface\nYou don\'t have to write complex `if` statements asserting whether user has given role or policy. \nAll of that happens automatically in one small `@guard` decorator, which can be attached to \nany function/method within your codebase and easily removed if needed. \n\n# Usage\n\n## Execution flow\n\nThe following diagram is a high level representation of the execution flow:\n\n![Execution Flow](./docs/targe@2x-2.png)\n\n\nWhen protected function gets called, `targe.Auth` class is checking whether `actor` is accessible \n(`targe.Auth.init` is responsible for providing valid instance of `targe.Actor`). \n\nIf for some reason `actor` is not accessible (`Auth.init` was not called or `targe.ActorProvider` has failed) \n`UnauthorizedError` exception is raised. \n\nWhen `actor` is present, library will attempt to resolve `scope` and assert whether `actor` has required\npolicy or access to given scope. Once this is done library will automatically generate an audit log, \nwhich can be persisted in database for future reference.\n\nFinally, if actor has sufficient rights guarded function is executed, otherwise `targe.AccessDeniedError` is raised.\n\n## Actor\nActor represents authenticated user in your application. Other important characteristics of an actor are:\n- an actor aggregates permissions and roles\n- an actor encapsulates its state and may act upon its change  \n- actor knows whether is can access given scope\n- actor\'s id is referenced in audit log  \n- actor can be extended further to encapsulate your application logic \n\n### Creating new actor\n\n```python\nfrom targe import Actor\n\nmy_actor = Actor("actor_id")\n```\n\n### Assigning policies\n\n```python\nfrom targe import Actor, Policy\n\nmy_actor = Actor("actor_id")\n\n# assign policies \nmy_actor.policies.append(Policy.allow("articles : update"))\n```\n\n> Note: whitespaces in scope are irrelevant, both `articles:update`, `articles : update`\n> are equal from the library points of view.\n\n### Assigning roles\n\n```python\nfrom targe import Actor, Policy, Role\n\nmy_actor = Actor("actor_id")\n\n# simple role\nuser_manager = Role("user_manager")\nuser_manager.policies.append(Policy.allow("user:*"))\n\n# assign role\nmy_actor.roles.append(user_manager)\n```\n\n### Providing an actor to auth system\nBy default, auth system does not know who is your actor and what it can do. \n\nTo provide information about your actor, you have to implement `targe.ActorProvider` protocol, \nplease consider the following example:\n\n```python\nfrom targe import ActorProvider, Actor, Auth\n\n\nclass MyActorProvider(ActorProvider):\n    def get_actor(self, actor_id: str) -> Actor:\n        ...  # you can query your database or do other relevant task to factory your instance of `targe.Actor`\n        return Actor(actor_id)\n\n\n# now we have to just instantiate auth and pass instance of our ActorProvider implementation\nauth = Auth(MyActorProvider())\n\n# The following line will cause auth system to use `MyActorProvider.get_actor` method.\nauth.authorize("actor_id")\n```\n\n## Policies\n\n**Policy** is an object representing logical rule that can either allow or deny accessing\ncertain areas in your application. \nOnce policies are created they can ba attached to a role, or a user to ensure fine-grained\naccess control.\n\nPolicies contain `scopes` and `effect`. The first ones holds an information how data is \nbeing accessed within your application (`read`, `write`, `update`, `etc`), the latter ones \ntell whether scope is accessible or not.\n\nYou can define a policy like in the example below:\n```python\nfrom targe import Policy\n\npolicy = Policy.allow(scope="articles : update")\n```\n\n### Scopes\n\nScope is a string representing a logical boundary within your application in which data is being \naccessed and/or manipulated. This string must follow the pattern [a-zA-Z][a-zA-Z0-9-_]*, whitespaces are ignored.\nCharacters like: `:`, `,`, `*` are allowed in the scope, but they have specific meaning:\n\n- `:` is used to build namespaced scopes\n- `,` is used to match multiple scope sections\n- `*` is used for pattern matching expressions\n\nScopes can be used in policies and in guarded functions. Use scopes in policies to set rules for existing logical \nboundaries in our application. To define those boundaries scopes must be provided in `Auth.guard` decorator, which\ndecorates given function or method.\n\nThe following is a list of valid policy scopes:\n```\narticles\narticles : update\narticles : update : article_id\narticles : create, upate, delete\narticles : *\narticles : meta : set-*\narticles : meta : *Name\narticles : update : * : tags\n```\n\n#### Pattern matching\n\nLet\'s review the following code snippet which defines multiple policies:\n\n```python\nfrom targe import Policy\n\nPolicy.allow("article : meta : setKeywords")\nPolicy.allow("article : meta : setVersion")\nPolicy.allow("article : meta : setCategory")\nPolicy.allow("article : meta : getKeywords")\nPolicy.allow("article : meta : getVersion")\nPolicy.allow("article : meta : getCategory")\n```\n\nIt is quite repetitive task which can be simplified by using pattern matching in policy\'s scopes:\n\n```python\nfrom targe import Policy\n\nPolicy.allow("article : meta : set*")\nPolicy.allow("article : meta : get*")\n```\n\nAn asterisk at the end of each scope tells **Targe** to use pattern matching mechanism. \nFirst policy might be interpreted as "match all the scopes which start with `article` namespace followed by `meta` \nnamespace followed by namespace that starts with a `set` word". Second one is very similar but last namespace \nhas to start with `get` word instead. \n\nHere are some examples to help you understand how pattern matching works:\n```\narticle : *\n```\nMatch all scopes that start with article namespace.\n```\narticle : * : id\n```\nMatch all scopes that start with an `article`, has `any` namespace after that and ends with a `id`.\n```\narticle : *Name\n```\nMatch all scopes that start with an `article` namespace and are followed by namespace that ends with `Name`\n\nLet\'s now go back last time to our example that we simplified with pattern matching. This can be simplified \neven further with grouping. Let\'s consider the following code snippet:\n\n```python\nfrom targe import Policy\n\nPolicy.allow("article : meta : set*, get*")\n```\n\nNow with the above policy we can match all the scopes that were presented at the beginning of this chapter.\n\n## Roles\n\nRole is a collection of policies with a unique name. Roles can be also \nused to build Role-based access control (RBAC), which is a simplified mechanism\nfor regulating access to part of your application based on the roles of an individual actor.\n\nThe following is an example code, where `user_manager` role is defined:\n\n```python\nfrom targe import Role, Policy\n\nrole = Role("user_manager")\n\n# You can also attach policies to a role, but it is not needed in RBAC scenario\nrole.policies.append(Policy.allow("user : create, update, delete, read"))\n```\n\n> Role names must follow [a-z][a-z0-9_-]+ pattern. Role name is also its identifier, \n> thus they should be unique across your application.\n\n## Guarding function\n\nProtecting function from unauthorized access is one of the **Targe**\'s main objectives.\n\nWe can protect function from unauthorized execution in two styles:\n- acl based style\n- rbac style\n\nUse rbac style in scenarios where you have to just assert if actor has given role, use acl based style in other cases.\nACL based style is not only giving you more control over your resources but also automatically enables audit log. \n\n### Guarding function - rbac style example\n\nTo protect function from unauthorized execution use `Auth.guard(rbac=[...])` decorator with `rbac` argument. The `rbac`\nargument accepts list of strings where each string is a role name that is required in to execute annotated function.\n\n> If more than one role is passed in the `rbac` argument, this means actor has to own all the required roles\n> to execute annotated function.\n\n```python\nfrom targe import ActorProvider, Actor, Auth\nfrom targe.errors import AccessDeniedError\n\n\nclass MyActorProvider(ActorProvider):\n    def get_actor(self, actor_id: str) -> Actor:\n        return Actor(actor_id)\n\n\nauth = Auth(MyActorProvider())\n\nauth.authorize("actor_id")\n\n\n@auth.guard(requires=["user_manager"])  # Here we use `Auth.guard` decorator to protect `create_user` function\ndef create_user() -> None:\n    ...\n\n\ntry:\n    create_user()\nexcept AccessDeniedError:\n    print("`create_user` is protected from unauthorized access.")\n```\n\n> Keep in mind you can still take advantage of audit log in rbac mode, \n> the only requirement is to provide `scope` argument in `Auth.guard` decorator.\n\n### Guarding function - acl style example\n\n```python\nfrom targe import ActorProvider, Actor, Auth\nfrom targe.errors import AccessDeniedError\n\nclass MyActorProvider(ActorProvider):\n    def get_actor(self, actor_id: str) -> Actor:\n        return Actor(actor_id)\n    \nauth = Auth(MyActorProvider())\n\nauth.authorize("actor_id")\n\n@auth.guard(scope="user : create") \ndef create_user() -> None:\n    ...\n\ntry:\n    create_user()\nexcept AccessDeniedError:\n    print("`create_user` is protected from unauthorized access.") \n```\n\n#### Dynamic scopes\n\nSometimes you might run in scenarios where you would like to limit access to given entity\nor group of entities. In the scenarios like that dynamic scopes might come in handy. \nDynamic scope contains placeholders for values held by arguments passed to guarded function.\nEverytime function is being called placeholders are replaced with corresponding values.\n\n```python\nfrom targe import ActorProvider, Actor, Auth, Policy\nfrom targe.errors import AccessDeniedError\n\nclass MyActorProvider(ActorProvider):\n    def get_actor(self, actor_id: str) -> Actor:\n        # lest initialize actor and allow access to `articles:update:allowed_id` scope.\n        actor = Actor(actor_id)\n        actor.policies.append(Policy.allow("articles:update:allowed_id"))\n        \n        return actor\n    \nauth = Auth(MyActorProvider())\n\nauth.authorize("actor_id")\n\nclass Article:\n    def __init__(self, article_id: str, body: str):\n        self.article_id = article_id\n        self.body = body\n\n# Here we define dynamic scope that holds reference to function\'s parameter `article` \n# and tries to access its property `article_id`\n@auth.guard(scope="article : update : { article.article_id }") \ndef update_article(article: Article) -> None:\n    print("article updated")\n\n# the following attempt will fail as actor has access only to `article:update:allowed_id` scope\ntry:\n    update_article(Article("other_id", "Lorem Ipsum"))\nexcept AccessDeniedError:\n    print("`update_article` is protected from unauthorized access.") \n\n# this line will succeed\nupdate_article(Article("allowed_id", "Lorem Ipsum"))\n```\n\n### Overriding function guarding mechanism\n\nYou can override default behavior of guard mechanism in scenarios when it denies access to guarded\nfunction. In order to do that pass a callable object to `Auth` initializer, like below:\n\n```python\nfrom targe import ActorProvider, Actor, Auth\n\nclass MyActorProvider(ActorProvider):\n    def get_actor(self, actor_id: str) -> Actor:\n        return Actor(actor_id)\n    \ndef on_guard(actor: Actor, scope: str) -> bool:\n    if scope == "user:create":\n        return True\n    \n    return False\n    \nauth = Auth(MyActorProvider(), on_guard=on_guard)\nauth.authorize("actor_id")\n\n@auth.guard(scope="user:create") \ndef create_user() -> None:\n    ...\n\ncreate_user()\n```\n\nCallable object must return `bool` value (`True` in order to allow access, `False` to deny access) and accept two parameters:\n- `actor: targe.Actor` - an actor that is currently authorized in the system\n- `scope: str` - scope assigned to guarded function\n\n\n## Audit log\n\nAudit log might be useful if you need to track actor\'s activities in your application.\nBy default, all actor\'s actions against guarded functions are automatically recorded and stored\nin memory as long as `scope` attribute is provided in the `Auth.guard` decorator. \n\n> `InMemoryAuditStore` class is a default in-memory implementation of `AuditStore` protocol, which\n> is instantiated by `Auth` class if no other implementation is provided.\n\n\n### AuditEntry structure\n\n`targe.AuditEntry` is a representation of a single actor\'s action against guarded function in your application.\n\nEach audit entry contains the following information:\n- **`entry_id`**: `str` - unique identifier 16 characters long\n- **`actor_id`**: `str` - id of authenticated actor may reference to a user in your application\n- **`scope`**: `str` - scope in which function was executed, defined in guard decorator\n- **`status`**: `targe.AuditStatus` - tells whether execution of a guarded function was successful or not\n- **`created_on`**: `datetime.datetime` - the time when action was initialized\n\n### Persisting audit log\n\nAs we already discussed by default audit log is stored only in the memory, this behaviour \ncan be simply amended by implementing `targe.AuditStore` protocol and passing instance\nof new implementation into `targe.Auth`\'s initializer:\n\n```python\nfrom targe import AuditStore, AuditEntry, Auth, ActorProvider, Actor\nimport sqlite3\n\n\nclass MyActorProvider(ActorProvider):\n    def get_actor(self, actor_id: str) -> Actor:\n        return Actor(actor_id)\n\n    \nclass MyAuditStore(AuditStore):\n    def __init__(self, connection: sqlite3.Connection):\n        self.connection = connection\n        \n    def append(self, log: AuditEntry) -> None:\n        query = self.connection.cursor()\n        query.execute(\n            "INSERT INTO audit_log VALUES(?, ?, ?, ?, ?)",\n            (\n                log.entry_id,\n                log.actor_id,\n                log.scope,\n                log.status,\n                log.created_on\n             )\n        )\n        \ndb_connection = sqlite3.connect("audit_log.db")\nauth = Auth(actor_provider=MyActorProvider(), audit_store=MyAuditStore(db_connection))\nauth.authorize("actor_id")\n...\n```\n',
    'author': 'Dawid Kraczkowski',
    'author_email': 'dawid.kraczkowski@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/kodemore/targe',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
