"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const acm = require("@aws-cdk/aws-certificatemanager");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
const app = new cdk.App();
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const stack = new cdk.Stack(app, 'demo-stack-tiny', { env });
const zoneName = `${stack.stackName}.local`;
const internalAlbRecordName = 'internal';
const internalALBEndpoint = `http://${internalAlbRecordName}.${zoneName}`;
// order service
const orderTask = new ecs.FargateTaskDefinition(stack, 'orderTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
orderTask.addContainer('order', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/OrderService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'order',
        versionNum: '1.0',
    },
});
// customer service
const customerTask = new ecs.FargateTaskDefinition(stack, 'customerTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
customerTask.addContainer('customer', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'customer',
        versionNum: '1.0',
    },
});
// product service
const productTask = new ecs.FargateTaskDefinition(stack, 'productTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
productTask.addContainer('product', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'product',
        versionNum: '1.0',
    },
});
const certArn = stack.node.tryGetContext('ACM_CERT_ARN');
const cert = certArn ? acm.Certificate.fromCertificateArn(stack, 'Cert', certArn) : undefined;
new index_1.DualAlbFargateService(stack, 'ALBService', {
    spot: true,
    enableExecuteCommand: true,
    tasks: [
        // The order service with both external/internal access
        {
            task: orderTask,
            desiredCount: 2,
            internal: { port: 80 },
            external: cert ? { port: 443, certificate: [cert] } : { port: 80 },
            // customize the service autoscaling policy
            scalingPolicy: {
                maxCapacity: 20,
                requestPerTarget: 1000,
                targetCpuUtilization: 50,
            },
        },
        {
            // The customer service(internal only)
            task: customerTask,
            desiredCount: 1,
            internal: { port: 8080 },
            capacityProviderStrategy: [
                {
                    capacityProvider: 'FARGATE',
                    base: 1,
                    weight: 1,
                },
                {
                    capacityProvider: 'FARGATE_SPOT',
                    base: 0,
                    weight: 3,
                },
            ],
        },
        // The produce service(internal only)
        {
            task: productTask,
            desiredCount: 1,
            internal: { port: 9090 },
        },
    ],
    route53Ops: {
        zoneName,
    },
});
const nlbService = new index_1.DualNlbFargateService(stack, 'NLBService', {
    spot: true,
    enableExecuteCommand: true,
    tasks: [
        // The order service with both external/internal access
        {
            task: orderTask,
            desiredCount: 2,
            internal: { port: 80 },
            external: cert ? { port: 443, certificate: [cert] } : { port: 80 },
            // customize the service autoscaling policy
            scalingPolicy: {
                maxCapacity: 20,
                requestPerTarget: 1000,
                targetCpuUtilization: 50,
            },
        },
        {
            // The customer service(internal only)
            task: customerTask,
            desiredCount: 1,
            internal: { port: 8080 },
            capacityProviderStrategy: [
                {
                    capacityProvider: 'FARGATE',
                    base: 1,
                    weight: 1,
                },
                {
                    capacityProvider: 'FARGATE_SPOT',
                    base: 0,
                    weight: 3,
                },
            ],
        },
        // The produce service(internal only)
        {
            task: productTask,
            desiredCount: 1,
            internal: { port: 9090 },
        },
    ],
    route53Ops: {
        zoneName: `${stack.stackName}-nlb.local`,
    },
});
// we need this to allow ingress traffic from public internet only for the order service
nlbService.service[0].connections.allowFromAnyIpv4(ec2.Port.tcp(8080));
// allow from VPC
nlbService.service.forEach(s => {
    s.connections.allowFrom(ec2.Peer.ipv4(nlbService.vpc.vpcCidrBlock), ec2.Port.tcp(8080));
});
//# sourceMappingURL=data:application/json;base64,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