#import grpc-package
from grpc import insecure_channel

#import class Sequence and OutoutState for advanced sequence building
from pulsestreamer.sequence import Sequence, OutputState

try:
    #import the python wrapper generated by the protocol buffer compiler (source: pulse_streamer.proto)
    from pulsestreamer.grpc.pulse_streamer_pb2 import VoidMessage, PulseMessage, SequenceMessage, ClockMessage, DrpClkSetMessage, TriggerMessage, SetNetworkMessage, EnableStaticIPMessage, HostnameMessage, PulseStreamerStub, AnalogCalibrationData
    from pulsestreamer.grpc.pulse_streamer_pb2 import GetSerialMessage # used for backwards compatibility reasons if client software detects firmware version < 1.1.0
except Exception as e:
    print('Exception: '+ str(e))
    assert False, \
"""
Failed to import simple wrapper generated by the protobuf compiler (source: pulse_streamer.proto)
- Ensure pulse_streamer_pb2.py is in the search path.
Required packages: grpcio, protobuf
Ensure that you have it installed by typing
> pip install grpcio
> pip install protobuf
"""

from pulsestreamer.enums import ClockSource, TriggerStart, TriggerRearm
from pulsestreamer.version import __CLIENT_VERSION__, _compare_version_number

#stream binary
import socket
import struct

class PulseStreamer():
    """
    Simple python wrapper for a PulseStreamer 8/2
    that describes sequences in the form of sequence steps as (time, [0,1,3], 0.8, -0.4),
    where time is an integer in ns (clock ticks),
    [0,1,3] is a list numbering the channels that should be high
    the last two numbers specify the analog outputs in volt.
    For advanced sequence creation use the method createSequence() and the functionality of
    the class Sequence described in the documentation of the Pulse Streamer 8/2.
    """

    TIMEOUT = 200
    REPEAT_INFINITELY=-1
    
    def __init__(self, ip_hostname='pulsestreamer'):
        print("Connect to Pulse Streamer via google-RPC.")
        print("IP / Hostname:", ip_hostname)
        self.ip_address = ip_hostname
        try:
            channel = insecure_channel(ip_hostname + ':50051')
            self.stub = PulseStreamerStub(channel)
            try:
                void = VoidMessage()
                self.stub.getSerial(void, timeout=PulseStreamer.TIMEOUT)
            except:
                try:
                    void = VoidMessage()
                    self.stub.isRunning(void, timeout=PulseStreamer.TIMEOUT)
                    assert False, "Pulse Streamer class not compatible with current firmware. Please update your firmware." \
                        "For detailed information visit https://www.swabianinstruments.com/pulse-streamer-8-2-firmware/ " \
                        "or contact support@swabianinstruments.com"
                except AssertionError:
                    raise
                except:
                    assert False, "No Pulse Streamer found at IP/Host-address: "+ip_hostname
        except AssertionError:
            raise
        except:
            assert False, "No Pulse Streamer found at IP/Host-address: "+ip_hostname
        void = VoidMessage()
        firmware_version=self.stub.getFirmwareVersion(void, timeout=PulseStreamer.TIMEOUT).string_value
        self.__version_1_1_or_higher=True if _compare_version_number(firmware_version, '1.1.0') >=0 else False
        print("Pulse Streamer 8/2 firmware: v" + firmware_version)
        print("Client software: v" + __CLIENT_VERSION__)
        if(_compare_version_number(firmware_version) >= 0):
            print("The Pulse Streamer 8/2 firmware is more up to date than your client software. We recommend to update your client software.")
            print("For detailed information visit https://www.swabianinstruments.com/support/downloads/ or contact support@swabianinstruments.com")
        elif(_compare_version_number('.'.join(__CLIENT_VERSION__.split('.')[0:2]+['0']),'.'.join(firmware_version.split('.')[0:2]+['0'])) > 0):
            print("Your client software is more up to date than the Pulse Streamer 8/2 firmware. We recommend to update the firmware of your Pulse Streamer 8/2.")
            print("For detailed information visit https://www.swabianinstruments.com/support/downloads/ or contact support@swabianinstruments.com")


    def reset(self):
        void = VoidMessage()
        return self.stub.reset(void, timeout=PulseStreamer.TIMEOUT).value
        
    def constant(self, state=OutputState.ZERO()):
        if isinstance(state, OutputState):
            state=state.getData()
            sequence_step = PulseMessage(ticks=0, digi=state[0], ao0=state[1], ao1=state[2])
        else:
            if isinstance(state[0], list):
                sequence_step = self.convert_sequence_step((0, state[0], state[1], state[2]))
            else:
                chans, a0, a1 = state
                sequence_step = PulseMessage(ticks=0, digi=chans, ao0=a0, ao1=a1)
        self.stub.constant(sequence_step, timeout=PulseStreamer.TIMEOUT)

    def forceFinal(self):
        void = VoidMessage()
        return self.stub.forceFinal(void, timeout=PulseStreamer.TIMEOUT).value

    def createOutputState(self, digi, A0=0.0, A1=0.0):
        output=OutputState(digi=digi,A0=A0, A1=A1)
        return output

    def createSequence(self):
        seq = Sequence()
        return seq

    def stream(self, seq, n_runs=REPEAT_INFINITELY, final=OutputState.ZERO()):
        if (self.__version_1_1_or_higher):
            #print("Debug information: Stream binary")
            #return self.__stream_binary(seq, n_runs, final)
            self.__stream_binary(seq, n_runs, final)
        else:
            #print("Debug information: Stream grpc")
            self.__stream_grpc(seq, n_runs, final)
        
    def __stream_grpc(self, seq, n_runs=REPEAT_INFINITELY, final=OutputState.ZERO()):
        if isinstance(seq, Sequence):
            sequence=seq.getData()
        else:
            sequence= seq

        s = SequenceMessage()
        
        if type(sequence[0][1])== list:
            for i in range(len(sequence)):
                t, chans, a0, a1 = sequence[i]
                if t > 0xffffffff:
                    mod=t%0xffffffff
                    count=t//0xffffffff
                    for i in range(count):
                        t =0xffffffff
                        s.pulse.extend([self.convert_sequence_step((t, chans, a0, a1))])
                    else:
                        t=mod
                s.pulse.extend([self.convert_sequence_step((t, chans, a0, a1))])
        else:
            for i in range(len(sequence)):
                t, chans, a0, a1 = sequence[i]
                if t > 0xffffffff:
                    mod=t%0xffffffff
                    count=t//0xffffffff
                    for i in range(count):
                        t =0xffffffff
                        s.pulse.extend([PulseMessage(ticks=t, digi=chans, ao0=a0, ao1=a1)])
                    else:
                        t=mod
                s.pulse.extend([PulseMessage(ticks=t, digi=chans, ao0=a0, ao1=a1)])

        assert len(s.pulse)<=2e6, "The resulting sequence length exceeds the limit of two million sequnence steps"

        s.n_runs = n_runs

        if isinstance(final, OutputState):
            final=final.getData()
            conv_final = PulseMessage(ticks=0, digi=final[0],ao0=final[1], ao1=final[2])
        else:
            if isinstance(final[0], list):
                conv_final = self.convert_sequence_step((0,final[0], final[1], final[2]))
            else:
                chans, a0, a1 = final
                conv_final = PulseMessage(ticks=0, digi=chans, ao0=a0, ao1=a1)

        s.final.ticks = conv_final.ticks
        s.final.digi = conv_final.digi
        s.final.ao0 = conv_final.ao0
        s.final.ao1 = conv_final.ao1
  
        self.stub.stream(s, timeout=PulseStreamer.TIMEOUT)

    def __stream_binary(self, seq, n_runs=REPEAT_INFINITELY, final=OutputState.ZERO()):
        HOST = self.ip_address #'192.168.0.101' #''127.0.0.1'  # The server's hostname or IP address
        PORT = 21328        # The port used by the server
        MAGIC_TOKEN = 0x53504953
        COMMAND=0
        
        if isinstance(final, OutputState):
            state=final.getData()
            final =(0, state[0], state[1], state[2])
        else:
            final = self.convert_sequence_step((0, final[0], final[1], final[2]), binary=True)

        
        if isinstance(seq, Sequence):
            s_enc = self.enc_binary(seq.getData())
        else:
            s_enc = self.enc_binary(seq)
        
        s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        s.connect((HOST, PORT))
        pulses=len(s_enc)//4
        pad=32-((pulses*9)%32)
        #pad_list=pad*[0x0]
        CommandHeader=[MAGIC_TOKEN, 1, COMMAND, 0x0, pulses*9+32+pad, 0]
        StreamHighLevelHeader=[n_runs, pulses, final[2], final[3], final[1], 0x0, 0x0, 0x0, 0]
        s.sendall(struct.pack('<4I2Q', *CommandHeader))
        s.sendall(struct.pack('<qQ2h4BQ', *StreamHighLevelHeader))
        fmt = '<' + len(s_enc)//4*'IBhh'
        s.sendall(struct.pack(fmt, *s_enc))
        s.sendall(b'\0' * pad)
        data = s.recv(32)
        s.close()

        return_value=struct.unpack('<4I2Q', data)
        return return_value
            
    def isStreaming(self):
        void = VoidMessage()
        return self.stub.isStreaming(void, timeout=PulseStreamer.TIMEOUT).value

    def hasSequence(self):
        void = VoidMessage()
        return self.stub.hasSequence(void, timeout=PulseStreamer.TIMEOUT).value

    def hasFinished(self):
        void = VoidMessage()
        return self.stub.hasFinished(void, timeout=PulseStreamer.TIMEOUT).value

    def startNow(self):
        void = VoidMessage()
        return self.stub.startNow(void, timeout=PulseStreamer.TIMEOUT).value

    def selectClock(self, source):
        if not isinstance(source, ClockSource):
            raise TypeError("source must be an instance of ClockSource Enum")
        else:
            c = ClockMessage()
            c.clock_source = source.value
            return self.stub.selectClock(c, timeout=PulseStreamer.TIMEOUT).value
    
    def getClock(self):
        void = VoidMessage()
        return ClockSource(self.stub.getClock(void, timeout=PulseStreamer.TIMEOUT).value)

    def getFirmwareVersion(self):
        void = VoidMessage()
        return self.stub.getFirmwareVersion(void, timeout=PulseStreamer.TIMEOUT).string_value

    def flash_lic(self):
        void = VoidMessage()
        return self.stub.flash_lic(void, timeout=PulseStreamer.TIMEOUT).string_value

    def getSerial(self):
        if (self.__version_1_1_or_higher):
            void = VoidMessage()
            return self.stub.getSerial(void, timeout=PulseStreamer.TIMEOUT).string_value
        else:
            s=GetSerialMessage()
            s.serial = 1
            return self.stub.getSerial(s, timeout=PulseStreamer.TIMEOUT).string_value

    def getFPGAID(self):
        if (self.__version_1_1_or_higher):
            void = VoidMessage()
            return self.stub.getFPGAID(void, timeout=PulseStreamer.TIMEOUT).string_value
        else:
            s=GetSerialMessage()
            s.serial = 0
            return self.stub.getSerial(s, timeout=PulseStreamer.TIMEOUT).string_value

    def setClkParam(self, value=0):
        assert value in range(58)
        v= DrpClkSetMessage()
        v.value = value
        return self.stub.setClkParam(v, timeout=PulseStreamer.TIMEOUT).value

    def setTrigger(self, start, rearm=TriggerRearm.AUTO):
        if not isinstance(start, TriggerStart):
            raise TypeError("start must be an instance of TriggerStart Enum")
        else:
            if not isinstance(rearm, TriggerRearm):
                raise TypeError("rearm must be an instance of TriggerRearm Enum")
            else:
                t = TriggerMessage()
                t.start = start.value
                t.mode = rearm.value
                return self.stub.setTrigger(t, timeout=PulseStreamer.TIMEOUT).value

    def getTriggerStart(self):
        void = VoidMessage()
        return TriggerStart(self.stub.getTriggerStart(void, timeout=PulseStreamer.TIMEOUT).value)

    def getTriggerRearm(self):
        void = VoidMessage()
        return TriggerRearm(self.stub.getTriggerRearm(void, timeout=PulseStreamer.TIMEOUT).value)

    def rearm(self):
        void = VoidMessage()
        return self.stub.rearm(void, timeout=PulseStreamer.TIMEOUT).value

    def setAnalogCalibration(self, dc_offset_a0=0, dc_offset_a1=0, slope_a0=1, slope_a1=1):
        a=AnalogCalibrationData()
        a.dc_offset_a0=dc_offset_a0
        a.dc_offset_a1=dc_offset_a1
        a.slope_a0=slope_a0
        a.slope_a1=slope_a1
        return self.stub.setAnalogCalibration(a,timeout=PulseStreamer.TIMEOUT).value
    
    def getAnalogCalibration(self):
        void = VoidMessage()
        response = self.stub.getAnalogCalibration(void, timeout=PulseStreamer.TIMEOUT)
        key_list=['dc_offset_a0', 'dc_offset_a1', 'slope_a0', 'slope_a1']
        return dict( zip(key_list, [response.dc_offset_a0, response.dc_offset_a1, response.slope_a0, response.slope_a1] ))
        
    
    def setNetworkConf(self, ip, netmask, gateway):
        n=SetNetworkMessage()
        n.ip=ip
        n.netmask=netmask
        n.gateway=gateway
        return self.stub.setNetworkConf(n, timeout=PulseStreamer.TIMEOUT).string_value

    def getNetworkConf(self):
        void = VoidMessage()
        return self.stub.getNetworkConf(void, timeout=PulseStreamer.TIMEOUT).string_value

    def testNetworkConf(self):
        void = VoidMessage()
        return self.stub.testNetworkConf(void, timeout=PulseStreamer.TIMEOUT).value

    def enableStaticIP(self, permanent=False):
        assert permanent in [True, False]
        e=EnableStaticIPMessage()
        e.permanent=permanent
        return self.stub.enableStaticIP(e, timeout=PulseStreamer.TIMEOUT).string_value

    def setHostname(self, hostname):
        h=HostnameMessage()
        h.hostname=hostname
        return self.stub.setHostname(h, timeout=PulseStreamer.TIMEOUT).value

    def getHostname(self):
        void = VoidMessage()
        return self.stub.getHostname(void, timeout=PulseStreamer.TIMEOUT).string_value

    def getTemperature(self):
        void = VoidMessage()
        return self.stub.getTemperature(void, timeout=PulseStreamer.TIMEOUT).string_value

    def getUnderflow(self):
        void = VoidMessage()
        return self.stub.getUnderflow(void, timeout=PulseStreamer.TIMEOUT).value

    def getDebugRegister(self):
        void = VoidMessage()
        return self.stub.getDebugRegister(void, timeout=PulseStreamer.PulseStreamer.TIMEOUT).value

    def chans_to_mask(self, chans):
        mask = 0
        for chan in chans:
            assert chan in range(8),"Pulse Streamer 8/2 supports "\
            "up to eight digital channels"
            mask |= 1<<chan
        return mask

    def convert_sequence_step(self, sequence_step, binary=False):
        t, chans, a0, a1 = sequence_step
        assert (abs(a0)<=1 and abs(a1)<=1), "Pulse Streamer 8/2 supports "\
                "analog voltage range of +/-1V" #check hardware
        assert t>=0
        if binary:
            return (t, self.chans_to_mask(chans), int(round(0x7fff*a0)), int(round(0x7fff*a1)))
        else:
            return PulseMessage(ticks=t, digi=self.chans_to_mask(chans), ao0=int(round(0x7fff*a0)), ao1=int(round(0x7fff*a1)))

    def enc_binary(self, seq):
        """
        Convert a human readable python sequence to a base64 encoded string and split sequence steps with duration exceeding limit of unsigned int
        """
        s = b''
        convert_list = []
        if type(seq[0][1])== list:
            for sequence_step in seq:
                if sequence_step[0] > 0xffffffff:
                    mod=sequence_step[0]%0xffffffff
                    count=sequence_step[0]//0xffffffff
                    for i in range(count):
                        sequence_step = (0xffffffff, sequence_step[1], sequence_step[2], sequence_step[3])
                        convert_list.extend(self.convert_sequence_step(sequence_step, binary=True))
                    else:
                        sequence_step = (mod, sequence_step[1], sequence_step[2], sequence_step[3])
                convert_list.extend(self.convert_sequence_step(sequence_step, binary=True))    
        else:
            for sequence_step in seq:
                if sequence_step[0] > 0xffffffff:
                    mod=sequence_step[0]%0xffffffff
                    count=sequence_step[0]//0xffffffff
                    for i in range(count):
                        sequence_step = (0xffffffff, sequence_step[1], sequence_step[2], sequence_step[3])
                        convert_list.extend(sequence_step)
                    else:
                        sequence_step = (mod, sequence_step[1], sequence_step[2], sequence_step[3])
                convert_list.extend(sequence_step)

        assert len(convert_list)//4<=2e6, "The resulting sequence length exceeds the limit of two million sequnence steps"

        return convert_list

"""---------Test-Code-------------------------------"""

if __name__ == '__main__':
    pulser = PulseStreamer(ip_hostname='pulsestreamer')

    print("Serial number:", pulser.getSerial())
    print("Firmware Version:", pulser.getFirmwareVersion())

    