'''
FastFD - GPU Accelerated Finite Differences Simulation Library
==============================================================

Copyright (c) 2020-2021 Stefan Meili
MIT License
'''



import math
from functools import lru_cache


from . import sparse_lib
from . import DiscretizedScalar


def gen_coefficients(derivative, accuracy):
    # Generate stencil offsets
    n_forward = derivative + accuracy # Number of terms required for forward difference
    
    offsets = [sparse_lib.np.arange(-i, n_forward-i, dtype = int) for i in range(n_forward)] # Simple offsets
    
    c_accuracy = max(accuracy // 2 * 2, 2) #round accuracy down to the nearest even number
    n_central = max((derivative - 1) // 2 * 2 + 1 + c_accuracy, 3) #round derivative down to the nearest odd, add accuracy
    
    if n_central < n_forward:
        offsets = offsets[:n_central//2] \
            + [sparse_lib.np.arange(n_central, dtype = int) - n_central//2] \
            + offsets[-n_central//2 + 1:]
            
    # Solve for Finite Difference coefficients per:
    # https://en.wikipedia.org/wiki/Finite_difference_coefficient#Arbitrary_stencil_points
    coefficients = []
    for off in offsets:
        M = sparse_lib.np.power(off, sparse_lib.np.arange(len(off), dtype = int).reshape(-1,1))
        C = sparse_lib.np.zeros(len(off))
        C[derivative] = math.factorial(derivative)
        coefficients.append(sparse_lib.np.linalg.solve(M, C))
        
    return coefficients, offsets, n_forward
    

#@lru_cache()
def gen_coeff_matrix(shape, dim_idx, derivative, accuracy):
    '''
    Called by Scalar.d() to build a coefficient matrix for a DiscretizedScalar.
    
    Inputs:
        shape = tuple(x, y, z, ...)
            Size of final coefficient matrix
        
        dim_idx = int
            Index in shape that the scalar is discretized over.
            
        derivative = int
            Derivative being approximated by the coefficients being produced
            
        accuracy = int
            Accuracy of the derivative approximation
            
    This method first generates index offsets from the coefficient matrix diagonal based on the inputs. Coefficients are
    then derived and assigned to a sparse 2D matrix that discretizes the scalar along dim_idx. Finally, the matrix is
    replicated to match the final output scalar shape.
    
    '''
    
    
    if dim_idx < 0:
        dim_idx = len(shape) + dim_idx
    if dim_idx < 0 or dim_idx >= len(shape):
        raise IndexError(f'dim_idx index out of range of shape')
        
    n_rows = shape[dim_idx]
    
    coefficients, offsets, n_forward = gen_coefficients(derivative, accuracy)
    
    if n_rows < n_forward:
        raise ValueError('Grid resolution is less than required')
    
        
    # Build a 1D finite difference matrix by expanding coefficients
    if n_rows > len(coefficients):
        central_idx = len(coefficients)//2 # index of central FD approximation
        
        forward = sparse_lib.sparse.csr_matrix(sparse_lib.np.array(coefficients[:central_idx]), shape = (central_idx, n_rows))
        
        central = sparse_lib.sparse.diags(
            coefficients[central_idx],
            offsets[central_idx] + central_idx,
            shape=(n_rows - 2*central_idx , n_rows),
            format='csr')
        
        backwards = sparse_lib.sparse.hstack([
            sparse_lib.sparse.csr_matrix((central_idx, n_rows - n_forward)),
            sparse_lib.sparse.csr_matrix(sparse_lib.np.array(coefficients[-central_idx:]))
        ])
        
        fd_matrix = sparse_lib.sparse.vstack([forward, central, backwards])
        
    else:
        fd_matrix = sparse_lib.sparse.csr_matrix(sparse_lib.np.stack(coefficients))
    
    # Expand 1D fd_matrix dimensions to match shape, replicating coefficients using identity and kron
    A = sparse_lib.sparse.identity(math.prod(shape[:dim_idx]))
    B = sparse_lib.sparse.identity(math.prod(shape[dim_idx + 1:]))
    
    return sparse_lib.sparse.kron(sparse_lib.sparse.kron(A, fd_matrix), B, format = 'csr')
    
    
class Scalar:
    '''
    Generates appropriately indexed coefficient matrices to build a finite difference simulation with.
    
    Inputs:
        name = 'string'
            Names must be unique to each FDModel, as they are used to index blocks of variables in the model coefficient
            matrix.
        
        axes = [list of Axis objects]
            Specifies the volume over which the Scalar will be discretized. The order of axes in the list defines the
            block of indices in the final coefficient matrix and constraint vector.
            
        accuracy = int (optional)
            Specifies the default approximation accuracy of derivatives generated by Scalar.d()
    '''
    
    def __init__(self, name, axes, accuracy = 2):
        self.name = name
        self.axes = {}
        for a in axes:
            if a.name in self.axes:
                raise ValueError(f'Axis {a.name} is duplicated in Scalar {name}')
            self.axes.update({a.name:a})
        
        self.accuracy = accuracy
        
        self.shape = tuple(a.num_points for a in self.axes.values())
        self.size = math.prod(self.shape)
        
        self.model = None
        
        self.identity = sparse_lib.sparse.identity(self.size, format='csr')
        
        self.coords = sparse_lib.np.meshgrid(*tuple([a.coords for a in self.axes.values()]), indexing = 'ij')
        self.coords = {axis:coord for axis, coord in zip(self.axes.keys(), self.coords)}
        
        self.timestep = None
    
    def _check_model(self):
        if self.model is None:
            raise Exception(f'Scaler {self.name} has not been assigned to a model')
        
    
    @property
    def i(self):
        '''
        Return an identity matrix that can be indexed within a model scope. Useful for masking and setting boundary
        conditions.
        '''
        self._check_model()
        return DiscretizedScalar(self.identity, self)
    
    
    
    def d(self, axis_name, derivative = 1, accuracy = None):
        '''
        Returns a DiscretizedScalar coefficient matrix that approximates the derivative of the Scalar along the selected
        Axis.
        
        Inputs:
            axis_name = 'string'
                Key used to specify the axis the Scalar is discretized over.
                
            derivative = int
                Derivative being approximated.
                
            accuracy = int
                Order of approximation accuracy.
        '''
        
        #TODO Adjust coefficients for variable grid step size?
        #TODO Adjust coefficients to allow polar or spherical coordinates?
        
        self._check_model()
            
        if accuracy is None:
            accuracy = self.accuracy
            
        if accuracy < 1:
            raise ValueError('Derivative approximation accuracy must be at least 1')
            
        if derivative < 1:
            raise ValueError("Derivative must be at least 1. Were you looking for 'Scalar.i' instead?") 
        
        if axis_name in self.axes.keys(): 
            dim_idx = list(self.axes.keys()).index(axis_name)
        else:
            raise ValueError(f'Derivative axis {axis_name} is not registered: {list(self.axes.keys())}')
                
        coeff_matrix = gen_coeff_matrix(self.shape, dim_idx, derivative, accuracy).copy()
        coeff_matrix /= self.axes[axis_name].delta**derivative
        
        return DiscretizedScalar(coeff_matrix, self)
        
        
    def dt(self, dt_type, derivative = 1, accuracy = None):
        '''
        Returns a DiscretizedScalar coefficient matrix that approximates the derivative of the Scalar along the selected
        Axis.
        
        Inputs:
            dt_type = 'string'
                Must be either 'lhs' or 'rhs' depending on if it's used in the coefficient matrix or constraint
                vector.
                
            derivative = int
                Derivative being approximated.
                
            accuracy = int
                Order of approximation accuracy.
        '''
        
        #TODO Adjust coefficients for variable grid step size?
        #TODO Adjust coefficients to allow polar or spherical coordinates?
        
        self._check_model()
            
        if accuracy is None:
            accuracy = self.accuracy
            
        if accuracy < 1:
            raise ValueError('Derivative approximation accuracy must be at least 1')
            
        if derivative < 1:
            raise ValueError("Derivative must be at least 1. Were you looking for 'Scalar.i' instead?") 
        
        if self.timestep is None:
            raise ValueError('Timestep is not set. Create model with FDModel(scalars = [?], timestep = ?)')
        
        coefficients, *_ = gen_coefficients(derivative, accuracy)
        coefficients = coefficients[-1] / self.timestep**derivative
        
        if dt_type == 'lhs':
            return self.i * float(coefficients[-1])
        elif dt_type == 'rhs':
            return sparse_lib.sparse.kron(sparse_lib.sparse.identity(self.size), sparse_lib.sparse.csr_matrix(coefficients[:-1]), format = 'csr')
        else:
            raise ValueError(f"dt_type must be either 'scalar' or 'constraint' for {self.name}. Got {dt_type}.")
        
