"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const glue_generated_1 = require("./glue.generated");
/**
 * A Glue database.
 */
class Database extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.databaseName,
        });
        let databaseInput = {
            name: props.databaseName,
        };
        if (props.locationUri !== undefined) {
            validateLocationUri(props.locationUri);
            this.locationUri = props.locationUri;
            databaseInput = {
                locationUri: this.locationUri,
                ...databaseInput,
            };
        }
        this.catalogId = core_1.Stack.of(this).account;
        const resource = new glue_generated_1.CfnDatabase(this, 'Resource', {
            catalogId: this.catalogId,
            databaseInput,
        });
        // see https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html#data-catalog-resource-arns
        this.databaseName = this.getResourceNameAttribute(resource.ref);
        this.databaseArn = this.stack.formatArn({
            service: 'glue',
            resource: 'database',
            resourceName: this.databaseName,
        });
        // catalogId is implicitly the accountId, which is why we don't pass the catalogId here
        this.catalogArn = core_1.Stack.of(this).formatArn({
            service: 'glue',
            resource: 'catalog'
        });
    }
    static fromDatabaseArn(scope, id, databaseArn) {
        const stack = core_1.Stack.of(scope);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.databaseArn = databaseArn;
                this.databaseName = stack.parseArn(databaseArn).resourceName;
                this.catalogArn = stack.formatArn({ service: 'glue', resource: 'catalog' });
                this.catalogId = stack.account;
            }
        }
        return new Import(scope, id);
    }
}
exports.Database = Database;
function validateLocationUri(locationUri) {
    if (locationUri.length < 1 || locationUri.length > 1024) {
        throw new Error(`locationUri length must be (inclusively) between 1 and 1024, but was ${locationUri.length}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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