"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Absolute class name of the Hadoop `InputFormat` to use when reading table files.
 */
class InputFormat {
    constructor(className) {
        this.className = className;
    }
}
exports.InputFormat = InputFormat;
/**
 * InputFormat for Avro files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/avro/AvroContainerInputFormat.html
 */
InputFormat.AVRO = new InputFormat('org.apache.hadoop.hive.ql.io.avro.AvroContainerInputFormat');
/**
 * InputFormat for Cloudtrail Logs.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
 */
InputFormat.CLOUDTRAIL = new InputFormat('com.amazon.emr.cloudtrail.CloudTrailInputFormat');
/**
 * InputFormat for Orc files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcInputFormat.html
 */
InputFormat.ORC = new InputFormat('org.apache.hadoop.hive.ql.io.orc.OrcInputFormat');
/**
 * InputFormat for Parquet files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/MapredParquetInputFormat.html
 */
InputFormat.PARQUET = new InputFormat('org.apache.hadoop.hive.ql.io.parquet.MapredParquetInputFormat');
/**
 * An InputFormat for plain text files. Files are broken into lines. Either linefeed or
 * carriage-return are used to signal end of line. Keys are the position in the file, and
 * values are the line of text.
 * JSON & CSV files are examples of this InputFormat
 *
 * @see https://hadoop.apache.org/docs/stable/api/org/apache/hadoop/mapred/TextInputFormat.html
 */
InputFormat.TEXT = new InputFormat('org.apache.hadoop.mapred.TextInputFormat');
/**
 * Absolute class name of the Hadoop `OutputFormat` to use when writing table files.
 */
class OutputFormat {
    constructor(className) {
        this.className = className;
    }
}
exports.OutputFormat = OutputFormat;
/**
 * Writes text data with a null key (value only).
 *
 * @see https://hive.apache.org/javadocs/r2.2.0/api/org/apache/hadoop/hive/ql/io/HiveIgnoreKeyTextOutputFormat.html
 */
OutputFormat.HIVE_IGNORE_KEY_TEXT = new OutputFormat('org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat');
/**
 * OutputFormat for Avro files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/avro/AvroContainerOutputFormat.html
 */
OutputFormat.AVRO = new InputFormat('org.apache.hadoop.hive.ql.io.avro.AvroContainerOutputFormat');
/**
 * OutputFormat for Orc files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcOutputFormat.html
 */
OutputFormat.ORC = new InputFormat('org.apache.hadoop.hive.ql.io.orc.OrcOutputFormat');
/**
 * OutputFormat for Parquet files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/MapredParquetOutputFormat.html
 */
OutputFormat.PARQUET = new OutputFormat('org.apache.hadoop.hive.ql.io.parquet.MapredParquetOutputFormat');
/**
 * Serialization library to use when serializing/deserializing (SerDe) table records.
 *
 * @see https://cwiki.apache.org/confluence/display/Hive/SerDe
 */
class SerializationLibrary {
    constructor(className) {
        this.className = className;
    }
}
exports.SerializationLibrary = SerializationLibrary;
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/avro/AvroSerDe.html
 */
SerializationLibrary.AVRO = new SerializationLibrary('org.apache.hadoop.hive.serde2.avro.AvroSerDe');
/**
 * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
 */
SerializationLibrary.CLOUDTRAIL = new SerializationLibrary('com.amazon.emr.hive.serde.CloudTrailSerde');
/**
 * @see https://docs.aws.amazon.com/athena/latest/ug/grok.html
 */
SerializationLibrary.GROK = new SerializationLibrary('com.amazonaws.glue.serde.GrokSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hive/hcatalog/data/JsonSerDe.html
 */
SerializationLibrary.HIVE_JSON = new SerializationLibrary('org.apache.hive.hcatalog.data.JsonSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/lazy/LazySimpleSerDe.html
 */
SerializationLibrary.LAZY_SIMPLE = new SerializationLibrary('org.apache.hadoop.hive.serde2.lazy.LazySimpleSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/OpenCSVSerde.html
 */
SerializationLibrary.OPEN_CSV = new SerializationLibrary('org.apache.hadoop.hive.serde2.OpenCSVSerde');
/**
 * @see https://github.com/rcongiu/Hive-JSON-Serde
 */
SerializationLibrary.OPENX_JSON = new SerializationLibrary('org.openx.data.jsonserde.JsonSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcSerde.html
 */
SerializationLibrary.ORC = new SerializationLibrary('org.apache.hadoop.hive.ql.io.orc.OrcSerde');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/serde/ParquetHiveSerDe.html
 */
SerializationLibrary.PARQUET = new SerializationLibrary('org.apache.hadoop.hive.ql.io.parquet.serde.ParquetHiveSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/RegexSerDe.html
 */
SerializationLibrary.REGEXP = new SerializationLibrary('org.apache.hadoop.hive.serde2.RegexSerDe');
var DataFormat;
(function (DataFormat) {
    /**
     * DataFormat for Apache Web Server Logs. Also works for CloudFront logs
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/apache.html
     */
    DataFormat.ApacheLogs = {
        inputFormat: InputFormat.TEXT,
        outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
        serializationLibrary: SerializationLibrary.REGEXP
    };
    /**
     * DataFormat for Apache Avro
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/avro.html
     */
    DataFormat.Avro = {
        inputFormat: InputFormat.AVRO,
        outputFormat: OutputFormat.AVRO,
        serializationLibrary: SerializationLibrary.AVRO
    };
    /**
     * DataFormat for CloudTrail logs stored on S3
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
     */
    DataFormat.CloudTrailLogs = {
        inputFormat: InputFormat.CLOUDTRAIL,
        outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
        serializationLibrary: SerializationLibrary.CLOUDTRAIL
    };
    /**
     * DataFormat for CSV Files
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/csv.html
     */
    DataFormat.CSV = {
        inputFormat: InputFormat.TEXT,
        outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
        serializationLibrary: SerializationLibrary.OPEN_CSV
    };
    /**
     * Stored as plain text files in JSON format.
     * Uses OpenX Json SerDe for serialization and deseralization.
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/json.html
     */
    DataFormat.Json = {
        inputFormat: InputFormat.TEXT,
        outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
        serializationLibrary: SerializationLibrary.OPENX_JSON
    };
    /**
     * DataFormat for Logstash Logs, using the GROK SerDe
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/grok.html
     */
    DataFormat.Logstash = {
        inputFormat: InputFormat.TEXT,
        outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
        serializationLibrary: SerializationLibrary.GROK
    };
    /**
     * DataFormat for Apache ORC (Optimized Row Columnar)
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/orc.html
     */
    DataFormat.Orc = {
        inputFormat: InputFormat.ORC,
        outputFormat: OutputFormat.ORC,
        serializationLibrary: SerializationLibrary.ORC
    };
    /**
     * DataFormat for Apache Parquet
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/parquet.html
     */
    DataFormat.Parquet = {
        inputFormat: InputFormat.PARQUET,
        outputFormat: OutputFormat.PARQUET,
        serializationLibrary: SerializationLibrary.PARQUET
    };
    /**
     * DataFormat for TSV (Tab-Separated Values)
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/lazy-simple-serde.html
     */
    DataFormat.TSV = {
        inputFormat: InputFormat.TEXT,
        outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
        serializationLibrary: SerializationLibrary.LAZY_SIMPLE
    };
})(DataFormat = exports.DataFormat || (exports.DataFormat = {}));
//# sourceMappingURL=data:application/json;base64,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