/**
 * Absolute class name of the Hadoop `InputFormat` to use when reading table files.
 */
export declare class InputFormat {
    readonly className: string;
    /**
     * InputFormat for Avro files.
     *
     * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/avro/AvroContainerInputFormat.html
     */
    static readonly AVRO: InputFormat;
    /**
     * InputFormat for Cloudtrail Logs.
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
     */
    static readonly CLOUDTRAIL: InputFormat;
    /**
     * InputFormat for Orc files.
     *
     * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcInputFormat.html
     */
    static readonly ORC: InputFormat;
    /**
     * InputFormat for Parquet files.
     *
     * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/MapredParquetInputFormat.html
     */
    static readonly PARQUET: InputFormat;
    /**
     * An InputFormat for plain text files. Files are broken into lines. Either linefeed or
     * carriage-return are used to signal end of line. Keys are the position in the file, and
     * values are the line of text.
     * JSON & CSV files are examples of this InputFormat
     *
     * @see https://hadoop.apache.org/docs/stable/api/org/apache/hadoop/mapred/TextInputFormat.html
     */
    static readonly TEXT: InputFormat;
    constructor(className: string);
}
/**
 * Absolute class name of the Hadoop `OutputFormat` to use when writing table files.
 */
export declare class OutputFormat {
    readonly className: string;
    /**
     * Writes text data with a null key (value only).
     *
     * @see https://hive.apache.org/javadocs/r2.2.0/api/org/apache/hadoop/hive/ql/io/HiveIgnoreKeyTextOutputFormat.html
     */
    static readonly HIVE_IGNORE_KEY_TEXT: OutputFormat;
    /**
     * OutputFormat for Avro files.
     *
     * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/avro/AvroContainerOutputFormat.html
     */
    static readonly AVRO: InputFormat;
    /**
     * OutputFormat for Orc files.
     *
     * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcOutputFormat.html
     */
    static readonly ORC: InputFormat;
    /**
     * OutputFormat for Parquet files.
     *
     * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/MapredParquetOutputFormat.html
     */
    static readonly PARQUET: OutputFormat;
    constructor(className: string);
}
/**
 * Serialization library to use when serializing/deserializing (SerDe) table records.
 *
 * @see https://cwiki.apache.org/confluence/display/Hive/SerDe
 */
export declare class SerializationLibrary {
    readonly className: string;
    /**
     * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/avro/AvroSerDe.html
     */
    static readonly AVRO: SerializationLibrary;
    /**
     * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
     */
    static readonly CLOUDTRAIL: SerializationLibrary;
    /**
     * @see https://docs.aws.amazon.com/athena/latest/ug/grok.html
     */
    static readonly GROK: SerializationLibrary;
    /**
     * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hive/hcatalog/data/JsonSerDe.html
     */
    static readonly HIVE_JSON: SerializationLibrary;
    /**
     * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/lazy/LazySimpleSerDe.html
     */
    static readonly LAZY_SIMPLE: SerializationLibrary;
    /**
     * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/OpenCSVSerde.html
     */
    static readonly OPEN_CSV: SerializationLibrary;
    /**
     * @see https://github.com/rcongiu/Hive-JSON-Serde
     */
    static readonly OPENX_JSON: SerializationLibrary;
    /**
     * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcSerde.html
     */
    static readonly ORC: SerializationLibrary;
    /**
     * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/serde/ParquetHiveSerDe.html
     */
    static readonly PARQUET: SerializationLibrary;
    /**
     * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/RegexSerDe.html
     */
    static readonly REGEXP: SerializationLibrary;
    constructor(className: string);
}
/**
 * Defines the input/output formats and ser/de for a single DataFormat.
 */
export interface DataFormat {
    /**
     * `InputFormat` for this data format.
     */
    readonly inputFormat: InputFormat;
    /**
     * `OutputFormat` for this data format.
     */
    readonly outputFormat: OutputFormat;
    /**
     * Serialization library for this data format.
     */
    readonly serializationLibrary: SerializationLibrary;
}
export declare namespace DataFormat {
    /**
     * DataFormat for Apache Web Server Logs. Also works for CloudFront logs
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/apache.html
     */
    const ApacheLogs: DataFormat;
    /**
     * DataFormat for Apache Avro
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/avro.html
     */
    const Avro: DataFormat;
    /**
     * DataFormat for CloudTrail logs stored on S3
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
     */
    const CloudTrailLogs: DataFormat;
    /**
     * DataFormat for CSV Files
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/csv.html
     */
    const CSV: DataFormat;
    /**
     * Stored as plain text files in JSON format.
     * Uses OpenX Json SerDe for serialization and deseralization.
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/json.html
     */
    const Json: DataFormat;
    /**
     * DataFormat for Logstash Logs, using the GROK SerDe
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/grok.html
     */
    const Logstash: DataFormat;
    /**
     * DataFormat for Apache ORC (Optimized Row Columnar)
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/orc.html
     */
    const Orc: DataFormat;
    /**
     * DataFormat for Apache Parquet
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/parquet.html
     */
    const Parquet: DataFormat;
    /**
     * DataFormat for TSV (Tab-Separated Values)
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/lazy-simple-serde.html
     */
    const TSV: DataFormat;
}
