#!/usr/bin/env python

# compare the skyrmion configuration generated by this code (asd)
# and Spirit.configurations
# the lattice, sites, and spin configurations are carefully compared
# Shunhong Zhang <szhang2@ustc.edu.cn>
# July 10, 2021


import matplotlib.pyplot as plt
import numpy as np
from asd.core import spin_configurations
from asd.utility.spin_visualize_tools import *
from asd.core.geometry import *
import os


def asd_gen_skyrmion(nx,ny,radius,return_skyr_idx=False,repeat_x=3,repeat_y=3,center_pos=[0,0]):
    latt,sites = build_latt('honeycomb',nx,ny,1,return_neigh=False)
    nat = sites.shape[2]
    sp_lat = np.zeros((nx,ny,nat,3),float)
    kwargs=  dict(theta_cycle=1,
    display=False,
    return_skyr_idx=True,
    center_pos=center_pos,
    use_pbc=True,
    )
    sp_lat,skyr_idx = spin_configurations.init_spin_latt_skyrmion(sp_lat,latt,sites,radius,**kwargs)

    sites_cart = np.dot(sites,latt)

    if repeat_x>1 and repeat_y>1:
        sites = get_repeated_sites(sites,repeat_x,repeat_y)
        sp_lat = get_repeated_conf(sp_lat,repeat_x,repeat_y)
        sites_cart = np.dot(sites,latt)
        slatt = np.dot([[nx,0],[0,ny]],latt)
        #fig,ax,scat,qv,tl = plot_spin_2d(sites_cart,sp_lat,show=True,scatter_size=30,latt=latt,superlatt=slatt)
    
    if return_skyr_idx: return sp_lat,skyr_idx,sites_cart,latt
    else: return sp_lat,None,sites_cart,latt


def gen_confs_for_target_Mz(nx,ny,target_Mz_list,r_step=0.01,outdir='ovfs',ani=True,center_pos=[0,0]):
    radius_list = np.arange(1,nx/2.+r_step,r_step)
    mz_list=[]
    for radius in radius_list:
        sp_lat,skyr_idx,sites_cart,latt = asd_gen_skyrmion(nx,ny,radius,repeat_x=1,repeat_y=1,center_pos=center_pos)
        mz = np.average(sp_lat[...,2])
        mz_list.append(mz)
        #print ('R = {:9.5f}, Mz = {:9.5f}'.format(radius,mz))
    mz_list = np.array(mz_list)
    print ('')
    target_idx=[]
    for target_Mz in target_Mz_list:
        idx = np.argmin(abs(mz_list-target_Mz))
        target_idx.append(idx)
        fmt = 'target_Mz = {:9.5f}, optimal radius = {:9.5f}, mz = {:9.5f}'
        print (fmt.format(target_Mz,radius_list[idx],mz_list[idx]))
    print ('')

    target_idx=np.array(target_idx)

    fig,ax=plt.subplots(1,1)
    ax.plot(radius_list,mz_list,'r-')
    for ii in target_idx:
        ax.axhline(mz_list[ii],ls='--',c='m',alpha=0.5,zorder=-1)
        ax.axvline(radius_list[ii],ls='--',alpha=0.5,zorder=-1)
    ax.set_xlabel('R')
    ax.set_ylabel('$M_z$')
    plt.show()

    anim_kwargs = dict(interval=1e3,
    scatter_size=30,
    quiver_kws = dict(scale=1,units='x',pivot='mid'),
    superlatt = np.dot([[nx,0],[0,ny]],latt),
    colorbar_axes_position=[0.1,0.2,0.02,0.25],
    colorbar_orientation='vertical',
    colorbar_shrink=0.3,
    )

    if os.path.isdir(outdir): os.system('rm -r {}'.format(outdir))
    os.mkdir(outdir)
    all_confs = []
    for i,ii in enumerate(target_idx):
        sp_lat,skyr_idx,sites_cart,latt = asd_gen_skyrmion(nx,ny,radius_list[ii],repeat_x=1,repeat_y=1,center_pos=center_pos)
        all_confs.append(sp_lat)
        add_desc = [
        'Radius    = {:8.5f}'.format(radius_list[ii]),
        'Target Mz = {:8.5f}'.format(target_Mz_list[i]),
        'Real   Mz = {:8.5f}'.format(mz_list[ii])]
        params = gen_params_for_ovf(nx,ny,1,additional_desc=add_desc)
        spins = np.swapaxes(sp_lat,0,1).reshape(-1,3)
        fil = '{}/spin_confs_{}.ovf'.format(outdir,i)
        write_ovf(params,spins,filename=fil)
    if ani:
        all_confs = np.array(all_confs)[::-1]
        sites = np.dot(sites_cart,np.linalg.inv(latt))
        sites = get_repeated_sites(sites,2,2)
        sites_cart = np.dot(sites,latt)
        all_confs = np.tile(all_confs,(1,2,2,1,1))
        anim_kwargs.update(titles = ['n = {}, Mz = {:6.3f}'.format(nx,mz) for mz in mz_list[::-1]])
        make_ani(sites_cart,all_confs,**anim_kwargs)


target_Mz_list = np.append(0.46,np.arange(0.5,0.95,0.05))

for n in range(9,12):
    if n%2==0: center_pos = [0.5,0.0]
    else: center_pos = [0.,0.5]
    gen_confs_for_target_Mz(n,n,target_Mz_list,outdir='n_{}'.format(n),center_pos=center_pos)
