#!/usr/bin/env python

# compare the skyrmion configuration generated by this code (asd)
# and Spirit.configurations
# the lattice, sites, and spin configurations are carefully compared
# Shunhong Zhang <szhang2@ustc.edu.cn>
# July 10, 2021


import matplotlib.pyplot as plt
import numpy as np
from spirit import state,configuration,geometry,io,system
from asd.core import spin_configurations
from asd.utility.spin_visualize_tools import *
from asd.core.geometry import *


def asd_gen_skyrmion(nx,ny,radius,return_skyr_idx=False):
    latt,sites = build_latt('honeycomb',nx,ny,1,return_neigh=False)
    nat = sites.shape[2]
    sp_lat = np.zeros((nx,ny,nat,3),float)
    kwargs=  dict(theta_cycle=-1,
    display=False,
    return_skyr_idx=True)
    sp_lat,skyr_idx = spin_configurations.init_spin_latt_skyrmion(sp_lat,latt,sites,radius,**kwargs)
    sites_cart = np.dot(sites,latt)
    plot_spin_2d(sites_cart,sp_lat,scatter_size=20,quiver_kws=quiver_kws,show=False,title='Config by asd')
    if return_skyr_idx: return sp_lat,skyr_idx,sites_cart
    else: return sp_lat,None,sites_cart


def spirit_gen_skyrmion(nx,ny,radius,display=False):
    with state.State('input.cfg',quiet=True) as p_state:
        geometry.set_n_cells(p_state,[nx,ny,1],idx_image=0)
        configuration.plus_z(p_state,idx_image=0)
        pos = geometry.get_positions(p_state,idx_image=0)
        latt = geometry.get_bravais_vectors(p_state)
        latt = np.array([vec for vec in latt]).T[:2,:2]

        mm = np.min(pos,axis=0)
        nn = np.max(pos,axis=0)
        center = (mm+nn)/2

        cc = [0,np.sqrt(3)/12]
        configuration.skyrmion(p_state,1.5,idx_image=0,pos=cc)
        system.update_data(p_state,idx_image=0)
        io.image_write(p_state,'skyr_spirit.ovf',idx_image=0) 

    spins = parse_ovf('skyr_spirit.ovf')[1]
    if display:
        fig,ax,scat,qv,tl = plot_spin_2d(pos,spins,scatter_size=40,quiver_kws=quiver_kws,title='Config by Spirit')
        ax.scatter(center[0],center[1],c='brown',s=60,marker='*')
        plt.show()
    nat = spins.shape[0]//(nx*ny)
    sp_lat = np.swapaxes(spins.reshape(ny,nx,nat,3),0,1)
    skyr_idx = np.array(np.where(sp_lat[...,2]!=1.)).T
    sites_cart = np.swapaxes(pos.reshape(ny,nx,nat,3),0,1)[...,:2]
    return sp_lat,skyr_idx,sites_cart


def compare_sites(sites_cart_1,sites_cart_2):
    fig,ax=plt.subplots()
    ax.scatter(sites_cart_1[...,0],sites_cart_1[...,1],c='g',label='asd',s=5)
    ax.scatter(sites_cart_2[...,0],sites_cart_2[...,1],facecolor='none',edgecolor='r',label='spirit',s=50)
    ax.legend(scatterpoints=1)
    ax.set_aspect('equal')
    ax.set_axis_off()
    fig.tight_layout()
    plt.show()
    for sites_cart in (sites_cart_1,sites_cart_2):
        mm=np.min(sites_cart,axis=(0,1,2))
        nn=np.max(sites_cart,axis=(0,1,2))
        print ('{:8.5f} {:8.5f}'.format(*tuple(mm)))
        print ('{:8.5f} {:8.5f}'.format(*tuple(nn)))
    print ('sites    consistency: ',np.allclose(sites_cart_1,sites_cart_2,atol=1e-6))


nx=10
ny=nx
radius=1.5
quiver_kws = dict(scale=1.2,units='x',pivot='mid')



if __name__=='__main__':
    sp_lat,skyr_idx,sites_cart, = asd_gen_skyrmion(nx,ny,radius,return_skyr_idx=True)
    sp_lat_spirit,skyr_idx_spirit,sites_cart_spirit = spirit_gen_skyrmion(nx,ny,radius,display=True)

    #compare_sites(sites_cart,sites_cart_spirit)
    print ('\n')
    print ('skyr_idx consistency: ',np.allclose(skyr_idx,skyr_idx_spirit))
    print ('\n')

    fmt0 = '{:3d} '*3
    fmt1 = '{:8.4f} '*3
    print ('-'*90)
    print ('{} | {}'.format('asd'.center(40),'Spirit'.center(40)))
    print ('-'*90)
    for ii in range(len(skyr_idx)):
        ix,iy,iat = skyr_idx[ii]
        print (fmt0.format(ix,iy,iat),fmt1.format(*tuple(sp_lat[ix,iy,iat])),end=' | ')
        ix,iy,iat = skyr_idx_spirit[ii]
        print (fmt0.format(ix,iy,iat),fmt1.format(*tuple(sp_lat_spirit[ix,iy,iat])))
    print ('-'*90+'\n')
