#!/usr/bin/env python

# generate and display a domain wall by asd tools
# the type of domain wall can be determined by helicity
# 0 for Neel and 1 for Bloch type DWs
# Shunhong Zhang <szhang2@ustc.edu.cn>
# September 22, 2021


import matplotlib.pyplot as plt
import numpy as np
from spirit import state,configuration,geometry,io,system
from asd.core.spin_configurations import *
from asd.utility.spin_visualize_tools import *
from asd.core.geometry import *


def asd_gen_dw(nx=8,ny=8,normal_vec=[1,0,0],width=6,helicity=1,display=True,lat_type='square'):
    latt,sites = build_latt(lat_type,nx,ny,1,return_neigh=False)
    nat = sites.shape[-2]
    sp_lat = init_domain_wall(latt,sites,normal_vec,width,helicity=helicity)
    shape = sp_lat.shape
    sites_cart = np.dot(sites,latt)
    if display:
        fmt = '[{:4.2f}, '*(len(normal_vec)-1)+'{:4.2f}]'
        title = 'Domain Wall generated by asd tools\n'
        title+= '\n$\mathbf{n}$ = '
        title+= fmt.format(*tuple(normal_vec))
        title+= ', width = {:4.2f}, helicity = {:4.2f}'.format(width,helicity)
        
        kwargs = dict(title=title,
        scatter_size=20,
        quiver_kws = dict(
        units='x',
        pivot='mid',
        scale=1.2,
        headwidth=2,
        headlength=3),
        show=display,
        colorbar_axes_position=[0.8,0.4,0.015,0.25],
        colorbar_orientation='vertical',
        )

        fig,ax,qv,tl,scat = plot_spin_2d(sites_cart,sp_lat,**kwargs)
        
    params = gen_params_for_ovf(*tuple(shape[:-1]))
    spins = np.swapaxes(sp_lat,0,1).reshape(-1,3)
    write_ovf(params,spins,'DW_conf.ovf')

    # for double checking
    #spins = parse_ovf('DW_conf.ovf')[1]
    #sp_lat = np.swapaxes(spins.reshape(ny,nx,nat,3),0,1)
    #fig,ax,qv,tl,scat = plot_spin_2d(sites_cart,sp_lat,title=title,**kwargs)

    return sp_lat,sites_cart


if __name__=='__main__':

    kwargs = dict(
    nx=10,
    ny=10,
    normal_vec=[np.sqrt(3)/2,0.5],
    width=6,
    helicity=1,
    display=True,
    lat_type='honeycomb')

    sp_lat_1,sites_cart_1 = asd_gen_dw(**kwargs)
