# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['mrsal', 'mrsal.config']

package_data = \
{'': ['*']}

install_requires = \
['colorlog>=6.7.0,<7.0.0', 'pika>=1.3.0,<2.0.0', 'retry>=0.9.2,<0.10.0']

setup_kwargs = {
    'name': 'mrsal',
    'version': '0.3.0a0',
    'description': 'Mrsal is a simple to use message broker abstraction on top of RabbitMQ and Pika.',
    'long_description': '# MRSAL  <img align="right" width="125" alt="20200907_104224" src="https://user-images.githubusercontent.com/29639563/187228621-af1d695d-29a3-4940-9a8c-c19bcd6421a5.png">\n<img src="https://img.shields.io/badge/release-v0.1.0--alpha-blue" height="20" /> [![Python 3.9](https://img.shields.io/badge/python-3.9-blue.svg)](https://www.python.org/downloads/release/python-390/)\n[![Python 3.8](https://img.shields.io/badge/python-3.8-blue.svg)](https://www.python.org/downloads/release/python-380/) \n\n## Intro\nMrsal is a simple to use message broker abstraction on top of [RabbitMQ](https://www.rabbitmq.com/) and [Pika](https://pika.readthedocs.io/en/stable/index.html). The goal is to make Mrsal trivial to re-use in all services of a distributed system and to make the use of advanced message queing protocols easy and safe. No more big chunks of repetive code across your services or bespoke solutions to handle dead letters. \n\n###### Mrsal is Arabic for a small arrow and is used to describe something that performs a task with lightness and speed. \n\n## Quick Start guide\n\n### 0. Install\n\nFirst things first: \n\n```bash\npip install mrsal\n```\n\nWe need to install RabbitMQ to use Mrsal. Head over to [install](https://www.rabbitmq.com/download.html) RabbitMQ. Make sure to stick to the configuration that you give the installation throughout this guide. You can also use the [Dockerfile](https://github.com/NeoMedSys/mrsal/blob/main/Dockerfile) and the [docker-compose](https://github.com/NeoMedSys/mrsal/blob/main/docker-compose.yml) that we are using in the full guide.\n\nNext set the default username, password and servername for your RabbitMQ setup. It\'s advisable to use a `.env` script or the rc file for persistence.\n\n```bash\n[RabbitEnvVars]\nRABBITMQ_DEFAULT_USER=******\nRABBITMQ_DEFAULT_PASS=******\nRABBITMQ_DEFAULT_VHOST=******\nRABBITMQ_DOMAIN=******\nRABBITMQ_DOMAIN_TLS=******\n\nRABBITMQ_GUI_PORT=******\nRABBITMQ_PORT=******\nRABBITMQ_PORT_TLS=******\n\n# FOR TLS\nRABBITMQ_CAFILE=/path/to/file\nRABBITMQ_CERT=/path/to/file\nRABBITMQ_KEY=/path/to/file\n```\n\nPlease read the [full guide](https://github.com/NeoMedSys/mrsal/blob/main/FullGuide.md) to understand what Mrsal currently can and can\'t do.\n\n###### Mrsal was first developed by NeoMedSys and the research group [CRAI](https://crai.no/) at the univeristy hospital of Oslo.\n\n### 1. Setup and connect\n\n\nThe first thing we need to do is to setup our rabbit server before we can subscribe and publish to it. Lets set up a server on our localhost with the port and credentials we used when spinning up the docker-compose\n\n```python\nimport json\nimport pika\nfrom mrsal.mrsal import Mrsal\n\n# If you want to use SSL for external listening then set it to True\nSSL = False\n\n# Note RabbitMQ container is listening on:\n# 1. When SSL is False the default port 5672 which is exposed to RABBITMQ_PORT in docker-compose\n# 2. When SSL is True the default port 5671 which is exposed to RABBITMQ_PORT_TLS in docker-compose\nport = RABBITMQ_PORT_TLS if SSL else RABBITMQ_PORT\nhost = RABBITMQ_DOMAIN_TLS if SSL else RABBITMQ_DOMAIN\n\n# It should match with the env specifications (RABBITMQ_DEFAULT_USER, RABBITMQ_DEFAULT_PASS)\ncredentials=(RABBITMQ_DEFAULT_USER, RABBITMQ_DEFAULT_PASS)\n\n# It should match with the env specifications (RABBITMQ_DEFAULT_VHOST)\nv_host = RABBITMQ_DEFAULT_VHOST\n\nmrsal = Mrsal(\n    host=host,\n    port=port,\n    credentials=credentials,\n    virtual_host=v_host,\n    ssl=SSL\n)\n\nmrsal.connect_to_server()\n```\n\n### 2 Publish\nNow lets publish our message of friendship on the friendship exchange.\nNote: When `fast_setup=True` that means Mrsal will create the specified `exchange` and `queue`, then bind them together using `routing_key`.\n\n```python\n# BasicProperties is used to set the message properties\nprop = pika.BasicProperties(\n        app_id=\'friendship_app\',\n        message_id=\'friendship_msg\',\n        content_type=\'text/plain\',\n        content_encoding=\'utf-8\',\n        delivery_mode=pika.DeliveryMode.Persistent,\n        headers=None)\n\nmessage_body = \'Hello\'\n\n# Publish the message to the exchange to be routed to queue\nmrsal.publish_message(exchange=\'friendship\',\n                        exchange_type=\'direct\',\n                        queue=\'friendship_queue\',\n                        routing_key=\'friendship_key\',\n                        message=json.dumps(message_body), \n                        prop=prop,\n                        fast_setup=True)\n```\n\n### 3 Consume\n\nNow lets setup a consumer that will listen to our very important messages. If you are using scripts rather than notebooks then it\'s advisable to run consume and publish separately. We are going to need a callback function which is triggered upon receiving the message from the queue we subscribe to. You can use the callback function to activate something in your system.\n\nNote: \n- If you start a consumer with `callback_with_delivery_info=True` then your callback function should have at least these params `(method_frame: pika.spec.Basic.Deliver, properties: pika.spec.BasicProperties, message_param: str)`. \n- If not, then it should have at least `(message_param: str)`\n\n```python\nimport json\n\ndef consumer_callback_with_delivery_info(host_param: str, queue_param: str, method_frame: pika.spec.Basic.Deliver, properties: pika.spec.BasicProperties, message_param: str):\n    str_message = json.loads(message_param).replace(\'"\', \'\')\n    if \'Hello\' in str_message:\n        app_id = properties.app_id\n        msg_id = properties.message_id\n        print(f\'app_id={app_id}, msg_id={msg_id}\')\n        print(\'Salaam habibi\')\n        return True  # Consumed message processed correctly\n    return False\n\ndef consumer_callback(host_param: str, queue_param: str, message_param: str):\n    str_message = json.loads(message_param).replace(\'"\', \'\')\n    if \'Hello\' in str_message:\n        print(\'Salaam habibi\')\n        return True  # Consumed message processed correctly\n    return False\n\nmrsal.start_consumer(\n        queue=\'friendship_queue\',\n        callback=consumer_callback_with_delivery_info,\n        callback_args=(test_config.HOST, \'friendship_queue\'),\n        inactivity_timeout=1,\n        requeue=False,\n        callback_with_delivery_info=True\n    )\n```\n\nDone! Your first message of friendship has been sent to the friendship queue on the exchange of friendship.\n\nThat simple! You have now setup a full advanced message queueing protocol that you can use to promote friendship or other necessary communication between your services.\n\n###### Note! Please refer to the full guide on how to use customize Mrsal to meet specific needs. There are many parameters and settings that you can use to set up a more sophisticated communication protocol.\n',
    'author': 'Raafat',
    'author_email': 'rafatzahran90@gmail.com',
    'maintainer': 'Raafat',
    'maintainer_email': 'rafatzahran90@gmail.com',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
