def ler_POF(path, header=True):
    '''
    Realiza a leitura dos microdados da POF 2017/2018 diretamente do arquivo .zip baixado do site do IBGE
    e exporta os dados como um arquivo .csv.
    :param path: caminho para o arquivo .zip
    :param header: boolean, Default True - acrescenta o código da variável como nome de cada coluna.
    https://github.com/otavio-s-s/lerMicrodados
     '''

    import pandas as pd
    from zipfile import ZipFile

    pof_2017 = {"ALUGUEL_ESTIMADO": [[2, 4, 1, 9, 2, 1, 2, 7, 2, 10, 2, 2, 12, 10, 1, 2, 14, 14, 10],
                                      ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC',
                                       'QUADRO', 'V9001', 'V9002', 'V8000', 'V9010', 'V9011', 'DEFLATOR', 'V8000_DEFLA',
                                       'COD_IMPUT_VALOR', 'FATOR_ANUALIZACAO', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL']],
                 "CADERNETA_COLETIVA": [[2, 4, 1, 9, 2, 1, 2, 3, 7, 2, 10, 12, 10, 1, 2, 14, 14, 10, 9, 4, 5, 9],
                                        ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC',
                                         'QUADRO', 'SEQ', 'V9001', 'V9002', 'V8000', 'DEFLATOR', 'V8000_DEFLA',
                                         'COD_IMPUT_VALOR', 'FATOR_ANUALIZACAO', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL',
                                         'V9005', 'V9007', 'V9009', 'QTD_FINAL']],
                 "DESPESA_COLETIVA": [
                     [2, 4, 1, 9, 2, 1, 2, 2, 7, 2, 4, 10, 2, 2, 1, 10, 1, 12, 10, 10, 1, 1, 2, 14, 14, 10],
                     ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC', 'QUADRO', 'SEQ',
                      'V9001', 'V9002', 'V9005', 'V8000', 'V9010', 'V9011', 'V9012', 'V1904', 'V1905', 'DEFLATOR',
                      'V8000_DEFLA', 'V1904_DEFLA', 'COD_IMPUT_VALOR', 'COD_IMPUT_QUANTIDADE', 'FATOR_ANUALIZACAO',
                      'PESO', 'PESO_FINAL', 'RENDA_TOTAL']],
                 "DESPESA_INDIVIDUAL": [[2, 4, 1, 9, 2, 1, 2, 2, 2, 7, 2, 10, 2, 2, 1, 1, 1, 12, 10, 1, 2, 14, 14, 10],
                                        ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC',
                                         'COD_INFORMANTE', 'QUADRO', 'SEQ', 'V9001', 'V9002', 'V8000', 'V9010', 'V9011',
                                         'V9012', 'V4104', 'V4105', 'DEFLATOR', 'V8000_DEFLA', 'COD_IMPUT_VALOR',
                                         'FATOR_ANUALIZACAO', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL']],
                 "DOMICILIO": [
                     [2, 4, 1, 9, 2, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                      1, 1, 14, 14],
                     ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'V0201', 'V0202', 'V0203',
                      'V0204', 'V0205', 'V0206', 'V0207', 'V0208', 'V0209', 'V02101', 'V02102', 'V02103', 'V02104',
                      'V02105', 'V02111', 'V02112', 'V02113', 'V0212', 'V0213', 'V02141', 'V02142', 'V0215', 'V02161',
                      'V02162', 'V02163', 'V02164', 'V0217', 'V0219', 'V0220', 'V0221', 'PESO', 'PESO_FINAL']],
                 "MORADOR": [
                     [2, 4, 1, 9, 2, 1, 2, 2, 1, 2, 2, 4, 3, 1, 1, 1, 1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                      1, 1, 2, 1, 1, 2, 1, 1, 2, 1, 1, 1, 2, 1, 2, 14, 14, 10],
                     ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC', 'COD_INFORMANTE',
                      'V0306', 'V0401', 'V04021', 'V04022', 'V04023', 'V0403', 'V0404', 'V0405', 'V0406', 'V0407',
                      'V0408', 'V0409', 'V0410', 'V0411', 'V0412', 'V0413', 'V0414', 'V0415', 'V0416', 'V041711',
                      'V041712', 'V041721', 'V041722', 'V041731', 'V041732', 'V041741', 'V041742', 'V0418', 'V0419',
                      'V0420', 'V0421', 'V0422', 'V0423', 'V0424', 'V0425', 'V0426', 'V0427', 'V0428', 'V0429', 'V0430',
                      'ANOS_ESTUDO', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL']],
                 "OUTROS_RENDIMENTOS": [[2, 4, 1, 9, 2, 1, 2, 2, 2, 7, 10, 10, 2, 2, 12, 10, 10, 1, 1, 14, 14, 10],
                                        ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC',
                                         'COD_INFORMANTE', 'QUADRO', 'SEQ', 'V9001', 'V8500', 'V8501', 'V9010', 'V9011',
                                         'DEFLATOR', 'V8500_DEFLA', 'V8501_DEFLA', 'COD_IMPUT_VALOR',
                                         'FATOR_ANUALIZACAO', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL']],
                 "RENDIMENTO_TRABALHO": [
                     [2, 4, 1, 9, 2, 1, 2, 2, 1, 1, 7, 1, 1, 1, 1, 1, 1, 7, 7, 7, 7, 2, 2, 3, 1, 12, 10, 10, 10, 10, 1,
                      1, 14, 14, 10],
                     ['UF', 'ESTRATO_POF', 'TIPO_SITUACAO_REG', 'COD_UPA', 'NUM_DOM', 'NUM_UC', 'COD_INFORMANTE',
                      'QUADRO', 'SUB_QUADRO', 'SEQ', 'V9001', 'V5302', 'V53021', 'V5303', 'V5304', 'V5305', 'V5307',
                      'V8500', 'V531112', 'V531122', 'V531132', 'V9010', 'V9011', 'V5314', 'V5315', 'DEFLATOR',
                      'V8500_DEFLA', 'V531112_DEFLA', 'V531122_DEFLA', 'V531132_DEFLA', 'COD_IMPUT_VALOR',
                      'FATOR_ANUALIZACAO', 'PESO', 'PESO_FINAL', 'RENDA_TOTAL']]}

    zip_file = ZipFile(path, mode='r')
    for text_file in zip_file.infolist():
        file = text_file.filename
        name = '{}'.format(text_file.filename.split('.')[0])
        new_file = pd.read_fwf(zip_file.open(file), widths=pof_2017[name][0], index=False, header=None, dtype=str)
        if header:
            new_file.columns = pof_2017[name][1]
        new_file.to_csv('{}.csv'.format(name), index=False)
        print('{}.csv exportado com sucesso.'.format(name))


def ler_PNAD(path, ano, header=True):
    '''
    Realiza a leitura dos microdados da PNAD 2013, 2014 ou 2015 diretamente do arquivo .zip baixado do site do IBGE
    e exporta os dados como um arquivo .csv.
    :param path: aminho para o arquivo .zip
    :param ano: ano da PNAD.
    :param header:  boolean, Default True - acrescenta o código da variável como nome de cada coluna.
    https://github.com/otavio-s-s/lerMicrodados
    '''

    import pandas as pd
    from zipfile import ZipFile

    anos = [2013, 2014, 2015]

    if ano not in anos:
        raise ValueError('Ano inválido. Os anos aceitavéis são: {}'.format(anos))

    pnads = {
    'pnad_2013': {'PES2013': [[4, 8, 3, 2, 1, 2, 2, 4, 3, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 4, 5, 1, 4, 5, 1, 1, 12, 1, 12, 1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 2, 1, 1, 1, 1, 1, 1, 1, 11, 7, 1, 11, 7, 1, 11, 7, 1, 1, 1, 1, 1, 11, 7, 1, 11, 7, 1, 11, 7, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 12, 1, 12, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 2, 2, 1, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 2, 4, 5, 1, 1, 1, 1, 1, 1, 1, 12, 1, 12, 1, 1, 1, 2, 1, 12, 1, 12, 1, 1, 1, 1, 2, 1, 1, 1, 2, 2, 4, 5, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 1, 1, 2, 2, 2, 2, 1, 1, 2, 2, 1, 1, 1, 2, 4, 1, 1, 2, 2, 1, 1, 1, 2, 2, 2, 1, 1, 2, 1, 1, 2, 2, 1, 1, 1, 1, 2, 2, 2, 12, 12, 12, 12, 12, 2, 2, 1, 1, 5, 5, 1, 1, 1, 2, 12, 2, 1, 1, 1, 1, 1, 12, 8], ['V0101',  'V0102', 'V0103', 'V0301', 'V0302', 'V3031', 'V3032', 'V3033', 'V8005', 'V0401', 'V0402', 'V0403', 'V0404', 'V0405', 'V0406', 'V0407', 'V0408', 'V0409', 'V0410', 'V4111', 'V4112', 'V4011', 'V0412', 'V0501', 'V0502', 'V5030', 'V0504', 'V0505', 'V5061', 'V5062', 'V5063', 'V5064', 'V5065', 'V0507', 'V5080', 'V5090', 'V0510', 'V0511', 'V5121', 'V5122', 'V5123', 'V5124', 'V5125', 'V5126', 'V0601', 'V0602', 'V6002', 'V6020', 'V6003', 'V6030', 'V0604', 'V0605', 'V0606', 'V6007', 'V6070', 'V0608', 'V0609', 'V0610', 'V0611', 'V06111', 'V061111', 'V061112', 'V061113', 'V061114', 'V061115', 'V061116', 'V06112', 'V0612', 'V0701', 'V0702', 'V0703', 'V0704', 'V0705', 'V7060', 'V7070', 'V0708', 'V7090', 'V7100', 'V0711', 'V7121', 'V7122', 'V7124', 'V7125', 'V7127', 'V7128', 'V0713', 'V0714', 'V0715', 'V0716', 'V9001', 'V9002', 'V9003', 'V9004', 'V9005', 'V9906', 'V9907', 'V9008', 'V9009', 'V9010', 'V9011', 'V9012', 'V9013', 'V9014', 'V9151', 'V9152', 'V9154', 'V9156', 'V9157', 'V9159', 'V9161', 'V9162', 'V9164', 'V9016', 'V9017', 'V9018', 'V9019', 'V9201', 'V9202', 'V9204', 'V9206', 'V9207', 'V9209', 'V9211', 'V9212', 'V9214', 'V9021', 'V9022', 'V9023', 'V9024', 'V9025', 'V9026', 'V9027', 'V9028', 'V9029', 'V9030', 'V9031', 'V9032', 'V9033', 'V9034', 'V9035', 'V9036', 'V9037', 'V9038', 'V9039', 'V9040', 'V9041', 'V9042', 'V9043', 'V9044', 'V9045', 'V9046', 'V9047', 'V9048', 'V9049', 'V9050', 'V9051', 'V9052', 'V9531', 'V9532', 'V9534', 'V9535', 'V9537', 'V90531', 'V90532', 'V90533', 'V9054', 'V9055', 'V9056', 'V9057', 'V9058', 'V9059', 'V9060', 'V9611', 'V9612', 'V9062', 'V9063', 'V9064', 'V9065', 'V9066', 'V9067', 'V9068', 'V9069', 'V9070', 'V9971', 'V9972', 'V9073', 'V9074', 'V9075', 'V9076', 'V9077', 'V9078', 'V9079', 'V9080', 'V9081', 'V9082', 'V9083', 'V9084', 'V9085', 'V9861', 'V9862', 'V9087', 'V9088', 'V9891', 'V9892', 'V9990', 'V9991', 'V9092', 'V9093', 'V9094', 'V9095', 'V9096', 'V9097', 'V9981', 'V9982', 'V9984', 'V9985', 'V9987', 'V9099', 'V9100', 'V9101', 'V1021', 'V1022', 'V1024', 'V1025', 'V1027', 'V1028', 'V9103', 'V9104', 'V9105', 'V9106', 'V9107', 'V9108', 'V1091', 'V1092', 'V9910', 'V9911', 'V9112', 'V9113', 'V9114', 'V9115', 'V9116', 'V9117', 'V9118', 'V9119', 'V9120', 'V9121', 'V9921', 'V9122', 'V9123', 'V9124', 'V1251', 'V1252', 'V1254', 'V1255', 'V1257', 'V1258', 'V1260', 'V1261', 'V1263', 'V1264', 'V1266', 'V1267', 'V1269', 'V1270', 'V1272', 'V1273', 'V9126', 'V1101', 'V1141', 'V1142', 'V1151', 'V1152', 'V1153', 'V1154', 'V1161', 'V1162', 'V1163', 'V1164', 'V1107', 'V1181', 'V1182', 'V1109', 'V1110', 'V1111', 'V1112', 'V1113', 'V1114', 'V1115', 'V4801', 'V4802', 'V4803', 'V4704', 'V4805', 'V4706', 'V4707', 'V4808', 'V4809', 'V4810', 'V4711', 'V4812', 'V4713', 'V4814', 'V4715', 'V4816', 'V4817', 'V4718', 'V4719', 'V4720', 'V4721', 'V4722', 'V4723', 'V4724', 'V4727', 'V4728', 'V4729', 'V4732', 'V4735', 'V4838', 'V6502', 'V4741', 'V4742', 'V4743', 'V4745', 'V4746', 'V4747', 'V4748', 'V4749', 'V4750', 'V9993']],
               'DOM2013': [[4, 8, 3, 2, 2, 2, 1, 1, 1, 1, 2, 2, 1, 12, 12, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 4, 2, 12, 3, 12, 6, 9, 3, 5, 12, 4, 7, 7, 2, 12, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 8], ['V0101', 'V0102', 'V0103', 'V0104', 'V0105', 'V0106', 'V0201', 'V0202', 'V0203', 'V0204', 'V0205', 'V0206', 'V0207', 'V0208', 'V0209', 'V0210', 'V0211', 'V0212', 'V0213', 'V0214', 'V0215', 'V0216', 'V2016', 'V0217', 'V0218', 'V0219', 'V0220', 'V2020', 'V0221', 'V0222', 'V0223', 'V0224', 'V0225', 'V0226', 'V0227', 'V02270', 'V02271', 'V02272', 'V02273', 'V02274', 'V2027', 'V0228', 'V0229', 'V0230', 'V0231', 'V0232', 'V02321', 'V02322', 'V02323', 'V02324', 'V02325', 'V02326', 'V02327', 'V02424', 'V02425', 'V02426', 'V2032', 'V4105', 'V4107', 'V4600', 'V4601', 'V4602', 'V4604', 'V4605', 'V4606', 'V4607', 'V4608', 'V4609', 'V4610', 'V4611', 'V4614', 'UPA', 'V4617', 'V4618', 'V4620', 'V4621', 'V4622', 'V4623A', 'V4624', 'V2101', 'V2103', 'V2105', 'V2107', 'V2109', 'V2113', 'V2115', 'V2117', 'V2121', 'V2123', 'V2124', 'V2126', 'V2128', 'V2130', 'V2132', 'V2134', 'V2138', 'V2139', 'V9992']]},

    'pnad_2014': {'PES2014': [[4, 8, 3, 2, 1, 2, 2, 4, 3, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 4, 5, 1, 4, 5, 1, 1, 12, 1, 12, 1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 2, 1, 1, 1, 1, 1, 1, 1, 11, 7, 1, 11, 7, 1, 11, 7, 1, 1, 1, 1, 1, 11, 7, 1, 11, 7, 1, 11, 7, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 12, 1, 12, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 2, 2, 1, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 2, 4, 5, 1, 1, 1, 1, 1, 1, 1, 12, 1, 12, 1, 1, 1, 2, 1, 12, 1, 12, 1, 1, 1, 1, 2, 1, 1, 1, 2, 2, 4, 5, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 1, 1, 2, 2, 2, 2, 1, 1, 2, 2, 1, 1, 1, 2, 4, 1, 1, 2, 2, 1, 1, 1, 2, 2, 2, 1, 1, 2, 1, 1, 2, 2, 1, 1, 1, 1, 2, 2, 2, 12, 12, 12, 12, 12, 2, 2, 1, 1, 5, 5, 1, 1, 1, 2, 12, 2, 1, 1, 1, 1, 1, 12, 1, 2, 1, 2, 1, 1, 4, 5, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 1, 1, 1, 5, 5, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 2, 1, 1, 1, 2, 1, 8], ['V0101', 'V0102', 'V0103', 'V0301', 'V0302', 'V3031', 'V3032', 'V3033', 'V8005', 'V0401', 'V0402', 'V0403', 'V0404', 'V0405', 'V0406', 'V0407', 'V0408', 'V0409', 'V0410', 'V4111', 'V4112', 'V4011', 'V0412', 'V0501', 'V0502', 'V5030', 'V0504', 'V0505', 'V5061', 'V5062', 'V5063', 'V5064', 'V5065', 'V0507', 'V5080', 'V5090', 'V0510', 'V0511', 'V5121', 'V5122', 'V5123', 'V5124', 'V5125', 'V5126', 'V0601', 'V0602', 'V6002', 'V6020', 'V6003', 'V6030', 'V0604', 'V0605', 'V0606', 'V6007', 'V6070', 'V0608', 'V0609', 'V0610', 'V0611', 'V06111', 'V061111', 'V061112', 'V061113', 'V061114', 'V061115', 'V061116', 'V06112', 'V0612', 'V0701', 'V0702', 'V0703', 'V0704', 'V0705', 'V7060', 'V7070', 'V0708', 'V7090', 'V7100', 'V0711', 'V7121', 'V7122', 'V7124', 'V7125', 'V7127', 'V7128', 'V0713', 'V0714', 'V0715', 'V0716', 'V9001', 'V9002', 'V9003', 'V9004', 'V9005', 'V9906', 'V9907', 'V9008', 'V9009', 'V9010', 'V9011', 'V9012', 'V9013', 'V9014', 'V9151', 'V9152', 'V9154', 'V9156', 'V9157', 'V9159', 'V9161', 'V9162', 'V9164', 'V9016', 'V9017', 'V9018', 'V9019', 'V9201', 'V9202', 'V9204', 'V9206', 'V9207', 'V9209', 'V9211', 'V9212', 'V9214', 'V9021', 'V9022', 'V9023', 'V9024', 'V9025', 'V9026', 'V9027', 'V9028', 'V90281', 'V90282', 'V90283', 'V90284', 'V90285', 'V9029', 'V9030', 'V9031', 'V9032', 'V9033', 'V9034', 'V9035', 'V9036', 'V9037', 'V9038', 'V9039', 'V9040', 'V9041', 'V9042', 'V9043', 'V9044', 'V9045', 'V9046', 'V9047', 'V9048', 'V9049', 'V9050', 'V9051', 'V9052', 'V90521', 'V90522', 'V90523', 'V90524', 'V90525', 'V90526', 'V9531', 'V9532', 'V9534', 'V9535', 'V9537', 'V90531', 'V90532', 'V90533', 'V9054', 'V9055', 'V9056', 'V9057', 'V9058', 'V9059', 'V9060', 'V9611', 'V9612', 'V9062', 'V9063', 'V9064', 'V9065', 'V9066', 'V9067', 'V9068', 'V9069', 'V9070', 'V9971', 'V9972', 'V9073', 'V9074', 'V9075', 'V9076', 'V9077', 'V9078', 'V9079', 'V9080', 'V9081', 'V9082', 'V9083', 'V9084', 'V9085', 'V9861', 'V9862', 'V9087', 'V9088', 'V9891', 'V9892', 'V9990', 'V9991', 'V9092', 'V9093', 'V9094', 'V9095', 'V9096', 'V9097', 'V9981', 'V9982', 'V9984', 'V9985', 'V9987', 'V9099', 'V9100', 'V9101', 'V1021', 'V1022', 'V1024', 'V1025', 'V1027', 'V1028', 'V9103', 'V9104', 'V9105', 'V9106', 'V9107', 'V9108', 'V1091', 'V1092', 'V9910', 'V9911', 'V9112', 'V9113', 'V9114', 'V9115', 'V9116', 'V9117', 'V9118', 'V9119', 'V9120', 'V9121', 'V9921', 'V9122', 'V9123', 'V9124', 'V1251', 'V1252', 'V1254', 'V1255', 'V1257', 'V1258', 'V1260', 'V1261', 'V1263', 'V1264', 'V1266', 'V1267', 'V1269', 'V1270', 'V1272', 'V1273', 'V9126', 'V1101', 'V1141', 'V1142', 'V1151', 'V1152', 'V1153', 'V1154', 'V1161', 'V1162', 'V1163', 'V1164', 'V1107', 'V1181', 'V1182', 'V1109', 'V1110', 'V1111', 'V1112', 'V1113', 'V1114', 'V1115', 'V4801', 'V4802', 'V4803', 'V4704', 'V4805', 'V4706', 'V4707', 'V4808', 'V4809', 'V4810', 'V4711', 'V4812', 'V4713', 'V4814', 'V4715', 'V4816', 'V4817', 'V4718', 'V4719', 'V4720', 'V4721', 'V4722', 'V4723', 'V4724', 'V4727', 'V4728', 'V4729', 'V4732', 'V4735', 'V4838', 'V6502', 'V4741', 'V4742', 'V4743', 'V4745', 'V4746', 'V4747', 'V4748', 'V4749', 'V4750', 'V0305', 'V32000', 'V32001', 'V32002', 'V32003', 'V32004', 'V32005', 'V32006', 'V32007', 'V32008', 'V32009', 'V32010', 'V32011', 'V32012', 'V32013', 'V32014', 'V32015', 'V32016', 'V32017', 'V32018', 'V32019', 'V32020', 'V32021', 'V32022', 'V32023', 'V32024', 'V32025', 'V32026', 'V32027', 'V32028', 'V32029', 'V32030', 'V32031', 'V32032', 'V32033', 'V32034', 'V32035', 'V32036', 'V32037', 'V32038', 'V32039', 'V31001', 'V31002', 'V31003', 'V31004', 'V31005', 'V31007', 'V31008', 'V31009', 'V31010', 'V31012', 'V31013', 'V31015', 'V31016', 'V31017', 'V31018', 'V31020', 'V31025', 'V31026', 'V31027', 'V31028', 'V31029', 'V31030', 'V31032', 'V31033', 'V31034', 'V31035', 'V31037', 'V31038', 'V31039', 'V31040', 'V31042', 'V31043', 'V31044', 'V31045', 'V31046', 'V31047', 'V31048', 'V31050', 'V31055', 'V31056', 'V31057', 'V31058', 'V31059', 'V31060', 'V31061', 'V31062', 'V31064', 'V31065', 'V31066', 'V31067', 'V31068', 'V31069', 'V31070', 'V31071', 'V31072', 'V31073', 'V31074', 'V31076', 'V31077', 'V31078', 'V31079', 'V31080', 'V31081', 'V31082', 'V31083', 'V31084', 'V31085', 'V31086', 'V31087', 'V31088', 'V31089', 'V31090', 'V31091', 'V31092', 'V31093', 'V31094', 'V9993']],
                'DOM2014': [[4, 8, 3, 2, 2, 2, 1, 1, 1, 1, 2, 2, 1, 12, 12, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 4, 2, 12, 3, 12, 6, 9, 3, 5, 12, 4, 7, 7, 2, 12, 2, 1, 1, 9, 5, 1, 1, 1, 1, 8], ['V0101', 'V0102', 'V0103', 'V0104', 'V0105', 'V0106', 'V0201', 'V0202', 'V0203', 'V0204', 'V0205', 'V0206', 'V0207', 'V0208', 'V0209', 'V0210', 'V0211', 'V0212', 'V0213', 'V0214', 'V0215', 'V0216', 'V2016', 'V0217', 'V0218', 'V0219', 'V0220', 'V2020', 'V0221', 'V0222', 'V0223', 'V0224', 'V0225', 'V0226', 'V0227', 'V02270', 'V02271', 'V02272', 'V02273', 'V02274', 'V2027', 'V0228', 'V0229', 'V0230', 'V0231', 'V0232', 'V02321', 'V02322', 'V02323', 'V02324', 'V02325', 'V02326', 'V02327', 'V02424', 'V02425', 'V02426', 'V2032', 'V4105', 'V4107', 'V4600', 'V4601', 'V4602', 'V4604', 'V4605', 'V4606', 'V4607', 'V4608', 'V4609', 'V4610', 'V4611', 'V4614', 'UPA', 'V4617', 'V4618', 'V4620', 'V4621', 'V4622', 'V4624', 'V4625', 'V4626', 'V4627', 'V33001', 'V33002', 'V33003', 'V33004', 'V9992']]},

    'pnad_2015': {'PES2015': [[4, 8, 3, 2, 1, 2, 2, 4, 3, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 4, 5, 1, 4, 5, 1, 1, 12, 1, 12, 1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 11, 7, 1, 11, 7, 1, 11, 7, 1, 1, 1, 1, 1, 11, 7, 1, 11, 7, 1, 11, 7, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 12, 1, 12, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 2, 2, 1, 1, 2, 1, 1, 1, 1, 1, 1, 4, 5, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 4, 5, 1, 1, 1, 1, 1, 1, 1, 12, 1, 12, 1, 1, 1, 2, 1, 12, 1, 12, 1, 1, 1, 1, 2, 1, 1, 1, 2, 2, 4, 5, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 2, 12, 1, 1, 2, 2, 2, 2, 1, 1, 2, 2, 1, 1, 1, 2, 4, 1, 1, 2, 2, 1, 1, 1, 2, 2, 2, 1, 1, 2, 1, 1, 2, 2, 1, 1, 1, 1, 2, 2, 2, 12, 12, 12, 12, 12, 2, 2, 1, 1, 5, 5, 1, 1, 1, 2, 12, 2, 1, 1, 1, 1, 1, 12, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 5, 5, 1, 1, 1, 1, 1, 2, 1, 1, 5, 1, 1, 1, 1, 1, 1, 1, 1, 1, 5, 2, 1, 1, 1, 1, 2, 1, 5, 1, 1, 1, 1, 1, 1, 1, 1, 1, 5, 5, 8], ['V0101', 'V0102', 'V0103', 'V0301', 'V0302', 'V3031', 'V3032', 'V3033', 'V8005', 'V0401', 'V0402', 'V0403', 'V0404', 'V0405', 'V0406', 'V0407', 'V0408', 'V4111', 'V4112', 'V4011', 'V0412', 'V0501', 'V0502', 'V5030', 'V0504', 'V0505', 'V5061', 'V5062', 'V5063', 'V5064', 'V5065', 'V0507', 'V5080', 'V5090', 'V0510', 'V0511', 'V5121', 'V5122', 'V5123', 'V5124', 'V5125', 'V5126', 'V0601', 'V0602', 'V6002', 'V6020', 'V6003', 'V6030', 'V0604', 'V0605', 'V0606', 'V6007', 'V6070', 'V0608', 'V0609', 'V0610', 'V0611', 'V06111', 'V061111', 'V061112', 'V061113', 'V061114', 'V061115', 'V061116', 'V06112', 'V0612', 'V0701', 'V0702', 'V0703', 'V0704', 'V0705', 'V7060', 'V7070', 'V0708', 'V7090', 'V7100', 'V0711', 'V7121', 'V7122', 'V7124', 'V7125', 'V7127', 'V7128', 'V0713', 'V0714', 'V0715', 'V0716', 'V9001', 'V9002', 'V9003', 'V9004', 'V9005', 'V9906', 'V9907', 'V9008', 'V9009', 'V9010', 'V90101', 'V9011', 'V9012', 'V90121', 'V9013', 'V9014', 'V9151', 'V9152', 'V9154', 'V9156', 'V9157', 'V9159', 'V9161', 'V9162', 'V9164', 'V9016', 'V9017', 'V9018', 'V9019', 'V9201', 'V9202', 'V9204', 'V9206', 'V9207', 'V9209', 'V9211', 'V9212', 'V9214', 'V9021', 'V9022', 'V9023', 'V9024', 'V9025', 'V9026', 'V9027', 'V9028', 'V9029', 'V9030', 'V9031', 'V9032', 'V9033', 'V9034', 'V9035', 'V9036', 'V9037', 'V9038', 'V9039', 'V9040', 'V9041', 'V9042', 'V9043', 'V9044', 'V9045', 'V9046', 'V9047', 'V9048', 'V9049', 'V9050', 'V9051', 'V9052', 'V9531', 'V9532', 'V9534', 'V9535', 'V9537', 'V90531', 'V90532', 'V90533', 'V9054', 'V9055', 'V9056', 'V9057', 'V9058', 'V9059', 'V9060', 'V9611', 'V9612', 'V9062', 'V9063', 'V9064', 'V9065', 'V9066', 'V9067', 'V9068', 'V9069', 'V9070', 'V9971', 'V9972', 'V9073', 'V9074', 'V9075', 'V9076', 'V9077', 'V9078', 'V9079', 'V9080', 'V9081', 'V9082', 'V9083', 'V9084', 'V9085', 'V9861', 'V9862', 'V9087', 'V90871', 'V908721', 'V908722', 'V908723', 'V908724', 'V908725', 'V908726', 'V90873', 'V90874', 'V9088', 'V90881', 'V90882', 'V908831', 'V908832', 'V908833', 'V908834', 'V908835', 'V908836', 'V908837', 'V90884', 'V908851', 'V908852', 'V908853', 'V908854', 'V908855', 'V908856', 'V90886', 'V90887', 'V908881', 'V908882', 'V908883', 'V908884', 'V908885', 'V908886', 'V908887', 'V9891', 'V9892', 'V9990', 'V9991', 'V9092', 'V9093', 'V9094', 'V9095', 'V9096', 'V9097', 'V9981', 'V9982', 'V9984', 'V9985', 'V9987', 'V9099', 'V9100', 'V9101', 'V1021', 'V1022', 'V1024', 'V1025', 'V1027', 'V1028', 'V9103', 'V9104', 'V9105', 'V9106', 'V9107', 'V9108', 'V1091', 'V1092', 'V9910', 'V9911', 'V9112', 'V9113', 'V9114', 'V9115', 'V9116', 'V9117', 'V9118', 'V9119', 'V9120', 'V9121', 'V9921', 'V9122', 'V9123', 'V9124', 'V1251', 'V1252', 'V1254', 'V1255', 'V1257', 'V1258', 'V1260', 'V1261', 'V1263', 'V1264', 'V1266', 'V1267', 'V1269', 'V1270', 'V1272', 'V1273', 'V9126', 'V1101', 'V1141', 'V1142', 'V1151', 'V1152', 'V1153', 'V1154', 'V1161', 'V1162', 'V1163', 'V1164', 'V1107', 'V1181', 'V1182', 'V1109', 'V1110', 'V1111', 'V1112', 'V1113', 'V1114', 'V1115', 'V4801', 'V4802', 'V4803', 'V4704', 'V4805', 'V4706', 'V4707', 'V4808', 'V4809', 'V4810', 'V4711', 'V4812', 'V4713', 'V4814', 'V4715', 'V4816', 'V4817', 'V4718', 'V4719', 'V4720', 'V4721', 'V4722', 'V4723', 'V4724', 'V4727', 'V4728', 'V4729', 'V4732', 'V4735', 'V4838', 'V6502', 'V4741', 'V4742', 'V4743', 'V4745', 'V4746', 'V4747', 'V4748', 'V4749', 'V4750', 'V38011', 'V38012', 'V3802', 'V3803', 'V3804', 'V3805', 'V3806', 'V3807', 'V3808', 'V3809', 'V37001', 'V37002', 'V3701', 'V3702', 'V3703', 'V3704', 'V3705', 'V3706', 'V37071', 'V37072', 'V37073', 'V37074', 'V37075', 'V37091', 'V37092', 'V37093', 'V37094', 'V37095', 'V37096', 'V37097', 'V37098', 'V3719', 'V3720', 'V36001', 'V36002', 'V3601', 'V3602', 'V3603', 'V3604', 'V3605', 'V3606', 'V3607', 'V3608', 'V3609', 'V3610', 'V3611', 'V3612', 'V3613', 'V3614', 'V3615', 'V3616', 'V3617', 'V3618', 'V3619', 'V3620', 'V3621', 'V3622', 'V3623', 'V3624', 'V3625', 'V3626', 'V3627', 'V3628', 'V3629', 'V3630', 'V3631', 'V3632', 'V3633', 'V3634', 'V3637', 'V3638', 'V9993']],
               'DOM2015': [[4, 8, 3, 2, 2, 2, 1, 1, 1, 1, 2, 2, 1, 12, 12, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 4, 2, 12, 3, 12, 6, 9, 3, 5, 12, 4, 7, 7, 2, 12, 2, 1, 1, 7, 5, 7, 5, 8], ['V0101', 'V0102', 'V0103', 'V0104', 'V0105', 'V0106', 'V0201', 'V0202', 'V0203', 'V0204', 'V0205', 'V0206', 'V0207', 'V0208', 'V0209', 'V0210', 'V0211', 'V0212', 'V0213', 'V0214', 'V0215', 'V0216', 'V2016', 'V0217', 'V0218', 'V0219', 'V0220', 'V2020', 'V0221', 'V0222', 'V0223', 'V0224', 'V0225', 'V0226', 'V0227', 'V02270', 'V02271', 'V02272', 'V02273', 'V02274', 'V2027', 'V0228', 'V0229', 'V0230', 'V0231', 'V0232', 'V02321', 'V02322', 'V02323', 'V02324', 'V02325', 'V02326', 'V02327', 'V02424', 'V02425', 'V02426', 'V2032', 'V4105', 'V4107', 'V4600', 'V4601', 'V4602', 'V4604', 'V4605', 'V4606', 'V4607', 'V4608', 'V4609', 'V4610', 'V4611', 'V4614', 'UPA', 'V4617', 'V4618', 'V4620', 'V4621', 'V4622', 'V4624', 'V4628', 'V4629', 'V4630', 'V4632', 'V4633', 'V9992']]}}

    zip_file = ZipFile(path)
    for text_file in zip_file.infolist():
        dict_name = 'pnad_{}'.format(ano)
        file = text_file.filename
        name = '{}'.format(file.split('.')[0]).split('/')[1]
        new_file = pd.read_fwf(zip_file.open(file), widths=pnads[dict_name][name][0], index=False, header=None, dtype=str)
        if header:
            new_file.columns = pnads[dict_name][name][1]
        new_file.to_csv('{}.csv'.format(name), index=False)
        print('{}.csv exportado com sucesso.'.format(name))