import sys
import unittest
import urllib.error
import urllib.parse
import urllib.request

from cloudshell.cp.vcenter.common.vcenter.ovf_service import OvfImageDeployerService
from cloudshell.cp.vcenter.models.VCenterConnectionDetails import (
    VCenterConnectionDetails,
)

if sys.version_info >= (3, 0):
    from unittest.mock import MagicMock, patch
else:
    from mock import MagicMock, patch


PROCESS = MagicMock()


class ProccesMock:
    @staticmethod
    def Popen(*args, **kwargs):
        PROCESS.args = args[0]

        return PROCESS


class TestOvfImageService(unittest.TestCase):
    @patch("subprocess.Popen", ProccesMock.Popen)
    def test_deploy_image_success(self):
        PROCESS.communicate = MagicMock(return_value=["Completed successfully"])
        expected_args = [
            "dummypath/ovftool.exe",
            "--noSSLVerify",
            "--acceptAllEulas",
            "--powerOffTarget",
            "--name=raz_deploy_image_integration_test",
            "--datastore=aa",
            "--vmFolder=Raz",
            '--vlan="anetwork"',
            "http://192.168.65.88/ovf/Debian 64 - Yoav.ovf",
            "vi://vcenter%20user:password%20to%20vcenter@venter.host.com/QualiSB/host/QualiSB%20Cluster/Resources/LiverPool",
        ]
        ovf = OvfImageDeployerService("dummypath/ovftool.exe")
        image_params = MagicMock()
        image_params.connectivity = VCenterConnectionDetails(
            "venter.host.com", "vcenter user", "password to vcenter"
        )

        image_params.datacenter = "QualiSB"
        image_params.cluster = "QualiSB Cluster"
        image_params.resource_pool = "LiverPool"
        image_params.vm_name = "raz_deploy_image_integration_test"
        image_params.datastore = "aa"
        image_params.power_on = "false"
        image_params.vm_folder = "Raz"
        # image_params.image_url = "C:\\images\\test\\OVAfile121_QS\\OVAfile121_QS.ovf"
        image_params.image_url = "http://192.168.65.88/ovf/Debian 64 - Yoav.ovf"
        image_params.user_arguments = '--vlan="anetwork"'

        vcenter_data_model = MagicMock()
        vcenter_data_model.ovf_tool_path = "dummypath/ovftool.exe"

        ovf._validate_url_exists = MagicMock(return_value=True)

        res = ovf.deploy_image(vcenter_data_model, image_params, logger=MagicMock())

        self.assertTrue(res)
        self.assertEqual(PROCESS.args, expected_args)
        self.assertTrue(PROCESS.stdin.close.called)

    @patch("subprocess.Popen", ProccesMock.Popen)
    def test_deploy_image_no_communication(self):

        PROCESS.communicate = MagicMock(return_value=None)
        ovf = OvfImageDeployerService("dummypath/ovftool.exe")
        image_params = MagicMock()
        image_params.connectivity = VCenterConnectionDetails(
            "venter.host.com", "vcenter user", "password to vcenter"
        )

        image_params.datacenter = "QualiSB"
        image_params.cluster = "QualiSB Cluster"
        image_params.vm_name = "raz_deploy_image_integration_test"
        image_params.resource_pool = "LiverPool"
        image_params.datastore = "aa"
        image_params.image_url = "http://192.168.65.88/ovf/Debian 64 - Yoav.ovf"
        image_params.user_arguments = '--vlan="anetwork"'

        vcenter_data_model = MagicMock()
        vcenter_data_model.ovf_tool_path = "dummypath/ovftool.exe"
        ovf._validate_url_exists = MagicMock(return_value=True)

        self.assertRaises(Exception, ovf.deploy_image, vcenter_data_model, image_params)

    @patch("subprocess.Popen", ProccesMock.Popen)
    def test_deploy_image_error(self):

        PROCESS.communicate = MagicMock(return_value=["error"])
        ovf = OvfImageDeployerService("dummypath/ovftool.exe")
        image_params = MagicMock()
        image_params.connectivity = VCenterConnectionDetails(
            "venter.host.com", "vcenter user", "password to vcenter"
        )
        image_params.datacenter = "QualiSB"
        image_params.cluster = "QualiSB Cluster"
        image_params.resource_pool = "LiverPool"
        image_params.vm_name = "raz_deploy_image_integration_test"
        image_params.datastore = "aa"
        image_params.image_url = "http://192.168.65.88/ovf/Debian 64 - Yoav.ovf"

        image_params.user_arguments = '--vlan="anetwork"'

        ovf._validate_url_exists = MagicMock(return_value=True)

        vcenter_data_model = MagicMock()
        vcenter_data_model.ovf_tool_path = "dummypath/ovftool.exe"

        try:
            ovf.deploy_image(vcenter_data_model, image_params, logger=MagicMock())
            # should not reach here
            self.assertTrue(False)
        except Exception as inst:
            self.assertTrue(str(inst).find("password to vcenter") == -1)
            self.assertTrue(str(inst).find(urllib.parse.quote_plus("******")) > -1)
            self.assertTrue(PROCESS.stdin.close.called)
