# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['proxycurl_py_test',
 'proxycurl_py_test.asyncio',
 'proxycurl_py_test.gevent',
 'proxycurl_py_test.twisted']

package_data = \
{'': ['*']}

extras_require = \
{'asyncio': ['asyncio>=3.4.3,<4.0.0', 'aiohttp>=3.7.4,<4.0.0'],
 'gevent': ['gevent>=21.1.1,<22.0.0', 'requests>=2.25.0,<3.0.0'],
 'twisted': ['Twisted>=21.7.0,<22.0.0', 'treq>=21.5.0,<22.0.0']}

setup_kwargs = {
    'name': 'proxycurl-py-test-2',
    'version': '0.0.17',
    'description': '',
    'long_description_content_type':"text/markdown",
    'long_description': '# `proxycurl-py` -  The official Python client for Proxycurl API to scrape and enrich LinkedIn profiles\n\n[TOC]\n\n\n\n## What is Proxycurl?\n\n**Proxycurl** is an enrichment API to fetch fresh data on people and businesses. We are a fully-managed API that sits between your application and raw data so that you can focus on building the application; instead of worrying about building a web-scraping team and processing data at scale.\n\nWith Proxycurl, you can programatically:\n\n- Enrich profiles on people and companies\n- Lookup people and companies\n- Lookup contact information on people and companies\n- Check if an email address is of a disposable nature\n- [And more..](https://nubela.co/proxycurl/docs#explain-it-to-me-like-i-39-m-5)\n\nVisit [Proxycurl\'s website](https://nubela.co/proxycurl) for more details.\n\n\n\n## Before you install\n\nYou should understand that `proxycurl-py` was designed with concurrency as a first class citizen from ground-up. To install `proxycurl-py`, *you have to pick a concurency model*.\n\nWe support the following concurrency models:\n\n* [asyncio](https://docs.python.org/3/library/asyncio.html) - See implementation example [here](https://github.com/nubelaco/proxycurl-linkedin-scraper/blob/main/examples/lib-asyncio.py).\n* [gevent](https://www.gevent.org/) - See implementation example [here](https://github.com/nubelaco/proxycurl-linkedin-scraper/blob/main/examples/lib-gevent.py).\n* [twisted](https://twisted.org/) - See implementation example [here](https://github.com/nubelaco/proxycurl-linkedin-scraper/blob/main/examples/lib-twisted.py).\n\nThe right way to use Proxycurl API is to make API calls concurrently. In fact, making API requests concurrently is the only way to achieve a high rate of throughput. On the default rate limit, you can enrich up to 432,000 profiles per day. See [this blog post](https://nubela.co/blog/how-to-maximize-throughput-on-proxycurl/) for context.\n\n\n\n## Installation and supported Python versions\n\n`proxycurl-py` is [available on PyPi](https://pypi.org/project/proxycurl/). For which you can install into your project with the following command:\n\n```bash\n# install proxycurl-py with asyncio\n$ pip install \'proxycurl[asyncio]\'\n\n# install proxycurl-py with gevent\n$ pip install \'proxycurl[gevent]\'\n\n# install proxycurl-py with twisted\n$ pip install \'proxycurl[twisted]\'\n```\n\n`proxycurl-py` is tested on Python `3.7`, `3.8` and `3.9`.\n\n\n\n## Initializing `proxycurl-py` with an API Key\n\nYou can get an API key by [registering an account](https://nubela.co/proxycurl/auth/register) with Proxycurl. The API Key can be retrieved from the dashboard.\n\nTo use Proxycurl with the API Key:\n\n* You can run your script with  the `PROXYCURL_API_KEY` environment variable set. \n* Or, you can prepend your script with the API key injected into the environment. See `proxycurl/config.py` for an example.\n\n\n\n## Usage with examples\n\nI will be using `proxycurl-py` with the *asyncio* concurrency model to illustrate some examples on what you can do with Proxycurl and how the code will look with this library.\n\nForexamples with other concurrency models such as:\n\n* *gevent*, see `examples/lib-gevent.py`.\n* *twisted*, see `examples/lib-twisted`.\n\n\n\n### Enrich a Person Profile\n\nGiven a *LinkedIn Member Profile URL*, you can get the entire profile back in structured data with Proxycurl\'s [Person Profile API Endpoint](https://nubela.co/proxycurl/docs#people-api-person-profile-endpoint).\n\n```python\nfrom proxycurl.asyncio import Proxycurl, do_bulk\nimport asyncio\nimport csv\n\nproxycurl = Proxycurl()\nperson = asyncio.run(proxycurl.linkedin.person.get(\n    url=\'https://www.linkedin.com/in/williamhgates/\'\n))\nprint(\'Person Result:\', person)\n```\n\n\n\n### Enrich a Company Profile\n\nGiven a *LinkedIn Company Profile URL*, enrich the URL with it\'s full profile with Proxycurl\'s [Company Profile API Endpoint](https://nubela.co/proxycurl/docs#company-api-company-profile-endpoint).\n\n```python\ncompany = asyncio.run(proxycurl.linkedin.company.get(\n    url=\'https://www.linkedin.com/company/tesla-motors\'\n))\nprint(\'Company Result:\', company)\n```\n\n\n\n### Lookup a person\n\nGiven a first name and a company name or domain, lookup a person with Proxycurl\'s [Person Lookup API Endpoint](https://nubela.co/proxycurl/docs#people-api-person-lookup-endpoint).\n\n```python\nlookup_results = asyncio.run(proxycurl.linkedin.person.resolve(first_name="bill", last_name="gates", company_domain="microsoft"))\nprint(\'Person Lookup Result:\', lookup_results)\n```\n\n\n\n### Lookup a company\n\nGiven a company name or a domain, lookup a company with Proxycurl\'s [Company Lookup API Endpoint](https://nubela.co/proxycurl/docs#company-api-company-lookup-endpoint).\n\n```python\ncompany_lookup_results = asyncio.run(proxycurl.linkedin.company.resolve(company_name="microsoft", company_domain="microsoft.com"))\nprint(\'Company Lookup Result:\', company_lookup_results)\n```\n\n\n\n### Lookup a LinkedIn Profile URL from a work email address\n\nGiven a work email address, lookup a LinkedIn Profile URL with Proxycurl\'s [Reverse Work Email Lookup Endpoint](https://nubela.co/proxycurl/docs#contact-api-reverse-work-email-lookup-endpoint).\n\n```python\nlookup_results = asyncio.run(proxycurl.linkedin.person.resolve_by_email(work_email="anthony.tan@grab.com"))\nprint(\'Reverse Work Email Lookup Result:\', lookup_results)\n```\n\n\n\n### Enrich LinkedIn member profiles in bulk (from a CSV)\n\nGiven a CSV file with a list of LinkedIn member profile URLs, you can enrich the list in the following manner:\n\n```python\n# PROCESS BULK WITH CSV\nbulk_linkedin_person_data = []\nwith open(\'sample.csv\', \'r\') as file:\n    reader = csv.reader(file)\n    next(reader, None)\n    for row in reader:\n        bulk_linkedin_person_data.append(\n            (proxycurl.linkedin.person.get, {\'url\': row[0]})\n        )\nresults = asyncio.run(do_bulk(bulk_linkedin_person_data))\n\nprint(\'Bulk:\', results)\n```\n\n\n\n### More *asyncio* examples\n\nMore *asyncio* examples can be found at `examples/lib-asyncio.py`\n\n\n\n## Rate limit and error handling\n\nThere is no need for you to handle rate limits (`429` HTTP status error). The [library handles rate limits automatically with exponential backoff](https://github.com/nubelaco/proxycurl-linkedin-scraper/blob/main/proxycurl/asyncio/base.py#L109).\n\nHowever, there is a need for you to handle other error codes. Errors will be returned in the form of `ProxycurlException`. The [list of possible errors](https://nubela.co/proxycurl/docs#overview-errors) is listed in our API documentation.\n\n\n\n## API Endpoints and their corresponding documentation\n\nHere we list the possible API endpoints and their corresponding library functions. Do refer to each endpoint\'s relevant API documentation to find out the required arguments that needs to be fed into the function.\n\n| Function                                     | Endpoint                                                     | API                                                         |\n| -------------------------------------------- | ------------------------------------------------------------ | ----------------------------------------------------------- |\n| `linkedin.company.employee_count(**kwargs)`  | [Employee Count Endpoint](https://nubela.co/proxycurl/docs#company-api-employee-count-endpoint) | [Company API](https://nubela.co/proxycurl/docs#company-api) |\n| `linkedin.company.resolve(**kwargs)`         | [Company Lookup Endpoint](https://nubela.co/proxycurl/docs#company-api-company-profile-endpoint) | [Company API](https://nubela.co/proxycurl/docs#company-api) |\n| `linkedin.company.employee_list(**kwargs)`   | [Employee Listing Endpoint](https://nubela.co/proxycurl/docs#company-api-employee-listing-endpoint) | [Company API](https://nubela.co/proxycurl/docs#company-api) |\n| `linkedin.company.get(**kwargs)`             | [Company Profile Endpoint](https://nubela.co/proxycurl/docs#company-api-company-profile-endpoint) | [Company API](https://nubela.co/proxycurl/docs#company-api) |\n| `linkedin.person.resolve_by_email(**kwargs)` | [Reverse Work Email Lookup Endpoint](https://nubela.co/proxycurl/docs#contact-api-reverse-work-email-lookup-endpoint) | [Contact API](https://nubela.co/proxycurl/docs#contact-api) |\n| `linkedin.person.lookup_email(**kwargs)`     | [Work Email Lookup Endpoint](https://nubela.co/proxycurl/docs#contact-api-work-email-lookup-endpoint) | [Contact API](https://nubela.co/proxycurl/docs#contact-api) |\n| `linkedin.person.personal_contact(**kwargs)` | [Personal Contact Number Lookup Endpoint](https://nubela.co/proxycurl/docs#contact-api-personal-contact-number-lookup-endpoint) | [Contact API](https://nubela.co/proxycurl/docs#contact-api) |\n| `linkedin.person.personal_email(**kwargs)`   | [Personal Email Lookup Endpoint](https://nubela.co/proxycurl/docs#contact-api-personal-email-lookup-endpoint) | [Contact API](https://nubela.co/proxycurl/docs#contact-api) |\n| `linkedin.disposable_email(**kwargs)`        | [Disposable Email Address Check Endpoint](https://nubela.co/proxycurl/docs#contact-api-disposable-email-address-check-endpoint) | [Contact API](https://nubela.co/proxycurl/docs#contact-api)\n| `linkedin.company.find_job(**kwargs)`        | [Job Listings Endpoint](https://nubela.co/proxycurl/docs#jobs-api-jobs-listing-endpoint) | [Jobs API](https://nubela.co/proxycurl/docs#jobs-api)       |\n| `linkedin.job.get(**kwargs)`                 | [Jobs Profile Endpoint](https://nubela.co/proxycurl/docs#jobs-api-job-profile-endpoint) | [Jobs API](https://nubela.co/proxycurl/docs#jobs-api)       |\n| `linkedin.person.resolve(**kwargs)`          | [Person Lookup Endpoint](https://nubela.co/proxycurl/docs#people-api-person-lookup-endpoint) | [People API](https://nubela.co/proxycurl/docs#people-api)   |\n| `linkedin.company.role_lookup(**kwargs)`     | [Role Lookup Endpoint](https://nubela.co/proxycurl/docs#people-api-role-lookup-endpoint) | [People API](https://nubela.co/proxycurl/docs#people-api) |\n| `linkedin.person.get(**kwargs)`              | [Person Profile Endpoint](https://nubela.co/proxycurl/docs#people-api-person-profile-endpoint) | [People API](https://nubela.co/proxycurl/docs#people-api)   |\n| `linkedin.school.get(**kwargs)`              | [School Profile Endpoint](https://nubela.co/proxycurl/docs#school-api-school-profile-endpoint) | [School API](https://nubela.co/proxycurl/docs#school-api)   |\n| `linkedin.company.reveal`                    | [Reveal Endpoint](https://nubela.co/proxycurl/docs#reveal-api-reveal-endpoint) | [Reveal API](https://nubela.co/proxycurl/docs#reveal-api) |\n| `get_balance(**kwargs)`                      | [View Credit Balance Endpoint](https://nubela.co/proxycurl/docs#meta-api-view-credit-balance-endpoint) | [Meta API](https://nubela.co/proxycurl/docs#meta-api)    |',
    'author': 'Nubela',
    'author_email': 'tech@nubela.co',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'package_dir': {'': '.'},
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
