"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../../lib");
let stack;
let clusterRole;
let serviceRole;
let autoScalingRole;
beforeEach(() => {
    var _a;
    // GIVEN
    stack = new cdk.Stack();
    clusterRole = new iam.Role(stack, 'ClusterRole', {
        assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
    });
    serviceRole = new iam.Role(stack, 'ServiceRole', {
        assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
    });
    autoScalingRole = new iam.Role(stack, 'AutoScalingRole', {
        assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
    });
    (_a = autoScalingRole.assumeRolePolicy) === null || _a === void 0 ? void 0 : _a.addStatements(new iam.PolicyStatement({
        effect: iam.Effect.ALLOW,
        principals: [
            new iam.ServicePrincipal('application-autoscaling.amazonaws.com'),
        ],
        actions: [
            'sts:AssumeRole',
        ],
    }));
});
test('Create Cluster with FIRE_AND_FORGET integrationPattern', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
        },
    });
});
test('Create Cluster with SYNC integrationPattern', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
        },
    });
});
test('Create Cluster with clusterConfiguration Name from payload', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: sfn.TaskInput.fromDataAt('$.ClusterName').value,
        serviceRole,
        autoScalingRole,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            'Name.$': '$.ClusterName',
            'Instances': {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            'VisibleToAllUsers': true,
            'JobFlowRole': {
                Ref: 'ClusterRoleD9CA7471',
            },
            'ServiceRole': {
                Ref: 'ServiceRole4288B192',
            },
            'AutoScalingRole': {
                Ref: 'AutoScalingRole015ADA0A',
            },
        },
    });
});
test('Create Cluster with Tags', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        tags: {
            key: 'value',
        },
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
            Tags: [{
                    Key: 'key',
                    Value: 'value',
                }],
        },
    });
});
test('Create Cluster with Applications', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        applications: [
            { name: 'Hive', version: '0.0' },
            { name: 'Spark', version: '0.0' },
        ],
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
            Applications: [
                { Name: 'Hive', Version: '0.0' },
                { Name: 'Spark', Version: '0.0' },
            ],
        },
    });
});
test('Create Cluster with Bootstrap Actions', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        bootstrapActions: [{
                name: 'Bootstrap',
                scriptBootstrapAction: {
                    path: 's3://null',
                    args: ['Arg'],
                },
            }],
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
            BootstrapActions: [{
                    Name: 'Bootstrap',
                    ScriptBootstrapAction: {
                        Path: 's3://null',
                        Args: ['Arg'],
                    },
                }],
        },
    });
});
test('Create Cluster with Configurations', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        configurations: [{
                classification: 'classification',
                properties: {
                    Key: 'Value',
                },
            }],
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
            Configurations: [{
                    Classification: 'classification',
                    Properties: {
                        Key: 'Value',
                    },
                }],
        },
    });
});
test('Create Cluster with KerberosAttributes', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        kerberosAttributes: {
            realm: 'realm',
            adDomainJoinPassword: 'password1',
            adDomainJoinUser: 'user',
            crossRealmTrustPrincipalPassword: 'password2',
            kdcAdminPassword: 'password3',
        },
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
            KerberosAttributes: {
                Realm: 'realm',
                ADDomainJoinPassword: 'password1',
                ADDomainJoinUser: 'user',
                CrossRealmTrustPrincipalPassword: 'password2',
                KdcAdminPassword: 'password3',
            },
        },
    });
});
test('Create Cluster without Roles', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        name: 'Cluster',
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'TaskInstanceRoleB72072BF',
            },
            ServiceRole: {
                Ref: 'TaskServiceRoleBF55F61E',
            },
            AutoScalingRole: {
                Ref: 'TaskAutoScalingRoleD06F8423',
            },
        },
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Principal: { Service: 'elasticmapreduce.amazonaws.com' },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                },
            ],
        },
    });
    // The stack renders the ec2.amazonaws.com Service principal id with a
    // Join to the URLSuffix
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Principal: {
                        Service: {
                            'Fn::Join': [
                                '',
                                [
                                    'ec2.',
                                    {
                                        Ref: 'AWS::URLSuffix',
                                    },
                                ],
                            ],
                        },
                    },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                },
            ],
        },
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Principal: { Service: 'elasticmapreduce.amazonaws.com' },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                },
                {
                    Principal: { Service: 'application-autoscaling.amazonaws.com' },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                },
            ],
        },
    });
});
test('Create Cluster with Instances configuration', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {
            additionalMasterSecurityGroups: ['MasterGroup'],
            additionalSlaveSecurityGroups: ['SlaveGroup'],
            ec2KeyName: 'Ec2KeyName',
            ec2SubnetId: 'Ec2SubnetId',
            ec2SubnetIds: ['Ec2SubnetId'],
            emrManagedMasterSecurityGroup: 'MasterGroup',
            emrManagedSlaveSecurityGroup: 'SlaveGroup',
            hadoopVersion: 'HadoopVersion',
            instanceCount: 1,
            masterInstanceType: 'MasterInstanceType',
            placement: {
                availabilityZone: 'AvailabilityZone',
                availabilityZones: ['AvailabilityZone'],
            },
            serviceAccessSecurityGroup: 'ServiceAccessGroup',
            slaveInstanceType: 'SlaveInstanceType',
            terminationProtected: true,
        },
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                AdditionalMasterSecurityGroups: ['MasterGroup'],
                AdditionalSlaveSecurityGroups: ['SlaveGroup'],
                Ec2KeyName: 'Ec2KeyName',
                Ec2SubnetId: 'Ec2SubnetId',
                Ec2SubnetIds: ['Ec2SubnetId'],
                EmrManagedMasterSecurityGroup: 'MasterGroup',
                EmrManagedSlaveSecurityGroup: 'SlaveGroup',
                HadoopVersion: 'HadoopVersion',
                InstanceCount: 1,
                KeepJobFlowAliveWhenNoSteps: true,
                MasterInstanceType: 'MasterInstanceType',
                Placement: {
                    AvailabilityZone: 'AvailabilityZone',
                    AvailabilityZones: ['AvailabilityZone'],
                },
                ServiceAccessSecurityGroup: 'ServiceAccessGroup',
                SlaveInstanceType: 'SlaveInstanceType',
                TerminationProtected: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
        },
    });
});
test('Create Cluster with InstanceFleet', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {
            instanceFleets: [{
                    instanceFleetType: lib_1.EmrCreateCluster.InstanceRoleType.MASTER,
                    instanceTypeConfigs: [{
                            bidPrice: '1',
                            bidPriceAsPercentageOfOnDemandPrice: 1,
                            configurations: [{
                                    classification: 'Classification',
                                    properties: {
                                        Key: 'Value',
                                    },
                                }],
                            ebsConfiguration: {
                                ebsBlockDeviceConfigs: [{
                                        volumeSpecification: {
                                            iops: 1,
                                            volumeSize: cdk.Size.gibibytes(1),
                                            volumeType: lib_1.EmrCreateCluster.EbsBlockDeviceVolumeType.STANDARD,
                                        },
                                        volumesPerInstance: 1,
                                    }],
                                ebsOptimized: true,
                            },
                            instanceType: 'm5.xlarge',
                            weightedCapacity: 1,
                        }],
                    launchSpecifications: {
                        spotSpecification: {
                            blockDurationMinutes: 1,
                            timeoutAction: lib_1.EmrCreateCluster.SpotTimeoutAction.TERMINATE_CLUSTER,
                            timeoutDurationMinutes: 1,
                        },
                    },
                    name: 'Master',
                    targetOnDemandCapacity: 1,
                    targetSpotCapacity: 1,
                }],
        },
        clusterRole,
        name: 'Cluster',
        serviceRole,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
                InstanceFleets: [{
                        InstanceFleetType: 'MASTER',
                        InstanceTypeConfigs: [{
                                BidPrice: '1',
                                BidPriceAsPercentageOfOnDemandPrice: 1,
                                Configurations: [{
                                        Classification: 'Classification',
                                        Properties: {
                                            Key: 'Value',
                                        },
                                    }],
                                EbsConfiguration: {
                                    EbsBlockDeviceConfigs: [{
                                            VolumeSpecification: {
                                                Iops: 1,
                                                SizeInGB: 1,
                                                VolumeType: 'standard',
                                            },
                                            VolumesPerInstance: 1,
                                        }],
                                    EbsOptimized: true,
                                },
                                InstanceType: 'm5.xlarge',
                                WeightedCapacity: 1,
                            }],
                        LaunchSpecifications: {
                            SpotSpecification: {
                                BlockDurationMinutes: 1,
                                TimeoutAction: 'TERMINATE_CLUSTER',
                                TimeoutDurationMinutes: 1,
                            },
                        },
                        Name: 'Master',
                        TargetOnDemandCapacity: 1,
                        TargetSpotCapacity: 1,
                    }],
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
        },
    });
});
test('Create Cluster with InstanceGroup', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {
            instanceGroups: [{
                    autoScalingPolicy: {
                        constraints: {
                            maxCapacity: 2,
                            minCapacity: 1,
                        },
                        rules: [{
                                action: {
                                    market: lib_1.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                                    simpleScalingPolicyConfiguration: {
                                        adjustmentType: lib_1.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                                        coolDown: 1,
                                        scalingAdjustment: 1,
                                    },
                                },
                                description: 'Description',
                                name: 'Name',
                                trigger: {
                                    cloudWatchAlarmDefinition: {
                                        comparisonOperator: lib_1.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN,
                                        dimensions: [{
                                                key: 'Key',
                                                value: 'Value',
                                            }],
                                        evaluationPeriods: 1,
                                        metricName: 'Name',
                                        namespace: 'Namespace',
                                        period: cdk.Duration.seconds(300),
                                        statistic: lib_1.EmrCreateCluster.CloudWatchAlarmStatistic.AVERAGE,
                                        threshold: 1,
                                        unit: lib_1.EmrCreateCluster.CloudWatchAlarmUnit.NONE,
                                    },
                                },
                            }, {
                                action: {
                                    market: lib_1.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                                    simpleScalingPolicyConfiguration: {
                                        adjustmentType: lib_1.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                                        coolDown: 1,
                                        scalingAdjustment: 1,
                                    },
                                },
                                description: 'Description',
                                name: 'Name',
                                trigger: {
                                    cloudWatchAlarmDefinition: {
                                        comparisonOperator: lib_1.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN,
                                        dimensions: [{
                                                key: 'Key',
                                                value: 'Value',
                                            }],
                                        evaluationPeriods: 1,
                                        metricName: 'Name',
                                        namespace: 'Namespace',
                                        period: cdk.Duration.seconds(sfn.JsonPath.numberAt('$.CloudWatchPeriod')),
                                        statistic: lib_1.EmrCreateCluster.CloudWatchAlarmStatistic.AVERAGE,
                                        threshold: 1,
                                        unit: lib_1.EmrCreateCluster.CloudWatchAlarmUnit.NONE,
                                    },
                                },
                            }],
                    },
                    bidPrice: '1',
                    configurations: [{
                            classification: 'Classification',
                            properties: {
                                Key: 'Value',
                            },
                        }],
                    ebsConfiguration: {
                        ebsBlockDeviceConfigs: [{
                                volumeSpecification: {
                                    iops: 1,
                                    volumeSize: cdk.Size.gibibytes(1),
                                    volumeType: lib_1.EmrCreateCluster.EbsBlockDeviceVolumeType.STANDARD,
                                },
                                volumesPerInstance: 1,
                            }],
                        ebsOptimized: true,
                    },
                    instanceCount: 1,
                    instanceRole: lib_1.EmrCreateCluster.InstanceRoleType.MASTER,
                    instanceType: 'm5.xlarge',
                    market: lib_1.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                    name: 'Name',
                }],
        },
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
                InstanceGroups: [{
                        AutoScalingPolicy: {
                            Constraints: {
                                MaxCapacity: 2,
                                MinCapacity: 1,
                            },
                            Rules: [{
                                    Action: {
                                        Market: 'ON_DEMAND',
                                        SimpleScalingPolicyConfiguration: {
                                            AdjustmentType: 'CHANGE_IN_CAPACITY',
                                            CoolDown: 1,
                                            ScalingAdjustment: 1,
                                        },
                                    },
                                    Description: 'Description',
                                    Name: 'Name',
                                    Trigger: {
                                        CloudWatchAlarmDefinition: {
                                            ComparisonOperator: 'GREATER_THAN',
                                            Dimensions: [{
                                                    Key: 'Key',
                                                    Value: 'Value',
                                                }],
                                            EvaluationPeriods: 1,
                                            MetricName: 'Name',
                                            Namespace: 'Namespace',
                                            Period: 300,
                                            Statistic: 'AVERAGE',
                                            Threshold: 1,
                                            Unit: 'NONE',
                                        },
                                    },
                                }, {
                                    Action: {
                                        Market: 'ON_DEMAND',
                                        SimpleScalingPolicyConfiguration: {
                                            AdjustmentType: 'CHANGE_IN_CAPACITY',
                                            CoolDown: 1,
                                            ScalingAdjustment: 1,
                                        },
                                    },
                                    Description: 'Description',
                                    Name: 'Name',
                                    Trigger: {
                                        CloudWatchAlarmDefinition: {
                                            'ComparisonOperator': 'GREATER_THAN',
                                            'Dimensions': [{
                                                    Key: 'Key',
                                                    Value: 'Value',
                                                }],
                                            'EvaluationPeriods': 1,
                                            'MetricName': 'Name',
                                            'Namespace': 'Namespace',
                                            'Period.$': '$.CloudWatchPeriod',
                                            'Statistic': 'AVERAGE',
                                            'Threshold': 1,
                                            'Unit': 'NONE',
                                        },
                                    },
                                }],
                        },
                        BidPrice: '1',
                        Configurations: [{
                                Classification: 'Classification',
                                Properties: {
                                    Key: 'Value',
                                },
                            }],
                        EbsConfiguration: {
                            EbsBlockDeviceConfigs: [{
                                    VolumeSpecification: {
                                        Iops: 1,
                                        SizeInGB: 1,
                                        VolumeType: 'standard',
                                    },
                                    VolumesPerInstance: 1,
                                }],
                            EbsOptimized: true,
                        },
                        InstanceCount: 1,
                        InstanceRole: 'MASTER',
                        InstanceType: 'm5.xlarge',
                        Market: 'ON_DEMAND',
                        Name: 'Name',
                    }],
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new lib_1.EmrCreateCluster(stack, 'Task', {
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        });
    }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE,RUN_JOB. Received: WAIT_FOR_TASK_TOKEN/);
});
//# sourceMappingURL=data:application/json;base64,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