"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
let vpc;
let cluster;
beforeEach(() => {
    // GIVEN
    stack = new core_1.Stack();
    vpc = new ec2.Vpc(stack, 'Vpc');
    cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
    cluster.addCapacity('Capacity', {
        instanceType: new ec2.InstanceType('t3.medium'),
    });
});
test('Cannot create a Fargate task with a fargate-incompatible task definition', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        memoryMiB: '512',
        cpu: '256',
        compatibility: ecs.Compatibility.EC2,
    });
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    expect(() => new tasks.RunEcsFargateTask({ cluster, taskDefinition })).toThrowError(/not configured for compatibility with Fargate/);
});
test('Cannot create a Fargate task without a default container', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        memoryMiB: '512',
        cpu: '256',
        compatibility: ecs.Compatibility.FARGATE,
    });
    expect(() => new tasks.RunEcsFargateTask({ cluster, taskDefinition })).toThrowError(/must have at least one essential container/);
});
test('Running a Fargate Task', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        memoryMiB: '512',
        cpu: '256',
        compatibility: ecs.Compatibility.FARGATE,
    });
    const containerDefinition = taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    // WHEN
    const runTask = new sfn.Task(stack, 'RunFargate', {
        task: new tasks.RunEcsFargateTask({
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
            cluster,
            taskDefinition,
            containerOverrides: [
                {
                    containerDefinition,
                    environment: [
                        { name: 'SOME_KEY', value: sfn.JsonPath.stringAt('$.SomeKey') },
                    ],
                },
            ],
        }),
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: runTask,
    });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'FARGATE',
            NetworkConfiguration: {
                AwsvpcConfiguration: {
                    SecurityGroups: [{ 'Fn::GetAtt': ['RunFargateSecurityGroup709740F2', 'GroupId'] }],
                    Subnets: [{ Ref: 'VpcPrivateSubnet1Subnet536B997A' }, { Ref: 'VpcPrivateSubnet2Subnet3788AAA1' }],
                },
            },
            TaskDefinition: { Ref: 'TD49C78F36' },
            Overrides: {
                ContainerOverrides: [
                    {
                        Environment: [
                            {
                                'Name': 'SOME_KEY',
                                'Value.$': '$.SomeKey',
                            },
                        ],
                        Name: 'TheContainer',
                    },
                ],
            },
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'ecs:RunTask',
                    Effect: 'Allow',
                    Resource: { Ref: 'TD49C78F36' },
                },
                {
                    Action: ['ecs:StopTask', 'ecs:DescribeTasks'],
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: [{ 'Fn::GetAtt': ['TDTaskRoleC497AFFC', 'Arn'] }],
                },
                {
                    Action: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                { Ref: 'AWS::Partition' },
                                ':events:',
                                { Ref: 'AWS::Region' },
                                ':',
                                { Ref: 'AWS::AccountId' },
                                ':rule/StepFunctionsGetEventsForECSTaskRule',
                            ],
                        ],
                    },
                },
            ],
        },
    });
});
test('Running an EC2 Task with bridge network', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        compatibility: ecs.Compatibility.EC2,
    });
    const containerDefinition = taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    // WHEN
    const runTask = new sfn.Task(stack, 'Run', {
        task: new tasks.RunEcsEc2Task({
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
            cluster,
            taskDefinition,
            containerOverrides: [
                {
                    containerDefinition,
                    environment: [
                        { name: 'SOME_KEY', value: sfn.JsonPath.stringAt('$.SomeKey') },
                    ],
                },
            ],
        }),
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: runTask,
    });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'EC2',
            TaskDefinition: { Ref: 'TD49C78F36' },
            Overrides: {
                ContainerOverrides: [
                    {
                        Environment: [
                            {
                                'Name': 'SOME_KEY',
                                'Value.$': '$.SomeKey',
                            },
                        ],
                        Name: 'TheContainer',
                    },
                ],
            },
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'ecs:RunTask',
                    Effect: 'Allow',
                    Resource: { Ref: 'TD49C78F36' },
                },
                {
                    Action: ['ecs:StopTask', 'ecs:DescribeTasks'],
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: [{ 'Fn::GetAtt': ['TDTaskRoleC497AFFC', 'Arn'] }],
                },
                {
                    Action: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                { Ref: 'AWS::Partition' },
                                ':events:',
                                { Ref: 'AWS::Region' },
                                ':',
                                { Ref: 'AWS::AccountId' },
                                ':rule/StepFunctionsGetEventsForECSTaskRule',
                            ],
                        ],
                    },
                },
            ],
        },
    });
});
test('Running an EC2 Task with placement strategies', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        compatibility: ecs.Compatibility.EC2,
    });
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    const ec2Task = new tasks.RunEcsEc2Task({
        integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
        cluster,
        taskDefinition,
        placementStrategies: [ecs.PlacementStrategy.spreadAcrossInstances(), ecs.PlacementStrategy.packedByCpu(), ecs.PlacementStrategy.randomly()],
        placementConstraints: [ecs.PlacementConstraint.memberOf('blieptuut')],
    });
    // WHEN
    const runTask = new sfn.Task(stack, 'Run', { task: ec2Task });
    new sfn.StateMachine(stack, 'SM', {
        definition: runTask,
    });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'EC2',
            TaskDefinition: { Ref: 'TD49C78F36' },
            PlacementConstraints: [{ Type: 'memberOf', Expression: 'blieptuut' }],
            PlacementStrategy: [{ Field: 'instanceId', Type: 'spread' }, { Field: 'cpu', Type: 'binpack' }, { Type: 'random' }],
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
});
test('Running an EC2 Task with overridden number values', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        compatibility: ecs.Compatibility.EC2,
    });
    const containerDefinition = taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    const ec2Task = new tasks.RunEcsEc2Task({
        integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
        cluster,
        taskDefinition,
        containerOverrides: [
            {
                containerDefinition,
                command: sfn.JsonPath.listAt('$.TheCommand'),
                cpu: 5,
                memoryLimit: sfn.JsonPath.numberAt('$.MemoryLimit'),
            },
        ],
    });
    // WHEN
    const runTask = new sfn.Task(stack, 'Run', { task: ec2Task });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'EC2',
            TaskDefinition: { Ref: 'TD49C78F36' },
            Overrides: {
                ContainerOverrides: [
                    {
                        'Command.$': '$.TheCommand',
                        'Cpu': 5,
                        'Memory.$': '$.MemoryLimit',
                        'Name': 'TheContainer',
                    },
                ],
            },
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
});
//# sourceMappingURL=data:application/json;base64,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