from io import IOBase
import os


def parse_molecule(inp, fmt=None, component=None):
    """
    Parse the molecule for `data` parameter of molstar viewer.

    Parameters
    ----------
    `inp` — str | file-like object
        The file path to molecule or the file content of molecule.

        If `inp` was set to file path, the format can be automatically determined.
        Otherwise the format has to be specified manually.
    `fmt` — str (optional)
        Format of the input molecule. 
        Supported formats include `cif`, `cifcore`, `pdb`, `pdbqt`, `gro`, `xyz`, `mol`, `sdf`, `mol2` (default: `None`)
    `component` — dict | List[dict] (optional)
        Component to be created in molstar. 
        If is not specified, molstar will use its default settings. (default: `None`)

        Use helper function `create_component` to generate correct data for this parameter.

    Returns
    -------
    `dict`
        The value for the `data` parameter.

    Raises
    ------
    `RuntimeError`
        If the input format is not supported by molstar viewer, raises RuntimeError.
    """
    # provided a filename as input
    if os.path.isfile(inp):
        # if format if not specified, deduce it from filename
        if not fmt:
            name, fmt = os.path.splitext(inp)
        with open(inp, 'r') as f:
            data = f.read()
    else:
        # provided a file-like object as input
        if isinstance(inp, IOBase):
            data = inp.read()
        # provided file content as input
        else:
            data = inp
        if type(data) == bytes: data = data.decode()
    if fmt == None: raise RuntimeError("The format must be specified if you haven't provided a file name.")
    fmt = fmt.strip('.').lower()
    if fmt not in ["cif", "cifcore", "pdb", "pdbqt", "gro", "xyz", "mol", "sdf", "mol2"]:
        raise RuntimeError(f"The input file format \"{fmt}\" is not supported by molstar.")
    if fmt == 'cif': fmt = 'mmcif'
    if fmt == 'cifcore': fmt = 'cifCore'
    d = {
        "type": 'mol',
        "data": data,
        "format": fmt
    }
    if component: d['component'] = component
    return d

def get_cube():
    pass

def get_sphere(center, radius):
    pass

def get_targets(chain, residue=None):
    """
    Select residues from a given chain. If no residue was specified, the entire chain will be selected.

    Parameters
    ----------
    `chain` — str
        Name of the target chain
    `residue` — int | List[int] (optional)
        Residue index of the target residues, started from 0. (default: `None`)

    Returns
    -------
    `dict`
        Selected residues
    """
    if not residue:
        target = {'chain_name': chain}
    else:
        if type(residue) != list: residue = [residue]
        residues = []
        for res in residue:
            if type(res) == int: residues.append(res)
            elif type(res) == str:
                try:
                    num = eval(res)
                    residues.append(num)
                except SyntaxError:
                    if res[-1].isalpha():
                        num = eval(res[:-1])
                        residues.append(num)
        target = {
            'chain_name': chain,
            'residue_numbers': residues
        }
    return target

def create_component(label, targets, representation='cartoon'):
    """
    Generate the component information for selected targets.

    Parameters
    ----------
    `label` — str
        Name of the component
    `targets` — dict | List[dict]
        List of targets, whose value should be generated by helper function `get_targets`
    `representation` — str (optional)
        The default representation for this component (default: `'cartoon'`)

        Optional representations include `label`, `line`, `cartoon`,
        `backbone`, `ball-and-stick`, `carbohydrate`,
        `ellipsoid`, `gaussian-surface`, `gaussian-volume`,
        `molecular-surface`, `orientation`, `point`,
        `putty` and `spacefill`

    Returns
    -------
    `dict`
        Component information that can be passed to function `parse_molecule`

    Raises
    ------
    `RuntimeError`
        If specified an invalid representation, raises `RuntimeError`
    """
    r = representation.lower()
    if r not in ["label", "line", "cartoon",
                 "backbone", "ball-and-stick", "carbohydrate",
                 "ellipsoid", "gaussian-surface", "gaussian-volume",
                 "molecular-surface", "orientation", "point",
                 "putty", "spacefill"]: raise RuntimeError(f"Invalid representation type \"{representation}\"!")
    if type(targets) != list: targets = [targets]
    return {
        'label': label,
        'targets': targets,
        'representation': r
    }

def get_selection(targets, select=True, add=False):
    """
    Select specific targets in the molstar viewer.

    Parameters
    ----------
    `targets` — dict | List[dict]
        List of targets, whose value should be generated by helper function `get_targets`
    `select` — bool (optional)
        True for 'select' mode and False for 'hover' mode. (default: `True`)

        If set to True, the targets will be "selected" in the viewer.
        Otherwise they will be highlighted but not selected, as you hover on the structure.
    `add` — bool (optional)
        If set to False, the viewer will clear the selections in corresponding mode before adding new selections.
        Otherwise the new selections will be added to existed ones. (default: `False`)

    Returns
    -------
    `dict`
        Selection data for callbacks.
    """
    if select: mode = 'select'
    else: mode = 'hover'

    if add: modifier = 'add'
    else: modifier = 'set'

    return {
        'targets': targets,
        'mode': mode,
        'modifier': modifier
    }

def get_focus(targets, analyse=False):
    """
    Let the camera focus on the specified targets. If `analyse` were set to true, non-covalent interactions within 5 angstroms will be analyzed.

    Parameters
    ----------
    `targets` — dict | List[dict]
        List of targets, whose value should be generated by helper function `get_targets`
    `analyse` — bool (optional)
        Whether to analyse the non-covalent interactions of targets to its surroundings within 5 angstroms  (default: `False`)

    Returns
    -------
    `dict`
        Focus data for callbacks
    """
    return {
        'targets': targets,
        'analyse': analyse
    }
