"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginRequestQueryStringBehavior = exports.OriginRequestHeaderBehavior = exports.OriginRequestCookieBehavior = exports.OriginRequestPolicy = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
/**
 * A Origin Request Policy configuration.
 *
 * @resource AWS::CloudFront::OriginRequestPolicy
 */
class OriginRequestPolicy extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.originRequestPolicyName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cloudfront_OriginRequestPolicyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, OriginRequestPolicy);
            }
            throw error;
        }
        const originRequestPolicyName = props.originRequestPolicyName ?? core_1.Names.uniqueId(this);
        if (!core_1.Token.isUnresolved(originRequestPolicyName) && !originRequestPolicyName.match(/^[\w-]+$/i)) {
            throw new Error(`'originRequestPolicyName' can only include '-', '_', and alphanumeric characters, got: '${props.originRequestPolicyName}'`);
        }
        const cookies = props.cookieBehavior ?? OriginRequestCookieBehavior.none();
        const headers = props.headerBehavior ?? OriginRequestHeaderBehavior.none();
        const queryStrings = props.queryStringBehavior ?? OriginRequestQueryStringBehavior.none();
        const resource = new cloudfront_generated_1.CfnOriginRequestPolicy(this, 'Resource', {
            originRequestPolicyConfig: {
                name: originRequestPolicyName,
                comment: props.comment,
                cookiesConfig: {
                    cookieBehavior: cookies.behavior,
                    cookies: cookies.cookies,
                },
                headersConfig: {
                    headerBehavior: headers.behavior,
                    headers: headers.headers,
                },
                queryStringsConfig: {
                    queryStringBehavior: queryStrings.behavior,
                    queryStrings: queryStrings.queryStrings,
                },
            },
        });
        this.originRequestPolicyId = resource.ref;
    }
    /** Imports a Origin Request Policy from its id. */
    static fromOriginRequestPolicyId(scope, id, originRequestPolicyId) {
        return new class extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.originRequestPolicyId = originRequestPolicyId;
            }
        }(scope, id);
    }
    /** Use an existing managed origin request policy. */
    static fromManagedOriginRequestPolicy(managedOriginRequestPolicyId) {
        return new class {
            constructor() {
                this.originRequestPolicyId = managedOriginRequestPolicyId;
            }
        }();
    }
}
exports.OriginRequestPolicy = OriginRequestPolicy;
_a = JSII_RTTI_SYMBOL_1;
OriginRequestPolicy[_a] = { fqn: "@aws-cdk/aws-cloudfront.OriginRequestPolicy", version: "1.191.0" };
/** This policy includes only the User-Agent and Referer headers. It doesn’t include any query strings or cookies. */
OriginRequestPolicy.USER_AGENT_REFERER_HEADERS = OriginRequestPolicy.fromManagedOriginRequestPolicy('acba4595-bd28-49b8-b9fe-13317c0390fa');
/** This policy includes the header that enables cross-origin resource sharing (CORS) requests when the origin is a custom origin. */
OriginRequestPolicy.CORS_CUSTOM_ORIGIN = OriginRequestPolicy.fromManagedOriginRequestPolicy('59781a5b-3903-41f3-afcb-af62929ccde1');
/** This policy includes the headers that enable cross-origin resource sharing (CORS) requests when the origin is an Amazon S3 bucket. */
OriginRequestPolicy.CORS_S3_ORIGIN = OriginRequestPolicy.fromManagedOriginRequestPolicy('88a5eaf4-2fd4-4709-b370-b4c650ea3fcf');
/** This policy includes all values (query strings, headers, and cookies) in the viewer request. */
OriginRequestPolicy.ALL_VIEWER = OriginRequestPolicy.fromManagedOriginRequestPolicy('216adef6-5c7f-47e4-b989-5492eafa07d3');
/** This policy is designed for use with an origin that is an AWS Elemental MediaTailor endpoint. */
OriginRequestPolicy.ELEMENTAL_MEDIA_TAILOR = OriginRequestPolicy.fromManagedOriginRequestPolicy('775133bc-15f2-49f9-abea-afb2e0bf67d2');
/**
 * Determines whether any cookies in viewer requests (and if so, which cookies)
 * are included in requests that CloudFront sends to the origin.
 */
class OriginRequestCookieBehavior {
    constructor(behavior, cookies) {
        this.behavior = behavior;
        this.cookies = cookies;
    }
    /**
     * Cookies in viewer requests are not included in requests that CloudFront sends to the origin.
     * Any cookies that are listed in a CachePolicy are still included in origin requests.
     */
    static none() { return new OriginRequestCookieBehavior('none'); }
    /** All cookies in viewer requests are included in requests that CloudFront sends to the origin. */
    static all() { return new OriginRequestCookieBehavior('all'); }
    /** Only the provided `cookies` are included in requests that CloudFront sends to the origin. */
    static allowList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to allow must be provided');
        }
        return new OriginRequestCookieBehavior('whitelist', cookies);
    }
}
exports.OriginRequestCookieBehavior = OriginRequestCookieBehavior;
_b = JSII_RTTI_SYMBOL_1;
OriginRequestCookieBehavior[_b] = { fqn: "@aws-cdk/aws-cloudfront.OriginRequestCookieBehavior", version: "1.191.0" };
/**
 * Determines whether any HTTP headers (and if so, which headers) are included in requests that CloudFront sends to the origin.
 */
class OriginRequestHeaderBehavior {
    constructor(behavior, headers) {
        this.behavior = behavior;
        this.headers = headers;
    }
    /**
     * HTTP headers are not included in requests that CloudFront sends to the origin.
     * Any headers that are listed in a CachePolicy are still included in origin requests.
     */
    static none() { return new OriginRequestHeaderBehavior('none'); }
    /**
     * All HTTP headers in viewer requests are included in requests that CloudFront sends to the origin.
     * Additionally, any additional CloudFront headers provided are included; the additional headers are added by CloudFront.
     * @see https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-cloudfront-headers.html
     */
    static all(...cloudfrontHeaders) {
        if (cloudfrontHeaders.length > 0) {
            if (!cloudfrontHeaders.every(header => header.startsWith('CloudFront-'))) {
                throw new Error('additional CloudFront headers passed to `OriginRequestHeaderBehavior.all()` must begin with \'CloudFront-\'');
            }
            return new OriginRequestHeaderBehavior('allViewerAndWhitelistCloudFront', cloudfrontHeaders);
        }
        else {
            return new OriginRequestHeaderBehavior('allViewer');
        }
    }
    /** Listed headers are included in requests that CloudFront sends to the origin. */
    static allowList(...headers) {
        if (headers.length === 0) {
            throw new Error('At least one header to allow must be provided');
        }
        if (headers.map(header => header.toLowerCase()).some(header => ['authorization', 'accept-encoding'].includes(header))) {
            throw new Error('you cannot pass `Authorization` or `Accept-Encoding` as header values; use a CachePolicy to forward these headers instead');
        }
        return new OriginRequestHeaderBehavior('whitelist', headers);
    }
}
exports.OriginRequestHeaderBehavior = OriginRequestHeaderBehavior;
_c = JSII_RTTI_SYMBOL_1;
OriginRequestHeaderBehavior[_c] = { fqn: "@aws-cdk/aws-cloudfront.OriginRequestHeaderBehavior", version: "1.191.0" };
/**
 * Determines whether any URL query strings in viewer requests (and if so, which query strings)
 * are included in requests that CloudFront sends to the origin.
 */
class OriginRequestQueryStringBehavior {
    constructor(behavior, queryStrings) {
        this.behavior = behavior;
        this.queryStrings = queryStrings;
    }
    /**
     * Query strings in viewer requests are not included in requests that CloudFront sends to the origin.
     * Any query strings that are listed in a CachePolicy are still included in origin requests.
     */
    static none() { return new OriginRequestQueryStringBehavior('none'); }
    /** All query strings in viewer requests are included in requests that CloudFront sends to the origin. */
    static all() { return new OriginRequestQueryStringBehavior('all'); }
    /** Only the provided `queryStrings` are included in requests that CloudFront sends to the origin. */
    static allowList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to allow must be provided');
        }
        return new OriginRequestQueryStringBehavior('whitelist', queryStrings);
    }
}
exports.OriginRequestQueryStringBehavior = OriginRequestQueryStringBehavior;
_d = JSII_RTTI_SYMBOL_1;
OriginRequestQueryStringBehavior[_d] = { fqn: "@aws-cdk/aws-cloudfront.OriginRequestQueryStringBehavior", version: "1.191.0" };
//# sourceMappingURL=data:application/json;base64,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