# -*- coding: utf-8 -*-
# Copyright (c) 2020 Stephen Wasilewski, HSLU and EPFL
# =======================================================================
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.
# =======================================================================
import os.path
import re
import sys

import numpy as np
from scipy.interpolate import RegularGridInterpolator
from scipy.spatial import cKDTree
from clasp.script_tools import try_mkdir, pipeline

from raytraverse import io
from raytraverse.mapper import ViewMapper
from raytraverse.renderer import SpRenderer
from raytraverse.sampler import draw
from raytraverse.sampler.basesampler import BaseSampler, filterdict
from raytraverse.evaluate import SamplingMetrics
from raytraverse.lightfield import LightPlaneKD
from raytraverse.formatter import RadianceFormatter as Fmt


class SamplerArea(BaseSampler):
    """wavelet based area sampling class

    Parameters
    ----------
    scene: raytraverse.scene.Scene
        scene class containing geometry and formatter compatible with engine
    engine: raytraverse.sampler.SamplerPt
        point sampler
    accuracy: float, optional
        parameter to set threshold at sampling level relative to final level
        threshold (smaller number will increase sampling, default is 1.0)
    nlev: int, optional
        number of levels to sample
    jitter: bool, optional
        jitter samples
    edgemode: {‘reflect’, ‘constant’, ‘nearest’, ‘mirror’, ‘wrap’}, optional
        default: 'constant', if 'constant' value is set to -self.t1, so edge is
        always seen as detail. Internal edges (resulting from PlanMapper
        borders) will behave like 'nearest' for all options except 'constant'
    metricclass: raytraverse.evaluate.BaseMetricSet, optional
        the metric calculator used to compute weights
    metricset: iterable, optional
        list of metrics (must be recognized by metricclass. metrics containing
        "lum" will be normalized to 0-1)
    metricfunc: func, optional
        takes detail array as an argument, shape: (len(metricset),N, M) and an
        axis=0 keyword argument, returns shape (N, M). could be np.max, np.sum
        np.average or us custom function following the same pattern.
    """

    #: initial sampling threshold coefficient
    t0 = .1
    #: final sampling threshold coefficient
    t1 = .9
    #: upper bound for drawing from pdf
    ub = 100

    def __init__(self, scene, engine, accuracy=1.0, nlev=3, jitter=True,
                 edgemode='constant', metricclass=SamplingMetrics,
                 metricset=('avglum', 'loggcr', 'xpeak', 'ypeak'),
                 metricfunc=np.max, **kwargs):
        super().__init__(scene, engine, accuracy, engine.stype, **kwargs)
        if "sun" in self.stype:
            self._gcrnorm = 8
        else:
            self._gcrnorm = 2
        self.engine._slevel = self._slevel + 1
        self.nlev = nlev
        self.jitter = jitter
        #: raytraverse.mapper.PlanMapper
        self._mapper = None
        self._name = None
        modes = ('reflect', 'constant', 'nearest', 'mirror', 'wrap')
        if edgemode not in modes:
            raise ValueError(f"edgemode={edgemode} not a valid option"
                             " must be one of: {modes}")
        self._specguide = None
        self._edgemode = edgemode
        self._mask = slice(None)
        self._candidates = None
        self._plotpchild = False
        self.slices = []
        #: raytraverse.evaluate.BaseMetricSet
        self.metricclass = metricclass
        #: iterable
        self.metricset = metricset
        #: numpy func takes weights and axis=0 argument to reduce metric set
        self._metricfunc = metricfunc
        #: int:
        self.features = len(metricset)

    def sampling_scheme(self, mapper):
        """calculate sampling scheme"""
        return np.array([mapper.shape(i) for i in range(self.nlev)])

    def run(self, mapper, name=None, specguide=None, plotp=False,
            find_reflections=False, **kwargs):
        """adapively sample an area defined by mapper

        Parameters
        ----------
        mapper: raytraverse.mapper.PlanMapper
            the pointset to build/run
        name: str, optional
        specguide: raytraverse.lightfield.LightPlaneKD
            sky source lightfield to use as specular guide for sampling (used by
            engine of type raytraverse.sampler.SunSamplerPt)
        plotp: bool, optional
            plot weights, detail and vectors for each level
        find_reflections: bool, optional
            write file for zone with potential reflection normals
        kwargs:
            passed to self.run()

        Returns
        -------
        raytraverse.lightlplane.LightPlaneKD
        """
        if name is None:
            name = mapper.name
        try_mkdir(f"{self.scene.outdir}/{name}")
        self._mapper = mapper
        self._name = name
        reflf = f"{self.scene.outdir}/{name}/reflection_normals.txt"
        if specguide is True:
            self._specguide = reflf
        else:
            self._specguide = specguide
        levels = self.sampling_scheme(mapper)
        plotpthis = plotp and len(levels) > 1
        self._plotpchild = plotp and not plotpthis
        super().run(mapper, name, levels, plotp=plotpthis, **kwargs)
        if find_reflections:
            refl = self.reflection_search(self.vecs)
            if refl.size > 0:
                np.savetxt(reflf, refl)
        return LightPlaneKD(self.scene, self.vecs, self._mapper, self.stype)

    def reflection_search(self, vecs, res=5):
        # plastic reflections do not work in a frozen octree, so need to try
        # and recompile.
        octf = f"{self.scene.outdir}/scene_reflections.oct"
        if not os.path.isfile(octf):
            header = pipeline([f"getinfo {self.engine.engine.scene}"])
            oconvf = re.findall(r"\s*oconv (.+)", header)[0]
            files = [i for i in oconvf.split() if re.match(r".+\.rad", i)]
            hasoct = [i for i in oconvf.split() if re.match(r".+\.oct", i)]
            if len(hasoct) == 0 and np.all([os.path.isfile(i) for i in files]):
                skyf = f"{self.scene.outdir}/reflections_sky.rad"
                f = open(skyf, 'w')
                f.write(Fmt.get_skydef((1, 1, 1), ground=False))
                f.close()
                pipeline([f"oconv -w {' '.join(files)} {skyf}"], outfile=octf,
                         writemode='wb')
            else:
                print(f"Warning, scene made from frozen octree or source scene "
                      f"files can no longer be located, reflection search will"
                      f"miss specular plastic", file=sys.stderr)
                octf = self.engine.engine.scene
        reflengine = SpRenderer("-ab 0 -w- -lr 1 -ss 0 -st .001 -otndM -h",
                                octf)
        vm = ViewMapper()
        side = 2**res
        uv = np.stack(np.unravel_index(np.arange(side*side*2),
                                       (2*side, side))).T/side
        uv += np.random.default_rng().random(uv.shape) * (.5 / side)
        xyz = vm.uv2xyz(uv)
        pvecs = np.concatenate(np.broadcast_arrays(vecs[:, None], xyz[None, :]), 2)
        a = reflengine(pvecs.reshape(-1, 6))
        # count tabs to get level
        level = np.array([len(i) - len(i.lstrip()) for i in a.splitlines(False)])
        # normal, direction, modifier
        a = np.array(a.split()).reshape(-1, 7)
        mod = a[:, -1]
        a = a[:, 0:6].astype(float)
        sky = np.argwhere(mod == "skyglow").ravel()
        skyl = level[sky]
        leva, levs = np.broadcast_arrays(level, skyl[:, None])
        idxa, idxs = np.broadcast_arrays(np.arange(len(level)), sky[:, None])
        # filter indices before sky index in each row
        leva = np.where(idxa - idxs <= 0, -3, leva)
        # filter indices not one level up from sky
        leva = np.where(leva != (levs - 1), -2, leva)
        # this returns the first value, our candidate reflection
        candidate = np.argmax(leva, 1)
        # check if this is a reflection
        acos = np.einsum("ij,ij->i", a[candidate, 3:], a[sky, 3:])
        normals = a[candidate, 0:3][acos < .99]
        # find unique
        return np.array(list(set(zip(*normals.T))))

    def repeat(self, guide, stype):
        """repeat the sampling of a guide LightPlane (to match all rays)

        Parameters
        ----------
        guide: LightPlaneKD
        stype: str, optional
            alternate stype name for samplerpt. raises a ValueError if it
            matches the guide.
        Returns
        -------

        """
        self._mapper = guide.pm
        self._name = self._mapper.name
        if stype == guide.src:
            raise ValueError("stype cannot match guide.src, as it would "
                             "overwrite data")
        ostype = self.stype
        self.stype = stype

        self.vecs = None
        self.lum = []

        gvecs = guide.vecs
        self._dump_vecs(gvecs)
        pbar = self.scene.progress_bar(self, list(enumerate(gvecs)),
                                       level=self._slevel, message="resampling")
        for posidx, point in pbar:
            gpt = guide.data[posidx]
            self.engine.repeat(gpt, stype)
        lp = LightPlaneKD(self.scene, self.vecs, self._mapper, self.stype)
        self.stype = ostype
        return lp

    def draw(self, level):
        """draw samples based on detail calculated from weights

        Returns
        -------
        pdraws: np.array
            index array of flattened samples chosen to sample at next level
        p: np.array
            computed probabilities
        """
        dres = self.levels[level]
        # sample all if weights is not set or all even
        if level == 0 and np.var(self.weights) < 1e-9:
            pdraws = np.arange(int(np.prod(dres)))[self._mask]
            p = np.ones(dres).ravel()
            p[np.logical_not(self._mask)] = 0
        else:
            nweights = self._normed_weights(mask=self._edgemode == 'constant')
            p = draw.get_detail(nweights, *filterdict[self.detailfunc],
                                mode=self._edgemode, cval=-self.t1)
            p = self._metricfunc(p.reshape(self.weights.shape), axis=0)
            # zero out cells of previous samples
            if self.vecs is not None:
                pxy = self._mapper.ray2pixel(self.vecs, self.weights.shape[1:])
                x = self.weights.shape[1] - 1 - pxy[:, 0]
                y = pxy[:, 1]
                p[x, y] = 0
            # zero out oob
            p = p.ravel()
            p[np.logical_not(self._mask)] = 0
            # draw on pdf
            pdraws = draw.from_pdf(p, self._threshold(level),
                                   lb=self.lb, ub=self.ub)
        return pdraws, p

    def sample_to_uv(self, pdraws, shape):
        """generate samples vectors from flat draw indices

        Parameters
        ----------
        pdraws: np.array
            flat index positions of samples to generate
        shape: tuple
            shape of level samples

        Returns
        -------
        si: np.array
            index array of draws matching samps.shape
        vecs: np.array
            sample vectors
        """
        if len(pdraws) == 0:
            return np.empty(0), np.empty(0)
        si = np.stack(np.unravel_index(pdraws, shape))
        return si, self._candidates[pdraws]

    def sample(self, vecs):
        """call rendering engine to sample rays

        Parameters
        ----------
        vecs: np.array
            sample vectors (subclasses can choose which to use)

        Returns
        -------
        lum: np.array
            array of shape (N,) to update weights
        """
        self._dump_vecs(vecs)
        idx = self.slices[-1].indices(self.slices[-1].stop)
        lums = []
        lpargs = dict(parent=self._name)
        kwargs = dict(metricset=self.metricset, lmin=1e-8)
        specguide = None
        sgpt = None
        if hasattr(self.engine, "_load_specguide"):
            specguide = self._specguide
        level_desc = f"Level {len(self.slices)} of {len(self.levels)}"
        if self.nlev == 1 and len(vecs) == 1:
            logpoint = 'err'
        else:
            logpoint = None
        if logpoint is not None:
            self.engine._slevel -= 1
            pbar = list(zip(range(*idx), vecs))
        elif self._slevel > 0:
            pbar = list(zip(range(*idx), vecs))
        else:
            pbar = self.scene.progress_bar(self, list(zip(range(*idx), vecs)),
                                           level=self._slevel, message=level_desc)
        for posidx, point in pbar:
            if specguide is not None:
                sgpt = self._load_specguide(point)
            lp = self.engine.run(point, posidx, specguide=sgpt, lpargs=lpargs,
                                 log=logpoint, plotp=self._plotpchild, pfish=False)
            vol = lp.evaluate(1, includeviews=True)
            metric = self.metricclass(*vol, lp.vm, gcrnorm=self._gcrnorm,
                                      **kwargs)
            lums.append(metric())
        if logpoint is not None:
            self.engine._slevel += 1
        if len(self.lum) == 0:
            self.lum = np.array(lums)
        else:
            self.lum = np.concatenate((self.lum, lums), 0)
        return np.array(lums)

    def _load_specguide(self, point):
        """find the 3 nearest lightpoints in the specular sampling guide"""
        if hasattr(self._specguide, "kd"):
            d, i = self._specguide.kd.query(point, 3)
            idxs = i[d <= self._mapper.ptres * np.sqrt(2)]
            specguide = [self._specguide.data[j] for j in idxs]
            if len(specguide) > 0:
                return specguide
            else:
                return None
        else:
            return self._specguide

    def _update_weights(self, si, lum):
        """only used by _plot_weights, weights are recomputed from spatial
        query"""
        wv = np.moveaxis(self.weights, 0, 2)
        wv[tuple(si)] = lum

    def _lift_weights(self, i):
        """because areas can have an arbitrary border, upsamppling is not
        effective. If we mask the weights, than the UV edges (which are padded)
        will be treated differently from the inside edges. by remapping each
        level with the nearest sample value and then masking after calculating
        the detail we avoid these issues."""
        self._candidates = self._mapper.point_grid_uv(jitter=self.jitter,
                                                      level=i, masked=False,
                                                      snap=self.nlev-1)
        self._mask = self._mapper.in_view_uv(self._candidates, False)
        if self.vecs is not None:
            wvecs = self._mapper.uv2xyz(self._candidates)
            d, idx = cKDTree(self.vecs).query(wvecs)
            weights = self.lum[idx].reshape(*self.levels[i], self.features)
            self.weights = np.moveaxis(weights, 2, 0)

    def _normed_weights(self, mask=False):
        normi = [i for i, j in enumerate(self.metricset) if 'lum' in j]
        nweights = np.copy(self.weights)
        for i in normi:
            nmin = np.min(self.weights[i])
            norm = np.max(self.weights[i]) - nmin
            if norm > 0:
                nweights[i] = (nweights[i] - nmin)/norm
        if mask:
            # when using a maskedplanmapper, we don't want to mask excluded
            # points before calculating detail
            mk = self._mapper.in_view_uv(self._candidates, False, usemask=False)
            for nw in nweights:
                nw.flat[np.logical_not(mk)] = -self.t1
        return nweights

    def _dump_vecs(self, vecs):
        if self.vecs is None:
            self.vecs = vecs
            v0 = 0
        else:
            self.vecs = np.concatenate((self.vecs, vecs))
            v0 = self.slices[-1].stop
        self.slices.append(slice(v0, v0 + len(vecs)))
        vfile = (f"{self.scene.outdir}/{self._name}/{self.stype}"
                 f"_points.tsv")
        idx = np.arange(len(self.vecs))[:, None]
        level = np.zeros_like(idx, dtype=int)
        for sl in self.slices[1:]:
            level[sl.start:] += 1
        idxvecs = np.hstack((level, idx, self.vecs))
        np.savetxt(vfile, idxvecs, ("%d", "%d", "%.4f", "%.4f", "%.4f"))

    def _wshape(self, level):
        return np.concatenate(([self.features], self.levels[level]))

    def _plot_dist(self, ps, outf):
        shp = ps.shape
        pixels = self._mapper.pixels(512)
        x = (np.arange(shp[0]) + .5)*pixels.shape[0]/shp[0]
        y = (np.arange(shp[1]) + .5)*pixels.shape[1]/shp[1]
        pinterp = RegularGridInterpolator((x, y), ps, bounds_error=False,
                                          method='nearest', fill_value=None)
        outpar = pinterp(pixels.reshape(-1, 2)).reshape(pixels.shape[:-1])
        io.array2hdr(outpar[-1::-1], outf)

    def _plot_p(self, p, level, vm, name, suffix=".hdr", **kwargs):
        outp = (f"{self.scene.outdir}_{name}_{self.stype}_detail_"
                f"{level:02d}{suffix}")
        ps = p.reshape(self.weights.shape[1:])
        self._plot_dist(ps, outp)

    def _plot_weights(self, level, vm, name, suffix=".hdr", **kwargs):
        normw = self._normed_weights()
        for i, w in zip(self.metricset, normw):
            outw = (f"{self.scene.outdir}_{name}_{self.stype}_weight_{i}_"
                    f"{level:02d}{suffix}")
            w.flat[np.logical_not(self._mask)] = 0
            self._plot_dist(w, outw)

    def _plot_vecs(self, vecs, level, vm, name, suffix=".hdr", **kwargs):
        img = np.zeros((3, *self._mapper.framesize(512)))
        img = self._mapper.add_vecs_to_img(img, vecs, grow=2)
        outv = (f"{self.scene.outdir}_{name}_{self.stype}_samples_"
                f"{level:02d}{suffix}")
        io.carray2hdr(img, outv)
