##
# UEFI Authenticated Variable Structure Support Library Unit Test
#
# Copyright (c) Microsoft Corporation
#
# SPDX-License-Identifier: BSD-2-Clause-Patent
##

# spell-checker: ignore deduplicated, deduplication, Dedpulication

import unittest
import uuid
import io
from edk2toollib.uefi.authenticated_variables_structure_support import \
    EfiSignatureDatabase, EfiSignatureList, EfiSignatureDataEfiCertSha256,\
    EfiSignatureDataEfiCertX509, EfiSignatureDataFactory, EFiVariableAuthentication2
# import logging

VERBOSE = False

DEBIAN_CERT = '308202FC308201E4A003020102020500A7468DEF300D06092A864886F70D01010B05003020311E301C060355040313154465' \
              '6269616E2053656375726520426F6F74204341301E170D3136303831363138323235305A170D323630383136313832323530' \
              '5A3024312230200603550403131944656269616E2053656375726520426F6F74205369676E657230820122300D06092A8648' \
              '86F70D01010105000382010F003082010A0282010100D3D183900FDA65A22F075A6095EBF7C7867C2086DA65A3A612EB5B3B' \
              'CEC8FB3FA1724B9EDF50C50333A40C2B5FD641040DB6CF9548ED8AB2ADD6E501374E60CDB24A3804B3448094AF9F6E54DBA8' \
              '1F3CB74B30DE21816F09A366BA6A2B96D69A61770CD4ED3CD071BBAD8CF0225C3E25CC6D222E619795AF9B2E4D58B67E7802' \
              'C30EB9FAB25B27DE7DA2BE0C14AC73EC97B0155EEDEDE5A5753F78E071CE2FCE83ED533130984EE6F901A28888A623087C0D' \
              'B7543A1695ED5E795E904EFECDAADE82FCF696714E4949B9D3E9B0AB7FD72A47B75330277CDC6698096FD17EF57F3D3ED4A2' \
              '6A8859022F2F3DC8C628DE42FED9523D24C2FC409811F676BF8CBB650203010001A3393037301106096086480186F8420101' \
              '04040302041030150603551D25040E300C060A2B0601040182370A0301300B0603551D0F040403020780300D06092A864886' \
              'F70D01010B05000382010100571BA4604C29E9F27D6B5C93DBCC6C9F183F69489A75DE64F3834A09A92621EEE9565DE13ED9' \
              '75CBCC7FBF4DE4E8893D7E11428740C3D5E07179DC006CE17162C798C2CB270B2F9FCCECFA8BB2F30B9EF3F2C3C99FDB2593' \
              '90A4CDBB01E58EF4D755A8B4754131FD4E5D0318A0C2ACC5DE46E7DC1CCF12D59DE8479D938C32CD44D574C7309A57A556D0' \
              '7ECF0511B4F4F329F9DB9B53D2BD2FAD6A75264564BABA2896878EB7F07957FA7A0E3C4A3892BCF295F2E728D0F7D8981A5E' \
              '399EB56580BDF3DA123F507667299FD10B0A1E87975C72DBF301744ADD07BA76E96AFCDD22DB4602D7AF0AC5ED15BC0F2BA9' \
              'DB8DBF7F6FADA2B7C54D4A47B3C15690B617'

UBUNTU_CERT = '3082042030820308A003020102020101300D06092A864886F70D01010B0500308184310B3009060355040613024742311430' \
              '1206035504080C0B49736C65206F66204D616E3110300E06035504070C07446F75676C617331173015060355040A0C0E4361' \
              '6E6F6E6963616C204C74642E3134303206035504030C2B43616E6F6E6963616C204C74642E204D6173746572204365727469' \
              '66696361746520417574686F72697479301E170D3132303431323131333930385A170D3432303431313131333930385A307F' \
              '310B30090603550406130247423114301206035504080C0B49736C65206F66204D616E31173015060355040A0C0E43616E6F' \
              '6E6963616C204C74642E31143012060355040B0C0B53656375726520426F6F74312B302906035504030C2243616E6F6E6963' \
              '616C204C74642E2053656375726520426F6F74205369676E696E6730820122300D06092A864886F70D01010105000382010F' \
              '003082010A0282010100C95F9B628F0BB06482ACBEC9E262E34BD29F1E8AD5611A2B5D38F4B7CEB99AB843B8439777AB4F7F' \
              '0C70460BFC7F6DC66DEA805E01D2B7661E87DE0D6DD04197A8A5AF0C634FF77CC252CCA031A9BB895D991E466F5573B97669' \
              'ECD7C1FC21D6C607E74FBD22DEE4A85B2DDB95341997D6284B214CCABB1D79A6177F5AF967E65C78453D106DB017592611C5' \
              '57E37F4E82BAF62C4EC8374DFF85158447E0ED3B7C7FBCAFE90105A70C6FC3E98DA3CEBEA6E3CD3CB5582C9EC2031C602237' \
              '39FF4102C129A46551FF3334AA4215F99578FC2DF5DA8A857C829DFB372C6BA5A8DF7C550B802E3CB063E1CD384889E81406' \
              '0B82BCFDD407681B0F3ED915DD94111B0203010001A381A030819D300C0603551D130101FF04023000301F0603551D250418' \
              '301606082B06010505070303060A2B0601040182370A0306302C06096086480186F842010D041F161D4F70656E53534C2047' \
              '656E657261746564204365727469666963617465301D0603551D0E0416041461482AA2830D0AB2AD5AF10B7250DA9033DDCE' \
              'F0301F0603551D23041830168014AD91990BC22AB1F517048C23B6655A268E345A63300D06092A864886F70D01010B050003' \
              '820101008F8AA1061F29B70A4AD5C5FD81AB25EAC07DE2FC6A96A0799367EE050E251225E45AF6AA1AF112F3058D875EF15A' \
              '5CCB8D2373651D15B9DE226BD64967C9A3C6D7624E5CB5F903834081DC879C3C3F1C0D519F94650A844867E4A2F8A64AF0E7' \
              'CDCDBD94E309D25D2D161B05150BCB44B43E614222C42A5C4EC51DA3E2E052B2EBF48B2BDC38395DFB88A156655F2B4F26FF' \
              '06781012EB8C5D32E3C645AF259BA0FF8EEF4709A3E98B37929269767E343B9205674EB025EDBC5E5F8FB4D6CA40FFE4E231' \
              '230C8525AE0C5501ECE5475EDF5BBC1433E3C6F518B6D9F7DDB3B4A131D35A5C5D7D3EBF0AE4E4E8B4597D3BB48CA31BB520' \
              'A3B93E846F8C2100C339'

HASHSTR = [
    '0000000000000000000000000000000000000000000000000000000000000000',
    '1111111111111111111111111111111111111111111111111111111111111111',
    '2222222222222222222222222222222222222222222222222222222222222222',
    '3333333333333333333333333333333333333333333333333333333333333333',
    '4444444444444444444444444444444444444444444444444444444444444444',
    '5555555555555555555555555555555555555555555555555555555555555555',
    '6666666666666666666666666666666666666666666666666666666666666666',
    '7777777777777777777777777777777777777777777777777777777777777777',
    '8888888888888888888888888888888888888888888888888888888888888888',
    '9999999999999999999999999999999999999999999999999999999999999999'
]

# DBXFILE contains Dbx contents downloaded from UEFI.org's Revocation List on 2023.01.30 (x64)
DBXFILE = 'da070306131115000000000000000000'\
          'f60c00000002f10e9dd2af4adf68ee49'\
          '8aa9347d375665a730820cda02010131'\
          '0f300d06096086480165030402010500'\
          '300b06092a864886f70d010701a0820a'\
          'ed308204fd308203e5a0030201020213'\
          '330000002596d20c5c53120043000000'\
          '000025300d06092a864886f70d01010b'\
          '0500308180310b300906035504061302'\
          '5553311330110603550408130a576173'\
          '68696e67746f6e3110300e0603550407'\
          '13075265646d6f6e64311e301c060355'\
          '040a13154d6963726f736f667420436f'\
          '72706f726174696f6e312a3028060355'\
          '040313214d6963726f736f667420436f'\
          '72706f726174696f6e204b454b204341'\
          '2032303131301e170d32313039303231'\
          '38323433315a170d3232303930313138'\
          '323433315a308186310b300906035504'\
          '0613025553311330110603550408130a'\
          '57617368696e67746f6e3110300e0603'\
          '55040713075265646d6f6e64311e301c'\
          '060355040a13154d6963726f736f6674'\
          '20436f72706f726174696f6e3130302e'\
          '060355040313274d6963726f736f6674'\
          '2057696e646f77732055454649204b65'\
          '792045786368616e6765204b65793082'\
          '0122300d06092a864886f70d01010105'\
          '000382010f003082010a0282010100ca'\
          'be8aef9b0f8692546db68e66e1585f14'\
          'ebebe6eed227c8fa323674b22e24efba'\
          'fa169e7a5fa4c2e6f252076ab245a4f8'\
          'cf358d0d9c618cafae8ad1807cd50c19'\
          '23a741a3525a4e32e78d21b83b063517'\
          '03349de5635e77b2ac3e502fb7231c59'\
          'e3f3c977139eeb65e44324f6cf04701f'\
          'd5962995d0f573012769eea20942a7be'\
          'c1f7e67d4b2e6ed8a958a98e4043aa6f'\
          '620b676becebab447f6ecbf174fda343'\
          'ba10cbfc1e05c2d652d3a139626f4a18'\
          'e7f8bf3fa412f63fc13af332040049d2'\
          'b26ea270c5cf914f56661de0d54ce3bc'\
          '3399e01e016190a03dc8826a87b032e0'\
          '1526dadd767ef07c0ba72953d021115f'\
          '87264bf21c6cb8dbe6c7f067087cb702'\
          '03010001a38201663082016230140603'\
          '551d25040d300b06092b060104018237'\
          '4f01301d0603551d0e04160414753d9c'\
          'ab3265b73cafcc22f9b16ba1e38dcf8d'\
          '6130450603551d11043e303ca43a3038'\
          '311e301c060355040b13154d6963726f'\
          '736f667420436f72706f726174696f6e'\
          '311630140603550405130d3232393936'\
          '312b343637353933301f0603551d2304'\
          '183016801462fc43cda03ea4cb6712d2'\
          '5bd955ac7bccb68a5f30530603551d1f'\
          '044c304a3048a046a044864268747470'\
          '3a2f2f7777772e6d6963726f736f6674'\
          '2e636f6d2f706b696f70732f63726c2f'\
          '4d6963436f724b454b4341323031315f'\
          '323031312d30362d32342e63726c3060'\
          '06082b06010505070101045430523050'\
          '06082b06010505073002864468747470'\
          '3a2f2f7777772e6d6963726f736f6674'\
          '2e636f6d2f706b696f70732f63657274'\
          '732f4d6963436f724b454b4341323031'\
          '315f323031312d30362d32342e637274'\
          '300c0603551d130101ff04023000300d'\
          '06092a864886f70d01010b0500038201'\
          '010069662ba664b30e7f47e3308e9cbf'\
          '0f6c1250bd8eecd227c8a223c5ced041'\
          'a35bef9d8aa41a31ceee757163a15d4f'\
          'dc921518daa170873922e85324a6d2a7'\
          'c2a74b483a488f7ef0262324568d6391'\
          '4a9d66915ee9f6a4c1c0f07592689c16'\
          '833a87006ce0f650b15314b0f4f3bb0b'\
          '3d623db77a4d526e8dc1d9358f2a3a6a'\
          '76661a41677f4d00a2979e8493e27106'\
          'e3b5414d11dc91d0c0dbe83ea3b18895'\
          '9b48889350293669c8bf9315b34d2aae'\
          '0f64b78e36051c341cde7d77ec06aa47'\
          'c6aa35f0bf6f36455fbc17c431b6048b'\
          'c2003c88b48f04b352da0fbef900a2f2'\
          '0262d867a54d4dd841605e1d4b0024d4'\
          '6f6c79cd4fd5b3c022107ac921ed71e0'\
          '8a59308205e8308203d0a00302010202'\
          '0a610ad188000000000003300d06092a'\
          '864886f70d01010b0500308191310b30'\
          '09060355040613025553311330110603'\
          '550408130a57617368696e67746f6e31'\
          '10300e060355040713075265646d6f6e'\
          '64311e301c060355040a13154d696372'\
          '6f736f667420436f72706f726174696f'\
          '6e313b3039060355040313324d696372'\
          '6f736f667420436f72706f726174696f'\
          '6e205468697264205061727479204d61'\
          '726b6574706c61636520526f6f74301e'\
          '170d3131303632343230343132395a17'\
          '0d3236303632343230353132395a3081'\
          '80310b30090603550406130255533113'\
          '30110603550408130a57617368696e67'\
          '746f6e3110300e060355040713075265'\
          '646d6f6e64311e301c060355040a1315'\
          '4d6963726f736f667420436f72706f72'\
          '6174696f6e312a302806035504031321'\
          '4d6963726f736f667420436f72706f72'\
          '6174696f6e204b454b20434120323031'\
          '3130820122300d06092a864886f70d01'\
          '010105000382010f003082010a028201'\
          '0100c4e8b58abfad5726b026c3eae7fb'\
          '577a44025d070dda4ae5742ae6b00fec'\
          '6debec7fb9e35a63327c11174f0ee30b'\
          'a73815938ec6f5e084b19a9b2ce7f5b7'\
          '91d609e1e2c004a8ac301cdf48f30650'\
          '9a64a7517fc8854f8f2086cefe2fe19f'\
          'ff82c0ede9cdcef4536a623a0b43b9e2'\
          '25fdfe05f9d4c414ab11e223898d70b7'\
          'a41d4decaee59cfa16c2d7c1cbd4e8c4'\
          '2fe599ee248b03ec8df28beac34afb43'\
          '11120b7eb547926cdce60489ebf53304'\
          'eb10012a71e5f983133cff25092f6876'\
          '46ffba4fbedcad712a58aafb0ed2793d'\
          'e49b653bcc292a9ffc7259a2ebae92ef'\
          'f6351380c602ece45fcc9d76cdef6392'\
          'c1af79408479877fe352a8e89d7b0769'\
          '8f150203010001a382014f3082014b30'\
          '1006092b060104018237150104030201'\
          '00301d0603551d0e0416041462fc43cd'\
          'a03ea4cb6712d25bd955ac7bccb68a5f'\
          '301906092b0601040182371402040c1e'\
          '0a00530075006200430041300b060355'\
          '1d0f040403020186300f0603551d1301'\
          '01ff040530030101ff301f0603551d23'\
          '04183016801445665243e17e5811bfd6'\
          '4e9e2355083b3a226aa8305c0603551d'\
          '1f045530533051a04fa04d864b687474'\
          '703a2f2f63726c2e6d6963726f736f66'\
          '742e636f6d2f706b692f63726c2f7072'\
          '6f64756374732f4d6963436f72546869'\
          '5061724d6172526f6f5f323031302d31'\
          '302d30352e63726c306006082b060105'\
          '0507010104543052305006082b060105'\
          '050730028644687474703a2f2f777777'\
          '2e6d6963726f736f66742e636f6d2f70'\
          '6b692f63657274732f4d6963436f7254'\
          '68695061724d6172526f6f5f32303130'\
          '2d31302d30352e637274300d06092a86'\
          '4886f70d01010b05000382020100d484'\
          '88f514941802ca2a3cfb2a921c0cd7a0'\
          'd1f1e85266a8eea2b5757a9000aa2da4'\
          '765aea79b7b9376a517b1064f6e164f2'\
          '0267bef7a81b78bdbace8858640cd657'\
          'c819a35f05d6dbc6d069ce484b32b7eb'\
          '5dd230f5c0f5b8ba7807a32bfe9bdb34'\
          '5684ec82caae4125709c6be9fe900fd7'\
          '961fe5e7941fb22a0c8d4bff2829107b'\
          'f7d77ca5d176b905c879ed0f90929cc2'\
          'fedf6f7e6c0f7bd4c145dd345196390f'\
          'e55e56d8180596f407a642b3a077fd08'\
          '19f27156cc9f8623a487cba6fd587ed4'\
          '696715917e81f27f13e50d8b8a3c8784'\
          'ebe3cebd43e5ad2d84938e6a2b5a7c44'\
          'fa52aa81c82d1cbbe052df0011f89a3d'\
          'c160b0e133b5a388d165190a1ae7ac7c'\
          'a4c182874e38b12f0dc514876ffd8d2e'\
          'bc39b6e7e6c3e0e4cd2784ef9442ef29'\
          '8b9046413b811b67d8f9435965cb0dbc'\
          'fd00924ff4753ba7a924fc50414079e0'\
          '2d4f0a6a27766e52ed96697baf0ff787'\
          '05d045c2ad5314811ffb3004aa373661'\
          'da4a691b34d868edd602cf6c940cd3cf'\
          '6c2279adb1f0bc03a24660a9c407c221'\
          '82f1fdf2e8793260bfd8aca522144bca'\
          'c1d84beb7d3f5735b2e64f75b4b06003'\
          '2253ae91791dd69b411f15865470b2de'\
          '0d350f7cb03472ba97603bf079eba2b2'\
          '1c5da216b887c5e91bf6b597256f389f'\
          'e391fa8a7998c3690eb7a31c200597f8'\
          'ca14ae00d7c4f3c01410756b34a01bb5'\
          '9960f35cb0c5574e36d23284bf9e3182'\
          '01c4308201c002010130819830818031'\
          '0b300906035504061302555331133011'\
          '0603550408130a57617368696e67746f'\
          '6e3110300e060355040713075265646d'\
          '6f6e64311e301c060355040a13154d69'\
          '63726f736f667420436f72706f726174'\
          '696f6e312a3028060355040313214d69'\
          '63726f736f667420436f72706f726174'\
          '696f6e204b454b204341203230313102'\
          '13330000002596d20c5c531200430000'\
          '00000025300d06096086480165030402'\
          '010500300d06092a864886f70d010101'\
          '05000482010002fcf6d3fb1c3e7af2bf'\
          'f3a028b0d2ddace519e9d3bf9eaccc0b'\
          '89e69ba782e42a6b7b723cbca7f93130'\
          'b79053db51f5c9e30b0607cb6144170a'\
          '1e2053ba1edd1553c44cf694c094e594'\
          'e904d341d157789acebbf327b52a9027'\
          '113195403625a47137dc8d779414d3cf'\
          '434d0077d61de393a960f1525bbb8c90'\
          'b6db9ffa1763bffc50e9f89a33c8a58f'\
          'c3af6a6174d18ee42b60b32ed0edf014'\
          '124e0cb65943b573b72b9c36b837b686'\
          'f1026874428c6b61cb6fec305fd51dde'\
          '53b32ead7575d8a9b8de4079d6476b16'\
          'db7d1b029ad59f76a1ed7b532d05e508'\
          '70eb873cd86c843b19a9e2ae412ad0f1'\
          'c768802626098dc8e1513529e6c7e062'\
          '801e1ee7daa02616c4c14c509240aca9'\
          '41f936934328cc280000000000003000'\
          '0000bd9afa775903324dbd6028f4e78f'\
          '784b80b4d96931bf0d02fd91a61e19d1'\
          '4f1da452e66db2408ca8604d411f9265'\
          '9f0abd9afa775903324dbd6028f4e78f'\
          '784bf52f83a3fa9cfbd6920f722824db'\
          'e4034534d25b8507246b3b957dac6e1b'\
          'ce7abd9afa775903324dbd6028f4e78f'\
          '784bc5d9d8a186e2c82d09afaa2a6f7f'\
          '2e73870d3e64f72c4e08ef67796a840f'\
          '0fbdbd9afa775903324dbd6028f4e78f'\
          '784b1aec84b84b6c65a51220a9be7181'\
          '965230210d62d6d33c48999c6b295a2b'\
          '0a06bd9afa775903324dbd6028f4e78f'\
          '784bc3a99a460da464a057c3586d83ce'\
          'f5f4ae08b7103979ed8932742df0ed53'\
          '0c66bd9afa775903324dbd6028f4e78f'\
          '784b58fb941aef95a25943b3fb5f2510'\
          'a0df3fe44c58c95e0ab80487297568ab'\
          '9771bd9afa775903324dbd6028f4e78f'\
          '784b5391c3a2fb112102a6aa1edc25ae'\
          '77e19f5d6f09cd09eeb2509922bfcd59'\
          '92eabd9afa775903324dbd6028f4e78f'\
          '784bd626157e1d6a718bc124ab8da27c'\
          'bb65072ca03a7b6b257dbdcbbd60f65e'\
          'f3d1bd9afa775903324dbd6028f4e78f'\
          '784bd063ec28f67eba53f1642dbf7dff'\
          '33c6a32add869f6013fe162e2c32f1cb'\
          'e56dbd9afa775903324dbd6028f4e78f'\
          '784b29c6eb52b43c3aa18b2cd8ed6ea8'\
          '607cef3cfae1bafe1165755cf2e61484'\
          '4a44bd9afa775903324dbd6028f4e78f'\
          '784b90fbe70e69d633408d3e170c6832'\
          'dbb2d209e0272527dfb63d49d29572a6'\
          'f44cbd9afa775903324dbd6028f4e78f'\
          '784b106faceacfecfd4e303b74f480a0'\
          '8098e2d0802b936f8ec774ce21f31686'\
          '689cbd9afa775903324dbd6028f4e78f'\
          '784b174e3a0b5b43c6a607bbd3404f05'\
          '341e3dcf396267ce94f8b50e2e23a9da'\
          '920cbd9afa775903324dbd6028f4e78f'\
          '784b2b99cf26422e92fe365fbf4bc30d'\
          '27086c9ee14b7a6fff44fb2f6b900169'\
          '9939bd9afa775903324dbd6028f4e78f'\
          '784b2e70916786a6f773511fa7181fab'\
          '0f1d70b557c6322ea923b2a8d3b92b51'\
          'af7dbd9afa775903324dbd6028f4e78f'\
          '784b3fce9b9fdf3ef09d5452b0f95ee4'\
          '81c2b7f06d743a737971558e70136ace'\
          '3e73bd9afa775903324dbd6028f4e78f'\
          '784b47cc086127e2069a86e03a6bef2c'\
          'd410f8c55a6d6bdb362168c31b2ce32a'\
          '5adfbd9afa775903324dbd6028f4e78f'\
          '784b71f2906fd222497e54a34662ab24'\
          '97fcc81020770ff51368e9e3d9bfcbfd'\
          '6375bd9afa775903324dbd6028f4e78f'\
          '784b82db3bceb4f60843ce9d97c3d187'\
          'cd9b5941cd3de8100e586f2bda563757'\
          '5f67bd9afa775903324dbd6028f4e78f'\
          '784b8ad64859f195b5f58dafaa940b6a'\
          '6167acd67a886e8f469364177221c559'\
          '45b9bd9afa775903324dbd6028f4e78f'\
          '784b8d8ea289cfe70a1c07ab7365cb28'\
          'ee51edd33cf2506de888fbadd60ebf80'\
          '481cbd9afa775903324dbd6028f4e78f'\
          '784baeebae3151271273ed95aa2e6711'\
          '39ed31a98567303a332298f83709a9d5'\
          '5aa1bd9afa775903324dbd6028f4e78f'\
          '784bc409bdac4775add8db92aa22b5b7'\
          '18fb8c94a1462c1fe9a416b95d8a3388'\
          'c2fcbd9afa775903324dbd6028f4e78f'\
          '784bc617c1a8b1ee2a811c28b5a81b4c'\
          '83d7c98b5b0c27281d610207ebe692c2'\
          '967fbd9afa775903324dbd6028f4e78f'\
          '784bc90f336617b8e7f983975413c997'\
          'f10b73eb267fd8a10cb9e3bdbfc667ab'\
          'db8bbd9afa775903324dbd6028f4e78f'\
          '784b64575bd912789a2e14ad56f6341f'\
          '52af6bf80cf94400785975e9f04e2d64'\
          'd745bd9afa775903324dbd6028f4e78f'\
          '784b45c7c8ae750acfbb48fc37527d64'\
          '12dd644daed8913ccd8a24c94d856967'\
          'df8ebd9afa775903324dbd6028f4e78f'\
          '784b81d8fb4c9e2e7a8225656b4b8273'\
          'b7cba4b03ef2e9eb20e0a0291624eca1'\
          'ba86bd9afa775903324dbd6028f4e78f'\
          '784bb92af298dc08049b78c77492d655'\
          '1b710cd72aada3d77be54609e43278ef'\
          '6e4dbd9afa775903324dbd6028f4e78f'\
          '784be19dae83c02e6f281358d4ebd11d'\
          '7723b4f5ea0e357907d5443decc5f93c'\
          '1e9dbd9afa775903324dbd6028f4e78f'\
          '784b39dbc2288ef44b5f95332cb777e3'\
          '1103e840dba680634aa806f5c9b10006'\
          '1802bd9afa775903324dbd6028f4e78f'\
          '784b32f5940ca29dd812a2c145e6fc89'\
          '646628ffcc7c7a42cae512337d8d29c4'\
          '0bbdbd9afa775903324dbd6028f4e78f'\
          '784b10d45fcba396aef3153ee8f6ecae'\
          '58afe8476a280a2026fc71f6217dcf49'\
          'ba2fbd9afa775903324dbd6028f4e78f'\
          '784b4b8668a5d465bcdd9000aa8dfcff'\
          '42044fcbd0aece32fc7011a83e9160e8'\
          '9f09bd9afa775903324dbd6028f4e78f'\
          '784b89f3d1f6e485c334cd059d0995e3'\
          'cdfdc00571b1849854847a44dc5548e2'\
          'dcfbbd9afa775903324dbd6028f4e78f'\
          '784bc9ec350406f26e559affb4030de2'\
          'ebde5435054c35a998605b8fcf04972d'\
          '8d55bd9afa775903324dbd6028f4e78f'\
          '784bb3e506340fbf6b5786973393079f'\
          '24b66ba46507e35e911db0362a2acde9'\
          '7049bd9afa775903324dbd6028f4e78f'\
          '784b9f1863ed5717c394b42ef10a6607'\
          'b144a65ba11fb6579df94b8eb2f0c4cd'\
          '60c1bd9afa775903324dbd6028f4e78f'\
          '784bdd59af56084406e38c63fbe0850f'\
          '30a0cd1277462a2192590fb05bc259e6'\
          '1273bd9afa775903324dbd6028f4e78f'\
          '784bdbaf9e056d3d5b38b68553304abc'\
          '88827ebc00f80cb9c7e197cdbc5822cd'\
          '316cbd9afa775903324dbd6028f4e78f'\
          '784b65f3c0a01b8402d362b9722e98f7'\
          '5e5e991e6c186e934f7b2b2e6be6dec8'\
          '00ecbd9afa775903324dbd6028f4e78f'\
          '784b5b248e913d71853d3da5aedd8d9a'\
          '4bc57a917126573817fb5fcb2d86a2f1'\
          'c886bd9afa775903324dbd6028f4e78f'\
          '784b2679650fe341f2cf1ea883460b35'\
          '56aaaf77a70d6b8dc484c9301d1b746c'\
          'f7b5bd9afa775903324dbd6028f4e78f'\
          '784bbb1dd16d530008636f232303a7a8'\
          '6f3dff969f848815c0574b12c2d787fe'\
          'c93fbd9afa775903324dbd6028f4e78f'\
          '784b0ce02100f67c7ef85f4eed368f02'\
          'bf7092380a3c23ca91fd7f19430d94b0'\
          '0c19bd9afa775903324dbd6028f4e78f'\
          '784b95049f0e4137c790b0d2767195e5'\
          '6f73807d123adcf8f6e7bf2d4d991d30'\
          '5f89bd9afa775903324dbd6028f4e78f'\
          '784b02e6216acaef6401401fa555ecbe'\
          'd940b1a5f2569aed92956137ae58482e'\
          'f1b7bd9afa775903324dbd6028f4e78f'\
          '784b6efefe0b5b01478b7b944c10d3a8'\
          'aca2cca4208888e2059f8a06cb5824d7'\
          'bab0bd9afa775903324dbd6028f4e78f'\
          '784b9d00ae4cd47a41c783dc48f342c0'\
          '76c2c16f3413f4d2df50d181ca3bb5ad'\
          '859dbd9afa775903324dbd6028f4e78f'\
          '784bd8d4e6ddf6e42d74a6a536ea62fd'\
          '1217e4290b145c9e5c3695a31b42efb5'\
          'f5a4bd9afa775903324dbd6028f4e78f'\
          '784bf277af4f9bdc918ae89fa35cc1b3'\
          '4e34984c04ae9765322c3cb049574d36'\
          '509cbd9afa775903324dbd6028f4e78f'\
          '784b0dc24c75eb1aef56b9f13ab9de60'\
          'e2eca1c4510034e290bbb36cf60a549b'\
          '234cbd9afa775903324dbd6028f4e78f'\
          '784b835881f2a5572d7059b5c8635018'\
          '552892e945626f115fc9ca07acf7bde8'\
          '57a4bd9afa775903324dbd6028f4e78f'\
          '784bbadff5e4f0fea711701ca8fb22e4'\
          'c43821e31e210cf52d1d4f74dd50f1d0'\
          '39bcbd9afa775903324dbd6028f4e78f'\
          '784bc452ab846073df5ace25cca64d6b'\
          '7a09d906308a1a65eb5240e3c4ebcaa9'\
          'cc0cbd9afa775903324dbd6028f4e78f'\
          '784bf1863ec8b7f43f94ad14fb0b8b4a'\
          '69497a8c65ecbc2a55e0bb420e772b8c'\
          'dc91bd9afa775903324dbd6028f4e78f'\
          '784b7bc9cb5463ce0f011fb5085eb8ba'\
          '77d1acd283c43f4a57603cc113f22ceb'\
          'c579bd9afa775903324dbd6028f4e78f'\
          '784be800395dbe0e045781e8005178b4'\
          'baf5a257f06e159121a67c595f6ae225'\
          '06fdbd9afa775903324dbd6028f4e78f'\
          '784b1cb4dccaf2c812cfa7b4938e1371'\
          'fe2b96910fe407216fd95428672d6c7e'\
          '7316bd9afa775903324dbd6028f4e78f'\
          '784b3ece27cbb3ec4438cce523b927c4'\
          'f05fdc5c593a3766db984c5e437a3ff6'\
          'a16bbd9afa775903324dbd6028f4e78f'\
          '784b68ee4632c7be1c66c83e89dd93ea'\
          'ee1294159abf45b4c2c72d7dc7499aa2'\
          'a043bd9afa775903324dbd6028f4e78f'\
          '784be24b315a551671483d8b9073b32d'\
          'e11b4de1eb2eab211afd2d9c319ff55e'\
          '08d0bd9afa775903324dbd6028f4e78f'\
          '784be7c20b3ab481ec885501eca52937'\
          '81d84b5a1ac24f88266b5270e7ecb4aa'\
          '2538bd9afa775903324dbd6028f4e78f'\
          '784bdccc3ce1c00ee4b0b10487d372a0'\
          'fa47f5c26f57a359be7b27801e144eac'\
          'bac4bd9afa775903324dbd6028f4e78f'\
          '784b0257ff710f2a16e489b37493c076'\
          '04a7cda96129d8a8fd68d2b6af633904'\
          '315dbd9afa775903324dbd6028f4e78f'\
          '784b3a91f0f9e5287fa2994c7d930b2c'\
          '1a5ee14ce8e1c8304ae495adc58cc445'\
          '3c0cbd9afa775903324dbd6028f4e78f'\
          '784b495300790e6c9bf2510daba59db3'\
          'd57e9d2b85d7d7640434ec75baa3851c'\
          '74e5bd9afa775903324dbd6028f4e78f'\
          '784b81a8b2c9751aeb1faba7dbde5ee9'\
          '691dc0eaee2a31c38b1491a8146756a6'\
          'b770bd9afa775903324dbd6028f4e78f'\
          '784b8e53efdc15f852cee5a6e92931bc'\
          '42e6163cd30ff649cca7e87252c3a459'\
          '960bbd9afa775903324dbd6028f4e78f'\
          '784b992d359aa7a5f789d268b94c11b9'\
          '485a6b1ce64362b0edb4441ccc187c39'\
          '647bbd9afa775903324dbd6028f4e78f'\
          '784b9fa4d5023fd43ecaff4200ba7e8d'\
          '4353259d2b7e5e72b5096eff8027d66d'\
          '1043bd9afa775903324dbd6028f4e78f'\
          '784bd372c0d0f4fdc9f52e9e1f23fc56'\
          'ee72414a17f350d0cea6c26a35a6c321'\
          '7a13bd9afa775903324dbd6028f4e78f'\
          '784b5c5805196a85e93789457017d4f9'\
          'eb6828b97c41cb9ba6d3dc1fcc115f52'\
          '7a55bd9afa775903324dbd6028f4e78f'\
          '784b03f64a29948a88beffdb035e0b09'\
          'a7370ccf0cd9ce6bcf8e640c2107318f'\
          'ab87bd9afa775903324dbd6028f4e78f'\
          '784b05d87e15713454616f5b0ed7849a'\
          'b5c1712ab84f02349478ec2a38f970c0'\
          '1489bd9afa775903324dbd6028f4e78f'\
          '784b06eb5badd26e4fae65f9a42358de'\
          'ef7c18e52cc05fbb7fc76776e69d1b98'\
          '2a14bd9afa775903324dbd6028f4e78f'\
          '784b08bb2289e9e91b4d20ff3f156251'\
          '6ab07e979b2c6cefe2ab70c6dfc1199f'\
          '8da5bd9afa775903324dbd6028f4e78f'\
          '784b0928f0408bf725e61d67d87138a8'\
          'eebc52962d2847f16e3587163b160e41'\
          'b6adbd9afa775903324dbd6028f4e78f'\
          '784b09f98aa90f85198c0d73f89ba77e'\
          '87ec6f596c491350fb8f8bba80a62fbb'\
          '914bbd9afa775903324dbd6028f4e78f'\
          '784b0a75ea0b1d70eaa4d3f374246db5'\
          '4fc7b43e7f596a353309b9c36b4fd975'\
          '725ebd9afa775903324dbd6028f4e78f'\
          '784b0c51d7906fc4931149765da88682'\
          '426b2cfe9e6aa4f27253eab400111432'\
          'e3a7bd9afa775903324dbd6028f4e78f'\
          '784b0fa3a29ad05130d7fe5bf4d25965'\
          '63cded1d874096aacc181069932a2e49'\
          '519abd9afa775903324dbd6028f4e78f'\
          '784b147730b42f11fe493fe902b6251e'\
          '97cd2b6f34d36af59330f11d02a42f94'\
          '0d07bd9afa775903324dbd6028f4e78f'\
          '784b148fe18f715a9fcfe1a444ce0fff'\
          '7f85869eb422330dc04b314c0f295d6d'\
          'a79ebd9afa775903324dbd6028f4e78f'\
          '784b1b909115a8d473e51328a87823bd'\
          '621ce655dfae54fa2bfa72fdc0298611'\
          'd6b8bd9afa775903324dbd6028f4e78f'\
          '784b1d8b58c1fdb8da8b33ccee1e5f97'\
          '3af734d90ef317e33f5db1573c2ba088'\
          'a80cbd9afa775903324dbd6028f4e78f'\
          '784b1f179186efdf5ef2de018245ba0e'\
          'ae8134868601ba0d35ff3d9865c1537c'\
          'ed93bd9afa775903324dbd6028f4e78f'\
          '784b270c84b29d86f16312b06aaae4eb'\
          'b8dff8de7d080d825b8839ff1766274e'\
          'ff47bd9afa775903324dbd6028f4e78f'\
          '784b29cca4544ea330d61591c784695c'\
          '149c6b040022ac7b5b89cbd72800d108'\
          '40eabd9afa775903324dbd6028f4e78f'\
          '784b2b2298eaa26b9dc4a4558ae92e7b'\
          'b0e4f85cf34bf848fdf636c0c11fbec4'\
          '9897bd9afa775903324dbd6028f4e78f'\
          '784b2dcf8e8d817023d1e8e1451a3d68'\
          'd6ec30d9bed94cbcb87f19ddc1cc0116'\
          'ac1abd9afa775903324dbd6028f4e78f'\
          '784b311a2ac55b50c09b30b3cc93b994'\
          'a119153eeeac54ef892fc447bbbd9610'\
          '1aa1bd9afa775903324dbd6028f4e78f'\
          '784b32ad3296829bc46dcfac5eddcb9d'\
          'bf2c1eed5c11f83b2210cf9c6e60c798'\
          'd4a7bd9afa775903324dbd6028f4e78f'\
          '784b340da32b58331c8e2b561baf300c'\
          'a9dfd6b91cd2270ee0e2a34958b1c625'\
          '9e85bd9afa775903324dbd6028f4e78f'\
          '784b362ed31d20b1e00392281231a96f'\
          '0a0acfde02618953e695c9ef2eb0bac3'\
          '7550bd9afa775903324dbd6028f4e78f'\
          '784b367a31e5838831ad2c074647886a'\
          '6cdff217e6b1ba910bff85dc7a87ae9b'\
          '5e98bd9afa775903324dbd6028f4e78f'\
          '784b3765d769c05bf98b427b3511903b'\
          '2137e8a49b6f859d0af159ed6a86786a'\
          'a634bd9afa775903324dbd6028f4e78f'\
          '784b386d695cdf2d4576e01bcaccf5e4'\
          '9e78da51af9955c0b8fa7606373b0079'\
          '94b3bd9afa775903324dbd6028f4e78f'\
          '784b3a4f74beafae2b9383ad8215d233'\
          'a6cf3d057fb3c7e213e897beef4255fa'\
          'ee9dbd9afa775903324dbd6028f4e78f'\
          '784b3ae76c45ca70e9180c1559981f42'\
          '622dd251bca1fbe6b901c52ec11673b0'\
          '3514bd9afa775903324dbd6028f4e78f'\
          '784b3be8e7eb348d35c1928f19c76984'\
          '6788991641d1f6cf09514ca10269934f'\
          '7359bd9afa775903324dbd6028f4e78f'\
          '784b3e3926f0b8a15ad5a14167bb647a'\
          '843c3d4321e35dbc44dce8c837417f2d'\
          '28b0bd9afa775903324dbd6028f4e78f'\
          '784b400ac66d59b7b094a9e30b01a6bd'\
          '013aff1d30570f83e7592f421dbe5ff4'\
          'ba8fbd9afa775903324dbd6028f4e78f'\
          '784b4185821f6dab5ba8347b78a22b5f'\
          '9a0a7570ca5c93a74d478a793d83bac4'\
          '9805bd9afa775903324dbd6028f4e78f'\
          '784b41d1eeb177c0324e17dd6557f384'\
          'e532de0cf51a019a446b01efb351bc25'\
          '9d77bd9afa775903324dbd6028f4e78f'\
          '784b45876b4dd861d45b3a9480077402'\
          '7a5db45a48b2a729410908b6412f8a87'\
          'e95dbd9afa775903324dbd6028f4e78f'\
          '784b4667bf250cd7c1a06b8474c613cd'\
          'b1df648a7f58736fbf57d05d6f755dab'\
          '67f4bd9afa775903324dbd6028f4e78f'\
          '784b47ff1b63b140b6fc04ed79131331'\
          'e651da5b2e2f170f5daef4153dc2fbc5'\
          '32b1bd9afa775903324dbd6028f4e78f'\
          '784b57e6913afacc5222bd76cdaf31f8'\
          'ed88895464255374ef097a82d7f59ad3'\
          '9596bd9afa775903324dbd6028f4e78f'\
          '784b5890fa227121c76d90ed9e63c87e'\
          '3a6533eea0f6f0a1a23f1fc445139bc6'\
          'bcdfbd9afa775903324dbd6028f4e78f'\
          '784b5d1e9acbbb4a7d024b6852df0259'\
          '70e2ced66ff622ee019cd0ed7fd841cc'\
          'ad02bd9afa775903324dbd6028f4e78f'\
          '784b61cec4a377bf5902c0feaee37034'\
          'bf97d5bc6e0615e23a1cdfbae6e3f5fb'\
          '3cfdbd9afa775903324dbd6028f4e78f'\
          '784b631f0857b41845362c90c6980b4b'\
          '10c4b628e23dbe24b6e96c128ae3dcb0'\
          'd5acbd9afa775903324dbd6028f4e78f'\
          '784b65b2e7cc18d903c331df1152df73'\
          'ca0dc932d29f17997481c56f3087b2dd'\
          '3147bd9afa775903324dbd6028f4e78f'\
          '784b66aa13a0edc219384d9c425d3927'\
          'e6ed4a5d1940c5e7cd4dac88f5770103'\
          'f2f1bd9afa775903324dbd6028f4e78f'\
          '784b6873d2f61c29bd52e954eeff5977'\
          'aa8367439997811a62ff212c948133c6'\
          '8d97bd9afa775903324dbd6028f4e78f'\
          '784b6dbbead23e8c860cf8b47f74fbfc'\
          'a5204de3e28b881313bb1d1eccdc4747'\
          '934ebd9afa775903324dbd6028f4e78f'\
          '784b6dead13257dfc3ccc6a4b37016ba'\
          '91755fe9e0ec1f415030942e5abc47f0'\
          '7c88bd9afa775903324dbd6028f4e78f'\
          '784b70a1450af2ad395569ad0afeb1d9'\
          'c125324ee90aec39c258880134d4892d'\
          '51abbd9afa775903324dbd6028f4e78f'\
          '784b72c26f827ceb92989798961bc6ae'\
          '748d141e05d3ebcfb65d9041b266c920'\
          'be82bd9afa775903324dbd6028f4e78f'\
          '784b781764102188a8b4b173d4a8f5ec'\
          '94d828647156097f99357a581e624b37'\
          '7509bd9afa775903324dbd6028f4e78f'\
          '784b788383a4c733bb87d2bf51673dc7'\
          '3e92df15ab7d51dc715627ae77686d8d'\
          '23bcbd9afa775903324dbd6028f4e78f'\
          '784b78b4edcaabc8d9093e20e217802c'\
          'aeb4f09e23a3394c4acc6e87e8f35395'\
          '310fbd9afa775903324dbd6028f4e78f'\
          '784b7f49ccb309323b1c7ab11c93c955'\
          'b8c744f0a2b75c311f495e1890607050'\
          '0027bd9afa775903324dbd6028f4e78f'\
          '784b82acba48d5236ccff7659afc1459'\
          '4dee902bd6082ef1a30a0b9b508628cf'\
          '34f4bd9afa775903324dbd6028f4e78f'\
          '784b894d7839368f3298cc915ae8742e'\
          'f330d7a26699f459478cf22c2b6bb285'\
          '0166bd9afa775903324dbd6028f4e78f'\
          '784b8c0349d708571ae5aa21c1136348'\
          '2332073297d868f29058916529efc520'\
          'ef70bd9afa775903324dbd6028f4e78f'\
          '784b8d93d60c691959651476e5dc464b'\
          'e12a85fa5280b6f524d4a1c3fcc9d048'\
          'cfadbd9afa775903324dbd6028f4e78f'\
          '784b9063f5fbc5e57ab6de6c94881460'\
          '20e172b176d5ab57d4c89f0f600e17fe'\
          '2de2bd9afa775903324dbd6028f4e78f'\
          '784b91656aa4ef493b3824a0b7263248'\
          'e4e2d657a5c8488d880cb65b01730932'\
          'fb53bd9afa775903324dbd6028f4e78f'\
          '784b91971c1497bf8e5bc68439acc48d'\
          '63ebb8faabfd764dcbe82f3ba977cac8'\
          'cf6abd9afa775903324dbd6028f4e78f'\
          '784b947078f97c6196968c3ae99c9a5d'\
          '58667e86882cf6c8c9d58967a496bb7a'\
          'f43cbd9afa775903324dbd6028f4e78f'\
          '784b96e4509450d380dac362ff8e2955'\
          '89128a1f1ce55885d20d89c27ba2a9d0'\
          '0909bd9afa775903324dbd6028f4e78f'\
          '784b9783b5ee4492e9e891c655f1f480'\
          '35959dad453c0e623af0fe7bf2c0a578'\
          '85e3bd9afa775903324dbd6028f4e78f'\
          '784b97a51a094444620df38cd8c6512c'\
          'ac909a75fd437ae1e4d2292980766123'\
          '8127bd9afa775903324dbd6028f4e78f'\
          '784b97a8c5ba11d61fefbb5d6a05da4e'\
          '15ba472dc4c6cd4972fc1a035de32134'\
          '2fe4bd9afa775903324dbd6028f4e78f'\
          '784b992820e6ec8c41daae4bd8ab48f5'\
          '8268e943a670d35ca5e2bdcd3e7c4c94'\
          'a072bd9afa775903324dbd6028f4e78f'\
          '784b9954a1a99d55e8b189ab1bca414b'\
          '91f6a017191f6c40a86b6f3ef368dd86'\
          '0031bd9afa775903324dbd6028f4e78f'\
          '784b9baf4f76d76bf5d6a897bfbd5f42'\
          '9ba14d04e08b48c3ee8d76930a828fff'\
          '3891bd9afa775903324dbd6028f4e78f'\
          '784b9c259fcb301d5fc7397ed5759963'\
          'e0ef6b36e42057fd73046e6bd08b149f'\
          '751cbd9afa775903324dbd6028f4e78f'\
          '784b9dd2dcb72f5e741627f2e9e03ab1'\
          '8503a3403cf6a904a479a4db05d97e22'\
          '50a9bd9afa775903324dbd6028f4e78f'\
          '784b9ed33f0fbc180bc032f8909ca2c4'\
          'ab3418edc33a45a50d2521a3b5876aa3'\
          'ea2cbd9afa775903324dbd6028f4e78f'\
          '784ba4d978b7c4bda15435d508f8b959'\
          '2ec2a5adfb12ea7bad146a35ecb53094'\
          '642fbd9afa775903324dbd6028f4e78f'\
          '784ba924d3cad6da42b7399b96a095a0'\
          '6f18f6b1aba5b873b0d5f3a0ee2173b4'\
          '8b6cbd9afa775903324dbd6028f4e78f'\
          '784bad3be589c0474e97de5bb2bf3353'\
          '4948b76bb80376dfdc58b1fed767b5a1'\
          '5bfcbd9afa775903324dbd6028f4e78f'\
          '784bb8d6b5e7857b45830e017c7be3d8'\
          '56adeb97c7290eb0665a3d473a4beb51'\
          'dcf3bd9afa775903324dbd6028f4e78f'\
          '784bb93f0699598f8b20fa0dacc12cfc'\
          'fc1f2568793f6e779e04795e6d7c2253'\
          '0f75bd9afa775903324dbd6028f4e78f'\
          '784bbb01da0333bb639c7e1c806db056'\
          '1dc98a5316f22fef1090fb8d0be46dae'\
          '499abd9afa775903324dbd6028f4e78f'\
          '784bbc75f910ff320f5cb5999e66bbd4'\
          '034f4ae537a42fdfef35161c5348e366'\
          'e216bd9afa775903324dbd6028f4e78f'\
          '784bbdd01126e9d85710d3fe75af1cc1'\
          '702a29f081b4f6fdf6a2b2135c0297a9'\
          'cec5bd9afa775903324dbd6028f4e78f'\
          '784bbe435df7cd28aa2a7c8db4fc8173'\
          '475b77e5abf392f76b7c76fa3f698cb7'\
          '1a9abd9afa775903324dbd6028f4e78f'\
          '784bbef7663be5ea4dbfd8686e24701e'\
          '036f4c03fb7fcd67a6c566ed94ce09c4'\
          '4470bd9afa775903324dbd6028f4e78f'\
          '784bc2469759c1947e14f4b65f72a9f5'\
          'b3af8b6f6e727b68bb0d91385cbf4217'\
          '6a8abd9afa775903324dbd6028f4e78f'\
          '784bc3505bf3ec10a51dace417c76b8b'\
          'd10939a065d1f34e75b8a3065ee31cc6'\
          '9b96bd9afa775903324dbd6028f4e78f'\
          '784bc42d11c70ccf5e8cf3fb91fdf21d'\
          '884021ad836ca68adf2cbb7995c10bf5'\
          '88d4bd9afa775903324dbd6028f4e78f'\
          '784bc69d64a5b839e41ba16742527e17'\
          '056a18ce3c276fd26e34901a1bc7d0e3'\
          '2219bd9afa775903324dbd6028f4e78f'\
          '784bcb340011afeb0d74c4a588b36eba'\
          'a441961608e8d2fa80dca8c13872c850'\
          '796bbd9afa775903324dbd6028f4e78f'\
          '784bcc8eec6eb9212cbf897a5ace7e8a'\
          'beece1079f1a6def0a789591cb1547f1'\
          'f084bd9afa775903324dbd6028f4e78f'\
          '784bcf13a243c1cd2e3c8ceb7e701003'\
          '87cecbfb830525bbf9d0b70c79adf3e8'\
          '4128bd9afa775903324dbd6028f4e78f'\
          '784bd89a11d16c488dd4fbbc541d4b07'\
          'faf8670d660994488fe54b1fbff2704e'\
          '4288bd9afa775903324dbd6028f4e78f'\
          '784bd9668ab52785086786c134b5e4bd'\
          'dbf72452813b6973229ab92aa1a54d20'\
          '1bf5bd9afa775903324dbd6028f4e78f'\
          '784bda3560fd0c32b54c83d4f2ff8690'\
          '03d2089369acf2c89608f8afa7436bfa'\
          '4655bd9afa775903324dbd6028f4e78f'\
          '784bdf02aab48387a9e1d4c65228089c'\
          'b6abe196c8f4b396c7e4bbc395de1369'\
          '77f6bd9afa775903324dbd6028f4e78f'\
          '784bdf91ac85a94fcd0cfb8155bd7cbe'\
          'faac14b8c5ee7397fe2cc85984459e2e'\
          'a14ebd9afa775903324dbd6028f4e78f'\
          '784be051b788ecbaeda53046c70e6af6'\
          '058f95222c046157b8c4c1b9c2cfc65f'\
          '46e5bd9afa775903324dbd6028f4e78f'\
          '784be36dfc719d2114c2e39aea88849e'\
          '2845ab326f6f7fe74e0e539b7e54d81f'\
          '3631bd9afa775903324dbd6028f4e78f'\
          '784be39891f48bbcc593b8ed86ce82ce'\
          '666fc1145b9fcbfd2b07bad0a89bf4c7'\
          'bfbfbd9afa775903324dbd6028f4e78f'\
          '784be6856f137f79992dc94fa2f43297'\
          'ec32d2d9a76f7be66114c6a13efc3bcd'\
          'f5c8bd9afa775903324dbd6028f4e78f'\
          '784beaff8c85c208ba4d5b6b8046f5d6'\
          '081747d779bada7768e649d047ff9b1f'\
          '660cbd9afa775903324dbd6028f4e78f'\
          '784bee83a566496109a74f6ac6e410df'\
          '00bb29a290e0021516ae3b8a23288e7e'\
          '2e72bd9afa775903324dbd6028f4e78f'\
          '784beed7e0eff2ed559e2a79ee361f99'\
          '62af3b1e999131e30bb7fd07546fae0a'\
          '7267bd9afa775903324dbd6028f4e78f'\
          '784bf1b4f6513b0d544a688d13adc291'\
          'efa8c59f420ca5dcb23e0b5a06fa7e0d'\
          '083dbd9afa775903324dbd6028f4e78f'\
          '784bf2a16d35b554694187a70d40ca68'\
          '2959f4f35c2ce0eab8fd64f7ac2ab9f5'\
          'c24abd9afa775903324dbd6028f4e78f'\
          '784bf31fd461c5e99510403fc97c1da2'\
          'd8a9cbe270597d32badf8fd66b77495f'\
          '8d94bd9afa775903324dbd6028f4e78f'\
          '784bf48e6dd8718e953b60a24f2cbea6'\
          '0a9521deae67db25425b7d3ace3c517d'\
          'd9b7bd9afa775903324dbd6028f4e78f'\
          '784bc805603c4fa038776e42f263c604'\
          'b49d96840322e1922d5606a9b0bbb5bf'\
          'fe6fbd9afa775903324dbd6028f4e78f'\
          '784b1f16078cce009df62edb9e7170e6'\
          '6caae670bce71b8f92d38280c56aa372'\
          '031dbd9afa775903324dbd6028f4e78f'\
          '784b37a480374daf6202ce790c318a2b'\
          'b8aa3797311261160a8e30558b7dea78'\
          'c7a6bd9afa775903324dbd6028f4e78f'\
          '784b408b8b3df5abb043521a49352502'\
          '3175ab1261b1de21064d6bf247ce1421'\
          '53b9bd9afa775903324dbd6028f4e78f'\
          '784b540801dd345dc1c33ef431b35bf4'\
          'c0e68bd319b577b9abe1a9cff1cbc39f'\
          '548fbd9afa775903324dbd6028f4e78f'\
          '784b040b3bc339e9b6f9acd828b88f34'\
          '82a5c3f64e67e5a714ba1da8a70453b3'\
          '4af6bd9afa775903324dbd6028f4e78f'\
          '784b1142a0cc7c9004dff64c5948484d'\
          '6a7ec3514e176f5ca6bdeed7a093940b'\
          '93ccbd9afa775903324dbd6028f4e78f'\
          '784b288878f12e8b9c6ccbf601c73d5f'\
          '4e985cac0ff3fcb0c24e4414912b3eb9'\
          '1f15bd9afa775903324dbd6028f4e78f'\
          '784b2ea4cb6a1f1eb1d3dce82d54fde2'\
          '6ded243ba3e18de7c6d211902a594fe5'\
          '6788bd9afa775903324dbd6028f4e78f'\
          '784b40d6cae02973789080cf4c3a9ad1'\
          '1b5a0a4d8bba4438ab96e276cc784454'\
          'dee7bd9afa775903324dbd6028f4e78f'\
          '784b4f0214fce4fa8897d0c80a46d6da'\
          'b4124726d136fc2492efd01bfedfa388'\
          '7a9cbd9afa775903324dbd6028f4e78f'\
          '784b5c2afe34bd8a7aebbb439c251dfb'\
          '6a424f00e535ac4df61ec19745b6f10e'\
          '893abd9afa775903324dbd6028f4e78f'\
          '784b99d7ada0d67e5233108dbd76702f'\
          '4b168087cfc4ec65494d6ca8aba858fe'\
          'badabd9afa775903324dbd6028f4e78f'\
          '784ba608a87f51bdf7532b4b80fa95ea'\
          'dfdf1bf8b0cbb58a7d3939c9f11c12e7'\
          '1c85bd9afa775903324dbd6028f4e78f'\
          '784bbdd4086c019f5d388453c6d93475'\
          'd39a576572baff75612c321b46a35a53'\
          '29b1bd9afa775903324dbd6028f4e78f'\
          '784bcb994b400590b66cbf55fc663555'\
          'caf0d4f1ce267464d0452c2361e05ee1'\
          'cd50bd9afa775903324dbd6028f4e78f'\
          '784bd6ee8db782e36caffb4d9f820790'\
          '0487de930aabcc1d196fa455fbfd6f37'\
          '273dbd9afa775903324dbd6028f4e78f'\
          '784bdda0121dcf167db1e2622d10f454'\
          '701837ac6af304a03ec06b3027904988'\
          'c56bbd9afa775903324dbd6028f4e78f'\
          '784be42572afac720f5d4a1c7aaaf802'\
          'f094daceb682f4e92783b2bb3fa00862'\
          'af7fbd9afa775903324dbd6028f4e78f'\
          '784be6236dc1ee074c077c7a1c9b3965'\
          '947430847be125f7aeb71d91a128133a'\
          'ea7fbd9afa775903324dbd6028f4e78f'\
          '784bef87be89a413657de8721498552c'\
          'f9e0f3c1f71bc62dfa63b9f25bbc66e8'\
          '6494bd9afa775903324dbd6028f4e78f'\
          '784bf5e892dd6ec4c2defa4a495c0921'\
          '9b621379b64da3d1b2e34adf4b5f1102'\
          'bd39bd9afa775903324dbd6028f4e78f'\
          '784bd4241190cd5a369d8c344c660e24'\
          'f3027fb8e7064fab33770e93fa765ffb'\
          '152ebd9afa775903324dbd6028f4e78f'\
          '784b23142e14424fb3ff4efc75d00b63'\
          '867727841aba5005149070ee2417df8a'\
          'b799bd9afa775903324dbd6028f4e78f'\
          '784b91721aa76266b5bb2f8009f11885'\
          '10a36e54afd56e967387ea7d0b114d78'\
          '2089bd9afa775903324dbd6028f4e78f'\
          '784bdc8aff7faa9d1a00a3e32eefbf89'\
          '9b3059cbb313a48b82fa9c8d931fd58f'\
          'b69dbd9afa775903324dbd6028f4e78f'\
          '784b9959ed4e05e548b59f219308a455'\
          '63ea85bb224c1ad96dec0e96c0e71ffc'\
          'cd81bd9afa775903324dbd6028f4e78f'\
          '784b47b31a1c7867644b2ee8093b2d5f'\
          'be21e21f77c1617a2c08812f57ace085'\
          '0e9fbd9afa775903324dbd6028f4e78f'\
          '784bfabc379df395e6f52472b44fa508'\
          '2f9f0e0da480f05198c66814b7055b03'\
          'f446bd9afa775903324dbd6028f4e78f'\
          '784be37ff3fc0eff20bfc1c060a4bf56'\
          '885e1efd55a8e9ce3c5f4869444cacff'\
          'ad0bbd9afa775903324dbd6028f4e78f'\
          '784b4cdae3920a512c9c052a8b4aba90'\
          '96969b0a0197b614031e4c64a5d898cb'\
          '09b9bd9afa775903324dbd6028f4e78f'\
          '784b5b89f1aa2435a03d18d9b203d17f'\
          'b4fba4f8f5076cf1f9b8d6d9b8262222'\
          '35c1bd9afa775903324dbd6028f4e78f'\
          '784b007f4c95125713b112093e21663e'\
          '2d23e3c1ae9ce4b5de0d58a297332336'\
          'a2d8bd9afa775903324dbd6028f4e78f'\
          '784be060da09561ae00dcfb1769d6e8e'\
          '846868a1e99a54b14aa5d0689f2840ce'\
          'c6dfbd9afa775903324dbd6028f4e78f'\
          '784b48f4584de1c5ec650c25e6c62363'\
          '5ce101bd82617fc400d4150f0aee2355'\
          'b4cabd9afa775903324dbd6028f4e78f'\
          '784baf79b14064601bc0987d4747af1e'\
          '914a228c05d622ceda03b7a4f67014fe'\
          'e767bd9afa775903324dbd6028f4e78f'\
          '784bc3d65e174d47d3772cb431ea599b'\
          'ba76b8670bfaa51081895796432e2ef6'\
          '461fbd9afa775903324dbd6028f4e78f'\
          '784b1e918f170a796b4b0b1400bb9bda'\
          'e75be1cf86705c2d0fc8fb9dd0c5016b'\
          '933bbd9afa775903324dbd6028f4e78f'\
          '784b66d0803e2550d9e790829ae1b5f8'\
          '1547cc9bfbe69b51817068ecb5dabb7a'\
          '89fcbd9afa775903324dbd6028f4e78f'\
          '784b284153e7d04a9f187e5c3dbfe17b'\
          '2672ad2fbdd119f27bec789417b79198'\
          '53ecbd9afa775903324dbd6028f4e78f'\
          '784bedd2cb55726e10abedec9de8ca5d'\
          'ed289ad793ab3b6919d163c875fec120'\
          '9cd5bd9afa775903324dbd6028f4e78f'\
          '784b90aec5c4995674a849c1d1384463'\
          'f3b02b5aa625a5c320fc4fe7d9bb58a6'\
          '2398'


def sha256_esl_builder(initList: []):
    """
    Creates an EfiSignatureList of type EFI_CERT_SHA256 from a list of (hash, sigowner) tuples

    initList is [ (hashString1, ownerGuidString1), ...]
    """

    Esl = EfiSignatureList(typeguid=EfiSignatureDataFactory.EFI_CERT_SHA256_GUID)
    SignatureSize = EfiSignatureDataEfiCertSha256.STATIC_STRUCT_SIZE if initList else 0
    Esl.AddSignatureHeader(SigHeader=None, SigSize=SignatureSize)
    for entry in initList:
        (hashStr, ownerGuidStr) = entry
        hashBytes = bytes.fromhex(hashStr)
        Esl.AddSignatureData(EfiSignatureDataEfiCertSha256(digest=hashBytes,
                                                           sigowner=uuid.UUID(ownerGuidStr)))
    return Esl


def x509_esl_builder(initList: []):
    """
    Creates an EfiSignatureList of type EFI_CERT_X509 from a tuple containing x509 and signatureowner hex strings

    initList is [ (hashString1, ownerGuidString1), ...]
    NOTE: x509 EfiSignatureLists only support 1 data element, so initList should either be empty or 1 element.
    The interface is a List for consistency with other esl_builders
    """

    listSize = len(initList)

    if (listSize > 1):
        raise Exception("initList length > 1, unsupported by type")

    Esl = EfiSignatureList(typeguid=EfiSignatureDataFactory.EFI_CERT_X509_GUID)

    if (listSize == 0):
        Esl.AddSignatureHeader(SigHeader=None, SigSize=0)
    else:
        (certHexStr, ownerGuidStr) = initList[0]
        certBytes = bytes.fromhex(certHexStr)
        Esl.AddSignatureHeader(SigHeader=None, SigSize=EfiSignatureDataEfiCertX509.STATIC_STRUCT_SIZE + len(certBytes))
        Esl.AddSignatureData(EfiSignatureDataEfiCertX509(cert=certBytes,
                                                         sigowner=uuid.UUID(ownerGuidStr)))
    return Esl


def basicEfiSignatureDatabase():
    owner = uuid.uuid4().hex
    certInput = [(DEBIAN_CERT, owner)]
    cert = x509_esl_builder(certInput)

    hashEslInput = [
        (HASHSTR[8], owner),
        (HASHSTR[0], owner),
        (HASHSTR[0], owner),
        (HASHSTR[5], owner),
        (HASHSTR[5], owner),
        (HASHSTR[5], owner),
        (HASHSTR[1], owner),
        (HASHSTR[2], owner),
        (HASHSTR[1], owner),
        (HASHSTR[2], owner)
    ]

    hashes = sha256_esl_builder(hashEslInput)

    EslList = [
        cert, hashes
    ]

    return EfiSignatureDatabase(EslList=EslList)


def BootHoleData():
    """ Returns a tuple of the (rawData, expectedDupes, expectedCanonical) for the BootHole dbx generated by EDK2 """

    owner = '77fa9abd-0359-4d32-bd60-28f4e78f784b'

    # First prepare EfiSignatureDatabase that is unsorted, unmerged, and contains duplicates
    originalHashes = [
        ('80B4D96931BF0D02FD91A61E19D14F1DA452E66DB2408CA8604D411F92659F0A', owner),
        ('F52F83A3FA9CFBD6920F722824DBE4034534D25B8507246B3B957DAC6E1BCE7A', owner),
        ('C5D9D8A186E2C82D09AFAA2A6F7F2E73870D3E64F72C4E08EF67796A840F0FBD', owner),
        ('363384D14D1F2E0B7815626484C459AD57A318EF4396266048D058C5A19BBF76', owner),
        ('1AEC84B84B6C65A51220A9BE7181965230210D62D6D33C48999C6B295A2B0A06', owner),
        ('E6CA68E94146629AF03F69C2F86E6BEF62F930B37C6FBCC878B78DF98C0334E5', owner),
        ('C3A99A460DA464A057C3586D83CEF5F4AE08B7103979ED8932742DF0ED530C66', owner),
        ('58FB941AEF95A25943B3FB5F2510A0DF3FE44C58C95E0AB80487297568AB9771', owner),
        ('5391C3A2FB112102A6AA1EDC25AE77E19F5D6F09CD09EEB2509922BFCD5992EA', owner),
        ('D626157E1D6A718BC124AB8DA27CBB65072CA03A7B6B257DBDCBBD60F65EF3D1', owner),
        ('D063EC28F67EBA53F1642DBF7DFF33C6A32ADD869F6013FE162E2C32F1CBE56D', owner),
        ('29C6EB52B43C3AA18B2CD8ED6EA8607CEF3CFAE1BAFE1165755CF2E614844A44', owner),
        ('90FBE70E69D633408D3E170C6832DBB2D209E0272527DFB63D49D29572A6F44C', owner),
        ('075EEA060589548BA060B2FEED10DA3C20C7FE9B17CD026B94E8A683B8115238', owner),
        ('07E6C6A858646FB1EFC67903FE28B116011F2367FE92E6BE2B36999EFF39D09E', owner),
        ('09DF5F4E511208EC78B96D12D08125FDB603868DE39F6F72927852599B659C26', owner),
        ('0BBB4392DAAC7AB89B30A4AC657531B97BFAAB04F90B0DAFE5F9B6EB90A06374', owner),
        ('0C189339762DF336AB3DD006A463DF715A39CFB0F492465C600E6C6BD7BD898C', owner),
        ('0D0DBECA6F29ECA06F331A7D72E4884B12097FB348983A2A14A0D73F4F10140F', owner),
        ('0DC9F3FB99962148C3CA833632758D3ED4FC8D0B0007B95B31E6528F2ACD5BFC', owner),
        ('106FACEACFECFD4E303B74F480A08098E2D0802B936F8EC774CE21F31686689C', owner),
        ('174E3A0B5B43C6A607BBD3404F05341E3DCF396267CE94F8B50E2E23A9DA920C', owner),
        ('18333429FF0562ED9F97033E1148DCEEE52DBE2E496D5410B5CFD6C864D2D10F', owner),
        ('2B99CF26422E92FE365FBF4BC30D27086C9EE14B7A6FFF44FB2F6B9001699939', owner),
        ('2BBF2CA7B8F1D91F27EE52B6FB2A5DD049B85A2B9B529C5D6662068104B055F8', owner),
        ('2C73D93325BA6DCBE589D4A4C63C5B935559EF92FBF050ED50C4E2085206F17D', owner),
        ('2E70916786A6F773511FA7181FAB0F1D70B557C6322EA923B2A8D3B92B51AF7D', owner),
        ('306628FA5477305728BA4A467DE7D0387A54F569D3769FCE5E75EC89D28D1593', owner),
        ('3608EDBAF5AD0F41A414A1777ABF2FAF5E670334675EC3995E6935829E0CAAD2', owner),
        ('3841D221368D1583D75C0A02E62160394D6C4E0A6760B6F607B90362BC855B02', owner),
        ('3FCE9B9FDF3EF09D5452B0F95EE481C2B7F06D743A737971558E70136ACE3E73', owner),
        ('4397DACA839E7F63077CB50C92DF43BC2D2FB2A8F59F26FC7A0E4BD4D9751692', owner),
        ('47CC086127E2069A86E03A6BEF2CD410F8C55A6D6BDB362168C31B2CE32A5ADF', owner),
        ('518831FE7382B514D03E15C621228B8AB65479BD0CBFA3C5C1D0F48D9C306135', owner),
        ('5AE949EA8855EB93E439DBC65BDA2E42852C2FDF6789FA146736E3C3410F2B5C', owner),
        ('6B1D138078E4418AA68DEB7BB35E066092CF479EEB8CE4CD12E7D072CCB42F66', owner),
        ('6C8854478DD559E29351B826C06CB8BFEF2B94AD3538358772D193F82ED1CA11', owner),
        ('6F1428FF71C9DB0ED5AF1F2E7BBFCBAB647CC265DDF5B293CDB626F50A3A785E', owner),
        ('71F2906FD222497E54A34662AB2497FCC81020770FF51368E9E3D9BFCBFD6375', owner),
        ('726B3EB654046A30F3F83D9B96CE03F670E9A806D1708A0371E62DC49D2C23C1', owner),
        ('72E0BD1867CF5D9D56AB158ADF3BDDBC82BF32A8D8AA1D8C5E2F6DF29428D6D8', owner),
        ('7827AF99362CFAF0717DADE4B1BFE0438AD171C15ADDC248B75BF8CAA44BB2C5', owner),
        ('81A8B965BB84D3876B9429A95481CC955318CFAA1412D808C8A33BFD33FFF0E4', owner),
        ('82DB3BCEB4F60843CE9D97C3D187CD9B5941CD3DE8100E586F2BDA5637575F67', owner),
        ('895A9785F617CA1D7ED44FC1A1470B71F3F1223862D9FF9DCC3AE2DF92163DAF', owner),
        ('8AD64859F195B5F58DAFAA940B6A6167ACD67A886E8F469364177221C55945B9', owner),
        ('8BF434B49E00CCF71502A2CD900865CB01EC3B3DA03C35BE505FDF7BD563F521', owner),
        ('8D8EA289CFE70A1C07AB7365CB28EE51EDD33CF2506DE888FBADD60EBF80481C', owner),
        ('9998D363C491BE16BD74BA10B94D9291001611736FDCA643A36664BC0F315A42', owner),
        ('9E4A69173161682E55FDE8FEF560EB88EC1FFEDCAF04001F66C0CAF707B2B734', owner),
        ('A6B5151F3655D3A2AF0D472759796BE4A4200E5495A7D869754C4848857408A7', owner),
        ('A7F32F508D4EB0FEAD9A087EF94ED1BA0AEC5DE6F7EF6FF0A62B93BEDF5D458D', owner),
        ('AD6826E1946D26D3EAF3685C88D97D85DE3B4DCB3D0EE2AE81C70560D13C5720', owner),
        ('AEEBAE3151271273ED95AA2E671139ED31A98567303A332298F83709A9D55AA1', owner),
        ('AFE2030AFB7D2CDA13F9FA333A02E34F6751AFEC11B010DBCD441FDF4C4002B3', owner),
        ('B54F1EE636631FAD68058D3B0937031AC1B90CCB17062A391CCA68AFDBE40D55', owner),
        ('B8F078D983A24AC433216393883514CD932C33AF18E7DD70884C8235F4275736', owner),
        ('B97A0889059C035FF1D54B6DB53B11B9766668D9F955247C028B2837D7A04CD9', owner),
        ('BC87A668E81966489CB508EE805183C19E6ACD24CF17799CA062D2E384DA0EA7', owner),
        ('C409BDAC4775ADD8DB92AA22B5B718FB8C94A1462C1FE9A416B95D8A3388C2FC', owner),
        ('C617C1A8B1EE2A811C28B5A81B4C83D7C98B5B0C27281D610207EBE692C2967F', owner),
        ('C90F336617B8E7F983975413C997F10B73EB267FD8A10CB9E3BDBFC667ABDB8B', owner),
        ('CB6B858B40D3A098765815B592C1514A49604FAFD60819DA88D7A76E9778FEF7', owner),
        ('CE3BFABE59D67CE8AC8DFD4A16F7C43EF9C224513FBC655957D735FA29F540CE', owner),
        ('D8CBEB9735F5672B367E4F96CDC74969615D17074AE96C724D42CE0216F8F3FA', owner),
        ('E92C22EB3B5642D65C1EC2CAF247D2594738EEBB7FB3841A44956F59E2B0D1FA', owner),
        ('FDDD6E3D29EA84C7743DAD4A1BDBC700B5FEC1B391F932409086ACC71DD6DBD8', owner),
        ('FE63A84F782CC9D3FCF2CCF9FC11FBD03760878758D26285ED12669BDC6E6D01', owner),
        ('FECFB232D12E994B6D485D2C7167728AA5525984AD5CA61E7516221F079A1436', owner),
        ('CA171D614A8D7E121C93948CD0FE55D39981F9D11AA96E03450A415227C2C65B', owner),
        ('55B99B0DE53DBCFE485AA9C737CF3FB616EF3D91FAB599AA7CAB19EDA763B5BA', owner),
        ('77DD190FA30D88FF5E3B011A0AE61E6209780C130B535ECB87E6F0888A0B6B2F', owner),
        ('C83CB13922AD99F560744675DD37CC94DCAD5A1FCBA6472FEE341171D939E884', owner),
        ('3B0287533E0CC3D0EC1AA823CBF0A941AAD8721579D1C499802DD1C3A636B8A9', owner),
        ('939AEEF4F5FA51E23340C3F2E49048CE8872526AFDF752C3A7F3A3F2BC9F6049', owner),
        ('64575BD912789A2E14AD56F6341F52AF6BF80CF94400785975E9F04E2D64D745', owner),
        ('45C7C8AE750ACFBB48FC37527D6412DD644DAED8913CCD8A24C94D856967DF8E', owner)
    ]
    originalEsl = sha256_esl_builder(originalHashes)

    cert1 = [(UBUNTU_CERT, owner)]
    cert1_esl = x509_esl_builder(cert1)

    cert2 = [(DEBIAN_CERT, owner)]
    cert2_esl = x509_esl_builder(cert2)

    newHashes = [
        ('81D8FB4C9E2E7A8225656B4B8273B7CBA4B03EF2E9EB20E0A0291624ECA1BA86', owner),
        ('B92AF298DC08049B78C77492D6551B710CD72AADA3D77BE54609E43278EF6E4D', owner),
        ('E19DAE83C02E6F281358D4EBD11D7723B4F5EA0E357907D5443DECC5F93C1E9D', owner),
        ('39DBC2288EF44B5F95332CB777E31103E840DBA680634AA806F5C9B100061802', owner),
        ('32F5940CA29DD812A2C145E6FC89646628FFCC7C7A42CAE512337D8D29C40BBD', owner),
        ('10D45FCBA396AEF3153EE8F6ECAE58AFE8476A280A2026FC71F6217DCF49BA2F', owner),
        ('4B8668A5D465BCDD9000AA8DFCFF42044FCBD0AECE32FC7011A83E9160E89F09', owner),
        ('89F3D1F6E485C334CD059D0995E3CDFDC00571B1849854847A44DC5548E2DCFB', owner),
        ('C9EC350406F26E559AFFB4030DE2EBDE5435054C35A998605B8FCF04972D8D55', owner),
        ('B3E506340FBF6B5786973393079F24B66BA46507E35E911DB0362A2ACDE97049', owner),
        ('9F1863ED5717C394B42EF10A6607B144A65BA11FB6579DF94B8EB2F0C4CD60C1', owner),
        ('DD59AF56084406E38C63FBE0850F30A0CD1277462A2192590FB05BC259E61273', owner),
        ('DBAF9E056D3D5B38B68553304ABC88827EBC00F80CB9C7E197CDBC5822CD316C', owner),
        ('65F3C0A01B8402D362B9722E98F75E5E991E6C186E934F7B2B2E6BE6DEC800EC', owner),
        ('5B248E913D71853D3DA5AEDD8D9A4BC57A917126573817FB5FCB2D86A2F1C886', owner),
        ('2679650FE341F2CF1EA883460B3556AAAF77A70D6B8DC484C9301D1B746CF7B5', owner),
        ('BB1DD16D530008636F232303A7A86F3DFF969F848815C0574B12C2D787FEC93F', owner),
        ('0CE02100F67C7EF85F4EED368F02BF7092380A3C23CA91FD7F19430D94B00C19', owner),
        ('95049F0E4137C790B0D2767195E56F73807D123ADCF8F6E7BF2D4D991D305F89', owner),
        ('02E6216ACAEF6401401FA555ECBED940B1A5F2569AED92956137AE58482EF1B7', owner),
        ('6EFEFE0B5B01478B7B944C10D3A8ACA2CCA4208888E2059F8A06CB5824D7BAB0', owner),
        ('9D00AE4CD47A41C783DC48F342C076C2C16F3413F4D2DF50D181CA3BB5AD859D', owner),
        ('D8D4E6DDF6E42D74A6A536EA62FD1217E4290B145C9E5C3695A31B42EFB5F5A4', owner),
        ('F277AF4F9BDC918AE89FA35CC1B34E34984C04AE9765322C3CB049574D36509C', owner),
        ('0DC24C75EB1AEF56B9F13AB9DE60E2ECA1C4510034E290BBB36CF60A549B234C', owner),
        ('835881F2A5572D7059B5C8635018552892E945626F115FC9CA07ACF7BDE857A4', owner),
        ('BADFF5E4F0FEA711701CA8FB22E4C43821E31E210CF52D1D4F74DD50F1D039BC', owner),
        ('C452AB846073DF5ACE25CCA64D6B7A09D906308A1A65EB5240E3C4EBCAA9CC0C', owner),
        ('F1863EC8B7F43F94AD14FB0B8B4A69497A8C65ECBC2A55E0BB420E772B8CDC91', owner),
        ('7BC9CB5463CE0F011FB5085EB8BA77D1ACD283C43F4A57603CC113F22CEBC579', owner),
        ('E800395DBE0E045781E8005178B4BAF5A257F06E159121A67C595F6AE22506FD', owner),
        ('1CB4DCCAF2C812CFA7B4938E1371FE2B96910FE407216FD95428672D6C7E7316', owner),
        ('3ECE27CBB3EC4438CCE523B927C4F05FDC5C593A3766DB984C5E437A3FF6A16B', owner),
        ('68EE4632C7BE1C66C83E89DD93EAEE1294159ABF45B4C2C72D7DC7499AA2A043', owner),
        ('E24B315A551671483D8B9073B32DE11B4DE1EB2EAB211AFD2D9C319FF55E08D0', owner),
        ('E7C20B3AB481EC885501ECA5293781D84B5A1AC24F88266B5270E7ECB4AA2538', owner),
        ('7EAC80A915C84CD4AFEC638904D94EB168A8557951A4D539B0713028552B6B8C', owner),
        ('E7681F153121EA1E67F74BBCB0CDC5E502702C1B8CC55FB65D702DFBA948B5F4', owner),
        ('DCCC3CE1C00EE4B0B10487D372A0FA47F5C26F57A359BE7B27801E144EACBAC4', owner),
        ('0257FF710F2A16E489B37493C07604A7CDA96129D8A8FD68D2B6AF633904315D', owner),
        ('3A91F0F9E5287FA2994C7D930B2C1A5EE14CE8E1C8304AE495ADC58CC4453C0C', owner),
        ('495300790E6C9BF2510DABA59DB3D57E9D2B85D7D7640434EC75BAA3851C74E5', owner),
        ('81A8B2C9751AEB1FABA7DBDE5EE9691DC0EAEE2A31C38B1491A8146756A6B770', owner),
        ('8E53EFDC15F852CEE5A6E92931BC42E6163CD30FF649CCA7E87252C3A459960B', owner),
        ('992D359AA7A5F789D268B94C11B9485A6B1CE64362B0EDB4441CCC187C39647B', owner),
        ('9FA4D5023FD43ECAFF4200BA7E8D4353259D2B7E5E72B5096EFF8027D66D1043', owner),
        ('D372C0D0F4FDC9F52E9E1F23FC56EE72414A17F350D0CEA6C26A35A6C3217A13', owner),
        ('5C5805196A85E93789457017D4F9EB6828B97C41CB9BA6D3DC1FCC115F527A55', owner),
        ('804E354C6368BB27A90FAE8E498A57052B293418259A019C4F53A2007254490F', owner),
        ('03F64A29948A88BEFFDB035E0B09A7370CCF0CD9CE6BCF8E640C2107318FAB87', owner),
        ('05D87E15713454616F5B0ED7849AB5C1712AB84F02349478EC2A38F970C01489', owner),
        ('06EB5BADD26E4FAE65F9A42358DEEF7C18E52CC05FBB7FC76776E69D1B982A14', owner),
        ('08BB2289E9E91B4D20FF3F1562516AB07E979B2C6CEFE2AB70C6DFC1199F8DA5', owner),
        ('0928F0408BF725E61D67D87138A8EEBC52962D2847F16E3587163B160E41B6AD', owner),
        ('09F98AA90F85198C0D73F89BA77E87EC6F596C491350FB8F8BBA80A62FBB914B', owner),
        ('0A75EA0B1D70EAA4D3F374246DB54FC7B43E7F596A353309B9C36B4FD975725E', owner),
        ('0C51D7906FC4931149765DA88682426B2CFE9E6AA4F27253EAB400111432E3A7', owner),
        ('0FA3A29AD05130D7FE5BF4D2596563CDED1D874096AACC181069932A2E49519A', owner),
        ('147730B42F11FE493FE902B6251E97CD2B6F34D36AF59330F11D02A42F940D07', owner),
        ('148FE18F715A9FCFE1A444CE0FFF7F85869EB422330DC04B314C0F295D6DA79E', owner),
        ('1B909115A8D473E51328A87823BD621CE655DFAE54FA2BFA72FDC0298611D6B8', owner),
        ('1D8B58C1FDB8DA8B33CCEE1E5F973AF734D90EF317E33F5DB1573C2BA088A80C', owner),
        ('1F179186EFDF5EF2DE018245BA0EAE8134868601BA0D35FF3D9865C1537CED93', owner),
        ('270C84B29D86F16312B06AAAE4EBB8DFF8DE7D080D825B8839FF1766274EFF47', owner),
        ('29CCA4544EA330D61591C784695C149C6B040022AC7B5B89CBD72800D10840EA', owner),
        ('2B2298EAA26B9DC4A4558AE92E7BB0E4F85CF34BF848FDF636C0C11FBEC49897', owner),
        ('2DCF8E8D817023D1E8E1451A3D68D6EC30D9BED94CBCB87F19DDC1CC0116AC1A', owner),
        ('311A2AC55B50C09B30B3CC93B994A119153EEEAC54EF892FC447BBBD96101AA1', owner),
        ('32AD3296829BC46DCFAC5EDDCB9DBF2C1EED5C11F83B2210CF9C6E60C798D4A7', owner),
        ('340DA32B58331C8E2B561BAF300CA9DFD6B91CD2270EE0E2A34958B1C6259E85', owner),
        ('362ED31D20B1E00392281231A96F0A0ACFDE02618953E695C9EF2EB0BAC37550', owner),
        ('367A31E5838831AD2C074647886A6CDFF217E6B1BA910BFF85DC7A87AE9B5E98', owner),
        ('3765D769C05BF98B427B3511903B2137E8A49B6F859D0AF159ED6A86786AA634', owner),
        ('386D695CDF2D4576E01BCACCF5E49E78DA51AF9955C0B8FA7606373B007994B3', owner),
        ('3A4F74BEAFAE2B9383AD8215D233A6CF3D057FB3C7E213E897BEEF4255FAEE9D', owner),
        ('3AE76C45CA70E9180C1559981F42622DD251BCA1FBE6B901C52EC11673B03514', owner),
        ('3BE8E7EB348D35C1928F19C769846788991641D1F6CF09514CA10269934F7359', owner),
        ('3E3926F0B8A15AD5A14167BB647A843C3D4321E35DBC44DCE8C837417F2D28B0', owner),
        ('400AC66D59B7B094A9E30B01A6BD013AFF1D30570F83E7592F421DBE5FF4BA8F', owner),
        ('4185821F6DAB5BA8347B78A22B5F9A0A7570CA5C93A74D478A793D83BAC49805', owner),
        ('41D1EEB177C0324E17DD6557F384E532DE0CF51A019A446B01EFB351BC259D77', owner),
        ('45876B4DD861D45B3A94800774027A5DB45A48B2A729410908B6412F8A87E95D', owner),
        ('4667BF250CD7C1A06B8474C613CDB1DF648A7F58736FBF57D05D6F755DAB67F4', owner),
        ('47FF1B63B140B6FC04ED79131331E651DA5B2E2F170F5DAEF4153DC2FBC532B1', owner),
        ('47FF1B63B140B6FC04ED79131331E651DA5B2E2F170F5DAEF4153DC2FBC532B1', owner),
        ('57E6913AFACC5222BD76CDAF31F8ED88895464255374EF097A82D7F59AD39596', owner),
        ('5890FA227121C76D90ED9E63C87E3A6533EEA0F6F0A1A23F1FC445139BC6BCDF', owner),
        ('5D1E9ACBBB4A7D024B6852DF025970E2CED66FF622EE019CD0ED7FD841CCAD02', owner),
        ('61341E07697978220EA61E85DCD2421343F2C1BF35CC5B8D0AD2F0226F391479', owner),
        ('61CEC4A377BF5902C0FEAEE37034BF97D5BC6E0615E23A1CDFBAE6E3F5FB3CFD', owner),
        ('631F0857B41845362C90C6980B4B10C4B628E23DBE24B6E96C128AE3DCB0D5AC', owner),
        ('65B2E7CC18D903C331DF1152DF73CA0DC932D29F17997481C56F3087B2DD3147', owner),
        ('66AA13A0EDC219384D9C425D3927E6ED4A5D1940C5E7CD4DAC88F5770103F2F1', owner),
        ('6873D2F61C29BD52E954EEFF5977AA8367439997811A62FF212C948133C68D97', owner),
        ('6DBBEAD23E8C860CF8B47F74FBFCA5204DE3E28B881313BB1D1ECCDC4747934E', owner),
        ('6DEAD13257DFC3CCC6A4B37016BA91755FE9E0EC1F415030942E5ABC47F07C88', owner),
        ('70A1450AF2AD395569AD0AFEB1D9C125324EE90AEC39C258880134D4892D51AB', owner),
        ('72C26F827CEB92989798961BC6AE748D141E05D3EBCFB65D9041B266C920BE82', owner),
        ('781764102188A8B4B173D4A8F5EC94D828647156097F99357A581E624B377509', owner),
        ('788383A4C733BB87D2BF51673DC73E92DF15AB7D51DC715627AE77686D8D23BC', owner),
        ('78B4EDCAABC8D9093E20E217802CAEB4F09E23A3394C4ACC6E87E8F35395310F', owner),
        ('7F49CCB309323B1C7AB11C93C955B8C744F0A2B75C311F495E18906070500027', owner),
        ('82ACBA48D5236CCFF7659AFC14594DEE902BD6082EF1A30A0B9B508628CF34F4', owner),
        ('894D7839368F3298CC915AE8742EF330D7A26699F459478CF22C2B6BB2850166', owner),
        ('8C0349D708571AE5AA21C11363482332073297D868F29058916529EFC520EF70', owner),
        ('8D93D60C691959651476E5DC464BE12A85FA5280B6F524D4A1C3FCC9D048CFAD', owner),
        ('9063F5FBC5E57AB6DE6C9488146020E172B176D5AB57D4C89F0F600E17FE2DE2', owner),
        ('91656AA4EF493B3824A0B7263248E4E2D657A5C8488D880CB65B01730932FB53', owner),
        ('91971C1497BF8E5BC68439ACC48D63EBB8FAABFD764DCBE82F3BA977CAC8CF6A', owner),
        ('947078F97C6196968C3AE99C9A5D58667E86882CF6C8C9D58967A496BB7AF43C', owner),
        ('96E4509450D380DAC362FF8E295589128A1F1CE55885D20D89C27BA2A9D00909', owner),
        ('9783B5EE4492E9E891C655F1F48035959DAD453C0E623AF0FE7BF2C0A57885E3', owner),
        ('97A51A094444620DF38CD8C6512CAC909A75FD437AE1E4D22929807661238127', owner),
        ('97A8C5BA11D61FEFBB5D6A05DA4E15BA472DC4C6CD4972FC1A035DE321342FE4', owner),
        ('992820E6EC8C41DAAE4BD8AB48F58268E943A670D35CA5E2BDCD3E7C4C94A072', owner),
        ('992D359AA7A5F789D268B94C11B9485A6B1CE64362B0EDB4441CCC187C39647B', owner),
        ('9954A1A99D55E8B189AB1BCA414B91F6A017191F6C40A86B6F3EF368DD860031', owner),
        ('9BAF4F76D76BF5D6A897BFBD5F429BA14D04E08B48C3EE8D76930A828FFF3891', owner),
        ('9C259FCB301D5FC7397ED5759963E0EF6B36E42057FD73046E6BD08B149F751C', owner),
        ('9DD2DCB72F5E741627F2E9E03AB18503A3403CF6A904A479A4DB05D97E2250A9', owner),
        ('9ED33F0FBC180BC032F8909CA2C4AB3418EDC33A45A50D2521A3B5876AA3EA2C', owner),
        ('A4D978B7C4BDA15435D508F8B9592EC2A5ADFB12EA7BAD146A35ECB53094642F', owner),
        ('A924D3CAD6DA42B7399B96A095A06F18F6B1ABA5B873B0D5F3A0EE2173B48B6C', owner),
        ('AD3BE589C0474E97DE5BB2BF33534948B76BB80376DFDC58B1FED767B5A15BFC', owner),
        ('B8D6B5E7857B45830E017C7BE3D856ADEB97C7290EB0665A3D473A4BEB51DCF3', owner),
        ('B93F0699598F8B20FA0DACC12CFCFC1F2568793F6E779E04795E6D7C22530F75', owner),
        ('BB01DA0333BB639C7E1C806DB0561DC98A5316F22FEF1090FB8D0BE46DAE499A', owner),
        ('BC75F910FF320F5CB5999E66BBD4034F4AE537A42FDFEF35161C5348E366E216', owner),
        ('BDD01126E9D85710D3FE75AF1CC1702A29F081B4F6FDF6A2B2135C0297A9CEC5', owner),
        ('BE435DF7CD28AA2A7C8DB4FC8173475B77E5ABF392F76B7C76FA3F698CB71A9A', owner),
        ('BEF7663BE5EA4DBFD8686E24701E036F4C03FB7FCD67A6C566ED94CE09C44470', owner),
        ('C2469759C1947E14F4B65F72A9F5B3AF8B6F6E727B68BB0D91385CBF42176A8A', owner),
        ('C3505BF3EC10A51DACE417C76B8BD10939A065D1F34E75B8A3065EE31CC69B96', owner),
        ('C42D11C70CCF5E8CF3FB91FDF21D884021AD836CA68ADF2CBB7995C10BF588D4', owner),
        ('C452AB846073DF5ACE25CCA64D6B7A09D906308A1A65EB5240E3C4EBCAA9CC0C', owner),
        ('C69D64A5B839E41BA16742527E17056A18CE3C276FD26E34901A1BC7D0E32219', owner),
        ('CB340011AFEB0D74C4A588B36EBAA441961608E8D2FA80DCA8C13872C850796B', owner),
        ('CC8EEC6EB9212CBF897A5ACE7E8ABEECE1079F1A6DEF0A789591CB1547F1F084', owner),
        ('CF13A243C1CD2E3C8CEB7E70100387CECBFB830525BBF9D0B70C79ADF3E84128', owner),
        ('D89A11D16C488DD4FBBC541D4B07FAF8670D660994488FE54B1FBFF2704E4288', owner),
        ('D9668AB52785086786C134B5E4BDDBF72452813B6973229AB92AA1A54D201BF5', owner),
        ('DA3560FD0C32B54C83D4F2FF869003D2089369ACF2C89608F8AFA7436BFA4655', owner),
        ('DF02AAB48387A9E1D4C65228089CB6ABE196C8F4B396C7E4BBC395DE136977F6', owner),
        ('DF91AC85A94FCD0CFB8155BD7CBEFAAC14B8C5EE7397FE2CC85984459E2EA14E', owner),
        ('E051B788ECBAEDA53046C70E6AF6058F95222C046157B8C4C1B9C2CFC65F46E5', owner),
        ('E051B788ECBAEDA53046C70E6AF6058F95222C046157B8C4C1B9C2CFC65F46E5', owner),
        ('E36DFC719D2114C2E39AEA88849E2845AB326F6F7FE74E0E539B7E54D81F3631', owner),
        ('E39891F48BBCC593B8ED86CE82CE666FC1145B9FCBFD2B07BAD0A89BF4C7BFBF', owner),
        ('E6856F137F79992DC94FA2F43297EC32D2D9A76F7BE66114C6A13EFC3BCDF5C8', owner),
        ('EAFF8C85C208BA4D5B6B8046F5D6081747D779BADA7768E649D047FF9B1F660C', owner),
        ('EE83A566496109A74F6AC6E410DF00BB29A290E0021516AE3B8A23288E7E2E72', owner),
        ('EED7E0EFF2ED559E2A79EE361F9962AF3B1E999131E30BB7FD07546FAE0A7267', owner),
        ('F1B4F6513B0D544A688D13ADC291EFA8C59F420CA5DCB23E0B5A06FA7E0D083D', owner),
        ('F2A16D35B554694187A70D40CA682959F4F35C2CE0EAB8FD64F7AC2AB9F5C24A', owner),
        ('F31FD461C5E99510403FC97C1DA2D8A9CBE270597D32BADF8FD66B77495F8D94', owner),
        ('F48E6DD8718E953B60A24F2CBEA60A9521DEAE67DB25425B7D3ACE3C517DD9B7', owner),
        ('C805603C4FA038776E42F263C604B49D96840322E1922D5606A9B0BBB5BFFE6F', owner),
        ('1F16078CCE009DF62EDB9E7170E66CAAE670BCE71B8F92D38280C56AA372031D', owner),
        ('37A480374DAF6202CE790C318A2BB8AA3797311261160A8E30558B7DEA78C7A6', owner),
        ('408B8B3DF5ABB043521A493525023175AB1261B1DE21064D6BF247CE142153B9', owner),
        ('540801DD345DC1C33EF431B35BF4C0E68BD319B577B9ABE1A9CFF1CBC39F548F', owner)
    ]
    newHash_esl = sha256_esl_builder(newHashes)

    # EDK2 appended cert1, cert2, and the partially de-duped new hashes to the original dbx
    esd = EfiSignatureDatabase(EslList=[originalEsl, cert1_esl, cert2_esl, newHash_esl])

    # Now prepare the duplicate data, these were the 4 dupes in the new list that were not de-duped by EDK2
    expected_dupes = [
        ('47FF1B63B140B6FC04ED79131331E651DA5B2E2F170F5DAEF4153DC2FBC532B1', owner),
        ('992D359AA7A5F789D268B94C11B9485A6B1CE64362B0EDB4441CCC187C39647B', owner),
        ('C452AB846073DF5ACE25CCA64D6B7A09D906308A1A65EB5240E3C4EBCAA9CC0C', owner),
        ('E051B788ECBAEDA53046C70E6AF6058F95222C046157B8C4C1B9C2CFC65F46E5', owner)
    ]
    expected_dupes_esl = sha256_esl_builder(expected_dupes)
    expected_dupes_esd = EfiSignatureDatabase(EslList=[expected_dupes_esl])

    # Now prepare the canonicalized data
    canonicalHashes = [
        ('0257FF710F2A16E489B37493C07604A7CDA96129D8A8FD68D2B6AF633904315D', owner),
        ('02E6216ACAEF6401401FA555ECBED940B1A5F2569AED92956137AE58482EF1B7', owner),
        ('03F64A29948A88BEFFDB035E0B09A7370CCF0CD9CE6BCF8E640C2107318FAB87', owner),
        ('05D87E15713454616F5B0ED7849AB5C1712AB84F02349478EC2A38F970C01489', owner),
        ('06EB5BADD26E4FAE65F9A42358DEEF7C18E52CC05FBB7FC76776E69D1B982A14', owner),
        ('075EEA060589548BA060B2FEED10DA3C20C7FE9B17CD026B94E8A683B8115238', owner),
        ('07E6C6A858646FB1EFC67903FE28B116011F2367FE92E6BE2B36999EFF39D09E', owner),
        ('08BB2289E9E91B4D20FF3F1562516AB07E979B2C6CEFE2AB70C6DFC1199F8DA5', owner),
        ('0928F0408BF725E61D67D87138A8EEBC52962D2847F16E3587163B160E41B6AD', owner),
        ('09DF5F4E511208EC78B96D12D08125FDB603868DE39F6F72927852599B659C26', owner),
        ('09F98AA90F85198C0D73F89BA77E87EC6F596C491350FB8F8BBA80A62FBB914B', owner),
        ('0A75EA0B1D70EAA4D3F374246DB54FC7B43E7F596A353309B9C36B4FD975725E', owner),
        ('0BBB4392DAAC7AB89B30A4AC657531B97BFAAB04F90B0DAFE5F9B6EB90A06374', owner),
        ('0C189339762DF336AB3DD006A463DF715A39CFB0F492465C600E6C6BD7BD898C', owner),
        ('0C51D7906FC4931149765DA88682426B2CFE9E6AA4F27253EAB400111432E3A7', owner),
        ('0CE02100F67C7EF85F4EED368F02BF7092380A3C23CA91FD7F19430D94B00C19', owner),
        ('0D0DBECA6F29ECA06F331A7D72E4884B12097FB348983A2A14A0D73F4F10140F', owner),
        ('0DC24C75EB1AEF56B9F13AB9DE60E2ECA1C4510034E290BBB36CF60A549B234C', owner),
        ('0DC9F3FB99962148C3CA833632758D3ED4FC8D0B0007B95B31E6528F2ACD5BFC', owner),
        ('0FA3A29AD05130D7FE5BF4D2596563CDED1D874096AACC181069932A2E49519A', owner),
        ('106FACEACFECFD4E303B74F480A08098E2D0802B936F8EC774CE21F31686689C', owner),
        ('10D45FCBA396AEF3153EE8F6ECAE58AFE8476A280A2026FC71F6217DCF49BA2F', owner),
        ('147730B42F11FE493FE902B6251E97CD2B6F34D36AF59330F11D02A42F940D07', owner),
        ('148FE18F715A9FCFE1A444CE0FFF7F85869EB422330DC04B314C0F295D6DA79E', owner),
        ('174E3A0B5B43C6A607BBD3404F05341E3DCF396267CE94F8B50E2E23A9DA920C', owner),
        ('18333429FF0562ED9F97033E1148DCEEE52DBE2E496D5410B5CFD6C864D2D10F', owner),
        ('1AEC84B84B6C65A51220A9BE7181965230210D62D6D33C48999C6B295A2B0A06', owner),
        ('1B909115A8D473E51328A87823BD621CE655DFAE54FA2BFA72FDC0298611D6B8', owner),
        ('1CB4DCCAF2C812CFA7B4938E1371FE2B96910FE407216FD95428672D6C7E7316', owner),
        ('1D8B58C1FDB8DA8B33CCEE1E5F973AF734D90EF317E33F5DB1573C2BA088A80C', owner),
        ('1F16078CCE009DF62EDB9E7170E66CAAE670BCE71B8F92D38280C56AA372031D', owner),
        ('1F179186EFDF5EF2DE018245BA0EAE8134868601BA0D35FF3D9865C1537CED93', owner),
        ('2679650FE341F2CF1EA883460B3556AAAF77A70D6B8DC484C9301D1B746CF7B5', owner),
        ('270C84B29D86F16312B06AAAE4EBB8DFF8DE7D080D825B8839FF1766274EFF47', owner),
        ('29C6EB52B43C3AA18B2CD8ED6EA8607CEF3CFAE1BAFE1165755CF2E614844A44', owner),
        ('29CCA4544EA330D61591C784695C149C6B040022AC7B5B89CBD72800D10840EA', owner),
        ('2B2298EAA26B9DC4A4558AE92E7BB0E4F85CF34BF848FDF636C0C11FBEC49897', owner),
        ('2B99CF26422E92FE365FBF4BC30D27086C9EE14B7A6FFF44FB2F6B9001699939', owner),
        ('2BBF2CA7B8F1D91F27EE52B6FB2A5DD049B85A2B9B529C5D6662068104B055F8', owner),
        ('2C73D93325BA6DCBE589D4A4C63C5B935559EF92FBF050ED50C4E2085206F17D', owner),
        ('2DCF8E8D817023D1E8E1451A3D68D6EC30D9BED94CBCB87F19DDC1CC0116AC1A', owner),
        ('2E70916786A6F773511FA7181FAB0F1D70B557C6322EA923B2A8D3B92B51AF7D', owner),
        ('306628FA5477305728BA4A467DE7D0387A54F569D3769FCE5E75EC89D28D1593', owner),
        ('311A2AC55B50C09B30B3CC93B994A119153EEEAC54EF892FC447BBBD96101AA1', owner),
        ('32AD3296829BC46DCFAC5EDDCB9DBF2C1EED5C11F83B2210CF9C6E60C798D4A7', owner),
        ('32F5940CA29DD812A2C145E6FC89646628FFCC7C7A42CAE512337D8D29C40BBD', owner),
        ('340DA32B58331C8E2B561BAF300CA9DFD6B91CD2270EE0E2A34958B1C6259E85', owner),
        ('3608EDBAF5AD0F41A414A1777ABF2FAF5E670334675EC3995E6935829E0CAAD2', owner),
        ('362ED31D20B1E00392281231A96F0A0ACFDE02618953E695C9EF2EB0BAC37550', owner),
        ('363384D14D1F2E0B7815626484C459AD57A318EF4396266048D058C5A19BBF76', owner),
        ('367A31E5838831AD2C074647886A6CDFF217E6B1BA910BFF85DC7A87AE9B5E98', owner),
        ('3765D769C05BF98B427B3511903B2137E8A49B6F859D0AF159ED6A86786AA634', owner),
        ('37A480374DAF6202CE790C318A2BB8AA3797311261160A8E30558B7DEA78C7A6', owner),
        ('3841D221368D1583D75C0A02E62160394D6C4E0A6760B6F607B90362BC855B02', owner),
        ('386D695CDF2D4576E01BCACCF5E49E78DA51AF9955C0B8FA7606373B007994B3', owner),
        ('39DBC2288EF44B5F95332CB777E31103E840DBA680634AA806F5C9B100061802', owner),
        ('3A4F74BEAFAE2B9383AD8215D233A6CF3D057FB3C7E213E897BEEF4255FAEE9D', owner),
        ('3A91F0F9E5287FA2994C7D930B2C1A5EE14CE8E1C8304AE495ADC58CC4453C0C', owner),
        ('3AE76C45CA70E9180C1559981F42622DD251BCA1FBE6B901C52EC11673B03514', owner),
        ('3B0287533E0CC3D0EC1AA823CBF0A941AAD8721579D1C499802DD1C3A636B8A9', owner),
        ('3BE8E7EB348D35C1928F19C769846788991641D1F6CF09514CA10269934F7359', owner),
        ('3E3926F0B8A15AD5A14167BB647A843C3D4321E35DBC44DCE8C837417F2D28B0', owner),
        ('3ECE27CBB3EC4438CCE523B927C4F05FDC5C593A3766DB984C5E437A3FF6A16B', owner),
        ('3FCE9B9FDF3EF09D5452B0F95EE481C2B7F06D743A737971558E70136ACE3E73', owner),
        ('400AC66D59B7B094A9E30B01A6BD013AFF1D30570F83E7592F421DBE5FF4BA8F', owner),
        ('408B8B3DF5ABB043521A493525023175AB1261B1DE21064D6BF247CE142153B9', owner),
        ('4185821F6DAB5BA8347B78A22B5F9A0A7570CA5C93A74D478A793D83BAC49805', owner),
        ('41D1EEB177C0324E17DD6557F384E532DE0CF51A019A446B01EFB351BC259D77', owner),
        ('4397DACA839E7F63077CB50C92DF43BC2D2FB2A8F59F26FC7A0E4BD4D9751692', owner),
        ('45876B4DD861D45B3A94800774027A5DB45A48B2A729410908B6412F8A87E95D', owner),
        ('45C7C8AE750ACFBB48FC37527D6412DD644DAED8913CCD8A24C94D856967DF8E', owner),
        ('4667BF250CD7C1A06B8474C613CDB1DF648A7F58736FBF57D05D6F755DAB67F4', owner),
        ('47CC086127E2069A86E03A6BEF2CD410F8C55A6D6BDB362168C31B2CE32A5ADF', owner),
        ('47FF1B63B140B6FC04ED79131331E651DA5B2E2F170F5DAEF4153DC2FBC532B1', owner),
        ('495300790E6C9BF2510DABA59DB3D57E9D2B85D7D7640434EC75BAA3851C74E5', owner),
        ('4B8668A5D465BCDD9000AA8DFCFF42044FCBD0AECE32FC7011A83E9160E89F09', owner),
        ('518831FE7382B514D03E15C621228B8AB65479BD0CBFA3C5C1D0F48D9C306135', owner),
        ('5391C3A2FB112102A6AA1EDC25AE77E19F5D6F09CD09EEB2509922BFCD5992EA', owner),
        ('540801DD345DC1C33EF431B35BF4C0E68BD319B577B9ABE1A9CFF1CBC39F548F', owner),
        ('55B99B0DE53DBCFE485AA9C737CF3FB616EF3D91FAB599AA7CAB19EDA763B5BA', owner),
        ('57E6913AFACC5222BD76CDAF31F8ED88895464255374EF097A82D7F59AD39596', owner),
        ('5890FA227121C76D90ED9E63C87E3A6533EEA0F6F0A1A23F1FC445139BC6BCDF', owner),
        ('58FB941AEF95A25943B3FB5F2510A0DF3FE44C58C95E0AB80487297568AB9771', owner),
        ('5AE949EA8855EB93E439DBC65BDA2E42852C2FDF6789FA146736E3C3410F2B5C', owner),
        ('5B248E913D71853D3DA5AEDD8D9A4BC57A917126573817FB5FCB2D86A2F1C886', owner),
        ('5C5805196A85E93789457017D4F9EB6828B97C41CB9BA6D3DC1FCC115F527A55', owner),
        ('5D1E9ACBBB4A7D024B6852DF025970E2CED66FF622EE019CD0ED7FD841CCAD02', owner),
        ('61341E07697978220EA61E85DCD2421343F2C1BF35CC5B8D0AD2F0226F391479', owner),
        ('61CEC4A377BF5902C0FEAEE37034BF97D5BC6E0615E23A1CDFBAE6E3F5FB3CFD', owner),
        ('631F0857B41845362C90C6980B4B10C4B628E23DBE24B6E96C128AE3DCB0D5AC', owner),
        ('64575BD912789A2E14AD56F6341F52AF6BF80CF94400785975E9F04E2D64D745', owner),
        ('65B2E7CC18D903C331DF1152DF73CA0DC932D29F17997481C56F3087B2DD3147', owner),
        ('65F3C0A01B8402D362B9722E98F75E5E991E6C186E934F7B2B2E6BE6DEC800EC', owner),
        ('66AA13A0EDC219384D9C425D3927E6ED4A5D1940C5E7CD4DAC88F5770103F2F1', owner),
        ('6873D2F61C29BD52E954EEFF5977AA8367439997811A62FF212C948133C68D97', owner),
        ('68EE4632C7BE1C66C83E89DD93EAEE1294159ABF45B4C2C72D7DC7499AA2A043', owner),
        ('6B1D138078E4418AA68DEB7BB35E066092CF479EEB8CE4CD12E7D072CCB42F66', owner),
        ('6C8854478DD559E29351B826C06CB8BFEF2B94AD3538358772D193F82ED1CA11', owner),
        ('6DBBEAD23E8C860CF8B47F74FBFCA5204DE3E28B881313BB1D1ECCDC4747934E', owner),
        ('6DEAD13257DFC3CCC6A4B37016BA91755FE9E0EC1F415030942E5ABC47F07C88', owner),
        ('6EFEFE0B5B01478B7B944C10D3A8ACA2CCA4208888E2059F8A06CB5824D7BAB0', owner),
        ('6F1428FF71C9DB0ED5AF1F2E7BBFCBAB647CC265DDF5B293CDB626F50A3A785E', owner),
        ('70A1450AF2AD395569AD0AFEB1D9C125324EE90AEC39C258880134D4892D51AB', owner),
        ('71F2906FD222497E54A34662AB2497FCC81020770FF51368E9E3D9BFCBFD6375', owner),
        ('726B3EB654046A30F3F83D9B96CE03F670E9A806D1708A0371E62DC49D2C23C1', owner),
        ('72C26F827CEB92989798961BC6AE748D141E05D3EBCFB65D9041B266C920BE82', owner),
        ('72E0BD1867CF5D9D56AB158ADF3BDDBC82BF32A8D8AA1D8C5E2F6DF29428D6D8', owner),
        ('77DD190FA30D88FF5E3B011A0AE61E6209780C130B535ECB87E6F0888A0B6B2F', owner),
        ('781764102188A8B4B173D4A8F5EC94D828647156097F99357A581E624B377509', owner),
        ('7827AF99362CFAF0717DADE4B1BFE0438AD171C15ADDC248B75BF8CAA44BB2C5', owner),
        ('788383A4C733BB87D2BF51673DC73E92DF15AB7D51DC715627AE77686D8D23BC', owner),
        ('78B4EDCAABC8D9093E20E217802CAEB4F09E23A3394C4ACC6E87E8F35395310F', owner),
        ('7BC9CB5463CE0F011FB5085EB8BA77D1ACD283C43F4A57603CC113F22CEBC579', owner),
        ('7EAC80A915C84CD4AFEC638904D94EB168A8557951A4D539B0713028552B6B8C', owner),
        ('7F49CCB309323B1C7AB11C93C955B8C744F0A2B75C311F495E18906070500027', owner),
        ('804E354C6368BB27A90FAE8E498A57052B293418259A019C4F53A2007254490F', owner),
        ('80B4D96931BF0D02FD91A61E19D14F1DA452E66DB2408CA8604D411F92659F0A', owner),
        ('81A8B2C9751AEB1FABA7DBDE5EE9691DC0EAEE2A31C38B1491A8146756A6B770', owner),
        ('81A8B965BB84D3876B9429A95481CC955318CFAA1412D808C8A33BFD33FFF0E4', owner),
        ('81D8FB4C9E2E7A8225656B4B8273B7CBA4B03EF2E9EB20E0A0291624ECA1BA86', owner),
        ('82ACBA48D5236CCFF7659AFC14594DEE902BD6082EF1A30A0B9B508628CF34F4', owner),
        ('82DB3BCEB4F60843CE9D97C3D187CD9B5941CD3DE8100E586F2BDA5637575F67', owner),
        ('835881F2A5572D7059B5C8635018552892E945626F115FC9CA07ACF7BDE857A4', owner),
        ('894D7839368F3298CC915AE8742EF330D7A26699F459478CF22C2B6BB2850166', owner),
        ('895A9785F617CA1D7ED44FC1A1470B71F3F1223862D9FF9DCC3AE2DF92163DAF', owner),
        ('89F3D1F6E485C334CD059D0995E3CDFDC00571B1849854847A44DC5548E2DCFB', owner),
        ('8AD64859F195B5F58DAFAA940B6A6167ACD67A886E8F469364177221C55945B9', owner),
        ('8BF434B49E00CCF71502A2CD900865CB01EC3B3DA03C35BE505FDF7BD563F521', owner),
        ('8C0349D708571AE5AA21C11363482332073297D868F29058916529EFC520EF70', owner),
        ('8D8EA289CFE70A1C07AB7365CB28EE51EDD33CF2506DE888FBADD60EBF80481C', owner),
        ('8D93D60C691959651476E5DC464BE12A85FA5280B6F524D4A1C3FCC9D048CFAD', owner),
        ('8E53EFDC15F852CEE5A6E92931BC42E6163CD30FF649CCA7E87252C3A459960B', owner),
        ('9063F5FBC5E57AB6DE6C9488146020E172B176D5AB57D4C89F0F600E17FE2DE2', owner),
        ('90FBE70E69D633408D3E170C6832DBB2D209E0272527DFB63D49D29572A6F44C', owner),
        ('91656AA4EF493B3824A0B7263248E4E2D657A5C8488D880CB65B01730932FB53', owner),
        ('91971C1497BF8E5BC68439ACC48D63EBB8FAABFD764DCBE82F3BA977CAC8CF6A', owner),
        ('939AEEF4F5FA51E23340C3F2E49048CE8872526AFDF752C3A7F3A3F2BC9F6049', owner),
        ('947078F97C6196968C3AE99C9A5D58667E86882CF6C8C9D58967A496BB7AF43C', owner),
        ('95049F0E4137C790B0D2767195E56F73807D123ADCF8F6E7BF2D4D991D305F89', owner),
        ('96E4509450D380DAC362FF8E295589128A1F1CE55885D20D89C27BA2A9D00909', owner),
        ('9783B5EE4492E9E891C655F1F48035959DAD453C0E623AF0FE7BF2C0A57885E3', owner),
        ('97A51A094444620DF38CD8C6512CAC909A75FD437AE1E4D22929807661238127', owner),
        ('97A8C5BA11D61FEFBB5D6A05DA4E15BA472DC4C6CD4972FC1A035DE321342FE4', owner),
        ('992820E6EC8C41DAAE4BD8AB48F58268E943A670D35CA5E2BDCD3E7C4C94A072', owner),
        ('992D359AA7A5F789D268B94C11B9485A6B1CE64362B0EDB4441CCC187C39647B', owner),
        ('9954A1A99D55E8B189AB1BCA414B91F6A017191F6C40A86B6F3EF368DD860031', owner),
        ('9998D363C491BE16BD74BA10B94D9291001611736FDCA643A36664BC0F315A42', owner),
        ('9BAF4F76D76BF5D6A897BFBD5F429BA14D04E08B48C3EE8D76930A828FFF3891', owner),
        ('9C259FCB301D5FC7397ED5759963E0EF6B36E42057FD73046E6BD08B149F751C', owner),
        ('9D00AE4CD47A41C783DC48F342C076C2C16F3413F4D2DF50D181CA3BB5AD859D', owner),
        ('9DD2DCB72F5E741627F2E9E03AB18503A3403CF6A904A479A4DB05D97E2250A9', owner),
        ('9E4A69173161682E55FDE8FEF560EB88EC1FFEDCAF04001F66C0CAF707B2B734', owner),
        ('9ED33F0FBC180BC032F8909CA2C4AB3418EDC33A45A50D2521A3B5876AA3EA2C', owner),
        ('9F1863ED5717C394B42EF10A6607B144A65BA11FB6579DF94B8EB2F0C4CD60C1', owner),
        ('9FA4D5023FD43ECAFF4200BA7E8D4353259D2B7E5E72B5096EFF8027D66D1043', owner),
        ('A4D978B7C4BDA15435D508F8B9592EC2A5ADFB12EA7BAD146A35ECB53094642F', owner),
        ('A6B5151F3655D3A2AF0D472759796BE4A4200E5495A7D869754C4848857408A7', owner),
        ('A7F32F508D4EB0FEAD9A087EF94ED1BA0AEC5DE6F7EF6FF0A62B93BEDF5D458D', owner),
        ('A924D3CAD6DA42B7399B96A095A06F18F6B1ABA5B873B0D5F3A0EE2173B48B6C', owner),
        ('AD3BE589C0474E97DE5BB2BF33534948B76BB80376DFDC58B1FED767B5A15BFC', owner),
        ('AD6826E1946D26D3EAF3685C88D97D85DE3B4DCB3D0EE2AE81C70560D13C5720', owner),
        ('AEEBAE3151271273ED95AA2E671139ED31A98567303A332298F83709A9D55AA1', owner),
        ('AFE2030AFB7D2CDA13F9FA333A02E34F6751AFEC11B010DBCD441FDF4C4002B3', owner),
        ('B3E506340FBF6B5786973393079F24B66BA46507E35E911DB0362A2ACDE97049', owner),
        ('B54F1EE636631FAD68058D3B0937031AC1B90CCB17062A391CCA68AFDBE40D55', owner),
        ('B8D6B5E7857B45830E017C7BE3D856ADEB97C7290EB0665A3D473A4BEB51DCF3', owner),
        ('B8F078D983A24AC433216393883514CD932C33AF18E7DD70884C8235F4275736', owner),
        ('B92AF298DC08049B78C77492D6551B710CD72AADA3D77BE54609E43278EF6E4D', owner),
        ('B93F0699598F8B20FA0DACC12CFCFC1F2568793F6E779E04795E6D7C22530F75', owner),
        ('B97A0889059C035FF1D54B6DB53B11B9766668D9F955247C028B2837D7A04CD9', owner),
        ('BADFF5E4F0FEA711701CA8FB22E4C43821E31E210CF52D1D4F74DD50F1D039BC', owner),
        ('BB01DA0333BB639C7E1C806DB0561DC98A5316F22FEF1090FB8D0BE46DAE499A', owner),
        ('BB1DD16D530008636F232303A7A86F3DFF969F848815C0574B12C2D787FEC93F', owner),
        ('BC75F910FF320F5CB5999E66BBD4034F4AE537A42FDFEF35161C5348E366E216', owner),
        ('BC87A668E81966489CB508EE805183C19E6ACD24CF17799CA062D2E384DA0EA7', owner),
        ('BDD01126E9D85710D3FE75AF1CC1702A29F081B4F6FDF6A2B2135C0297A9CEC5', owner),
        ('BE435DF7CD28AA2A7C8DB4FC8173475B77E5ABF392F76B7C76FA3F698CB71A9A', owner),
        ('BEF7663BE5EA4DBFD8686E24701E036F4C03FB7FCD67A6C566ED94CE09C44470', owner),
        ('C2469759C1947E14F4B65F72A9F5B3AF8B6F6E727B68BB0D91385CBF42176A8A', owner),
        ('C3505BF3EC10A51DACE417C76B8BD10939A065D1F34E75B8A3065EE31CC69B96', owner),
        ('C3A99A460DA464A057C3586D83CEF5F4AE08B7103979ED8932742DF0ED530C66', owner),
        ('C409BDAC4775ADD8DB92AA22B5B718FB8C94A1462C1FE9A416B95D8A3388C2FC', owner),
        ('C42D11C70CCF5E8CF3FB91FDF21D884021AD836CA68ADF2CBB7995C10BF588D4', owner),
        ('C452AB846073DF5ACE25CCA64D6B7A09D906308A1A65EB5240E3C4EBCAA9CC0C', owner),
        ('C5D9D8A186E2C82D09AFAA2A6F7F2E73870D3E64F72C4E08EF67796A840F0FBD', owner),
        ('C617C1A8B1EE2A811C28B5A81B4C83D7C98B5B0C27281D610207EBE692C2967F', owner),
        ('C69D64A5B839E41BA16742527E17056A18CE3C276FD26E34901A1BC7D0E32219', owner),
        ('C805603C4FA038776E42F263C604B49D96840322E1922D5606A9B0BBB5BFFE6F', owner),
        ('C83CB13922AD99F560744675DD37CC94DCAD5A1FCBA6472FEE341171D939E884', owner),
        ('C90F336617B8E7F983975413C997F10B73EB267FD8A10CB9E3BDBFC667ABDB8B', owner),
        ('C9EC350406F26E559AFFB4030DE2EBDE5435054C35A998605B8FCF04972D8D55', owner),
        ('CA171D614A8D7E121C93948CD0FE55D39981F9D11AA96E03450A415227C2C65B', owner),
        ('CB340011AFEB0D74C4A588B36EBAA441961608E8D2FA80DCA8C13872C850796B', owner),
        ('CB6B858B40D3A098765815B592C1514A49604FAFD60819DA88D7A76E9778FEF7', owner),
        ('CC8EEC6EB9212CBF897A5ACE7E8ABEECE1079F1A6DEF0A789591CB1547F1F084', owner),
        ('CE3BFABE59D67CE8AC8DFD4A16F7C43EF9C224513FBC655957D735FA29F540CE', owner),
        ('CF13A243C1CD2E3C8CEB7E70100387CECBFB830525BBF9D0B70C79ADF3E84128', owner),
        ('D063EC28F67EBA53F1642DBF7DFF33C6A32ADD869F6013FE162E2C32F1CBE56D', owner),
        ('D372C0D0F4FDC9F52E9E1F23FC56EE72414A17F350D0CEA6C26A35A6C3217A13', owner),
        ('D626157E1D6A718BC124AB8DA27CBB65072CA03A7B6B257DBDCBBD60F65EF3D1', owner),
        ('D89A11D16C488DD4FBBC541D4B07FAF8670D660994488FE54B1FBFF2704E4288', owner),
        ('D8CBEB9735F5672B367E4F96CDC74969615D17074AE96C724D42CE0216F8F3FA', owner),
        ('D8D4E6DDF6E42D74A6A536EA62FD1217E4290B145C9E5C3695A31B42EFB5F5A4', owner),
        ('D9668AB52785086786C134B5E4BDDBF72452813B6973229AB92AA1A54D201BF5', owner),
        ('DA3560FD0C32B54C83D4F2FF869003D2089369ACF2C89608F8AFA7436BFA4655', owner),
        ('DBAF9E056D3D5B38B68553304ABC88827EBC00F80CB9C7E197CDBC5822CD316C', owner),
        ('DCCC3CE1C00EE4B0B10487D372A0FA47F5C26F57A359BE7B27801E144EACBAC4', owner),
        ('DD59AF56084406E38C63FBE0850F30A0CD1277462A2192590FB05BC259E61273', owner),
        ('DF02AAB48387A9E1D4C65228089CB6ABE196C8F4B396C7E4BBC395DE136977F6', owner),
        ('DF91AC85A94FCD0CFB8155BD7CBEFAAC14B8C5EE7397FE2CC85984459E2EA14E', owner),
        ('E051B788ECBAEDA53046C70E6AF6058F95222C046157B8C4C1B9C2CFC65F46E5', owner),
        ('E19DAE83C02E6F281358D4EBD11D7723B4F5EA0E357907D5443DECC5F93C1E9D', owner),
        ('E24B315A551671483D8B9073B32DE11B4DE1EB2EAB211AFD2D9C319FF55E08D0', owner),
        ('E36DFC719D2114C2E39AEA88849E2845AB326F6F7FE74E0E539B7E54D81F3631', owner),
        ('E39891F48BBCC593B8ED86CE82CE666FC1145B9FCBFD2B07BAD0A89BF4C7BFBF', owner),
        ('E6856F137F79992DC94FA2F43297EC32D2D9A76F7BE66114C6A13EFC3BCDF5C8', owner),
        ('E6CA68E94146629AF03F69C2F86E6BEF62F930B37C6FBCC878B78DF98C0334E5', owner),
        ('E7681F153121EA1E67F74BBCB0CDC5E502702C1B8CC55FB65D702DFBA948B5F4', owner),
        ('E7C20B3AB481EC885501ECA5293781D84B5A1AC24F88266B5270E7ECB4AA2538', owner),
        ('E800395DBE0E045781E8005178B4BAF5A257F06E159121A67C595F6AE22506FD', owner),
        ('E92C22EB3B5642D65C1EC2CAF247D2594738EEBB7FB3841A44956F59E2B0D1FA', owner),
        ('EAFF8C85C208BA4D5B6B8046F5D6081747D779BADA7768E649D047FF9B1F660C', owner),
        ('EE83A566496109A74F6AC6E410DF00BB29A290E0021516AE3B8A23288E7E2E72', owner),
        ('EED7E0EFF2ED559E2A79EE361F9962AF3B1E999131E30BB7FD07546FAE0A7267', owner),
        ('F1863EC8B7F43F94AD14FB0B8B4A69497A8C65ECBC2A55E0BB420E772B8CDC91', owner),
        ('F1B4F6513B0D544A688D13ADC291EFA8C59F420CA5DCB23E0B5A06FA7E0D083D', owner),
        ('F277AF4F9BDC918AE89FA35CC1B34E34984C04AE9765322C3CB049574D36509C', owner),
        ('F2A16D35B554694187A70D40CA682959F4F35C2CE0EAB8FD64F7AC2AB9F5C24A', owner),
        ('F31FD461C5E99510403FC97C1DA2D8A9CBE270597D32BADF8FD66B77495F8D94', owner),
        ('F48E6DD8718E953B60A24F2CBEA60A9521DEAE67DB25425B7D3ACE3C517DD9B7', owner),
        ('F52F83A3FA9CFBD6920F722824DBE4034534D25B8507246B3B957DAC6E1BCE7A', owner),
        ('FDDD6E3D29EA84C7743DAD4A1BDBC700B5FEC1B391F932409086ACC71DD6DBD8', owner),
        ('FE63A84F782CC9D3FCF2CCF9FC11FBD03760878758D26285ED12669BDC6E6D01', owner),
        ('FECFB232D12E994B6D485D2C7167728AA5525984AD5CA61E7516221F079A1436', owner)
    ]
    canonical_hashes_esl = sha256_esl_builder(canonicalHashes)
    expected_canonical_esd = EfiSignatureDatabase(EslList=[canonical_hashes_esl, cert2_esl, cert1_esl])

    return (esd, expected_dupes_esd, expected_canonical_esd)


def MixedCertsData():
    """ Returns a tuple of (rawData, expectedDupes, expectedCanonical) for a mix of unsorted certs with dupes """

    owner = '77fa9abd-0359-4d32-bd60-28f4e78f784b'

    esls = [
        x509_esl_builder([(DEBIAN_CERT, owner)]),
        x509_esl_builder([(UBUNTU_CERT, owner)])
    ]

    input = [
        esls[1],
        esls[0],
        esls[1],
        esls[0],
        esls[1]
    ]
    esd = EfiSignatureDatabase(EslList=input)

    dupes = [
        esls[0],
        esls[1],
        esls[1]
    ]
    expected_dupes_esd = EfiSignatureDatabase(EslList=dupes)

    canonical = [
        esls[0],
        esls[1],
    ]
    expected_canonical_esd = EfiSignatureDatabase(EslList=canonical)

    return (esd, expected_dupes_esd, expected_canonical_esd)


class AuthVarSupportLibraryUnitTests(unittest.TestCase):

    def test_print(self):
        esd = basicEfiSignatureDatabase()
        print("\nPrint test:")
        esd.Print(compact=True)
        esd.Print(compact=False)

    def test_deserializeEqualsSerialize(self):
        # root = logging.getLogger()
        # root.setLevel(logging.INFO)

        original_esd = basicEfiSignatureDatabase()
        original_fs = io.BytesIO()
        original_esd.Write(original_fs)
        original_fs.seek(0)

        new_esd = EfiSignatureDatabase(filestream=original_fs)
        original_fs.close()

        self.assertEqual(original_esd.GetBytes(), new_esd.GetBytes())

    def test_EfiSignatureList_Sort_and_Deduplication_x509(self):
        owner = uuid.uuid4().hex

        subTestList = [
            (
                "SubTest_Empty",
                # input
                [],
                # expected after sort
                [],
                # expected duplicates
                [],
            ),
            (
                "SubTest_One_Entry",
                # input
                [(DEBIAN_CERT, owner)],
                # expected after sort
                [(DEBIAN_CERT, owner)],
                # expected duplicates
                [],
            )
        ]

        for subTest in subTestList:
            (testName, input, expected_sort, expected_dupes) = subTest
            with self.subTest(msg=testName):
                testEsl = x509_esl_builder(input)
                expected_sort_esl = x509_esl_builder(expected_sort)
                expected_dupes_esl = x509_esl_builder(expected_dupes)

                output_dupes = testEsl.SortBySignatureDataValue(deduplicate=True)

                self.assertEqual(output_dupes.GetBytes(), expected_dupes_esl.GetBytes())
                self.assertEqual(testEsl.GetBytes(), expected_sort_esl.GetBytes())

    def test_EfiSignatureList_Sort_and_Deduplication_sha256(self):

        owner1 = uuid.uuid4().hex
        owner2 = uuid.uuid4().hex

        subTestList = [
            (
                "SubTest_Empty",
                # input
                [],
                # expected after sort
                [],
                # expected duplicates
                [],
            ),
            (
                "SubTest_One_Entry",
                # input
                [
                    (HASHSTR[1], owner1),
                ],
                # expected after sort
                [
                    (HASHSTR[1], owner1),
                ],
                # expected duplicates
                [],
            ),
            (
                "SubTest_321",
                # input
                [
                    (HASHSTR[3], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[1], owner1),
                ],
                # expected after sort
                [
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner1),
                ],
                # expected duplicates
                [],
            ),
            (
                "SubTest_32123",
                # input
                [
                    (HASHSTR[3], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner1),
                ],
                # expected after sort
                [
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner1),
                ],
                # expected duplicates
                [
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner1),
                ],
            ),
            (
                "SubTest_122333221",
                # input
                [
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner1),
                    (HASHSTR[3], owner1),
                    (HASHSTR[3], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[1], owner1),
                ],
                # expected after sort
                [
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner1),
                ],
                # expected duplicates
                [
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner1),
                    (HASHSTR[3], owner1),
                ],
            ),
            (
                "SubTest_8005551212",
                # input
                [
                    (HASHSTR[8], owner1),
                    (HASHSTR[0], owner1),
                    (HASHSTR[0], owner1),
                    (HASHSTR[5], owner1),
                    (HASHSTR[5], owner1),
                    (HASHSTR[5], owner1),
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1)
                ],
                # expected after sort
                [
                    (HASHSTR[0], owner1),
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[5], owner1),
                    (HASHSTR[8], owner1),
                ],
                # expected duplicates
                [
                    (HASHSTR[0], owner1),
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[5], owner1),
                    (HASHSTR[5], owner1),
                ],
            ),
            (
                "SubTest_122333221_mixed_sigowner",
                # input
                [
                    (HASHSTR[1], owner2),
                    (HASHSTR[2], owner2),
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner2),
                    (HASHSTR[3], owner2),
                    (HASHSTR[3], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[2], owner2),
                    (HASHSTR[1], owner1),
                ],
                # expected after sort
                [
                    (HASHSTR[1], owner2),
                    (HASHSTR[2], owner2),
                    (HASHSTR[3], owner2),
                ],
                # expected duplicates
                [
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[2], owner2),
                    (HASHSTR[3], owner2),
                    (HASHSTR[3], owner1),
                ],
            ),
        ]

        for subTest in subTestList:
            (testName, input, expected_sort, expected_dupes) = subTest
            with self.subTest(msg=testName):
                testEsl = sha256_esl_builder(input)
                expected_sort_esl = sha256_esl_builder(expected_sort)
                expected_dupes_esl = sha256_esl_builder(expected_dupes)

                output_dupes_esl = testEsl.SortBySignatureDataValue(deduplicate=True)

                self.assertEqual(testEsl.GetBytes(), expected_sort_esl.GetBytes())
                self.assertEqual(output_dupes_esl.GetBytes(), expected_dupes_esl.GetBytes())

    def test_EfiSignatureDatabase_Sort_and_Deduplication(self):
        subTestList = [
            ("MixedCerts", MixedCertsData()),
            ("BootHole", BootHoleData()),
        ]

        for subTest in subTestList:
            (testName, (Esd, ExpectedDupesEsd, ExpectedCanonicalEsd)) = subTest
            with self.subTest(msg=testName):

                (output_canonical_esd, output_dupes_esd) = Esd.GetCanonicalAndDupes()

                if VERBOSE:
                    print("Starting EfiSignatureDatabase: ")
                    Esd.Print(compact=True)

                    print("Dupes: ")
                    output_dupes_esd.Print(compact=True)

                    print("Canonical: ")
                    output_canonical_esd.Print(compact=True)

                self.assertEqual(output_dupes_esd.GetBytes(), ExpectedDupesEsd.GetBytes())
                self.assertEqual(output_canonical_esd.GetBytes(), ExpectedCanonicalEsd.GetBytes())

    def test_EFiVariableAuthentication2_load_from_file(self):
        # translate the DBXFILE to a ByteIo stream
        DbxIo = io.BytesIO(bytes.fromhex(DBXFILE))

        # Use EFiVariableAuthentication2 to decode the DBX file
        EfiAuthVar = EFiVariableAuthentication2(decodefs=DbxIo)

        # Write the decoded Dbx File back into a ByteIo stream
        Output = io.BytesIO()
        EfiAuthVar.Write(Output)

        # Assert if the Decoded version does not match the original
        self.assertEqual(Output.getvalue(), DbxIo.getvalue())
