"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabContainerRunner = exports.InstanceInterruptionBehavior = exports.BlockDuration = void 0;
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
/**
 * @experimental
 */
var BlockDuration;
(function (BlockDuration) {
    BlockDuration[BlockDuration["ONE_HOUR"] = 60] = "ONE_HOUR";
    BlockDuration[BlockDuration["TWO_HOURS"] = 120] = "TWO_HOURS";
    BlockDuration[BlockDuration["THREE_HOURS"] = 180] = "THREE_HOURS";
    BlockDuration[BlockDuration["FOUR_HOURS"] = 240] = "FOUR_HOURS";
    BlockDuration[BlockDuration["FIVE_HOURS"] = 300] = "FIVE_HOURS";
    BlockDuration[BlockDuration["SIX_HOURS"] = 360] = "SIX_HOURS";
    BlockDuration[BlockDuration["SEVEN_HOURS"] = 420] = "SEVEN_HOURS";
    BlockDuration[BlockDuration["EIGHT_HOURS"] = 480] = "EIGHT_HOURS";
    BlockDuration[BlockDuration["NINE_HOURS"] = 540] = "NINE_HOURS";
    BlockDuration[BlockDuration["TEN_HOURS"] = 600] = "TEN_HOURS";
    BlockDuration[BlockDuration["ELEVEN_HOURS"] = 660] = "ELEVEN_HOURS";
    BlockDuration[BlockDuration["TWELVE_HOURS"] = 720] = "TWELVE_HOURS";
    BlockDuration[BlockDuration["THIRTEEN_HOURS"] = 780] = "THIRTEEN_HOURS";
    BlockDuration[BlockDuration["FOURTEEN_HOURS"] = 840] = "FOURTEEN_HOURS";
    BlockDuration[BlockDuration["FIFTEEN_HOURS"] = 900] = "FIFTEEN_HOURS";
    BlockDuration[BlockDuration["SIXTEEN_HOURS"] = 960] = "SIXTEEN_HOURS";
    BlockDuration[BlockDuration["SEVENTEEN_HOURS"] = 1020] = "SEVENTEEN_HOURS";
    BlockDuration[BlockDuration["EIGHTTEEN_HOURS"] = 1080] = "EIGHTTEEN_HOURS";
    BlockDuration[BlockDuration["NINETEEN_HOURS"] = 1140] = "NINETEEN_HOURS";
    BlockDuration[BlockDuration["TWENTY_HOURS"] = 1200] = "TWENTY_HOURS";
})(BlockDuration = exports.BlockDuration || (exports.BlockDuration = {}));
/**
 * @experimental
 */
var InstanceInterruptionBehavior;
(function (InstanceInterruptionBehavior) {
    InstanceInterruptionBehavior["HIBERNATE"] = "hibernate";
    InstanceInterruptionBehavior["STOP"] = "stop";
    InstanceInterruptionBehavior["TERMINATE"] = "terminate";
})(InstanceInterruptionBehavior = exports.InstanceInterruptionBehavior || (exports.InstanceInterruptionBehavior = {}));
/**
 * @experimental
 */
class GitlabContainerRunner extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j;
        super(scope, id);
        const spotFleetId = id;
        const defaultProps = {
            gitlabRunnerImage: 'public.ecr.aws/gitlab/gitlab-runner:alpine',
            gitlaburl: 'https://gitlab.com/',
            ec2type: 't3.micro',
            tags: ['gitlab', 'awscdk', 'runner'],
        };
        const runnerProps = { ...defaultProps, ...props };
        const runnerBucket = new aws_s3_1.Bucket(this, 'runnerBucket', {
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        const shell = aws_ec2_1.UserData.forLinux();
        shell.addCommands(...this.createUserData(runnerProps, runnerBucket.bucketName));
        this.runnerRole = (_a = runnerProps.ec2iamrole) !== null && _a !== void 0 ? _a : new aws_iam_1.Role(this, 'runner-role', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            description: 'For Gitlab EC2 Runner Role',
        });
        this.validUntil = runnerProps.validUntil;
        const instanceProfile = new aws_iam_1.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.runnerRole.roleName],
        });
        runnerBucket.grantWrite(this.runnerRole);
        this.vpc = (_b = runnerProps.selfvpc) !== null && _b !== void 0 ? _b : new aws_ec2_1.Vpc(this, 'VPC', {
            cidr: '10.0.0.0/16',
            maxAzs: 2,
            subnetConfiguration: [
                {
                    cidrMask: 26,
                    name: 'RunnerVPC',
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                },
            ],
            natGateways: 0,
        });
        this.defaultRunnerSG = new aws_ec2_1.SecurityGroup(this, 'SpotFleetSg', {
            vpc: this.vpc,
        });
        this.defaultRunnerSG.connections.allowFromAnyIpv4(aws_ec2_1.Port.tcp(22));
        const spotOrOnDemand = (_c = runnerProps.spotFleet) !== null && _c !== void 0 ? _c : false;
        if (spotOrOnDemand) {
            //throw new Error('yes new spotfleet');
            const imageId = aws_ec2_1.MachineImage.latestAmazonLinux({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }).getImage(this).imageId;
            const lt = new aws_ec2_1.CfnLaunchTemplate(this, 'LaunchTemplate', {
                launchTemplateData: {
                    imageId,
                    instanceType: runnerProps.ec2type,
                    blockDeviceMappings: [
                        {
                            deviceName: '/dev/xvda',
                            ebs: {
                                volumeSize: (_d = runnerProps.ebsSize) !== null && _d !== void 0 ? _d : 60,
                            },
                        },
                    ],
                    userData: core_1.Fn.base64(shell.render()),
                    keyName: runnerProps.keyName,
                    tagSpecifications: [
                        {
                            resourceType: 'instance',
                            tags: [
                                {
                                    key: 'Name',
                                    value: `${core_1.Stack.of(this).stackName}/spotFleetGitlabRunner/${spotFleetId}`,
                                },
                            ],
                        },
                    ],
                    instanceMarketOptions: {
                        marketType: 'spot',
                        spotOptions: {
                            blockDurationMinutes: (_e = runnerProps.blockDuration) !== null && _e !== void 0 ? _e : BlockDuration.ONE_HOUR,
                            instanceInterruptionBehavior: (_f = runnerProps.instanceInterruptionBehavior) !== null && _f !== void 0 ? _f : InstanceInterruptionBehavior.TERMINATE,
                        },
                    },
                    securityGroupIds: this.defaultRunnerSG.connections.securityGroups.map((m) => m.securityGroupId),
                    iamInstanceProfile: {
                        arn: instanceProfile.attrArn,
                    },
                },
            });
            const spotFleetRole = new aws_iam_1.Role(this, 'FleetRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
                ],
            });
            const vpcSubnetSelection = (_g = runnerProps.vpcSubnet) !== null && _g !== void 0 ? _g : {
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            };
            const subnetConfig = this.vpc
                .selectSubnets(vpcSubnetSelection)
                .subnets.map((s) => ({
                subnetId: s.subnetId,
            }));
            const cfnSpotFleet = new aws_ec2_1.CfnSpotFleet(this, id, {
                spotFleetRequestConfigData: {
                    launchTemplateConfigs: [
                        {
                            launchTemplateSpecification: {
                                launchTemplateId: lt.ref,
                                version: lt.attrLatestVersionNumber,
                            },
                            overrides: subnetConfig,
                        },
                    ],
                    iamFleetRole: spotFleetRole.roleArn,
                    targetCapacity: 1,
                    validUntil: core_1.Lazy.stringValue({ produce: () => this.validUntil }),
                    terminateInstancesWithExpiration: true,
                },
            });
            const onEvent = new lambda.Function(this, 'OnEvent', {
                code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
                handler: 'index.on_event',
                runtime: lambda.Runtime.PYTHON_3_8,
                timeout: core_1.Duration.seconds(60),
            });
            const isComplete = new lambda.Function(this, 'IsComplete', {
                code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
                handler: 'index.is_complete',
                runtime: lambda.Runtime.PYTHON_3_8,
                timeout: core_1.Duration.seconds(60),
                role: onEvent.role,
            });
            const myProvider = new cr.Provider(this, 'MyProvider', {
                onEventHandler: onEvent,
                isCompleteHandler: isComplete,
                logRetention: logs.RetentionDays.ONE_DAY,
            });
            onEvent.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ['ec2:DescribeSpotFleetInstances'],
                resources: ['*'],
            }));
            const fleetInstances = new core_1.CustomResource(this, 'GetInstanceId', {
                serviceToken: myProvider.serviceToken,
                properties: {
                    SpotFleetRequestId: cfnSpotFleet.ref,
                },
            });
            fleetInstances.node.addDependency(cfnSpotFleet);
            this.spotFleetInstanceId = core_1.Token.asString(fleetInstances.getAtt('InstanceId'));
            this.spotFleetRequestId = core_1.Token.asString(fleetInstances.getAtt('SpotInstanceRequestId'));
            new core_1.CfnOutput(this, 'InstanceId', { value: this.spotFleetInstanceId });
            new core_1.CfnOutput(this, 'SpotFleetId', { value: cfnSpotFleet.ref });
        }
        else {
            this.runnerEc2 = new aws_ec2_1.Instance(this, 'GitlabRunner', {
                instanceType: new aws_ec2_1.InstanceType(runnerProps.ec2type),
                instanceName: 'Gitlab-Runner',
                vpc: this.vpc,
                vpcSubnets: (_h = runnerProps.vpcSubnet) !== null && _h !== void 0 ? _h : {
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                },
                machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({
                    generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
                }),
                role: this.runnerRole,
                userData: shell,
                securityGroup: this.defaultRunnerSG,
                blockDevices: [
                    {
                        deviceName: '/dev/xvda',
                        volume: aws_ec2_1.BlockDeviceVolume.ebs((_j = runnerProps.ebsSize) !== null && _j !== void 0 ? _j : 60),
                    },
                ],
            });
            new core_1.CfnOutput(this, 'Runner-Instance-ID', {
                value: this.runnerEc2.instanceId,
            });
        }
        const unregisterRunnerOnEvent = new lambda.Function(this, 'unregisterRunnerOnEvent', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'unregister_runner.on_event',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: core_1.Duration.seconds(60),
        });
        const unregisterRunnerProvider = new cr.Provider(this, 'unregisterRunnerProvider', {
            onEventHandler: unregisterRunnerOnEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        const unregisterRunnerCR = new core_1.CustomResource(this, 'unregisterRunnerCR', {
            resourceType: 'Custom::unregisterRunnerProvider',
            serviceToken: unregisterRunnerProvider.serviceToken,
            properties: {
                BucketName: runnerBucket.bucketName,
                GitlabUrl: runnerProps.gitlaburl,
            },
        });
        runnerBucket.grantReadWrite(unregisterRunnerOnEvent);
        unregisterRunnerCR.node.addDependency(runnerBucket);
        this.runnerRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        new core_1.CfnOutput(this, 'Runner-Role-Arn', {
            value: this.runnerRole.roleArn,
        });
    }
    /**
     * @default - !!! only support spotfleet runner !!! .
     * @experimental
     */
    expireAfter(duration) {
        const date = new Date();
        date.setSeconds(date.getSeconds() + duration.toSeconds());
        this.validUntil = date.toISOString();
    }
    dockerVolumesList(dockerVolume) {
        let tempString = '--docker-volumes "/var/run/docker.sock:/var/run/docker.sock"';
        if (dockerVolume) {
            let tempList = [];
            dockerVolume.forEach(e => {
                tempList.push(`"${e.hostPath}:${e.containerPath}"`);
            });
            tempList.forEach(e => {
                tempString = `${tempString} --docker-volumes ${e}`;
            });
        }
        return tempString;
    }
    /**
     * @experimental
     */
    createUserData(props, bucketName) {
        var _a;
        return [
            'yum update -y ',
            'sleep 15 && yum install docker git -y && systemctl start docker && usermod -aG docker ec2-user && chmod 777 /var/run/docker.sock',
            'systemctl restart docker && systemctl enable docker',
            `docker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock \
      --name gitlab-runner-register ${props.gitlabRunnerImage} register --non-interactive --url ${props.gitlaburl} --registration-token ${props.gitlabtoken} \
      --docker-pull-policy if-not-present ${this.dockerVolumesList(props === null || props === void 0 ? void 0 : props.dockerVolumes)} \
      --executor docker --docker-image "alpine:latest" --description "Docker Runner" \
      --tag-list "${(_a = props.tags) === null || _a === void 0 ? void 0 : _a.join(',')}" --docker-privileged`,
            `sleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner ${props.gitlabRunnerImage}`,
            `TOKEN=$(cat /home/ec2-user/.gitlab-runner/config.toml | grep token | cut -d '"' -f 2) && echo '{"token": "TOKEN"}' > /tmp/runnertoken.txt && sed -i s/TOKEN/$TOKEN/g /tmp/runnertoken.txt && aws s3 cp /tmp/runnertoken.txt s3://${bucketName}`,
        ];
    }
}
exports.GitlabContainerRunner = GitlabContainerRunner;
//# sourceMappingURL=data:application/json;base64,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