"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LegendPosition = exports.Color = exports.Shading = exports.SingleValueWidget = exports.GraphWidget = exports.GraphWidgetView = exports.AlarmWidget = void 0;
const cdk = require("@aws-cdk/core");
const rendering_1 = require("./private/rendering");
const widget_1 = require("./widget");
/**
 * Display the metric associated with an alarm, including the alarm line.
 *
 * @stability stable
 */
class AlarmWidget extends widget_1.ConcreteWidget {
    /**
     * @stability stable
     */
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    /**
     * Return the widget JSON for use in the dashboard.
     *
     * @stability stable
     */
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    annotations: {
                        alarms: [this.props.alarm.alarmArn],
                    },
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                    },
                },
            }];
    }
}
exports.AlarmWidget = AlarmWidget;
/**
 * Types of view.
 *
 * @stability stable
 */
var GraphWidgetView;
(function (GraphWidgetView) {
    GraphWidgetView["TIME_SERIES"] = "timeSeries";
    GraphWidgetView["BAR"] = "bar";
    GraphWidgetView["PIE"] = "pie";
})(GraphWidgetView = exports.GraphWidgetView || (exports.GraphWidgetView = {}));
/**
 * A dashboard widget that displays metrics.
 *
 * @stability stable
 */
class GraphWidget extends widget_1.ConcreteWidget {
    /**
     * @stability stable
     */
    constructor(props) {
        var _a, _b;
        super(props.width || 6, props.height || 6);
        this.props = props;
        this.leftMetrics = (_a = props.left) !== null && _a !== void 0 ? _a : [];
        this.rightMetrics = (_b = props.right) !== null && _b !== void 0 ? _b : [];
    }
    /**
     * Add another metric to the left Y axis of the GraphWidget.
     *
     * @param metric the metric to add.
     * @stability stable
     */
    addLeftMetric(metric) {
        this.leftMetrics.push(metric);
    }
    /**
     * Add another metric to the right Y axis of the GraphWidget.
     *
     * @param metric the metric to add.
     * @stability stable
     */
    addRightMetric(metric) {
        this.rightMetrics.push(metric);
    }
    /**
     * Return the widget JSON for use in the dashboard.
     *
     * @stability stable
     */
    toJson() {
        var _a;
        const horizontalAnnotations = [
            ...(this.props.leftAnnotations || []).map(mapAnnotation('left')),
            ...(this.props.rightAnnotations || []).map(mapAnnotation('right')),
        ];
        const metrics = rendering_1.allMetricsGraphJson(this.leftMetrics, this.rightMetrics);
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: (_a = this.props.view) !== null && _a !== void 0 ? _a : GraphWidgetView.TIME_SERIES,
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    stacked: this.props.stacked,
                    metrics: metrics.length > 0 ? metrics : undefined,
                    annotations: horizontalAnnotations.length > 0 ? { horizontal: horizontalAnnotations } : undefined,
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                        right: this.props.rightYAxis !== undefined ? this.props.rightYAxis : undefined,
                    },
                    legend: this.props.legendPosition !== undefined ? { position: this.props.legendPosition } : undefined,
                    liveData: this.props.liveData,
                },
            }];
    }
}
exports.GraphWidget = GraphWidget;
/**
 * A dashboard widget that displays the most recent value for every metric.
 *
 * @stability stable
 */
class SingleValueWidget extends widget_1.ConcreteWidget {
    /**
     * @stability stable
     */
    constructor(props) {
        super(props.width || 6, props.height || 3);
        this.props = props;
    }
    /**
     * Return the widget JSON for use in the dashboard.
     *
     * @stability stable
     */
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'singleValue',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    metrics: rendering_1.allMetricsGraphJson(this.props.metrics, []),
                    setPeriodToTimeRange: this.props.setPeriodToTimeRange,
                    singleValueFullPrecision: this.props.fullPrecision,
                },
            }];
    }
}
exports.SingleValueWidget = SingleValueWidget;
/**
 * Fill shading options that will be used with an annotation.
 *
 * @stability stable
 */
var Shading;
(function (Shading) {
    Shading["NONE"] = "none";
    Shading["ABOVE"] = "above";
    Shading["BELOW"] = "below";
})(Shading = exports.Shading || (exports.Shading = {}));
/**
 * A set of standard colours that can be used in annotations in a GraphWidget.
 *
 * @stability stable
 */
class Color {
}
exports.Color = Color;
/**
 * blue - hex #1f77b4.
 *
 * @stability stable
 */
Color.BLUE = '#1f77b4';
/**
 * brown - hex #8c564b.
 *
 * @stability stable
 */
Color.BROWN = '#8c564b';
/**
 * green - hex #2ca02c.
 *
 * @stability stable
 */
Color.GREEN = '#2ca02c';
/**
 * grey - hex #7f7f7f.
 *
 * @stability stable
 */
Color.GREY = '#7f7f7f';
/**
 * orange - hex #ff7f0e.
 *
 * @stability stable
 */
Color.ORANGE = '#ff7f0e';
/**
 * pink - hex #e377c2.
 *
 * @stability stable
 */
Color.PINK = '#e377c2';
/**
 * purple - hex #9467bd.
 *
 * @stability stable
 */
Color.PURPLE = '#9467bd';
/**
 * red - hex #d62728.
 *
 * @stability stable
 */
Color.RED = '#d62728';
/**
 * The position of the legend on a GraphWidget.
 *
 * @stability stable
 */
var LegendPosition;
(function (LegendPosition) {
    LegendPosition["BOTTOM"] = "bottom";
    LegendPosition["RIGHT"] = "right";
    LegendPosition["HIDDEN"] = "hidden";
})(LegendPosition = exports.LegendPosition || (exports.LegendPosition = {}));
function mapAnnotation(yAxis) {
    return (a) => {
        return { ...a, yAxis };
    };
}
//# sourceMappingURL=data:application/json;base64,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