"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AlarmRule = exports.AlarmState = void 0;
/**
 * Enumeration indicates state of Alarm used in building Alarm Rule.
 *
 * @stability stable
 */
var AlarmState;
(function (AlarmState) {
    AlarmState["ALARM"] = "ALARM";
    AlarmState["OK"] = "OK";
    AlarmState["INSUFFICIENT_DATA"] = "INSUFFICIENT_DATA";
})(AlarmState = exports.AlarmState || (exports.AlarmState = {}));
/**
 * Enumeration of supported Composite Alarms operators.
 */
var Operator;
(function (Operator) {
    Operator["AND"] = "AND";
    Operator["OR"] = "OR";
    Operator["NOT"] = "NOT";
})(Operator || (Operator = {}));
/**
 * Class with static functions to build AlarmRule for Composite Alarms.
 *
 * @stability stable
 */
class AlarmRule {
    /**
     * function to join all provided AlarmRules with AND operator.
     *
     * @param operands IAlarmRules to be joined with AND operator.
     * @stability stable
     */
    static allOf(...operands) {
        return this.concat(Operator.AND, ...operands);
    }
    /**
     * function to join all provided AlarmRules with OR operator.
     *
     * @param operands IAlarmRules to be joined with OR operator.
     * @stability stable
     */
    static anyOf(...operands) {
        return this.concat(Operator.OR, ...operands);
    }
    /**
     * function to wrap provided AlarmRule in NOT operator.
     *
     * @param operand IAlarmRule to be wrapped in NOT operator.
     * @stability stable
     */
    static not(operand) {
        return new class {
            renderAlarmRule() {
                return `(NOT (${operand.renderAlarmRule()}))`;
            }
        };
    }
    /**
     * function to build TRUE/FALSE intent for Rule Expression.
     *
     * @param value boolean value to be used in rule expression.
     * @stability stable
     */
    static fromBoolean(value) {
        return new class {
            renderAlarmRule() {
                return `${String(value).toUpperCase()}`;
            }
        };
    }
    /**
     * function to build Rule Expression for given IAlarm and AlarmState.
     *
     * @param alarm IAlarm to be used in Rule Expression.
     * @param alarmState AlarmState to be used in Rule Expression.
     * @stability stable
     */
    static fromAlarm(alarm, alarmState) {
        return new class {
            renderAlarmRule() {
                return `${alarmState}(${alarm.alarmArn})`;
            }
        };
    }
    /**
     * function to build Rule Expression for given Alarm Rule string.
     *
     * @param alarmRule string to be used in Rule Expression.
     * @stability stable
     */
    static fromString(alarmRule) {
        return new class {
            renderAlarmRule() {
                return alarmRule;
            }
        };
    }
    static concat(operator, ...operands) {
        return new class {
            renderAlarmRule() {
                const expression = operands
                    .map(operand => `${operand.renderAlarmRule()}`)
                    .join(` ${operator} `);
                return `(${expression})`;
            }
        };
    }
}
exports.AlarmRule = AlarmRule;
//# sourceMappingURL=data:application/json;base64,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