from typing import List, Optional

from gateau_api.game_ram.cartridge_info import CartridgeInfo, ChangeMeaning
from gateau_api.game_ram.pokemon import constants

# Gold sha1 d8b8a3600a465308c9953dfa04f0081c05bdcb94

#: Bit to meaning
BIT_TO_MEANING = {
    0xDBE4: {
        7: constants.BULBASAUR_OWNED,
        6: constants.IVYSAUR_OWNED,
        5: constants.VENUSAUR_OWNED,
        4: constants.CHARMANDER_OWNED,
        3: constants.CHARMELEON_OWNED,
        2: constants.CHARIZARD_OWNED,
        1: constants.SQUIRTLE_OWNED,
        0: constants.WARTORTLE_OWNED,
    },
    0xDBE5: {
        7: constants.BLASTOISE_OWNED,
        6: constants.CATERPIE_OWNED,
        5: constants.METAPOD_OWNED,
        4: constants.BUTTERFREE_OWNED,
        3: constants.WEEDLE_OWNED,
        2: constants.KAKUNA_OWNED,
        1: constants.BEEDRILL_OWNED,
        0: constants.PIDGEY_OWNED,
    },
    0xDBE6: {
        7: constants.PIDGEOTTO_OWNED,
        6: constants.PIDGEOT_OWNED,
        5: constants.RATTATA_OWNED,
        4: constants.RATICATE_OWNED,
        3: constants.SPEAROW_OWNED,
        2: constants.FEAROW_OWNED,
        1: constants.EKANS_OWNED,
        0: constants.ARBOK_OWNED,
    },
    0xDBE7: {
        7: constants.PIKACHU_OWNED,
        6: constants.RAICHU_OWNED,
        5: constants.SANDSHREW_OWNED,
        4: constants.SANDSLASH_OWNED,
        3: constants.NIDORAN_F_OWNED,
        2: constants.NIDORINA_OWNED,
        1: constants.NIDOQUEEN_OWNED,
        0: constants.NIDORAN_M_OWNED,
    },
    0xDBE8: {
        7: constants.NIDORINO_OWNED,
        6: constants.NIDOKING_OWNED,
        5: constants.CLEFAIRY_OWNED,
        4: constants.CLEFABLE_OWNED,
        3: constants.VULPIX_OWNED,
        2: constants.NINETALES_OWNED,
        1: constants.JIGGLYPUFF_OWNED,
        0: constants.WIGGLYTUFF_OWNED,
    },
    0xDBE9: {
        7: constants.ZUBAT_OWNED,
        6: constants.GOLBAT_OWNED,
        5: constants.ODDISH_OWNED,
        4: constants.GLOOM_OWNED,
        3: constants.VILEPLUME_OWNED,
        2: constants.PARAS_OWNED,
        1: constants.PARASECT_OWNED,
        0: constants.VENONAT_OWNED,
    },
    0xDBEA: {
        7: constants.VENOMOTH_OWNED,
        6: constants.DIGLETT_OWNED,
        5: constants.DUGTRIO_OWNED,
        4: constants.MEOWTH_OWNED,
        3: constants.PERSIAN_OWNED,
        2: constants.PSYDUCK_OWNED,
        1: constants.GOLDUCK_OWNED,
        0: constants.MANKEY_OWNED,
    },
    0xDBEB: {
        7: constants.PRIMEAPE_OWNED,
        6: constants.GROWLITHE_OWNED,
        5: constants.ARCANINE_OWNED,
        4: constants.POLIWAG_OWNED,
        3: constants.POLIWHIRL_OWNED,
        2: constants.POLIWRATH_OWNED,
        1: constants.ABRA_OWNED,
        0: constants.KADABRA_OWNED,
    },
    0xDBEC: {
        7: constants.ALAKAZAM_OWNED,
        6: constants.MACHOP_OWNED,
        5: constants.MACHOKE_OWNED,
        4: constants.MACHAMP_OWNED,
        3: constants.BELLSPROUT_OWNED,
        2: constants.WEEPINBELL_OWNED,
        1: constants.VICTREEBEL_OWNED,
        0: constants.TENTACOOL_OWNED,
    },
    0xDBED: {
        7: constants.TENTACRUEL_OWNED,
        6: constants.GEODUDE_OWNED,
        5: constants.GRAVELER_OWNED,
        4: constants.GOLEM_OWNED,
        3: constants.PONYTA_OWNED,
        2: constants.RAPIDASH_OWNED,
        1: constants.SLOWPOKE_OWNED,
        0: constants.SLOWBRO_OWNED,
    },
    0xDBEE: {
        7: constants.MAGNEMITE_OWNED,
        6: constants.MAGNETON_OWNED,
        5: constants.FARFETCHD_OWNED,
        4: constants.DODUO_OWNED,
        3: constants.DODRIO_OWNED,
        2: constants.SEEL_OWNED,
        1: constants.DEWGONG_OWNED,
        0: constants.GRIMER_OWNED,
    },
    0xDBEF: {
        7: constants.MUK_OWNED,
        6: constants.SHELLDER_OWNED,
        5: constants.CLOYSTER_OWNED,
        4: constants.GASTLY_OWNED,
        3: constants.HAUNTER_OWNED,
        2: constants.GENGAR_OWNED,
        1: constants.ONIX_OWNED,
        0: constants.DROWZEE_OWNED,
    },
    0xDBF0: {
        7: constants.HYPNO_OWNED,
        6: constants.KRABBY_OWNED,
        5: constants.KINGLER_OWNED,
        4: constants.VOLTORB_OWNED,
        3: constants.ELECTRODE_OWNED,
        2: constants.EXEGGCUTE_OWNED,
        1: constants.EXEGGUTOR_OWNED,
        0: constants.CUBONE_OWNED,
    },
    0xDBF1: {
        7: constants.MAROWAK_OWNED,
        6: constants.HITMONLEE_OWNED,
        5: constants.HITMONCHAN_OWNED,
        4: constants.LICKITUNG_OWNED,
        3: constants.KOFFING_OWNED,
        2: constants.WEEZING_OWNED,
        1: constants.RHYHORN_OWNED,
        0: constants.RHYDON_OWNED,
    },
    0xDBF2: {
        7: constants.CHANSEY_OWNED,
        6: constants.TANGELA_OWNED,
        5: constants.KANGASKHAN_OWNED,
        4: constants.HORSEA_OWNED,
        3: constants.SEADRA_OWNED,
        2: constants.GOLDEEN_OWNED,
        1: constants.SEAKING_OWNED,
        0: constants.STARYU_OWNED,
    },
    0xDBF3: {
        7: constants.STARMIE_OWNED,
        6: constants.MRMIME_OWNED,
        5: constants.SCYTHER_OWNED,
        4: constants.JYNX_OWNED,
        3: constants.ELECTABUZZ_OWNED,
        2: constants.MAGMAR_OWNED,
        1: constants.PINSIR_OWNED,
        0: constants.TAUROS_OWNED,
    },
    0xDBF4: {
        7: constants.MAGIKARP_OWNED,
        6: constants.GYARADOS_OWNED,
        5: constants.LAPRAS_OWNED,
        4: constants.DITTO_OWNED,
        3: constants.EEVEE_OWNED,
        2: constants.VAPOREON_OWNED,
        1: constants.JOLTEON_OWNED,
        0: constants.FLAREON_OWNED,
    },
    0xDBF5: {
        7: constants.PORYGON_OWNED,
        6: constants.OMANYTE_OWNED,
        5: constants.OMASTAR_OWNED,
        4: constants.KABUTO_OWNED,
        3: constants.KABUTOPS_OWNED,
        2: constants.AERODACTYL_OWNED,
        1: constants.SNORLAX_OWNED,
        0: constants.ARTICUNO_OWNED,
    },
    0xDBF6: {
        7: constants.ZAPDOS_OWNED,
        6: constants.MOLTRES_OWNED,
        5: constants.DRATINI_OWNED,
        4: constants.DRAGONAIR_OWNED,
        3: constants.DRAGONITE_OWNED,
        2: constants.MEWTWO_OWNED,
        1: constants.MEW_OWNED,
        0: constants.CHIKORITA_OWNED,
    },
    0xDBF7: {
        7: constants.BAYLEEF_OWNED,
        6: constants.MEGANIUM_OWNED,
        5: constants.CYNDAQUIL_OWNED,
        4: constants.QUILAVA_OWNED,
        3: constants.TYPHLOSION_OWNED,
        2: constants.TOTODILE_OWNED,
        1: constants.CROCONAW_OWNED,
        0: constants.FERALIGATR_OWNED,
    },
    0xDBF8: {
        7: constants.SENTRET_OWNED,
        6: constants.FURRET_OWNED,
        5: constants.HOOTHOOT_OWNED,
        4: constants.NOCTOWL_OWNED,
        3: constants.LEDYBA_OWNED,
        2: constants.LEDIAN_OWNED,
        1: constants.SPINARAK_OWNED,
        0: constants.ARIADOS_OWNED,
    },
    0xDBF9: {
        7: constants.CROBAT_OWNED,
        6: constants.CHINCHOU_OWNED,
        5: constants.LANTURN_OWNED,
        4: constants.PICHU_OWNED,
        3: constants.CLEFFA_OWNED,
        2: constants.IGGLYBUFF_OWNED,
        1: constants.TOGEPI_OWNED,
        0: constants.TOGETIC_OWNED,
    },
    0xDBFA: {
        7: constants.NATU_OWNED,
        6: constants.XATU_OWNED,
        5: constants.MAREEP_OWNED,
        4: constants.FLAAFFY_OWNED,
        3: constants.AMPHAROS_OWNED,
        2: constants.BELLOSSOM_OWNED,
        1: constants.MARILL_OWNED,
        0: constants.AZUMARILL_OWNED,
    },
    0xDBFB: {
        7: constants.SUDOWOODO_OWNED,
        6: constants.POLITOED_OWNED,
        5: constants.HOPPIP_OWNED,
        4: constants.SKIPLOOM_OWNED,
        3: constants.JUMPLUFF_OWNED,
        2: constants.AIPOM_OWNED,
        1: constants.SUNKERN_OWNED,
        0: constants.SUNFLORA_OWNED,
    },
    0xDBFC: {
        7: constants.YANMA_OWNED,
        6: constants.WOOPER_OWNED,
        5: constants.QUAGSIRE_OWNED,
        4: constants.ESPEON_OWNED,
        3: constants.UMBREON_OWNED,
        2: constants.MURKROW_OWNED,
        1: constants.SLOWKING_OWNED,
        0: constants.MISDREAVUS_OWNED,
    },
    0xDBFD: {
        7: constants.UNOWN_OWNED,
        6: constants.WOBBUFFET_OWNED,
        5: constants.GIRAFARIG_OWNED,
        4: constants.PINECO_OWNED,
        3: constants.FORRETRESS_OWNED,
        2: constants.DUNSPARCE_OWNED,
        1: constants.GLIGAR_OWNED,
        0: constants.STEELIX_OWNED,
    },
    0xDBFE: {
        7: constants.SNUBBULL_OWNED,
        6: constants.GRANBULL_OWNED,
        5: constants.QWILFISH_OWNED,
        4: constants.SCIZOR_OWNED,
        3: constants.SHUCKLE_OWNED,
        2: constants.HERACROSS_OWNED,
        1: constants.SNEASEL_OWNED,
        0: constants.TEDDIURSA_OWNED,
    },
    0xDBFF: {
        7: constants.URSARING_OWNED,
        6: constants.SLUGMA_OWNED,
        5: constants.MAGCARGO_OWNED,
        4: constants.SWINUB_OWNED,
        3: constants.PILOSWINE_OWNED,
        2: constants.CORSOLA_OWNED,
        1: constants.REMORAID_OWNED,
        0: constants.OCTILLERY_OWNED,
    },
    0xDC00: {
        7: constants.DELIBIRD_OWNED,
        6: constants.MANTINE_OWNED,
        5: constants.SKARMORY_OWNED,
        4: constants.HOUNDOUR_OWNED,
        3: constants.HOUNDOOM_OWNED,
        2: constants.KINGDRA_OWNED,
        1: constants.PHANPY_OWNED,
        0: constants.DONPHAN_OWNED,
    },
    0xDC01: {
        7: constants.PORYGON2_OWNED,
        6: constants.STANTLER_OWNED,
        5: constants.SMEARGLE_OWNED,
        4: constants.TYROGUE_OWNED,
        3: constants.HITMONTOP_OWNED,
        2: constants.SMOOCHUM_OWNED,
        1: constants.ELEKID_OWNED,
        0: constants.MAGBY_OWNED,
    },
    0xDC02: {
        7: constants.MILTANK_OWNED,
        6: constants.BLISSEY_OWNED,
        5: constants.RAIKOU_OWNED,
        4: constants.ENTEI_OWNED,
        3: constants.SUICUNE_OWNED,
        2: constants.LARVITAR_OWNED,
        1: constants.PUPITAR_OWNED,
        0: constants.TYRANITAR_OWNED,
    },
    0xDC03: {
        7: constants.LUGIA_OWNED,
        6: constants.HO_OH_OWNED,
        5: constants.CELEBI_OWNED,
    },
    0xDC04: {
        7: constants.BULBASAUR_SEEN,
        6: constants.IVYSAUR_SEEN,
        5: constants.VENUSAUR_SEEN,
        4: constants.CHARMANDER_SEEN,
        3: constants.CHARMELEON_SEEN,
        2: constants.CHARIZARD_SEEN,
        1: constants.SQUIRTLE_SEEN,
        0: constants.WARTORTLE_SEEN,
    },
    0xDC05: {
        7: constants.BLASTOISE_SEEN,
        6: constants.CATERPIE_SEEN,
        5: constants.METAPOD_SEEN,
        4: constants.BUTTERFREE_SEEN,
        3: constants.WEEDLE_SEEN,
        2: constants.KAKUNA_SEEN,
        1: constants.BEEDRILL_SEEN,
        0: constants.PIDGEY_SEEN,
    },
    0xDC06: {
        7: constants.PIDGEOTTO_SEEN,
        6: constants.PIDGEOT_SEEN,
        5: constants.RATTATA_SEEN,
        4: constants.RATICATE_SEEN,
        3: constants.SPEAROW_SEEN,
        2: constants.FEAROW_SEEN,
        1: constants.EKANS_SEEN,
        0: constants.ARBOK_SEEN,
    },
    0xDC07: {
        7: constants.PIKACHU_SEEN,
        6: constants.RAICHU_SEEN,
        5: constants.SANDSHREW_SEEN,
        4: constants.SANDSLASH_SEEN,
        3: constants.NIDORAN_F_SEEN,
        2: constants.NIDORINA_SEEN,
        1: constants.NIDOQUEEN_SEEN,
        0: constants.NIDORAN_M_SEEN,
    },
    0xDC08: {
        7: constants.NIDORINO_SEEN,
        6: constants.NIDOKING_SEEN,
        5: constants.CLEFAIRY_SEEN,
        4: constants.CLEFABLE_SEEN,
        3: constants.VULPIX_SEEN,
        2: constants.NINETALES_SEEN,
        1: constants.JIGGLYPUFF_SEEN,
        0: constants.WIGGLYTUFF_SEEN,
    },
    0xDC09: {
        7: constants.ZUBAT_SEEN,
        6: constants.GOLBAT_SEEN,
        5: constants.ODDISH_SEEN,
        4: constants.GLOOM_SEEN,
        3: constants.VILEPLUME_SEEN,
        2: constants.PARAS_SEEN,
        1: constants.PARASECT_SEEN,
        0: constants.VENONAT_SEEN,
    },
    0xDC0A: {
        7: constants.VENOMOTH_SEEN,
        6: constants.DIGLETT_SEEN,
        5: constants.DUGTRIO_SEEN,
        4: constants.MEOWTH_SEEN,
        3: constants.PERSIAN_SEEN,
        2: constants.PSYDUCK_SEEN,
        1: constants.GOLDUCK_SEEN,
        0: constants.MANKEY_SEEN,
    },
    0xDC0B: {
        7: constants.PRIMEAPE_SEEN,
        6: constants.GROWLITHE_SEEN,
        5: constants.ARCANINE_SEEN,
        4: constants.POLIWAG_SEEN,
        3: constants.POLIWHIRL_SEEN,
        2: constants.POLIWRATH_SEEN,
        1: constants.ABRA_SEEN,
        0: constants.KADABRA_SEEN,
    },
    0xDC0C: {
        7: constants.ALAKAZAM_SEEN,
        6: constants.MACHOP_SEEN,
        5: constants.MACHOKE_SEEN,
        4: constants.MACHAMP_SEEN,
        3: constants.BELLSPROUT_SEEN,
        2: constants.WEEPINBELL_SEEN,
        1: constants.VICTREEBEL_SEEN,
        0: constants.TENTACOOL_SEEN,
    },
    0xDC0D: {
        7: constants.TENTACRUEL_SEEN,
        6: constants.GEODUDE_SEEN,
        5: constants.GRAVELER_SEEN,
        4: constants.GOLEM_SEEN,
        3: constants.PONYTA_SEEN,
        2: constants.RAPIDASH_SEEN,
        1: constants.SLOWPOKE_SEEN,
        0: constants.SLOWBRO_SEEN,
    },
    0xDC0E: {
        7: constants.MAGNEMITE_SEEN,
        6: constants.MAGNETON_SEEN,
        5: constants.FARFETCHD_SEEN,
        4: constants.DODUO_SEEN,
        3: constants.DODRIO_SEEN,
        2: constants.SEEL_SEEN,
        1: constants.DEWGONG_SEEN,
        0: constants.GRIMER_SEEN,
    },
    0xDC0F: {
        7: constants.MUK_SEEN,
        6: constants.SHELLDER_SEEN,
        5: constants.CLOYSTER_SEEN,
        4: constants.GASTLY_SEEN,
        3: constants.HAUNTER_SEEN,
        2: constants.GENGAR_SEEN,
        1: constants.ONIX_SEEN,
        0: constants.DROWZEE_SEEN,
    },
    0xDC10: {
        7: constants.HYPNO_SEEN,
        6: constants.KRABBY_SEEN,
        5: constants.KINGLER_SEEN,
        4: constants.VOLTORB_SEEN,
        3: constants.ELECTRODE_SEEN,
        2: constants.EXEGGCUTE_SEEN,
        1: constants.EXEGGUTOR_SEEN,
        0: constants.CUBONE_SEEN,
    },
    0xDC11: {
        7: constants.MAROWAK_SEEN,
        6: constants.HITMONLEE_SEEN,
        5: constants.HITMONCHAN_SEEN,
        4: constants.LICKITUNG_SEEN,
        3: constants.KOFFING_SEEN,
        2: constants.WEEZING_SEEN,
        1: constants.RHYHORN_SEEN,
        0: constants.RHYDON_SEEN,
    },
    0xDC12: {
        7: constants.CHANSEY_SEEN,
        6: constants.TANGELA_SEEN,
        5: constants.KANGASKHAN_SEEN,
        4: constants.HORSEA_SEEN,
        3: constants.SEADRA_SEEN,
        2: constants.GOLDEEN_SEEN,
        1: constants.SEAKING_SEEN,
        0: constants.STARYU_SEEN,
    },
    0xDC13: {
        7: constants.STARMIE_SEEN,
        6: constants.MRMIME_SEEN,
        5: constants.SCYTHER_SEEN,
        4: constants.JYNX_SEEN,
        3: constants.ELECTABUZZ_SEEN,
        2: constants.MAGMAR_SEEN,
        1: constants.PINSIR_SEEN,
        0: constants.TAUROS_SEEN,
    },
    0xDC14: {
        7: constants.MAGIKARP_SEEN,
        6: constants.GYARADOS_SEEN,
        5: constants.LAPRAS_SEEN,
        4: constants.DITTO_SEEN,
        3: constants.EEVEE_SEEN,
        2: constants.VAPOREON_SEEN,
        1: constants.JOLTEON_SEEN,
        0: constants.FLAREON_SEEN,
    },
    0xDC15: {
        7: constants.PORYGON_SEEN,
        6: constants.OMANYTE_SEEN,
        5: constants.OMASTAR_SEEN,
        4: constants.KABUTO_SEEN,
        3: constants.KABUTOPS_SEEN,
        2: constants.AERODACTYL_SEEN,
        1: constants.SNORLAX_SEEN,
        0: constants.ARTICUNO_SEEN,
    },
    0xDC16: {
        7: constants.ZAPDOS_SEEN,
        6: constants.MOLTRES_SEEN,
        5: constants.DRATINI_SEEN,
        4: constants.DRAGONAIR_SEEN,
        3: constants.DRAGONITE_SEEN,
        2: constants.MEWTWO_SEEN,
        1: constants.MEW_SEEN,
        0: constants.CHIKORITA_SEEN,
    },
    0xDC17: {
        7: constants.BAYLEEF_SEEN,
        6: constants.MEGANIUM_SEEN,
        5: constants.CYNDAQUIL_SEEN,
        4: constants.QUILAVA_SEEN,
        3: constants.TYPHLOSION_SEEN,
        2: constants.TOTODILE_SEEN,
        1: constants.CROCONAW_SEEN,
        0: constants.FERALIGATR_SEEN,
    },
    0xDC18: {
        7: constants.SENTRET_SEEN,
        6: constants.FURRET_SEEN,
        5: constants.HOOTHOOT_SEEN,
        4: constants.NOCTOWL_SEEN,
        3: constants.LEDYBA_SEEN,
        2: constants.LEDIAN_SEEN,
        1: constants.SPINARAK_SEEN,
        0: constants.ARIADOS_SEEN,
    },
    0xDC19: {
        7: constants.CROBAT_SEEN,
        6: constants.CHINCHOU_SEEN,
        5: constants.LANTURN_SEEN,
        4: constants.PICHU_SEEN,
        3: constants.CLEFFA_SEEN,
        2: constants.IGGLYBUFF_SEEN,
        1: constants.TOGEPI_SEEN,
        0: constants.TOGETIC_SEEN,
    },
    0xDC1A: {
        7: constants.NATU_SEEN,
        6: constants.XATU_SEEN,
        5: constants.MAREEP_SEEN,
        4: constants.FLAAFFY_SEEN,
        3: constants.AMPHAROS_SEEN,
        2: constants.BELLOSSOM_SEEN,
        1: constants.MARILL_SEEN,
        0: constants.AZUMARILL_SEEN,
    },
    0xDC1B: {
        7: constants.SUDOWOODO_SEEN,
        6: constants.POLITOED_SEEN,
        5: constants.HOPPIP_SEEN,
        4: constants.SKIPLOOM_SEEN,
        3: constants.JUMPLUFF_SEEN,
        2: constants.AIPOM_SEEN,
        1: constants.SUNKERN_SEEN,
        0: constants.SUNFLORA_SEEN,
    },
    0xDC1C: {
        7: constants.YANMA_SEEN,
        6: constants.WOOPER_SEEN,
        5: constants.QUAGSIRE_SEEN,
        4: constants.ESPEON_SEEN,
        3: constants.UMBREON_SEEN,
        2: constants.MURKROW_SEEN,
        1: constants.SLOWKING_SEEN,
        0: constants.MISDREAVUS_SEEN,
    },
    0xDC1D: {
        7: constants.UNOWN_SEEN,
        6: constants.WOBBUFFET_SEEN,
        5: constants.GIRAFARIG_SEEN,
        4: constants.PINECO_SEEN,
        3: constants.FORRETRESS_SEEN,
        2: constants.DUNSPARCE_SEEN,
        1: constants.GLIGAR_SEEN,
        0: constants.STEELIX_SEEN,
    },
    0xDC1E: {
        7: constants.SNUBBULL_SEEN,
        6: constants.GRANBULL_SEEN,
        5: constants.QWILFISH_SEEN,
        4: constants.SCIZOR_SEEN,
        3: constants.SHUCKLE_SEEN,
        2: constants.HERACROSS_SEEN,
        1: constants.SNEASEL_SEEN,
        0: constants.TEDDIURSA_SEEN,
    },
    0xDC1F: {
        7: constants.URSARING_SEEN,
        6: constants.SLUGMA_SEEN,
        5: constants.MAGCARGO_SEEN,
        4: constants.SWINUB_SEEN,
        3: constants.PILOSWINE_SEEN,
        2: constants.CORSOLA_SEEN,
        1: constants.REMORAID_SEEN,
        0: constants.OCTILLERY_SEEN,
    },
    0xDC20: {
        7: constants.DELIBIRD_SEEN,
        6: constants.MANTINE_SEEN,
        5: constants.SKARMORY_SEEN,
        4: constants.HOUNDOUR_SEEN,
        3: constants.HOUNDOOM_SEEN,
        2: constants.KINGDRA_SEEN,
        1: constants.PHANPY_SEEN,
        0: constants.DONPHAN_SEEN,
    },
    0xDC21: {
        7: constants.PORYGON2_SEEN,
        6: constants.STANTLER_SEEN,
        5: constants.SMEARGLE_SEEN,
        4: constants.TYROGUE_SEEN,
        3: constants.HITMONTOP_SEEN,
        2: constants.SMOOCHUM_SEEN,
        1: constants.ELEKID_SEEN,
        0: constants.MAGBY_SEEN,
    },
    0xDC22: {
        7: constants.MILTANK_SEEN,
        6: constants.BLISSEY_SEEN,
        5: constants.RAIKOU_SEEN,
        4: constants.ENTEI_SEEN,
        3: constants.SUICUNE_SEEN,
        2: constants.LARVITAR_SEEN,
        1: constants.PUPITAR_SEEN,
        0: constants.TYRANITAR_SEEN,
    },
    0xDC23: {
        7: constants.LUGIA_SEEN,
        6: constants.HO_OH_SEEN,
        5: constants.CELEBI_SEEN,
    },
}

#: Bit-value meaning to byte address and bit
MEANING_TO_BIT = {}
for address, values in BIT_TO_MEANING.items():
    for bit, meaning in values.items():
        MEANING_TO_BIT[meaning] = (address, bit)


class PokemonGoldSilverInfo(CartridgeInfo):
    @staticmethod
    def byte_for_meaning(meaning: str) -> Optional[int]:
        """
        Get the address corresponding to a specific meaning for a given
        cartridge.

        If the meaning corresponds to a single bit within a byte, return the
        address of that byte

        Returns None if there is no corresponding byte.
        """

        if meaning in MEANING_TO_BIT:
            address, bit = MEANING_TO_BIT[meaning]
            return address

        return None

    @staticmethod
    def meaning_for_byte_change(
        address: int,
        old_value: Optional[int],
        new_value: int,
    ) -> List[ChangeMeaning]:
        """
        Meanings corresponding to a change in byte value from one to another.

        The return value is a list of tuples containing the meaning of the
        change, and whether it's a positive or negative result
        """

        if address in BIT_TO_MEANING:
            new_bits = [int(i) for i in "{0:08b}".format(new_value)]
            old_bits = (
                [int(i) for i in "{0:08b}".format(old_value)]
                if old_value is not None
                else [not i for i in new_bits]
            )
            changed_bits = [
                location
                for location in range(len(new_bits))
                if new_bits[location] != old_bits[location]
            ]

            results = []
            for change in changed_bits:
                try:
                    meaning = BIT_TO_MEANING[address][change]
                except KeyError:
                    continue
                positive = new_bits[change] == 1
                results.append(ChangeMeaning(meaning=meaning, value=positive))

            return results

        return []
