import { IOrigin, Behavior, BehaviorOptions, AddBehaviorOptions } from '@aws-cdk/aws-cloudfront';
import { OAuthScope, UserPoolClientIdentityProvider, IUserPool, IUserPoolClient } from '@aws-cdk/aws-cognito';
import { Construct } from '@aws-cdk/core';
import { LogLevel } from '@cloudcomponents/cdk-lambda-at-edge-pattern';
import { AuthFlow, RedirectPaths } from './auth-flow';
export interface UserPoolClientCallbackUrls {
    /**
     * A list of allowed redirect (callback) URLs for the identity providers.
     */
    readonly callbackUrls: string[];
    /**
     * A list of allowed logout URLs for the identity providers.
     */
    readonly logoutUrls: string[];
}
export interface IAuthorization {
    readonly redirectPaths: RedirectPaths;
    readonly signOutUrlPath: string;
    updateUserPoolClientCallbacks(redirects: UserPoolClientCallbackUrls): void;
    createDefaultBehavior(origin: IOrigin, options?: AddBehaviorOptions): BehaviorOptions;
    createAdditionalBehaviors(origin: IOrigin, options?: AddBehaviorOptions): Record<string, BehaviorOptions>;
    createLegacyDefaultBehavior(): Behavior;
    createLegacyAdditionalBehaviors(): Behavior[];
}
export interface AuthorizationProps {
    readonly userPool: IUserPool;
    readonly redirectPaths?: RedirectPaths;
    readonly signOutUrl?: string;
    readonly httpHeaders?: Record<string, string>;
    readonly logLevel?: LogLevel;
    readonly oauthScopes?: OAuthScope[];
    readonly cookieSettings?: Record<string, string>;
    readonly identityProviders?: UserPoolClientIdentityProvider[];
}
export declare abstract class Authorization extends Construct {
    readonly redirectPaths: RedirectPaths;
    readonly signOutUrlPath: string;
    readonly authFlow: AuthFlow;
    readonly userPoolClient: IUserPoolClient;
    protected readonly userPool: IUserPool;
    protected readonly oauthScopes: OAuthScope[];
    protected readonly cookieSettings: Record<string, string> | undefined;
    protected readonly httpHeaders: Record<string, string>;
    protected readonly nonceSigningSecret: string;
    protected readonly cognitoAuthDomain: string;
    protected readonly identityProviders: UserPoolClientIdentityProvider[];
    constructor(scope: Construct, id: string, props: AuthorizationProps);
    protected abstract createUserPoolClient(): IUserPoolClient;
    protected abstract createAuthFlow(logLevel: LogLevel): AuthFlow;
    updateUserPoolClientCallbacks(redirects: UserPoolClientCallbackUrls): void;
    createDefaultBehavior(origin: IOrigin, options?: AddBehaviorOptions): BehaviorOptions;
    createLegacyDefaultBehavior(options?: Behavior): Behavior;
    createAdditionalBehaviors(origin: IOrigin, options?: AddBehaviorOptions): Record<string, BehaviorOptions>;
    createLegacyAdditionalBehaviors(options?: Behavior): Behavior[];
    private generateNonceSigningSecret;
    private retrieveCognitoAuthDomain;
}
export interface ISpaAuthorization extends IAuthorization {
    readonly mode: Mode.SPA;
}
export declare type SpaAuthorizationProps = AuthorizationProps;
export declare class SpaAuthorization extends Authorization implements ISpaAuthorization {
    readonly mode = Mode.SPA;
    constructor(scope: Construct, id: string, props: SpaAuthorizationProps);
    protected createUserPoolClient(): IUserPoolClient;
    protected createAuthFlow(logLevel: LogLevel): AuthFlow;
}
export interface IStaticSiteAuthorization extends IAuthorization {
    readonly mode: Mode.STATIC_SITE;
}
export declare type StaticSiteAuthorizationProps = AuthorizationProps;
export declare class StaticSiteAuthorization extends Authorization implements IStaticSiteAuthorization {
    readonly mode = Mode.STATIC_SITE;
    constructor(scope: Construct, id: string, props: StaticSiteAuthorizationProps);
    protected createUserPoolClient(): IUserPoolClient;
    protected createAuthFlow(logLevel: LogLevel): AuthFlow;
    private retrieveUserPoolClientSecret;
}
export declare enum Mode {
    SPA = "SPA",
    STATIC_SITE = "STATIC_SITE"
}
