"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RouteSpec = void 0;
const shared_interfaces_1 = require("./shared-interfaces");
/**
 * (experimental) Used to generate specs with different protocols for a RouteSpec.
 *
 * @experimental
 */
class RouteSpec {
    /**
     * (experimental) Creates an HTTP Based RouteSpec.
     *
     * @experimental
     */
    static http(options) {
        return new HttpRouteSpec(options, shared_interfaces_1.Protocol.HTTP);
    }
    /**
     * (experimental) Creates an HTTP2 Based RouteSpec.
     *
     * @experimental
     */
    static http2(options) {
        return new HttpRouteSpec(options, shared_interfaces_1.Protocol.HTTP2);
    }
    /**
     * (experimental) Creates a TCP Based RouteSpec.
     *
     * @experimental
     */
    static tcp(options) {
        return new TcpRouteSpec(options);
    }
    /**
     * (experimental) Creates a GRPC Based RouteSpec.
     *
     * @experimental
     */
    static grpc(options) {
        return new GrpcRouteSpec(options);
    }
}
exports.RouteSpec = RouteSpec;
class HttpRouteSpec extends RouteSpec {
    constructor(props, protocol) {
        super();
        this.protocol = protocol;
        this.match = props.match;
        this.weightedTargets = props.weightedTargets;
    }
    bind(_scope) {
        const prefixPath = this.match ? this.match.prefixPath : '/';
        if (prefixPath[0] != '/') {
            throw new Error(`Prefix Path must start with \'/\', got: ${prefixPath}`);
        }
        const httpConfig = {
            action: {
                weightedTargets: renderWeightedTargets(this.weightedTargets),
            },
            match: {
                prefix: prefixPath,
            },
        };
        return {
            httpRouteSpec: this.protocol === shared_interfaces_1.Protocol.HTTP ? httpConfig : undefined,
            http2RouteSpec: this.protocol === shared_interfaces_1.Protocol.HTTP2 ? httpConfig : undefined,
        };
    }
}
class TcpRouteSpec extends RouteSpec {
    constructor(props) {
        super();
        this.weightedTargets = props.weightedTargets;
    }
    bind(_scope) {
        return {
            tcpRouteSpec: {
                action: {
                    weightedTargets: renderWeightedTargets(this.weightedTargets),
                },
            },
        };
    }
}
class GrpcRouteSpec extends RouteSpec {
    constructor(props) {
        super();
        this.weightedTargets = props.weightedTargets;
        this.match = props.match;
    }
    bind(_scope) {
        return {
            grpcRouteSpec: {
                action: {
                    weightedTargets: renderWeightedTargets(this.weightedTargets),
                },
                match: {
                    serviceName: this.match.serviceName,
                },
            },
        };
    }
}
/**
* Utility method to add weighted route targets to an existing route
*/
function renderWeightedTargets(weightedTargets) {
    const renderedTargets = [];
    for (const t of weightedTargets) {
        renderedTargets.push({
            virtualNode: t.virtualNode.virtualNodeName,
            weight: t.weight || 1,
        });
    }
    return renderedTargets;
}
//# sourceMappingURL=data:application/json;base64,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