# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['_ptcsvp', 'pytest_csv_params']

package_data = \
{'': ['*']}

install_requires = \
['pytest>=7.1.2,<8.0.0']

entry_points = \
{'pytest11': ['pytest-csv-params = pytest_csv_params.plugin']}

setup_kwargs = {
    'name': 'pytest-csv-params',
    'version': '0.4.0',
    'description': 'Pytest plugin for Test Case Parametrization with CSV files',
    'long_description': '![pytest-csv-params](docs/icon/pytest-csv-params.png)\n\n# pytest-csv-params\n\nA pytest plugin to parametrize data-driven tests by CSV files.\n\n[![PyPI - Downloads](https://img.shields.io/pypi/dw/pytest-csv-params?label=PyPI%20downloads&style=for-the-badge)](https://pypi.org/project/pytest-csv-params/)\n[![PyPI - Version](https://img.shields.io/pypi/v/pytest-csv-params?label=PyPI%20version&style=for-the-badge)](https://pypi.org/project/pytest-csv-params/)\n[![PyPI - Status](https://img.shields.io/pypi/status/pytest-csv-params?label=PyPI%20status&style=for-the-badge)](https://pypi.org/project/pytest-csv-params/)\n[![PyPI - Format](https://img.shields.io/pypi/format/pytest-csv-params?label=PyPI%20format&style=for-the-badge)](https://pypi.org/project/pytest-csv-params/)\n\n## Requirements\n \n- Python 3.8, 3.9 or 3.10\n- pytest >= 7.1\n\nThere\'s no operating system dependent code in this plugin, so it should run anywhere where pytest runs.\n\n## Installation\n\nSimply install it with pip...\n\n```bash\npip install pytest-csv-params\n```\n\n... or poetry ...\n\n```bash\npoetry add --dev pytest-csv-params\n```\n\n## Documentation / User Guide\n\n**Detailed documentation can be found under\n[docs.codebau.dev/pytest-plugins/pytest-csv-params/](https://docs.codebau.dev/pytest-plugins/pytest-csv-params/)**\n\n## Usage: Command Line Argument\n\n| Argument                | Required      | Description                                                          | Example                                      |\n|-------------------------|---------------|----------------------------------------------------------------------|----------------------------------------------|\n| `--csv-params-base-dir` | no (optional) | Define a base dir for all relative-path CSV data files (since 0.1.0) | `pytest --csv-params-base-dir /var/testdata` |\n\n## Usage: Decorator\n\nSimply decorate your test method with `@csv_params` (`pytest_csv_params.decorator.csv_params`) and the following parameters:\n\n| Parameter        | Type                     | Description                                                                                                                            | Example                                                                                        |\n|------------------|--------------------------|----------------------------------------------------------------------------------------------------------------------------------------|------------------------------------------------------------------------------------------------|\n| `data_file`      | `str`                    | The CSV file to use, relative or absolute path                                                                                         | `"/var/testdata/test1.csv"`                                                                    |\n| `base_dir`       | `str` (optional)         | Directory to look up relative CSV files (see `data_file`); overrides the command line argument                                         | `join(dirname(__file__), "assets")`                                                            |\n| `id_col`         | `str` (optional)         | Column name of the CSV that contains test case IDs                                                                                     | `"ID#"`                                                                                        |\n| `dialect`        | `csv.Dialect` (optional) | CSV Dialect definition (see [Python CSV Documentation](https://docs.python.org/3/library/csv.html#dialects-and-formatting-parameters)) | `csv.excel_tab`                                                                                |\n| `data_casts`     | `dict` (optional)        | Cast Methods for the CSV Data (see "Data Casting" below)                                                                               | `{ "a": int, "b": float }`                                                                     |\n| `header_renames` | `dict` (optional)        | Replace headers from the CSV file, so that they can be used as parameters for the test function (since 0.3.0)                          | `{ "Annual Amount of Bananas": "banana_count", "Cherry export price": "cherry_export_price" }` | \n\n## CSV Format\n\nThe default CSV format is:\n\n- `\\r\\n` as line ending\n- All non-numeric fields are surrounded by `"`\n- If you need a `"` in the value, use `""` (double quote)\n- Fields are separated by comma (`,`)\n\n## Usage Example\n\nThis example uses the CSV example from above.\n\n```python\nfrom pytest_csv_params.decorator import csv_params\n\n@csv_params(\n    data_file="/data/test-lib/cases/addition.csv",\n    id_col="ID#",\n    data_casts={\n        "part_a": int,\n        "part_b": int,\n        "expected_result": int,\n    },\n)\ndef test_addition(part_a, part_b, expected_result):\n    assert part_a + part_b == expected_result\n```\n\nShorthand example (no ID col, only string values):\n\n```python\nfrom pytest_csv_params.decorator import csv_params\n\n@csv_params("/data/test-lib/cases/texts.csv")\ndef test_texts(text_a, text_b, text_c):\n    assert f"{text_a}:{text_b}" == text_c\n```\n\n### More complex example\n\nThis example features nearly all things the plugin has to offer. You find this example also in the test cases, see `tests/test_complex_example.py`.\n\nThe CSV file (`tests/assets/example.csv`):\n\n```text\n"Test ID","Bananas shipped","Single Banana Weight","Apples shipped","Single Apple Weight","Container Size"\n"Order-7","1503","0.5","2545","0.25","1500"\n"Order-15","101","0.55","1474","0.33","550"\n```\n\nThe Test (`tests/test_complex_example.py`):\n\n```python\nfrom math import ceil\nfrom os.path import join, dirname\n\nfrom pytest_csv_params.decorator import csv_params\n\n\n@csv_params(\n    data_file="example.csv",\n    base_dir=join(dirname(__file__), "assets"),\n    id_col="Test ID",\n    header_renames={\n        "Bananas shipped": "bananas_shipped",\n        "Single Banana Weight": "banana_weight",\n        "Apples shipped": "apples_shipped",\n        "Single Apple Weight": "apple_weight",\n        "Container Size": "container_size",\n    },\n    data_casts={\n        "bananas_shipped": int,\n        "banana_weight": float,\n        "apples_shipped": int,\n        "apple_weight": float,\n        "container_size": int,\n    },\n)\ndef test_container_size_is_big_enough(\n    bananas_shipped: int, banana_weight: float, apples_shipped: int, apple_weight: float, container_size: int\n) -> None:\n    """\n    This is just an example test case for the documentation.\n    """\n\n    gross_weight = (banana_weight * bananas_shipped) + (apple_weight * apples_shipped)\n    assert ceil(gross_weight) <= container_size\n```\n\nIf you decide not to rename the columns, the test would look like this:\n\n```python\n@csv_params(\n    data_file="example.csv",\n    base_dir=join(dirname(__file__), "assets"),\n    id_col="Test ID",\n    data_casts={\n        "Bananas_Shipped": int,\n        "Single_Banana_Weight": float,\n        "Apples_Shipped": int,\n        "Single_Apple_Weight": float,\n        "Container_Size": int,\n    },\n)\ndef test_container_size_is_big_enough(\n    Bananas_Shipped: int, Single_Banana_Weight: float, Apples_Shipped: int, Single_Apple_Weight: float, Container_Size: int\n) -> None:\n    ...\n```\n\n## Changelog\n\n- A detailed changelog is here:\n  [docs.codebau.dev/pytest-plugins/pytest-csv-params/pages/changelog.html](https://docs.codebau.dev/pytest-plugins/pytest-csv-params/pages/changelog.html)\n\n## Bugs etc.\n\nPlease send your issues to `csv-params_issues` (at) `jued.de`. Please include the following:\n\n- Plugin Version used\n- Pytest version\n- Python version with operating system\n\nIt would be great if you could include example code that clarifies your issue.\n\nSee [CONTRIBUTING.md](CONTRIBUTING.md) for details.\n\n## Pull Requests\n\nPull requests are always welcome. Since this Gitea instance is not open to public, just send an e-mail to discuss options.\n\nAny changes that are made are to be backed by tests. Please give me a sign if you\'re going to break the existing API and let us discuss ways to handle that.\n\nSee [CONTRIBUTING.md](CONTRIBUTING.md) for details.\n\n## Where are the sources?\n\nThe source code is available under [git.codebau.dev/pytest-plugins/pytest-csv-params](https://git.codebau.dev/pytest-plugins/pytest-csv-params).\n',
    'author': 'Juergen Edelbluth',
    'author_email': 'csv_params@jued.de',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://git.codebau.dev/pytest-plugins/pytest-csv-params',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
